%% 
% Code used in the processing the data and generation of the figures for
% the paper "Portable system for flexible micro-endoscopy using a
% multi-mode fiber" 
%
% Each section generates the figures stated in the section's title. Note
% that time consuming sub-routines have been commented out in favour of
% reading in the preprocessed data. These can also be run should the user
% wish to make use of them. 
%
% Produced by Simon Peter Mekhail 2025
%
%% Figure 3 and 4
folder = [pwd,'\BendImaging'];

% --------------------------Start parameters-----------------------------
numObj = 6;
object = 1;
imageFiles = dir([folder,'\Fibre\*.jpg']);
numLoc = (length(imageFiles));
% -------------------------------------------------------------------------

% % -----------------Find fibre shapes at imaging locations------------------
% FS = cell(numLoc,1);
% fibreLength = 250;
% numPoints = 100;
% rc = 2*fibreLength/numPoints;
% for i = 1:numLoc
%     D = imread([folder,'\Fibre\',imageFiles(i).name]);
%     D = D(1051:1870,851:3020,:);
% 
%     C = rgb2hsv(D);
%     mask = and(and(C(:,:,1)>0.0,C(:,:,1)<0.09),C(:,:,2)>0.5);
%     for j = 1:10
%         filtMask = imfilter(mask+0,[1,1,1;1,0,1;1,1,1]);
%         mask = and(mask,filtMask>4);
%     end
%     img = imresize(mask+0,100/820);
%     [x,y] = meshgrid(1:265,1:100);
% 
%     F = nan*zeros(2,1.2*numPoints);
%     F(:,1) = [264;4];
%     grad = [-1;0];
%     stepSize = rc;
%     j = 1;
%     while stepSize>rc/10
%         imgCopy = img;
%         [~,r] = cart2pol(x-F(1,j)-rc*grad(1)/2,y-F(2,j)-rc*grad(2)/2);
%         imgCopy(r>rc) = 0;
%         F(:,j+1) = [sum(imgCopy(:).*x(:))/sum(imgCopy(:));sum(imgCopy(:).*y(:))/sum(imgCopy(:))];
%         stepSize = norm(F(:,j+1)-F(:,j));
%         grad = (F(:,j+1)-F(:,j))/stepSize;
%         j = j+1;
% 
%         figure(100)
%         subplot(2,1,1)
%         imagesc(cat(3,img,imgCopy,0*img))
%         hold on
%         plot(F(1,:),F(2,:),'bo')
%         hold off
%         title([num2str(j),'   ',num2str(stepSize)])
%         subplot(2,1,2)
%         quiver(0,0,grad(1),-grad(2)); axis([-1,1,-1,1]); axis square
%         title([num2str(grad(1)),',',num2str(-grad(2))])
%     end
%     FS{i} = reshape(F(~isnan(F)),2,[]);
% end
% save([folder,'\Fibre\fibreShapes.mat'],'FS')
% % -------------------------------------------------------------------------

% % -----------Crop and rotate GT images to match fibre images---------------
% objNames = {'Checkerboard','Magpie','Panda','Quantic'};
% res = 64;
% mask = makeImagingMask(res);
% top = [8,4,10,6];
% left = [14,8,15,22];
% siz = [470,550,545,652];
% ang = [0,0,0,0];
% GT = zeros(res,res,numObj-2);
% for i = 1:4
%     trueImage = mean(double(imread([folder,'\Objects\',objNames{i},'.png'])),3);
%     if i==1; trueImage = rot90(trueImage,1); end
%     fibImage = nan*zeros(res);
%     fibImage(mask) = readLabviewData([folder,'\TM1\Loc6\Image',num2str(i-1),'.dat'],1,'double');
% 
%     trueImage = imrotate(trueImage,ang(i));
%     trueImage = trueImage(top(i):(siz(i)+top(i)-1),left(i):(siz(i)+left(i)-1));
%     trueImage = imresize(trueImage,res/siz(i));
% 
%     trueImage(~mask) = nan;
% 
%     trueImage = mat2gray(trueImage);
%     fibImage = mat2gray(fibImage');
% 
%     pic = cat(3,trueImage,fibImage,zeros(res));
% 
%     figure(101);
%     imagesc(pic); axis off; axis square;
% 
%     trueImage(~mask) = nan;
%     GT(:,:,i) = trueImage;
% end
% save([folder,'\Objects\GT.mat'],'GT')
% % -------------------------------------------------------------------------

% % ------------------Determine contrast and SSIM measurments----------------
% numTM = 6;
% numPix = sum(mask(:));
% p = 0.01;
% res = 64;
% mask = makeImagingMask(res);
% load([folder,'\Objects\GT.mat'])
% GT = GT(:,:,object)';
% A = zeros(res*numTM,res*numLoc);
% data = zeros(numPix,3);
% s = zeros(numTM,numLoc);
% c = zeros(numTM,numLoc);
% for i = 0:numTM-1
%     for j = 0:numLoc-1
%         m = 1;
%         for k = [object-1,numObj-2,numObj-1]
%             imFile = ['\TM',num2str(i),'\Loc',num2str(j),'\Image',num2str(k),'.dat'];
%             data(:,m) = readLabviewData([folder,imFile],1,'double');
%             m = m+1;
%         end
%         data(:,1:2) = data(:,1:2)-repmat(data(:,3),1,2);
%         temp = nan*zeros(res);
%         temp(mask) = data(:,1)./data(:,2);
%         A(res*i+1:res*(i+1),res*j+1:res*(j+1)) = temp';
%         s(i+1,j+1) = ssim(data(:,1),GT(mask),'Exponents',[0,0,1]);
%         temp = sort(data(:,1));
%         lo = mean(temp(1:round(p*numPix)));
%         hi = mean(temp(numPix-round(p*numPix)+1:numPix));
%         c(i+1,j+1) = (hi-lo)/(hi+lo);
%     end
% end
% displayInds = [1,3:3:30];
% A = A(:,reshape(bsxfun(@plus,(displayInds-1)*res,(1:res)'),1,[]));
% x = readmatrix([folder,'\BendPoints.csv']);
% save([folder,'\FigureData.mat'],'A','x','s','c')
% % -------------------------------------------------------------------------

load([folder,'\Fibre\fibreShapes.mat'])
load([folder,'\FigureData.mat'])

figure(1); clf
set(gcf,'Units','Normalized','Position',[0,0.037,1,0.8917],'color','w')
subplot(8,8,1)
h1(1) = plotFibre(FS,3);
subplot(8,8,9)
h1(2) = plotFibre(FS,8);
subplot(8,8,17)
h1(3) = plotFibre(FS,13);
subplot(8,8,25)
h1(4) = plotFibre(FS,18);
subplot(8,8,33)
h1(5) = plotFibre(FS,23);
subplot(8,8,41)
h1(6) = plotFibre(FS,28);
subplot(8,8,2)
h1(7) = plotFibre(FS,1);
subplot(8,8,5)
h1(8) = plotFibre(FS,12);
subplot(8,8,8)
h1(9) = plotFibre(FS,30);
subplot(2,2,4)
imagesc(mat2gray(A)); axis image; colormap gray; box off
h1(10) = gca;
text(h1(10),0.35,1.2,'Imaging Position (\Deltaz mm)','FontSize',24,'Units','normalized','fontweight','bold');
text(h1(10),-0.2,0.2,'Calibration Position (\Deltaz mm)','FontSize',24,'Units','normalized','Rotation',90,'fontweight','bold');
set(h1(1),'position',[0.07,0.66,0.1,0.1])
set(h1(2),'position',[0.07,0.54,0.1,0.1])
set(h1(3),'position',[0.07,0.44,0.1,0.1])
set(h1(4),'position',[0.07,0.32,0.1,0.1])
set(h1(5),'position',[0.07,0.20,0.1,0.1])
set(h1(6),'position',[0.07,0.08,0.1,0.1])
set(h1(7),'position',[0.16,0.78,0.1,0.1])
set(h1(8),'position',[0.44,0.80,0.1,0.1])
set(h1(9),'position',[0.73,0.81,0.1,0.1])
set(h1(10),'position',[0.15,0.06,0.72,0.72],...
    'YTick',64*(1:6)-10,...
    'YTickLabel',{'8.6','60.4','117.9','158.2','187.0','204.2'},...
    'XTick',64*(0:10)+32,...
    'XTickLabel',{'0','8.6','37.4','71.9','106.4','135.2','158.2','175.5','192.7','201.4','210'}, ...
    'XaxisLocation','Top',...
    'fontsize',18,'ticklength',[0,0],'fontweight','bold')
h1(10).XAxis.Color = [1,1,1];
h1(10).XAxis.TickLabelColor = [0,0,0];
h1(10).YAxis.Color = [1,1,1];
h1(10).YAxis.TickLabelColor = [0,0,0];


figure(2); clf
subplot(2,6,7:9)
set(gcf,'Units','Normalized','Position',[0.01,0.15,0.48,0.45],'color','w')
palette = hsv2rgb([0.08,0.4,1.0; ...
    0.08,0.7,1.0; ...
    0.08,1.0,1.0; ...
    0.08,1.0,0.8; ...
    0.08,1.0,0.5; ...
    0.08,1.0,0.2]);
yyaxis left
colororder(palette)
plot(210*x/max(x),s','linewidth',1.5);
axis([0,210,round(min(s(:))-0.01,2),round(max(s(:))+0.01,2)])
ylabel('SSIM','fontsize',14)
yyaxis right
colororder(palette)
plot(210*x/max(x),c','--','linewidth',1.5);
axis([0,210,round(min(c(:))-0.01,2),round(max(c(:))+0.01,2)])
ylabel('Contrast','fontsize',14)
h2(1) = gca;
set(h2(1),'fontsize',14,'TickDir','out','linewidth',1.5)
h2(1).YAxis(1).Color = 'k';
h2(1).YAxis(2).Color = 'k';
h2(1).YAxis(1).TickValues = [round(min(s(:))-0.01,2),round(max(s(:))+0.01,2)];
h2(1).YAxis(2).TickValues = [round(min(c(:))-0.01,2),round(max(c(:))+0.01,2)];
h2(1).XAxis.TickValues = [0,105,210];
xlabel('Deformation distance \Delta z (mm)','fontsize',14)
box off
subplot(2,6,1)
h2(2) = plotFibre(FS,3,1);
subplot(2,6,2)
h2(3) = plotFibre(FS,8,1);
subplot(2,6,3)
h2(4) = plotFibre(FS,13,1);
subplot(2,6,4)
h2(5) = plotFibre(FS,18,1);
subplot(2,6,5)
h2(6) = plotFibre(FS,23,1);
subplot(2,6,6)
h2(7) = plotFibre(FS,28,1);
subplot(2,6,10)
h2(8) = plotFibre(FS,1,1);
subplot(2,6,11)
h2(9) = plotFibre(FS,30,1);
subplot(2,6,12)
plot([0,0],[0,1],'--w','linewidth',2.1); axis equal; axis off
h2(10) = gca;
text(h2(1),0.2,1.15,'TM performance under fibre deformation','FontSize',20,'Units','normalized')
fSiz = 0.06; fTop = 0.81;
set(h2(1),'position',[0.10,0.15,0.8,0.7])
set(h2(2),'position',[0.15,fTop,fSiz,fSiz])
set(h2(3),'position',[0.29,fTop,fSiz,fSiz])
set(h2(4),'position',[0.49,fTop,fSiz,fSiz])
set(h2(5),'position',[0.61,fTop,fSiz,fSiz])
set(h2(6),'position',[0.73,fTop,fSiz,fSiz])
set(h2(7),'position',[0.81,fTop,fSiz,fSiz])
set(h2(8),'position',[0.07,0.02,fSiz,fSiz])
set(h2(9),'position',[0.85,0.02,fSiz,fSiz])
set(h2(10),'position',[0.55,0.165,0.7,0.7])

%% Figure 5a and Video 1
folder = [pwd,'\BendSeries'];
numTM = 5;
numFrame = 456;
res = 60;
numPix = sum(makeImagingMask(res),'all');
frame = 53;
video = 1==0;

% % --------Determine contrast metrics for all pixels and n x n------------
% A = zeros(res,res,numFrame,numTM);
% for i = 1:numTM
%     temp = readLabviewData([folder,'\Videos\Flythrough',num2str(i-1),'.dat'],2,'double');
%     A(:,:,:,i) = rot90(temp(:,:,1:numFrame));
% end
% indsAll=find(~isnan(A(:,:,2,1)));
% pAll = 0.01;
% inds7 = makeGridInCircle(res,7);
% p7 = 1/7;
% C = zeros(numFrame,numTM,2);
% for j = 1:numFrame
%     for k = 1:numTM
%         img = A(:,:,j,k);
%         ePoints = sort(img(indsAll));
%         lower = sum(ePoints(1:round(pAll*length(indsAll))));
%         upper = sum(ePoints(round((1-pAll)*length(indsAll)):length(indsAll)));
%         C(j,k,1) = (upper-lower)/(lower+upper);
% 
%         ePoints = sort(img(inds7));
%         lower = sum(ePoints(1:round(p7*length(inds7))));
%         upper = sum(ePoints(round((1-p7)*length(inds7)):length(inds7)));
%         C(j,k,2) = (upper-lower)/(lower+upper);
%     end
% end
% 
% % ----------------Find fibre shapes for chess scans----------------------
% dataIn.vFibre = VideoReader([folder,'\Videos\FlythroughPath.mp4']);
% dataIn.startTFibre = 7.9;
% dataIn.leftFibre = 131;
% dataIn.rightFibre = 1330;
% dataIn.topFibre = 461;
% dataIn.bottomFibre = 620;
% dataIn.frameRate = 2e4/numPix;
% dataIn.numFrame = numFrame;
% dataIn.fibreLength = 250;
% dataIn.numPoints = 100;
% dataIn.figureNumber = 12;
% 
% FP = fibreShapesFromVideo(dataIn);
% 
% save([folder,'\Videos\multiFlythrough.mat'],'FP','C','A')
% % -----------------------------------------------------------------------

load([folder,'\Videos\multiFlythrough.mat'],'FP','C','A')

vChess = VideoReader([folder,'\Videos\ChessGT.mp4']);
startT = 1.8;
left = 130;
top = 50;
imWidth = 420;
mask = makeImagingMask(imWidth+1);
frameRate = 2e4/numPix;

inds = makeGridInCircle(res,7);
inds = reshape([inds';inds'+res^2;inds'+2*res^2],[],1);
ePoints = reshape(repmat(hsv2rgb([0.08,0.8,1])',1,49),[],1);

[~,IAll] = max(C(:,:,1),[],2);
[~,I7] = max(C(:,:,2),[],2);

CB = mean(imread([folder,'\chessBoard.jpg']),3)/255;
CB = CB(170:570,35:435)';
cXc = 5;
cYc = -40;
cS = 80;
xC = cS*linspace(-1,1,401)+cXc;
yC = cS*linspace(-1,1,401)+cYc;

NA = 0.26;
tLength = 250;
tVertices = [tLength*tan(asin(NA))*[1,-1];-tLength*[1,1]];
Q = @(x) [sin(x),cos(x);cos(x),-sin(x)];

if video
    v = VideoWriter([folder,'\Videos\ChessManual']);
    v.FrameRate = frameRate;
    open(v)
    vStart = 1;
    vEnd = numFrame;
    plotCol = 'k';
    textCol = 'w';
    fibNanMask = ~isnan(A(:,:,2,1));
else
    vStart = frame;
    vEnd = frame;
    plotCol = 'w';
    textCol = 'k';
end

w = 0.2;
pos = [0.02,0.3,w,w;
    0.24,0.3 w,w;
    0.45,0.3 w,w;
    0.13,0.05,w,w;
    0.34,0.05,w,w];

figure(3); clf;
set(gcf,'Color',plotCol,'units','normalized','Position',[0,0.04,0.7,0.88]);
h3 = repmat(gca,5,1);

for frame = vStart:vEnd

    % Plot the 5 runs
    for i = 1:numTM
        subplot(4,3,i+6);
        if i==I7(frame)
            img = repmat(mat2gray(A(:,:,frame,i)),1,1,3);
            if video; img = img.*repmat(fibNanMask,1,1,3); end
            img(inds) = ePoints;
            imagesc(img); axis image; axis off;
        else
            img = mat2gray(A(:,:,frame,i));
            if video; img = img.*fibNanMask; end
            imagesc(img); axis image; axis off; colormap gray
        end
        title(['TM ',num2str(i-1)],'Color',textCol,'fontsize',18);
        h3(i) = gca;
    end

    % Plot orange square around the best image (full frame)
    hold(h3(IAll(frame)),'on')
    plot(h3(IAll(frame)),res*[0,1,1,0,0]+0.5,res*[0,0,1,1,0]+0.5,'color',hsv2rgb([0.08,0.8,1]),'linewidth',2)
    hold(h3(IAll(frame)),'off')

    % Plot the best image large
    subplot(4,3,2);
    img = mat2gray(A(:,:,frame,I7(frame)));
    if video; img = img.*fibNanMask; end
    imagesc(img); axis image; axis off; colormap gray
    title('Highest Contrast (Grid)','Color',textCol,'fontsize',24)
    h3(6) = gca;

    % Plot the ground truth
    vChess.currentTime = startT+(frame-1)/frameRate;
    V = flipud(readFrame(vChess));
    V = mean(V(top:top+imWidth,left:left+imWidth,:),3)/255;
    V(~mask) = 1-video;
    subplot(4,3,1);
    imshow(V); axis image; axis off; colormap gray
    title('Ground Truth','Color',textCol,'fontsize',24)
    h3(7) = gca;

    % Find vertices of triangle and needle
    x = FP{frame}(1,:)/2; % X direction is twice Y from stepper motor definition for video
    if(and(~video,frame==53)); k = 15; x(end-k:end) = x(end-k:end)+(0:k)*3.5/k; end
    y = FP{frame}(2,:);
    nNear = [x(end-2);y(end-2)];
    g = [x(end)-x(end-10);y(end)-y(end-10)]; g = g/norm(g);
    nFar = nNear+20*g;
    tV = nFar-Q(angle(g(1)-1i*g(2)))*tVertices;

    % Plot fibre and chess board position
    subplot(4,3,3)
    imagesc(xC,yC,CB); colormap gray;
    hold on
    plot([nNear(1),nFar(1)],[nNear(2),nFar(2)],'color',0.4*[1,1,1],'linewidth',2)
    plot(x,y,'color',hsv2rgb([0.08,0.8,1]),'linewidth',2.5); axis equal; axis([cXc-cS,cXc+cS,cYc-cS,300]); axis off
    p1 = patch([nFar(1),tV(1,1),tV(1,2)],[nFar(2),tV(2,1),tV(2,2)],[0.8,0,0.2],'edgecolor','none');
    p1.FaceVertexAlphaData = [0.5;0;0];
    p1.FaceAlpha = 'interp';
    set(gca,'Ydir','reverse')
    plotArrow([45,290],2.5,23.2,0,textCol)
    plotArrow([46.25,291.25],2.5,23.2,-90,textCol)
    hold off
    title('Setup','Color',textCol,'fontsize',24)
    h3(8) = gca;

    for i = 1:5
        set(h3(i),'Position',pos(i,:))
    end
    set(h3(6),'Position',[0.32,0.56,0.38,0.38])
    set(h3(7),'Position',[-0.02,0.56,0.38,0.38])
    set(h3(8),'Position',[0.40,0.05,0.89,0.89])

    text(h3(8),0.9,0.04,'x','FontSize',16,'Units','normalized','fontweight','bold','color',textCol)
    text(h3(8),0.77,0.09,'z','FontSize',16,'Units','normalized','fontweight','bold','color',textCol)
    drawnow

    if video
        if frame>1
            thisFrame = getframe(gcf);
            writeVideo(v,thisFrame)
        end
    end
end

if video; close(v); end

%% Figure 5b
folder = [pwd,'\BendSeries'];
load([folder,'\Videos\multiFlythrough.mat'],'C')
res = 60;
numFrame = 455;
numTM = 5;
numPix = sum(makeImagingMask(res),'all');
frameRate = 2e4/numPix;
plotWidth = 10;
C = C(2:end,:,:);

palette = hsv2rgb([0.08,0.4,1.0; ...
    0.08,0.7,1.0; ...
    0.08,1.0,1.0; ...
    0.08,1.0,0.6; ...
    0.08,1.0,0.2]);


[X,Y] = meshgrid((0:numFrame-1)/frameRate,0:plotWidth-1);
[~,indAll] = max(C(:,:,1),[],2);
[~,ind7] = max(C(:,:,2),[],2);

sortAll = sort(C(:,:,1),2,'descend');
sort7 = sort(C(:,:,2),2,'descend');
diffAll = (sortAll(:,1)-sortAll(:,2))./mean(sortAll,2);
diff7 = (sort7(:,1)-sort7(:,2))./mean(sort7,2);

figure(4); clf
set(gcf,'color','w','units','normalized','Position',[0.02,0.3,0.45,0.3])
for i = 1:numTM
    TMAll = 1*(indAll==i);
    TMAll = repmat(TMAll',plotWidth,1);

    TM7 = 1*(ind7==i);
    TM7 = repmat(TM7',plotWidth,1);

    colour = repmat(permute(palette(i,:),[1,3,2]),plotWidth,numFrame,1);

    subplot(4,1,1)
    hold on
    surf(X,Y,TMAll,colour,'edgecolor','none'); axis([0,(numFrame-1)/frameRate,0,plotWidth-1,0.5,1]); view(2)
    ylabel({'Full frame','TM Selection'})
    hold off
    h4(1) = gca;

    subplot(4,1,3)
    hold on
    surf(X,Y,TM7,colour,'edgecolor','none'); axis([0,(numFrame-1)/frameRate,0,plotWidth-1,0.5,1]); view(2)
    ylabel({'Restricted grid','TM selection'})
    xlabel('Video time stamp (s)')
    hold off
    h4(2) = gca;
end

subplot(4,1,2)
imagesc(repmat(diffAll',plotWidth,1)); axis off; colormap gray
h4(3) = gca;

subplot(4,1,4)
imagesc(repmat(diff7',plotWidth,1)); axis off; colormap gray
h4(4) = gca;

set(h4(1),'fontsize',12,'Xtick',[],'Ytick',[],'Position',[0.06,0.63,0.8,0.3])
set(h4(2),'fontsize',12,'Xtick',[0,32,64],'Ytick',[],'Position',[0.06,0.2,0.8,0.3])
set(h4(3),'Position',[0.06,0.63,0.8,0.1])
set(h4(4),'Position',[0.06,0.2,0.8,0.1],'Clim',[0,0.2516])
legend(h4(1),'TM0','TM1','TM2','TM3','TM4','Position',[0.89,0.69,0.07,0.2],'fontsize',10,'box','off')
h4c = colorbar(h4(4),'Position',[0.88,0.2,0.02,0.35],'fontsize',10,'ticks',[0,0.25]);
h4c.Label.String = {'Normalised distance','to next best TM'};
h4c.FontSize = 12;
h4c.Label.FontSize = 12;

%% Video 2
folder = [pwd,'\BendSeries'];

res = 60;
numFrame = 429;
numTM = 5;
numPix = sum(makeImagingMask(res),'all');
frameRate = 2e4/(numPix+numTM*49);

% % ----------------Find fibre shapes for chess scans----------------------
% dataIn.vFibre = VideoReader([folder,'\Videos\FlythroughPath.mp4']);
% dataIn.startTFibre = 7.9;
% dataIn.leftFibre = 131;
% dataIn.rightFibre = 1330;
% dataIn.topFibre = 461;
% dataIn.bottomFibre = 620;
% dataIn.frameRate = frameRate;
% dataIn.numFrame = numFrame;
% dataIn.fibreLength = 250;
% dataIn.numPoints = 100;
% dataIn.figureNumber = 13;
% 
% FP = fibreShapesFromVideo(dataIn);
% 
% save([folder,'\Videos\autoFlythrough.mat'],'FP')
% % -------------------------------------------------------------------------

load([folder,'\Videos\autoFlythrough.mat'],'FP')
A = rot90(readLabviewData([folder,'\Videos\Flythrough.dat'],2,'double'));
numFrame = min([numFrame,length(FP)]);

CB = mean(imread([folder,'\chessBoard.jpg']),3)/255;
CB = CB(170:570,35:435)';
cXc = 10;
cYc = -50;
cS = 80;
xC = cS*linspace(-1,1,401)+cXc;
yC = cS*linspace(-1,1,401)+cYc;

NA = 0.275;
tLength = 250;
tVertices = [tLength*tan(asin(NA))*[1,-1];-tLength*[1,1]];
Q = @(x) [sin(x),cos(x);cos(x),-sin(x)];

vChess = VideoReader([folder,'\Videos\ChessFlythroughGT.mp4']);
vStartT = 1.35;
vLeft = 60;%65
vTop = 45;%50
vWidth = 430;%420
mask = makeImagingMask(vWidth+1);

v = VideoWriter([folder,'\Videos\ChessAuto']);
v.FrameRate = frameRate;
open(v)

figure(5)
set(gcf,'Color','k','units','normalized','position',[0.01,0.03,0.45,0.9])

for i = 1:numFrame
    % Plot ground truth
    vChess.currentTime = vStartT+(i-1)/frameRate;
    V = flipud(readFrame(vChess));
    V = mask.*mean(V(vTop:vTop+vWidth,vLeft:vLeft+vWidth,:),3)/255;
    subplot(3,1,2);
    imshow(V); axis image; axis off; colormap gray
    title('Ground Truth','Color','w','fontsize',24)
    h5(2) = gca;

    % Plot endoscope image
    subplot(3,1,1);
    imagesc(A(:,:,i)); axis image; axis off; colormap gray
    title('Fibre Endoscope','Color','w','fontsize',24)
    h5(1) = gca;

    % Find vertices of triangle and needle
    x = FP{i}(1,:)/2; % X direction is twice Y from stepper motor definition for video
    y = FP{i}(2,:);
    nNear = [x(end-2);y(end-2)];
    g = [x(end)-x(end-10);y(end)-y(end-10)]; g = g/norm(g);
    nFar = nNear+20*g;
    tV = nFar-Q(angle(g(1)-1i*g(2)))*tVertices;

    % Plot fibre and chess board position
    subplot(3,1,3)
    imagesc(xC,yC,CB); colormap gray;
    hold on
    plot([nNear(1),nFar(1)],[nNear(2),nFar(2)],'color',0.4*[1,1,1],'linewidth',2)
    plot(x,y,'color',hsv2rgb([0.08,0.8,1]),'linewidth',2.5); axis equal; axis([cXc-cS,cXc+cS,cYc-cS,300]); axis off
    p1 = patch([nFar(1),tV(1,1),tV(1,2)],[nFar(2),tV(2,1),tV(2,2)],[0.8,0,0.2],'edgecolor','none');
    p1.FaceVertexAlphaData = [0.5;0;0];
    p1.FaceAlpha = 'interp';
    set(gca,'Ydir','reverse')
    plotArrow([45,290],2.5,23.2,0,'w')
    plotArrow([46.25,291.25],2.5,23.2,-90,'w')
    hold off
    title('Setup','Color','w','fontsize',24)
    h5(3) = gca;

    set(h5(1),'Position',[0.13,0.52,0.4,0.4])
    set(h5(2),'Position',[0.13,0.02,0.4,0.4])
    set(h5(3),'Position',[0.59,0.05,0.4,0.85])

    text(h5(3),0.9,0.04,'x','FontSize',16,'Units','normalized','fontweight','bold','color','w')
    text(h5(3),0.77,0.09,'z','FontSize',16,'Units','normalized','fontweight','bold','color','w')

    drawnow

    if i>3
        frame = getframe(gcf);
        writeVideo(v,frame)
    end
end
close(v)

%% Figure 6
folder = [pwd,'\BendSeries'];

res = 60;
numTM = 5;
numPix = sum(makeImagingMask(res),'all');
frameRate = 2e4/(numPix+numTM*49);

load([folder,'\Videos\autoFlythrough.mat'],'FP')
A = rot90(readLabviewData([folder,'\Videos\Flythrough.dat'],2,'double'));

CB = mean(imread([folder,'\chessBoard.jpg']),3)/255;
CB = CB(170:570,35:435)';
cXc = 10;
cYc = -40;
cS = 80;
xC = cS*linspace(-1,1,401)+cXc;
yC = cS*linspace(-1,1,401)+cYc;

NA = 0.275;
tLength = 250;
tVertices = [tLength*tan(asin(NA))*[1,-1];-tLength*[1,1]];
Q = @(x) [sin(x),cos(x);cos(x),-sin(x)];

vChess = VideoReader([folder,'\Videos\ChessFlythroughGT.mp4']);
vStartT = 1.35;
vLeft = 70;
vTop = 45;
vWidth = 430;
mask = makeImagingMask(vWidth+1);

wMask = makeImagingMask(res+2);

figure(6); clf
set(gcf,'Color','w','units','normalized','position',[0.01,0.1,0.70,0.68])
h6 = repmat(gca,3,5);
j = 1;
for i = [10,30,60,120,324]
    % Plot ground truth
    vChess.currentTime = vStartT+(i-1)/frameRate;
    V = flipud(readFrame(vChess));
    V = mean(V(vTop:vTop+vWidth,vLeft:vLeft+vWidth,:),3)/255;
    subplot(3,5,j);
    imagesc(V,'AlphaData',mask); axis image; axis off; colormap gray
    title([num2str(round(i/frameRate,1),'%4.1f'),' s'],'Fontsize',22)
    h6(1,j) = gca;

    % Plot endoscope image
    subplot(3,5,5+j);
    temp = wMask+0;
    temp(2:res+1,2:res+1) = mat2gray(A(:,:,i));
    imagesc(temp,'AlphaData',wMask); axis image; axis off; colormap gray
    h6(2,j) = gca;

    % Find vertices of triangle and needle
    x = FP{i}(1,:)/2; % X direction is twice Y from stepper motor definition for video
    y = FP{i}(2,:);
    if(and(i>45,i<70)); k = 15; x(end-k:end) = x(end-k:end)+(0:k)*1.5/k; end
    nNear = [x(end-2);y(end-2)];
    g = [x(end)-x(end-10);y(end)-y(end-10)]; g = g/norm(g);
    nFar = nNear+20*g;
    tV = nFar-Q(angle(g(1)-1i*g(2)))*tVertices;

    % Plot fibre and chess board position
    subplot(3,5,10+j)
    imagesc(xC,yC,CB); colormap gray;
    hold on
    plot([nNear(1),nFar(1)],[nNear(2),nFar(2)],'color',0.4*[1,1,1],'linewidth',2)
    plot(x,y,'color',hsv2rgb([0.08,0.8,1]),'linewidth',2.5); axis equal; axis([cXc-cS,cXc+cS,cYc-cS,300]); axis off
    p1 = patch([nFar(1),tV(1,1),tV(1,2)],[nFar(2),tV(2,1),tV(2,2)],[0.8,0,0.2],'edgecolor','none');
    p1.FaceVertexAlphaData = [0.5;0;0];
    p1.FaceAlpha = 'interp';
    set(gca,'Ydir','reverse')
    hold off
    h6(3,j) = gca;

    j = j+1;
end

hold(h6(3,5),'on')
plotArrow([30,280],4,37.1,0,[0,0,0],h6(3,5))
plotArrow([32,282],4,37.1,-90,[0,0,0],h6(3,5))
hold(h6(3,5),'off')

step = 0.18;
for j = 0:4
    set(h6(1,j+1),'position',[0.08+j*step,0.71,0.23,0.23])
    set(h6(2,j+1),'position',[0.075+j*step,0.52,0.24,0.24])
    set(h6(3,j+1),'position',[0.116+j*step,0.00,0.16,0.52])
end
text(h6(1,1),-0.75,0.55,{'Ground','Truth'},'Units','Normalized','Fontsize',22,'FontWeight','bold')
text(h6(2,1),-0.75,0.55,{'Fibre','Image'},'Units','Normalized','Fontsize',22,'FontWeight','bold')
text(h6(2,1),-0.75,-0.4,{'Setup','Position'},'Units','Normalized','Fontsize',22,'FontWeight','bold')
text(h6(3,5),0.82,0.08,'x','FontSize',15,'Units','normalized','fontweight','bold','color','k')
text(h6(3,5),0.67,0.14,'z','FontSize',15,'Units','normalized','fontweight','bold','color','k')

%% Figure 7
folder = [pwd,'\BendSeries'];

res = 60;
numTM = 5;
numPix = sum(makeImagingMask(res),'all');
frameRate = 2e4/(numPix+numTM*49);

oC = imread([folder,'\Videos\OfficeCookie.png']);
oC = oC(230:480,550:1200,:);
oP = imread([folder,'\Videos\OfficePepper.png']);
oP = oP(230:480,550:1200,:);
oO = imread([folder,'\Videos\OfficeObjects.png']);
oO = oO(230:480,540:1190,:);

aChannel = [repmat(linspace(0,1,651),45,1);ones(206,651)];

fC = readLabviewData([folder,'\Videos\OfficeCookie.dat'],2,'double');
fP = readLabviewData([folder,'\Videos\OfficePepper.dat'],2,'double');
fO = readLabviewData([folder,'\Videos\OfficeObjects.dat'],2,'double');

save([folder,'\Videos\OfficeVideos.mat'],'fC','fP','fO')

inds = [22,69,104,149,273,301;
    31,144,187,239,320,606;
    43,168,237,311,335,490];

numFrame = size(inds,2);
res = size(fC,1);
wMask = makeImagingMask(res+2);

figure(7); clf
h7 = repmat(gca,3,numFrame+1);
set(gcf,'color','w','units','normalized','position',[0.01,0.1,0.88,0.66])

subplot(3,numFrame+1,1)
imagesc(oC,'AlphaData',aChannel); axis image; axis off
h7(1,1) = gca;
subplot(3,numFrame+1,numFrame+2)
imagesc(oP,'AlphaData',aChannel); axis image; axis off
h7(2,1) = gca;
subplot(3,numFrame+1,2*numFrame+3)
imagesc(oO,'AlphaData',aChannel); axis image; axis off
h7(3,1) = gca;

for i = 1:numFrame
    if i==1
        sl = 1;
        sr = 4;
    elseif i==numFrame
        sl = 4;
        sr = 1;
    else
        sl = 4;
        sr = 4;
    end
    subplot(3,numFrame+1,i+1);
    imagesc(fC(sl:res-sr+1,:,inds(1,i))','AlphaData',~isnan(fC(sl:res-sr+1,:,inds(1,i))'));
    title([num2str(round(inds(1,i)/frameRate,1),'%4.1f') ' s'],'Fontsize',19,'fontweight','bold')
    axis image; axis off; colormap gray
    h7(1,i+1) = gca;

    subplot(3,numFrame+1,i+numFrame+2);
    imagesc(fP(sl:res-sr+1,:,inds(2,i))','AlphaData',~isnan(fP(sl:res-sr+1,:,inds(2,i))'));
    title([num2str(round(inds(2,i)/frameRate,1),'%4.1f') ' s'],'Fontsize',19,'fontweight','bold')
    axis image; axis off; colormap gray
    h7(2,i+1) = gca;

    subplot(3,numFrame+1,i+2*numFrame+3);
    imagesc(fO(sl:res-sr+1,:,inds(3,i))','AlphaData',~isnan(fO(sl:res-sr+1,:,inds(3,i))'));
    title([num2str(round(inds(3,i)/frameRate,1),'%4.1f') ' s'],'Fontsize',19,'fontweight','bold')
    axis image; axis off; colormap gray
    h7(3,i+1) = gca;
end

scale = 60/55;

set(h7(1,1),'position',[-0.01,0.69,scale*0.35,scale*0.28])
set(h7(2,1),'position',[-0.01,0.35,scale*0.35,scale*0.28])
set(h7(3,1),'position',[-0.01,0.01,scale*0.35,scale*0.28])

for i = 0:numFrame-1
    if i<2
        start = 0.27;
        step = 0.1085;
    elseif i==numFrame-1
        start = 0.2755;
        step = 0.1059;
    else
        start = 0.2725;
        step = 0.106;
    end
    set(h7(1,i+2),'position',[start+i*step,0.685,scale*0.25,scale*0.25])
    set(h7(2,i+2),'position',[start+i*step,0.345,scale*0.25,scale*0.25])
    set(h7(3,i+2),'position',[start+i*step,0.005,scale*0.25,scale*0.25])
end

text(h7(1,1),0.01,0.91,'Fortune Cookie','Units','Normalized','Fontsize',22,'FontWeight','bold')
text(h7(2,1),0.01,0.91,'Bell Pepper','Units','Normalized','Fontsize',22,'FontWeight','bold')
text(h7(3,1),0.01,0.91,'Common Targets','Units','Normalized','Fontsize',22,'FontWeight','bold')

%% Figure S3a
% ------------------------ Alignment Parameters ---------------------------
trueTop = 60;
trueLeft = 38;
trueSize = 370;
trueAngle = 2;
% -------------------------------------------------------------------------

folder = [pwd,'\BendSeries\'];

% Get the true image for comparison
trueImage = double(imread([folder,'magpie.png']));
fibImage = readLabviewData([folder,'Images\TM0_S0.dat'],2,'double');
fibImage = mean(permute(fibImage,[2,1,3]),3,'omitnan');
fibImage(1) = NaN;

trueImage = imrotate(trueImage,trueAngle);
trueImage = trueImage(trueTop:(trueSize+trueTop-1),trueLeft:(trueSize+trueLeft-1));
trueImage = imresize(trueImage,60/trueSize);

mask = ones(60);
mask(isnan(fibImage)) = nan;

trueImage = mat2gray(mask.*trueImage);
fibImage = mat2gray(fibImage);

% figure(102);
% pic = cat(3,trueImage,fibImage,zeros(60));
% imagesc(pic); axis off; axis square; colormap('gray')

ssims = zeros(13,4);
% figure(103)
% set(gcf,'Position',[1,41,1920,963]);
for i = 0:3
    for j = 0:25:300
        temp = readLabviewData(...
            [folder,'Images\TM',num2str(i),'_S',num2str(j),'.dat'],2,'double');
        temp = mean(permute(temp,[2,1,3]),3,'omitnan');
        temp(1) = NaN;
        ssims(j/25+1,i+1) = ssim(mat2gray(temp),trueImage);

%         subplot(4,13,13*i+j/25+1)
%         imagesc(temp); axis off; axis square; colormap('gray')
%         title(num2str(ssims(j/25+1,i+1)))
    end
end

% Determine fibre shapes
FS = cell(2,1);
fibreLength = 250;
numPoints = 100;
rc = 2*fibreLength/numPoints;
for i = 1:2
    D = rgb2hsv(imread([folder,'fib',num2str((i-1)*300),'.jpg']));
    D = D(363:722,380:1579,:);
    mask = and(and(D(:,:,1)<0.09,D(:,:,1)>0.05),and(D(:,:,2)>0.35,D(:,:,3)>0.3));
    img = imresize(mask+0,1/6,'bilinear');

    [x,y] = meshgrid(1:200,1:60);
    F = nan*zeros(2,1.2*numPoints);
    F(:,1) = [200;2];
    grad = [-1;0];
    stepSize = rc;
    j = 1;
    while stepSize>rc/10
        imgCopy = img;
        [~,r] = cart2pol(x-F(1,j)-rc*grad(1)/2,y-F(2,j)-rc*grad(2)/2);
        imgCopy(r>rc) = 0;
        F(:,j+1) = [sum(imgCopy(:).*x(:))/sum(imgCopy(:));sum(imgCopy(:).*y(:))/sum(imgCopy(:))];
        stepSize = norm(F(:,j+1)-F(:,j));
        grad = (F(:,j+1)-F(:,j))/stepSize;
        j = j+1;

%         figure(104)
%         subplot(2,1,1)
%         imagesc(cat(3,img,imgCopy,0*img))
%         hold on
%         plot(F(1,:),F(2,:),'bo')
%         hold off
%         title([num2str(j),'   ',num2str(stepSize)])
%         subplot(2,1,2)
%         quiver(0,0,grad(1),-grad(2)); axis([-1,1,-1,1]); axis square
%         title([num2str(grad(1)),',',num2str(-grad(2))])
    end
    FS{i} = reshape(F(~isnan(F)),2,[]);
end

% Plot SSIM
figure(8)
palette = hsv2rgb([0.08,0.4,1.0; ...
    0.08,0.9,1.0; ...
    0.08,1.0,0.6; ...
    0.08,1.0,0.3]);
clf; set(gcf,'Position',[20,450,600,470])
set(gcf,'color','w')
colororder(palette);
h8(1) = subplot(2,2,1:2);
plot(0:12.5:150,ssims,'-x','LineWidth',2);
box off
subplot(2,2,3);
h8(2) = plotFibre(FS,1);
subplot(2,2,4);
h8(3) = plotFibre(FS,2);
title(h8(1),'Imaging Fidelity Over a Large Displacement','Fontsize',18);
legend(h8(1),{'TM 0','TM 50','TM 100','TM 150'},'box','off')
xlabel(h8(1),'Fibre Tip Displacement (mm)')
ylabel(h8(1),'SSIM with Ground Truth')
set(h8(1),'fontsize',13)
set(h8(1),'Position',[0.13,0.13,0.8,0.8])
set(h8(2),'Position',[0.05,-0.09,0.2,0.2])
set(h8(3),'Position',[0.8,-0.08,0.2,0.2])

%% Figure S3b
folder = [pwd,'\BendImaging'];
load([folder,'\Fibre\fibreShapes.mat'])

n = 101;
imSiz = 120;
extent = 4200;
folder = [pwd,'\SpeckleData\'];

inds = linspace(0,extent,n);
S = zeros(imSiz^2,n);
for i = 1:n
    temp = readLabviewData([folder,'Speckle',num2str(inds(i)),'.dat'],2,'single');
    S(:,i) = temp(:)/norm(temp(:));
end
C = S'*S;

figure(9);clf
palette = hsv2rgb([0.08*ones(256,1),[ones(128,1);linspace(1,0,128)'],[linspace(0,1,128)';ones(128,1)]]);
clf; set(gcf,'Position',[225,50,580,470],'color','w')
h9(1) = subplot(1,5,1);
imagesc(C); axis image;
colormap(palette)
title({'Output Speckle Intensitiy Cross-Correlation',''})
hold on
i = 1;
threshold = 0.92;
a = ones(n,1);
iOld = 1;
j = ones(n,1);
count = 1;
while j(count)<n
    while a(count) > threshold
        if i>=n; break; end
        i = i+1;
        a(count) = C(i,j(count));
    end
    if i>=n; break; end
    plot([j(count),j(count)],[iOld,i],'color',[0.2,0.5,0.9],'Linewidth',2);
    iOld = i;
    plot([j(count),i],[i,i],'color',[0.2,0.5,0.9],'Linewidth',2);
    count = count+1;
    j(count) = i;
    a(count) = C(i,j(count));
end
hold off
hc = colorbar('Ticks',[0.7,0.8,0.9,1],'TickLabels',{'0.7','0.8','0.9','1.0'});
hc.Label.String = 'Correlation Magnitude';
hc.FontSize = 13;
set(gca,'Xtick',[1,101],'Ytick',[1,101],'Ztick',[0.7,0.8,0.9,1],'fontsize',12)
xlabel('Fibre Position')
ylabel('Fibre Position')
subplot(1,5,2)
h9(2) = plotFibre(FS,1);
subplot(1,5,3)
h9(3) = plotFibre(FS,30);
subplot(1,5,4)
h9(4) = plotFibre(FS,1);
subplot(1,5,5)
h9(5) = plotFibre(FS,30);
set(h9(1),'Position',[0.05,0.11,0.82,0.78])
set(h9(2),'Position',[0.01,0.81,0.1,0.1])
set(h9(3),'Position',[-0.02,0.07,0.1,0.1])
set(h9(4),'Position',[0.1,-0.04,0.1,0.1])
set(h9(5),'Position',[0.69,-0.02,0.1,0.1])

%% Figure S1
x = linspace(0,pi/2);
y2 = (-4/pi^2)*x.^2+(4/pi)*x;
y3 = (-4/pi^3)*x.^3+(3/pi)*x;
yt = (-1/6)*x.^3+x;
ys = sin(x);

figure(10); clf
set(gcf,'Position',[90,340,780,360],'color','w')
plot(x,y2,'b',x,y3,'r',x,yt,x,ys,'--k','linewidth',2); box off;
axis([0,pi/2,0,1])
legend('2^{nd} Order Polynomial','3^{rd} Order Polynomial','3^{rd} Order Taylor','sin(x)','location','best','box','off')
title('Sine Approximations')
xlabel('x'); ylabel('f(x)')
set(gca,'xtick',[0,pi/2],'xticklabel',{'0','\pi/2'},'ytick',[0,1],'fontsize',16,'fontname','Times')

%% Video 3
folder = [pwd,'\BendSeries'];
load([folder,'\Videos\OfficeVideos.mat'],'fC')
fC = permute(fC,[2,1,3]);

res = 60;
numTM = 5;
numPix = sum(makeImagingMask(res),'all');
frameRate = 1/(44e-6*(numPix+numTM*49));
multiplier = 3;
buff = 2;

vC = VideoReader([folder,'\Videos\OfficeFullCookie.mp4']);
vCStart = 3.95365-1/frameRate;
vCLeft = 60;
vCWidth = 1200;
vCHeight = 320;
vCTArray = 150*ones(multiplier*(size(fC,3)+2*buff),1);

vCOut = VideoWriter([folder,'\Videos\OfficeCookie']);

makeOfficeVideo(vC,'k',vCWidth,vCHeight,vCStart,vCTArray,vCLeft,frameRate,fC,buff,multiplier,11,vCOut,'Fortune Cookie',0)

%% Video 4
folder = [pwd,'\BendSeries'];
load([folder,'\Videos\OfficeVideos.mat'],'fP')
fP = permute(fP,[2,1,3]);
fP = fP(:,:,1:610);

res = 60;
numTM = 5;
numPix = sum(makeImagingMask(res),'all');
frameRate = 1/(44e-6*(numPix+numTM*49));
multiplier = 3;
buff = 0;

vP = VideoReader([folder,'\Videos\OfficeFullPepper.mp4']);
vPStart = 3.95365-3/frameRate;
vPLeft = 60;
vPWidth = 1200;
vPHeight = 320;
p = [45,80,300,330];
tMin = 160;
tMax = 360;
vPTArray = round([tMin*ones(1,p(1)),...
    linspace(tMin,tMax,p(2)-p(1)),...
    tMax*ones(1,p(3)-p(2)),...
    linspace(tMax,tMin,p(4)-p(3)),...
    tMin*ones(1,multiplier*(size(fP,3)+2*buff)-p(4))]);

vPOut = VideoWriter([folder,'\Videos\OfficePepper']);

makeOfficeVideo(vP,'k',vPWidth,vPHeight,vPStart,vPTArray,vPLeft,frameRate,fP,buff,multiplier,12,vPOut,'Bell Pepper',0)

%% Video 5
folder = [pwd,'\BendSeries'];
load([folder,'\Videos\OfficeVideos.mat'],'fO')
fO = permute(fO,[2,1,3]);

res = 60;
numTM = 5;
numPix = sum(makeImagingMask(res),'all');
frameRate = 1/(44e-6*(numPix+numTM*49));
multiplier = 3;
buff = 2;

vO = VideoReader([folder,'\Videos\OfficeFullObjects.mp4']);
vOStart = 3.95365+7/frameRate;
vOLeft = 60;
vOWidth = 1200;
vOHeight = 320;
vOTArray = 160*ones(multiplier*(size(fO,3)+2*buff),1);

vOOut = VideoWriter([folder,'\Videos\OfficeObjects']);

makeOfficeVideo(vO,'k',vOWidth,vOHeight,vOStart,vOTArray,vOLeft,frameRate,fO,buff,multiplier,13,vOOut,'Imaging Targets',0)

%% Find fibre shapes
fibreShapesFromVideo;
function result = fibreShapesFromVideo(dataIn)
if(nargin>0)
    vFibre = dataIn.vFibre;
    startTFibre = dataIn.startTFibre;
    leftFibre = dataIn.leftFibre;
    rightFibre = dataIn.rightFibre;
    topFibre = dataIn.topFibre;
    bottomFibre = dataIn.bottomFibre;
    frameRate = dataIn.frameRate;
    fibreLength = dataIn.fibreLength;
    numPoints = dataIn.numPoints;
    numFrame = dataIn.numFrame;
    figureNumber = dataIn.figureNumber;

    count = 0;
    FP = cell(numFrame,1);
    rc = 2*fibreLength/numPoints;
    [x,y] = meshgrid(1:40,1:300);
    figure(figureNumber); clf;
    set(gcf,'Color','w','units','normalized','Position',[0,0.04,0.5,0.88])
    for i = 1:numFrame
        if (vFibre.Duration>(startTFibre+(i-1)/frameRate))
            vFibre.currentTime = startTFibre+(i-1)/frameRate;
            VF = readFrame(vFibre);
            VF = VF(topFibre:bottomFibre,leftFibre:rightFibre,:);
            VF = rot90(VF);
            img = rgb2hsv(VF);
            img = and(and(and(img(:,:,1)>0.0,img(:,:,1)<0.1),img(:,:,2)>0.3),img(:,:,3)>0.25);
            for j = 1:10
                filtMask = imfilter(img+0,[1,1,1;1,0,1;1,1,1]);
                img = and(img,filtMask>4);
            end
            img = imresize(img+0,0.25);
            figure(figureNumber)
            subplot(1,4,1);
            imshow(VF); axis image; axis off;
            title(['Frame ',num2str(i)])
            subplot(1,4,2);
            imshow(img); axis image; axis off;
            title({'Colour Keyed Mask','After Erode and Build'})
            F = nan*zeros(2,1.2*numPoints);
            F(:,1) = [15;298];
            grad = [0;-1];
            stepSize = rc;
            j = 1;
            while stepSize>rc/10
                imgCopy = img;
                [~,r] = cart2pol(x-F(1,j)-rc*grad(1)/2,y-F(2,j)-rc*grad(2)/2);
                imgCopy(r>rc) = 0;
                F(:,j+1) = [sum(imgCopy(:).*x(:))/sum(imgCopy(:));sum(imgCopy(:).*y(:))/sum(imgCopy(:))];
                stepSize = norm(F(:,j+1)-F(:,j));
                grad = (F(:,j+1)-F(:,j))/stepSize;
                j = j+1;
                figure(figureNumber)
                subplot(1,4,3)
                imagesc(cat(3,img,imgCopy,0*img)); axis equal; axis off
                hold on
                plot(F(1,:),F(2,:),'bo')
                hold off
                title({['Point ',num2str(j)],['Step Size ',num2str(stepSize)]})
                subplot(1,4,4)
                quiver(0,0,grad(1),-grad(2)); axis([-1,1,-1,1]); axis square
                title({'Heading (x,z)',[num2str(round(grad(1),3)),',',num2str(round(-grad(2),3))]})%,'fontsize',16,'color','w')
            end
            FP{i} = reshape(F(~isnan(F)),2,[]);
        end
        count = count + ~isempty(FP{i});
    end
    result = FP(1:count);
end
end

%% Plot an arrow
function plotArrow(s,w,l,rot,col,h)
if nargin<6
    h = gca;
    if nargin<5
        col = [0,0,0];
        if nargin<4
            rot = 0;
        end
    end
end
t = deg2rad(rot);
Q = [cos(t),-sin(t);sin(t),cos(t)];
x = [0,l-3*w,l-3*w,l,l-3*w,l-3*w,0,0];
y = [-w/2,-w/2,-w,0,w,w/2,w/2,-w/2];
v = Q*[x;y];
patch(h,v(1,:)+s(1),v(2,:)+s(2),col,'edgecolor','none');
end

%% Make office video
function makeOfficeVideo(vIn,figCol,vW,vH,vS,vTArray,vL,frameRate,F,buff,multiplier,figNum,vOut,vTitle,preview)

switch figCol
    case 'k'
        cFig = 'k';
        cText = 'w';
    case'w'
        cFig = 'w';
        cText = 'k';
    otherwise
        error('Pick b(black) or w(white) for figure colour')
end

tHeight = 50;
aChannel = [repmat(linspace(0,1,vW+1),tHeight,1);ones(vH-tHeight+1,vW+1)];

if ~preview
    vOut.FrameRate = multiplier*frameRate;
    open(vOut)
end

figure(figNum)
set(gcf,'units','normalized','position',[0.01,0.2,0.75,0.3],'color',cFig)
for i = multiplier*(1-buff):multiplier*(size(F,3)+buff)
    vT = vTArray(i+(buff-1)*multiplier+1);

    vIn.currentTime = vS+(i-1)/(frameRate*multiplier);
    V = readFrame(vIn);
    V = V(vT:vT+vH,vL:vL+vW,:);

    figure(figNum)
    subplot(2,1,1)
    imagesc(V,'AlphaData',aChannel); axis image; axis off;
    h(1) = gca;

    j = floor(i/3);
    if j<1
        k=1;
    elseif j>size(F,3)
        k=size(F,3);
    else
        k=j;
    end
    subplot(2,1,2)
    imagesc(F(:,:,k),'AlphaData',~isnan(F(:,:,k)));
    axis image; axis off; colormap gray
    h(2) = gca;

    set(h(1),'Position',[0.01,0.025,0.75,0.95])
    set(h(2),'Position',[0.42,0.040,0.92,0.92])
    text(h(1),0.01,0.93,vTitle,'Units','Normalized','Fontsize',24,'FontWeight','bold','color',cText)

    if ~preview
        frame = getframe(gcf);
        writeVideo(vOut,frame)
    end
end
if ~preview
    close(vOut)
end
end

%% Square grid in circle
function result = makeGridInCircle(siz,nPoints)
nPoints = nPoints-1;
maxEdge = floor((siz-1)/sqrt(2));
pixStep = floor(maxEdge/nPoints);
pixEdge = pixStep*nPoints;
lb = floor((siz-pixEdge)/2)+1;
ub = lb+pixEdge;
[x,y] = meshgrid(lb:pixStep:ub);
result = sub2ind([siz,siz],x,y);
result = result(:);
end

%% Plot fibre profile
function result = plotFibre(FS,n,t)
if nargin<3; t = 2; end
x = FS{n}(1,:);
y = FS{n}(2,:);
nX = [x(end)-35,x(end)];
nY = [y(end),y(end)];
plot(nX,nY,'color',0.4*[1,1,1],'linewidth',t);
hold on
plot(x,y,'color',hsv2rgb([0.06,1,1]),'linewidth',t+0.5);
hold off
axis equal; axis([-25,265,1,100]); axis off
set(gca,'ydir','reverse')
result = gca;
end

%% Make imaging mask
function result = makeImagingMask(n)
edge = (n-1)/2;
[x,y] = meshgrid(linspace(-edge,edge,n));
result = abs(x+1i*y)<(n/2);
end

%% General read data from LabView .dat file
function result = readLabviewData(filePath,numDim,precision,complex)
if nargin<4;complex = false;end

% Get file size
s = dir(filePath);
fsize = s.bytes;

% Get array size
fid = fopen(filePath);
arraySize = fread(fid,numDim,'int32','ieee-be');

% Set read loaction back to 0
fseek(fid,-ftell(fid),0);

% Convert precision to number of bytes (This is trash in Matlab)
switch precision
    case {'logical','bool','char','int8','uint8'}
        nBytes = 1;
    case {'int16','uint16'}
        nBytes = 2;
    case {'int','int32','uint32','float','single'}
        nBytes = 4;
    case {'int64','uint64','double'}
        nBytes = 8;
    otherwise
        error(['Unrecognised data type: ',precision])
end

% Get nuber of frames
nFrames = fsize/(nBytes*prod(arraySize)+4*numDim/(1+complex))/(1+complex);

% Initialise frame array
A = zeros(prod(arraySize),nFrames);

% Loop through frames
for i = 1:nFrames
    % Skip size definition (Only works for frames of the same size)
    fseek(fid,4*numDim,0);

    % Read frame
    frame = fread(fid,prod(arraySize)*(1+complex),precision,'ieee-be');

    % Convert to complex number if complex
    if complex; frame = frame(1:2:end)+1i*frame(2:2:end); end

    % Write frame to array
    A(:,i) = frame;
end

fclose(fid);

% reshape into (n+1)D array for Matlab
result = reshape(A,[flip(arraySize'),nFrames]);
end