# -*- coding: utf-8 -*-

'''
This module provides an access to the HITRAN data.
Data is downloaded and cached.

This module serves as a simple database manager front end.

API is aimed to be RESTful, which means that interaction
between local API and remote data-server will be held 
via sending RESTful queries (API->remote) and
receiving data preferably in text format (remote->API).

Object are supposed to be implemented by structures/dicts
as they are present in almost any programming language.

Trying to retain functional style for this API. 
'''

import sys
import json
import os, os.path
import re
from os import listdir
import numpy as np
from numpy import zeros,array,setdiff1d,ndarray,arange
from numpy import place,where,real,polyval
from numpy import complex128,int64,float64,float32
from numpy import sqrt,abs,exp,pi,log,sin,cos,tan
from numpy import convolve
from numpy import flipud
from numpy.fft import fft,fftshift
from numpy import linspace,floor
from numpy import any,minimum,maximum
from numpy import sort as npsort
from bisect import bisect
from warnings import warn,simplefilter
from time import time
import pydoc

# Enable warning repetitions
simplefilter('always', UserWarning)

# Python 3 compatibility
try:
    import urllib.request as urllib2
except ImportError:
    import urllib2
if 'io' in sys.modules: # define open using Linux-style line endings
    import io
    def open_(*args,**argv):
        argv.update(dict(newline='\n'))
        return io.open(*args,**argv)
else:
    open_ = open

HAPI_VERSION = '1.2.2.4'; __version__ = HAPI_VERSION
HAPI_HISTORY = [
'FIXED GRID BUG (ver. 1.1.0.1)',
'FIXED OUTPUT FORMAT FOR CROSS-SECTIONS (ver. 1.1.0.1)',
'ADDED CPF BY SCHREIER (JQSRT_112_2011) (ver. 1.1.0.2)',
'OPTIMIZED EXPRESSION EVALUATIONS FOR SELECT (ver. 1.1.0.3)',
'ADDED SUPPORT FOR MIXTURES (ver. 1.1.0.4)',
'ADDED SUPPORT FOR USER-DEFINED ENV DEPENDENCES (ver. 1.1.0.5)',
'ADDED PROFILE SELECTION (ALPHA) (ver. 1.1.0.6)',
'ADDED METADATA FOR HTP, FIXED NORMALIZATION IN CONVOLVESPECTRUMSAME (ver. 1.1.0.7)',
'FIXED A "LONELY HEADER" BUG IN CACHE2STORAGE (ver. 1.1.0.7.1)',
'ADDED SUPPORT FOR PHOSGENE AND CYANOGEN (ver. 1.1.0.7.2)',
'OPTIMIZED STORAGE2CACHE (by Nils-Holger Loeber) (ver. 1.1.0.7.3)',
'ADDED SKIPABLE PARAMETERS IN HEADERS (ver. 1.1.0.7.4)',
'ADDED SUPPORT FOR FORTRAN D-NOTATION (ver. 1.1.0.7.5)',
'ADDED SUPPORT FOR WEIRD-FORMATTED INTENSITY VALUES E.G. "2.700-164" (ver. 1.1.0.7.6)',
'ADDED TIPS-2017 (ver. 1.1.0.8)',
'ADDED SUPPORT FOR CUSTOM EXTENSIONS OF THE DATA FILES (ver. 1.1.0.8.1)',
'FIXED LINK TO (2,0) ISOTOPOLOGUE IN TIPS-2017 (ver. 1.1.0.8.2)',
'ADDED SAVEHEADER FUNCTION (ver. 1.1.0.8.3)',
'ADDED METADATA FOR SF6 (ver. 1.1.0.8.4)',
'ADDED D2O ISOTOPOLOGUE OF WATER TO DESCRIPTION (ver. 1.1.0.8.5)',
'FIXED LINE ENDINGS IN STORAGE2CACHE AND QUERYHITRAN (ver. 1.1.0.8.6)',
'ADDED SUPPORT FOR NON-INTEGER LOCAL ISO IDS (ver. 1.1.0.8.7)',
'FIXED PARAMETER NAME CASE BUG (by Robert J. Hargreaves) (ver. 1.1.0.8.8)',
'CAST LOCAL_ISO_ID=0 TO 10 FOR CARBON DIOXIDE (ver. 1.1.0.8.9)',
'USING NUMPY.ARRAYS FOR NUMERIC COLUMNS OF LOCAL_TABLE_CACHE (ver. 1.1.0.9.0)',
'ADDED DESCRIPTIONS FOR BROADENING BY H2O (ver. 1.1.0.9.1)',
'ADDED PROXY SUPPORT IN FETCH AND FETCH_BY_IDS (ver. 1.1.0.9.2)',
'ADDED LIMIT FOR NUMBER OF LINES DURING TABLE READ (ver. 1.1.0.9.3)',
'FIXED ABSOLUTE PATH BUG IN TABLE NAMES (ver. 1.1.0.9.4)',
'CORRECTED ABUNDANCE OF THE HD ISOTOPOLOGUE (ver. 1.1.0.9.5)',
'ADDED UNIFIED INTERFACES FOR ABSCOEF AND XSC CALCULATIONS (ver. 1.1.0.9.6)',
'ADDED PARLISTS FOR LINE MIXING (VOIGT AND SDVOIGT) (ver. 1.1.0.9.7)',
'ADDED SUPPORT FOR ROSENKRANZ LM PARAMETERS TO PCQSDHC AND LORENTZ (ver. 1.1.1.0)',
'FIXED THE TYPEERROR IN ARANGE (ver. 1.1.2.0)',
'ADDED NEW FUNCTIONAL INTERFACES FOR ALL CROSS-SECTION CALCULATING ROUTINES (ver. 1.2.0.0)',
'ADDED CALCULATION OF THE ISO_ID TABLE ON STARTUP (ver. 1.2.1.0)',
'ADDED SUPPORT FOR TIPS-2021 (ver. 1.2.2.0)',
'FIXED BUG WITH WAVENUMBERGRID (ver. 1.2.2.1)',
'FIXED BUG WITH ZEROING OUT LINES WITH NON-STANDARD PARAMETERS (ver. 1.2.2.2)',
'ADDED SUPPORT FOR PARAMETERS ABSENT IN PARAMETER_META (ver. 1.2.2.3)',
'ADDED SUPPORT FOR 403 ERROR MESSAGE (QUERY LIMIT EXCEEDED)  (ver. 1.2.2.4)',
]

# version header
print('HAPI version: %s' % HAPI_VERSION)
print('To get the most up-to-date version please check http://hitran.org/hapi')
print('ATTENTION: Python versions of partition sums from TIPS-2021 are now available in HAPI code')
#print('ATTENTION: Python versions of partition sums from TIPS-2017 are available at http://hitran.org/suppl/TIPS/')
#print('           To use them in HAPI ver. 1.1.0.7, use partitionFunction parameter of the absorptionCoefficient_ routine.')
print('')
print('           MIT license: Copyright 2021 HITRAN team, see more at http://hitran.org. ')
print('')
print('           If you use HAPI in your research or software development,')
print('           please cite it using the following reference:')
print('           R.V. Kochanov, I.E. Gordon, L.S. Rothman, P. Wcislo, C. Hill, J.S. Wilzewski,')
print('           HITRAN Application Programming Interface (HAPI): A comprehensive approach')
print('           to working with spectroscopic data, J. Quant. Spectrosc. Radiat. Transfer 177, 15-30 (2016)')
print('           DOI: 10.1016/j.jqsrt.2016.03.005')
print('')
print('           ATTENTION: This is the core version of the HITRAN Application Programming Interface.')
print('                      For more efficient implementation of the absorption coefficient routine, ')
print('                      as well as for new profiles, parameters and other functional,')
print('                      please consider using HAPI2 extension library.')
print('                      HAPI2 package is available at http://github.com/hitranonline/hapi2')
print('')


# define precision
__ComplexType__ = complex128
__IntegerType__ = int64
__FloatType__ = float64

# define zero
cZero = __FloatType__(0.)

# physical constants
cBolts = 1.380648813E-16 # erg/K, CGS
cc = 2.99792458e10 # cm/s, CGS
hh = 6.626196e-27 # erg*s, CGS

# computational constants
cSqrtLn2divSqrtPi = 0.469718639319144059835
cLn2 = 0.6931471805599
cSqrtLn2 = 0.8325546111577
cSqrt2Ln2 = 1.1774100225

# initialize global variables
VARIABLES = {}

VARIABLES['DEBUG'] = False
if VARIABLES['DEBUG']: warn('DEBUG is set to True!')

GLOBAL_DEBUG = False
if GLOBAL_DEBUG: warn('GLOBAL_DEBUG is set to True!')

FLAG_DEBUG_PROFILE = False
FLAG_DEBUG_LADDER = False

LOCAL_HOST = 'http://localhost'

# DEBUG switch
if GLOBAL_DEBUG:
   GLOBAL_HOST = LOCAL_HOST+':8000' # localhost
else:
   GLOBAL_HOST = 'http://hitran.org'

# default PARAMETER_META for custom user parameters
PMETA_DEFAULT = {
    "default_fmt" : "%s",
}

VARIABLES['PROXY'] = {}
# EXAMPLE OF PROXY:
# VARIABLES['PROXY'] = {'http': '127.0.0.1:80'}
   
# make it changeable
VARIABLES['GLOBAL_HOST'] = GLOBAL_HOST

# display the fetch URL (debug)
VARIABLES['DISPLAY_FETCH_URL'] = False

# In this "robust" version of arange the grid doesn't suffer 
# from the shift of the nodes due to error accumulation.
# This effect is pronounced only if the step is sufficiently small.
def arange_(lower,upper,step):
    npnt = floor((upper-lower)/step)+1
    npnt = int(npnt) # cast to integer to avoid type errors
    upper_new = lower + step*(npnt-1)
    if abs((upper-upper_new)-step) < 1e-10:
        upper_new += step
        npnt += 1    
    return linspace(lower,upper_new,npnt)

# ---------------------------------------------------------------
# ---------------------------------------------------------------
# LOCAL DATABASE MANAGEMENT SYSTEM
# ---------------------------------------------------------------
# ---------------------------------------------------------------

# ---------------------------------------------------------------
# DATABASE BACKEND: simple text files, parsed into a python lists
# Use a directory as a database. Each table is stored in a 
# separate text file. Parameters in text are position-fixed.

BACKEND_DATABASE_NAME_DEFAULT = '.'

VARIABLES['BACKEND_DATABASE_NAME'] = BACKEND_DATABASE_NAME_DEFAULT

# For this node local DB is schema-dependent!
LOCAL_TABLE_CACHE = {
   'sampletab' : { # table
      'header' : { # header
         'order' : ('column1','column2','column3'),
         'format' : {
            'column1' : '%10d',
            'column2' : '%20f',
            'column3' : '%30s' 
         },
         'default' : {
            'column1' : 0,
            'column2' : 0.0,
            'column3' : ''
         },
         'number_of_rows' : 3,
         'size_in_bytes' : None,
         'table_name' : 'sampletab',
         'table_type' : 'strict'
      }, # /header
      'data' : {
         'column1' : [1,2,3],
         'column2' : [10.5,11.5,12.5],
         'column3' : ['one','two','three']
      }, # /data
   } # /table
} # hash-map of tables

# FORMAT CONVERSION LAYER

# converts between TRANSPORT_FORMAT and OBJECT_FORMAT
HITRAN_FORMAT_160 = {
   'M'          : {'pos' :   1,   'len' :  2,   'format' : '%2d' },
   'I'          : {'pos' :   3,   'len' :  1,   'format' : '%1d' },
   'nu'         : {'pos' :   4,   'len' : 12,   'format' : '%12f'},
   'S'          : {'pos' :  16,   'len' : 10,   'format' : '%10f'},
   'R'          : {'pos' :  26,   'len' :  0,   'format' : '%0f' },
   'A'          : {'pos' :  26,   'len' : 10,   'format' : '%10f'},
   'gamma_air'  : {'pos' :  36,   'len' :  5,   'format' : '%5f' },
   'gamma_self' : {'pos' :  41,   'len' :  5,   'format' : '%5f' },
   'E_'         : {'pos' :  46,   'len' : 10,   'format' : '%10f'},
   'n_air'      : {'pos' :  56,   'len' :  4,   'format' : '%4f' },
   'delta_air'  : {'pos' :  60,   'len' :  8,   'format' : '%8f' },
   'V'          : {'pos' :  68,   'len' : 15,   'format' : '%15s'},
   'V_'         : {'pos' :  83,   'len' : 15,   'format' : '%15s'},
   'Q'          : {'pos' :  98,   'len' : 15,   'format' : '%15s'},
   'Q_'         : {'pos' : 113,   'len' : 15,   'format' : '%15s'},
   'Ierr'       : {'pos' : 128,   'len' :  6,   'format' : '%6s' },
   'Iref'       : {'pos' : 134,   'len' : 12,   'format' : '%12s'},
   'flag'       : {'pos' : 146,   'len' :  1,   'format' : '%1s' },
   'g'          : {'pos' : 147,   'len' :  7,   'format' : '%7f' },
   'g_'         : {'pos' : 154,   'len' :  7,   'format' : '%7f' }
}

# This should be generating from the server's response
HITRAN_DEFAULT_HEADER = {
  "table_type": "column-fixed", 
  "size_in_bytes": -1, 
  "table_name": "###", 
  "number_of_rows": -1, 
  "order": [
    "molec_id", 
    "local_iso_id", 
    "nu", 
    "sw", 
    "a", 
    "gamma_air", 
    "gamma_self", 
    "elower", 
    "n_air", 
    "delta_air", 
    "global_upper_quanta", 
    "global_lower_quanta", 
    "local_upper_quanta", 
    "local_lower_quanta", 
    "ierr", 
    "iref", 
    "line_mixing_flag", 
    "gp", 
    "gpp"
  ],
  "format": {
    "a": "%10.3E", 
    "gamma_air": "%5.4f", 
    "gp": "%7.1f", 
    "local_iso_id": "%1d", 
    "molec_id": "%2d", 
    "sw": "%10.3E", 
    "local_lower_quanta": "%15s", 
    "local_upper_quanta": "%15s", 
    "gpp": "%7.1f", 
    "elower": "%10.4f", 
    "n_air": "%4.2f", 
    "delta_air": "%8.6f", 
    "global_upper_quanta": "%15s", 
    "iref": "%12s", 
    "line_mixing_flag": "%1s", 
    "ierr": "%6s", 
    "nu": "%12.6f", 
    "gamma_self": "%5.3f", 
    "global_lower_quanta": "%15s"
  }, 
  "default": {
    "a": 0.0, 
    "gamma_air": 0.0, 
    "gp": "FFF", 
    "local_iso_id": 0, 
    "molec_id": 0, 
    "sw": 0.0, 
    "local_lower_quanta": "000", 
    "local_upper_quanta": "000", 
    "gpp": "FFF", 
    "elower": 0.0, 
    "n_air": 0.0, 
    "delta_air": 0.0, 
    "global_upper_quanta": "000", 
    "iref": "EEE", 
    "line_mixing_flag": "EEE", 
    "ierr": "EEE", 
    "nu": 0.0, 
    "gamma_self": 0.0, 
    "global_lower_quanta": "000"
  },
  "description": {
    "a": "Einstein A-coefficient in s-1", 
    "gamma_air": "Air-broadened Lorentzian half-width at half-maximum at p = 1 atm and T = 296 K", 
    "gp": "Upper state degeneracy", 
    "local_iso_id": "Integer ID of a particular Isotopologue, unique only to a given molecule, in order or abundance (1 = most abundant)", 
    "molec_id": "The HITRAN integer ID for this molecule in all its isotopologue forms", 
    "sw": "Line intensity, multiplied by isotopologue abundance, at T = 296 K", 
    "local_lower_quanta": "Rotational, hyperfine and other quantum numbers and labels for the lower state of a transition", 
    "local_upper_quanta": "Rotational, hyperfine and other quantum numbers and labels for the upper state of a transition", 
    "gpp": "Lower state degeneracy", 
    "elower": "Lower-state energy", 
    "n_air": "Temperature exponent for the air-broadened HWHM", 
    "delta_air": "Pressure shift induced by air, referred to p=1 atm", 
    "global_upper_quanta": "Electronic and vibrational quantum numbers and labels for the upper state of a transition", 
    "iref": "Ordered list of reference identifiers for transition parameters", 
    "line_mixing_flag": "A flag indicating the presence of additional data and code relating to line-mixing", 
    "ierr": "Ordered list of indices corresponding to uncertainty estimates of transition parameters", 
    "nu": "Transition wavenumber", 
    "gamma_self": "Self-broadened HWHM at 1 atm pressure and 296 K", 
    "global_lower_quanta": "Electronic and vibrational quantum numbers and labels for the lower state of a transition"
  },
  "position": {
    "molec_id": 0,
    "local_iso_id": 2,
    "nu": 3,
    "sw": 15,
    "a": 25,
    "gamma_air": 35,
    "gamma_self": 40,
    "elower": 45,
    "n_air": 55,
    "delta_air": 59,
    "global_upper_quanta": 67,
    "global_lower_quanta": 82,
    "local_upper_quanta": 97,
    "local_lower_quanta": 112,
    "ierr": 127,
    "iref": 133,
    "line_mixing_flag": 145,
    "gp": 146,
    "gpp": 153,
  },
  'cast': {
    "molec_id": "uint8",
    "local_iso_id": "uint8",
    "nu": "float32",
    "sw": "float62",
    "a": "float62",
    "gamma_air": "float16",
    "gamma_self": "float16",
    "elower": "float32",
    "n_air": "float16",
    "delta_air": "float16",
    "global_upper_quanta": "str",
    "global_lower_quanta": "str",
    "local_upper_quanta": "str",
    "local_upper_quanta": "str",
    "ierr": "str",
    "iref": "str",
    "line_mixing_flag": "str",
    "gp": "int16",
    "gpp": "int16",  
  }
}

#class CaselessDict(dict):                   
#    def __getitem__(self,key):
#        return super(CaselessDict,self).__getitem__(key.lower())
#    def __setitem__(self,key,val):
#        super(CaselessDict,self).__setitem__(key.lower(),val)
#    def __contains__(self,key):
#        return super(CaselessDict,self).__contains__(key.lower())
#    def has_key(self,key):
#        return super(CaselessDict,self).has_key(key.lower())
#    def get(self,key,default=None):
#        return super(CaselessDict,self).get(key.lower(),default)
#    def __init__(self,dct=None):
#        if dct is not None:
#            for key in dct:
#                super(CaselessDict,self).__setitem__(key.lower(),dct[key])

class CaseInsensitiveDict(dict):
    """
    https://gist.github.com/m000/acbb31b9eca92c1da795 (c) Manolis Stamatogiannakis.
    """
    @classmethod
    def _k(cls, key):
        return key.lower() if isinstance(key, str) else key

    def __init__(self, *args, **kwargs):
        super(CaseInsensitiveDict, self).__init__(*args, **kwargs)
        self._convert_keys()
    def __getitem__(self, key):
        return super(CaseInsensitiveDict, self).__getitem__(self.__class__._k(key))
    def __setitem__(self, key, value):
        super(CaseInsensitiveDict, self).__setitem__(self.__class__._k(key), value)
    def __delitem__(self, key):
        return super(CaseInsensitiveDict, self).__delitem__(self.__class__._k(key))
    def __contains__(self, key):
        return super(CaseInsensitiveDict, self).__contains__(self.__class__._k(key))
    def has_key(self, key):
        return super(CaseInsensitiveDict, self).has_key(self.__class__._k(key))
    def pop(self, key, *args, **kwargs):
        return super(CaseInsensitiveDict, self).pop(self.__class__._k(key), *args, **kwargs)
    def get(self, key, *args, **kwargs):
        return super(CaseInsensitiveDict, self).get(self.__class__._k(key), *args, **kwargs)
    def setdefault(self, key, *args, **kwargs):
        return super(CaseInsensitiveDict, self).setdefault(self.__class__._k(key), *args, **kwargs)
    def update(self, E, **F):
        super(CaseInsensitiveDict, self).update(self.__class__(E))
        super(CaseInsensitiveDict, self).update(self.__class__(**F))
    def _convert_keys(self):
        for k in list(self.keys()):
            v = super(CaseInsensitiveDict, self).pop(k)
            self.__setitem__(k, v)

CaselessDict = CaseInsensitiveDict
            
PARAMETER_META = CaselessDict(
{
  "global_iso_id" : {
    "default_fmt" : "%5d",
  },
  "molec_id" : {
    "default_fmt" : "%2d",
  },
  "local_iso_id" : {
    "default_fmt" : "%1d",
  },
  "nu" : {
    "default_fmt" : "%12.6f",
  },
  "sw" : {
    "default_fmt" : "%10.3e",
  },
  "a" : {
    "default_fmt" : "%10.3e",
  },
  "gamma_air" : {
    "default_fmt" : "%6.4f",
  },
  "gamma_self" : {
    "default_fmt" : "%5.3f",
  },
  "n_air" : {
    "default_fmt" : "%7.4f",
  },
  "delta_air" : {
    "default_fmt" : "%9.6f",
  },
  "elower" : {
    "default_fmt" : "%10.4f",
  },
  "gp" : {
    "default_fmt" : "%5d",
  },
  "gpp" : {
    "default_fmt" : "%5d",
  },
  "global_upper_quanta" : {
    "default_fmt" : "%15s",
  },
  "global_lower_quanta" : {
    "default_fmt" : "%15s",
  },
  "local_upper_quanta" : {
    "default_fmt" : "%15s",
  },
  "local_lower_quanta" : {
    "default_fmt" : "%15s",
  },
  "line_mixing_flag" : {
    "default_fmt" : "%1s",
  },
  "ierr" : {
    "default_fmt" : "%s",
  },
  "iref" : {
    "default_fmt" : "%s",
  },
  "deltap_air" : {
    "default_fmt" : "%10.3e",
  },
  "n_self" : {
    "default_fmt" : "%7.4f",
  },
  "delta_self" : {
    "default_fmt" : "%9.6f",
  },
  "deltap_self" : {
    "default_fmt" : "%10.3e",
  },
  "SD_air" : {
    "default_fmt" : "%9.6f",
  },
  "SD_self" : {
    "default_fmt" : "%9.6f",
  },
  "beta_g_air" : {
    "default_fmt" : "%9.6f",
  },
  "y_self" : {
    "default_fmt" : "%10.3e",
  },
  "y_air" : {
    "default_fmt" : "%10.3e",
  },
  "statep" : {
    "default_fmt" : "%256s",
  },
  "statepp" : {
    "default_fmt" : "%256s",
  },
  "beta_g_self" : {
    "default_fmt" : "%9.6f",
  },
  "trans_id" : {
    "default_fmt" : "%12d",
  },
  "par_line" : {
    "default_fmt" : "%160s",
  },
  "gamma_H2" : {
    "default_fmt" : "%6.4f",
  },
  "n_H2" : {
    "default_fmt" : "%7.4f",
  },
  "delta_H2" : {
    "default_fmt" : "%9.6f",
  },
  "deltap_H2" : {
    "default_fmt" : "%10.3e",
  },
  "gamma_He": {
    "default_fmt" : "%6.4f",
  },
  "n_He" : {
    "default_fmt" : "%7.4f",
  },
  "delta_He" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_CO2" : {
    "default_fmt" : "%6.4f",
  },
  "n_CO2" : {
    "default_fmt" : "%7.4f",
  },
  "delta_CO2" : {
    "default_fmt" : "%9.6f",
  }, 
  "gamma_HT_0_self_50" : {
    "default_fmt" : "%6.4f",
  },
  "n_HT_self_50" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_HT_2_self_50" : {
    "default_fmt" : "%6.4f",
  },
  "delta_HT_0_self_50" : {
    "default_fmt" : "%9.6f",
  },
  "deltap_HT_self_50" : {
    "default_fmt" : "%9.6f",
  },
  "delta_HT_2_self_50" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_HT_0_self_150" : {
    "default_fmt" : "%6.4f",
  },
  "n_HT_self_150" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_HT_2_self_150" : {
    "default_fmt" : "%6.4f",
  },
  "delta_HT_0_self_150" : {
    "default_fmt" : "%9.6f",
  },
  "deltap_HT_self_150" : {
    "default_fmt" : "%9.6f",
  },
  "delta_HT_2_self_150" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_HT_0_self_296" : {
    "default_fmt" : "%6.4f",
  },
  "n_HT_self_296" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_HT_2_self_296" : {
    "default_fmt" : "%6.4f",
  },
  "delta_HT_0_self_296" : {
    "default_fmt" : "%9.6f",
  },
  "deltap_HT_self_296" : {
    "default_fmt" : "%9.6f",
  },
  "delta_HT_2_self_296" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_HT_0_self_700" : {
    "default_fmt" : "%6.4f",
  },
  "n_HT_self_700" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_HT_2_self_700" : {
    "default_fmt" : "%6.4f",
  },
  "delta_HT_0_self_700" : {
    "default_fmt" : "%9.6f",
  },
  "deltap_HT_self_700" : {
    "default_fmt" : "%9.6f",
  },
  "delta_HT_2_self_700" : {
    "default_fmt" : "%9.6f",
  },
  "nu_HT_self" : {
    "default_fmt" : "%6.4f",
  },
  "kappa_HT_self" : {
    "default_fmt" : "%9.6f",
  },
  "eta_HT_self" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_HT_0_air_50" : {
    "default_fmt" : "%6.4f",
  },
  "n_HT_air_50" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_HT_2_air_50" : {
    "default_fmt" : "%6.4f",
  },
  "delta_HT_0_air_50" : {
    "default_fmt" : "%9.6f",
  },
  "deltap_HT_air_50" : {
    "default_fmt" : "%9.6f",
  },
  "delta_HT_2_air_50" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_HT_0_air_150" : {
    "default_fmt" : "%6.4f",
  },
  "n_HT_air_150" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_HT_2_air_150" : {
    "default_fmt" : "%6.4f",
  },
  "delta_HT_0_air_150" : {
    "default_fmt" : "%9.6f",
  },
  "deltap_HT_air_150" : {
    "default_fmt" : "%9.6f",
  },
  "delta_HT_2_air_150" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_HT_0_air_296" : {
    "default_fmt" : "%6.4f",
  },
  "n_HT_air_296" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_HT_2_air_296" : {
    "default_fmt" : "%6.4f",
  },
  "delta_HT_0_air_296" : {
    "default_fmt" : "%9.6f",
  },
  "deltap_HT_air_296" : {
    "default_fmt" : "%9.6f",
  },
  "delta_HT_2_air_296" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_HT_0_air_700" : {
    "default_fmt" : "%6.4f",
  },
  "n_HT_air_700" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_HT_2_air_700" : {
    "default_fmt" : "%6.4f",
  },
  "delta_HT_0_air_700" : {
    "default_fmt" : "%9.6f",
  },
  "deltap_HT_air_700" : {
    "default_fmt" : "%9.6f",
  },
  "delta_HT_2_air_700" : {
    "default_fmt" : "%9.6f",
  },
  "nu_HT_air" : {
    "default_fmt" : "%6.4f",
  },
  "kappa_HT_air" : {
    "default_fmt" : "%9.6f",
  },
  "eta_HT_air" : {
    "default_fmt" : "%9.6f",
  },  
  "gamma_H2O" : {
    "default_fmt" : "%6.4f",
  },
  "n_H2O" : {
    "default_fmt" : "%9.6f",
  },
  "Y_SDV_air_296" : {
    "default_fmt" : "%10.3e",
  },
  "Y_SDV_self_296" : {
    "default_fmt" : "%10.3e",
  },
  "n_Y_SDV_air_296" : {
    "default_fmt" : "%6.4e",
  },
  "n_Y_SDV_self_296" : {
    "default_fmt" : "%6.4e",
  },
  "Y_HT_air_296" : {
    "default_fmt" : "%10.3e",
  },
  "Y_HT_self_296" : {
    "default_fmt" : "%10.3e",
  },
  "gamma_SDV_0_air_296" : {
    "default_fmt" : "%6.4f",
  },
  "gamma_SDV_0_self_296" : {
    "default_fmt" : "%6.4f",
  },
  "n_SDV_air_296" : {
    "default_fmt" : "%9.6f",
  },
  "n_SDV_self_296" : {
    "default_fmt" : "%9.6f",
  },
  "gamma_SDV_2_air_296" : {
    "default_fmt" : "%6.4f",
  },
  "gamma_SDV_2_self_296" : {
    "default_fmt" : "%6.4f",
  },
  "n_gamma_SDV_2_air_296" : {
    "default_fmt" : "%6.4f",
  },
  "n_gamma_SDV_2_self_296" : {
    "default_fmt" : "%6.4f",
  },
  "delta_SDV_0_air_296" : {
    "default_fmt" : "%9.6f",
  },
  "delta_SDV_0_self_296" : {
    "default_fmt" : "%9.6f",
  },
  "deltap_SDV_air_296" : {
    "default_fmt" : "%9.6f",
  },
  "deltap_SDV_self_296" : {
    "default_fmt" : "%9.6f",
  },
})

def getFullTableAndHeaderName(TableName,ext=None):
    #print('TableName=',TableName)
    if ext is None: ext = 'data'
    flag_abspath = False # check if the supplied table name already contains absolute path
    if os.path.isabs(TableName): flag_abspath = True        
    fullpath_data = TableName + '.' + ext
    if not flag_abspath: fullpath_data = os.path.join(VARIABLES['BACKEND_DATABASE_NAME'],fullpath_data)
    if not os.path.isfile(fullpath_data):
        fullpath_data = VARIABLES['BACKEND_DATABASE_NAME'] + '/' + TableName + '.par'
        if not os.path.isfile(fullpath_data) and TableName!='sampletab':
            raise Exception('Lonely header \"%s\"' % fullpath_data)
    fullpath_header = TableName + '.header'
    if not flag_abspath: fullpath_header = os.path.join(VARIABLES['BACKEND_DATABASE_NAME'],fullpath_header)
    return fullpath_data,fullpath_header

def getParameterFormat(ParameterName,TableName):
    return LOCAL_TABLE_CACHE[TableName]['header']['format']

def getTableHeader(TableName):
    return LOCAL_TABLE_CACHE[TableName]['header']

def getRowObject(RowID,TableName):
    # return RowObject from TableObject in CACHE
    RowObject = []
    for par_name in LOCAL_TABLE_CACHE[TableName]['header']['order']:
        par_value = LOCAL_TABLE_CACHE[TableName]['data'][par_name][RowID]
        par_format = LOCAL_TABLE_CACHE[TableName]['header']['format'][par_name]
        RowObject.append((par_name,par_value,par_format))
    return RowObject

# INCREASE ROW COUNT
def addRowObject(RowObject,TableName):
    #print 'addRowObject: '
    #print 'RowObject: '+str(RowObject)
    #print 'TableName:'+TableName
    for par_name,par_value,par_format in RowObject:
        #print 'par_name,par_value,par_format: '+str((par_name,par_value,par_format))
        #print '>>> '+ str(LOCAL_TABLE_CACHE[TableName]['data'][par_name])
        #LOCAL_TABLE_CACHE[TableName]['data'][par_name] += [par_value]
        LOCAL_TABLE_CACHE[TableName]['data'][par_name].append(par_value)

def setRowObject(RowID,RowObject,TableName):
    number_of_rows = LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows']
    if RowID >= 0 and RowID < number_of_rows:
       for par_name,par_value,par_format in RowObject:
           LOCAL_TABLE_CACHE[TableName]['data'][par_name][RowID] = par_value
    else:
       # !!! XXX ATTENTION: THIS IS A TEMPORARY INSERTION XXX !!!
       LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows'] += 1
       addRowObject(RowObject,TableName)

def getDefaultRowObject(TableName):
    # get a default RowObject from a table
    RowObject = []
    for par_name in LOCAL_TABLE_CACHE[TableName]['header']['order']:
        par_value = LOCAL_TABLE_CACHE[TableName]['header']['default'][par_name]
        par_format = LOCAL_TABLE_CACHE[TableName]['header']['format'][par_name]
        RowObject.append((par_name,par_value,par_format))
    return RowObject

def subsetOfRowObject(ParameterNames,RowObject):
    # return a subset of RowObject according to 
    #RowObjectNew = []
    #for par_name,par_value,par_format in RowObject:
    #     if par_name in ParameterNames:
    #        RowObjectNew.append((par_name,par_value,par_format))
    #return RowObjectNew
    dct = {}
    for par_name,par_value,par_format in RowObject:
        dct[par_name] = (par_name,par_value,par_format)
    RowObjectNew = []
    for par_name in ParameterNames:
        RowObjectNew.append(dct[par_name])
    return RowObjectNew

#FORMAT_PYTHON_REGEX = '^\%([0-9]*)\.?([0-9]*)([dfs])$'
FORMAT_PYTHON_REGEX = '^\%(\d*)(\.(\d*))?([edfsEDFS])$'

# Fortran string formatting
#  based on a pythonic format string
def formatString(par_format,par_value,lang='FORTRAN'):
    # Fortran format rules:
    #  %M.NP
    #        M - total field length (optional)
    #             (minus sign included in M)
    #        . - decimal ceparator (optional)
    #        N - number of digits after . (optional)
    #        P - [dfs] int/float/string
    # PYTHON RULE: if N is abcent, default value is 6
    regex = FORMAT_PYTHON_REGEX
    (lng,trail,lngpnt,ty) = re.search(regex,par_format).groups()
    if type(par_value) is np.ma.core.MaskedConstant:
        result = '%%%ss' % lng % '#'
        return result
    result = par_format % par_value
    if ty.lower() in set(['f','e']):
       lng = int(lng) if lng else 0
       lngpnt = int(lngpnt) if lngpnt else 0
       result = par_format % par_value
       res = result.strip()
       if lng==lngpnt+1:
          if res[0:1]=='0':
             result =  '%%%ds' % lng % res[1:]
       if par_value<0:
          if res[1:2]=='0':
             result = '%%%ds' % lng % (res[0:1]+res[2:])
    return result

def putRowObjectToString(RowObject):
    # serialize RowObject to string
    # TODO: support different languages (C,Fortran)
    output_string = ''
    for par_name,par_value,par_format in RowObject:
        # Python formatting
        #output_string += par_format % par_value
        # Fortran formatting
        #print 'par_name,par_value,par_format: '+str((par_name,par_value,par_format))
        output_string += formatString(par_format,par_value)
    return output_string

# Parameter nicknames are hard-coded.
PARAMETER_NICKNAMES = {
    "a": "A", 
    "gamma_air": "gair", 
    "gp": "g", 
    "local_iso_id": "I", 
    "molec_id": "M", 
    "sw": "S", 
    "local_lower_quanta": "Q_", 
    "local_upper_quanta": "Q", 
    "gpp": "g_", 
    "elower": "E_", 
    "n_air": "nair", 
    "delta_air": "dair", 
    "global_upper_quanta": "V", 
    "iref": "Iref", 
    "line_mixing_flag": "f", 
    "ierr": "ierr", 
    "nu": "nu", 
    "gamma_self": "gsel", 
    "global_lower_quanta": "V_"
}  

def putTableHeaderToString(TableName):
    output_string = ''
    regex = FORMAT_PYTHON_REGEX
    for par_name in LOCAL_TABLE_CACHE[TableName]['header']['order']:
        par_format = LOCAL_TABLE_CACHE[TableName]['header']['format'][par_name]
        (lng,trail,lngpnt,ty) = re.search(regex,par_format).groups()
        fmt = '%%%ss' % lng
        try:
            par_name_short = PARAMETER_NICKNAMES[par_name]
        except:
            par_name_short = par_name
        #output_string += fmt % par_name
        output_string += (fmt % par_name_short)[:int(lng)]
    return output_string

def getRowObjectFromString(input_string,TableName):
    # restore RowObject from string, get formats and names in TableName
    #print 'getRowObjectFromString:'
    pos = 0
    RowObject = []
    for par_name in LOCAL_TABLE_CACHE[TableName]['header']['order']:
        par_format = LOCAL_TABLE_CACHE[TableName]['header']['format'][par_name]
        regex = '^\%([0-9]+)\.?[0-9]*([dfs])$' #
        regex = FORMAT_PYTHON_REGEX
        (lng,trail,lngpnt,ty) = re.search(regex,par_format).groups()
        lng = int(lng)
        par_value = input_string[pos:(pos+lng)]
        if ty=='d': # integer value
           par_value = int(par_value)
        elif ty.lower() in set(['e','f']): # float value
           par_value = float(par_value)
        elif ty=='s': # string value
           pass # don't strip string value
        else:
           print('err1')
           raise Exception('Format \"%s\" is unknown' % par_format)
        RowObject.append((par_name,par_value,par_format))
        pos += lng
    # Do the same but now for extra (comma-separated) parameters
    if 'extra' in set(LOCAL_TABLE_CACHE[TableName]['header']):
        csv_chunks = input_string.split(LOCAL_TABLE_CACHE[TableName]['header'].\
                                        get('extra_separator',','))
        # Disregard the first "column-fixed" container if it presents:
        if LOCAL_TABLE_CACHE[TableName]['header'].get('order',[]):
            pos = 1
        else:
            pos = 0
        for par_name in LOCAL_TABLE_CACHE[TableName]['header']['extra']:
            par_format = LOCAL_TABLE_CACHE[TableName]['header']['extra_format'][par_name]
            #regex = '^\%([0-9]+)\.?[0-9]*([dfs])$' #
            regex = FORMAT_PYTHON_REGEX
            (lng,trail,lngpnt,ty) = re.search(regex,par_format).groups()
            lng = int(lng) 
            par_value = csv_chunks[pos]
            if ty=='d': # integer value
                try:
                    par_value = int(par_value)
                except ValueError:
                    #par_value = 0
                    par_value = np.nan
            elif ty.lower() in set(['e','f']): # float value
                try:
                    par_value = float(par_value)
                except ValueError:
                    #par_value = 0.0
                    par_value = np.nan
            elif ty=='s': # string value
                pass # don't strip string value
            else:
                print('err')
                raise Exception('Format \"%s\" is unknown' % par_format)
            RowObject.append((par_name,par_value,par_format))
            pos += 1   
    return RowObject

# Conversion between OBJECT_FORMAT and STORAGE_FORMAT
# This will substitute putTableToStorage and getTableFromStorage
def cache2storage(TableName):
    try:
       os.mkdir(VARIABLES['BACKEND_DATABASE_NAME'])
    except:
       pass
    #fullpath_data,fullpath_header = getFullTableAndHeaderName(TableName) # "lonely header" bug
    fullpath_data = VARIABLES['BACKEND_DATABASE_NAME'] + '/' + TableName + '.data' # bugfix
    fullpath_header = VARIABLES['BACKEND_DATABASE_NAME'] + '/' + TableName + '.header' # bugfix
    OutfileData = open(fullpath_data,'w')
    OutfileHeader = open(fullpath_header,'w')
    # write table data
    line_count = 1
    line_number = LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows']
    for RowID in range(0,line_number):
        line_count += 1
        RowObject = getRowObject(RowID,TableName)
        raw_string = putRowObjectToString(RowObject)
        OutfileData.write(raw_string+'\n')
    # write table header
    TableHeader = getTableHeader(TableName)
    OutfileHeader.write(json.dumps(TableHeader,indent=2))
    
def storage2cache(TableName,cast=True,ext=None,nlines=None,pos=None):
    """ edited by NHL
    TableName: name of the HAPI table to read in
    ext: file extension
    nlines: number of line in the block; if None, read all line at once 
    pos: file position to seek
    """
    #print 'storage2cache:'
    #print('TableName',TableName)
    if nlines is not None:
        print('WARNING: storage2cache is reading the block of maximum %d lines'%nlines)
    fullpath_data,fullpath_header = getFullTableAndHeaderName(TableName,ext)
    if TableName in LOCAL_TABLE_CACHE and \
       'filehandler' in LOCAL_TABLE_CACHE[TableName] and \
       LOCAL_TABLE_CACHE[TableName]['filehandler'] is not None:
        InfileData = LOCAL_TABLE_CACHE[TableName]['filehandler']
    else:
        InfileData = open_(fullpath_data,'r')            
    InfileHeader = open(fullpath_header,'r')
    #try:
    header_text = InfileHeader.read()
    try:
        Header = json.loads(header_text)
    except:
        print('HEADER:')
        print(header_text)
        raise Exception('Invalid header')
    #print 'Header:'+str(Header)
    LOCAL_TABLE_CACHE[TableName] = {}
    LOCAL_TABLE_CACHE[TableName]['header'] = Header
    LOCAL_TABLE_CACHE[TableName]['data'] = CaselessDict()
    LOCAL_TABLE_CACHE[TableName]['filehandler'] = InfileData
    # Check if Header['order'] and Header['extra'] contain
    #  parameters with same names, raise exception if true.
    #intersct = set(Header['order']).intersection(set(Header.get('extra',[])))
    intersct = set(Header.get('order',[])).intersection(set(Header.get('extra',[])))
    if intersct:
        raise Exception('Parameters with the same names: {}'.format(intersct))
    # initialize empty data to avoid problems
    glob_order = []; glob_format = {}; glob_default = {}
    if "order" in LOCAL_TABLE_CACHE[TableName]['header'].keys():
        glob_order += LOCAL_TABLE_CACHE[TableName]['header']['order']
        glob_format.update(LOCAL_TABLE_CACHE[TableName]['header']['format'])
        glob_default.update(LOCAL_TABLE_CACHE[TableName]['header']['default'])
        for par_name in LOCAL_TABLE_CACHE[TableName]['header']['order']:
            LOCAL_TABLE_CACHE[TableName]['data'][par_name] = []
    if "extra" in LOCAL_TABLE_CACHE[TableName]['header'].keys():
        glob_order += LOCAL_TABLE_CACHE[TableName]['header']['extra']
        glob_format.update(LOCAL_TABLE_CACHE[TableName]['header']['extra_format'])
        for par_name in LOCAL_TABLE_CACHE[TableName]['header']['extra']:
            glob_default[par_name] = PARAMETER_META.get(par_name,PMETA_DEFAULT)['default_fmt']
            LOCAL_TABLE_CACHE[TableName]['data'][par_name] = []
    
    header = LOCAL_TABLE_CACHE[TableName]['header']
    if 'extra' in header and header['extra']:
        line_count = 0
        flag_EOF = False
        #line_number = LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows']
        #for line in InfileData:
        while True:
            #print '%d line from %d' % (line_count,line_number)
            #print 'line: '+line #
            if nlines is not None and line_count>=nlines: break
            line = InfileData.readline()
            if line=='': # end of file is represented by an empty string
                flag_EOF = True
                break 
            try:
                RowObject = getRowObjectFromString(line,TableName)
                line_count += 1
            except:
                continue
            #print 'RowObject: '+str(RowObject)
            addRowObject(RowObject,TableName)
        #except:
        #    raise Exception('TABLE FETCHING ERROR')
        LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows'] = line_count
    else:
        quantities = header['order']
        formats = [header['format'][qnt].lower() for qnt in quantities]
        types = {'d':int, 'f':float, 'e':float, 's':str}
        converters = []
        end = 0
        for qnt, fmt in zip(quantities, formats):
            # pre-defined positions are needed to skip the existing parameters in headers (new feature)
            if 'position' in header:
                start = header['position'][qnt]
            else:
                start = end
            dtype = types[fmt[-1]]
            aux = fmt[fmt.index('%')+1:-1]
            if '.' in aux:
                aux = aux[:aux.index('.')]
            size = int(aux)
            end = start + size
            def cfunc(line, dtype=dtype, start=start, end=end, qnt=qnt):
                # return dtype(line[start:end]) # this will fail on the float number with D exponent (Fortran notation)
                if dtype in (float,int): # assign NaN if value is hashtagged
                    if line[start:end].strip()=='#':
                        return np.nan
                if dtype==float:
                    try:
                        return dtype(line[start:end])
                    except ValueError: # possible D exponent instead of E 
                        try:
                            return dtype(line[start:end].replace('D','E'))
                        except ValueError: # this is a special case and it should not be in the main version tree!
                            # Dealing with the weird and unparsable intensity format such as "2.700-164, i.e with no E or D characters.
                            res = re.search('(\d\.\d\d\d)\-(\d\d\d)',line[start:end])
                            if res:
                                return dtype(res.group(1)+'E-'+res.group(2))
                            else:
                                raise Exception('PARSE ERROR: unknown format of the par value (%s)'%line[start:end])
                elif dtype==int and qnt=='local_iso_id':
                    if line[start:end]=='0': return 10
                    try:
                        return dtype(line[start:end])
                    except ValueError:
                        # convert letters to numbers: A->11, B->12, etc... ; .par file must be in ASCII or Unicode.
                        return 11+ord(line[start:end])-ord('A')
                else:
                    return dtype(line[start:end])
            #cfunc.__doc__ = 'converter {} {}'.format(qnt, fmt) # doesn't work in earlier versions of Python
            converters.append(cfunc)
            #start = end
        #data_matrix = [[cvt(line) for cvt in converters] for line in InfileData]
        flag_EOF = False
        line_count = 0
        data_matrix = []
        while True:
            if nlines is not None and line_count>=nlines: break   
            line = InfileData.readline()
            if line=='': # end of file is represented by an empty string
                flag_EOF = True
                break 
            data_matrix.append([cvt(line) for cvt in converters])
            line_count += 1
        data_columns = zip(*data_matrix)
        for qnt, col in zip(quantities, data_columns):
            #LOCAL_TABLE_CACHE[TableName]['data'][qnt].extend(col) # old code
            if type(col[0]) in {int,float}:
                LOCAL_TABLE_CACHE[TableName]['data'][qnt] = np.array(col) # new code
            else:
                LOCAL_TABLE_CACHE[TableName]['data'][qnt].extend(col) # old code
            #LOCAL_TABLE_CACHE[TableName]['data'][qnt] = list(col)
            #LOCAL_TABLE_CACHE[TableName]['data'][qnt] = col
        header['number_of_rows'] = line_count = (
            len(LOCAL_TABLE_CACHE[TableName]['data'][quantities[0]]))
                        
    # Convert all columns to numpy arrays
    par_names = LOCAL_TABLE_CACHE[TableName]['header']['order'].copy()
    if 'extra' in header and header['extra']:
        par_names += LOCAL_TABLE_CACHE[TableName]['header']['extra']
    for par_name in par_names:
        column = LOCAL_TABLE_CACHE[TableName]['data'][par_name]
        LOCAL_TABLE_CACHE[TableName]['data'][par_name] = np.array(column)                    
            
    # Additionally: convert numeric arrays of the LOCAL_TABLE_CACHE to masked arrays.
    # This is done to avoid "nan" values in the arithmetic ope  rations involving these columns.
    for par_name in LOCAL_TABLE_CACHE[TableName]['header']['order']:
        par_format = LOCAL_TABLE_CACHE[TableName]['header']['format'][par_name]
        regex = FORMAT_PYTHON_REGEX
        (lng,trail,lngpnt,ty) = re.search(regex,par_format).groups()
        if ty.lower() in ['d','e','f']:
            column = LOCAL_TABLE_CACHE[TableName]['data'][par_name]
            colmask = np.isnan(column)
            LOCAL_TABLE_CACHE[TableName]['data'][par_name] = np.ma.array(column,mask=colmask)
    
    if 'extra' in header and header['extra']:
        for par_name in LOCAL_TABLE_CACHE[TableName]['header']['extra']:
            par_format = LOCAL_TABLE_CACHE[TableName]['header']['extra_format'][par_name]
            regex = FORMAT_PYTHON_REGEX
            (lng,trail,lngpnt,ty) = re.search(regex,par_format).groups()
            if ty.lower() in ['d','e','f']:
                column = LOCAL_TABLE_CACHE[TableName]['data'][par_name]
                colmask = np.isnan(column)
                LOCAL_TABLE_CACHE[TableName]['data'][par_name] = np.ma.array(column,mask=colmask)
    
    # Delete all character-separated values, treat them as column-fixed.
    try:
        del LOCAL_TABLE_CACHE[TableName]['header']['extra']
        del LOCAL_TABLE_CACHE[TableName]['header']['extra_format']
        del LOCAL_TABLE_CACHE[TableName]['header']['extra_separator']
    except:
        pass
    # Update header.order/format with header.extra/format if exist.
    LOCAL_TABLE_CACHE[TableName]['header']['order'] = glob_order
    LOCAL_TABLE_CACHE[TableName]['header']['format'] = glob_format
    LOCAL_TABLE_CACHE[TableName]['header']['default'] = glob_default
    if flag_EOF:
        InfileData.close()
        LOCAL_TABLE_CACHE[TableName]['filehandler'] = None
    InfileHeader.close()
    print('                     Lines parsed: %d' % line_count)
    return flag_EOF    
    
## old version based on regular expressions    
#def storage2cache(TableName):
#    fullpath_data,fullpath_header = getFullTableAndHeaderName(TableName)
#    InfileData = open(fullpath_data,'r')
#    InfileHeader = open(fullpath_header,'r')
#    #try:
#    header_text = InfileHeader.read()
#    try:
#        Header = json.loads(header_text)
#    except:
#        print('HEADER:')
#        print(header_text)
#        raise Exception('Invalid header')
#    LOCAL_TABLE_CACHE[TableName] = {}
#    LOCAL_TABLE_CACHE[TableName]['header'] = Header
#    LOCAL_TABLE_CACHE[TableName]['data'] = {}
#    # Check if Header['order'] and Header['extra'] contain
#    #  parameters with same names, raise exception if true.
#    intersct = set(Header.get('order',[])).intersection(set(Header.get('extra',[])))
#    if intersct:
#        raise Exception('Parameters with the same names: {}'.format(intersct))
#    # initialize empty data to avoid problems
#    glob_order = []; glob_format = {}; glob_default = {}
#    if "order" in LOCAL_TABLE_CACHE[TableName]['header'].keys():
#        glob_order += LOCAL_TABLE_CACHE[TableName]['header']['order']
#        glob_format.update(LOCAL_TABLE_CACHE[TableName]['header']['format'])
#        glob_default.update(LOCAL_TABLE_CACHE[TableName]['header']['default'])
#        for par_name in LOCAL_TABLE_CACHE[TableName]['header']['order']:
#            LOCAL_TABLE_CACHE[TableName]['data'][par_name] = []
#    if "extra" in LOCAL_TABLE_CACHE[TableName]['header'].keys():
#        glob_order += LOCAL_TABLE_CACHE[TableName]['header']['extra']
#        glob_format.update(LOCAL_TABLE_CACHE[TableName]['header']['extra_format'])
#        for par_name in LOCAL_TABLE_CACHE[TableName]['header']['extra']:
#            glob_default[par_name] = PARAMETER_META[par_name]['default_fmt']
#            LOCAL_TABLE_CACHE[TableName]['data'][par_name] = []
#    line_count = 0
#    for line in InfileData:
#        try:
#            RowObject = getRowObjectFromString(line,TableName)
#            line_count += 1
#        except:
#            continue
#        addRowObject(RowObject,TableName)
#    LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows'] = line_count
#    # Delete all character-separated values, treat them as column-fixed.
#    try:
#        del LOCAL_TABLE_CACHE[TableName]['header']['extra']
#        del LOCAL_TABLE_CACHE[TableName]['header']['extra_format']
#        del LOCAL_TABLE_CACHE[TableName]['header']['extra_separator']
#    except:
#        pass
#    # Update header.order/format with header.extra/format if exist.
#    LOCAL_TABLE_CACHE[TableName]['header']['order'] = glob_order
#    LOCAL_TABLE_CACHE[TableName]['header']['format'] = glob_format
#    LOCAL_TABLE_CACHE[TableName]['header']['default'] = glob_default
#    InfileData.close()
#    InfileHeader.close()
#    print('                     Lines parsed: %d' % line_count)
#    pass

# / FORMAT CONVERSION LAYER    
    
def getTableNamesFromStorage(StorageName):
    file_names = listdir(StorageName)
    table_names = []
    for file_name in file_names:
        matchObject = re.search('(.+)\.header$',file_name)
        if matchObject:
           table_names.append(matchObject.group(1))
    return table_names

# FIX POSSIBLE BUG: SIMILAR NAMES OF .PAR AND .DATA FILES
# BUG FIXED BY INTRODUCING A PRIORITY:
#   *.data files have more priority than *.par files
#   See getFullTableAndHeaderName function for explanation
def scanForNewParfiles(StorageName):
    file_names = listdir(StorageName)
    headers = {} # without extensions!
    parfiles_without_header = []
    for file_name in file_names:
        # create dictionary of unique headers
        try:
            fname,fext = re.search('(.+)\.(\w+)',file_name).groups()
        except:
            continue
        if fext == 'header': headers[fname] = True
    for file_name in file_names:
        # check if extension is 'par' and the header is absent
        try:
            fname,fext = re.search('(.+)\.(\w+)',file_name).groups()
        except:
            continue
        if fext == 'par' and fname not in headers:
            parfiles_without_header.append(fname)
    return parfiles_without_header

def createHeader(TableName):
    fname = TableName+'.header'
    fp = open(VARIABLES['BACKEND_DATABASE_NAME']+'/'+fname,'w')
    if os.path.isfile(TableName):
        raise Exception('File \"%s\" already exists!' % fname)
    fp.write(json.dumps(HITRAN_DEFAULT_HEADER,indent=2))
    fp.close()

def loadCache():
    print('Using '+VARIABLES['BACKEND_DATABASE_NAME']+'\n')
    LOCAL_TABLE_CACHE = {}
    table_names = getTableNamesFromStorage(VARIABLES['BACKEND_DATABASE_NAME'])
    parfiles_without_header = scanForNewParfiles(VARIABLES['BACKEND_DATABASE_NAME'])
    # create headers for new parfiles
    for tab_name in parfiles_without_header:
        # get name without 'par' extension
        createHeader(tab_name)
        table_names.append(tab_name)
    for TableName in table_names:
        print(TableName)
        storage2cache(TableName)

def saveCache():
    try:
        # delete query buffer
        del LOCAL_TABLE_CACHE[QUERY_BUFFER]
    except:
        pass
    for TableName in LOCAL_TABLE_CACHE:
        print(TableName)
        cache2storage(TableName)

# DB backend level, start transaction
def databaseBegin(db=None):
    if db:
       VARIABLES['BACKEND_DATABASE_NAME'] = db
    else:
       VARIABLES['BACKEND_DATABASE_NAME'] = BACKEND_DATABASE_NAME_DEFAULT
    if not os.path.exists(VARIABLES['BACKEND_DATABASE_NAME']):
       os.mkdir(VARIABLES['BACKEND_DATABASE_NAME'])
    loadCache()

# DB backend level, end transaction
def databaseCommit():
    saveCache()

# ----------------------------------------------------
# ----------------------------------------------------
# CONDITIONS
# ----------------------------------------------------
# ----------------------------------------------------
# ----------------------------------------------------
# hierarchic query.condition language:
# Conditions: CONS = ('and', ('=','p1','p2'), ('<','p1',13))
# String literals are distinguished from variable names 
#  by using the operation ('STRING','some_string')
# ----------------------------------------------------

# necessary conditions for hitranonline:
SAMPLE_CONDITIONS = ('AND',('SET','internal_iso_id',[1,2,3,4,5,6]),('>=','nu',0),('<=','nu',100))

# sample hitranonline protocol
# http://hitran.cloudapp.net/lbl/5?output_format_id=1&iso_ids_list=5&numin=0&numax=100&access=api&key=e20e4bd3-e12c-4931-99e0-4c06e88536bd

CONDITION_OPERATIONS = set(['AND','OR','NOT','RANGE','IN','<','>','<=','>=','==','!=','LIKE','STR','+','-','*','/','MATCH','SEARCH','FINDALL'])

# Operations used in Condition verification
# Basic scheme: operationXXX(args),
# where args - list/array of arguments (>=1)

def operationAND(args):
    # any number if arguments
    for arg in args:
        if not arg:
           return False
    return True

def operationOR(args):
    # any number of arguments
    for arg in args:
        if arg:
           return True
    return False

def operationNOT(arg):
    # one argument
    return not arg

def operationRANGE(x,x_min,x_max):
    return x_min <= x <= x_max
    
def operationSUBSET(arg1,arg2):
    # True if arg1 is subset of arg2
    # arg1 is an element
    # arg2 is a set
    return arg1 in arg2

def operationLESS(args):
    # any number of args
    for i in range(1,len(args)):
        if args[i-1] >= args[i]:
           return False
    return True

def operationMORE(args):
    # any number of args
    for i in range(1,len(args)):
        if args[i-1] <= args[i]:
           return False
    return True

def operationLESSOREQUAL(args):
    # any number of args
    for i in range(1,len(args)):
        if args[i-1] > args[i]:
           return False
    return True

def operationMOREOREQUAL(args):
    # any number of args
    for i in range(1,len(args)):
        if args[i-1] < args[i]:
           return False
    return True

def operationEQUAL(args):
    # any number of args
    for i in range(1,len(args)):
        if args[i] != args[i-1]:
           return False
    return True

def operationNOTEQUAL(arg1,arg2):
    return arg1 != arg2
    
def operationSUM(args):
    # any numbers of arguments
    if type(args[0]) in set([int,float]):
       result = 0
    elif type(args[0]) in set([str,unicode]):
       result = ''
    else:
       raise Exception('SUM error: unknown arg type')
    for arg in args:
        result += arg
    return result

def operationDIFF(arg1,arg2):
    return arg1-arg2

def operationMUL(args):
    # any numbers of arguments
    if type(args[0]) in set([int,float]):
       result = 1
    else:
       raise Exception('MUL error: unknown arg type')
    for arg in args:
        result *= arg
    return result

def operationDIV(arg1,arg2):
    return arg1/arg2

def operationSTR(arg):
    # transform arg to str
    if type(arg)!=str:
       raise Exception('Type mismatch: STR')
    return arg

def operationSET(arg):
    # transform arg to list
    if type(arg) not in set([list,tuple,set]):
        raise Exception('Type mismatch: SET')
    return list(arg)

def operationMATCH(arg1,arg2):
    # Match regex (arg1) and string (arg2)
    #return bool(re.match(arg1,arg2)) # works wrong
    return bool(re.search(arg1,arg2))

def operationSEARCH(arg1,arg2):
    # Search regex (arg1) in string (arg2)
    # Output list of entries
    group = re.search(arg1,arg2).groups()
    result = []
    for item in group:
        result.append(('STR',item))
    return result

def operationFINDALL(arg1,arg2):
    # Search all groups of a regex
    # Output a list of groups of entries
    # XXX: If a group has more than 1 entry,
    #    there could be potential problems
    list_of_groups = re.findall(arg1,arg2)
    result = []
    for item in list_of_groups:
        result.append(('STR',item))
    return result

def operationLIST(args):
    # args is a list: do nothing (almost)
    return list(args)

# /operations

# GROUPING ---------------------------------------------- 

GROUP_INDEX = {}
# GROUP_INDEX has the following structure:
#  GROUP_INDEX[KEY] = VALUE
#    KEY = table line values
#    VALUE = {'FUNCTIONS':DICT,'FLAG':LOGICAL,'ROWID':INTEGER}
#      FUNCTIONS = {'FUNC_NAME':DICT}
#            FUNC_NAME = {'FLAG':LOGICAL,'NAME':STRING}

# name and default value
GROUP_FUNCTION_NAMES = { 'COUNT' :  0,
                         'SUM'   :  0,
                         'MUL'   :  1,
                         'AVG'   :  0,
                         'MIN'   : +1e100,
                         'MAX'   : -1e100,
                         'SSQ'   : 0,
                       }

def clearGroupIndex():
    #GROUP_INDEX = {}
    for key in GROUP_INDEX.keys():
        del GROUP_INDEX[key]

def getValueFromGroupIndex(GroupIndexKey,FunctionName):
    # If no such index_key, create it and return a value
    if FunctionName not in GROUP_FUNCTION_NAMES:
       raise Exception('No such function \"%s\"' % FunctionName)
    # In the case if NewRowObjectDefault is requested
    if not GroupIndexKey:
       return GROUP_FUNCTION_NAMES[FunctionName]
    if FunctionName not in GROUP_INDEX[GroupIndexKey]['FUNCTIONS']:
       GROUP_INDEX[GroupIndexKey]['FUNCTIONS'][FunctionName] = {}
       GROUP_INDEX[GroupIndexKey]['FUNCTIONS'][FunctionName]['FLAG'] = True
       GROUP_INDEX[GroupIndexKey]['FUNCTIONS'][FunctionName]['VALUE'] = \
         GROUP_FUNCTION_NAMES[FunctionName]
    return GROUP_INDEX[GroupIndexKey]['FUNCTIONS'][FunctionName]['VALUE']

def setValueToGroupIndex(GroupIndexKey,FunctionName,Value):
    GROUP_INDEX[GroupIndexKey]['FUNCTIONS'][FunctionName]['VALUE'] = Value

GROUP_DESC = {}
def initializeGroup(GroupIndexKey):
    if GroupIndexKey not in GROUP_INDEX:
        print('GROUP_DESC[COUNT]='+str(GROUP_DESC['COUNT']))
        GROUP_INDEX[GroupIndexKey] = {}
        GROUP_INDEX[GroupIndexKey]['FUNCTIONS'] = {}
        GROUP_INDEX[GroupIndexKey]['ROWID'] = len(GROUP_INDEX) - 1
    for FunctionName in GROUP_FUNCTION_NAMES:
        # initialize function flags (UpdateFlag)
        if FunctionName in GROUP_INDEX[GroupIndexKey]['FUNCTIONS']:
           GROUP_INDEX[GroupIndexKey]['FUNCTIONS'][FunctionName]['FLAG'] = True
    print('initializeGroup: GROUP_INDEX='+str(GROUP_INDEX))

def groupCOUNT(GroupIndexKey):
    FunctionName = 'COUNT'
    Value = getValueFromGroupIndex(GroupIndexKey,FunctionName)
    if GroupIndexKey:
       if GROUP_INDEX[GroupIndexKey]['FUNCTIONS'][FunctionName]['FLAG']:
          GROUP_INDEX[GroupIndexKey]['FUNCTIONS'][FunctionName]['FLAG'] = False
          Value = Value + 1
          setValueToGroupIndex(GroupIndexKey,FunctionName,Value)
    return Value

def groupSUM():
    pass

def groupMUL():
    pass

def groupAVG():
    pass

def groupMIN():
    pass

def groupMAX():
    pass

def groupSSQ():
    pass

OPERATORS = {\
# List
'LIST' : lambda args : operationLIST(args),
# And
'&' : lambda args : operationAND(args),
'&&' : lambda args : operationAND(args),
'AND' : lambda args : operationAND(args),
# Or
'|' : lambda args : operationOR(args),
'||' : lambda args : operationOR(args),
'OR' : lambda args : operationOR(args),
# Not
'!' : lambda args : operationNOT(args[0]),
'NOT' : lambda args : operationNOT(args[0]),
# Between
'RANGE' : lambda args : operationRANGE(args[0],args[1],args[2]),
'BETWEEN' : lambda args : operationRANGE(args[0],args[1],args[2]),
# Subset
'IN' : lambda args : operationSUBSET(args[0],args[1]),
'SUBSET': lambda args : operationSUBSET(args[0],args[1]),
# Less
'<' : lambda args : operationLESS(args),
'LESS' : lambda args : operationLESS(args),
'LT'  : lambda args : operationLESS(args),
# More
'>' : lambda args : operationMORE(args),
'MORE' : lambda args : operationMORE(args),
'MT'   : lambda args : operationMORE(args),
# Less or equal
'<=' : lambda args : operationLESSOREQUAL(args),
'LESSOREQUAL' : lambda args : operationLESSOREQUAL(args),
'LTE' : lambda args : operationLESSOREQUAL(args),
# More or equal
'>=' : lambda args : operationMOREOREQUAL(args),
'MOREOREQUAL' : lambda args : operationMOREOREQUAL(args),
'MTE' : lambda args : operationMOREOREQUAL(args),
# Equal
'=' : lambda args : operationEQUAL(args),
'==' : lambda args : operationEQUAL(args),
'EQ' : lambda args : operationEQUAL(args),
'EQUAL' : lambda args : operationEQUAL(args),
'EQUALS' : lambda args : operationEQUAL(args),
# Not equal
'!=' : lambda args : operationNOTEQUAL(args[0],args[1]),
'<>' : lambda args : operationNOTEQUAL(args[0],args[1]),
'~=' : lambda args : operationNOTEQUAL(args[0],args[1]),
'NE' : lambda args : operationNOTEQUAL(args[0],args[1]),
'NOTEQUAL' : lambda args : operationNOTEQUAL(args[0],args[1]),
# Plus
'+' : lambda args : operationSUM(args),
'SUM' : lambda args : operationSUM(args),
# Minus
'-' : lambda args : operationDIFF(args[0],args[1]),
'DIFF' : lambda args : operationDIFF(args[0],args[1]),
# Mul
'*' : lambda args : operationMUL(args),
'MUL' : lambda args : operationMUL(args),
# Div
'/' : lambda args : operationDIV(args[0],args[1]),
'DIV' : lambda args : operationDIV(args[0],args[1]),
# Regexp match
'MATCH' : lambda args : operationMATCH(args[0],args[1]),
'LIKE' : lambda args : operationMATCH(args[0],args[1]),
# Regexp search
'SEARCH' : lambda args : operationSEARCH(args[0],args[1]),
# Regexp findal
'FINDALL' : lambda args : operationFINDALL(args[0],args[1]),
# Group count
'COUNT' : lambda args : groupCOUNT(args[0]),
}
    
# new evaluateExpression function,
#  accounting for groups
"""
def evaluateExpression(root,VarDictionary,GroupIndexKey=None):
    # input = local tree root
    # XXX: this could be very slow due to passing
    #      every time VarDictionary as a parameter
    # Two special cases: 1) root=varname
    #                    2) root=list/tuple
    # These cases must be processed in a separate way
    if type(root) in set([list,tuple]):
       # root is not a leaf
       head = root[0].upper()
       # string constants are treated specially
       if head in set(['STR','STRING']): # one arg
          return operationSTR(root[1])
       elif head in set(['SET']):
          return operationSET(root[1])
       tail = root[1:]
       args = []
       # evaluate arguments recursively
       for element in tail: # resolve tree by recursion
           args.append(evaluateExpression(element,VarDictionary,GroupIndexKey))
       # call functions with evaluated arguments
       if head in set(['LIST']): # list arg
          return operationLIST(args)
       elif head in set(['&','&&','AND']): # many args 
          return operationAND(args)
       elif head in set(['|','||','OR']): # many args
          return operationOR(args)
       elif head in set(['!','NOT']): # one args
          return operationNOT(args[0])
       elif head in set(['RANGE','BETWEEN']): # three args
          return operationRANGE(args[0],args[1],args[2])
       elif head in set(['IN','SUBSET']): # two args
          return operationSUBSET(args[0],args[1])
       elif head in set(['<','LESS','LT']): # many args
          return operationLESS(args)
       elif head in set(['>','MORE','MT']): # many args
          return operationMORE(args)
       elif head in set(['<=','LESSOREQUAL','LTE']): # many args
          return operationLESSOREQUAL(args)
       elif head in set(['>=','MOREOREQUAL','MTE']): # many args
          return operationMOREOREQUAL(args)
       elif head in set(['=','==','EQ','EQUAL','EQUALS']): # many args
          return operationEQUAL(args)
       elif head in set(['!=','<>','~=','NE','NOTEQUAL']): # two args
          return operationNOTEQUAL(args[0],args[1])
       elif head in set(['+','SUM']): # many args
          return operationSUM(args)
       elif head in set(['-','DIFF']): # two args
          return operationDIFF(args[0],args[1])
       elif head in set(['*','MUL']): # many args
          return operationMUL(args)
       elif head in set(['/','DIV']): # two args
          return operationDIV(args[0],args[1])
       elif head in set(['MATCH','LIKE']): # two args
          return operationMATCH(args[0],args[1])
       elif head in set(['SEARCH']): # two args
          return operationSEARCH(args[0],args[1])
       elif head in set(['FINDALL']): # two args
          return operationFINDALL(args[0],args[1])
       # --- GROUPING OPERATIONS ---
       elif head in set(['COUNT']):
          return groupCOUNT(GroupIndexKey)
       else:
          raise Exception('Unknown operator: %s' % root[0])
    elif type(root)==str:
       # root is a par_name
       return VarDictionary[root]
    else: 
       # root is a non-string constant
       return root
"""

def evaluateExpression(root,VarDictionary,GroupIndexKey=None):
    # input = local tree root
    # XXX: this could be very slow due to passing
    #      every time VarDictionary as a parameter
    # Two special cases: 1) root=varname
    #                    2) root=list/tuple
    # These cases must be processed in a separate way
    if type(root) in set([list,tuple]):
        # root is not a leaf
        head = root[0].upper()
        # string constants are treated specially
        if head in set(['STR','STRING']): # one arg
            return operationSTR(root[1])
        elif head in set(['SET']):
            return operationSET(root[1])
        tail = root[1:]
        args = []
        # evaluate arguments recursively
        for element in tail: # resolve tree by recursion
            args.append(evaluateExpression(element,VarDictionary,GroupIndexKey))
        # call functions with evaluated arguments
        try:
            return OPERATORS[head](args)
        except KeyError:
            raise Exception('Unknown operator: %s' % head)
    elif type(root)==str:
       # root is a par_name
       return VarDictionary[root]
    else: 
       # root is a non-string constant
       return root

def getVarDictionary(RowObject):
    # get VarDict from RowObject
    # VarDict: par_name => par_value
    VarDictionary = {}
    for par_name,par_value,par_format in RowObject:
        VarDictionary[par_name] = par_value
    return VarDictionary

def checkRowObject(RowObject,Conditions,VarDictionary):
    #VarDictionary = getVarDictionary(RowObject)   
    if Conditions:
       Flag = evaluateExpression(Conditions,VarDictionary)
    else:
       Flag=True
    return Flag

# ----------------------------------------------------
# /CONDITIONS
# ----------------------------------------------------


# ----------------------------------------------------
# PARAMETER NAMES (includeing creation of new ones)
# ----------------------------------------------------

# Bind an expression to a new parameter
#   in a form: ('BIND','new_par',('some_exp',...))
def operationBIND(parname,Expression,VarDictionary):
    pass

# This section is for more detailed processing of parlists. 

# Table creation must include not only subsets of 
#   existing parameters, but also new parameters
#   derived from functions on a special prefix language
# For this reason subsetOfRowObject(..) must be substituted
#   by newRowObject(ParameterNames,RowObject)

# For parsing use the function evaluateExpression

# Get names from expression.
#  Must merge this one with evaluateExrpression.
# This is VERY LIMITED version of what will be 
#  when make the language parser is implemented.
# For more ideas and info see LANGUAGE_REFERENCE

# more advansed version of expression evaluator
def evaluateExpressionPAR(ParameterNames,VarDictionary=None):
    # RETURN: 1) Upper-level Expression names
    #         2) Upper-level Expression values
    # Is it reasonable to pass a Context to every parse function?
    # For now the function does the following:
    #   1) iterates through all UPPER-LEVEL list elements
    #   2) if element is a par name: return par name
    #      if element is an BIND expression: return bind name
    #              (see operationBIND)
    #   3) if element is an anonymous expression: return #N(=1,2,3...)
    # N.B. Binds can be only on the 0-th level of Expression    
    pass

def getContextFormat(RowObject):
    # Get context format from the whole RowObject
    ContextFormat = {}
    for par_name,par_value,par_format in RowObject:
        ContextFormat[par_name] = par_format
    return ContextFormat

def getDefaultFormat(Type):
    if Type is int:
       return '%10d'
    elif Type is float:
       return '%25.15E'
    elif Type is str:
       return '%20s'
    elif Type is bool:
       return '%2d'
    else:
       raise Exception('Unknown type')
     
def getDefaultValue(Type):
    if Type is int:
       return 0
    elif Type is float:
       return 0.0
    elif Type is str:
       return ''
    elif Type is bool:
       return False
    else:
       raise Exception('Unknown type')

# VarDictionary = Context (this name is more suitable)

# GroupIndexKey is a key to special structure/dictionary GROUP_INDEX.
# GROUP_INDEX contains information needed to calculate streamed group functions
#  such as COUNT, AVG, MIN, MAX etc...

def newRowObject(ParameterNames,RowObject,VarDictionary,ContextFormat,GroupIndexKey=None):
    # Return a subset of RowObject according to 
    # ParameterNames include either par names
    #  or expressions containing par names literals
    # ContextFormat contains format for ParNames
    anoncount = 0
    RowObjectNew = []
    for expr in ParameterNames:
        if type(expr) in set([list,tuple]): # bind
           head = expr[0]
           if head in set(['let','bind','LET','BIND']):
              par_name = expr[1]
              par_expr = expr[2]
           else:
              par_name = "#%d" % anoncount
              anoncount += 1
              par_expr = expr
           par_value = evaluateExpression(par_expr,VarDictionary,GroupIndexKey)
           try:
              par_format = expr[3]
           except:
              par_format = getDefaultFormat(type(par_value))
        else: # parname
           par_name = expr
           par_value = VarDictionary[par_name]
           par_format = ContextFormat[par_name]
        RowObjectNew.append((par_name,par_value,par_format))
    return RowObjectNew

# ----------------------------------------------------
# /PARAMETER NAMES
# ----------------------------------------------------


# ----------------------------------------------------
# OPERATIONS ON TABLES
# ----------------------------------------------------

QUERY_BUFFER = '__BUFFER__'

def getTableList():
    return LOCAL_TABLE_CACHE.keys()

def describeTable(TableName):
    """
    INPUT PARAMETERS: 
        TableName: name of the table to describe
    OUTPUT PARAMETERS: 
        none
    ---
    DESCRIPTION:
        Print information about table, including 
        parameter names, formats and wavenumber range.
    ---
    EXAMPLE OF USAGE:
        describeTable('sampletab')
    ---
    """
    print('-----------------------------------------')
    print(TableName+' summary:')
    try:
       print('-----------------------------------------')
       print('Comment: \n'+LOCAL_TABLE_CACHE[TableName]['header']['comment'])
    except:
       pass
    print('Number of rows: '+str(LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows']))
    print('Table type: '+str(LOCAL_TABLE_CACHE[TableName]['header']['table_type']))
    print('-----------------------------------------')
    print('            PAR_NAME           PAR_FORMAT')
    print('')
    for par_name in LOCAL_TABLE_CACHE[TableName]['header']['order']:
        par_format = LOCAL_TABLE_CACHE[TableName]['header']['format'][par_name]
        print('%20s %20s' % (par_name,par_format))
    print('-----------------------------------------')

# Write a table to File or STDOUT
def outputTable(TableName,Conditions=None,File=None,Header=True):
    # Display or record table with condition checking
    if File:
       Header = False
       OutputFile = open(File,'w')
    if Header:
       headstr = putTableHeaderToString(TableName)
       if File:
          OutputFile.write(headstr)
       else:
          print(headstr)
    for RowID in range(0,LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows']):
        RowObject = getRowObject(RowID,TableName)
        VarDictionary = getVarDictionary(RowObject)
        VarDictionary['LineNumber'] = RowID
        if not checkRowObject(RowObject,Conditions,VarDictionary):
           continue
        raw_string = putRowObjectToString(RowObject)
        if File:
           OutputFile.write(raw_string+'\n')
        else:
           print(raw_string)

# Create table "prototype-based" way
def createTable(TableName,RowObjectDefault):
    # create a Table based on a RowObjectDefault
    LOCAL_TABLE_CACHE[TableName] = {}
    header_order = []
    header_format = {}
    header_default = {}
    data = {}
    for par_name,par_value,par_format in RowObjectDefault:
        header_order.append(par_name)
        header_format[par_name] = par_format
        header_default[par_name] = par_value
        data[par_name] = []
    #header_order = tuple(header_order) # XXX ?
    LOCAL_TABLE_CACHE[TableName]['header']={}
    LOCAL_TABLE_CACHE[TableName]['header']['order'] = header_order 
    LOCAL_TABLE_CACHE[TableName]['header']['format'] = header_format
    LOCAL_TABLE_CACHE[TableName]['header']['default'] = header_default
    LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows'] = 0
    LOCAL_TABLE_CACHE[TableName]['header']['size_in_bytes'] = 0
    LOCAL_TABLE_CACHE[TableName]['header']['table_name'] = TableName
    LOCAL_TABLE_CACHE[TableName]['header']['table_type'] = 'column-fixed'
    LOCAL_TABLE_CACHE[TableName]['data'] = data
    

# simple "drop table" capability
def dropTable(TableName):
    """
    INPUT PARAMETERS: 
        TableName:  name of the table to delete
    OUTPUT PARAMETERS: 
        none
    ---
    DESCRIPTION:
        Deletes a table from local database.
    ---
    EXAMPLE OF USAGE:
        dropTable('some_dummy_table')
    ---
    """
    # delete Table from both Cache and Storage
    try:
       #LOCAL_TABLE_CACHE[TableName] = {}
       del LOCAL_TABLE_CACHE[TableName]
    except:
       pass
    # delete from storage
    pass # TODO

# Returns a column corresponding to parameter name
def getColumn(TableName,ParameterName):
    """
    INPUT PARAMETERS: 
        TableName:      source table name     (required)
        ParameterName:  name of column to get (required)
    OUTPUT PARAMETERS: 
        ColumnData:     list of values from specified column 
    ---
    DESCRIPTION:
        Returns a column with a name ParameterName from
        table TableName. Column is returned as a list of values.
    ---
    EXAMPLE OF USAGE:
        p1 = getColumn('sampletab','p1')
    ---
    """
    return LOCAL_TABLE_CACHE[TableName]['data'][ParameterName]

# Returns a list of columns corresponding to parameter names
def getColumns(TableName,ParameterNames):
    """
    INPUT PARAMETERS: 
        TableName:       source table name           (required)
        ParameterNames:  list of column names to get (required)
    OUTPUT PARAMETERS: 
        ListColumnData:   tuple of lists of values from specified column 
    ---
    DESCRIPTION:
        Returns columns with a names in ParameterNames from
        table TableName. Columns are returned as a tuple of lists.
    ---
    EXAMPLE OF USAGE:
        p1,p2,p3 = getColumns('sampletab',('p1','p2','p3'))
    ---
    """
    Columns = []
    for par_name in ParameterNames:
        Columns.append(LOCAL_TABLE_CACHE[TableName]['data'][par_name])
    return Columns

def addColumn(TableName,ParameterName,Before=None,Expression=None,Type=None,Default=None,Format=None):
    if ParameterName in LOCAL_TABLE_CACHE[TableName]['header']['format']:
       raise Exception('Column \"%s\" already exists' % ParameterName)
    if not Type: Type = float
    if not Default: Default = getDefaultValue(Type)
    if not Format: Format = getDefaultFormat(Type)
    number_of_rows = LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows']
    # Mess with data
    if not Expression:
       LOCAL_TABLE_CACHE[TableName]['data'][ParameterName]=[Default for i in range(0,number_of_rows)]
    else:
       data = []
       for RowID in range(0,number_of_rows):
           RowObject = getRowObject(RowID,TableName)
           VarDictionary = getVarDictionary(RowObject)
           VarDictionary['LineNumber'] = RowID
           par_value = evaluateExpression(Expression,VarDictionary)
           data.append(par_value)
           LOCAL_TABLE_CACHE[TableName]['data'][ParameterName] = data
    # Mess with header
    header_order = LOCAL_TABLE_CACHE[TableName]['header']['order']
    if not Before: 
       header_order.append(ParameterName)
    else:
       #i = 0
       #for par_name in header_order:
       #    if par_name == Before: break
       #    i += 1
       i = header_order.index(Before)
       header_order = header_order[:i] + [ParameterName,] + header_order[i:]
    LOCAL_TABLE_CACHE[TableName]['header']['order'] = header_order
    LOCAL_TABLE_CACHE[TableName]['header']['format'][ParameterName] = Format
    LOCAL_TABLE_CACHE[TableName]['header']['default'][ParameterName] = Default
   

def deleteColumn(TableName,ParameterName):
    if ParameterName not in LOCAL_TABLE_CACHE[TableName]['header']['format']:
       raise Exception('No such column \"%s\"' % ParameterName)
    # Mess with data
    i = LOCAL_TABLE_CACHE[TableName]['header']['order'].index(ParameterName)
    del LOCAL_TABLE_CACHE[TableName]['header']['order'][i]
    del LOCAL_TABLE_CACHE[TableName]['header']['format'][ParameterName]
    del LOCAL_TABLE_CACHE[TableName]['header']['default'][ParameterName]
    if not LOCAL_TABLE_CACHE[TableName]['header']['order']:
       LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows'] = 0
    # Mess with header
    del LOCAL_TABLE_CACHE[TableName]['data'][ParameterName]

def deleteColumns(TableName,ParameterNames):
    if type(ParameterNames) not in set([list,tuple,set]):
       ParameterNames = [ParameterNames]
    for ParameterName in ParameterNames:
        deleteColumn(TableName,ParameterName)

def renameColumn(TableName,OldParameterName,NewParameterName):
    pass

def insertRow():
    pass

def deleteRows(TableName,ParameterNames,Conditions):
    pass

# select from table to another table
def selectInto(DestinationTableName,TableName,ParameterNames,Conditions):
    # TableName must refer to an existing table in cache!!
    # Conditions = Restrictables in specific format
    # Sample conditions: cond = {'par1':{'range',[b_lo,b_hi]},'par2':b}
    # return structure similar to TableObject and put it to QUERY_BUFFER
    # if ParameterNames is '*' then all parameters are used
    #table_columns = LOCAL_TABLE_CACHE[TableName]['data'].keys()
    #table_length = len(TableObject['header']['number_of_rows'])
    #if ParameterNames=='*':
    #   ParameterNames = table_columns
    # check if Conditions contain elements which are not in the TableObject
    #condition_variables = getConditionVariables(Conditions)
    #strange_pars = set(condition_variables)-set(table_variables)
    #if strange_pars: 
    #   raise Exception('The following parameters are not in the table \"%s\"' % (TableName,list(strange_pars)))
    # do full scan each time
    if DestinationTableName == TableName:
       raise Exception('Selecting into source table is forbidden')
    table_length = LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows']
    row_count = 0
    for RowID in range(0,table_length):
        RowObject = getRowObject(RowID,TableName)
        VarDictionary = getVarDictionary(RowObject)
        VarDictionary['LineNumber'] = RowID
        ContextFormat = getContextFormat(RowObject)
        RowObjectNew = newRowObject(ParameterNames,RowObject,VarDictionary,ContextFormat)
        if checkRowObject(RowObject,Conditions,VarDictionary):
           addRowObject(RowObjectNew,DestinationTableName)
           row_count += 1
    LOCAL_TABLE_CACHE[DestinationTableName]['header']['number_of_rows'] += row_count

def length(TableName):
    tab_len = LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows']
    #print(str(tab_len)+' rows in '+TableName)
    return tab_len

# Select parameters from a table with certain conditions.
# Parameters can be the names or expressions.
# Conditions contain a list of expressions in a special language.
# Set Output to False to suppress output
# Set File=FileName to redirect output to a file.
def select(TableName,DestinationTableName=QUERY_BUFFER,ParameterNames=None,Conditions=None,Output=True,File=None):
    """
    INPUT PARAMETERS: 
        TableName:            name of source table              (required)
        DestinationTableName: name of resulting table           (optional)
        ParameterNames:       list of parameters or expressions (optional)
        Conditions:           list of logincal expressions      (optional)
        Output:   enable (True) or suppress (False) text output (optional)
        File:     enable (True) or suppress (False) file output (optional)
    OUTPUT PARAMETERS: 
        none
    ---
    DESCRIPTION:
        Select or filter the data in some table 
        either to standard output or to file (if specified)
    ---
    EXAMPLE OF USAGE:
        select('sampletab',DestinationTableName='outtab',ParameterNames=(p1,p2),
                Conditions=(('and',('>=','p1',1),('<',('*','p1','p2'),20))))
        Conditions means (p1>=1 and p1*p2<20)
    ---
    """
    # TODO: Variables defined in ParameterNames ('LET') MUST BE VISIBLE IN Conditions !!
    # check if table exists
    if TableName not in LOCAL_TABLE_CACHE.keys():
        raise Exception('%s: no such table. Check tableList() for more info.' % TableName)
    if not ParameterNames: ParameterNames=LOCAL_TABLE_CACHE[TableName]['header']['order']
    LOCAL_TABLE_CACHE[DestinationTableName] = {} # clear QUERY_BUFFER for the new result
    RowObjectDefault = getDefaultRowObject(TableName)
    VarDictionary = getVarDictionary(RowObjectDefault)
    ContextFormat = getContextFormat(RowObjectDefault)
    RowObjectDefaultNew = newRowObject(ParameterNames,RowObjectDefault,VarDictionary,ContextFormat)
    dropTable(DestinationTableName) # redundant
    createTable(DestinationTableName,RowObjectDefaultNew)
    selectInto(DestinationTableName,TableName,ParameterNames,Conditions)
    if DestinationTableName!=QUERY_BUFFER:
        if File: outputTable(DestinationTableName,File=File)
    elif Output:
        outputTable(DestinationTableName,File=File)

# SORTING ===========================================================

def arrangeTable(TableName,DestinationTableName=None,RowIDList=None):
    #print 'AT/'
    #print 'AT: RowIDList = '+str(RowIDList)
    # make a subset of table rows according to RowIDList
    if not DestinationTableName:
       DestinationTableName = TableName
    if DestinationTableName != TableName:
       dropTable(DestinationTableName)
       LOCAL_TABLE_CACHE[DestinationTableName]['header']=LOCAL_TABLE_CACHE[TableName]['header']
       LOCAL_TABLE_CACHE[DestinationTableName]['data']={}
    LOCAL_TABLE_CACHE[DestinationTableName]['header']['number_of_rows'] = len(RowIDList)
    #print 'AT: RowIDList = '+str(RowIDList)
    for par_name in LOCAL_TABLE_CACHE[DestinationTableName]['header']['order']:
        par_data = LOCAL_TABLE_CACHE[TableName]['data'][par_name]
        LOCAL_TABLE_CACHE[DestinationTableName]['data'][par_name] = [par_data[i] for i in RowIDList]
    
def compareLESS(RowObject1,RowObject2,ParameterNames):
    #print 'CL/'
    # arg1 and arg2 are RowObjects
    # Compare them according to ParameterNames
    # Simple validity check:
    #if len(arg1) != len(arg2):
    #   raise Exception('Arguments have different lengths')
    #RowObject1Subset = subsetOfRowObject(ParameterNames,RowObject1)
    #RowObject2Subset = subsetOfRowObject(ParameterNames,RowObject2)
    #return RowObject1Subset < RowObject2Subset
    row1 = []
    row2 = []
    #n = len(RowObject1)
    #for i in range(0,n):
    #    par_name1 = RowObject1[i][0]
    #    if par_name1 in ParameterNames:
    #       par_value1 = RowObject1[i][1]
    #       par_value2 = RowObject2[i][1]
    #       row1 += [par_value1]
    #       row2 += [par_value2]
    VarDictionary1 = getVarDictionary(RowObject1)
    VarDictionary2 = getVarDictionary(RowObject2)
    for par_name in ParameterNames:
        par_value1 = VarDictionary1[par_name]
        par_value2 = VarDictionary2[par_name]
        row1 += [par_value1]
        row2 += [par_value2]
    Flag = row1 < row2
    return Flag

def quickSort(index,TableName,ParameterNames,Accending=True):
    # ParameterNames: names of parameters which are
    #  taking part in the sorting
    if index == []:
       return []
    else:
       PivotID = index[0]
       Pivot = getRowObject(PivotID,TableName)
       lesser_index = []
       greater_index = [];
       for RowID in index[1:]:
           RowObject = getRowObject(RowID,TableName)           
           if compareLESS(RowObject,Pivot,ParameterNames): 
              lesser_index += [RowID]
           else:
              greater_index += [RowID]
       lesser = quickSort(lesser_index,TableName,ParameterNames,Accending)
       greater = quickSort(greater_index,TableName,ParameterNames,Accending)
       if Accending:
          return lesser + [PivotID] + greater
       else:
          return greater + [PivotID] + lesser

# Sorting must work well on the table itself!
def sort(TableName,DestinationTableName=None,ParameterNames=None,Accending=True,Output=False,File=None):
    """
    INPUT PARAMETERS: 
        TableName:                name of source table          (required)
        DestinationTableName:     name of resulting table       (optional)
        ParameterNames:       list of parameters or expressions to sort by    (optional)
        Accending:       sort in ascending (True) or descending (False) order (optional)
        Output:   enable (True) or suppress (False) text output (optional)
        File:     enable (True) or suppress (False) file output (optional)
    OUTPUT PARAMETERS: 
        none
    ---
    DESCRIPTION:
        Sort a table by a list of it's parameters or expressions.
        The sorted table is saved in DestinationTableName (if specified).
    ---
    EXAMPLE OF USAGE:
        sort('sampletab',ParameterNames=(p1,('+',p1,p2)))
    ---
    """
    number_of_rows = LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows']
    index = range(0,number_of_rows)
    if not DestinationTableName:
       DestinationTableName = TableName
    # if names are not provided use all parameters in sorting
    if not ParameterNames:
       ParameterNames = LOCAL_TABLE_CACHE[TableName]['header']['order']
    elif type(ParameterNames) not in set([list,tuple]):
       ParameterNames = [ParameterNames] # fix of stupid bug where ('p1',) != ('p1')
    index_sorted = quickSort(index,TableName,ParameterNames,Accending)
    arrangeTable(TableName,DestinationTableName,index_sorted)
    if Output:
       outputTable(DestinationTableName,File=File)

# /SORTING ==========================================================
    

# GROUPING ==========================================================

# GROUP_INDEX global auxiliary structure is a Dictionary,
#   which has the following properties:
#      1) Each key is a composite variable:
#          [array of values of ParameterNames variable
#           STREAM_UPDATE_FLAG]
#      2) Each value is an index in LOCAL_TABLE_CACHE[TableName]['data'][...],
#          corresponding to this key
#   STREAM_UPDATE_FLAG = TRUE if value in GROUP_INDEX needs updating
#                      = FALSE otherwise
#   If no grouping variables are specified (GroupParameterNames==None)
#    than the following key is used: "__GLOBAL__"


def group(TableName,DestinationTableName=QUERY_BUFFER,ParameterNames=None,GroupParameterNames=None,File=None,Output=True):
    """
    INPUT PARAMETERS: 
        TableName:                name of source table          (required)
        DestinationTableName:     name of resulting table       (optional)
        ParameterNames:       list of parameters or expressions to take       (optional)
        GroupParameterNames:  list of parameters or expressions to group by   (optional)
        Accending:       sort in ascending (True) or descending (False) order (optional)
        Output:   enable (True) or suppress (False) text output (optional)
    OUTPUT PARAMETERS: 
        none
    ---
    DESCRIPTION:
        none
    ---
    EXAMPLE OF USAGE:
        group('sampletab',ParameterNames=('p1',('sum','p2')),GroupParameterNames=('p1'))
        ... makes grouping by p1,p2. For each group it calculates sum of p2 values.
    ---
    """
    # Implements such functions as:
    # count,sum,avg,min,max,ssq etc...
    # 1) ParameterNames can contain group functions
    # 2) GroupParameterNames can't contain group functions
    # 3) If ParameterNames contains parameters defined by LET directive,
    #    it IS visible in the sub-context of GroupParameterNames
    # 4) Parameters defined in GroupParameterNames are NOT visible in ParameterNames
    # 5) ParameterNames variable represents the structure of the resulting table/collection
    # 6) GroupParameterNames can contain either par_names or expressions with par_names
    # Clear old GROUP_INDEX value
    clearGroupIndex()
    # Consistency check
    if TableName == DestinationTableName:
       raise Exception('TableName and DestinationTableName must be different')
    #if not ParameterNames: ParameterNames=LOCAL_TABLE_CACHE[TableName]['header']['order']
    # Prepare the new DestinationTable
    RowObjectDefault = getDefaultRowObject(TableName)
    VarDictionary = getVarDictionary(RowObjectDefault)
    ContextFormat = getContextFormat(RowObjectDefault)
    RowObjectDefaultNew = newRowObject(ParameterNames,RowObjectDefault,VarDictionary,ContextFormat)
    dropTable(DestinationTableName) # redundant
    createTable(DestinationTableName,RowObjectDefaultNew)
    # Loop through rows of source Table
    # On each iteration group functions update GROUP_INDEX (see description above)
    number_of_rows = LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows']   
    # STAGE 1: CREATE GROUPS
    print('LOOP:')
    for RowID in range(0,number_of_rows):
        print('--------------------------------')
        print('RowID='+str(RowID))
        RowObject = getRowObject(RowID,TableName) # RowObject from source table
        VarDictionary = getVarDictionary(RowObject)
        print('VarDictionary='+str(VarDictionary))
        # This is a trick which makes evaluateExpression function
        #   not consider first expression as an operation
        GroupParameterNames_ = ['LIST'] + list(GroupParameterNames)
        GroupIndexKey = evaluateExpression(GroupParameterNames_,VarDictionary)
        # List is an unhashable type in Python!
        GroupIndexKey = tuple(GroupIndexKey)       
        initializeGroup(GroupIndexKey)
        print('GROUP_INDEX='+str(GROUP_INDEX))
        ContextFormat = getContextFormat(RowObject)
        RowObjectNew = newRowObject(ParameterNames,RowObject,VarDictionary,ContextFormat,GroupIndexKey)
        RowIDGroup = GROUP_INDEX[GroupIndexKey]['ROWID']
        setRowObject(RowIDGroup,RowObjectNew,DestinationTableName)
    # Output result if required
    if Output and DestinationTableName==QUERY_BUFFER:
       outputTable(DestinationTableName,File=File)

# /GROUPING =========================================================

# EXTRACTING ========================================================

REGEX_INTEGER = '[+-]?\d+'
REGEX_STRING = '[^\s]+'
REGEX_FLOAT_F = '[+-]?\d*\.?\d+'
REGEX_FLOAT_E = '[+-]?\d*\.?\d+[eEfF]?[+-]?\d+' 

REGEX_INTEGER_FIXCOL = lambda n: '\d{%d}' % n
REGEX_STRING_FIXCOL = lambda n: '[^\s]{%d}' % n
REGEX_FLOAT_F_FIXCOL = lambda n: '[\+\-\.\d]{%d}' % n
REGEX_FLOAT_E_FIXCOL = lambda n: '[\+\-\.\deEfF]{%d}' % n

# Extract sub-columns from string column
def extractColumns(TableName,SourceParameterName,ParameterFormats,ParameterNames=None,FixCol=False):
    """
    INPUT PARAMETERS: 
        TableName:             name of source table              (required)
        SourceParameterName:   name of source column to process  (required)
        ParameterFormats:      c formats of unpacked parameters  (required)
        ParameterNames:        list of resulting parameter names (optional)
        FixCol:      column-fixed (True) format of source column (optional)
    OUTPUT PARAMETERS: 
        none
    ---
    DESCRIPTION:
        Note, that this function is aimed to do some extra job on
        interpreting string parameters which is normally supposed
        to be done by the user.
    ---
    EXAMPLE OF USAGE:
        extractColumns('sampletab',SourceParameterName='p5',
                        ParameterFormats=('%d','%d','%d'),
                        ParameterNames=('p5_1','p5_2','p5_3'))
        This example extracts three integer parameters from
        a source column 'p5' and puts results in ('p5_1','p5_2','p5_3').
    ---
    """
    # ParameterNames = just the names without expressions
    # ParFormats contains python formats for par extraction
    # Example: ParameterNames=('v1','v2','v3')
    #          ParameterFormats=('%1s','%1s','%1s')
    # By default the format of parameters is column-fixed
    if type(LOCAL_TABLE_CACHE[TableName]['header']['default'][SourceParameterName]) not in set([str,unicode]):
       raise Exception('Source parameter must be a string')
    i=-1
    # bug when (a,) != (a)
    if ParameterNames and type(ParameterNames) not in set([list,tuple]):
       ParameterNames = [ParameterNames]
    if ParameterFormats and type(ParameterFormats) not in set([list,tuple]):
       ParameterFormats = [ParameterFormats]
    # if ParameterNames is empty, fill it with #1-2-3-...
    if not ParameterNames:
       ParameterNames = []
       # using naming convension #i, i=0,1,2,3...
       for par_format in ParameterFormats:
           while True:
                 i+=1
                 par_name = '#%d' % i
                 fmt = LOCAL_TABLE_CACHE[TableName]['header']['format'].get(par_name,None)
                 if not fmt: break
           ParameterNames.append(par_name)
    # check if ParameterNames are valid
    Intersection = set(ParameterNames).intersection(LOCAL_TABLE_CACHE[TableName]['header']['order'])
    if Intersection:
       raise Exception('Parameters %s already exist' % str(list(Intersection)))
    # loop over ParameterNames to prepare LOCAL_TABLE_CACHE
    i=0
    for par_name in ParameterNames:  
        par_format = ParameterFormats[i]     
        LOCAL_TABLE_CACHE[TableName]['header']['format'][par_name]=par_format
        LOCAL_TABLE_CACHE[TableName]['data'][par_name]=[] 
        i+=1
    # append new parameters in order list
    LOCAL_TABLE_CACHE[TableName]['header']['order'] += ParameterNames
    # cope with default values
    i=0
    format_regex = []
    format_types = []
    for par_format in ParameterFormats:
        par_name = ParameterNames[i]
        regex = FORMAT_PYTHON_REGEX
        (lng,trail,lngpnt,ty) = re.search(regex,par_format).groups()
        ty = ty.lower()
        if ty == 'd':
           par_type = int
           if FixCol:
              format_regex_part = REGEX_INTEGER_FIXCOL(lng)
           else:
              format_regex_part = REGEX_INTEGER
        elif ty == 's':
           par_type = str
           if FixCol:
              format_regex_part = REGEX_STRING_FIXCOL(lng)
           else:
              format_regex_part = REGEX_STRING
        elif ty == 'f':
           par_type = float
           if FixCol:
              format_regex_part = REGEX_FLOAT_F_FIXCOL(lng)
           else:
              format_regex_part = REGEX_FLOAT_F
        elif ty == 'e':
           par_type = float
           if FixCol:
              format_regex_part = REGEX_FLOAT_E_FIXCOL(lng)
           else:
              format_regex_part = REGEX_FLOAT_E
        else:
           raise Exception('Unknown data type')
        format_regex.append('('+format_regex_part+')')
        format_types.append(par_type)
        def_val = getDefaultValue(par_type)
        LOCAL_TABLE_CACHE[TableName]['header']['default'][par_name]=def_val
        i+=1
    format_regex = '\s*'.join(format_regex)
    # loop through values of SourceParameter
    for SourceParameterString in LOCAL_TABLE_CACHE[TableName]['data'][SourceParameterName]:
        try:
           ExtractedValues = list(re.search(format_regex,SourceParameterString).groups())
        except:
           raise Exception('Error with line \"%s\"' % SourceParameterString)
        i=0
        # loop through all parameters which are supposed to be extracted
        for par_name in ParameterNames:
            par_value = format_types[i](ExtractedValues[i])
            LOCAL_TABLE_CACHE[TableName]['data'][par_name].append(par_value)
            i+=1
    # explicitly check that number of rows are equal
    number_of_rows = LOCAL_TABLE_CACHE[TableName]['header']['number_of_rows']
    number_of_rows2 = len(LOCAL_TABLE_CACHE[TableName]['data'][SourceParameterName])
    number_of_rows3 = len(LOCAL_TABLE_CACHE[TableName]['data'][ParameterNames[0]])
    if not (number_of_rows == number_of_rows2 == number_of_rows3):
       raise Exception('Error while extracting parameters: check your regexp')

# Split string columns into sub-columns with given names
def splitColumn(TableName,SourceParameterName,ParameterNames,Splitter):
    pass

# /EXTRACTING =======================================================

# ---------------------------------------------------------------
# ---------------------------------------------------------------
# /LOCAL DATABASE MANAGEMENT SYSTEM
# ---------------------------------------------------------------
# ---------------------------------------------------------------


# --------------------------------------------------------------------------
# --------------------------------------------------------------------------
# GLOBAL API FUNCTIONS
# --------------------------------------------------------------------------
# --------------------------------------------------------------------------

def mergeParlist(*arg):
    # Merge parlists and remove duplicates.
    # Argument contains a list of lists/tuples.
    container = []
    for a in arg:
        container += list(a)
    result = []
    index = set()
    for par_name in container:
        if par_name not in index:
            index.add(par_name)
            result.append(par_name)
    return result

# Define parameter groups to simplify the usage of fetch_
# "Long term" core version includes templates for the Parlists instead of listing the broadeners explicitly.
# Each template parameter has '%s' in place of a broadener, i.e. 'gamma_%s' instead of 'gamma_air' 

# ======================================================
# CODE FOR GENERATING PARAMETER NAMES FOR LINE PROFILES
# NAME: GENERATE_PARLIST
# NOTE: THIS CODE DOESN'T COVER NON-PROFILE PARAMETERS 
#       SUCH AS QUANTA, LOCAL AND GLOBAL IDS ETC...
# NOTE2: THIS CODE DOESN'T GUARANTEE THAT THE GENERATED
#        PARAMETER NAMES WILL EXIST IN THE HITRANONLINE 
#        DATABASE!
#        TO GET THE REAL PARAMETER NAMES PLEASE EITHER
#        USE THE EXTENSION OF THE HITRAN APPLICATION
#        PROGRAMMING INTERFACE:
#             (http://github.org/hitranonline/hapi2) ...
#        ... OR LOOK FOR THE CUSTOM USER FORMAT WEB PAGE
#        ON HITRANONLINE WEBSITE (http://hitran.org).
# ======================================================

VOIGT_PROFILE_TEMPLATE = ['gamma_%s','n_%s','delta_%s','deltap_%s']

SDVOIGT_PROFILE_TEMPLATE = [
    'gamma_SDV_0_%s_%d','n_SDV_%s_%d', # HWHM AND ITS T-DEPENDENCE
    'gamma_SDV_2_%s_%d','n_gamma_SDV_2_%s_%d', # SPEED-DEPENDENCE OF HWHM AND ITS T-DEPENDENCE
    'delta_SDV_%s_%d','deltap_SDV_%s_%d', # SHIFT AND ITS T-DEPENDENCE
    'SD_%s' # UNITLESS SDV PARAMETER
    ]

HT_PROFILE_TEMPLATE = [
    'gamma_HT_0_%s_%d','n_HT_%s_%d', # HWHM AND ITS T-DEPENDENCE
    'gamma_HT_2_%s_%d','n_gamma_HT_2_%s_%d', # SPEED-DEPENDENCE OF HWHM AND ITS T-DEPENDENCE
    'delta_HT_%s_%d','deltap_HT_%s_%d', # SHIFT AND ITS T-DEPENDENCE
    ]

def apply_env(template,broadener,Tref):
    args = []
    if '%s' in template:
        args.append(broadener)
    if '%d'  in template:
        args.append(Tref)
    return template%tuple(args)

def generate_parlist(profile,broadener,Tref):
    PROFILE_MAP = {
        'voigt': VOIGT_PROFILE_TEMPLATE,
        'vp': VOIGT_PROFILE_TEMPLATE,
        'sdvoigt': SDVOIGT_PROFILE_TEMPLATE,
        'sdvp': SDVOIGT_PROFILE_TEMPLATE,
        'ht': HT_PROFILE_TEMPLATE,
        'htp': HT_PROFILE_TEMPLATE,
    }
    return [apply_env(template,broadener,Tref) \
        for template in PROFILE_MAP[profile.lower()]] 
    
# generate_parlist('Voigt','air',296)  =>   gamma_air,
    
# ====================================================================        
# PARLISTS FOR EACH BROADENER EXPLICITLY (FOR BACKWARDS COMPATIBILITY)
# ====================================================================        

# Define parameter groups to simplify the usage of fetch_
PARLIST_DOTPAR = ['par_line',]
PARLIST_ID = ['trans_id',]
PARLIST_STANDARD = ['molec_id','local_iso_id','nu','sw','a','elower','gamma_air',
                    'delta_air','gamma_self','n_air','n_self','gp','gpp']
PARLIST_LABELS = ['statep','statepp']
#PARLIST_LINEMIXING = ['y_air','y_self']

PARLIST_VOIGT_AIR = ['gamma_air','delta_air','deltap_air','n_air']
PARLIST_VOIGT_SELF = ['gamma_self','delta_self','deltap_self','n_self']
PARLIST_VOIGT_H2 = ['gamma_H2','delta_H2','deltap_H2','n_H2']
PARLIST_VOIGT_CO2 = ['gamma_CO2','delta_CO2','n_CO2']
PARLIST_VOIGT_HE = ['gamma_He','delta_He','n_He']
PARLIST_VOIGT_H2O = ['gamma_H2O','n_H2O']
PARLIST_VOIGT_LINEMIXING_AIR = ['y_air']
PARLIST_VOIGT_LINEMIXING_SELF = ['y_self']
PARLIST_VOIGT_LINEMIXING_ALL = mergeParlist(PARLIST_VOIGT_LINEMIXING_AIR,
                                            PARLIST_VOIGT_LINEMIXING_SELF)
PARLIST_VOIGT_ALL = mergeParlist(PARLIST_VOIGT_AIR,PARLIST_VOIGT_SELF,
                                 PARLIST_VOIGT_H2,PARLIST_VOIGT_CO2,
                                 PARLIST_VOIGT_HE,PARLIST_VOIGT_H2O,
                                 PARLIST_VOIGT_LINEMIXING_ALL)

#PARLIST_SDVOIGT_AIR = ['gamma_air','delta_air','deltap_air','n_air','SD_air']
#PARLIST_SDVOIGT_AIR = ['gamma_SDV_0_air_296','n_SDV_air_296',
#                       'gamma_SDV_2_air_296','n_gamma_SDV_2_air_296', # n_SDV_2_air_296 ?
PARLIST_SDVOIGT_AIR = ['gamma_SDV_0_air_296',  # don't include temperature exponents while they are absent in the database
                       'gamma_SDV_2_air_296',  # don't include temperature exponents while they are absent in the database
                       'delta_SDV_0_air_296','deltap_SDV_air_296','SD_air']
#PARLIST_SDVOIGT_SELF = ['gamma_self','delta_self','deltap_self','n_self','SD_self']
#PARLIST_SDVOIGT_SELF = ['gamma_SDV_0_self_296','n_SDV_self_296',
#                       'gamma_SDV_2_self_296','n_gamma_SDV_2_self_296', # n_SDV_2_self_296 ?
PARLIST_SDVOIGT_SELF = ['gamma_SDV_0_self_296', # don't include temperature exponents while they are absent in the database
                       'gamma_SDV_2_self_296',  # don't include temperature exponents while they are absent in the database
                       'delta_SDV_0_self_296','deltap_SDV_self_296','SD_self']
PARLIST_SDVOIGT_H2 = []
PARLIST_SDVOIGT_CO2 = []
PARLIST_SDVOIGT_HE = []
#PARLIST_SDVOIGT_LINEMIXING_AIR = ['Y_SDV_air_296','n_Y_SDV_air_296']
PARLIST_SDVOIGT_LINEMIXING_AIR = ['Y_SDV_air_296'] # don't include temperature exponents while they are absent in the database
#PARLIST_SDVOIGT_LINEMIXING_SELF = ['Y_SDV_self_296','n_Y_SDV_self_296']
PARLIST_SDVOIGT_LINEMIXING_SELF = ['Y_SDV_self_296'] # don't include temperature exponents while they are absent in the database
PARLIST_SDVOIGT_LINEMIXING_ALL = mergeParlist(PARLIST_SDVOIGT_LINEMIXING_AIR,
                                              PARLIST_SDVOIGT_LINEMIXING_SELF)
PARLIST_SDVOIGT_ALL = mergeParlist(PARLIST_SDVOIGT_AIR,PARLIST_SDVOIGT_SELF,
                                   PARLIST_SDVOIGT_H2,PARLIST_SDVOIGT_CO2,
                                   PARLIST_SDVOIGT_HE,PARLIST_SDVOIGT_LINEMIXING_ALL)

PARLIST_GALATRY_AIR = ['gamma_air','delta_air','deltap_air','n_air','beta_g_air']
PARLIST_GALATRY_SELF = ['gamma_self','delta_self','deltap_self','n_self','beta_g_self']
PARLIST_GALATRY_H2 = []
PARLIST_GALATRY_CO2 = []
PARLIST_GALATRY_HE = []
PARLIST_GALATRY_ALL = mergeParlist(PARLIST_GALATRY_AIR,PARLIST_GALATRY_SELF,
                                   PARLIST_GALATRY_H2,PARLIST_GALATRY_CO2,
                                   PARLIST_GALATRY_HE)

PARLIST_HT_SELF = ['gamma_HT_0_self_50','n_HT_self_50','gamma_HT_2_self_50',
                   'delta_HT_0_self_50','deltap_HT_self_50','delta_HT_2_self_50',
                   'gamma_HT_0_self_150','n_HT_self_150','gamma_HT_2_self_150',
                   'delta_HT_0_self_150','deltap_HT_self_150','delta_HT_2_self_150',
                   'gamma_HT_0_self_296','n_HT_self_296','gamma_HT_2_self_296',
                   'delta_HT_0_self_296','deltap_HT_self_296','delta_HT_2_self_296',
                   'gamma_HT_0_self_700','n_HT_self_700','gamma_HT_2_self_700',
                   'delta_HT_0_self_700','deltap_HT_self_700','delta_HT_2_self_700',
                   'nu_HT_self','kappa_HT_self','eta_HT_self','Y_HT_self_296']
#PARLIST_HT_AIR = ['gamma_HT_0_air_50','n_HT_air_50','gamma_HT_2_air_50',
#                  'delta_HT_0_air_50','deltap_HT_air_50','delta_HT_2_air_50',
#                  'gamma_HT_0_air_150','n_HT_air_150','gamma_HT_2_air_150',
#                  'delta_HT_0_air_150','deltap_HT_air_150','delta_HT_2_air_150',
#                  'gamma_HT_0_air_296','n_HT_air_296','gamma_HT_2_air_296',
#                  'delta_HT_0_air_296','deltap_HT_air_296','delta_HT_2_air_296',
#                  'gamma_HT_0_air_700','n_HT_air_700','gamma_HT_2_air_700',
#                  'delta_HT_0_air_700','deltap_HT_air_700','delta_HT_2_air_700',
#                  'nu_HT_air','kappa_HT_air','eta_HT_air']
PARLIST_HT_AIR = ['gamma_HT_0_air_296','n_HT_air_296','gamma_HT_2_air_296',
                  'delta_HT_0_air_296','deltap_HT_air_296','delta_HT_2_air_296',
                  'nu_HT_air','kappa_HT_air','eta_HT_air','Y_HT_air_296']
PARLIST_HT_ALL = mergeParlist(PARLIST_HT_SELF,PARLIST_HT_AIR)
                                   
PARLIST_ALL = mergeParlist(PARLIST_ID,PARLIST_DOTPAR,PARLIST_STANDARD,
                           PARLIST_LABELS,PARLIST_VOIGT_ALL,
                           PARLIST_SDVOIGT_ALL,PARLIST_GALATRY_ALL,
                           PARLIST_HT_ALL)

# ====================================================================        
# PARLISTS FOR EACH BROADENER EXPLICITLY (FOR BACKWARDS COMPATIBILITY)
# ====================================================================        
                           
PARAMETER_GROUPS = {
  'par_line' : PARLIST_DOTPAR,
  '160-char' : PARLIST_DOTPAR,
  '.par' : PARLIST_DOTPAR,
  'id' : PARLIST_ID,
  'standard' : PARLIST_STANDARD,
  'labels' : PARLIST_LABELS,
  #'linemixing' : PARLIST_LINEMIXING,
  'voigt_air' : PARLIST_VOIGT_AIR,
  'voigt_self' : PARLIST_VOIGT_SELF,
  'voigt_h2' : PARLIST_VOIGT_H2,
  'voigt_co2' : PARLIST_VOIGT_CO2,
  'voigt_he' : PARLIST_VOIGT_HE,
  'voigt_h2o' : PARLIST_VOIGT_H2O,
  'voigt_linemixing_air': PARLIST_VOIGT_LINEMIXING_AIR,
  'voigt_linemixing_self': PARLIST_VOIGT_LINEMIXING_SELF,
  'voigt_linemixing': PARLIST_VOIGT_LINEMIXING_ALL,
  'voigt' : PARLIST_VOIGT_ALL,
  'sdvoigt_air' : PARLIST_SDVOIGT_AIR,
  'sdvoigt_self' : PARLIST_SDVOIGT_SELF,
  'sdvoigt_h2' : PARLIST_SDVOIGT_H2,
  'sdvoigt_co2' : PARLIST_SDVOIGT_CO2,
  'sdvoigt_he' : PARLIST_SDVOIGT_HE,
  'sdvoigt_linemixing_air': PARLIST_SDVOIGT_LINEMIXING_AIR,
  'sdvoigt_linemixing_self': PARLIST_SDVOIGT_LINEMIXING_SELF,
  'sdvoigt_linemixing': PARLIST_SDVOIGT_LINEMIXING_ALL,
  'sdvoigt' : PARLIST_SDVOIGT_ALL,
  'galatry_air' : PARLIST_GALATRY_AIR,
  'galatry_self' : PARLIST_GALATRY_SELF,
  'galatry_h2' : PARLIST_GALATRY_H2,
  'galatry_co2' : PARLIST_GALATRY_CO2,
  'galatry_he' : PARLIST_GALATRY_HE,
  'galatry' : PARLIST_GALATRY_ALL,
  'ht' : PARLIST_HT_ALL,
  'all' : PARLIST_ALL
}

def prepareParlist(pargroups=[],params=[],dotpar=True):
    # Apply defaults
    parlist_default = []
    if dotpar:
        parlist_default += ['par_line']
    #parlist_default += PARAMETER_GROUPS['id']
    
    # Make a dictionary of "assumed" parameters.
    ASSUMED_PARAMS = {}
    if 'par_line' in set(parlist_default):
        ASSUMED_PARAMS = HITRAN_DEFAULT_HEADER['format']
    
    parlist = parlist_default
    
    # Iterate over parameter groups.
    for pargroup in pargroups:
        pargroup = pargroup.lower()
        parlist += PARAMETER_GROUPS[pargroup]
        
    # Iterate over single parameters.
    for param in params:
        #param = param.lower()
        parlist.append(param)
        
    # Clean up parameter list.
    parlist = mergeParlist(parlist)
    result = []
    for param in parlist:
        if param not in ASSUMED_PARAMS:
            result.append(param)
    
    return result

def prepareHeader(parlist):
    HEADER = {'table_name':'','number_of_rows':-1,'format':{},
              'default':{},'table_type':'column-fixed',
              'size_in_bytes':-1,'order':[],'description':{}}
    
    # Add column-fixed 160-character part, if specified in parlist.
    if 'par_line' in set(parlist):
        HEADER['order'] = HITRAN_DEFAULT_HEADER['order']
        HEADER['format'] = HITRAN_DEFAULT_HEADER['format']
        HEADER['default'] = HITRAN_DEFAULT_HEADER['default']
        HEADER['description'] = HITRAN_DEFAULT_HEADER['description']
        HEADER['position'] = HITRAN_DEFAULT_HEADER['position']

    # Insert all other parameters in the "extra" section of the header.
    plist = [v for v in parlist if v!='par_line']
    HEADER['extra'] = []
    HEADER['extra_format'] = {}
    HEADER['extra_separator'] = ','
    for param in plist:
        param = param.lower()
        HEADER['extra'].append(param)
        HEADER['extra_format'][param] = PARAMETER_META.get(param,PMETA_DEFAULT)['default_fmt']
        
    return HEADER
        
def queryHITRAN(TableName,iso_id_list,numin,numax,pargroups=[],params=[],dotpar=True,head=False):
    ParameterList = prepareParlist(pargroups=pargroups,params=params,dotpar=dotpar)
    TableHeader = prepareHeader(ParameterList)
    TableHeader['table_name'] = TableName
    DataFileName = VARIABLES['BACKEND_DATABASE_NAME'] + '/' + TableName + '.data'
    HeaderFileName = VARIABLES['BACKEND_DATABASE_NAME'] + '/' + TableName + '.header'
    # create URL
    iso_id_list_str = [str(iso_id) for iso_id in iso_id_list]
    iso_id_list_str = ','.join(iso_id_list_str)
    print('\nData is fetched from %s\n'%VARIABLES['GLOBAL_HOST'])
    if pargroups or params: # custom par search
        url = VARIABLES['GLOBAL_HOST'] + '/lbl/api?' + \
        'iso_ids_list=' + iso_id_list_str + '&' + \
        'numin=' + str(numin) + '&' + \
        'numax=' + str(numax) + '&' + \
        'head=' + str(head) + '&' + \
        'fixwidth=0&sep=[comma]&' +\
        'request_params=' + ','.join(ParameterList)
    else: # old-fashioned .par search
        url = VARIABLES['GLOBAL_HOST'] + '/lbl/api?' + \
        'iso_ids_list=' + iso_id_list_str + '&' + \
        'numin=' + str(numin) + '&' + \
        'numax=' + str(numax)
    #raise Exception(url)
    # Download data by chunks.
    if VARIABLES['DISPLAY_FETCH_URL']: print(url+'\n')
    try:       
        # Proxy handling # https://stackoverflow.com/questions/1450132/proxy-with-urllib2
        if VARIABLES['PROXY']:
            print('Using proxy '+str(VARIABLES['PROXY']))
            proxy = urllib2.ProxyHandler(VARIABLES['PROXY'])
            opener = urllib2.build_opener(proxy)
            urllib2.install_opener(opener)            
        req = urllib2.urlopen(url)
    except urllib2.HTTPError as err: 
        if err.code == 403:
            raise Exception('You have exceeded the daily limit of API queries.')
        else:    
            raise Exception('Failed to retrieve data for given parameters.')            
    except urllib2.URLError:
        raise Exception('Cannot connect to %s. Try again or edit GLOBAL_HOST variable.' % GLOBAL_HOST)
    CHUNK = 64 * 1024
    print('BEGIN DOWNLOAD: '+TableName)
    with open_(DataFileName,'w') as fp:
       while True:
          chunk = req.read(CHUNK)
          if not chunk: break
          fp.write(chunk.decode('utf-8'))
          print('  %d bytes written to %s' % (CHUNK,DataFileName))
    with open(HeaderFileName,'w') as fp:       
       fp.write(json.dumps(TableHeader,indent=2))
       print('Header written to %s' % HeaderFileName)
    print('END DOWNLOAD')
    # Set comment
    # Get this table to LOCAL_TABLE_CACHE
    storage2cache(TableName)
    print('PROCESSED')

def saveHeader(TableName):
    ParameterList = prepareParlist(dotpar=True)    
    TableHeader = prepareHeader(ParameterList)
    with open(TableName+'.header','w') as fp:
       fp.write(json.dumps(TableHeader,indent=2))
    
# ---------- DATABASE FRONTEND END -------------

# simple implementation of getting a line list from a remote server
def getLinelist(local_name,query,api_key):
    return makeQuery(local_name)

# -------------------------------------------------------------------
# -------------------------------------------------------------------
# / GLOBABL API FUNCTIONS
# -------------------------------------------------------------------
# -------------------------------------------------------------------



# ---------------- FILTER ---------------------------------------------

def filter(TableName,Conditions):
    select(TableName=TableName,Conditions=Conditions,Output=False)

# ---------------------- ISO.PY ---------------------------------------

ISO_ID_INDEX = {

'M':0,
'I':1,
'iso_name':2,
'abundance':3,
'mass':4,
'mol_name':5

}

ISO_INDEX = {

'id':0,
'iso_name':1,
'abundance':2,
'mass':3,
'mol_name':4

}

#  M   I             id   iso_name                   abundance      mass           mol_name

ISO = {
(  1,  1 ):    [      1,  'H2(16O)',                 9.973173E-01,  1.801056E+01,  'H2O'         ], 
(  1,  2 ):    [      2,  'H2(18O)',                 1.999827E-03,  2.001481E+01,  'H2O'         ], 
(  1,  3 ):    [      3,  'H2(17O)',                 3.718841E-04,  1.901478E+01,  'H2O'         ], 
(  1,  4 ):    [      4,  'HD(16O)',                 3.106928E-04,  1.901674E+01,  'H2O'         ], 
(  1,  5 ):    [      5,  'HD(18O)',                 6.230031E-07,  2.102098E+01,  'H2O'         ], 
(  1,  6 ):    [      6,  'HD(17O)',                 1.158526E-07,  2.002096E+01,  'H2O'         ], 
(  1,  7 ):    [    129,  'D2(16O)',                 2.419741E-08,  2.002292E+01,  'H2O'         ], 
(  2,  1 ):    [      7,  '(12C)(16O)2',             9.842043E-01,  4.398983E+01,  'CO2'         ], 
(  2,  2 ):    [      8,  '(13C)(16O)2',             1.105736E-02,  4.499318E+01,  'CO2'         ], 
(  2,  3 ):    [      9,  '(16O)(12C)(18O)',         3.947066E-03,  4.599408E+01,  'CO2'         ], 
(  2,  4 ):    [     10,  '(16O)(12C)(17O)',         7.339890E-04,  4.499404E+01,  'CO2'         ], 
(  2,  5 ):    [     11,  '(16O)(13C)(18O)',         4.434456E-05,  4.699743E+01,  'CO2'         ], 
(  2,  6 ):    [     12,  '(16O)(13C)(17O)',         8.246233E-06,  4.599740E+01,  'CO2'         ], 
(  2,  7 ):    [     13,  '(12C)(18O)2',             3.957340E-06,  4.799832E+01,  'CO2'         ], 
(  2,  8 ):    [     14,  '(17O)(12C)(18O)',         1.471799E-06,  4.699829E+01,  'CO2'         ], 
(  2,  9 ):    [    121,  '(12C)(17O)2',             1.368466E-07,  4.599826E+01,  'CO2'         ], 
(  2, 10 ):    [     15,  '(13C)(18O)2',             4.446000E-08,  4.900167E+01,  'CO2'         ], 
(  2, 11 ):    [    120,  '(18O)(13C)(17O)',         1.653540E-08,  4.800165E+01,  'CO2'         ], 
(  2, 12 ):    [    122,  '(13C)(17O)2',             1.537446E-09,  4.700162E+01,  'CO2'         ], 
(  3,  1 ):    [     16,  '(16O)3',                  9.929009E-01,  4.798474E+01,  'O3'          ], 
(  3,  2 ):    [     17,  '(16O)(16O)(18O)',         3.981942E-03,  4.998899E+01,  'O3'          ], 
(  3,  3 ):    [     18,  '(16O)(18O)(16O)',         1.990971E-03,  4.998899E+01,  'O3'          ], 
(  3,  4 ):    [     19,  '(16O)(16O)(17O)',         7.404746E-04,  4.898896E+01,  'O3'          ], 
(  3,  5 ):    [     20,  '(16O)(17O)(16O)',         3.702373E-04,  4.898896E+01,  'O3'          ], 
(  4,  1 ):    [     21,  '(14N)2(16O)',             9.903328E-01,  4.400106E+01,  'N2O'         ], 
(  4,  2 ):    [     22,  '(14N)(15N)(16O)',         3.640926E-03,  4.499810E+01,  'N2O'         ], 
(  4,  3 ):    [     23,  '(15N)(14N)(16O)',         3.640926E-03,  4.499810E+01,  'N2O'         ], 
(  4,  4 ):    [     24,  '(14N)2(18O)',             1.985822E-03,  4.600531E+01,  'N2O'         ], 
(  4,  5 ):    [     25,  '(14N)2(17O)',             3.692797E-04,  4.500528E+01,  'N2O'         ], 
(  5,  1 ):    [     26,  '(12C)(16O)',              9.865444E-01,  2.799491E+01,  'CO'          ], 
(  5,  2 ):    [     27,  '(13C)(16O)',              1.108364E-02,  2.899827E+01,  'CO'          ], 
(  5,  3 ):    [     28,  '(12C)(18O)',              1.978224E-03,  2.999916E+01,  'CO'          ], 
(  5,  4 ):    [     29,  '(12C)(17O)',              3.678671E-04,  2.899913E+01,  'CO'          ], 
(  5,  5 ):    [     30,  '(13C)(18O)',              2.222500E-05,  3.100252E+01,  'CO'          ], 
(  5,  6 ):    [     31,  '(13C)(17O)',              4.132920E-06,  3.000249E+01,  'CO'          ], 
(  6,  1 ):    [     32,  '(12C)H4',                 9.882741E-01,  1.603130E+01,  'CH4'         ], 
(  6,  2 ):    [     33,  '(13C)H4',                 1.110308E-02,  1.703466E+01,  'CH4'         ], 
(  6,  3 ):    [     34,  '(12C)H3D',                6.157511E-04,  1.703748E+01,  'CH4'         ], 
(  6,  4 ):    [     35,  '(13C)H3D',                6.917852E-06,  1.804083E+01,  'CH4'         ], 
(  7,  1 ):    [     36,  '(16O)2',                  9.952616E-01,  3.198983E+01,  'O2'          ], 
(  7,  2 ):    [     37,  '(16O)(18O)',              3.991410E-03,  3.399408E+01,  'O2'          ], 
(  7,  3 ):    [     38,  '(16O)(17O)',              7.422352E-04,  3.299404E+01,  'O2'          ], 
(  8,  1 ):    [     39,  '(14N)(16O)',              9.939737E-01,  2.999799E+01,  'NO'          ], 
(  8,  2 ):    [     40,  '(15N)(16O)',              3.654311E-03,  3.099502E+01,  'NO'          ], 
(  8,  3 ):    [     41,  '(14N)(18O)',              1.993122E-03,  3.200223E+01,  'NO'          ], 
(  9,  1 ):    [     42,  '(32S)(16O)2',             9.456777E-01,  6.396190E+01,  'SO2'         ], 
(  9,  2 ):    [     43,  '(34S)(16O)2',             4.195028E-02,  6.595770E+01,  'SO2'         ], 
(  9,  3 ):    [    137,  '(33S)(16O)2',             7.464462E-03,  6.496129E+01,  'SO2'         ], 
(  9,  4 ):    [    138,  '(16O)(32S)(18O)',         3.792558E-03,  6.596615E+01,  'SO2'         ], 
( 10,  1 ):    [     44,  '(14N)(16O)2',             9.916160E-01,  4.599290E+01,  'NO2'         ], 
( 10,  2 ):    [    130,  '(15N)(16O)2',             3.645643E-03,  4.698994E+01,  'NO2'         ], 
( 11,  1 ):    [     45,  '(14N)H3',                 9.958716E-01,  1.702655E+01,  'NH3'         ], 
( 11,  2 ):    [     46,  '(15N)H3',                 3.661289E-03,  1.802358E+01,  'NH3'         ], 
( 12,  1 ):    [     47,  'H(14N)(16O)3',            9.891098E-01,  6.299564E+01,  'HNO3'        ], 
( 12,  2 ):    [    117,  'H(15N)(16O)3',            3.636429E-03,  6.399268E+01,  'HNO3'        ], 
( 13,  1 ):    [     48,  '(16O)H',                  9.974726E-01,  1.700274E+01,  'OH'          ], 
( 13,  2 ):    [     49,  '(18O)H',                  2.000138E-03,  1.900699E+01,  'OH'          ], 
( 13,  3 ):    [     50,  '(16O)D',                  1.553706E-04,  1.800891E+01,  'OH'          ], 
( 14,  1 ):    [     51,  'H(19F)',                  9.998443E-01,  2.000623E+01,  'HF'          ], 
( 14,  2 ):    [    110,  'D(19F)',                  1.557410E-04,  2.101240E+01,  'HF'          ], 
( 15,  1 ):    [     52,  'H(35Cl)',                 7.575870E-01,  3.597668E+01,  'HCl'         ], 
( 15,  2 ):    [     53,  'H(37Cl)',                 2.422573E-01,  3.797373E+01,  'HCl'         ], 
( 15,  3 ):    [    107,  'D(35Cl)',                 1.180050E-04,  3.698285E+01,  'HCl'         ], 
( 15,  4 ):    [    108,  'D(37Cl)',                 3.773502E-05,  3.897990E+01,  'HCl'         ], 
( 16,  1 ):    [     54,  'H(79Br)',                 5.067811E-01,  7.992616E+01,  'HBr'         ], 
( 16,  2 ):    [     55,  'H(81Br)',                 4.930632E-01,  8.192412E+01,  'HBr'         ], 
( 16,  3 ):    [    111,  'D(79Br)',                 7.893838E-05,  8.093234E+01,  'HBr'         ], 
( 16,  4 ):    [    112,  'D(81Br)',                 7.680162E-05,  8.293029E+01,  'HBr'         ], 
( 17,  1 ):    [     56,  'H(127I)',                 9.998443E-01,  1.279123E+02,  'HI'          ], 
( 17,  2 ):    [    113,  'D(127I)',                 1.557410E-04,  1.289185E+02,  'HI'          ], 
( 18,  1 ):    [     57,  '(35Cl)(16O)',             7.559077E-01,  5.096377E+01,  'ClO'         ], 
( 18,  2 ):    [     58,  '(37Cl)(16O)',             2.417203E-01,  5.296082E+01,  'ClO'         ], 
( 19,  1 ):    [     59,  '(16O)(12C)(32S)',         9.373947E-01,  5.996699E+01,  'OCS'         ], 
( 19,  2 ):    [     60,  '(16O)(12C)(34S)',         4.158284E-02,  6.196278E+01,  'OCS'         ], 
( 19,  3 ):    [     61,  '(16O)(13C)(32S)',         1.053146E-02,  6.097034E+01,  'OCS'         ], 
( 19,  4 ):    [     62,  '(16O)(12C)(33S)',         7.399083E-03,  6.096637E+01,  'OCS'         ], 
( 19,  5 ):    [     63,  '(18O)(12C)(32S)',         1.879670E-03,  6.197123E+01,  'OCS'         ], 
( 19,  6 ):    [    135,  '(16O)(13C)(34S)',         4.671757E-04,  6.296614E+01,  'OCS'         ], 
( 20,  1 ):    [     64,  'H2(12C)(16O)',            9.862371E-01,  3.001056E+01,  'H2CO'        ], 
( 20,  2 ):    [     65,  'H2(13C)(16O)',            1.108020E-02,  3.101392E+01,  'H2CO'        ], 
( 20,  3 ):    [     66,  'H2(12C)(18O)',            1.977609E-03,  3.201481E+01,  'H2CO'        ], 
( 21,  1 ):    [     67,  'H(16O)(35Cl)',            7.557900E-01,  5.197159E+01,  'HOCl'        ], 
( 21,  2 ):    [     68,  'H(16O)(37Cl)',            2.416826E-01,  5.396864E+01,  'HOCl'        ], 
( 22,  1 ):    [     69,  '(14N)2',                  9.926874E-01,  2.800615E+01,  'N2'          ], 
( 22,  2 ):    [    118,  '(14N)(15N)',              7.299165E-03,  2.900318E+01,  'N2'          ], 
( 23,  1 ):    [     70,  'H(12C)(14N)',             9.851143E-01,  2.701090E+01,  'HCN'         ], 
( 23,  2 ):    [     71,  'H(13C)(14N)',             1.106758E-02,  2.801425E+01,  'HCN'         ], 
( 23,  3 ):    [     72,  'H(12C)(15N)',             3.621740E-03,  2.800793E+01,  'HCN'         ], 
( 24,  1 ):    [     73,  '(12C)H3(35Cl)',           7.489369E-01,  4.999233E+01,  'CH3Cl'       ], 
( 24,  2 ):    [     74,  '(12C)H3(37Cl)',           2.394912E-01,  5.198938E+01,  'CH3Cl'       ], 
( 25,  1 ):    [     75,  'H2(16O)2',                9.949516E-01,  3.400548E+01,  'H2O2'        ], 
( 26,  1 ):    [     76,  '(12C)2H2',                9.775989E-01,  2.601565E+01,  'C2H2'        ], 
( 26,  2 ):    [     77,  '(12C)(13C)H2',            2.196629E-02,  2.701900E+01,  'C2H2'        ], 
( 26,  3 ):    [    105,  '(12C)2HD',                3.045499E-04,  2.702182E+01,  'C2H2'        ], 
( 27,  1 ):    [     78,  '(12C)2H6',                9.769900E-01,  3.004695E+01,  'C2H6'        ], 
( 27,  2 ):    [    106,  '(12C)H3(13C)H3',          2.195261E-02,  3.105031E+01,  'C2H6'        ], 
( 28,  1 ):    [     79,  '(31P)H3',                 9.995329E-01,  3.399724E+01,  'PH3'         ], 
( 29,  1 ):    [     80,  '(12C)(16O)(19F)2',        9.865444E-01,  6.599172E+01,  'COF2'        ], 
( 29,  2 ):    [    119,  '(13C)(16O)(19F)2',        1.108366E-02,  6.699508E+01,  'COF2'        ], 
( 30,  1 ):    [    126,  '(32S)(19F)6',             9.501800E-01,  1.459625E+02,  'SF6'         ], 
( 31,  1 ):    [     81,  'H2(32S)',                 9.498841E-01,  3.398772E+01,  'H2S'         ], 
( 31,  2 ):    [     82,  'H2(34S)',                 4.213687E-02,  3.598351E+01,  'H2S'         ], 
( 31,  3 ):    [     83,  'H2(33S)',                 7.497664E-03,  3.498710E+01,  'H2S'         ], 
( 32,  1 ):    [     84,  'H(12C)(16O)(16O)H',       9.838977E-01,  4.600548E+01,  'HCOOH'       ], 
( 33,  1 ):    [     85,  'H(16O)2',                 9.951066E-01,  3.299766E+01,  'HO2'         ], 
( 34,  1 ):    [     86,  '(16O)',                   9.976280E-01,  1.599492E+01,  'O'           ], 
( 35,  1 ):    [    127,  '(35Cl)(16O)(14N)(16O)2',  7.495702E-01,  9.695667E+01,  'ClONO2'      ], 
( 35,  2 ):    [    128,  '(37Cl)(16O)(14N)(16O)2',  2.396937E-01,  9.895372E+01,  'ClONO2'      ], 
( 36,  1 ):    [     87,  '(14N)(16O)+',             9.939737E-01,  2.999799E+01,  'NOp'         ], 
( 37,  1 ):    [     88,  'H(16O)(79Br)',            5.055790E-01,  9.592108E+01,  'HOBr'        ], 
( 37,  2 ):    [     89,  'H(16O)(81Br)',            4.918937E-01,  9.791903E+01,  'HOBr'        ], 
( 38,  1 ):    [     90,  '(12C)2H4',                9.772944E-01,  2.803130E+01,  'C2H4'        ], 
( 38,  2 ):    [     91,  '(12C)H2(13C)H2',          2.195946E-02,  2.903466E+01,  'C2H4'        ], 
( 39,  1 ):    [     92,  '(12C)H3(16O)H',           9.859299E-01,  3.202622E+01,  'CH3OH'       ], 
( 40,  1 ):    [     93,  '(12C)H3(79Br)',           5.009946E-01,  9.394181E+01,  'CH3Br'       ], 
( 40,  2 ):    [     94,  '(12C)H3(81Br)',           4.874334E-01,  9.593976E+01,  'CH3Br'       ], 
( 41,  1 ):    [     95,  '(12C)H3(12C)(14N)',       9.738662E-01,  4.102655E+01,  'CH3CN'       ], 
( 42,  1 ):    [     96,  '(12C)(19F)4',             9.888900E-01,  8.799362E+01,  'CF4'         ], 
( 43,  1 ):    [    116,  '(12C)4H2',                9.559980E-01,  5.001565E+01,  'C4H2'        ], 
( 44,  1 ):    [    109,  'H(12C)3(14N)',            9.633460E-01,  5.101090E+01,  'HC3N'        ], 
( 45,  1 ):    [    103,  'H2',                      9.996885E-01,  2.015650E+00,  'H2'          ], 
( 45,  2 ):    [    115,  'HD',                      3.114316E-04,  3.021825E+00,  'H2'          ], 
( 46,  1 ):    [     97,  '(12C)(32S)',              9.396236E-01,  4.397207E+01,  'CS'          ], 
( 46,  2 ):    [     98,  '(12C)(34S)',              4.168171E-02,  4.596787E+01,  'CS'          ], 
( 46,  3 ):    [     99,  '(13C)(32S)',              1.055650E-02,  4.497543E+01,  'CS'          ], 
( 46,  4 ):    [    100,  '(12C)(33S)',              7.416675E-03,  4.497146E+01,  'CS'          ], 
( 47,  1 ):    [    114,  '(32S)(16O)3',             9.434345E-01,  7.995682E+01,  'SO3'         ], 
( 48,  1 ):    [    123,  '(12C)2(14N)2',            9.707524E-01,  5.200615E+01,  'C2N2'        ], 
( 49,  1 ):    [    124,  '(12C)(16O)(35Cl)2',       5.663918E-01,  9.793262E+01,  'COCl2'       ], 
( 49,  2 ):    [    125,  '(12C)(16O)(35Cl)(37Cl)',  3.622350E-01,  9.992967E+01,  'COCl2'       ], 
( 50,  1 ):    [    146,  '(32S)(16O)',              9.479262E-01,  4.796699E+01,  'SO'          ], 
( 50,  2 ):    [    147,  '(34S)(16O)',              4.205002E-02,  4.996278E+01,  'SO'          ], 
( 50,  3 ):    [    148,  '(32S)(18O)',              1.900788E-03,  4.997123E+01,  'SO'          ], 
( 51,  1 ):    [    144,  '(12C)H3(19F)',            9.884280E-01,  3.402188E+01,  'CH3F'        ], 
( 52,  1 ):    [    139,  '(74Ge)H4',                3.651724E-01,  7.795248E+01,  'GeH4'        ], 
( 52,  2 ):    [    140,  '(72Ge)H4',                2.741292E-01,  7.595338E+01,  'GeH4'        ], 
( 52,  3 ):    [    141,  '(70Ge)H4',                2.050722E-01,  7.395555E+01,  'GeH4'        ], 
( 52,  4 ):    [    142,  '(73Ge)H4',                7.755167E-02,  7.695476E+01,  'GeH4'        ], 
( 52,  5 ):    [    143,  '(76Ge)H4',                7.755167E-02,  7.995270E+01,  'GeH4'        ], 
( 53,  1 ):    [    131,  '(12C)(32S)2',             8.928115E-01,  7.594414E+01,  'CS2'         ], 
( 53,  2 ):    [    132,  '(32S)(12C)(34S)',         7.921026E-02,  7.793994E+01,  'CS2'         ], 
( 53,  3 ):    [    133,  '(32S)(12C)(33S)',         1.409435E-02,  7.694353E+01,  'CS2'         ], 
( 53,  4 ):    [    134,  '(13C)(32S)2',             1.003057E-02,  7.694750E+01,  'CS2'         ], 
( 54,  1 ):    [    145,  '(12C)H3(127I)',           9.884280E-01,  1.419279E+02,  'CH3I'        ], 
( 55,  1 ):    [    136,  '(14N)(19F)3',             9.963370E-01,  7.099829E+01,  'NF3'         ], 
}

# calculate ISO_ID instead of repeating the same information twice
ISO_ID = {}
for mol_id,iso_id in ISO:
    ln = ISO[(mol_id,iso_id)]
    glob_iso_id = ln[0]
    ln_ = [mol_id,iso_id]+ln[1:]
    ISO_ID[glob_iso_id] = ln_

def print_iso():
    print('The dictionary \"ISO\" contains information on isotopologues in HITRAN\n')
    print('   M    I          id                  iso_name   abundance      mass        mol_name')
    for i in ISO:
        ab = ISO[i][ISO_INDEX['abundance']]
        ma = ISO[i][ISO_INDEX['mass']]
        ab = ab if ab else -1
        ma = ma if ma else -1
        print('%4i %4i     : %5i %25s %10f %10f %15s' % (i[0],i[1],ISO[i][ISO_INDEX['id']],ISO[i][ISO_INDEX['iso_name']],ab,ma,ISO[i][ISO_INDEX['mol_name']]))

def print_iso_id():
    print('The dictionary \"ISO_ID\" contains information on \"global\" IDs of isotopologues in HITRAN\n')
    print('   id            M    I                    iso_name       abundance       mass        mol_name')
    for i in ISO_ID:
        ab = ISO_ID[i][ISO_ID_INDEX['abundance']]
        ma = ISO_ID[i][ISO_ID_INDEX['mass']]
        ab = ab if ab else -1
        ma = ma if ma else -1
        print('%5i     :   %4i %4i   %25s %15.10f %10f %15s' % (i,ISO_ID[i][ISO_ID_INDEX['M']],ISO_ID[i][ISO_ID_INDEX['I']],ISO_ID[i][ISO_ID_INDEX['iso_name']],ab,ma,ISO_ID[i][ISO_ID_INDEX['mol_name']]))

profiles = 'profiles'
def print_profiles():
    print('Profiles available:')
    print('  HT        : PROFILE_HT')
    print('  SDRautian : PROFILE_SDRAUTIAN')
    print('  Rautian   : PROFILE_RAUTIAN')
    print('  SDVoigt   : PROFILE_SDVOIGT')
    print('  Voigt     : PROFILE_VOIGT')
    print('  Lorentz   : PROFILE_LORENTZ')
    print('  Doppler   : PROFILE_DOPPLER')

slit_functions = 'slit_functions'
def print_slit_functions():
    print('  RECTANGULAR : SLIT_RECTANGULAR')
    print('  TRIANGULAR  : SLIT_TRIANGULAR')
    print('  GAUSSIAN    : SLIT_GAUSSIAN')
    print('  DIFFRACTION : SLIT_DIFFRACTION')
    print('  MICHELSON   : SLIT_MICHELSON')
    print('  DISPERSION/LORENTZ : SLIT_DISPERSION')

tutorial='tutorial'
units='units'
index='index'
data='data'
spectra='spectra'
plotting='plotting'
python='python'

python_tutorial_text = \
"""
THIS TUTORIAL IS TAKEN FROM http://www.stavros.io/tutorials/python/
AUTHOR: Stavros Korokithakis


----- LEARN PYTHON IN 10 MINUTES -----


PRELIMINARY STUFF

So, you want to learn the Python programming language but can't find a concise 
and yet full-featured tutorial. This tutorial will attempt to teach you Python in 10 minutes. 
It's probably not so much a tutorial as it is a cross between a tutorial and a cheatsheet, 
so it will just show you some basic concepts to start you off. Obviously, if you want to 
really learn a language you need to program in it for a while. I will assume that you are 
already familiar with programming and will, therefore, skip most of the non-language-specific stuff. 
The important keywords will be highlighted so you can easily spot them. Also, pay attention because, 
due to the terseness of this tutorial, some things will be introduced directly in code and only 
briefly commented on.


PROPERTIES

Python is strongly typed (i.e. types are enforced), dynamically, implicitly typed (i.e. you don't 
have to declare variables), case sensitive (i.e. var and VAR are two different variables) and 
object-oriented (i.e. everything is an object). 


GETTING HELP

Help in Python is always available right in the interpreter. If you want to know how an object works, 
all you have to do is call help(<object>)! Also useful are dir(), which shows you all the object's methods, 
and <object>.__doc__, which shows you its documentation string: 

>>> help(5)
Help on int object:
(etc etc)

>>> dir(5)
['__abs__', '__add__', ...]

>>> abs.__doc__
'abs(number) -> number

Return the absolute value of the argument.'


SYNTAX

Python has no mandatory statement termination characters and blocks are specified by indentation. 
Indent to begin a block, dedent to end one. Statements that expect an indentation level end in a colon (:). 
Comments start with the pound (#) sign and are single-line, multi-line strings are used for multi-line comments. 
Values are assigned (in fact, objects are bound to names) with the _equals_ sign ("="), and equality testing is 
done using two _equals_ signs ("=="). You can increment/decrement values using the += and -= operators respectively 
by the right-hand amount. This works on many datatypes, strings included. You can also use multiple variables on one 
line. For example: 

>>> myvar = 3
>>> myvar += 2
>>> myvar
5

>>> myvar -= 1
>>> myvar
4

\"\"\"This is a multiline comment.
The following lines concatenate the two strings.\"\"\"

>>> mystring = "Hello"
>>> mystring += " world."
>>> print mystring
Hello world.

# This swaps the variables in one line(!).
# It doesn't violate strong typing because values aren't
# actually being assigned, but new objects are bound to
# the old names.
>>> myvar, mystring = mystring, myvar


DATA TYPES

The data structures available in python are lists, tuples and dictionaries. 
Sets are available in the sets library (but are built-in in Python 2.5 and later). 
Lists are like one-dimensional arrays (but you can also have lists of other lists), 
dictionaries are associative arrays (a.k.a. hash tables) and tuples are immutable 
one-dimensional arrays (Python "arrays" can be of any type, so you can mix e.g. integers, 
strings, etc in lists/dictionaries/tuples). The index of the first item in all array types is 0. 
Negative numbers count from the end towards the beginning, -1 is the last item. Variables 
can point to functions. The usage is as follows:

>>> sample = [1, ["another", "list"], ("a", "tuple")]
>>> mylist = ["List item 1", 2, 3.14]
>>> mylist[0] = "List item 1 again" # We're changing the item.
>>> mylist[-1] = 3.21 # Here, we refer to the last item.
>>> mydict = {"Key 1": "Value 1", 2: 3, "pi": 3.14}
>>> mydict["pi"] = 3.15 # This is how you change dictionary values.
>>> mytuple = (1, 2, 3)
>>> myfunction = len
>>> print myfunction(mylist)
3


You can access array ranges using a colon (:). Leaving the start index empty assumes the first item, 
leaving the end index assumes the last item. Negative indexes count from the last item backwards 
(thus -1 is the last item) like so:

>>> mylist = ["List item 1", 2, 3.14]
>>> print mylist[:]
['List item 1', 2, 3.1400000000000001]

>>> print mylist[0:2]
['List item 1', 2]

>>> print mylist[-3:-1]
['List item 1', 2]

>>> print mylist[1:]
[2, 3.14]

# Adding a third parameter, "step" will have Python step in
# N item increments, rather than 1.
# E.g., this will return the first item, then go to the third and
# return that (so, items 0 and 2 in 0-indexing).
>>> print mylist[::2]
['List item 1', 3.14]


STRINGS

Its strings can use either single or double quotation marks, and you can have quotation 
marks of one kind inside a string that uses the other kind (i.e. "He said 'hello'." is valid). 
Multiline strings are enclosed in _triple double (or single) quotes_ (\"\"\"). 
Python supports Unicode out of the box, using the syntax u"This is a unicode string". 
To fill a string with values, you use the % (modulo) operator and a tuple. 
Each %s gets replaced with an item from the tuple, left to right, and you can also use 
dictionary substitutions, like so:

>>>print "Name: %s\
Number: %s\
String: %s" % (myclass.name, 3, 3 * "-")

Name: Poromenos
Number: 3
String: ---

strString = \"\"\"This is
a multiline
string.\"\"\"

# WARNING: Watch out for the trailing s in "%(key)s".
>>> print "This %(verb)s a %(noun)s." % {"noun": "test", "verb": "is"}
This is a test.


FLOW CONTROL STATEMENTS

Flow control statements are if, for, and while. There is no select; instead, use if. 
Use for to enumerate through members of a list. To obtain a list of numbers, 
use range(<number>). These statements' syntax is thus:

rangelist = range(10)
>>> print rangelist
[0, 1, 2, 3, 4, 5, 6, 7, 8, 9]

>>> for number in rangelist:
        # Check if number is one of
        # the numbers in the tuple.
        if number in (3, 4, 7, 9):
            # "Break" terminates a for without
            # executing the "else" clause.
            break
        else:
            # "Continue" starts the next iteration
            # of the loop. It's rather useless here,
            # as it's the last statement of the loop.
            continue
    else:
        # The "else" clause is optional and is
        # executed only if the loop didn't "break".
        pass # Do nothing

>>> if rangelist[1] == 2:
        print "The second item (lists are 0-based) is 2"
    elif rangelist[1] == 3:
        print "The second item (lists are 0-based) is 3"
    else:
        print "Dunno"

>>> while rangelist[1] == 1:
        pass


FUNCTIONS

Functions are declared with the "def" keyword. Optional arguments are set in 
the function declaration after the mandatory arguments by being assigned a default 
value. For named arguments, the name of the argument is assigned a value. 
Functions can return a tuple (and using tuple unpacking you can effectively return 
multiple values). Lambda functions are ad hoc functions that are comprised of 
a single statement. Parameters are passed by reference, but immutable types (tuples, 
ints, strings, etc) *cannot be changed*. This is because only the memory location of 
the item is passed, and binding another object to a variable discards the old one, 
so immutable types are replaced. For example:

# Same as def funcvar(x): return x + 1
>>> funcvar = lambda x: x + 1
>>> print funcvar(1)
2

# an_int and a_string are optional, they have default values
# if one is not passed (2 and "A default string", respectively).
>>> def passing_example(a_list, an_int=2, a_string="A default string"):
        a_list.append("A new item")
        an_int = 4
        return a_list, an_int, a_string

>>> my_list = [1, 2, 3]
>>> my_int = 10
>>> print passing_example(my_list, my_int)
([1, 2, 3, 'A new item'], 4, "A default string")

>>> my_list
[1, 2, 3, 'A new item']

>>> my_int
10


CLASSES

Python supports a limited form of multiple inheritance in classes. 
Private variables and methods can be declared (by convention, this is not enforced 
by the language) by adding at least two leading underscores and at most one trailing 
one (e.g. "__spam"). We can also bind arbitrary names to class instances. 
An example follows:

>>> class MyClass(object):
        common = 10
        def __init__(self):
            self.myvariable = 3
        def myfunction(self, arg1, arg2):
            return self.myvariable

# This is the class instantiation
>>> classinstance = MyClass()
>>> classinstance.myfunction(1, 2)
3

# This variable is shared by all classes.
>>> classinstance2 = MyClass()
>>> classinstance.common
10

>>> classinstance2.common
10

# Note how we use the class name
# instead of the instance.
>>> MyClass.common = 30
>>> classinstance.common
30

>>> classinstance2.common
30

# This will not update the variable on the class,
# instead it will bind a new object to the old
# variable name.
>>> classinstance.common = 10
>>> classinstance.common
10

>>> classinstance2.common
30

>>> MyClass.common = 50
# This has not changed, because "common" is
# now an instance variable.
>>> classinstance.common
10

>>> classinstance2.common
50

# This class inherits from MyClass. The example
# class above inherits from "object", which makes
# it what's called a "new-style class".
# Multiple inheritance is declared as:
# class OtherClass(MyClass1, MyClass2, MyClassN)
>>> class OtherClass(MyClass):
        # The "self" argument is passed automatically
        # and refers to the class instance, so you can set
        # instance variables as above, but from inside the class.
        def __init__(self, arg1):
            self.myvariable = 3
            print arg1

>>> classinstance = OtherClass("hello")
hello

>>> classinstance.myfunction(1, 2)
3

# This class doesn't have a .test member, but
# we can add one to the instance anyway. Note
# that this will only be a member of classinstance.
>>> classinstance.test = 10
>>> classinstance.test
10


EXCEPTIONS

Exceptions in Python are handled with try-except [exceptionname] blocks:

>>> def some_function():
        try:
            # Division by zero raises an exception
            10 / 0
        except ZeroDivisionError:
            print "Oops, invalid."
        else:
            # Exception didn't occur, we're good.
            pass
        finally:
            # This is executed after the code block is run
            # and all exceptions have been handled, even
            # if a new exception is raised while handling.
            print "We're done with that."

>>> some_function()
Oops, invalid.

We're done with that.


IMPORTING:

External libraries are used with the import [libname] keyword. 
You can also use from [libname] import [funcname] for individual functions. 
Here is an example:

>>> import random
>>> from time import clock

>>> randomint = random.randint(1, 100)
>>> print randomint
64


FILE I/O

Python has a wide array of libraries built in. As an example, here is how serializing 
(converting data structures to strings using the pickle library) with file I/O is used:

>>> import pickle
>>> mylist = ["This", "is", 4, 13327]
# Open the file C:\\binary.dat for writing. The letter r before the
# filename string is used to prevent backslash escaping.
>>> yfile = open(r"C:\\binary.dat", "w")
>>> pickle.dump(mylist, myfile)
>>> myfile.close()

>>> myfile = open(r"C:\\text.txt", "w")
>>> myfile.write("This is a sample string")
>>> myfile.close()

>>> myfile = open(r"C:\\text.txt")
>>> print myfile.read()
'This is a sample string'

>>> myfile.close()

# Open the file for reading.
>>> myfile = open(r"C:\\binary.dat")
>>> loadedlist = pickle.load(myfile)
>>> myfile.close()
>>> print loadedlist
['This', 'is', 4, 13327]


MISCELLANEOUS

    -> Conditions can be chained. 1 < a < 3 checks 
       that a is both less than 3 and greater than 1.
    -> You can use del to delete variables or items in arrays.
    -> List comprehensions provide a powerful way to create 
       and manipulate lists. They consist of an expression 
       followed by a for clause followed by zero or more 
       if or for clauses, like so:

>>> lst1 = [1, 2, 3]
>>> lst2 = [3, 4, 5]
>>> print [x * y for x in lst1 for y in lst2]
[3, 4, 5, 6, 8, 10, 9, 12, 15]

>>> print [x for x in lst1 if 4 > x > 1]
[2, 3]

# Check if a condition is true for any items.
# "any" returns true if any item in the list is true.
>>> any([i % 3 for i in [3, 3, 4, 4, 3]])
True

# This is because 4 % 3 = 1, and 1 is true, so any()
# returns True.

# Check for how many items a condition is true.
>>> sum(1 for i in [3, 3, 4, 4, 3] if i == 4)
2

>>> del lst1[0]
>>> print lst1
[2, 3]

>>> del lst1



    -> Global variables are declared outside of functions 
       and can be read without any special declarations, 
       but if you want to write to them you must declare them 
       at the beginning of the function with the "global" keyword, 
       otherwise Python will bind that object to a new local 
       variable (be careful of that, it's a small catch that can 
       get you if you don't know it). For example:

>>> number = 5

>>> def myfunc():
        # This will print 5.
        print number

>>> def anotherfunc():
        # This raises an exception because the variable has not
        # been bound before printing. Python knows that it an
        # object will be bound to it later and creates a new, local
        # object instead of accessing the global one.
        print number
        number = 3

>>> def yetanotherfunc():
        global number
        # This will correctly change the global.
        number = 3


EPILOGUE

This tutorial is not meant to be an exhaustive list of all (or even a subset) of Python. 
Python has a vast array of libraries and much much more functionality which you will 
have to discover through other means, such as the excellent book Dive into Python. 
I hope I have made your transition in Python easier. Please leave comments if you believe 
there is something that could be improved or added or if there is anything else 
you would like to see (classes, error handling, anything). 

"""
def print_python_tutorial():
    pydoc.pager(python_tutorial_text)

data_tutorial_text = \
"""

ACCESS YOUR DATA!

Welcome to tutorial on retrieving and processing the data from HITRANonline.


  ///////////////
 /// PREFACE ///
///////////////

HITRANonline API is a set of routines in Python which is aimed to 
provide a remote access to functionality and data given by a new project 
HITRANonline (http://hitranazure.cloudapp.net).

At the present moment the API can download, filter and process data on 
molecular and atomic line-by-line spectra which is provided by HITRANonline portal.

One of the major purposes of introducing API is extending a functionality 
of the main site, particularly providing a possibility to calculate several 
types of high- and low-resolution spectra based on a flexible HT lineshape. 

Each feature of API is represented by a Python function with a set of parameters 
providing a flexible approach to the task.


  ///////////////////////
 /// FEATURE SUMMARY ///
///////////////////////

1) Downloading line-by-line data from the HITRANonline site to local database.
2) Filtering and processing the data in SQL-like fashion.
3) Conventional Python structures (lists, tuples, dictionaries) for representing 
   a spectroscopic data.
4) Possibility to use a large set of third-party Python libraries to work with a data
5) Python implementation of an HT (Hartmann-Tran [1]) lineshape which is used in spectra.
   simulations. This lineshape can also be reduced to a number of conventional 
   line profiles such as Gaussian (Doppler), Lorentzian, Voigt, Rautian, 
   Speed-dependent Voigt and Rautian.
6) Python implementation of total internal partition sums (TIPS-2011 [2]) 
   which is used in spectra simulations.
7) High-resolution spectra simulation accounting pressure, 
   temperature and optical path length. The following spectral functions 
   can be calculated:
      a) absorption coefficient
      b) absorption spectrum
      c) transmittance spectrum
      d) radiance spectrum
8) Low-resolution spectra simulation using a number of apparatus functions.
9) Possibility to extend with the user's functionality by adding custom lineshapes, 
   partitions sums and apparatus functions.

References:

[1] N.H. Ngo, D. Lisak, H. Tran, J.-M. Hartmann.
    An isolated line-shape model to go beyond the Voigt profile in 
    spectroscopic databases and radiative transfer codes.
    JQSRT, Volume 129, November 2013, Pages 89–100
    http://dx.doi.org/10.1016/j.jqsrt.2013.05.034

[2] A. L. Laraia, R. R. Gamache, J. Lamouroux, I. E. Gordon, L. S. Rothman.
    Total internal partition sums to support planetary remote sensing.
    Icarus, Volume 215, Issue 1, September 2011, Pages 391–400
    http://dx.doi.org/10.1016/j.icarus.2011.06.004

_______________________________________________________________________


This tutorial will give you an insight of how to use HAPI for Python.

First, let's choose a folder for our local database. Every time you start
your Python project, you have to specify explicitly the name of the 
database folder.

>>> db_begin('data')

So, let's download some data from the server and do some processing on it.
Suppose that we want to get line by line data on the main isotopologue of H2O.

For retrieving the data to the local database, user have to specify the following parameters:
1) Name of the local table which will store the downloaded data.
2) Either a pair of molecule and isotopologue HITRAN numbers (M and I), 
   or a "global" isotopologue ID (iso_id).
3) Wavenumber range (nu_min and nu_max)

N.B. If you specify the name which already exists in the database, 
the existing table with that name will be overrided. 

To get additional information on function fetch,
call getHelp:

>>> getHelp(fetch)
...

To download the data, simply call the function "fetch".
This will establish a connection with the main server and get the data using
the parameters listed above:

>>> fetch('H2O',1,1,3400,4100)
BEGIN DOWNLOAD: H2O
  65536 bytes written to data/H2O.data
  65536 bytes written to data/H2O.data
  65536 bytes written to data/H2O.data
...
  65536 bytes written to data/H2O.data
  65536 bytes written to data/H2O.data
  65536 bytes written to data/H2O.data
Header written to data/H2O.header
END DOWNLOAD
                     Lines parsed: 7524
PROCESSED

The output is shown right after the console line ">>>".
To check the file that you've just downloaded you can open the database
folder. The new plain text file should have a name "H2O.data" and
it should contain line-by-line data in HITRAN format.

N.B. If we want several isotopologues in one table, we should
use fetch_by_ids instead of just fetch. Fetch_by_ids takes a "global" 
isotopologue ID numbers as an input instead of HITRAN's "local" identification.
See getHelp(fetch_by_ids) to get more information on this.

To get a list of tables which are already in the database,
use tableList() function (it takes no arguments):
>>> tableList()

To learn about the table we just downloaded, let's use a function "describeTable".

>>> describeTable('H2O')
-----------------------------------------
H2O summary:
-----------------------------------------
Comment: 
Contains lines for H2(16O)
 in 3400.000-4100.000 wavenumber range
Number of rows: 7524
Table type: column-fixed
-----------------------------------------
            PAR_NAME           PAR_FORMAT

            molec_id                  %2d
        local_iso_id                  %1d
                  nu               %12.6f
                  sw               %10.3E
                   a               %10.3E
           gamma_air                %5.4f
          gamma_self                %5.3f
              elower               %10.4f
               n_air                %4.2f
           delta_air                %8.6f
 global_upper_quanta                 %15s
 global_lower_quanta                 %15s
  local_upper_quanta                 %15s
  local_lower_quanta                 %15s
                ierr                  %6s
                iref                 %12s
    line_mixing_flag                  %1s
                  gp                %7.1f
                 gpp                %7.1f
-----------------------------------------

This output tells how many rows are currenty in the table H2O, which 
wavenumber range was used by fetch(). Also this gives a basic information 
about parameters stored in the table.

So, having the table downloaded, one can perform different operations on it
using API.

Here is a list of operations currently available with API:
1) FILTERING 
2) OUTPUTTING
3) SORTING
4) GROUPING


  ////////////////////////////////
 /// FILTERING AND OUTPUTTING ///
////////////////////////////////

The table data can be filtered with the help of select() function.

Use simple select() call to output the table content:

>>> select('H2O')
MI          nu         S         A gair gsel        E_nair    dair  ...
11 1000.288940 1.957E-24 2.335E-02.07100.350 1813.22270.680.008260  ...
11 1000.532321 2.190E-28 1.305E-05.04630.281 2144.04590.39-.011030  ...
...

This will display the list of line parameters containing in the table "H2O".

That's the simplest way of using the function select(). Full information
on control parameters can be obtained via getHelp(select) statement.

Suppose that we need a lines from a table within some wavenumber range. 
That's what filtering is for. Let's apply a simple range filter on a table.

>>> select('H2O',Conditions=('between','nu',4000,4100))
MI          nu         S         A gair gsel        E_nair    dair     
 11 4000.188800 1.513E-25 1.105E-02.03340.298 1581.33570.51-.013910 ...
 11 4000.204070 3.482E-24 8.479E-03.08600.454  586.47920.61-.007000 ...
 11 4000.469910 3.268E-23 1.627E+00.05410.375 1255.91150.56-.013050 ...
......

As a result of this operation, we see a list of lines of H2O table,
whose wavenumbers lie between 4000 cm-1 and 4100 cm-1.
The condition is taken as an input parameter to API function "select".

To specify a subset of columns to display, use another control parameter - 
ParameterNames:

>>> select('H2O',ParameterNames=('nu','sw'),Conditions=('between','nu',4000,4100))

The usage of ParameterNames is outlined below in the section "Specifying a list 
of parameters". So far it worth mentioning that this parameter is a part 
of a powerful tool for displaying and processing tables from database.

In the next section we will show how to create quieries 
with more complex conditions.


  ////////////////////////////
 /// FILTERING CONDITIONS ///
////////////////////////////

Let's analyze the last example of filtering. Condition input variable is
as follows:

                    ('between','nu',4000,4100)

Thus, this is a python list (or tuple), containing logical expressions
defined under column names of the table. For example, 'nu' is a name of 
the column in 'H2O' table, and this column contains a transition wavenumber.
The structure of a simple condition is as follows:

                    (OPERATION,ARG1,ARG2,...)
                    
Where OPERATION must be in a set of predefined operations (see below),
and ARG1,ARG2 etc. are the arguments for this operation.
Conditions can be nested, i.e. ARG can itself be a condition (see examples).
The following operations are available in select (case insensitive):


DESCRIPTION                   LITERAL                     EXAMPLE
---------------------------------------------------------------------------------
Range:               'RANGE','BETWEEN':         ('BETWEEN','nu',0,1000)
Subset:              'IN','SUBSET':             ('IN','local_iso_id',[1,2,3,4])
And:                 '&','&&','AND':            ('AND',('<','nu',1000),('>','nu',10))
Or:                  '|','||','OR':             ('OR',('>','nu',1000),('<','nu',10))
Not:                 '!','NOT':                 ('NOT',('IN','local_iso_id',[1,2,3]))
Less than:           '<','LESS','LT':                 ('<','nu',1000)
More than:           '>','MORE','MT':                 ('>','sw',1.0e-20)
Less or equal than:  '<=','LESSOREQUAL','LTE':        ('<=','local_iso_id',10)
More or equal than   '>=','MOREOREQUAL','MTE':        ('>=','sw',1e-20)
Equal:               '=','==','EQ','EQUAL','EQUALS':  ('<=','local_iso_id',10)
Not equal:           '!=','<>','~=','NE','NOTEQUAL':  ('!=','local_iso_id',1)
Summation:           '+','SUM':                 ('+','v1','v2','v3')
Difference:          '-','DIFF':                ('-','nu','elow')
Multiplication:      '*','MUL':                 ('*','sw',0.98)
Division:            '/','DIV':                 ('/','A',2)
Cast to string:      'STR','STRING':            ('STR','some_string')
Cast to Python list  'LIST':                    ('LIST',[1,2,3,4,5])
Match regexp         'MATCH','LIKE':            ('MATCH','\w+','some string')
Search single match: 'SEARCH':                  ('SEARCH','\d \d \d','1 2 3 4')
Search all matches:  'FINDALL':                 ('FINDALL','\d','1 2 3 4 5')
Count within group:  'COUNT' :                  ('COUNT','local_iso_id')
---------------------------------------------------------------------------------
   
Let's create a query with more complex condition. Suppese that we are 
interested in all lines between 3500 and 4000 with 1e-19 intensity cutoff.
The query will look like this:

>>> Cond = ('AND',('BETWEEN','nu',3500,4000),('>=','Sw',1e-19))
>>> select('H2O',Conditions=Cond,DestinationTableName='tmp')

Here, apart from other parameters, we have used a new parameter 
DestinationTableName. This parameter contains a name of the table
where we want to put a result of the query. Thus we have chosen 
a name 'tmp' for a new table.


  ////////////////////////////////////
 /// ACCESSING COLUMNS IN A TABLE ///
////////////////////////////////////

To get an access to particular table column (or columns) all we need
is to get a column from a table and put it to Python variable.

For this purpose, there exist two functions:

  getColumn(...)
  getColumns(...)

The first one returns just one column at a time. The second one returns
a list of solumns.

So, here are some examples of how to use both:

>>> nu1 = getColumn('H2O','nu')
>>> nu2,sw2 = getColumns('H2O',['nu','sw'])

N.B. If you don't remember exact names of columns in a particular table,
use describeTable to get an info on it's structure!


  ///////////////////////////////////////
 /// SPECIFYING A LIST OF PARAMETERS ///
///////////////////////////////////////

Suppose that we want not only select a set of parameters/columns
from a table, but do a certain transformations with them (for example,
multiply column on a coefficient, or add one column to another etc...).
We can make it in two ways. First, we can extract a column from table
using one of the functions (getColumn or getColumns) and do the rest 
in Python. The second way is to do it on the level of select.
The select function has a control parameter "ParameterNames", which 
makes it possible to specify parameters we want to be selected, 
and evaluate some simple arithmetic expressions with them.

Assume that we need only wavenumber and intensity from H2O table.
Also we need to scale an intensity to the unitary abundance. To do so,
we must divide an 'sw' parameter by it's natural abundance (0.99731) for 
principal isotopologue of water).

Thus, we have to select two columns:  
wavenumber (nu) and scaled intensity (sw/0.99731)
>>> select('H2O',)


  ////////////////////////////
 /// SAVING QUERY TO DISK ///
////////////////////////////

To quickly save a result of a query to disk, the user can take an 
advantage of an additional parameter "File".
If this parameter is presented in function call, then the query is 
saved to file with the name which was specified in "File".

For example, select all lines from H2O and save the result in file 'H2O.txt':
>>> select('H2O',File='H2O.txt')


  ////////////////////////////////////////////
 /// GETTING INFORMATION ON ISOTOPOLOGUES ///
////////////////////////////////////////////

API provides the following auxillary information about isotopologues
present in HITRAN. Corresponding functions use the standard HITRAN
molecule-isotopologue notation:

1) Natural abundances
>>> abundance(mol_id,iso_id)

2) Molecular masses
>>> molecularMass(mol_id,iso_id)

3) Molecule names
>>> moleculeName(mol_id,iso_id)

4) Isotopologue names
>>> isotopologueName(mol_id,iso_id)

5) ISO_ID
>>> getHelp(ISO_ID)

The latter is a dictionary, which contain all information about 
isotopologues concentrated in one place.

"""
def print_data_tutorial():
    pydoc.pager(data_tutorial_text)

spectra_tutorial_text = \
"""

CALCULATE YOUR SPECTRA!

Welcome to tutorial on calculating a spectra from line-by-line data.


  ///////////////
 /// PREFACE ///
///////////////

This tutorial will demonstrate how to use different lineshapes and partition
functions, and how to calculate synthetic spectra with respect to different 
instruments. It will be shown how to combine different parameters of spectral 
calculation to achieve better precision and performance for cross sections.

API provides a powerful tool to calculate cross-sections based on line-by-line
data containing in HITRAN. This features:

*) Python implementation of an HT (Hartmann-Tran [1]) lineshape which is used in 
   spectra simulations. This lineshape can also be reduced to a number of 
   conventional    line profiles such as Gaussian (Doppler), Lorentzian, Voigt, 
   Rautian, Speed-dependent Voigt and Rautian.
*) Python implementation of total internal partition sums (TIPS-2011 [2]) 
   which is used in spectra simulations.
*) High-resolution spectra simulation accounting pressure, 
   temperature and optical path length. The following spectral functions 
   can be calculated:
      a) absorption coefficient
      b) absorption spectrum
      c) transmittance spectrum
      d) radiance spectrum
*) Low-resolution spectra simulation using a number of apparatus functions.
*) Possibility to extend with the user's functionality by adding custom lineshapes, 
   partitions sums and apparatus functions.
*) An approach to function code is aimed to be flexible enough yet hopefully 
   intuitive.

References:

[1] N.H. Ngo, D. Lisak, H. Tran, J.-M. Hartmann.
    An isolated line-shape model to go beyond the Voigt profile in 
    spectroscopic databases and radiative transfer codes.
    JQSRT, Volume 129, November 2013, Pages 89–100
    http://dx.doi.org/10.1016/j.jqsrt.2013.05.034

[2] A. L. Laraia, R. R. Gamache, J. Lamouroux, I. E. Gordon, L. S. Rothman.
    Total internal partition sums to support planetary remote sensing.
    Icarus, Volume 215, Issue 1, September 2011, Pages 391–400
    http://dx.doi.org/10.1016/j.icarus.2011.06.004

            
  ///////////////////////////
 /// USING LINE PROFILES ///
///////////////////////////

Several lineshape (line profile) families are currently available:
1) Gaussian (Doppler) profile
2) Lorentzian profile
3) Voigt profile
4) Speed-dependent Voigt profile
5) Rautian profile
6) Speed-dependent Rautian profile
7) HT profile (Hartmann-Tran)

Each profile has it's own uniwue set of parameters. Normally one should
use profile parameters only in conjunction with their "native" profiles.

So, let's start exploring the available profiles using getHelp:
>>> getHelp(profiles)
Profiles available:
  HTP       : PROFILE_HT
  SDRautian : PROFILE_SDRAUTIAN
  Rautian   : PROFILE_RAUTIAN
  SDVoigt   : PROFILE_SDVOIGT
  Voigt     : PROFILE_VOIGT
  Lorentz   : PROFILE_LORENTZ
  Doppler   : PROFILE_DOPPLER

Output gives all available profiles. We can get additional info on each
of them just by calling getHelp(ProfileName):
>>> getHelp(PROFILE_HT)

Line profiles, adapted for using with HAPI, are written in Python and
heavily using the numerical library "Numpy". This means that the user
can calculate multiple values of particular profile at once having just
pasted a numpy array as a wavenumber grid (array). Let's give a short 
example of how to calculate HT profile on a numpy array.

>>> from numpy import arange
    w0 = 1000.
    GammaD = 0.005
    Gamma0 = 0.2
    Gamma2 = 0.01 * Gamma0
    Delta0 = 0.002
    Delta2 = 0.001 * Delta0
    nuVC = 0.2
    eta = 0.5
    Dw = 1.
    ww = arange(w0-Dw, w0+Dw, 0.01)  # GRID WITH THE STEP 0.01 
    l1 = PROFILE_HT(w0,GammaD,Gamma0,Gamma2,Delta0,Delta2,nuVC,eta,ww)[0]
    # now l1 contains values of HT profile calculates on the grid ww
    
On additional information about parameters see getHelp(PROFILE_HT).

It worth noting that PROFILE_HT returns 2 entities: real and imaginary part
of lineshape (as it described in the article given in preface). Apart from
HT, all other profiles return just one entity (the real part).


  ////////////////////////////
 /// USING PARTITION SUMS ///
////////////////////////////

As it was mentioned in the preface to this tutorial, the partition sums
are taken from the TIPS-2011 (the link is given above). Partition sums 
are taken for those isotopologues, which are present in HITRAN and in
TIPS-2011 simultaneousely.

N.B. Partition sums are omitted for the following isotopologues which
are in HITRAN at the moment:

ID       M     I         ISO                MOL
--------------------------------------------------
117      12    2     H(15N)(16O)3           HNO3
110      14    2     D(19F)                 HF
107      15    3     D(35Cl)                HCl
108      15    4     D(37Cl)                HCl
111      16    3     D(79Br)                HBr
112      16    4     D(81Br)                HBr
113      17    2     D(127I)                HI
118      22    2     (14N)(15N)             N2
119      29    2     (13C)(16O)(19F)2       COF2
 86      34    1     (16O)                  O
 92      39    1     (12C)H3(16O)H          CH3OH
114      47    1     (32S)(16O)3            SO3
--------------------------------------------------

The data on these isotopologues is not present in TIPS-2011 but is 
present in HITRAN. We're planning to add these molecules after TIPS-2013
is released.

To calculate a partition sum for most of the isotopologues in HITRAN,
we will use a function partitionSum (use getHelp for detailed info).
Let's just mention that 
The syntax is as follows: partitionSum(M,I,T), where M,I - standard 
HITRAN molecule-isotopologue notation, T - definition of temperature
range.

Usecase 1: temperatuer is defined by a list:
>>> Q = partitionSum(1,1,[70,80,90])

Usecase 2: temperature is defined by bounds and the step:
>>> T,Q = partiionSum(1,1,[70,3000],step=1.0)

In the latter example we calculate a partition sum on a range of
temperatures from 70K to 3000K using a step 1.0 K, and having arrays 
of temperature (T) and partition sum (Q) at the output.


  ///////////////////////////////////////////
 /// CALCULATING ABSORPTION COEFFICIENTS ///
///////////////////////////////////////////

Currently API can calculate the following spectral function at arbitrary
thermodynamic parameters:

1) Absorption coefficient
2) Absorption spectrum
3) Transmittance spectrum
4) Radiance spectrum

All these functions can be calculated with or without accounting of 
an instrument properties (apparatus function, resolution, path length etc...)

As it well known, the spectral functions such as absorption,
transmittance, and radiance spectra, are calculated on the basis
of the absorption coefficient. By that resaon, absorption coefficient
is the most important part of simulating a cross section. This part of
tutorial is devoted to demonstration how to calculate absorption 
coefficient from the HITRAN line-by-line data. Here we give a brief 
insight on basic parameters of calculation procedure, talk about some 
useful practices and precautions.

To calculate an absorption coefficient, we can use one of the following
functions:

-> absorptionCoefficient_HT
-> absorptionCoefficient_Voigt
-> absorptionCoefficient_Lorentz
-> absorptionCoefficient_Doppler

Each of these function calculates cross sections using different
lineshapes (the names a quite self-explanatory).
You can get detailed information on using each of these functions
by calling getHelp(function_name).

Let's look more closely to the cross sections based on the Lorentz profile.
For doing that, let's have a table downloaded from HITRANonline.

# get data on CO2 main isotopologue in the range 2000-2100 cm-1
>>> fetch('CO2',2,1,2000,2100)

OK, now we're ready to run a fast example of how to calculate an
absorption coefficient cross section:

>>> nu,coef = absorptionCoefficient_Lorentz(SourceTables='CO2')

This example calculates a Lorentz cross section using the whole set of 
lines in the "co2" table. This is the simplest possible way to use these
functions, because major part of parameters bound to their default values.

If we have matplotlib installed, then we can visualize it using a plotter:
>>> from pylab import plot
>>> plot(nu,coef) 

API provides a flexible control over a calculation procedure. This control
can be achieved by using a number of input parameters. So, let's dig 
into the depth of the settings.

The input parameters of absorptionCoefficient_Lorentz are as follows:

Name                          Default value
-------------------------------------------------------------------
SourceTables                  '__BUFFER__'
Components                    All isotopologues in SourceTables 
partitionFunction             PYTIPS
Environment                   {'T':296.,'p':1.}
WavenumberRange               depends on Components
WavenumberStep                0.01 cm-1
WavenumberWing                10 cm-1
WavenumberWingHW              50 HWHMs
IntensityThreshold            0 cm/molec
GammaL                        'gamma_air'
HITRAN_units                  True 
File                          None
Format                        '%e %e'
-------------------------------------------------------------------

Newt we'll give a brief explanation for each parameter. After each description
we'll make some notes about the usage of the correspondent parameter.


SourceTables:     (required parameter)
   
  List of source tables to take line-by-line data from.
  NOTE: User must provide at least one table in the list.

Components:    (optional parameter)

  List of tuples (M,I,D) to consider in cross section calculation.
  M here is a molecule number, I is an isotopologue number, 
  D is an abundance of the component.
  NOTE: If this input contains more than one tuple, then the output 
        is an absorption coefficient for mixture of corresponding gases.
  NOTE2: If omitted, then all data from the source tables is involved.

partitionFunction:    (optional parameter)

  Instance of partition function of the following format:
  Func(M,I,T), where Func - numae of function, (M,I) - HITRAN numbers
  for molecule and isotopologue, T - temperature.
  Function must return only one output - value of partition sum.
  NOTE: Deafult value is PYTIPS - python version of TIPS-2011

Environment:    (optional parameter)

  Python dictionary containing value of pressure and temperature.
  The format is as follows: Environment = {'p':pval,'T':tval}, 
  where "pval" and "tval" are corresponding values in atm and K 
  respectively.
  NOTE: Default value is {'p':1.0,'T':296.0}

WavenumberRange:    (optional parameter)

  List containing minimum and maximum value of wavenumber to consider
  in cross-section calculation. All lines that are out of htese bounds
  will be skipped. The firmat is as follows: WavenumberRange=[wn_low,wn_high]
  NOTE: If this parameter os skipped, then min and max are taken 
  from the data from SourceTables. Deprecated name is OmegaRange.

WavenumberStep:    (optional parameter)

  Value for the wavenumber step. 
  NOTE: Default value is 0.01 cm-1.
  NOTE2: Normally user would want to take the step under 0.001 when
         calculating absorption coefficient with Doppler profile 
         because of very narrow spectral lines. Deprecated name is OmegaStep.

WavenumberWing:    (optional parameter)

  Absolute value of the line wing in cm-1, i.e. distance from the center 
  of each line to the most far point where the profile is considered 
  to be non zero. Deprecated name is OmegaStep.
  NOTE: if omitted, then only OmegaWingHW is taken into account.

WavenumberWingHW:    (optional parameter)

  Relative value of the line wing in halfwidths. Deprecated name is OmegaWingHW.
  NOTE: The resulting wing is a maximum value from both OmegaWing and
  OmegaWingHW.

IntensityThreshold:    (optional parameter)

  Absolute value of minimum intensity in cm/molec to consider.
  NOTE: default value is 0.

GammaL:    (optional parameter)

  This is the name of broadening parameter to consider a "Lorentzian"
  part in the Voigt profile. In the current 160-char format there is 
  a choise between "gamma_air" and "gamma_self".
  NOTE: If the table has custom columns with a broadening coefficients,
        the user can specify the name of this column in GammaL. This
        would let the function calculate an absorption with custom
        broadening parameter.

HITRAN_units:    (optional parameter)

  Logical flag for units, in which the absorption coefficient shoould be 
  calculated. Currently, the choises are: cm^2/molec (if True) and
  cm-1 (if False).
  NOTE: to calculate other spectral functions like transmitance,
  radiance and absorption spectra, user should set HITRAN_units to False.

File:    (optional parameter)

  The name of the file to save the calculated absorption coefficient.
  The file is saved only if this parameter is specified.

Format:    (optional parameter)

  C-style format for the text data to be saved. Default value is "%e %e".
  NOTE: C-style output format specification (which are mostly valid for Python) 
        can be found, for instance, by the link: 
  http://www.gnu.org/software/libc/manual/html_node/Formatted-Output.html


N.B. Other functions such as absorptionCoefficient_Voigt(_HT,_Doppler) have
identical parameter sets so the description is the same for each function.


  ///////////////////////////////////////////////////////////////////
 /// CALCULATING ABSORPTION, TRANSMITTANCE, AND RADIANCE SPECTRA ///
///////////////////////////////////////////////////////////////////

Let's calculate an absorption, transmittance, and radiance
spectra on the basis of apsorption coefficient. In order to be consistent
with internal API's units, we need to have an absorption coefficient cm-1:

>>> nu,coef = absorptionCoefficient_Lorentz(SourceTables='CO2',HITRAN_units=False)

To calculate absorption spectrum, use the function absorptionSpectrum():
>>> nu,absorp = absorptionSpectrum(nu,coef) 

To calculate transmittance spectrum, use function transmittanceSpectrum():
>>> nu,trans = transmittanceSpectrum(nu,coef) 

To calculate radiance spectrum, use function radianceSpectrum():
>>> nu,radi = radianceSpectrum(nu,coef) 


The last three commands used a default path length (1 m).
To see complete info on all three functions, look for section
"calculating spectra" in getHelp()

Generally, all these three functions use similar set of parameters:

Wavenumber:       (required parameter) 

  Wavenumber grid to for spectrum. Deprecated name is Omegas.

AbsorptionCoefficient        (optional parameter)

  Absorption coefficient as input.

Environment={'T': 296.0, 'l': 100.0}       (optional parameter) 

  Environmental parameters for calculating  spectrum.
  This parameter is a bit specific for each of functions:
  For absorptionSpectrum() and transmittanceSpectrum() the default
  value is as follows: Environment={'l': 100.0}
  For transmittanceSpectrum() the default value, besides path length,
  contains a temperature: Environment={'T': 296.0, 'l': 100.0}
  NOTE: temperature must be equal to that which was used in 
  absorptionCoefficient_ routine!

File         (optional parameter)

  Filename of output file for calculated spectrum.
  If omitted, then the file is not created.

Format        (optional parameter)

  C-style format for spectra output file.
  NOTE: Default value is as follows: Format='%e %e'


  ///////////////////////////////////////
 /// APPLYING INSTRUMENTAL FUNCTIONS ///
///////////////////////////////////////

For comparison of the theoretical spectra with the real-world 
instruments output it's necessary to take into account instrumental resolution.
For this purpose HAPI has a function convolveSpectrum() which can emulate
spectra with lower resolution using custom instrumental functions.

The following instrumental functions are available:
1) Rectangular
2) Triangular
3) Gaussian
4) Diffraction
5) Michelson
6) Dispersion
7) Lorentz

To get a description of each instrumental function we can use getHelp():
>>> getHelp(slit_functions)
  RECTANGULAR : SLIT_RECTANGULAR
  TRIANGULAR  : SLIT_TRIANGULAR
  GAUSSIAN    : SLIT_GAUSSIAN
  DIFFRACTION : SLIT_DIFFRACTION
  MICHELSON   : SLIT_MICHELSON
  DISPERSION/LORENTZ : SLIT_DISPERSION
  
For instance,
>>> getHelp(SLIT_MICHELSON)
... will give a datailed info about Michelson's instrumental function.


The function convolveSpectrum() convolutes a high-resulution spectrum
with one of supplied instrumental (slit) functions. The folowing 
parameters of this function are provided:

Wavenumber     (required parameter)
  
  Array of wavenumbers in high-resolution input spectrum.
  Deprecated name is Omega.

CrossSection     (required parameter)

  Values of high-resolution input spectrum.

Resolution     (optional parameter)

  This parameter is passed to the slit function. It represents
  the resolution of corresponding instrument.
  NOTE: default value is 0.1 cm-1

AF_wing     (optional parameter)

  Width of an instrument function where it is considered non-zero.
  NOTE: default value is 10.0 cm-1

SlitFunction     (optional parameter)

  Custom instrumental function to convolve with spectrum.
  Format of the instrumental function must be as follows:
  Func(x,g), where Func - function name, x - wavenumber,
  g - resolution.
  NOTE: if omitted, then the default value is SLIT_RECTANGULAR


Before using the convolution procedure it worth giving some practical 
advices and remarks: 
1) Quality of a convolution depends on many things: quality of calculated 
spectra, width of AF_wing and WavenumberRange, Resolution, WavenumberStep etc ...
Most of these factors are taken from previus stages of spectral calculation.
Right choise of all these factors is crucial for the correct computation.
2) Dispersion, Diffraction and Michelson AF's don't work well in narrow 
wavenumber range because of their broad wings.
3) Generally one must consider WavenumberRange and AF_wing as wide as possible.
4) After applying a convolution, the resulting spectral range for 
the lower-resolution spectra is reduced by the doubled value of AF_wing.
For this reason, try to make an initial spectral range for high-resolution
spectrum (absorption, transmittance, radiance) sufficiently broad.

The following command will calculate a lower-resolution spectra from 
the CO2 transmittance, which was calculated in a previous section. 
The Spectral resolution is 1 cm-1, 

>>> nu_,trans_,i1,i2,slit = convolveSpectrum(nu,trans)

The outputs are: 

nu_, trans_ - wavenumbers and transmittance for the resulting 
              low-resolution spectrum.

i1,i2 - indexes for initial nu,trans spectrum denoting the part of 
        wavenumber range which was taken for lower resolution spectrum.
        => Low-res spectrum is calculated on nu[i1:i2]

Note, than to achieve more flexibility, one have to specify most of 
the optional parameters. For instance, more complete call is as follows:
>>> nu_,trans_,i1,i2,slit = convolveSpectrum(nu,trans,SlitFunction=SLIT_MICHELSON,Resolution=1.0,AF_wing=20.0)

"""
def print_spectra_tutorial():
    pydoc.pager(spectra_tutorial_text)

plotting_tutorial_text = \
"""

PLOTTING THE SPECTRA WITH MATPLOTLIB

This tutorial briefly explains how to make plots using
the Matplotlib - Python library for plotting.

Prerequisites:
   To tun through this tutorial, user must have the following
   Python libraries installed:
   1) Matplotlib
       Matplotlib can be obtained by the link http://matplotlib.org/ 
   2) Numpy  (required by HAPI itself)
       Numpy can be obtained via pip:  
          sudo pip install numpy (under Linux and Mac)
          pip install numpy (under Windows)
       Or by the link http://www.numpy.org/
       
As an option, user can download one of the many scientific Python
distributions, such as Anaconda, Canopy etc...

So, let's calculate plot the basic entities which ar provided by HAPI.
To do so, we will do all necessary steps to download, filter and 
calculate cross sections "from scratch". To demonstrate the different
possibilities of matplotlib, we will mostly use Pylab - a part of 
Matplotlib with the interface similar to Matlab. Please note, that it's 
not the only way to use Matplotlib. More information can be found on it's site.

The next part is a step-by-step guide, demonstrating basic possilities
of HITRANonline API in conjunction with Matplotlib.

First, do some preliminary imports:
>>> from hapi import *
>>> from pylab import show,plot,subplot,xlim,ylim,title,legend,xlabel,ylabel,hold

Start the database 'data':
>>> db_begin('data') 

Download lines for main isotopologue of ozone in [3900,4050] range:
>>> fetch('O3',3,1,3900,4050)

PLot a sick spectrum using the function getStickXY()
>>> x,y = getStickXY('O3')
>>> plot(x,y); show()

Zoom in spectral region [4020,4035] cm-1:
>>> plot(x,y); xlim([4020,4035]); show()

Calculate and plot difference between Voigt and Lorentzian lineshape:
>>> wn = arange(3002,3008,0.01) # get wavenumber range of interest
>>> voi = PROFILE_VOIGT(3005,0.1,0.3,wn)[0]   # calc Voigt
>>> lor = PROFILE_LORENTZ(3005,0.3,wn)   # calc Lorentz
>>> diff = voi-lor    # calc difference
>>> subplot(2,1,1)   # upper panel
>>> plot(wn,voi,'red',wn,lor,'blue')  # plot both profiles
>>> legend(['Voigt','Lorentz'])   # show legend
>>> title('Voigt and Lorentz profiles')   # show title
>>> subplot(2,1,2)   # lower panel
>>> plot(wn,diff)   # plot diffenence
>>> title('Voigt-Lorentz residual')   # show title
>>> show()   # show all figures

Calculate and plot absorption coefficients for ozone using Voigt 
profile. Spectra are calculated for 4 cases of thermodynamic parameters: 
(1 atm, 296 K), (5 atm, 296 K), (1 atm, 500 K), and (5 atm, 500 K)
>>> nu1,coef1 = absorptionCoefficient_Voigt(((3,1),),'O3',
        WavenumberStep=0.01,HITRAN_units=False,GammaL='gamma_self',
        Environment={'p':1,'T':296.})
>>> nu2,coef2 = absorptionCoefficient_Voigt(((3,1),),'O3',
        WavenumberStep=0.01,HITRAN_units=False,GammaL='gamma_self',
        Environment={'p':5,'T':296.})
>>> nu3,coef3 = absorptionCoefficient_Voigt(((3,1),),'O3',
        WavenumberStep=0.01,HITRAN_units=False,GammaL='gamma_self',
        Environment={'p':1,'T':500.})
>>> nu4,coef4 = absorptionCoefficient_Voigt(((3,1),),'O3',
        WavenumberStep=0.01,HITRAN_units=False,GammaL='gamma_self',
        Environment={'p':5,'T':500.})
>>> subplot(2,2,1); plot(nu1,coef1); title('O3 k(w): p=1 atm, T=296K')
>>> subplot(2,2,2); plot(nu2,coef2); title('O3 k(w): p=5 atm, T=296K')
>>> subplot(2,2,3); plot(nu3,coef3); title('O3 k(w): p=1 atm, T=500K')
>>> subplot(2,2,4); plot(nu4,coef4); title('O3 k(w): p=5 atm, T=500K')
>>> show()

Calculate and plot absorption, transmittance and radiance spectra for 1 atm 
and 296K. Path length is set to 10 m.
>>> nu,absorp = absorptionSpectrum(nu1,coef1,Environment={'l':1000.})
>>> nu,transm = transmittanceSpectrum(nu1,coef1,Environment={'l':1000.})
>>> nu,radian = radianceSpectrum(nu1,coef1,Environment={'l':1000.,'T':296.})
>>> subplot(2,2,1); plot(nu1,coef1,'r'); title('O3 k(w): p=1 atm, T=296K')
>>> subplot(2,2,2); plot(nu,absorp,'g'); title('O3 absorption: p=1 atm, T=296K')
>>> subplot(2,2,3); plot(nu,transm,'b'); title('O3 transmittance: p=1 atm, T=296K')
>>> subplot(2,2,4); plot(nu,radian,'y'); title('O3 radiance: p=1 atm, T=296K')
>>> show()

Calculate and compare high resolution spectrum for O3 with lower resolution
spectrum convoluted with an instrumental function of ideal Michelson interferometer.
>>> nu_,trans_,i1,i2,slit = convolveSpectrum(nu,transm,SlitFunction=SLIT_MICHELSON,Resolution=1.0,AF_wing=20.0)
>>> plot(nu,transm,'red',nu_,trans_,'blue'); legend(['HI-RES','Michelson']); show()

"""
def print_plotting_tutorial():
    pydoc.pager(plotting_tutorial_text)

def getHelp(arg=None):
    """
    This function provides interactive manuals and tutorials.
    """
    if arg==None:
        print('--------------------------------------------------------------')
        print('Hello, this is an interactive help system of HITRANonline API.')
        print('--------------------------------------------------------------')
        print('Run getHelp(.) with one of the following arguments:')
        print('    tutorial  -  interactive tutorials on HAPI')
        print('    units     -  units used in calculations')
        print('    index     -  index of available HAPI functions')
    elif arg=='tutorial':
        print('-----------------------------------')
        print('This is a tutorial section of help.')
        print('-----------------------------------')
        print('Please choose the subject of tutorial:')
        print('    data      -  downloading the data and working with it')
        print('    spectra   -  calculating spectral functions')
        print('    plotting  -  visualizing data with matplotlib')
        print('    python    -  Python quick start guide')
    elif arg=='python':
        print_python_tutorial()
    elif arg=='data':
        print_data_tutorial()
    elif arg=='spectra':
        print_spectra_tutorial()
    elif arg=='plotting':
        print_plotting_tutorial()
    elif arg=='index':
        print('------------------------------')
        print('FETCHING DATA:')
        print('------------------------------')
        print('  fetch')
        print('  fetch_by_ids')
        print('')
        print('------------------------------')
        print('WORKING WITH DATA:')
        print('------------------------------')
        print('  db_begin')
        print('  db_commit')
        print('  tableList')
        print('  describe')
        print('  select')
        print('  sort')
        print('  extractColumns')
        print('  getColumn')
        print('  getColumns')
        print('  dropTable')
        print('')
        print('------------------------------')
        print('CALCULATING SPECTRA:')
        print('------------------------------')
        print('  profiles')
        print('  partitionSum')
        print('  absorptionCoefficient_HT')
        print('  absorptionCoefficient_Voigt')
        print('  absorptionCoefficient_SDVoigt')
        print('  absorptionCoefficient_Lorentz')
        print('  absorptionCoefficient_Doppler')
        print('  transmittanceSpectrum')
        print('  absorptionSpectrum')
        print('  radianceSpectrum')
        print('')
        print('------------------------------')
        print('CONVOLVING SPECTRA:')
        print('------------------------------')
        print('  convolveSpectrum')
        print('  slit_functions')
        print('')
        print('------------------------------')
        print('INFO ON ISOTOPOLOGUES:')
        print('------------------------------')
        print('  ISO_ID')
        print('  abundance')
        print('  molecularMass')
        print('  moleculeName')
        print('  isotopologueName')
        print('')
        print('------------------------------')
        print('MISCELLANEOUS:')
        print('------------------------------')
        print('  getStickXY')
        print('  read_hotw')
    elif arg == ISO:
        print_iso()
    elif arg == ISO_ID:
        print_iso_id()
    elif arg == profiles:
        print_profiles()
    elif arg == slit_functions:
        print_slit_functions()
    else:
       help(arg)

    

# Get atmospheric (natural) abundance
# for a specified isotopologue
# M - molecule number
# I - isotopologue number
def abundance(M,I):
    """
    INPUT PARAMETERS: 
        M: HITRAN molecule number
        I: HITRAN isotopologue number
    OUTPUT PARAMETERS: 
        Abbundance: natural abundance
    ---
    DESCRIPTION:
        Return natural (Earth) abundance of HITRAN isotolopogue.
    ---
    EXAMPLE OF USAGE:
        ab = abundance(1,1) # H2O
    ---
    """
    return ISO[(M,I)][ISO_INDEX['abundance']]

# Get molecular mass
# for a specified isotopologue
# M - molecule number
# I - isotopologue number
def molecularMass(M,I):
    """
    INPUT PARAMETERS: 
        M: HITRAN molecule number
        I: HITRAN isotopologue number
    OUTPUT PARAMETERS: 
        MolMass: molecular mass
    ---
    DESCRIPTION:
        Return molecular mass of HITRAN isotolopogue.
    ---
    EXAMPLE OF USAGE:
        mass = molecularMass(1,1) # H2O
    ---
    """
    return ISO[(M,I)][ISO_INDEX['mass']]

# Get molecule name
# for a specified isotopologue
# M - molecule number
# I - isotopologue number
def moleculeName(M):
    """
    INPUT PARAMETERS: 
        M: HITRAN molecule number
    OUTPUT PARAMETERS: 
        MolName: molecular name
    ---
    DESCRIPTION:
        Return name of HITRAN molecule.
    ---
    EXAMPLE OF USAGE:
        molname = moleculeName(1) # H2O
    ---
    """
    return ISO[(M,1)][ISO_INDEX['mol_name']]

# Get isotopologue name
# for a specified isotopologue
# M - molecule number
# I - isotopologue number
def isotopologueName(M,I):
    """
    INPUT PARAMETERS: 
        M: HITRAN molecule number
        I: HITRAN isotopologue number
    OUTPUT PARAMETERS: 
        IsoMass: isotopologue mass
    ---
    DESCRIPTION:
        Return name of HITRAN isotolopogue.
    ---
    EXAMPLE OF USAGE:
        isoname = isotopologueName(1,1) # H2O
    ---
    """
    return ISO[(M,I)][ISO_INDEX['iso_name']]

# ----------------------- table list ----------------------------------
def tableList():
    """
    INPUT PARAMETERS: 
        none
    OUTPUT PARAMETERS: 
        TableList: a list of available tables
    ---
    DESCRIPTION:
        Return a list of tables present in database.
    ---
    EXAMPLE OF USAGE:
        lst = tableList()
    ---
    """

    return getTableList()

# ----------------------- describe ----------------------------------
def describe(TableName):
    """
    INPUT PARAMETERS: 
        TableName: name of the table to describe
    OUTPUT PARAMETERS: 
        none
    ---
    DESCRIPTION:
        Print information about table, including 
        parameter names, formats and wavenumber range.
    ---
    EXAMPLE OF USAGE:
        describe('sampletab')
    ---
    """
    describeTable(TableName)

# ---------------------- /ISO.PY ---------------------------------------

def db_begin(db=None):
    """
    INPUT PARAMETERS: 
        db: database name (optional)
    OUTPUT PARAMETERS: 
        none
    ---
    DESCRIPTION:
        Open a database connection. A database is stored 
        in a folder given in db input parameter.
        Default=data
    ---
    EXAMPLE OF USAGE:
        db_begin('bar')
    ---
    """
    databaseBegin(db)

def db_commit():
    """
    INPUT PARAMETERS: 
        none
    OUTPUT PARAMETERS: 
        none
    ---
    DESCRIPTION:
        Commit all changes made to opened database.
        All tables will be saved in corresponding files.
    ---
    EXAMPLE OF USAGE:
        db_commit()
    ---
    """
    databaseCommit()

# ------------------ QUERY HITRAN ---------------------------------------

def comment(TableName,Comment):
    LOCAL_TABLE_CACHE[TableName]['header']['comment'] = Comment

def fetch_by_ids(TableName,iso_id_list,numin,numax,ParameterGroups=[],Parameters=[]):
    """
    INPUT PARAMETERS: 
        TableName:   local table name to fetch in (required)
        iso_id_list: list of isotopologue id's    (required)
        numin:       lower wavenumber bound       (required)
        numax:       upper wavenumber bound       (required)
    OUTPUT PARAMETERS: 
        none
    ---
    DESCRIPTION:
        Download line-by-line data from HITRANonline server
        and save it to local table. The input parameter iso_id_list
        contains list of "global" isotopologue Ids (see help on ISO_ID).
        Note: this function is required if user wants to download
        multiple species into single table.
    ---
    EXAMPLE OF USAGE:
        fetch_by_ids('water',[1,2,3,4],4000,4100)
    ---
    """
    if type(iso_id_list) not in set([list,tuple]):
       iso_id_list = [iso_id_list]
    queryHITRAN(TableName,iso_id_list,numin,numax,
                pargroups=ParameterGroups,params=Parameters)
    iso_names = [ISO_ID[i][ISO_ID_INDEX['iso_name']] for i in iso_id_list]
    Comment = 'Contains lines for '+','.join(iso_names)
    Comment += ('\n in %.3f-%.3f wavenumber range' % (numin,numax))
    comment(TableName,Comment)

#def queryHITRAN(TableName,iso_id_list,numin,numax):
def fetch(TableName,M,I,numin,numax,ParameterGroups=[],Parameters=[]):
    """
    INPUT PARAMETERS: 
        TableName:   local table name to fetch in (required)
        M:           HITRAN molecule number       (required)
        I:           HITRAN isotopologue number   (required)
        numin:       lower wavenumber bound       (required)
        numax:       upper wavenumber bound       (required)
    OUTPUT PARAMETERS: 
        none
    ---
    DESCRIPTION:
        Download line-by-line data from HITRANonline server
        and save it to local table. The input parameters M and I
        are the HITRAN molecule and isotopologue numbers.
        This function results in a table containing single 
        isotopologue specie. To have multiple species in a 
        single table use fetch_by_ids instead.
    ---
    EXAMPLE OF USAGE:
        fetch('HOH',1,1,4000,4100)
    ---
    """
    queryHITRAN(TableName,[ISO[(M,I)][ISO_INDEX['id']]],numin,numax,
                pargroups=ParameterGroups,params=Parameters)
    iso_name = ISO[(M,I)][ISO_INDEX['iso_name']]
    Comment = 'Contains lines for '+iso_name
    Comment += ('\n in %.3f-%.3f wavenumber range' % (numin,numax))
    comment(TableName,Comment)

# ------------------ partition sum --------------------------------------

# ------------------- LAGRANGE INTERPOLATION ----------------------

#def AtoB(aa,bb,A,B,npt)
def AtoB(aa,A,B,npt):
#***************************
#...LaGrange 3- and 4-point interpolation
#...arrays A and B are the npt data points,  given aa, a value of the 
#...A variable, the routine will find the corresponding bb value
#
#...input:  aa
#...output: bb 
    for I in range(2,npt+1):
        if A[I-1] >= aa:
            if I < 3 or I == npt:
                J = I
                if I < 3: J = 3
                if I == npt: J = npt
                J = J-1   # zero index correction
                A0D1=A[J-2]-A[J-1]
                if A0D1 == 0.0: A0D1=0.0001
                A0D2=A[J-2]-A[J]
                if A0D2 == 0.0: A0D2=0.0000
                A1D1=A[J-1]-A[J-2]
                if A1D1 == 0.0: A1D1=0.0001
                A1D2=A[J-1]-A[J]
                if A1D2 == 0.0: A1D2=0.0001
                A2D1=A[J]-A[J-2]
                if A2D1 == 0.0: A2D1=0.0001
                A2D2=A[J]-A[J-1]
                if A2D2 == 0.0: A2D2=0.0001

                A0=(aa-A[J-1])*(aa-A[J])/(A0D1*A0D2)
                A1=(aa-A[J-2])*(aa-A[J])/(A1D1*A1D2)
                A2=(aa-A[J-2])*(aa-A[J-1])/(A2D1*A2D2)

                bb = A0*B[J-2] + A1*B[J-1] + A2*B[J]

            else:
                J = I
                J = J-1   # zero index correction
                A0D1=A[J-2]-A[J-1]
                if A0D1 == 0.0: A0D1=0.0001
                A0D2=A[J-2]-A[J]
                if A0D2 == 0.0: A0D2=0.0001
                A0D3 = (A[J-2]-A[J+1])
                if A0D3 == 0.0: A0D3=0.0001
                A1D1=A[J-1]-A[J-2]
                if A1D1 == 0.0: A1D1=0.0001
                A1D2=A[J-1]-A[J]
                if A1D2 == 0.0: A1D2=0.0001
                A1D3 = A[J-1]-A[J+1]
                if A1D3 == 0.0: A1D3=0.0001

                A2D1=A[J]-A[J-2]
                if A2D1 == 0.0: A2D1=0.0001
                A2D2=A[J]-A[J-1]
                if A2D2 == 0.0: A2D2=0.0001
                A2D3 = A[J]-A[J+1]
                if A2D3 == 0.0: A2D3=0.0001

                A3D1 = A[J+1]-A[J-2]
                if A3D1 == 0.0: A3D1=0.0001
                A3D2 = A[J+1]-A[J-1]
                if A3D2 == 0.0: A3D2=0.0001
                A3D3 = A[J+1]-A[J]
                if A3D3 == 0.0: A3D3=0.0001

                A0=(aa-A[J-1])*(aa-A[J])*(aa-A[J+1])
                A0=A0/(A0D1*A0D2*A0D3)
                A1=(aa-A[J-2])*(aa-A[J])*(aa-A[J+1])
                A1=A1/(A1D1*A1D2*A1D3)
                A2=(aa-A[J-2])*(aa-A[J-1])*(aa-A[J+1])
                A2=A2/(A2D1*A2D2*A2D3)
                A3=(aa-A[J-2])*(aa-A[J-1])*(aa-A[J])
                A3=A3/(A3D1*A3D2*A3D3)

                bb = A0*B[J-2] + A1*B[J-1] + A2*B[J] + A3*B[J+1]

            break

    return bb


#  --------------- ISOTOPOLOGUE HASH ----------------------

TIPS_ISO_HASH = {}

#  --------------- STATISTICAL WEIGHT HASH ----------------------

TIPS_GSI_HASH = {}

#  --------------- INTERPOLATION NODES ----------------------

Tdat = __FloatType__(  [60.,  85., 110., 135., 160., 185., 210., 235.,
                       260., 285., 310., 335., 360., 385., 410., 435., 460., 485.,
                       510., 535., 560., 585., 610., 635., 660., 685., 710., 735.,
                       760., 785., 810., 835., 860., 885., 910., 935., 960., 985.,
                      1010.,1035.,1060.,1085.,1110.,1135.,1160.,1185.,1210.,1235.,
                      1260.,1285.,1310.,1335.,1360.,1385.,1410.,1435.,1460.,1485.,
                      1510.,1535.,1560.,1585.,1610.,1635.,1660.,1685.,1710.,1735.,
                      1760.,1785.,1810.,1835.,1860.,1885.,1910.,1935.,1960.,1985.,
                      2010.,2035.,2060.,2085.,2110.,2135.,2160.,2185.,2210.,2235.,
                      2260.,2285.,2310.,2335.,2360.,2385.,2410.,2435.,2460.,2485.,
                      2510.,2535.,2560.,2585.,2610.,2635.,2660.,2685.,2710.,2735.,
                      2760.,2785.,2810.,2835.,2860.,2885.,2910.,2935.,2960.,2985.,
                      3010.]  )

TIPS_NPT = len(Tdat)


# REMARK
# float32 gives exactly the same results as fortran TIPS, because 
# all constants in the fortran code given as xx.xxE+-XX, i.e. 
# in single precision. By this fact all unsignificant figures 
# over single precision are filled with digital garbage


#  --------------- H2O 161: M = 1, I = 1 ---------------------
M = 1
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.16824E+02, 0.27771E+02, 0.40408E+02,
      0.54549E+02, 0.70054E+02, 0.86817E+02, 0.10475E+03, 0.12380E+03,
      0.14391E+03, 0.16503E+03, 0.18714E+03, 0.21021E+03, 0.23425E+03,
      0.25924E+03, 0.28518E+03, 0.31209E+03, 0.33997E+03, 0.36883E+03,
      0.39870E+03, 0.42959E+03, 0.46152E+03, 0.49452E+03, 0.52860E+03,
      0.56380E+03, 0.60015E+03, 0.63766E+03, 0.67637E+03, 0.71631E+03,
      0.75750E+03, 0.79999E+03, 0.84380E+03, 0.88897E+03, 0.93553E+03,
      0.98353E+03, 0.10330E+04, 0.10840E+04, 0.11365E+04, 0.11906E+04,
      0.12463E+04, 0.13037E+04, 0.13628E+04, 0.14237E+04, 0.14863E+04,
      0.15509E+04, 0.16173E+04, 0.16856E+04, 0.17559E+04, 0.18283E+04,
      0.19028E+04, 0.19793E+04, 0.20581E+04, 0.21391E+04, 0.22224E+04,
      0.23080E+04, 0.24067E+04, 0.24975E+04, 0.25908E+04, 0.26867E+04,
      0.27853E+04, 0.28865E+04, 0.29904E+04, 0.30972E+04, 0.32068E+04,
      0.33194E+04, 0.34349E+04, 0.35535E+04, 0.36752E+04, 0.38001E+04,
      0.39282E+04, 0.40597E+04, 0.41945E+04, 0.43327E+04, 0.44745E+04,
      0.46199E+04, 0.47688E+04, 0.49215E+04, 0.50780E+04, 0.52384E+04,
      0.54027E+04, 0.55710E+04, 0.57434E+04, 0.59200E+04, 0.61008E+04,
      0.62859E+04, 0.64754E+04, 0.66693E+04, 0.68679E+04, 0.70710E+04,
      0.72788E+04, 0.74915E+04, 0.77090E+04, 0.79315E+04, 0.81590E+04,
      0.83917E+04, 0.86296E+04, 0.88728E+04, 0.91214E+04, 0.93755E+04,
      0.96351E+04, 0.99005E+04, 0.10171E+05, 0.10448E+05, 0.10731E+05,
      0.11020E+05, 0.11315E+05, 0.11617E+05, 0.11924E+05, 0.12238E+05,
      0.12559E+05, 0.12886E+05, 0.13220E+05, 0.13561E+05, 0.13909E+05,
      0.14263E+05, 0.14625E+05, 0.14995E+05, 0.15371E+05, 0.15755E+05,
      0.16147E+05])


#  --------------- H2O 181: M = 1, I = 2 ---------------------
M = 1
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.15960E+02, 0.26999E+02, 0.39743E+02,
      0.54003E+02, 0.69639E+02, 0.86543E+02, 0.10463E+03, 0.12384E+03,
      0.14412E+03, 0.16542E+03, 0.18773E+03, 0.21103E+03, 0.23531E+03,
      0.26057E+03, 0.28681E+03, 0.31406E+03, 0.34226E+03, 0.37130E+03,
      0.40135E+03, 0.43243E+03, 0.46456E+03, 0.49777E+03, 0.53206E+03,
      0.56748E+03, 0.60405E+03, 0.64179E+03, 0.68074E+03, 0.72093E+03,
      0.76238E+03, 0.80513E+03, 0.84922E+03, 0.89467E+03, 0.94152E+03,
      0.98982E+03, 0.10396E+04, 0.10909E+04, 0.11437E+04, 0.11982E+04,
      0.12543E+04, 0.13120E+04, 0.13715E+04, 0.14328E+04, 0.14959E+04,
      0.15608E+04, 0.16276E+04, 0.16964E+04, 0.17672E+04, 0.18401E+04,
      0.19151E+04, 0.19922E+04, 0.20715E+04, 0.21531E+04, 0.22370E+04,
      0.23232E+04, 0.24118E+04, 0.25030E+04, 0.25967E+04, 0.26929E+04,
      0.27918E+04, 0.28934E+04, 0.29978E+04, 0.31050E+04, 0.32151E+04,
      0.33281E+04, 0.34441E+04, 0.35632E+04, 0.36854E+04, 0.38108E+04,
      0.39395E+04, 0.40715E+04, 0.42070E+04, 0.43459E+04, 0.44883E+04,
      0.46343E+04, 0.47840E+04, 0.49374E+04, 0.50946E+04, 0.52558E+04,
      0.54209E+04, 0.55900E+04, 0.57632E+04, 0.59407E+04, 0.61224E+04,
      0.63084E+04, 0.64988E+04, 0.66938E+04, 0.68933E+04, 0.70975E+04,
      0.73064E+04, 0.75202E+04, 0.77389E+04, 0.79625E+04, 0.81913E+04,
      0.84252E+04, 0.86644E+04, 0.89089E+04, 0.91588E+04, 0.94143E+04,
      0.96754E+04, 0.99422E+04, 0.10215E+05, 0.10493E+05, 0.10778E+05,
      0.11068E+05, 0.11365E+05, 0.11668E+05, 0.11977E+05, 0.12293E+05,
      0.12616E+05, 0.12945E+05, 0.13281E+05, 0.13624E+05, 0.13973E+05,
      0.14330E+05, 0.14694E+05, 0.15066E+05, 0.15445E+05, 0.15831E+05,
      0.16225E+05])


#  --------------- H2O 171: M = 1, I = 3 ---------------------
M = 1
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.95371E+02, 0.16134E+03, 0.23750E+03,
      0.32273E+03, 0.41617E+03, 0.51722E+03, 0.62540E+03, 0.74036E+03,
      0.86185E+03, 0.98970E+03, 0.11238E+04, 0.12642E+04, 0.14097E+04,
      0.15599E+04, 0.17159E+04, 0.18777E+04, 0.20453E+04, 0.22188E+04,
      0.23983E+04, 0.25840E+04, 0.27760E+04, 0.29743E+04, 0.31792E+04,
      0.33907E+04, 0.36091E+04, 0.38346E+04, 0.40672E+04, 0.43072E+04,
      0.45547E+04, 0.48100E+04, 0.50732E+04, 0.53446E+04, 0.56244E+04,
      0.59128E+04, 0.62100E+04, 0.65162E+04, 0.68317E+04, 0.71567E+04,
      0.74915E+04, 0.78363E+04, 0.81914E+04, 0.85571E+04, 0.89335E+04,
      0.93211E+04, 0.97200E+04, 0.10131E+05, 0.10553E+05, 0.10988E+05,
      0.11435E+05, 0.11895E+05, 0.12368E+05, 0.12855E+05, 0.13356E+05,
      0.13870E+05, 0.14399E+05, 0.14943E+05, 0.15502E+05, 0.16076E+05,
      0.16666E+05, 0.17272E+05, 0.17895E+05, 0.18534E+05, 0.19191E+05,
      0.19865E+05, 0.20557E+05, 0.21267E+05, 0.21996E+05, 0.22744E+05,
      0.23512E+05, 0.24299E+05, 0.25106E+05, 0.25935E+05, 0.26784E+05,
      0.27655E+05, 0.28547E+05, 0.29462E+05, 0.30400E+05, 0.31361E+05,
      0.32345E+05, 0.33353E+05, 0.34386E+05, 0.35444E+05, 0.36527E+05,
      0.37637E+05, 0.38772E+05, 0.39934E+05, 0.41124E+05, 0.42341E+05,
      0.43587E+05, 0.44861E+05, 0.46165E+05, 0.47498E+05, 0.48862E+05,
      0.50256E+05, 0.51682E+05, 0.53139E+05, 0.54629E+05, 0.56152E+05,
      0.57708E+05, 0.59299E+05, 0.60923E+05, 0.62583E+05, 0.64279E+05,
      0.66011E+05, 0.67779E+05, 0.69585E+05, 0.71429E+05, 0.73312E+05,
      0.75234E+05, 0.77195E+05, 0.79197E+05, 0.81240E+05, 0.83325E+05,
      0.85452E+05, 0.87622E+05, 0.89835E+05, 0.92093E+05, 0.94395E+05,
      0.96743E+05])


#  --------------- H2O 162: M = 1, I = 4 ---------------------
M = 1
I = 4
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.75792E+02, 0.12986E+03, 0.19244E+03,
      0.26253E+03, 0.33942E+03, 0.42259E+03, 0.51161E+03, 0.60619E+03,
      0.70609E+03, 0.81117E+03, 0.92132E+03, 0.10365E+04, 0.11567E+04,
      0.12820E+04, 0.14124E+04, 0.15481E+04, 0.16891E+04, 0.18355E+04,
      0.19876E+04, 0.21455E+04, 0.23092E+04, 0.24791E+04, 0.26551E+04,
      0.28376E+04, 0.30268E+04, 0.32258E+04, 0.34288E+04, 0.36392E+04,
      0.38571E+04, 0.40828E+04, 0.43165E+04, 0.45584E+04, 0.48089E+04,
      0.50681E+04, 0.53363E+04, 0.56139E+04, 0.59009E+04, 0.61979E+04,
      0.65049E+04, 0.68224E+04, 0.71506E+04, 0.74898E+04, 0.78403E+04,
      0.82024E+04, 0.85765E+04, 0.89628E+04, 0.93618E+04, 0.97736E+04,
      0.10199E+05, 0.10637E+05, 0.11090E+05, 0.11557E+05, 0.12039E+05,
      0.12535E+05, 0.13047E+05, 0.13575E+05, 0.14119E+05, 0.14679E+05,
      0.15257E+05, 0.15851E+05, 0.16464E+05, 0.17094E+05, 0.17743E+05,
      0.18411E+05, 0.19098E+05, 0.19805E+05, 0.20532E+05, 0.21280E+05,
      0.22049E+05, 0.22840E+05, 0.23652E+05, 0.24487E+05, 0.25345E+05,
      0.26227E+05, 0.27132E+05, 0.28062E+05, 0.29016E+05, 0.29997E+05,
      0.31002E+05, 0.32035E+05, 0.33094E+05, 0.34180E+05, 0.35295E+05,
      0.36438E+05, 0.37610E+05, 0.38812E+05, 0.40044E+05, 0.41306E+05,
      0.42600E+05, 0.43926E+05, 0.45284E+05, 0.46675E+05, 0.48100E+05,
      0.49559E+05, 0.51053E+05, 0.52583E+05, 0.54148E+05, 0.55750E+05,
      0.57390E+05, 0.59067E+05, 0.60783E+05, 0.62539E+05, 0.64334E+05,
      0.66170E+05, 0.68047E+05, 0.69967E+05, 0.71929E+05, 0.73934E+05,
      0.75983E+05, 0.78078E+05, 0.80217E+05, 0.82403E+05, 0.84636E+05,
      0.86917E+05, 0.89246E+05, 0.91625E+05, 0.94053E+05, 0.96533E+05,
      0.99064E+05])


#  --------------- H2O 182: M = 1, I = 5 ---------------------
M = 1
I = 5
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.82770E+02, 0.13749E+03, 0.20083E+03,
      0.27176E+03, 0.34955E+03, 0.43370E+03, 0.52376E+03, 0.61944E+03,
      0.72050E+03, 0.82679E+03, 0.93821E+03, 0.10547E+04, 0.11763E+04,
      0.13031E+04, 0.14350E+04, 0.15723E+04, 0.17150E+04, 0.18633E+04,
      0.20172E+04, 0.21770E+04, 0.23429E+04, 0.25149E+04, 0.26934E+04,
      0.28784E+04, 0.30702E+04, 0.32690E+04, 0.34750E+04, 0.36885E+04,
      0.39096E+04, 0.41386E+04, 0.43758E+04, 0.46213E+04, 0.48755E+04,
      0.51386E+04, 0.54109E+04, 0.56927E+04, 0.59841E+04, 0.62856E+04,
      0.65973E+04, 0.69197E+04, 0.72529E+04, 0.75973E+04, 0.79533E+04,
      0.83210E+04, 0.87009E+04, 0.90933E+04, 0.94985E+04, 0.99168E+04,
      0.10348E+05, 0.10794E+05, 0.11254E+05, 0.11728E+05, 0.12217E+05,
      0.12722E+05, 0.13242E+05, 0.13778E+05, 0.14331E+05, 0.14900E+05,
      0.15486E+05, 0.16091E+05, 0.16713E+05, 0.17353E+05, 0.18012E+05,
      0.18691E+05, 0.19389E+05, 0.20108E+05, 0.20847E+05, 0.21607E+05,
      0.22388E+05, 0.23191E+05, 0.24017E+05, 0.24866E+05, 0.25738E+05,
      0.26633E+05, 0.27553E+05, 0.28498E+05, 0.29468E+05, 0.30464E+05,
      0.31486E+05, 0.32536E+05, 0.33612E+05, 0.34716E+05, 0.35849E+05,
      0.37011E+05, 0.38202E+05, 0.39424E+05, 0.40676E+05, 0.41959E+05,
      0.43274E+05, 0.44622E+05, 0.46002E+05, 0.47416E+05, 0.48864E+05,
      0.50348E+05, 0.51866E+05, 0.53421E+05, 0.55012E+05, 0.56640E+05,
      0.58307E+05, 0.60012E+05, 0.61757E+05, 0.63541E+05, 0.65366E+05,
      0.67233E+05, 0.69141E+05, 0.71092E+05, 0.73087E+05, 0.75125E+05,
      0.77209E+05, 0.79338E+05, 0.81513E+05, 0.83736E+05, 0.86006E+05,
      0.88324E+05, 0.90693E+05, 0.93111E+05, 0.95580E+05, 0.98100E+05,
      0.10067E+06])


#  --------------- H2O 172: M = 1, I = 6 ---------------------
M = 1
I = 6
TIPS_GSI_HASH[(M,I)] = __FloatType__(36.)
TIPS_ISO_HASH[(M,I)] = float32([0.49379E+03, 0.82021E+03, 0.11980E+04,
      0.16211E+04, 0.20851E+04, 0.25870E+04, 0.31242E+04, 0.36949E+04,
      0.42977E+04, 0.49317E+04, 0.55963E+04, 0.62911E+04, 0.70164E+04,
      0.77722E+04, 0.85591E+04, 0.93777E+04, 0.10228E+05, 0.11112E+05,
      0.12030E+05, 0.12983E+05, 0.13971E+05, 0.14997E+05, 0.16061E+05,
      0.17163E+05, 0.18306E+05, 0.19491E+05, 0.20719E+05, 0.21991E+05,
      0.23309E+05, 0.24673E+05, 0.26086E+05, 0.27549E+05, 0.29064E+05,
      0.30631E+05, 0.32254E+05, 0.33932E+05, 0.35669E+05, 0.37464E+05,
      0.39321E+05, 0.41242E+05, 0.43227E+05, 0.45279E+05, 0.47399E+05,
      0.49589E+05, 0.51852E+05, 0.54189E+05, 0.56602E+05, 0.59094E+05,
      0.61666E+05, 0.64320E+05, 0.67058E+05, 0.69883E+05, 0.72796E+05,
      0.75801E+05, 0.78899E+05, 0.82092E+05, 0.85382E+05, 0.88773E+05,
      0.92266E+05, 0.95863E+05, 0.99568E+05, 0.10338E+06, 0.10731E+06,
      0.11135E+06, 0.11551E+06, 0.11979E+06, 0.12419E+06, 0.12871E+06,
      0.13337E+06, 0.13815E+06, 0.14307E+06, 0.14812E+06, 0.15331E+06,
      0.15865E+06, 0.16412E+06, 0.16975E+06, 0.17553E+06, 0.18146E+06,
      0.18754E+06, 0.19379E+06, 0.20020E+06, 0.20678E+06, 0.21352E+06,
      0.22044E+06, 0.22753E+06, 0.23480E+06, 0.24226E+06, 0.24990E+06,
      0.25773E+06, 0.26575E+06, 0.27397E+06, 0.28239E+06, 0.29102E+06,
      0.29985E+06, 0.30889E+06, 0.31814E+06, 0.32762E+06, 0.33731E+06,
      0.34724E+06, 0.35739E+06, 0.36777E+06, 0.37840E+06, 0.38926E+06,
      0.40038E+06, 0.41174E+06, 0.42335E+06, 0.43523E+06, 0.44737E+06,
      0.45977E+06, 0.47245E+06, 0.48540E+06, 0.49863E+06, 0.51214E+06,
      0.52595E+06, 0.54005E+06, 0.55444E+06, 0.56914E+06, 0.58415E+06,
      0.59947E+06])


#  --------------- CO2 626: M = 2, I = 1 ---------------------
M = 2
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.53642E+02, 0.75947E+02, 0.98292E+02,
      0.12078E+03, 0.14364E+03, 0.16714E+03, 0.19160E+03, 0.21731E+03,
      0.24454E+03, 0.27355E+03, 0.30456E+03, 0.33778E+03, 0.37343E+03,
      0.41170E+03, 0.45280E+03, 0.49692E+03, 0.54427E+03, 0.59505E+03,
      0.64948E+03, 0.70779E+03, 0.77019E+03, 0.83693E+03, 0.90825E+03,
      0.98440E+03, 0.10656E+04, 0.11522E+04, 0.12445E+04, 0.13427E+04,
      0.14471E+04, 0.15580E+04, 0.16759E+04, 0.18009E+04, 0.19334E+04,
      0.20739E+04, 0.22225E+04, 0.23798E+04, 0.25462E+04, 0.27219E+04,
      0.29074E+04, 0.31032E+04, 0.33097E+04, 0.35272E+04, 0.37564E+04,
      0.39976E+04, 0.42514E+04, 0.45181E+04, 0.47985E+04, 0.50929E+04,
      0.54019E+04, 0.57260E+04, 0.60659E+04, 0.64221E+04, 0.67952E+04,
      0.71859E+04, 0.75946E+04, 0.80222E+04, 0.84691E+04, 0.89362E+04,
      0.94241E+04, 0.99335E+04, 0.10465E+05, 0.11020E+05, 0.11598E+05,
      0.12201E+05, 0.12828E+05, 0.13482E+05, 0.14163E+05, 0.14872E+05,
      0.15609E+05, 0.16376E+05, 0.17173E+05, 0.18001E+05, 0.18861E+05,
      0.19754E+05, 0.20682E+05, 0.21644E+05, 0.22643E+05, 0.23678E+05,
      0.24752E+05, 0.25865E+05, 0.27018E+05, 0.28212E+05, 0.29449E+05,
      0.30730E+05, 0.32055E+05, 0.33426E+05, 0.34845E+05, 0.36312E+05,
      0.37828E+05, 0.39395E+05, 0.41015E+05, 0.42688E+05, 0.44416E+05,
      0.46199E+05, 0.48041E+05, 0.49942E+05, 0.51902E+05, 0.53925E+05,
      0.56011E+05, 0.58162E+05, 0.60379E+05, 0.62664E+05, 0.65019E+05,
      0.67444E+05, 0.69942E+05, 0.72515E+05, 0.75163E+05, 0.77890E+05,
      0.80695E+05, 0.83582E+05, 0.86551E+05, 0.89605E+05, 0.92746E+05,
      0.95975E+05, 0.99294E+05, 0.10271E+06, 0.10621E+06, 0.10981E+06,
      0.11351E+06])

      
#  --------------- CO2 636: M = 2, I = 2 ---------------------
M = 2
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.10728E+03, 0.15189E+03, 0.19659E+03,
      0.24164E+03, 0.28753E+03, 0.33486E+03, 0.38429E+03, 0.43643E+03,
      0.49184E+03, 0.55104E+03, 0.61449E+03, 0.68263E+03, 0.75589E+03,
      0.83468E+03, 0.91943E+03, 0.10106E+04, 0.11085E+04, 0.12137E+04,
      0.13266E+04, 0.14477E+04, 0.15774E+04, 0.17163E+04, 0.18649E+04,
      0.20237E+04, 0.21933E+04, 0.23743E+04, 0.25673E+04, 0.27729E+04,
      0.29917E+04, 0.32245E+04, 0.34718E+04, 0.37345E+04, 0.40132E+04,
      0.43087E+04, 0.46218E+04, 0.49533E+04, 0.53041E+04, 0.56749E+04,
      0.60668E+04, 0.64805E+04, 0.69171E+04, 0.73774E+04, 0.78626E+04,
      0.83736E+04, 0.89114E+04, 0.94772E+04, 0.10072E+05, 0.10697E+05,
      0.11353E+05, 0.12042E+05, 0.12765E+05, 0.13523E+05, 0.14317E+05,
      0.15148E+05, 0.16019E+05, 0.16930E+05, 0.17883E+05, 0.18879E+05,
      0.19920E+05, 0.21008E+05, 0.22143E+05, 0.23328E+05, 0.24563E+05,
      0.25852E+05, 0.27195E+05, 0.28594E+05, 0.30051E+05, 0.31568E+05,
      0.33146E+05, 0.34788E+05, 0.36496E+05, 0.38271E+05, 0.40115E+05,
      0.42031E+05, 0.44021E+05, 0.46086E+05, 0.48230E+05, 0.50453E+05,
      0.52759E+05, 0.55150E+05, 0.57628E+05, 0.60195E+05, 0.62854E+05,
      0.65608E+05, 0.68459E+05, 0.71409E+05, 0.74461E+05, 0.77618E+05,
      0.80883E+05, 0.84258E+05, 0.87746E+05, 0.91350E+05, 0.95073E+05,
      0.98918E+05, 0.10289E+06, 0.10698E+06, 0.11121E+06, 0.11558E+06,
      0.12008E+06, 0.12472E+06, 0.12950E+06, 0.13443E+06, 0.13952E+06,
      0.14475E+06, 0.15015E+06, 0.15571E+06, 0.16143E+06, 0.16732E+06,
      0.17338E+06, 0.17962E+06, 0.18604E+06, 0.19264E+06, 0.19943E+06,
      0.20642E+06, 0.21360E+06, 0.22098E+06, 0.22856E+06, 0.23636E+06,
      0.24436E+06])
    
    
#  --------------- CO2 628: M = 2, I = 3 ---------------------
M = 2
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.11368E+03, 0.16096E+03, 0.20833E+03,
      0.25603E+03, 0.30452E+03, 0.35442E+03, 0.40640E+03, 0.46110E+03,
      0.51910E+03, 0.58093E+03, 0.64709E+03, 0.71804E+03, 0.79422E+03,
      0.87607E+03, 0.96402E+03, 0.10585E+04, 0.11600E+04, 0.12689E+04,
      0.13857E+04, 0.15108E+04, 0.16449E+04, 0.17883E+04, 0.19416E+04,
      0.21054E+04, 0.22803E+04, 0.24668E+04, 0.26655E+04, 0.28770E+04,
      0.31021E+04, 0.33414E+04, 0.35956E+04, 0.38654E+04, 0.41516E+04,
      0.44549E+04, 0.47761E+04, 0.51160E+04, 0.54755E+04, 0.58555E+04,
      0.62568E+04, 0.66804E+04, 0.71273E+04, 0.75982E+04, 0.80944E+04,
      0.86169E+04, 0.91666E+04, 0.97446E+04, 0.10352E+05, 0.10990E+05,
      0.11660E+05, 0.12363E+05, 0.13101E+05, 0.13874E+05, 0.14683E+05,
      0.15531E+05, 0.16418E+05, 0.17347E+05, 0.18317E+05, 0.19332E+05,
      0.20392E+05, 0.21499E+05, 0.22654E+05, 0.23859E+05, 0.25116E+05,
      0.26426E+05, 0.27792E+05, 0.29214E+05, 0.30695E+05, 0.32236E+05,
      0.33840E+05, 0.35508E+05, 0.37242E+05, 0.39045E+05, 0.40917E+05,
      0.42862E+05, 0.44881E+05, 0.46977E+05, 0.49152E+05, 0.51407E+05,
      0.53746E+05, 0.56171E+05, 0.58683E+05, 0.61286E+05, 0.63981E+05,
      0.66772E+05, 0.69661E+05, 0.72650E+05, 0.75742E+05, 0.78940E+05,
      0.82246E+05, 0.85664E+05, 0.89196E+05, 0.92845E+05, 0.96613E+05,
      0.10050E+06, 0.10452E+06, 0.10867E+06, 0.11295E+06, 0.11736E+06,
      0.12191E+06, 0.12661E+06, 0.13145E+06, 0.13643E+06, 0.14157E+06,
      0.14687E+06, 0.15232E+06, 0.15794E+06, 0.16372E+06, 0.16968E+06,
      0.17580E+06, 0.18211E+06, 0.18859E+06, 0.19526E+06, 0.20213E+06,
      0.20918E+06, 0.21643E+06, 0.22388E+06, 0.23154E+06, 0.23941E+06,
      0.24750E+06])
    
    
#  --------------- CO2 627: M = 2, I = 4 ---------------------
M = 2
I = 4
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.66338E+03, 0.93923E+03, 0.12156E+04,
      0.14938E+04, 0.17766E+04, 0.20676E+04, 0.23705E+04, 0.26891E+04,
      0.30267E+04, 0.33866E+04, 0.37714E+04, 0.41839E+04, 0.46267E+04,
      0.51023E+04, 0.56132E+04, 0.61618E+04, 0.67508E+04, 0.73827E+04,
      0.80603E+04, 0.87863E+04, 0.95636E+04, 0.10395E+05, 0.11284E+05,
      0.12233E+05, 0.13246E+05, 0.14326E+05, 0.15477E+05, 0.16702E+05,
      0.18005E+05, 0.19390E+05, 0.20861E+05, 0.22422E+05, 0.24077E+05,
      0.25832E+05, 0.27689E+05, 0.29655E+05, 0.31734E+05, 0.33931E+05,
      0.36250E+05, 0.38698E+05, 0.41280E+05, 0.44002E+05, 0.46869E+05,
      0.49886E+05, 0.53062E+05, 0.56400E+05, 0.59909E+05, 0.63594E+05,
      0.67462E+05, 0.71521E+05, 0.75777E+05, 0.80238E+05, 0.84911E+05,
      0.89804E+05, 0.94925E+05, 0.10028E+06, 0.10588E+06, 0.11173E+06,
      0.11785E+06, 0.12423E+06, 0.13090E+06, 0.13785E+06, 0.14510E+06,
      0.15265E+06, 0.16053E+06, 0.16873E+06, 0.17727E+06, 0.18615E+06,
      0.19540E+06, 0.20501E+06, 0.21501E+06, 0.22540E+06, 0.23619E+06,
      0.24740E+06, 0.25904E+06, 0.27112E+06, 0.28365E+06, 0.29664E+06,
      0.31012E+06, 0.32409E+06, 0.33856E+06, 0.35356E+06, 0.36908E+06,
      0.38516E+06, 0.40180E+06, 0.41902E+06, 0.43683E+06, 0.45525E+06,
      0.47429E+06, 0.49397E+06, 0.51431E+06, 0.53532E+06, 0.55702E+06,
      0.57943E+06, 0.60256E+06, 0.62644E+06, 0.65107E+06, 0.67648E+06,
      0.70269E+06, 0.72972E+06, 0.75758E+06, 0.78629E+06, 0.81588E+06,
      0.84636E+06, 0.87775E+06, 0.91008E+06, 0.94337E+06, 0.97763E+06,
      0.10129E+07, 0.10492E+07, 0.10865E+07, 0.11249E+07, 0.11644E+07,
      0.12050E+07, 0.12467E+07, 0.12896E+07, 0.13337E+07, 0.13789E+07,
      0.14255E+07])
    

#  --------------- CO2 638: M = 2, I = 5 ---------------------
M = 2
I = 5
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.22737E+03, 0.32194E+03, 0.41671E+03,
      0.51226E+03, 0.60963E+03, 0.71017E+03, 0.81528E+03, 0.92628E+03,
      0.10444E+04, 0.11707E+04, 0.13061E+04, 0.14518E+04, 0.16085E+04,
      0.17772E+04, 0.19588E+04, 0.21542E+04, 0.23644E+04, 0.25903E+04,
      0.28330E+04, 0.30934E+04, 0.33726E+04, 0.36717E+04, 0.39918E+04,
      0.43342E+04, 0.47001E+04, 0.50907E+04, 0.55074E+04, 0.59515E+04,
      0.64244E+04, 0.69276E+04, 0.74626E+04, 0.80310E+04, 0.86344E+04,
      0.92744E+04, 0.99528E+04, 0.10671E+05, 0.11432E+05, 0.12236E+05,
      0.13086E+05, 0.13984E+05, 0.14932E+05, 0.15932E+05, 0.16985E+05,
      0.18096E+05, 0.19265E+05, 0.20495E+05, 0.21788E+05, 0.23148E+05,
      0.24576E+05, 0.26075E+05, 0.27648E+05, 0.29298E+05, 0.31027E+05,
      0.32839E+05, 0.34736E+05, 0.36721E+05, 0.38798E+05, 0.40970E+05,
      0.43240E+05, 0.45611E+05, 0.48087E+05, 0.50671E+05, 0.53368E+05,
      0.56180E+05, 0.59111E+05, 0.62165E+05, 0.65347E+05, 0.68659E+05,
      0.72107E+05, 0.75694E+05, 0.79425E+05, 0.83303E+05, 0.87334E+05,
      0.91522E+05, 0.95872E+05, 0.10039E+06, 0.10507E+06, 0.10994E+06,
      0.11498E+06, 0.12021E+06, 0.12563E+06, 0.13125E+06, 0.13707E+06,
      0.14309E+06, 0.14933E+06, 0.15579E+06, 0.16247E+06, 0.16938E+06,
      0.17653E+06, 0.18392E+06, 0.19156E+06, 0.19946E+06, 0.20761E+06,
      0.21604E+06, 0.22473E+06, 0.23371E+06, 0.24298E+06, 0.25254E+06,
      0.26240E+06, 0.27258E+06, 0.28307E+06, 0.29388E+06, 0.30502E+06,
      0.31651E+06, 0.32834E+06, 0.34052E+06, 0.35307E+06, 0.36599E+06,
      0.37929E+06, 0.39298E+06, 0.40706E+06, 0.42155E+06, 0.43645E+06,
      0.45178E+06, 0.46753E+06, 0.48373E+06, 0.50038E+06, 0.51748E+06,
      0.53506E+06])


#  --------------- CO2 637: M = 2, I = 6 ---------------------
M = 2
I = 6
TIPS_GSI_HASH[(M,I)] = __FloatType__(12.)
TIPS_ISO_HASH[(M,I)] = float32([0.13267E+04, 0.18785E+04, 0.24314E+04,
      0.29888E+04, 0.35566E+04, 0.41426E+04, 0.47550E+04, 0.54013E+04,
      0.60886E+04, 0.68232E+04, 0.76109E+04, 0.84574E+04, 0.93678E+04,
      0.10348E+05, 0.11402E+05, 0.12536E+05, 0.13755E+05, 0.15065E+05,
      0.16471E+05, 0.17980E+05, 0.19598E+05, 0.21330E+05, 0.23184E+05,
      0.25166E+05, 0.27283E+05, 0.29543E+05, 0.31953E+05, 0.34521E+05,
      0.37256E+05, 0.40164E+05, 0.43256E+05, 0.46541E+05, 0.50026E+05,
      0.53723E+05, 0.57641E+05, 0.61790E+05, 0.66180E+05, 0.70823E+05,
      0.75729E+05, 0.80910E+05, 0.86378E+05, 0.92145E+05, 0.98224E+05,
      0.10463E+06, 0.11137E+06, 0.11846E+06, 0.12592E+06, 0.13375E+06,
      0.14198E+06, 0.15062E+06, 0.15969E+06, 0.16920E+06, 0.17916E+06,
      0.18959E+06, 0.20052E+06, 0.21196E+06, 0.22392E+06, 0.23642E+06,
      0.24949E+06, 0.26314E+06, 0.27740E+06, 0.29227E+06, 0.30779E+06,
      0.32398E+06, 0.34085E+06, 0.35842E+06, 0.37673E+06, 0.39579E+06,
      0.41563E+06, 0.43626E+06, 0.45772E+06, 0.48003E+06, 0.50322E+06,
      0.52730E+06, 0.55232E+06, 0.57829E+06, 0.60524E+06, 0.63320E+06,
      0.66219E+06, 0.69226E+06, 0.72342E+06, 0.75571E+06, 0.78916E+06,
      0.82380E+06, 0.85966E+06, 0.89678E+06, 0.93518E+06, 0.97490E+06,
      0.10160E+07, 0.10585E+07, 0.11023E+07, 0.11477E+07, 0.11946E+07,
      0.12430E+07, 0.12929E+07, 0.13445E+07, 0.13977E+07, 0.14526E+07,
      0.15093E+07, 0.15677E+07, 0.16280E+07, 0.16901E+07, 0.17541E+07,
      0.18200E+07, 0.18880E+07, 0.19579E+07, 0.20300E+07, 0.21042E+07,
      0.21805E+07, 0.22591E+07, 0.23400E+07, 0.24232E+07, 0.25087E+07,
      0.25967E+07, 0.26871E+07, 0.27801E+07, 0.28757E+07, 0.29739E+07,
      0.30747E+07])


#  --------------- CO2 828: M = 2, I = 7 ---------------------
M = 2
I = 7
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.60334E+02, 0.85430E+02, 0.11058E+03,
      0.13590E+03, 0.16167E+03, 0.18821E+03, 0.21588E+03, 0.24502E+03,
      0.27595E+03, 0.30896E+03, 0.34431E+03, 0.38225E+03, 0.42301E+03,
      0.46684E+03, 0.51397E+03, 0.56464E+03, 0.61907E+03, 0.67753E+03,
      0.74027E+03, 0.80753E+03, 0.87961E+03, 0.95676E+03, 0.10393E+04,
      0.11275E+04, 0.12217E+04, 0.13222E+04, 0.14293E+04, 0.15434E+04,
      0.16648E+04, 0.17940E+04, 0.19312E+04, 0.20769E+04, 0.22315E+04,
      0.23954E+04, 0.25691E+04, 0.27529E+04, 0.29474E+04, 0.31530E+04,
      0.33702E+04, 0.35995E+04, 0.38414E+04, 0.40965E+04, 0.43654E+04,
      0.46484E+04, 0.49464E+04, 0.52598E+04, 0.55892E+04, 0.59353E+04,
      0.62988E+04, 0.66803E+04, 0.70804E+04, 0.74998E+04, 0.79394E+04,
      0.83998E+04, 0.88817E+04, 0.93859E+04, 0.99132E+04, 0.10464E+05,
      0.11040E+05, 0.11642E+05, 0.12270E+05, 0.12925E+05, 0.13609E+05,
      0.14321E+05, 0.15064E+05, 0.15838E+05, 0.16643E+05, 0.17482E+05,
      0.18355E+05, 0.19263E+05, 0.20207E+05, 0.21188E+05, 0.22208E+05,
      0.23267E+05, 0.24366E+05, 0.25508E+05, 0.26692E+05, 0.27921E+05,
      0.29195E+05, 0.30516E+05, 0.31886E+05, 0.33304E+05, 0.34773E+05,
      0.36294E+05, 0.37869E+05, 0.39499E+05, 0.41185E+05, 0.42929E+05,
      0.44732E+05, 0.46596E+05, 0.48522E+05, 0.50513E+05, 0.52569E+05,
      0.54692E+05, 0.56884E+05, 0.59146E+05, 0.61481E+05, 0.63890E+05,
      0.66375E+05, 0.68937E+05, 0.71578E+05, 0.74301E+05, 0.77107E+05,
      0.79998E+05, 0.82976E+05, 0.86043E+05, 0.89201E+05, 0.92452E+05,
      0.95799E+05, 0.99242E+05, 0.10278E+06, 0.10643E+06, 0.11018E+06,
      0.11403E+06, 0.11799E+06, 0.12206E+06, 0.12625E+06, 0.13055E+06,
      0.13497E+06])


#  --------------- CO2 728: M = 2, I = 8 ---------------------
M = 2
I = 8
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.70354E+03, 0.99615E+03, 0.12893E+04,
      0.15846E+04, 0.18848E+04, 0.21940E+04, 0.25162E+04, 0.28554E+04,
      0.32152E+04, 0.35991E+04, 0.40099E+04, 0.44507E+04, 0.49242E+04,
      0.54332E+04, 0.59802E+04, 0.65681E+04, 0.71996E+04, 0.78776E+04,
      0.86050E+04, 0.93847E+04, 0.10220E+05, 0.11114E+05, 0.12070E+05,
      0.13091E+05, 0.14182E+05, 0.15345E+05, 0.16585E+05, 0.17906E+05,
      0.19311E+05, 0.20805E+05, 0.22393E+05, 0.24078E+05, 0.25865E+05,
      0.27760E+05, 0.29768E+05, 0.31893E+05, 0.34140E+05, 0.36516E+05,
      0.39025E+05, 0.41674E+05, 0.44469E+05, 0.47416E+05, 0.50520E+05,
      0.53789E+05, 0.57229E+05, 0.60847E+05, 0.64650E+05, 0.68645E+05,
      0.72840E+05, 0.77242E+05, 0.81859E+05, 0.86699E+05, 0.91770E+05,
      0.97081E+05, 0.10264E+06, 0.10846E+06, 0.11454E+06, 0.12090E+06,
      0.12754E+06, 0.13447E+06, 0.14171E+06, 0.14927E+06, 0.15715E+06,
      0.16536E+06, 0.17392E+06, 0.18284E+06, 0.19213E+06, 0.20179E+06,
      0.21185E+06, 0.22231E+06, 0.23319E+06, 0.24450E+06, 0.25625E+06,
      0.26845E+06, 0.28112E+06, 0.29427E+06, 0.30791E+06, 0.32206E+06,
      0.33674E+06, 0.35196E+06, 0.36772E+06, 0.38406E+06, 0.40098E+06,
      0.41850E+06, 0.43663E+06, 0.45539E+06, 0.47480E+06, 0.49488E+06,
      0.51564E+06, 0.53710E+06, 0.55928E+06, 0.58219E+06, 0.60586E+06,
      0.63029E+06, 0.65553E+06, 0.68157E+06, 0.70844E+06, 0.73616E+06,
      0.76476E+06, 0.79424E+06, 0.82464E+06, 0.85597E+06, 0.88826E+06,
      0.92153E+06, 0.95580E+06, 0.99108E+06, 0.10274E+07, 0.10648E+07,
      0.11033E+07, 0.11429E+07, 0.11837E+07, 0.12256E+07, 0.12687E+07,
      0.13131E+07, 0.13586E+07, 0.14055E+07, 0.14536E+07, 0.15031E+07,
      0.15539E+07])


#  --------------- CO2 727: M = 2, I = 9 ---------------------
M = 2
I = 9
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.20518E+04, 0.29051E+04, 0.37601E+04,
      0.46209E+04, 0.54961E+04, 0.63969E+04, 0.73353E+04, 0.83227E+04,
      0.93698E+04, 0.10486E+05, 0.11681E+05, 0.12962E+05, 0.14337E+05,
      0.15815E+05, 0.17403E+05, 0.19110E+05, 0.20942E+05, 0.22909E+05,
      0.25018E+05, 0.27278E+05, 0.29699E+05, 0.32290E+05, 0.35060E+05,
      0.38019E+05, 0.41177E+05, 0.44545E+05, 0.48135E+05, 0.51957E+05,
      0.56023E+05, 0.60346E+05, 0.64938E+05, 0.69812E+05, 0.74981E+05,
      0.80461E+05, 0.86264E+05, 0.92406E+05, 0.98902E+05, 0.10577E+06,
      0.11302E+06, 0.12067E+06, 0.12875E+06, 0.13726E+06, 0.14622E+06,
      0.15566E+06, 0.16559E+06, 0.17604E+06, 0.18702E+06, 0.19855E+06,
      0.21066E+06, 0.22336E+06, 0.23669E+06, 0.25065E+06, 0.26528E+06,
      0.28061E+06, 0.29664E+06, 0.31342E+06, 0.33096E+06, 0.34930E+06,
      0.36845E+06, 0.38845E+06, 0.40933E+06, 0.43111E+06, 0.45383E+06,
      0.47751E+06, 0.50219E+06, 0.52790E+06, 0.55466E+06, 0.58252E+06,
      0.61151E+06, 0.64166E+06, 0.67300E+06, 0.70558E+06, 0.73943E+06,
      0.77458E+06, 0.81108E+06, 0.84896E+06, 0.88827E+06, 0.92904E+06,
      0.97131E+06, 0.10151E+07, 0.10605E+07, 0.11076E+07, 0.11563E+07,
      0.12068E+07, 0.12590E+07, 0.13130E+07, 0.13689E+07, 0.14267E+07,
      0.14865E+07, 0.15483E+07, 0.16121E+07, 0.16781E+07, 0.17462E+07,
      0.18165E+07, 0.18892E+07, 0.19641E+07, 0.20415E+07, 0.21213E+07,
      0.22036E+07, 0.22884E+07, 0.23759E+07, 0.24661E+07, 0.25590E+07,
      0.26547E+07, 0.27533E+07, 0.28549E+07, 0.29594E+07, 0.30670E+07,
      0.31778E+07, 0.32918E+07, 0.34090E+07, 0.35296E+07, 0.36536E+07,
      0.37812E+07, 0.39123E+07, 0.40470E+07, 0.41855E+07, 0.43278E+07,
      0.44739E+07])


#  --------------- CO2 838: M = 2, I = 10 ---------------------
M = 2
I = 10
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.12066E+03, 0.17085E+03, 0.22116E+03,
      0.27190E+03, 0.32364E+03, 0.37711E+03, 0.43305E+03, 0.49219E+03,
      0.55516E+03, 0.62256E+03, 0.69492E+03, 0.77276E+03, 0.85657E+03,
      0.94685E+03, 0.10441E+04, 0.11488E+04, 0.12614E+04, 0.13826E+04,
      0.15127E+04, 0.16525E+04, 0.18024E+04, 0.19630E+04, 0.21351E+04,
      0.23191E+04, 0.25158E+04, 0.27260E+04, 0.29502E+04, 0.31892E+04,
      0.34438E+04, 0.37148E+04, 0.40031E+04, 0.43094E+04, 0.46346E+04,
      0.49797E+04, 0.53455E+04, 0.57331E+04, 0.61434E+04, 0.65775E+04,
      0.70364E+04, 0.75212E+04, 0.80330E+04, 0.85730E+04, 0.91424E+04,
      0.97423E+04, 0.10374E+05, 0.11039E+05, 0.11738E+05, 0.12474E+05,
      0.13246E+05, 0.14057E+05, 0.14908E+05, 0.15801E+05, 0.16737E+05,
      0.17717E+05, 0.18744E+05, 0.19819E+05, 0.20944E+05, 0.22120E+05,
      0.23349E+05, 0.24634E+05, 0.25975E+05, 0.27376E+05, 0.28837E+05,
      0.30361E+05, 0.31950E+05, 0.33605E+05, 0.35330E+05, 0.37126E+05,
      0.38996E+05, 0.40942E+05, 0.42965E+05, 0.45069E+05, 0.47256E+05,
      0.49528E+05, 0.51888E+05, 0.54338E+05, 0.56882E+05, 0.59521E+05,
      0.62259E+05, 0.65097E+05, 0.68040E+05, 0.71090E+05, 0.74249E+05,
      0.77522E+05, 0.80910E+05, 0.84417E+05, 0.88046E+05, 0.91801E+05,
      0.95684E+05, 0.99699E+05, 0.10385E+06, 0.10814E+06, 0.11257E+06,
      0.11715E+06, 0.12187E+06, 0.12675E+06, 0.13179E+06, 0.13699E+06,
      0.14235E+06, 0.14788E+06, 0.15358E+06, 0.15946E+06, 0.16552E+06,
      0.17176E+06, 0.17819E+06, 0.18482E+06, 0.19164E+06, 0.19867E+06,
      0.20590E+06, 0.21335E+06, 0.22101E+06, 0.22889E+06, 0.23699E+06,
      0.24533E+06, 0.25390E+06, 0.26271E+06, 0.27177E+06, 0.28108E+06,
      0.29064E+06])

#  --------------- CO2 838: M = 2, I = 0 ALIAS-----------------
TIPS_GSI_HASH[(M,0)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,0)] = TIPS_ISO_HASH[(M,I)]
      
#  --------------- CO2 837: M = 2, I = 11 ---------------------
M = 2
I = 11
TIPS_GSI_HASH[(M,I)] = __FloatType__(12.)
TIPS_ISO_HASH[(M,I)] = float32([0.14071E+04, 0.19923E+04, 0.25789E+04,
      0.31704E+04, 0.37733E+04, 0.43962E+04, 0.50477E+04, 0.57360E+04,
      0.64687E+04, 0.72525E+04, 0.80938E+04, 0.89984E+04, 0.99723E+04,
      0.11021E+05, 0.12150E+05, 0.13366E+05, 0.14673E+05, 0.16079E+05,
      0.17589E+05, 0.19211E+05, 0.20949E+05, 0.22812E+05, 0.24807E+05,
      0.26940E+05, 0.29221E+05, 0.31656E+05, 0.34254E+05, 0.37023E+05,
      0.39972E+05, 0.43111E+05, 0.46449E+05, 0.49996E+05, 0.53762E+05,
      0.57756E+05, 0.61991E+05, 0.66477E+05, 0.71226E+05, 0.76249E+05,
      0.81558E+05, 0.87167E+05, 0.93088E+05, 0.99334E+05, 0.10592E+06,
      0.11286E+06, 0.12016E+06, 0.12785E+06, 0.13594E+06, 0.14444E+06,
      0.15337E+06, 0.16274E+06, 0.17258E+06, 0.18290E+06, 0.19371E+06,
      0.20504E+06, 0.21691E+06, 0.22933E+06, 0.24233E+06, 0.25592E+06,
      0.27012E+06, 0.28496E+06, 0.30046E+06, 0.31663E+06, 0.33351E+06,
      0.35111E+06, 0.36946E+06, 0.38858E+06, 0.40850E+06, 0.42924E+06,
      0.45083E+06, 0.47329E+06, 0.49666E+06, 0.52095E+06, 0.54620E+06,
      0.57243E+06, 0.59967E+06, 0.62796E+06, 0.65732E+06, 0.68778E+06,
      0.71938E+06, 0.75214E+06, 0.78611E+06, 0.82131E+06, 0.85777E+06,
      0.89553E+06, 0.93463E+06, 0.97511E+06, 0.10170E+07, 0.10603E+07,
      0.11051E+07, 0.11514E+07, 0.11993E+07, 0.12488E+07, 0.12999E+07,
      0.13527E+07, 0.14073E+07, 0.14636E+07, 0.15217E+07, 0.15816E+07,
      0.16435E+07, 0.17072E+07, 0.17730E+07, 0.18408E+07, 0.19107E+07,
      0.19827E+07, 0.20569E+07, 0.21334E+07, 0.22121E+07, 0.22931E+07,
      0.23765E+07, 0.24624E+07, 0.25507E+07, 0.26416E+07, 0.27351E+07,
      0.28312E+07, 0.29301E+07, 0.30317E+07, 0.31361E+07, 0.32434E+07,
      0.33537E+07])

    
#  --------------- O3 666: M = 3, I = 1 ---------------------
M = 3
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.30333E+03, 0.51126E+03, 0.75274E+03,
      0.10241E+04, 0.13236E+04, 0.16508E+04, 0.20068E+04, 0.23935E+04,
      0.28136E+04, 0.32703E+04, 0.37672E+04, 0.43082E+04, 0.48975E+04,
      0.55395E+04, 0.62386E+04, 0.69996E+04, 0.78272E+04, 0.87264E+04,
      0.97026E+04, 0.10761E+05, 0.11907E+05, 0.13146E+05, 0.14485E+05,
      0.15929E+05, 0.17484E+05, 0.19158E+05, 0.20957E+05, 0.22887E+05,
      0.24956E+05, 0.27172E+05, 0.29541E+05, 0.32072E+05, 0.34773E+05,
      0.37652E+05, 0.40718E+05, 0.43979E+05, 0.47444E+05, 0.51123E+05,
      0.55026E+05, 0.59161E+05, 0.63540E+05, 0.68172E+05, 0.73069E+05,
      0.78240E+05, 0.83698E+05, 0.89453E+05, 0.95517E+05, 0.10190E+06,
      0.10862E+06, 0.11569E+06, 0.12311E+06, 0.13091E+06, 0.13909E+06,
      0.14767E+06, 0.15666E+06, 0.16608E+06, 0.17594E+06, 0.18626E+06,
      0.19706E+06, 0.20834E+06, 0.22012E+06, 0.23242E+06, 0.24526E+06,
      0.25866E+06, 0.27262E+06, 0.28717E+06, 0.30233E+06, 0.31811E+06,
      0.33453E+06, 0.35161E+06, 0.36937E+06, 0.38784E+06, 0.40702E+06,
      0.42694E+06, 0.44762E+06, 0.46909E+06, 0.49135E+06, 0.51444E+06,
      0.53838E+06, 0.56318E+06, 0.58887E+06, 0.61548E+06, 0.64303E+06,
      0.67153E+06, 0.70102E+06, 0.73153E+06, 0.76306E+06, 0.79566E+06,
      0.82934E+06, 0.86413E+06, 0.90006E+06, 0.93716E+06, 0.97545E+06,
      0.10150E+07, 0.10557E+07, 0.10977E+07, 0.11411E+07, 0.11858E+07,
      0.12318E+07, 0.12792E+07, 0.13281E+07, 0.13784E+07, 0.14302E+07,
      0.14835E+07, 0.15384E+07, 0.15948E+07, 0.16529E+07, 0.17126E+07,
      0.17740E+07, 0.18371E+07, 0.19020E+07, 0.19686E+07, 0.20371E+07,
      0.21074E+07, 0.21797E+07, 0.22538E+07, 0.23300E+07, 0.24081E+07,
      0.24883E+07])


#  --------------- O3 668: M = 3, I = 2 ---------------------
M = 3
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.64763E+03, 0.10916E+04, 0.16073E+04,
      0.21870E+04, 0.28271E+04, 0.35272E+04, 0.42900E+04, 0.51197E+04,
      0.60225E+04, 0.70057E+04, 0.80771E+04, 0.92455E+04, 0.10520E+05,
      0.11911E+05, 0.13427E+05, 0.15079E+05, 0.16878E+05, 0.18834E+05,
      0.20960E+05, 0.23267E+05, 0.25767E+05, 0.28472E+05, 0.31397E+05,
      0.34553E+05, 0.37957E+05, 0.41620E+05, 0.45559E+05, 0.49790E+05,
      0.54327E+05, 0.59187E+05, 0.64387E+05, 0.69944E+05, 0.75877E+05,
      0.82203E+05, 0.88943E+05, 0.96114E+05, 0.10374E+06, 0.11184E+06,
      0.12043E+06, 0.12954E+06, 0.13918E+06, 0.14939E+06, 0.16018E+06,
      0.17159E+06, 0.18362E+06, 0.19632E+06, 0.20970E+06, 0.22380E+06,
      0.23863E+06, 0.25423E+06, 0.27063E+06, 0.28786E+06, 0.30594E+06,
      0.32490E+06, 0.34478E+06, 0.36561E+06, 0.38743E+06, 0.41026E+06,
      0.43413E+06, 0.45909E+06, 0.48517E+06, 0.51241E+06, 0.54084E+06,
      0.57049E+06, 0.60141E+06, 0.63365E+06, 0.66722E+06, 0.70219E+06,
      0.73858E+06, 0.77644E+06, 0.81581E+06, 0.85674E+06, 0.89927E+06,
      0.94345E+06, 0.98932E+06, 0.10369E+07, 0.10863E+07, 0.11375E+07,
      0.11906E+07, 0.12457E+07, 0.13027E+07, 0.13618E+07, 0.14229E+07,
      0.14862E+07, 0.15517E+07, 0.16194E+07, 0.16894E+07, 0.17618E+07,
      0.18366E+07, 0.19139E+07, 0.19937E+07, 0.20761E+07, 0.21612E+07,
      0.22490E+07, 0.23395E+07, 0.24330E+07, 0.25293E+07, 0.26286E+07,
      0.27309E+07, 0.28363E+07, 0.29449E+07, 0.30568E+07, 0.31720E+07,
      0.32905E+07, 0.34125E+07, 0.35381E+07, 0.36672E+07, 0.38000E+07,
      0.39366E+07, 0.40770E+07, 0.42213E+07, 0.43696E+07, 0.45220E+07,
      0.46785E+07, 0.48392E+07, 0.50043E+07, 0.51737E+07, 0.53476E+07,
      0.55261E+07])


#  --------------- O3 686: M = 3, I = 3 ---------------------
M = 3
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.31656E+03, 0.53355E+03, 0.78557E+03,
      0.10688E+04, 0.13815E+04, 0.17235E+04, 0.20960E+04, 0.25011E+04,
      0.29420E+04, 0.34223E+04, 0.39459E+04, 0.45172E+04, 0.51408E+04,
      0.58213E+04, 0.65639E+04, 0.73735E+04, 0.82555E+04, 0.92152E+04,
      0.10259E+05, 0.11391E+05, 0.12619E+05, 0.13949E+05, 0.15387E+05,
      0.16940E+05, 0.18614E+05, 0.20417E+05, 0.22357E+05, 0.24440E+05,
      0.26675E+05, 0.29070E+05, 0.31633E+05, 0.34374E+05, 0.37299E+05,
      0.40420E+05, 0.43746E+05, 0.47285E+05, 0.51049E+05, 0.55047E+05,
      0.59289E+05, 0.63788E+05, 0.68554E+05, 0.73598E+05, 0.78932E+05,
      0.84568E+05, 0.90519E+05, 0.96796E+05, 0.10341E+06, 0.11039E+06,
      0.11772E+06, 0.12544E+06, 0.13356E+06, 0.14208E+06, 0.15103E+06,
      0.16041E+06, 0.17026E+06, 0.18057E+06, 0.19137E+06, 0.20268E+06,
      0.21450E+06, 0.22687E+06, 0.23979E+06, 0.25328E+06, 0.26736E+06,
      0.28206E+06, 0.29738E+06, 0.31336E+06, 0.33000E+06, 0.34733E+06,
      0.36537E+06, 0.38414E+06, 0.40366E+06, 0.42396E+06, 0.44505E+06,
      0.46696E+06, 0.48971E+06, 0.51332E+06, 0.53782E+06, 0.56323E+06,
      0.58958E+06, 0.61689E+06, 0.64518E+06, 0.67448E+06, 0.70482E+06,
      0.73623E+06, 0.76872E+06, 0.80234E+06, 0.83710E+06, 0.87303E+06,
      0.91017E+06, 0.94853E+06, 0.98816E+06, 0.10291E+07, 0.10713E+07,
      0.11149E+07, 0.11599E+07, 0.12063E+07, 0.12541E+07, 0.13034E+07,
      0.13542E+07, 0.14066E+07, 0.14606E+07, 0.15161E+07, 0.15733E+07,
      0.16322E+07, 0.16928E+07, 0.17552E+07, 0.18194E+07, 0.18854E+07,
      0.19532E+07, 0.20230E+07, 0.20947E+07, 0.21684E+07, 0.22441E+07,
      0.23219E+07, 0.24018E+07, 0.24838E+07, 0.25680E+07, 0.26545E+07,
      0.27432E+07])


#  --------------- O3 667: M = 3, I = 4 ---------------------
M = 3
I = 4
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.37657E+04, 0.63472E+04, 0.93454E+04,
      0.12715E+05, 0.16435E+05, 0.20502E+05, 0.24929E+05, 0.29742E+05,
      0.34975E+05, 0.40668E+05, 0.46868E+05, 0.53624E+05, 0.60990E+05,
      0.69018E+05, 0.77768E+05, 0.87296E+05, 0.97666E+05, 0.10894E+06,
      0.12118E+06, 0.13446E+06, 0.14885E+06, 0.16441E+06, 0.18123E+06,
      0.19938E+06, 0.21894E+06, 0.23998E+06, 0.26261E+06, 0.28690E+06,
      0.31295E+06, 0.34084E+06, 0.37068E+06, 0.40256E+06, 0.43659E+06,
      0.47287E+06, 0.51151E+06, 0.55262E+06, 0.59632E+06, 0.64272E+06,
      0.69194E+06, 0.74412E+06, 0.79937E+06, 0.85783E+06, 0.91963E+06,
      0.98492E+06, 0.10538E+07, 0.11265E+07, 0.12031E+07, 0.12837E+07,
      0.13686E+07, 0.14579E+07, 0.15517E+07, 0.16502E+07, 0.17536E+07,
      0.18621E+07, 0.19758E+07, 0.20949E+07, 0.22196E+07, 0.23501E+07,
      0.24866E+07, 0.26292E+07, 0.27783E+07, 0.29339E+07, 0.30963E+07,
      0.32658E+07, 0.34425E+07, 0.36266E+07, 0.38184E+07, 0.40181E+07,
      0.42260E+07, 0.44422E+07, 0.46671E+07, 0.49008E+07, 0.51437E+07,
      0.53959E+07, 0.56578E+07, 0.59296E+07, 0.62116E+07, 0.65040E+07,
      0.68071E+07, 0.71213E+07, 0.74468E+07, 0.77838E+07, 0.81328E+07,
      0.84939E+07, 0.88676E+07, 0.92541E+07, 0.96536E+07, 0.10067E+08,
      0.10493E+08, 0.10934E+08, 0.11390E+08, 0.11860E+08, 0.12345E+08,
      0.12846E+08, 0.13363E+08, 0.13895E+08, 0.14445E+08, 0.15011E+08,
      0.15595E+08, 0.16196E+08, 0.16815E+08, 0.17453E+08, 0.18110E+08,
      0.18786E+08, 0.19482E+08, 0.20198E+08, 0.20934E+08, 0.21691E+08,
      0.22470E+08, 0.23270E+08, 0.24093E+08, 0.24939E+08, 0.25807E+08,
      0.26699E+08, 0.27616E+08, 0.28556E+08, 0.29522E+08, 0.30514E+08,
      0.31531E+08])


#  --------------- O3 676: M = 3, I = 5 ---------------------
M = 3
I = 5
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.18608E+04, 0.31363E+04, 0.46177E+04,
      0.62826E+04, 0.81202E+04, 0.10129E+05, 0.12316E+05, 0.14693E+05,
      0.17277E+05, 0.20089E+05, 0.23153E+05, 0.26492E+05, 0.30133E+05,
      0.34103E+05, 0.38430E+05, 0.43145E+05, 0.48277E+05, 0.53858E+05,
      0.59920E+05, 0.66497E+05, 0.73624E+05, 0.81336E+05, 0.89671E+05,
      0.98668E+05, 0.10836E+06, 0.11880E+06, 0.13002E+06, 0.14207E+06,
      0.15500E+06, 0.16884E+06, 0.18365E+06, 0.19947E+06, 0.21636E+06,
      0.23438E+06, 0.25356E+06, 0.27398E+06, 0.29568E+06, 0.31873E+06,
      0.34318E+06, 0.36911E+06, 0.39656E+06, 0.42561E+06, 0.45632E+06,
      0.48877E+06, 0.52302E+06, 0.55914E+06, 0.59722E+06, 0.63732E+06,
      0.67952E+06, 0.72390E+06, 0.77055E+06, 0.81954E+06, 0.87097E+06,
      0.92491E+06, 0.98146E+06, 0.10407E+07, 0.11027E+07, 0.11677E+07,
      0.12356E+07, 0.13066E+07, 0.13807E+07, 0.14582E+07, 0.15390E+07,
      0.16233E+07, 0.17113E+07, 0.18029E+07, 0.18984E+07, 0.19978E+07,
      0.21012E+07, 0.22089E+07, 0.23208E+07, 0.24372E+07, 0.25581E+07,
      0.26837E+07, 0.28141E+07, 0.29494E+07, 0.30898E+07, 0.32354E+07,
      0.33864E+07, 0.35428E+07, 0.37049E+07, 0.38728E+07, 0.40466E+07,
      0.42264E+07, 0.44125E+07, 0.46050E+07, 0.48040E+07, 0.50098E+07,
      0.52224E+07, 0.54420E+07, 0.56689E+07, 0.59031E+07, 0.61449E+07,
      0.63943E+07, 0.66517E+07, 0.69172E+07, 0.71909E+07, 0.74731E+07,
      0.77639E+07, 0.80635E+07, 0.83721E+07, 0.86900E+07, 0.90172E+07,
      0.93541E+07, 0.97008E+07, 0.10058E+08, 0.10424E+08, 0.10802E+08,
      0.11190E+08, 0.11589E+08, 0.11999E+08, 0.12420E+08, 0.12853E+08,
      0.13298E+08, 0.13755E+08, 0.14223E+08, 0.14705E+08, 0.15199E+08,
      0.15706E+08])


#  --------------- O3 886: M = 3, I = 6 ---------------------
M = 3
I = 6
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.67639E+03, 0.11401E+04, 0.16787E+04,
      0.22843E+04, 0.29532E+04, 0.36856E+04, 0.44842E+04, 0.53545E+04,
      0.63030E+04, 0.73381E+04, 0.84686E+04, 0.97040E+04, 0.11054E+05,
      0.12530E+05, 0.14143E+05, 0.15903E+05, 0.17823E+05, 0.19915E+05,
      0.22190E+05, 0.24663E+05, 0.27346E+05, 0.30254E+05, 0.33400E+05,
      0.36800E+05, 0.40469E+05, 0.44423E+05, 0.48678E+05, 0.53251E+05,
      0.58160E+05, 0.63423E+05, 0.69058E+05, 0.75085E+05, 0.81524E+05,
      0.88395E+05, 0.95719E+05, 0.10352E+06, 0.11181E+06, 0.12063E+06,
      0.12999E+06, 0.13991E+06, 0.15043E+06, 0.16157E+06, 0.17335E+06,
      0.18580E+06, 0.19895E+06, 0.21283E+06, 0.22746E+06, 0.24288E+06,
      0.25911E+06, 0.27619E+06, 0.29415E+06, 0.31301E+06, 0.33283E+06,
      0.35362E+06, 0.37542E+06, 0.39827E+06, 0.42221E+06, 0.44726E+06,
      0.47348E+06, 0.50089E+06, 0.52954E+06, 0.55947E+06, 0.59072E+06,
      0.62332E+06, 0.65733E+06, 0.69279E+06, 0.72973E+06, 0.76821E+06,
      0.80827E+06, 0.84996E+06, 0.89332E+06, 0.93840E+06, 0.98526E+06,
      0.10339E+07, 0.10845E+07, 0.11370E+07, 0.11914E+07, 0.12479E+07,
      0.13065E+07, 0.13672E+07, 0.14302E+07, 0.14953E+07, 0.15628E+07,
      0.16327E+07, 0.17050E+07, 0.17798E+07, 0.18571E+07, 0.19371E+07,
      0.20197E+07, 0.21051E+07, 0.21933E+07, 0.22844E+07, 0.23785E+07,
      0.24755E+07, 0.25757E+07, 0.26790E+07, 0.27855E+07, 0.28954E+07,
      0.30086E+07, 0.31253E+07, 0.32455E+07, 0.33693E+07, 0.34967E+07,
      0.36280E+07, 0.37631E+07, 0.39021E+07, 0.40451E+07, 0.41922E+07,
      0.43435E+07, 0.44990E+07, 0.46589E+07, 0.48232E+07, 0.49920E+07,
      0.51654E+07, 0.53436E+07, 0.55265E+07, 0.57143E+07, 0.59071E+07,
      0.61050E+07])


#  --------------- O3 868: M = 3, I = 7 ---------------------
M = 3
I = 7
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.34615E+03, 0.58348E+03, 0.85915E+03,
      0.11692E+04, 0.15117E+04, 0.18868E+04, 0.22960E+04, 0.27419E+04,
      0.32278E+04, 0.37579E+04, 0.43366E+04, 0.49686E+04, 0.56591E+04,
      0.64134E+04, 0.72369E+04, 0.81354E+04, 0.91148E+04, 0.10181E+05,
      0.11341E+05, 0.12600E+05, 0.13966E+05, 0.15446E+05, 0.17046E+05,
      0.18775E+05, 0.20640E+05, 0.22649E+05, 0.24810E+05, 0.27132E+05,
      0.29624E+05, 0.32295E+05, 0.35154E+05, 0.38211E+05, 0.41475E+05,
      0.44958E+05, 0.48670E+05, 0.52621E+05, 0.56823E+05, 0.61288E+05,
      0.66026E+05, 0.71052E+05, 0.76376E+05, 0.82011E+05, 0.87972E+05,
      0.94271E+05, 0.10092E+06, 0.10794E+06, 0.11534E+06, 0.12313E+06,
      0.13134E+06, 0.13997E+06, 0.14905E+06, 0.15858E+06, 0.16859E+06,
      0.17909E+06, 0.19010E+06, 0.20164E+06, 0.21373E+06, 0.22638E+06,
      0.23962E+06, 0.25346E+06, 0.26792E+06, 0.28302E+06, 0.29879E+06,
      0.31524E+06, 0.33240E+06, 0.35029E+06, 0.36892E+06, 0.38833E+06,
      0.40853E+06, 0.42956E+06, 0.45142E+06, 0.47416E+06, 0.49778E+06,
      0.52233E+06, 0.54781E+06, 0.57427E+06, 0.60172E+06, 0.63019E+06,
      0.65971E+06, 0.69031E+06, 0.72201E+06, 0.75485E+06, 0.78886E+06,
      0.82405E+06, 0.86048E+06, 0.89815E+06, 0.93711E+06, 0.97739E+06,
      0.10190E+07, 0.10620E+07, 0.11065E+07, 0.11523E+07, 0.11997E+07,
      0.12485E+07, 0.12990E+07, 0.13510E+07, 0.14046E+07, 0.14599E+07,
      0.15169E+07, 0.15756E+07, 0.16361E+07, 0.16984E+07, 0.17626E+07,
      0.18287E+07, 0.18966E+07, 0.19666E+07, 0.20386E+07, 0.21126E+07,
      0.21887E+07, 0.22669E+07, 0.23474E+07, 0.24300E+07, 0.25150E+07,
      0.26022E+07, 0.26919E+07, 0.27839E+07, 0.28784E+07, 0.29753E+07,
      0.30749E+07])


#  --------------- O3 678: M = 3, I = 8 ---------------------
M = 3
I = 8
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.39745E+04, 0.66993E+04, 0.98642E+04,
      0.13422E+05, 0.17352E+05, 0.21652E+05, 0.26339E+05, 0.31442E+05,
      0.37000E+05, 0.43058E+05, 0.49669E+05, 0.56885E+05, 0.64766E+05,
      0.73372E+05, 0.82765E+05, 0.93011E+05, 0.10418E+06, 0.11633E+06,
      0.12955E+06, 0.14390E+06, 0.15946E+06, 0.17632E+06, 0.19455E+06,
      0.21424E+06, 0.23547E+06, 0.25835E+06, 0.28296E+06, 0.30939E+06,
      0.33776E+06, 0.36816E+06, 0.40070E+06, 0.43549E+06, 0.47264E+06,
      0.51228E+06, 0.55451E+06, 0.59947E+06, 0.64728E+06, 0.69807E+06,
      0.75198E+06, 0.80915E+06, 0.86971E+06, 0.93381E+06, 0.10016E+07,
      0.10733E+07, 0.11489E+07, 0.12287E+07, 0.13128E+07, 0.14015E+07,
      0.14948E+07, 0.15930E+07, 0.16961E+07, 0.18045E+07, 0.19183E+07,
      0.20378E+07, 0.21629E+07, 0.22942E+07, 0.24316E+07, 0.25754E+07,
      0.27258E+07, 0.28831E+07, 0.30475E+07, 0.32192E+07, 0.33984E+07,
      0.35855E+07, 0.37805E+07, 0.39838E+07, 0.41956E+07, 0.44162E+07,
      0.46458E+07, 0.48847E+07, 0.51332E+07, 0.53916E+07, 0.56601E+07,
      0.59390E+07, 0.62286E+07, 0.65292E+07, 0.68412E+07, 0.71647E+07,
      0.75002E+07, 0.78479E+07, 0.82081E+07, 0.85813E+07, 0.89676E+07,
      0.93676E+07, 0.97814E+07, 0.10209E+08, 0.10652E+08, 0.11110E+08,
      0.11583E+08, 0.12071E+08, 0.12576E+08, 0.13097E+08, 0.13635E+08,
      0.14190E+08, 0.14763E+08, 0.15354E+08, 0.15963E+08, 0.16592E+08,
      0.17239E+08, 0.17906E+08, 0.18593E+08, 0.19301E+08, 0.20030E+08,
      0.20780E+08, 0.21553E+08, 0.22347E+08, 0.23165E+08, 0.24006E+08,
      0.24870E+08, 0.25759E+08, 0.26673E+08, 0.27612E+08, 0.28577E+08,
      0.29568E+08, 0.30585E+08, 0.31631E+08, 0.32704E+08, 0.33805E+08,
      0.34936E+08])


#  --------------- O3 768: M = 3, I = 9 ---------------------
M = 3
I = 9
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.40228E+04, 0.67808E+04, 0.99842E+04,
      0.13586E+05, 0.17564E+05, 0.21919E+05, 0.26665E+05, 0.31833E+05,
      0.37461E+05, 0.43596E+05, 0.50286E+05, 0.57589E+05, 0.65562E+05,
      0.74264E+05, 0.83761E+05, 0.94115E+05, 0.10540E+06, 0.11767E+06,
      0.13102E+06, 0.14550E+06, 0.16121E+06, 0.17822E+06, 0.19661E+06,
      0.21646E+06, 0.23788E+06, 0.26094E+06, 0.28574E+06, 0.31239E+06,
      0.34097E+06, 0.37160E+06, 0.40437E+06, 0.43941E+06, 0.47683E+06,
      0.51673E+06, 0.55925E+06, 0.60451E+06, 0.65262E+06, 0.70374E+06,
      0.75799E+06, 0.81550E+06, 0.87643E+06, 0.94092E+06, 0.10091E+07,
      0.10812E+07, 0.11572E+07, 0.12375E+07, 0.13221E+07, 0.14112E+07,
      0.15050E+07, 0.16037E+07, 0.17074E+07, 0.18164E+07, 0.19307E+07,
      0.20507E+07, 0.21765E+07, 0.23084E+07, 0.24464E+07, 0.25909E+07,
      0.27421E+07, 0.29001E+07, 0.30652E+07, 0.32377E+07, 0.34177E+07,
      0.36055E+07, 0.38014E+07, 0.40055E+07, 0.42182E+07, 0.44397E+07,
      0.46703E+07, 0.49102E+07, 0.51597E+07, 0.54191E+07, 0.56886E+07,
      0.59686E+07, 0.62593E+07, 0.65611E+07, 0.68742E+07, 0.71989E+07,
      0.75356E+07, 0.78846E+07, 0.82461E+07, 0.86206E+07, 0.90083E+07,
      0.94097E+07, 0.98249E+07, 0.10254E+08, 0.10699E+08, 0.11158E+08,
      0.11632E+08, 0.12123E+08, 0.12629E+08, 0.13152E+08, 0.13691E+08,
      0.14248E+08, 0.14823E+08, 0.15416E+08, 0.16027E+08, 0.16657E+08,
      0.17307E+08, 0.17976E+08, 0.18665E+08, 0.19375E+08, 0.20106E+08,
      0.20858E+08, 0.21633E+08, 0.22430E+08, 0.23250E+08, 0.24093E+08,
      0.24960E+08, 0.25851E+08, 0.26767E+08, 0.27709E+08, 0.28676E+08,
      0.29670E+08, 0.30691E+08, 0.31739E+08, 0.32815E+08, 0.33919E+08,
      0.35053E+08])


#  --------------- O3 786: M = 3, I = 10 ---------------------
M = 3
I = 10
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.39315E+04, 0.66267E+04, 0.97569E+04,
      0.13276E+05, 0.17162E+05, 0.21414E+05, 0.26048E+05, 0.31094E+05,
      0.36590E+05, 0.42581E+05, 0.49120E+05, 0.56260E+05, 0.64061E+05,
      0.72580E+05, 0.81882E+05, 0.92031E+05, 0.10309E+06, 0.11514E+06,
      0.12824E+06, 0.14247E+06, 0.15791E+06, 0.17463E+06, 0.19272E+06,
      0.21226E+06, 0.23333E+06, 0.25604E+06, 0.28047E+06, 0.30673E+06,
      0.33490E+06, 0.36510E+06, 0.39743E+06, 0.43200E+06, 0.46892E+06,
      0.50831E+06, 0.55029E+06, 0.59498E+06, 0.64251E+06, 0.69301E+06,
      0.74662E+06, 0.80347E+06, 0.86370E+06, 0.92747E+06, 0.99491E+06,
      0.10662E+07, 0.11414E+07, 0.12208E+07, 0.13046E+07, 0.13928E+07,
      0.14856E+07, 0.15833E+07, 0.16860E+07, 0.17939E+07, 0.19072E+07,
      0.20261E+07, 0.21508E+07, 0.22814E+07, 0.24182E+07, 0.25614E+07,
      0.27112E+07, 0.28679E+07, 0.30316E+07, 0.32026E+07, 0.33811E+07,
      0.35674E+07, 0.37617E+07, 0.39642E+07, 0.41752E+07, 0.43950E+07,
      0.46237E+07, 0.48618E+07, 0.51094E+07, 0.53668E+07, 0.56343E+07,
      0.59123E+07, 0.62009E+07, 0.65005E+07, 0.68113E+07, 0.71338E+07,
      0.74681E+07, 0.78147E+07, 0.81737E+07, 0.85457E+07, 0.89308E+07,
      0.93295E+07, 0.97420E+07, 0.10169E+08, 0.10610E+08, 0.11066E+08,
      0.11538E+08, 0.12025E+08, 0.12528E+08, 0.13048E+08, 0.13584E+08,
      0.14138E+08, 0.14709E+08, 0.15298E+08, 0.15906E+08, 0.16532E+08,
      0.17178E+08, 0.17843E+08, 0.18528E+08, 0.19234E+08, 0.19961E+08,
      0.20710E+08, 0.21480E+08, 0.22272E+08, 0.23088E+08, 0.23926E+08,
      0.24789E+08, 0.25675E+08, 0.26587E+08, 0.27523E+08, 0.28485E+08,
      0.29474E+08, 0.30489E+08, 0.31532E+08, 0.32603E+08, 0.33701E+08,
      0.34829E+08])


#  --------------- O3 776: M = 3, I = 11 ---------------------
M = 3
I = 11
TIPS_GSI_HASH[(M,I)] = __FloatType__(36.)
TIPS_ISO_HASH[(M,I)] = float32([0.23106E+05, 0.38945E+05, 0.57342E+05,
      0.78021E+05, 0.10085E+06, 0.12582E+06, 0.15302E+06, 0.18262E+06,
      0.21482E+06, 0.24989E+06, 0.28812E+06, 0.32983E+06, 0.37535E+06,
      0.42501E+06, 0.47919E+06, 0.53825E+06, 0.60258E+06, 0.67256E+06,
      0.74862E+06, 0.83118E+06, 0.92069E+06, 0.10176E+07, 0.11223E+07,
      0.12354E+07, 0.13574E+07, 0.14887E+07, 0.16299E+07, 0.17816E+07,
      0.19443E+07, 0.21187E+07, 0.23052E+07, 0.25047E+07, 0.27176E+07,
      0.29447E+07, 0.31866E+07, 0.34441E+07, 0.37179E+07, 0.40087E+07,
      0.43173E+07, 0.46444E+07, 0.49910E+07, 0.53578E+07, 0.57456E+07,
      0.61554E+07, 0.65880E+07, 0.70444E+07, 0.75255E+07, 0.80322E+07,
      0.85656E+07, 0.91266E+07, 0.97163E+07, 0.10336E+08, 0.10986E+08,
      0.11668E+08, 0.12383E+08, 0.13133E+08, 0.13918E+08, 0.14739E+08,
      0.15598E+08, 0.16496E+08, 0.17435E+08, 0.18415E+08, 0.19438E+08,
      0.20505E+08, 0.21619E+08, 0.22779E+08, 0.23987E+08, 0.25246E+08,
      0.26556E+08, 0.27920E+08, 0.29337E+08, 0.30811E+08, 0.32343E+08,
      0.33934E+08, 0.35585E+08, 0.37300E+08, 0.39079E+08, 0.40924E+08,
      0.42837E+08, 0.44819E+08, 0.46873E+08, 0.49001E+08, 0.51203E+08,
      0.53483E+08, 0.55842E+08, 0.58282E+08, 0.60805E+08, 0.63414E+08,
      0.66109E+08, 0.68894E+08, 0.71770E+08, 0.74740E+08, 0.77806E+08,
      0.80970E+08, 0.84234E+08, 0.87600E+08, 0.91072E+08, 0.94651E+08,
      0.98339E+08, 0.10214E+09, 0.10605E+09, 0.11009E+09, 0.11424E+09,
      0.11851E+09, 0.12291E+09, 0.12744E+09, 0.13209E+09, 0.13688E+09,
      0.14180E+09, 0.14687E+09, 0.15207E+09, 0.15742E+09, 0.16291E+09,
      0.16855E+09, 0.17435E+09, 0.18030E+09, 0.18641E+09, 0.19268E+09,
      0.19912E+09])


#  --------------- O3 767: M = 3, I = 12 ---------------------
M = 3
I = 12
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.11692E+05, 0.19707E+05, 0.29017E+05,
      0.39482E+05, 0.51038E+05, 0.63680E+05, 0.77450E+05, 0.92432E+05,
      0.10873E+06, 0.12649E+06, 0.14584E+06, 0.16694E+06, 0.18996E+06,
      0.21507E+06, 0.24245E+06, 0.27229E+06, 0.30478E+06, 0.34013E+06,
      0.37853E+06, 0.42020E+06, 0.46536E+06, 0.51424E+06, 0.56708E+06,
      0.62411E+06, 0.68559E+06, 0.75178E+06, 0.82296E+06, 0.89939E+06,
      0.98137E+06, 0.10692E+07, 0.11631E+07, 0.12636E+07, 0.13708E+07,
      0.14851E+07, 0.16069E+07, 0.17365E+07, 0.18742E+07, 0.20206E+07,
      0.21758E+07, 0.23404E+07, 0.25148E+07, 0.26992E+07, 0.28943E+07,
      0.31004E+07, 0.33179E+07, 0.35474E+07, 0.37892E+07, 0.40440E+07,
      0.43121E+07, 0.45940E+07, 0.48904E+07, 0.52017E+07, 0.55285E+07,
      0.58713E+07, 0.62306E+07, 0.66071E+07, 0.70014E+07, 0.74140E+07,
      0.78456E+07, 0.82967E+07, 0.87681E+07, 0.92604E+07, 0.97742E+07,
      0.10310E+08, 0.10869E+08, 0.11452E+08, 0.12059E+08, 0.12691E+08,
      0.13348E+08, 0.14033E+08, 0.14745E+08, 0.15484E+08, 0.16253E+08,
      0.17052E+08, 0.17881E+08, 0.18741E+08, 0.19634E+08, 0.20560E+08,
      0.21520E+08, 0.22515E+08, 0.23546E+08, 0.24613E+08, 0.25718E+08,
      0.26862E+08, 0.28046E+08, 0.29270E+08, 0.30536E+08, 0.31845E+08,
      0.33197E+08, 0.34594E+08, 0.36037E+08, 0.37527E+08, 0.39065E+08,
      0.40652E+08, 0.42289E+08, 0.43977E+08, 0.45719E+08, 0.47514E+08,
      0.49363E+08, 0.51270E+08, 0.53233E+08, 0.55255E+08, 0.57337E+08,
      0.59480E+08, 0.61686E+08, 0.63956E+08, 0.66290E+08, 0.68691E+08,
      0.71160E+08, 0.73699E+08, 0.76307E+08, 0.78988E+08, 0.81743E+08,
      0.84572E+08, 0.87478E+08, 0.90462E+08, 0.93525E+08, 0.96669E+08,
      0.99896E+08])


#  --------------- O3 888: M = 3, I = 13 ---------------------
M = 3
I = 13
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.36175E+03, 0.60978E+03, 0.89790E+03,
      0.12219E+04, 0.15802E+04, 0.19728E+04, 0.24016E+04, 0.28696E+04,
      0.33807E+04, 0.39394E+04, 0.45506E+04, 0.52196E+04, 0.59521E+04,
      0.67538E+04, 0.76308E+04, 0.85894E+04, 0.96361E+04, 0.10777E+05,
      0.12021E+05, 0.13373E+05, 0.14841E+05, 0.16434E+05, 0.18158E+05,
      0.20023E+05, 0.22037E+05, 0.24208E+05, 0.26547E+05, 0.29061E+05,
      0.31762E+05, 0.34659E+05, 0.37762E+05, 0.41083E+05, 0.44632E+05,
      0.48421E+05, 0.52462E+05, 0.56766E+05, 0.61346E+05, 0.66215E+05,
      0.71386E+05, 0.76873E+05, 0.82688E+05, 0.88848E+05, 0.95365E+05,
      0.10226E+06, 0.10954E+06, 0.11722E+06, 0.12532E+06, 0.13387E+06,
      0.14286E+06, 0.15233E+06, 0.16229E+06, 0.17275E+06, 0.18374E+06,
      0.19528E+06, 0.20737E+06, 0.22006E+06, 0.23335E+06, 0.24726E+06,
      0.26182E+06, 0.27705E+06, 0.29297E+06, 0.30960E+06, 0.32696E+06,
      0.34509E+06, 0.36399E+06, 0.38371E+06, 0.40425E+06, 0.42566E+06,
      0.44794E+06, 0.47114E+06, 0.49527E+06, 0.52036E+06, 0.54644E+06,
      0.57354E+06, 0.60169E+06, 0.63091E+06, 0.66124E+06, 0.69270E+06,
      0.72533E+06, 0.75916E+06, 0.79421E+06, 0.83053E+06, 0.86814E+06,
      0.90708E+06, 0.94737E+06, 0.98907E+06, 0.10322E+07, 0.10768E+07,
      0.11229E+07, 0.11705E+07, 0.12197E+07, 0.12705E+07, 0.13230E+07,
      0.13771E+07, 0.14330E+07, 0.14906E+07, 0.15501E+07, 0.16114E+07,
      0.16745E+07, 0.17397E+07, 0.18067E+07, 0.18759E+07, 0.19470E+07,
      0.20203E+07, 0.20957E+07, 0.21733E+07, 0.22532E+07, 0.23353E+07,
      0.24198E+07, 0.25067E+07, 0.25960E+07, 0.26878E+07, 0.27821E+07,
      0.28790E+07, 0.29785E+07, 0.30807E+07, 0.31857E+07, 0.32934E+07,
      0.34040E+07])


#  --------------- O3 887: M = 3, I = 14 ---------------------
M = 3
I = 14
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.42000E+04, 0.70796E+04, 0.10424E+05,
      0.14186E+05, 0.18342E+05, 0.22896E+05, 0.27866E+05, 0.33285E+05,
      0.39199E+05, 0.45659E+05, 0.52720E+05, 0.60444E+05, 0.68895E+05,
      0.78139E+05, 0.88246E+05, 0.99288E+05, 0.11134E+06, 0.12447E+06,
      0.13877E+06, 0.15431E+06, 0.17119E+06, 0.18949E+06, 0.20930E+06,
      0.23071E+06, 0.25383E+06, 0.27875E+06, 0.30558E+06, 0.33442E+06,
      0.36539E+06, 0.39861E+06, 0.43418E+06, 0.47224E+06, 0.51291E+06,
      0.55632E+06, 0.60260E+06, 0.65189E+06, 0.70434E+06, 0.76008E+06,
      0.81927E+06, 0.88206E+06, 0.94862E+06, 0.10191E+07, 0.10937E+07,
      0.11725E+07, 0.12558E+07, 0.13436E+07, 0.14363E+07, 0.15340E+07,
      0.16368E+07, 0.17450E+07, 0.18588E+07, 0.19784E+07, 0.21040E+07,
      0.22358E+07, 0.23741E+07, 0.25190E+07, 0.26708E+07, 0.28297E+07,
      0.29961E+07, 0.31700E+07, 0.33518E+07, 0.35417E+07, 0.37400E+07,
      0.39469E+07, 0.41628E+07, 0.43878E+07, 0.46224E+07, 0.48667E+07,
      0.51210E+07, 0.53858E+07, 0.56611E+07, 0.59475E+07, 0.62451E+07,
      0.65544E+07, 0.68755E+07, 0.72089E+07, 0.75550E+07, 0.79139E+07,
      0.82861E+07, 0.86720E+07, 0.90719E+07, 0.94861E+07, 0.99151E+07,
      0.10359E+08, 0.10819E+08, 0.11294E+08, 0.11786E+08, 0.12294E+08,
      0.12820E+08, 0.13363E+08, 0.13924E+08, 0.14503E+08, 0.15101E+08,
      0.15719E+08, 0.16356E+08, 0.17013E+08, 0.17690E+08, 0.18389E+08,
      0.19109E+08, 0.19851E+08, 0.20616E+08, 0.21404E+08, 0.22215E+08,
      0.23050E+08, 0.23910E+08, 0.24794E+08, 0.25704E+08, 0.26640E+08,
      0.27603E+08, 0.28593E+08, 0.29610E+08, 0.30656E+08, 0.31731E+08,
      0.32835E+08, 0.33969E+08, 0.35133E+08, 0.36329E+08, 0.37556E+08,
      0.38816E+08])


#  --------------- O3 878: M = 3, I = 15 ---------------------
M = 3
I = 15
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.21250E+04, 0.35820E+04, 0.52744E+04,
      0.71778E+04, 0.92814E+04, 0.11586E+05, 0.14102E+05, 0.16845E+05,
      0.19839E+05, 0.23108E+05, 0.26680E+05, 0.30588E+05, 0.34861E+05,
      0.39534E+05, 0.44642E+05, 0.50219E+05, 0.56305E+05, 0.62937E+05,
      0.70155E+05, 0.78001E+05, 0.86516E+05, 0.95747E+05, 0.10574E+06,
      0.11653E+06, 0.12819E+06, 0.14075E+06, 0.15427E+06, 0.16881E+06,
      0.18441E+06, 0.20114E+06, 0.21906E+06, 0.23823E+06, 0.25871E+06,
      0.28056E+06, 0.30386E+06, 0.32867E+06, 0.35507E+06, 0.38312E+06,
      0.41291E+06, 0.44450E+06, 0.47799E+06, 0.51344E+06, 0.55095E+06,
      0.59060E+06, 0.63248E+06, 0.67667E+06, 0.72327E+06, 0.77238E+06,
      0.82409E+06, 0.87850E+06, 0.93571E+06, 0.99583E+06, 0.10590E+07,
      0.11252E+07, 0.11947E+07, 0.12675E+07, 0.13438E+07, 0.14237E+07,
      0.15072E+07, 0.15946E+07, 0.16859E+07, 0.17814E+07, 0.18810E+07,
      0.19849E+07, 0.20934E+07, 0.22064E+07, 0.23242E+07, 0.24469E+07,
      0.25747E+07, 0.27076E+07, 0.28459E+07, 0.29897E+07, 0.31391E+07,
      0.32944E+07, 0.34557E+07, 0.36231E+07, 0.37968E+07, 0.39770E+07,
      0.41639E+07, 0.43576E+07, 0.45583E+07, 0.47663E+07, 0.49816E+07,
      0.52045E+07, 0.54352E+07, 0.56739E+07, 0.59207E+07, 0.61759E+07,
      0.64396E+07, 0.67121E+07, 0.69936E+07, 0.72844E+07, 0.75845E+07,
      0.78943E+07, 0.82139E+07, 0.85436E+07, 0.88837E+07, 0.92342E+07,
      0.95956E+07, 0.99680E+07, 0.10352E+08, 0.10747E+08, 0.11154E+08,
      0.11573E+08, 0.12004E+08, 0.12448E+08, 0.12904E+08, 0.13374E+08,
      0.13857E+08, 0.14353E+08, 0.14864E+08, 0.15388E+08, 0.15927E+08,
      0.16481E+08, 0.17050E+08, 0.17634E+08, 0.18234E+08, 0.18849E+08,
      0.19481E+08])


#  --------------- O3 778: M = 3, I = 16 ---------------------
M = 3
I = 16
TIPS_GSI_HASH[(M,I)] = __FloatType__(36.)
TIPS_ISO_HASH[(M,I)] = float32([0.24692E+05, 0.41621E+05, 0.61284E+05,
      0.83394E+05, 0.10782E+06, 0.13457E+06, 0.16375E+06, 0.19554E+06,
      0.23020E+06, 0.26801E+06, 0.30930E+06, 0.35443E+06, 0.40375E+06,
      0.45763E+06, 0.51650E+06, 0.58075E+06, 0.65080E+06, 0.72711E+06,
      0.81012E+06, 0.90030E+06, 0.99815E+06, 0.11042E+07, 0.12189E+07,
      0.13428E+07, 0.14765E+07, 0.16206E+07, 0.17757E+07, 0.19423E+07,
      0.21212E+07, 0.23129E+07, 0.25181E+07, 0.27377E+07, 0.29721E+07,
      0.32223E+07, 0.34890E+07, 0.37729E+07, 0.40750E+07, 0.43959E+07,
      0.47365E+07, 0.50978E+07, 0.54807E+07, 0.58860E+07, 0.63147E+07,
      0.67678E+07, 0.72463E+07, 0.77512E+07, 0.82836E+07, 0.88445E+07,
      0.94351E+07, 0.10056E+08, 0.10710E+08, 0.11396E+08, 0.12117E+08,
      0.12873E+08, 0.13666E+08, 0.14497E+08, 0.15367E+08, 0.16279E+08,
      0.17232E+08, 0.18229E+08, 0.19271E+08, 0.20359E+08, 0.21495E+08,
      0.22681E+08, 0.23917E+08, 0.25206E+08, 0.26549E+08, 0.27948E+08,
      0.29404E+08, 0.30920E+08, 0.32496E+08, 0.34135E+08, 0.35838E+08,
      0.37608E+08, 0.39445E+08, 0.41353E+08, 0.43332E+08, 0.45385E+08,
      0.47514E+08, 0.49721E+08, 0.52007E+08, 0.54376E+08, 0.56829E+08,
      0.59367E+08, 0.61995E+08, 0.64712E+08, 0.67523E+08, 0.70429E+08,
      0.73432E+08, 0.76535E+08, 0.79740E+08, 0.83050E+08, 0.86467E+08,
      0.89993E+08, 0.93632E+08, 0.97385E+08, 0.10126E+09, 0.10525E+09,
      0.10936E+09, 0.11360E+09, 0.11796E+09, 0.12246E+09, 0.12709E+09,
      0.13186E+09, 0.13677E+09, 0.14182E+09, 0.14701E+09, 0.15236E+09,
      0.15785E+09, 0.16350E+09, 0.16931E+09, 0.17528E+09, 0.18141E+09,
      0.18771E+09, 0.19418E+09, 0.20082E+09, 0.20764E+09, 0.21465E+09,
      0.22183E+09])


#  --------------- O3 787: M = 3, I = 17 ---------------------
M = 3
I = 17
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.12211E+05, 0.20582E+05, 0.30305E+05,
      0.41237E+05, 0.53314E+05, 0.66536E+05, 0.80957E+05, 0.96672E+05,
      0.11380E+06, 0.13250E+06, 0.15292E+06, 0.17524E+06, 0.19965E+06,
      0.22632E+06, 0.25546E+06, 0.28728E+06, 0.32199E+06, 0.35980E+06,
      0.40094E+06, 0.44565E+06, 0.49417E+06, 0.54676E+06, 0.60366E+06,
      0.66516E+06, 0.73152E+06, 0.80305E+06, 0.88002E+06, 0.96276E+06,
      0.10516E+07, 0.11468E+07, 0.12488E+07, 0.13578E+07, 0.14743E+07,
      0.15987E+07, 0.17312E+07, 0.18723E+07, 0.20225E+07, 0.21820E+07,
      0.23514E+07, 0.25310E+07, 0.27214E+07, 0.29230E+07, 0.31362E+07,
      0.33616E+07, 0.35997E+07, 0.38509E+07, 0.41158E+07, 0.43949E+07,
      0.46887E+07, 0.49980E+07, 0.53231E+07, 0.56647E+07, 0.60234E+07,
      0.63998E+07, 0.67946E+07, 0.72084E+07, 0.76418E+07, 0.80955E+07,
      0.85702E+07, 0.90666E+07, 0.95854E+07, 0.10127E+08, 0.10693E+08,
      0.11284E+08, 0.11900E+08, 0.12542E+08, 0.13211E+08, 0.13907E+08,
      0.14633E+08, 0.15388E+08, 0.16173E+08, 0.16990E+08, 0.17838E+08,
      0.18720E+08, 0.19636E+08, 0.20586E+08, 0.21573E+08, 0.22596E+08,
      0.23657E+08, 0.24757E+08, 0.25896E+08, 0.27077E+08, 0.28299E+08,
      0.29565E+08, 0.30874E+08, 0.32229E+08, 0.33630E+08, 0.35079E+08,
      0.36576E+08, 0.38123E+08, 0.39721E+08, 0.41371E+08, 0.43075E+08,
      0.44833E+08, 0.46647E+08, 0.48518E+08, 0.50448E+08, 0.52438E+08,
      0.54489E+08, 0.56603E+08, 0.58780E+08, 0.61023E+08, 0.63332E+08,
      0.65710E+08, 0.68157E+08, 0.70676E+08, 0.73266E+08, 0.75931E+08,
      0.78672E+08, 0.81490E+08, 0.84386E+08, 0.87363E+08, 0.90422E+08,
      0.93564E+08, 0.96791E+08, 0.10011E+09, 0.10351E+09, 0.10700E+09,
      0.11059E+09])


#  --------------- O3 777: M = 3, I = 18 ---------------------
M = 3
I = 18
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.71750E+05, 0.12094E+06, 0.17807E+06,
      0.24230E+06, 0.31324E+06, 0.39088E+06, 0.47550E+06, 0.56764E+06,
      0.66800E+06, 0.77740E+06, 0.89677E+06, 0.10271E+07, 0.11694E+07,
      0.13249E+07, 0.14945E+07, 0.16796E+07, 0.18813E+07, 0.21009E+07,
      0.23396E+07, 0.25989E+07, 0.28801E+07, 0.31847E+07, 0.35140E+07,
      0.38698E+07, 0.42535E+07, 0.46669E+07, 0.51115E+07, 0.55893E+07,
      0.61019E+07, 0.66513E+07, 0.72393E+07, 0.78680E+07, 0.85395E+07,
      0.92558E+07, 0.10019E+08, 0.10832E+08, 0.11696E+08, 0.12614E+08,
      0.13588E+08, 0.14621E+08, 0.15716E+08, 0.16875E+08, 0.18100E+08,
      0.19395E+08, 0.20762E+08, 0.22205E+08, 0.23726E+08, 0.25328E+08,
      0.27015E+08, 0.28789E+08, 0.30654E+08, 0.32614E+08, 0.34671E+08,
      0.36830E+08, 0.39093E+08, 0.41465E+08, 0.43949E+08, 0.46549E+08,
      0.49269E+08, 0.52112E+08, 0.55084E+08, 0.58188E+08, 0.61428E+08,
      0.64809E+08, 0.68335E+08, 0.72010E+08, 0.75840E+08, 0.79828E+08,
      0.83979E+08, 0.88299E+08, 0.92792E+08, 0.97463E+08, 0.10232E+09,
      0.10736E+09, 0.11260E+09, 0.11803E+09, 0.12367E+09, 0.12952E+09,
      0.13559E+09, 0.14187E+09, 0.14839E+09, 0.15513E+09, 0.16212E+09,
      0.16935E+09, 0.17683E+09, 0.18457E+09, 0.19257E+09, 0.20085E+09,
      0.20940E+09, 0.21824E+09, 0.22736E+09, 0.23678E+09, 0.24651E+09,
      0.25655E+09, 0.26691E+09, 0.27759E+09, 0.28861E+09, 0.29997E+09,
      0.31167E+09, 0.32374E+09, 0.33616E+09, 0.34896E+09, 0.36214E+09,
      0.37571E+09, 0.38967E+09, 0.40404E+09, 0.41882E+09, 0.43403E+09,
      0.44966E+09, 0.46573E+09, 0.48226E+09, 0.49923E+09, 0.51668E+09,
      0.53460E+09, 0.55301E+09, 0.57191E+09, 0.59131E+09, 0.61123E+09,
      0.63167E+09])


#  --------------- N2O 446: M = 4, I = 1 ---------------------
M = 4
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(9.)
TIPS_ISO_HASH[(M,I)] = float32([0.89943E+03, 0.12734E+04, 0.16489E+04,
      0.20293E+04, 0.24205E+04, 0.28289E+04, 0.32609E+04, 0.37222E+04,
      0.42180E+04, 0.47529E+04, 0.53312E+04, 0.59572E+04, 0.66348E+04,
      0.73683E+04, 0.81616E+04, 0.90190E+04, 0.99450E+04, 0.10944E+05,
      0.12021E+05, 0.13180E+05, 0.14426E+05, 0.15766E+05, 0.17203E+05,
      0.18745E+05, 0.20396E+05, 0.22162E+05, 0.24051E+05, 0.26069E+05,
      0.28222E+05, 0.30517E+05, 0.32962E+05, 0.35564E+05, 0.38331E+05,
      0.41271E+05, 0.44393E+05, 0.47704E+05, 0.51214E+05, 0.54932E+05,
      0.58868E+05, 0.63030E+05, 0.67429E+05, 0.72075E+05, 0.76979E+05,
      0.82151E+05, 0.87604E+05, 0.93348E+05, 0.99395E+05, 0.10576E+06,
      0.11245E+06, 0.11948E+06, 0.12686E+06, 0.13461E+06, 0.14275E+06,
      0.15128E+06, 0.16021E+06, 0.16958E+06, 0.17938E+06, 0.18964E+06,
      0.20037E+06, 0.21159E+06, 0.22331E+06, 0.23556E+06, 0.24834E+06,
      0.26169E+06, 0.27561E+06, 0.29012E+06, 0.30525E+06, 0.32101E+06,
      0.33743E+06, 0.35452E+06, 0.37230E+06, 0.39080E+06, 0.41004E+06,
      0.43004E+06, 0.45082E+06, 0.47241E+06, 0.49483E+06, 0.51810E+06,
      0.54225E+06, 0.56730E+06, 0.59329E+06, 0.62022E+06, 0.64814E+06,
      0.67707E+06, 0.70703E+06, 0.73806E+06, 0.77018E+06, 0.80342E+06,
      0.83781E+06, 0.87338E+06, 0.91016E+06, 0.94818E+06, 0.98748E+06,
      0.10281E+07, 0.10700E+07, 0.11133E+07, 0.11581E+07, 0.12042E+07,
      0.12519E+07, 0.13010E+07, 0.13517E+07, 0.14040E+07, 0.14579E+07,
      0.15134E+07, 0.15707E+07, 0.16297E+07, 0.16905E+07, 0.17530E+07,
      0.18175E+07, 0.18838E+07, 0.19521E+07, 0.20224E+07, 0.20947E+07,
      0.21690E+07, 0.22455E+07, 0.23242E+07, 0.24050E+07, 0.24881E+07,
      0.25735E+07])


#  --------------- N2O 456: M = 4, I = 2 ---------------------
M = 4
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.59966E+03, 0.84903E+03, 0.10995E+04,
      0.13538E+04, 0.16158E+04, 0.18903E+04, 0.21815E+04, 0.24934E+04,
      0.28295E+04, 0.31927E+04, 0.35862E+04, 0.40128E+04, 0.44752E+04,
      0.49763E+04, 0.55189E+04, 0.61059E+04, 0.67404E+04, 0.74256E+04,
      0.81646E+04, 0.89609E+04, 0.98180E+04, 0.10740E+05, 0.11729E+05,
      0.12791E+05, 0.13930E+05, 0.15149E+05, 0.16453E+05, 0.17847E+05,
      0.19335E+05, 0.20922E+05, 0.22614E+05, 0.24416E+05, 0.26333E+05,
      0.28371E+05, 0.30535E+05, 0.32833E+05, 0.35269E+05, 0.37851E+05,
      0.40585E+05, 0.43478E+05, 0.46537E+05, 0.49769E+05, 0.53182E+05,
      0.56783E+05, 0.60580E+05, 0.64582E+05, 0.68796E+05, 0.73232E+05,
      0.77898E+05, 0.82803E+05, 0.87957E+05, 0.93369E+05, 0.99048E+05,
      0.10501E+06, 0.11125E+06, 0.11780E+06, 0.12465E+06, 0.13182E+06,
      0.13933E+06, 0.14718E+06, 0.15539E+06, 0.16396E+06, 0.17291E+06,
      0.18226E+06, 0.19201E+06, 0.20218E+06, 0.21278E+06, 0.22383E+06,
      0.23534E+06, 0.24733E+06, 0.25980E+06, 0.27278E+06, 0.28628E+06,
      0.30032E+06, 0.31491E+06, 0.33007E+06, 0.34581E+06, 0.36216E+06,
      0.37912E+06, 0.39673E+06, 0.41499E+06, 0.43392E+06, 0.45355E+06,
      0.47389E+06, 0.49496E+06, 0.51678E+06, 0.53937E+06, 0.56276E+06,
      0.58695E+06, 0.61199E+06, 0.63788E+06, 0.66464E+06, 0.69231E+06,
      0.72090E+06, 0.75044E+06, 0.78094E+06, 0.81244E+06, 0.84496E+06,
      0.87853E+06, 0.91316E+06, 0.94889E+06, 0.98573E+06, 0.10237E+07,
      0.10629E+07, 0.11033E+07, 0.11449E+07, 0.11877E+07, 0.12319E+07,
      0.12773E+07, 0.13241E+07, 0.13723E+07, 0.14219E+07, 0.14729E+07,
      0.15254E+07, 0.15793E+07, 0.16349E+07, 0.16919E+07, 0.17506E+07,
      0.18109E+07])


#  --------------- N2O 546: M = 4, I = 3 ---------------------
M = 4
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.62051E+03, 0.87856E+03, 0.11377E+04,
      0.14003E+04, 0.16705E+04, 0.19529E+04, 0.22518E+04, 0.25713E+04,
      0.29149E+04, 0.32859E+04, 0.36873E+04, 0.41220E+04, 0.45929E+04,
      0.51028E+04, 0.56547E+04, 0.62515E+04, 0.68963E+04, 0.75923E+04,
      0.83428E+04, 0.91511E+04, 0.10021E+05, 0.10956E+05, 0.11960E+05,
      0.13036E+05, 0.14190E+05, 0.15425E+05, 0.16746E+05, 0.18158E+05,
      0.19664E+05, 0.21271E+05, 0.22984E+05, 0.24806E+05, 0.26745E+05,
      0.28806E+05, 0.30995E+05, 0.33317E+05, 0.35780E+05, 0.38389E+05,
      0.41151E+05, 0.44073E+05, 0.47162E+05, 0.50425E+05, 0.53871E+05,
      0.57505E+05, 0.61338E+05, 0.65375E+05, 0.69628E+05, 0.74102E+05,
      0.78808E+05, 0.83755E+05, 0.88951E+05, 0.94407E+05, 0.10013E+06,
      0.10614E+06, 0.11243E+06, 0.11902E+06, 0.12593E+06, 0.13316E+06,
      0.14072E+06, 0.14862E+06, 0.15689E+06, 0.16552E+06, 0.17453E+06,
      0.18394E+06, 0.19376E+06, 0.20399E+06, 0.21466E+06, 0.22578E+06,
      0.23737E+06, 0.24942E+06, 0.26198E+06, 0.27503E+06, 0.28861E+06,
      0.30273E+06, 0.31741E+06, 0.33265E+06, 0.34848E+06, 0.36492E+06,
      0.38197E+06, 0.39967E+06, 0.41803E+06, 0.43706E+06, 0.45679E+06,
      0.47723E+06, 0.49840E+06, 0.52033E+06, 0.54303E+06, 0.56653E+06,
      0.59084E+06, 0.61599E+06, 0.64200E+06, 0.66888E+06, 0.69667E+06,
      0.72539E+06, 0.75506E+06, 0.78569E+06, 0.81733E+06, 0.84998E+06,
      0.88369E+06, 0.91846E+06, 0.95433E+06, 0.99132E+06, 0.10295E+07,
      0.10688E+07, 0.11093E+07, 0.11511E+07, 0.11941E+07, 0.12384E+07,
      0.12840E+07, 0.13310E+07, 0.13793E+07, 0.14291E+07, 0.14803E+07,
      0.15329E+07, 0.15871E+07, 0.16428E+07, 0.17000E+07, 0.17589E+07,
      0.18194E+07])


#  --------------- N2O 448: M = 4, I = 4 ---------------------
M = 4
I = 4
TIPS_GSI_HASH[(M,I)] = __FloatType__(9.)
TIPS_ISO_HASH[(M,I)] = float32([0.95253E+03, 0.13487E+04, 0.17465E+04,
      0.21498E+04, 0.25648E+04, 0.29986E+04, 0.34580E+04, 0.39493E+04,
      0.44779E+04, 0.50488E+04, 0.56669E+04, 0.63366E+04, 0.70625E+04,
      0.78488E+04, 0.87003E+04, 0.96216E+04, 0.10617E+05, 0.11692E+05,
      0.12852E+05, 0.14102E+05, 0.15447E+05, 0.16893E+05, 0.18446E+05,
      0.20112E+05, 0.21898E+05, 0.23811E+05, 0.25856E+05, 0.28042E+05,
      0.30377E+05, 0.32866E+05, 0.35520E+05, 0.38345E+05, 0.41351E+05,
      0.44545E+05, 0.47939E+05, 0.51540E+05, 0.55359E+05, 0.59405E+05,
      0.63689E+05, 0.68222E+05, 0.73015E+05, 0.78078E+05, 0.83424E+05,
      0.89064E+05, 0.95012E+05, 0.10128E+06, 0.10788E+06, 0.11482E+06,
      0.12213E+06, 0.12981E+06, 0.13788E+06, 0.14635E+06, 0.15524E+06,
      0.16456E+06, 0.17433E+06, 0.18457E+06, 0.19530E+06, 0.20652E+06,
      0.21827E+06, 0.23055E+06, 0.24338E+06, 0.25679E+06, 0.27079E+06,
      0.28541E+06, 0.30066E+06, 0.31656E+06, 0.33314E+06, 0.35042E+06,
      0.36841E+06, 0.38715E+06, 0.40666E+06, 0.42695E+06, 0.44805E+06,
      0.46999E+06, 0.49279E+06, 0.51649E+06, 0.54109E+06, 0.56664E+06,
      0.59315E+06, 0.62066E+06, 0.64919E+06, 0.67877E+06, 0.70943E+06,
      0.74121E+06, 0.77413E+06, 0.80822E+06, 0.84351E+06, 0.88004E+06,
      0.91783E+06, 0.95693E+06, 0.99737E+06, 0.10392E+07, 0.10824E+07,
      0.11270E+07, 0.11732E+07, 0.12208E+07, 0.12700E+07, 0.13208E+07,
      0.13732E+07, 0.14272E+07, 0.14830E+07, 0.15405E+07, 0.15999E+07,
      0.16610E+07, 0.17240E+07, 0.17890E+07, 0.18559E+07, 0.19248E+07,
      0.19957E+07, 0.20687E+07, 0.21439E+07, 0.22213E+07, 0.23009E+07,
      0.23828E+07, 0.24671E+07, 0.25537E+07, 0.26428E+07, 0.27343E+07,
      0.28284E+07])


#  --------------- N2O 447: M = 4, I = 5 ---------------------
M = 4
I = 5
TIPS_GSI_HASH[(M,I)] = __FloatType__(54.)
TIPS_ISO_HASH[(M,I)] = float32([0.55598E+04, 0.78718E+04, 0.10193E+05,
      0.12546E+05, 0.14966E+05, 0.17495E+05, 0.20171E+05, 0.23031E+05,
      0.26106E+05, 0.29426E+05, 0.33018E+05, 0.36908E+05, 0.41121E+05,
      0.45684E+05, 0.50622E+05, 0.55962E+05, 0.61731E+05, 0.67958E+05,
      0.74671E+05, 0.81902E+05, 0.89681E+05, 0.98043E+05, 0.10702E+06,
      0.11665E+06, 0.12697E+06, 0.13801E+06, 0.14983E+06, 0.16244E+06,
      0.17591E+06, 0.19028E+06, 0.20558E+06, 0.22188E+06, 0.23920E+06,
      0.25762E+06, 0.27718E+06, 0.29793E+06, 0.31993E+06, 0.34323E+06,
      0.36791E+06, 0.39401E+06, 0.42160E+06, 0.45074E+06, 0.48151E+06,
      0.51397E+06, 0.54819E+06, 0.58424E+06, 0.62221E+06, 0.66215E+06,
      0.70416E+06, 0.74832E+06, 0.79470E+06, 0.84340E+06, 0.89450E+06,
      0.94808E+06, 0.10042E+07, 0.10631E+07, 0.11247E+07, 0.11892E+07,
      0.12567E+07, 0.13272E+07, 0.14009E+07, 0.14779E+07, 0.15583E+07,
      0.16422E+07, 0.17298E+07, 0.18211E+07, 0.19163E+07, 0.20154E+07,
      0.21187E+07, 0.22263E+07, 0.23382E+07, 0.24546E+07, 0.25757E+07,
      0.27016E+07, 0.28324E+07, 0.29683E+07, 0.31095E+07, 0.32560E+07,
      0.34081E+07, 0.35659E+07, 0.37295E+07, 0.38991E+07, 0.40750E+07,
      0.42572E+07, 0.44459E+07, 0.46414E+07, 0.48437E+07, 0.50531E+07,
      0.52698E+07, 0.54939E+07, 0.57257E+07, 0.59653E+07, 0.62129E+07,
      0.64688E+07, 0.67331E+07, 0.70061E+07, 0.72880E+07, 0.75790E+07,
      0.78792E+07, 0.81891E+07, 0.85086E+07, 0.88382E+07, 0.91780E+07,
      0.95283E+07, 0.98893E+07, 0.10261E+08, 0.10644E+08, 0.11039E+08,
      0.11445E+08, 0.11864E+08, 0.12294E+08, 0.12738E+08, 0.13194E+08,
      0.13663E+08, 0.14145E+08, 0.14641E+08, 0.15151E+08, 0.15675E+08,
      0.16214E+08])


#  --------------- CO 26: M = 5, I = 1 ---------------------
M = 5
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.21948E+02, 0.30961E+02, 0.39980E+02,
      0.49004E+02, 0.58035E+02, 0.67071E+02, 0.76112E+02, 0.85160E+02,
      0.94213E+02, 0.10327E+03, 0.11234E+03, 0.12142E+03, 0.13050E+03,
      0.13960E+03, 0.14872E+03, 0.15787E+03, 0.16704E+03, 0.17624E+03,
      0.18548E+03, 0.19477E+03, 0.20411E+03, 0.21350E+03, 0.22295E+03,
      0.23248E+03, 0.24207E+03, 0.25175E+03, 0.26151E+03, 0.27136E+03,
      0.28130E+03, 0.29134E+03, 0.30148E+03, 0.31172E+03, 0.32207E+03,
      0.33253E+03, 0.34312E+03, 0.35381E+03, 0.36463E+03, 0.37557E+03,
      0.38663E+03, 0.39782E+03, 0.40914E+03, 0.42060E+03, 0.43218E+03,
      0.44389E+03, 0.45575E+03, 0.46774E+03, 0.47987E+03, 0.49213E+03,
      0.50454E+03, 0.51708E+03, 0.52978E+03, 0.54261E+03, 0.55559E+03,
      0.56871E+03, 0.58198E+03, 0.59540E+03, 0.60896E+03, 0.62267E+03,
      0.63653E+03, 0.65055E+03, 0.66470E+03, 0.67901E+03, 0.69347E+03,
      0.70808E+03, 0.72284E+03, 0.73776E+03, 0.75283E+03, 0.76805E+03,
      0.78342E+03, 0.79895E+03, 0.81463E+03, 0.83047E+03, 0.84646E+03,
      0.86260E+03, 0.87891E+03, 0.89536E+03, 0.91197E+03, 0.92874E+03,
      0.94566E+03, 0.96275E+03, 0.97998E+03, 0.99738E+03, 0.10149E+04,
      0.10326E+04, 0.10505E+04, 0.10685E+04, 0.10867E+04, 0.11051E+04,
      0.11236E+04, 0.11422E+04, 0.11611E+04, 0.11800E+04, 0.11992E+04,
      0.12185E+04, 0.12380E+04, 0.12576E+04, 0.12774E+04, 0.12973E+04,
      0.13174E+04, 0.13377E+04, 0.13581E+04, 0.13787E+04, 0.13994E+04,
      0.14203E+04, 0.14414E+04, 0.14627E+04, 0.14841E+04, 0.15056E+04,
      0.15273E+04, 0.15492E+04, 0.15713E+04, 0.15935E+04, 0.16159E+04,
      0.16384E+04, 0.16611E+04, 0.16840E+04, 0.17070E+04, 0.17302E+04,
      0.17536E+04])


#  --------------- CO 36: M = 5, I = 2 ---------------------
M = 5
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.45888E+02, 0.64745E+02, 0.83615E+02,
      0.10250E+03, 0.12139E+03, 0.14030E+03, 0.15921E+03, 0.17814E+03,
      0.19708E+03, 0.21604E+03, 0.23501E+03, 0.25400E+03, 0.27302E+03,
      0.29207E+03, 0.31117E+03, 0.33031E+03, 0.34952E+03, 0.36880E+03,
      0.38817E+03, 0.40764E+03, 0.42723E+03, 0.44694E+03, 0.46679E+03,
      0.48679E+03, 0.50696E+03, 0.52730E+03, 0.54783E+03, 0.56855E+03,
      0.58948E+03, 0.61061E+03, 0.63198E+03, 0.65357E+03, 0.67539E+03,
      0.69747E+03, 0.71979E+03, 0.74237E+03, 0.76521E+03, 0.78832E+03,
      0.81169E+03, 0.83534E+03, 0.85927E+03, 0.88348E+03, 0.90798E+03,
      0.93277E+03, 0.95784E+03, 0.98322E+03, 0.10089E+04, 0.10349E+04,
      0.10611E+04, 0.10877E+04, 0.11146E+04, 0.11418E+04, 0.11693E+04,
      0.11971E+04, 0.12253E+04, 0.12537E+04, 0.12825E+04, 0.13115E+04,
      0.13409E+04, 0.13707E+04, 0.14007E+04, 0.14311E+04, 0.14617E+04,
      0.14928E+04, 0.15241E+04, 0.15558E+04, 0.15877E+04, 0.16200E+04,
      0.16527E+04, 0.16857E+04, 0.17190E+04, 0.17526E+04, 0.17866E+04,
      0.18209E+04, 0.18555E+04, 0.18905E+04, 0.19258E+04, 0.19614E+04,
      0.19974E+04, 0.20337E+04, 0.20703E+04, 0.21073E+04, 0.21446E+04,
      0.21823E+04, 0.22203E+04, 0.22586E+04, 0.22973E+04, 0.23363E+04,
      0.23756E+04, 0.24153E+04, 0.24553E+04, 0.24957E+04, 0.25364E+04,
      0.25775E+04, 0.26189E+04, 0.26606E+04, 0.27027E+04, 0.27451E+04,
      0.27879E+04, 0.28310E+04, 0.28745E+04, 0.29183E+04, 0.29625E+04,
      0.30070E+04, 0.30518E+04, 0.30970E+04, 0.31425E+04, 0.31885E+04,
      0.32347E+04, 0.32813E+04, 0.33282E+04, 0.33755E+04, 0.34231E+04,
      0.34711E+04, 0.35194E+04, 0.35681E+04, 0.36172E+04, 0.36666E+04,
      0.37163E+04])


#  --------------- CO 28: M = 5, I = 3 ---------------------
M = 5
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.23030E+02, 0.32495E+02, 0.41966E+02,
      0.51443E+02, 0.60926E+02, 0.70415E+02, 0.79910E+02, 0.89410E+02,
      0.98918E+02, 0.10843E+03, 0.11795E+03, 0.12749E+03, 0.13703E+03,
      0.14659E+03, 0.15618E+03, 0.16579E+03, 0.17543E+03, 0.18511E+03,
      0.19483E+03, 0.20461E+03, 0.21444E+03, 0.22434E+03, 0.23430E+03,
      0.24435E+03, 0.25447E+03, 0.26468E+03, 0.27499E+03, 0.28540E+03,
      0.29591E+03, 0.30652E+03, 0.31725E+03, 0.32810E+03, 0.33906E+03,
      0.35014E+03, 0.36136E+03, 0.37270E+03, 0.38417E+03, 0.39577E+03,
      0.40752E+03, 0.41940E+03, 0.43142E+03, 0.44358E+03, 0.45589E+03,
      0.46834E+03, 0.48094E+03, 0.49369E+03, 0.50659E+03, 0.51964E+03,
      0.53284E+03, 0.54619E+03, 0.55971E+03, 0.57337E+03, 0.58719E+03,
      0.60117E+03, 0.61530E+03, 0.62959E+03, 0.64405E+03, 0.65866E+03,
      0.67343E+03, 0.68837E+03, 0.70346E+03, 0.71872E+03, 0.73414E+03,
      0.74972E+03, 0.76547E+03, 0.78138E+03, 0.79745E+03, 0.81369E+03,
      0.83010E+03, 0.84667E+03, 0.86341E+03, 0.88031E+03, 0.89738E+03,
      0.91462E+03, 0.93202E+03, 0.94960E+03, 0.96734E+03, 0.98524E+03,
      0.10033E+04, 0.10216E+04, 0.10400E+04, 0.10586E+04, 0.10773E+04,
      0.10962E+04, 0.11153E+04, 0.11346E+04, 0.11540E+04, 0.11737E+04,
      0.11934E+04, 0.12134E+04, 0.12335E+04, 0.12538E+04, 0.12743E+04,
      0.12949E+04, 0.13157E+04, 0.13367E+04, 0.13578E+04, 0.13792E+04,
      0.14007E+04, 0.14223E+04, 0.14442E+04, 0.14662E+04, 0.14884E+04,
      0.15108E+04, 0.15333E+04, 0.15560E+04, 0.15789E+04, 0.16020E+04,
      0.16252E+04, 0.16486E+04, 0.16722E+04, 0.16960E+04, 0.17199E+04,
      0.17441E+04, 0.17684E+04, 0.17928E+04, 0.18175E+04, 0.18423E+04,
      0.18673E+04])


#  --------------- CO 27: M = 5, I = 4 ---------------------
M = 5
I = 4
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.13505E+03, 0.19054E+03, 0.24606E+03,
      0.30161E+03, 0.35720E+03, 0.41283E+03, 0.46848E+03, 0.52418E+03,
      0.57991E+03, 0.63568E+03, 0.69149E+03, 0.74737E+03, 0.80332E+03,
      0.85937E+03, 0.91553E+03, 0.97183E+03, 0.10283E+04, 0.10850E+04,
      0.11420E+04, 0.11992E+04, 0.12568E+04, 0.13147E+04, 0.13730E+04,
      0.14318E+04, 0.14910E+04, 0.15507E+04, 0.16110E+04, 0.16718E+04,
      0.17332E+04, 0.17952E+04, 0.18579E+04, 0.19212E+04, 0.19852E+04,
      0.20499E+04, 0.21153E+04, 0.21815E+04, 0.22484E+04, 0.23161E+04,
      0.23846E+04, 0.24539E+04, 0.25240E+04, 0.25949E+04, 0.26666E+04,
      0.27392E+04, 0.28127E+04, 0.28869E+04, 0.29621E+04, 0.30381E+04,
      0.31150E+04, 0.31928E+04, 0.32715E+04, 0.33511E+04, 0.34316E+04,
      0.35129E+04, 0.35952E+04, 0.36785E+04, 0.37626E+04, 0.38477E+04,
      0.39336E+04, 0.40206E+04, 0.41084E+04, 0.41972E+04, 0.42869E+04,
      0.43776E+04, 0.44692E+04, 0.45618E+04, 0.46553E+04, 0.47498E+04,
      0.48452E+04, 0.49416E+04, 0.50390E+04, 0.51373E+04, 0.52366E+04,
      0.53368E+04, 0.54381E+04, 0.55403E+04, 0.56435E+04, 0.57476E+04,
      0.58527E+04, 0.59588E+04, 0.60659E+04, 0.61739E+04, 0.62829E+04,
      0.63930E+04, 0.65040E+04, 0.66160E+04, 0.67290E+04, 0.68429E+04,
      0.69579E+04, 0.70739E+04, 0.71908E+04, 0.73088E+04, 0.74277E+04,
      0.75477E+04, 0.76686E+04, 0.77905E+04, 0.79135E+04, 0.80374E+04,
      0.81624E+04, 0.82883E+04, 0.84153E+04, 0.85432E+04, 0.86722E+04,
      0.88022E+04, 0.89331E+04, 0.90651E+04, 0.91982E+04, 0.93322E+04,
      0.94672E+04, 0.96033E+04, 0.97404E+04, 0.98785E+04, 0.10018E+05,
      0.10158E+05, 0.10299E+05, 0.10441E+05, 0.10584E+05, 0.10728E+05,
      0.10874E+05])


#  --------------- CO 38: M = 5, I = 5 ---------------------
M = 5
I = 5
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.48264E+02, 0.68112E+02, 0.87974E+02,
      0.10785E+03, 0.12773E+03, 0.14763E+03, 0.16754E+03, 0.18747E+03,
      0.20741E+03, 0.22736E+03, 0.24733E+03, 0.26732E+03, 0.28735E+03,
      0.30741E+03, 0.32752E+03, 0.34770E+03, 0.36794E+03, 0.38828E+03,
      0.40871E+03, 0.42926E+03, 0.44994E+03, 0.47077E+03, 0.49175E+03,
      0.51290E+03, 0.53424E+03, 0.55578E+03, 0.57752E+03, 0.59948E+03,
      0.62166E+03, 0.64409E+03, 0.66676E+03, 0.68969E+03, 0.71287E+03,
      0.73633E+03, 0.76006E+03, 0.78407E+03, 0.80836E+03, 0.83295E+03,
      0.85784E+03, 0.88302E+03, 0.90851E+03, 0.93431E+03, 0.96042E+03,
      0.98686E+03, 0.10136E+04, 0.10407E+04, 0.10681E+04, 0.10958E+04,
      0.11238E+04, 0.11522E+04, 0.11809E+04, 0.12100E+04, 0.12393E+04,
      0.12691E+04, 0.12991E+04, 0.13295E+04, 0.13603E+04, 0.13914E+04,
      0.14228E+04, 0.14546E+04, 0.14867E+04, 0.15192E+04, 0.15520E+04,
      0.15852E+04, 0.16187E+04, 0.16526E+04, 0.16869E+04, 0.17215E+04,
      0.17564E+04, 0.17917E+04, 0.18274E+04, 0.18634E+04, 0.18998E+04,
      0.19365E+04, 0.19736E+04, 0.20111E+04, 0.20489E+04, 0.20871E+04,
      0.21256E+04, 0.21645E+04, 0.22038E+04, 0.22434E+04, 0.22834E+04,
      0.23238E+04, 0.23645E+04, 0.24056E+04, 0.24471E+04, 0.24889E+04,
      0.25311E+04, 0.25736E+04, 0.26166E+04, 0.26599E+04, 0.27035E+04,
      0.27476E+04, 0.27920E+04, 0.28368E+04, 0.28819E+04, 0.29275E+04,
      0.29733E+04, 0.30196E+04, 0.30662E+04, 0.31133E+04, 0.31606E+04,
      0.32084E+04, 0.32565E+04, 0.33050E+04, 0.33539E+04, 0.34032E+04,
      0.34528E+04, 0.35028E+04, 0.35532E+04, 0.36040E+04, 0.36551E+04,
      0.37067E+04, 0.37586E+04, 0.38108E+04, 0.38635E+04, 0.39165E+04,
      0.39699E+04])


#  --------------- CO 37: M = 5, I = 6 ---------------------
M = 5
I = 6
TIPS_GSI_HASH[(M,I)] = __FloatType__(12.)
TIPS_ISO_HASH[(M,I)] = float32([0.28271E+03, 0.39894E+03, 0.51524E+03,
      0.63162E+03, 0.74807E+03, 0.86459E+03, 0.98119E+03, 0.10979E+04,
      0.12146E+04, 0.13314E+04, 0.14484E+04, 0.15654E+04, 0.16826E+04,
      0.18000E+04, 0.19176E+04, 0.20355E+04, 0.21538E+04, 0.22725E+04,
      0.23916E+04, 0.25114E+04, 0.26318E+04, 0.27529E+04, 0.28749E+04,
      0.29977E+04, 0.31215E+04, 0.32463E+04, 0.33721E+04, 0.34991E+04,
      0.36274E+04, 0.37568E+04, 0.38876E+04, 0.40197E+04, 0.41533E+04,
      0.42882E+04, 0.44247E+04, 0.45626E+04, 0.47022E+04, 0.48433E+04,
      0.49860E+04, 0.51304E+04, 0.52763E+04, 0.54240E+04, 0.55735E+04,
      0.57246E+04, 0.58775E+04, 0.60321E+04, 0.61886E+04, 0.63468E+04,
      0.65068E+04, 0.66687E+04, 0.68324E+04, 0.69980E+04, 0.71654E+04,
      0.73347E+04, 0.75058E+04, 0.76789E+04, 0.78539E+04, 0.80307E+04,
      0.82096E+04, 0.83903E+04, 0.85729E+04, 0.87576E+04, 0.89441E+04,
      0.91326E+04, 0.93230E+04, 0.95154E+04, 0.97098E+04, 0.99061E+04,
      0.10104E+05, 0.10305E+05, 0.10507E+05, 0.10711E+05, 0.10918E+05,
      0.11126E+05, 0.11336E+05, 0.11549E+05, 0.11763E+05, 0.11979E+05,
      0.12198E+05, 0.12418E+05, 0.12640E+05, 0.12865E+05, 0.13091E+05,
      0.13320E+05, 0.13550E+05, 0.13783E+05, 0.14018E+05, 0.14254E+05,
      0.14493E+05, 0.14734E+05, 0.14977E+05, 0.15221E+05, 0.15468E+05,
      0.15718E+05, 0.15969E+05, 0.16222E+05, 0.16477E+05, 0.16734E+05,
      0.16994E+05, 0.17255E+05, 0.17519E+05, 0.17784E+05, 0.18052E+05,
      0.18322E+05, 0.18594E+05, 0.18868E+05, 0.19144E+05, 0.19422E+05,
      0.19703E+05, 0.19985E+05, 0.20270E+05, 0.20556E+05, 0.20845E+05,
      0.21136E+05, 0.21429E+05, 0.21724E+05, 0.22021E+05, 0.22320E+05,
      0.22622E+05])


#  --------------- CH4 211: M = 6, I = 1 ---------------------
M = 6
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.54800E+02, 0.91500E+02, 0.13410E+03,
      0.18180E+03, 0.23410E+03, 0.29070E+03, 0.35140E+03, 0.41600E+03,
      0.48450E+03, 0.55720E+03, 0.63420E+03, 0.71600E+03, 0.80310E+03,
      0.89590E+03, 0.99520E+03, 0.11017E+04, 0.12161E+04, 0.13393E+04,
      0.14721E+04, 0.16155E+04, 0.17706E+04, 0.19384E+04, 0.21202E+04,
      0.23172E+04, 0.25307E+04, 0.27624E+04, 0.30137E+04, 0.32864E+04,
      0.35823E+04, 0.39034E+04, 0.42519E+04, 0.46300E+04, 0.50402E+04,
      0.54853E+04, 0.59679E+04, 0.64913E+04, 0.70588E+04, 0.76739E+04,
      0.83404E+04, 0.90625E+04, 0.98446E+04, 0.10691E+05, 0.11608E+05,
      0.12600E+05, 0.13674E+05, 0.14835E+05, 0.16090E+05, 0.17447E+05,
      0.18914E+05, 0.20500E+05, 0.22212E+05, 0.24063E+05, 0.26061E+05,
      0.28218E+05, 0.30548E+05, 0.33063E+05, 0.35778E+05, 0.38708E+05,
      0.41871E+05, 0.45284E+05, 0.48970E+05, 0.52940E+05, 0.57230E+05,
      0.61860E+05, 0.66860E+05, 0.72250E+05, 0.78070E+05, 0.84350E+05,
      0.91130E+05, 0.98450E+05, 0.10635E+06, 0.11488E+06, 0.12408E+06,
      0.13403E+06, 0.14480E+06, 0.15640E+06, 0.16890E+06, 0.18240E+06,
      0.19700E+06, 0.21280E+06, 0.22980E+06, 0.24830E+06, 0.26820E+06,
      0.28970E+06, 0.31290E+06, 0.33800E+06, 0.36520E+06, 0.39450E+06,
      0.42600E+06, 0.46000E+06, 0.49700E+06, 0.53700E+06, 0.58100E+06,
      0.62700E+06, 0.67800E+06, 0.73300E+06, 0.79200E+06, 0.85600E+06,
      0.92500E+06, 0.10000E+07, 0.10800E+07, 0.11670E+07, 0.12610E+07,
      0.13620E+07, 0.14720E+07, 0.15910E+07, 0.17190E+07, 0.18600E+07,
      0.20100E+07, 0.21700E+07, 0.23400E+07, 0.25300E+07, 0.27300E+07,
      0.29500E+07, 0.31800E+07, 0.34300E+07, 0.37000E+07, 0.39900E+07,
      0.42856E+07])


#  --------------- CH4 311: M = 6, I = 2 ---------------------
M = 6
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.10958E+03, 0.18304E+03, 0.26818E+03,
      0.36356E+03, 0.46820E+03, 0.58141E+03, 0.70270E+03, 0.83186E+03,
      0.96893E+03, 0.11142E+04, 0.12682E+04, 0.14316E+04, 0.16055E+04,
      0.17909E+04, 0.19891E+04, 0.22016E+04, 0.24297E+04, 0.26752E+04,
      0.29399E+04, 0.32255E+04, 0.35342E+04, 0.38680E+04, 0.42294E+04,
      0.46208E+04, 0.50449E+04, 0.55046E+04, 0.60030E+04, 0.65434E+04,
      0.71293E+04, 0.77646E+04, 0.84535E+04, 0.92004E+04, 0.10010E+05,
      0.10888E+05, 0.11838E+05, 0.12869E+05, 0.13984E+05, 0.15193E+05,
      0.16501E+05, 0.17916E+05, 0.19448E+05, 0.21104E+05, 0.22895E+05,
      0.24830E+05, 0.26921E+05, 0.29180E+05, 0.31618E+05, 0.34250E+05,
      0.37090E+05, 0.40152E+05, 0.43454E+05, 0.47012E+05, 0.50845E+05,
      0.54973E+05, 0.59416E+05, 0.64197E+05, 0.69340E+05, 0.74870E+05,
      0.80813E+05, 0.87198E+05, 0.94055E+05, 0.10142E+06, 0.10932E+06,
      0.11779E+06, 0.12688E+06, 0.13662E+06, 0.14706E+06, 0.15824E+06,
      0.17021E+06, 0.18302E+06, 0.19673E+06, 0.21139E+06, 0.22706E+06,
      0.24381E+06, 0.26171E+06, 0.28082E+06, 0.30122E+06, 0.32299E+06,
      0.34621E+06, 0.37097E+06, 0.39737E+06, 0.42551E+06, 0.45548E+06,
      0.48739E+06, 0.52136E+06, 0.55752E+06, 0.59598E+06, 0.63688E+06,
      0.68036E+06, 0.72657E+06, 0.77566E+06, 0.82780E+06, 0.88316E+06,
      0.94191E+06, 0.10043E+07, 0.10704E+07, 0.11405E+07, 0.12148E+07,
      0.12936E+07, 0.13770E+07, 0.14654E+07, 0.15589E+07, 0.16579E+07,
      0.17627E+07, 0.18736E+07, 0.19908E+07, 0.21147E+07, 0.22456E+07,
      0.23840E+07, 0.25301E+07, 0.26844E+07, 0.28474E+07, 0.30193E+07,
      0.32007E+07, 0.33921E+07, 0.35939E+07, 0.38067E+07, 0.40310E+07,
      0.42673E+07])


#  --------------- CH4 212: M = 6, I = 3 ---------------------
M = 6
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(3.)
TIPS_ISO_HASH[(M,I)] = float32([0.44079E+03, 0.73786E+03, 0.10822E+04,
      0.14679E+04, 0.18913E+04, 0.23497E+04, 0.28415E+04, 0.33665E+04,
      0.39257E+04, 0.45211E+04, 0.51562E+04, 0.58349E+04, 0.65624E+04,
      0.73445E+04, 0.81872E+04, 0.90978E+04, 0.10084E+05, 0.11153E+05,
      0.12315E+05, 0.13579E+05, 0.14955E+05, 0.16455E+05, 0.18089E+05,
      0.19871E+05, 0.21816E+05, 0.23937E+05, 0.26251E+05, 0.28776E+05,
      0.31531E+05, 0.34535E+05, 0.37811E+05, 0.41384E+05, 0.45278E+05,
      0.49521E+05, 0.54144E+05, 0.59178E+05, 0.64657E+05, 0.70621E+05,
      0.77108E+05, 0.84161E+05, 0.91828E+05, 0.10016E+06, 0.10921E+06,
      0.11903E+06, 0.12968E+06, 0.14124E+06, 0.15378E+06, 0.16736E+06,
      0.18207E+06, 0.19800E+06, 0.21524E+06, 0.23389E+06, 0.25405E+06,
      0.27585E+06, 0.29939E+06, 0.32482E+06, 0.35226E+06, 0.38186E+06,
      0.41379E+06, 0.44821E+06, 0.48529E+06, 0.52522E+06, 0.56821E+06,
      0.61447E+06, 0.66422E+06, 0.71771E+06, 0.77519E+06, 0.83693E+06,
      0.90323E+06, 0.97438E+06, 0.10507E+07, 0.11326E+07, 0.12203E+07,
      0.13143E+07, 0.14150E+07, 0.15228E+07, 0.16382E+07, 0.17616E+07,
      0.18935E+07, 0.20346E+07, 0.21853E+07, 0.23463E+07, 0.25181E+07,
      0.27016E+07, 0.28973E+07, 0.31060E+07, 0.33284E+07, 0.35655E+07,
      0.38181E+07, 0.40870E+07, 0.43733E+07, 0.46780E+07, 0.50020E+07,
      0.53467E+07, 0.57130E+07, 0.61023E+07, 0.65158E+07, 0.69549E+07,
      0.74211E+07, 0.79158E+07, 0.84407E+07, 0.89973E+07, 0.95874E+07,
      0.10213E+08, 0.10875E+08, 0.11577E+08, 0.12320E+08, 0.13107E+08,
      0.13940E+08, 0.14820E+08, 0.15752E+08, 0.16736E+08, 0.17777E+08,
      0.18877E+08, 0.20038E+08, 0.21265E+08, 0.22560E+08, 0.23927E+08,
      0.25369E+08])


#  --------------- CH4 312: M = 6, I = 4 ---------------------
M = 6
I = 4
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.88231E+03, 0.14770E+04, 0.21661E+04,
      0.29384E+04, 0.37859E+04, 0.47034E+04, 0.56879E+04, 0.67388E+04,
      0.78581E+04, 0.90501E+04, 0.10321E+05, 0.11680E+05, 0.13136E+05,
      0.14702E+05, 0.16389E+05, 0.18212E+05, 0.20186E+05, 0.22328E+05,
      0.24654E+05, 0.27185E+05, 0.29941E+05, 0.32943E+05, 0.36216E+05,
      0.39786E+05, 0.43681E+05, 0.47930E+05, 0.52567E+05, 0.57625E+05,
      0.63144E+05, 0.69164E+05, 0.75730E+05, 0.82890E+05, 0.90693E+05,
      0.99198E+05, 0.10846E+06, 0.11855E+06, 0.12954E+06, 0.14149E+06,
      0.15450E+06, 0.16864E+06, 0.18402E+06, 0.20072E+06, 0.21886E+06,
      0.23856E+06, 0.25993E+06, 0.28312E+06, 0.30825E+06, 0.33550E+06,
      0.36501E+06, 0.39696E+06, 0.43155E+06, 0.46896E+06, 0.50942E+06,
      0.55315E+06, 0.60039E+06, 0.65141E+06, 0.70648E+06, 0.76589E+06,
      0.82997E+06, 0.89904E+06, 0.97346E+06, 0.10536E+07, 0.11399E+07,
      0.12327E+07, 0.13326E+07, 0.14400E+07, 0.15554E+07, 0.16793E+07,
      0.18124E+07, 0.19553E+07, 0.21085E+07, 0.22729E+07, 0.24490E+07,
      0.26378E+07, 0.28400E+07, 0.30565E+07, 0.32881E+07, 0.35360E+07,
      0.38010E+07, 0.40843E+07, 0.43870E+07, 0.47103E+07, 0.50555E+07,
      0.54239E+07, 0.58169E+07, 0.62361E+07, 0.66830E+07, 0.71592E+07,
      0.76666E+07, 0.82069E+07, 0.87820E+07, 0.93940E+07, 0.10045E+08,
      0.10737E+08, 0.11473E+08, 0.12256E+08, 0.13086E+08, 0.13969E+08,
      0.14905E+08, 0.15899E+08, 0.16954E+08, 0.18072E+08, 0.19258E+08,
      0.20515E+08, 0.21847E+08, 0.23257E+08, 0.24750E+08, 0.26331E+08,
      0.28004E+08, 0.29774E+08, 0.31646E+08, 0.33625E+08, 0.35716E+08,
      0.37926E+08, 0.40261E+08, 0.42726E+08, 0.45329E+08, 0.48077E+08,
      0.50975E+08])


#  --------------- O2 66: M = 7, I = 1 ---------------------
M = 7
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.44334E+02, 0.62460E+02, 0.80596E+02,
      0.98738E+02, 0.11688E+03, 0.13503E+03, 0.15319E+03, 0.17136E+03,
      0.18954E+03, 0.20775E+03, 0.22600E+03, 0.24431E+03, 0.26270E+03,
      0.28119E+03, 0.29981E+03, 0.31857E+03, 0.33750E+03, 0.35662E+03,
      0.37594E+03, 0.39550E+03, 0.41529E+03, 0.43535E+03, 0.45568E+03,
      0.47630E+03, 0.49722E+03, 0.51844E+03, 0.53998E+03, 0.56185E+03,
      0.58406E+03, 0.60660E+03, 0.62949E+03, 0.65274E+03, 0.67635E+03,
      0.70031E+03, 0.72465E+03, 0.74936E+03, 0.77444E+03, 0.79990E+03,
      0.82574E+03, 0.85197E+03, 0.87858E+03, 0.90558E+03, 0.93297E+03,
      0.96076E+03, 0.98895E+03, 0.10175E+04, 0.10465E+04, 0.10759E+04,
      0.11057E+04, 0.11359E+04, 0.11665E+04, 0.11976E+04, 0.12290E+04,
      0.12609E+04, 0.12931E+04, 0.13258E+04, 0.13590E+04, 0.13925E+04,
      0.14265E+04, 0.14609E+04, 0.14958E+04, 0.15311E+04, 0.15669E+04,
      0.16031E+04, 0.16397E+04, 0.16768E+04, 0.17144E+04, 0.17524E+04,
      0.17909E+04, 0.18298E+04, 0.18692E+04, 0.19091E+04, 0.19495E+04,
      0.19904E+04, 0.20318E+04, 0.20736E+04, 0.21160E+04, 0.21588E+04,
      0.22022E+04, 0.22461E+04, 0.22905E+04, 0.23354E+04, 0.23809E+04,
      0.24268E+04, 0.24734E+04, 0.25204E+04, 0.25680E+04, 0.26162E+04,
      0.26649E+04, 0.27142E+04, 0.27641E+04, 0.28145E+04, 0.28655E+04,
      0.29171E+04, 0.29693E+04, 0.30221E+04, 0.30755E+04, 0.31295E+04,
      0.31841E+04, 0.32393E+04, 0.32951E+04, 0.33516E+04, 0.34087E+04,
      0.34665E+04, 0.35249E+04, 0.35839E+04, 0.36436E+04, 0.37040E+04,
      0.37650E+04, 0.38267E+04, 0.38891E+04, 0.39522E+04, 0.40159E+04,
      0.40804E+04, 0.41455E+04, 0.42114E+04, 0.42780E+04, 0.43452E+04,
      0.44132E+04])


#  --------------- O2 68: M = 7, I = 2 ---------------------
M = 7
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.89206E+02, 0.12759E+03, 0.16600E+03,
      0.20442E+03, 0.24285E+03, 0.28128E+03, 0.31973E+03, 0.35821E+03,
      0.39672E+03, 0.43530E+03, 0.47398E+03, 0.51281E+03, 0.55183E+03,
      0.59108E+03, 0.63062E+03, 0.67051E+03, 0.71078E+03, 0.75148E+03,
      0.79265E+03, 0.83435E+03, 0.87659E+03, 0.91941E+03, 0.96285E+03,
      0.10069E+04, 0.10517E+04, 0.10971E+04, 0.11432E+04, 0.11901E+04,
      0.12377E+04, 0.12861E+04, 0.13352E+04, 0.13851E+04, 0.14358E+04,
      0.14872E+04, 0.15395E+04, 0.15926E+04, 0.16466E+04, 0.17013E+04,
      0.17569E+04, 0.18134E+04, 0.18706E+04, 0.19288E+04, 0.19877E+04,
      0.20476E+04, 0.21083E+04, 0.21698E+04, 0.22323E+04, 0.22956E+04,
      0.23598E+04, 0.24248E+04, 0.24908E+04, 0.25576E+04, 0.26253E+04,
      0.26940E+04, 0.27635E+04, 0.28339E+04, 0.29052E+04, 0.29775E+04,
      0.30506E+04, 0.31247E+04, 0.31997E+04, 0.32756E+04, 0.33524E+04,
      0.34302E+04, 0.35089E+04, 0.35885E+04, 0.36691E+04, 0.37506E+04,
      0.38331E+04, 0.39166E+04, 0.40010E+04, 0.40864E+04, 0.41727E+04,
      0.42601E+04, 0.43484E+04, 0.44377E+04, 0.45280E+04, 0.46193E+04,
      0.47116E+04, 0.48049E+04, 0.48992E+04, 0.49946E+04, 0.50909E+04,
      0.51883E+04, 0.52868E+04, 0.53863E+04, 0.54868E+04, 0.55884E+04,
      0.56911E+04, 0.57949E+04, 0.58997E+04, 0.60056E+04, 0.61126E+04,
      0.62207E+04, 0.63298E+04, 0.64401E+04, 0.65516E+04, 0.66641E+04,
      0.67778E+04, 0.68926E+04, 0.70085E+04, 0.71256E+04, 0.72439E+04,
      0.73633E+04, 0.74839E+04, 0.76056E+04, 0.77286E+04, 0.78527E+04,
      0.79781E+04, 0.81046E+04, 0.82324E+04, 0.83613E+04, 0.84915E+04,
      0.86229E+04, 0.87556E+04, 0.88895E+04, 0.90247E+04, 0.91611E+04,
      0.92988E+04])


#  --------------- O2 67: M = 7, I = 3 ---------------------
M = 7
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.52071E+03, 0.74484E+03, 0.96908E+03,
      0.11934E+04, 0.14177E+04, 0.16422E+04, 0.18667E+04, 0.20913E+04,
      0.23161E+04, 0.25413E+04, 0.27671E+04, 0.29936E+04, 0.32212E+04,
      0.34501E+04, 0.36806E+04, 0.39130E+04, 0.41476E+04, 0.43846E+04,
      0.46242E+04, 0.48668E+04, 0.51125E+04, 0.53615E+04, 0.56140E+04,
      0.58701E+04, 0.61300E+04, 0.63938E+04, 0.66617E+04, 0.69337E+04,
      0.72099E+04, 0.74904E+04, 0.77754E+04, 0.80647E+04, 0.83586E+04,
      0.86571E+04, 0.89602E+04, 0.92680E+04, 0.95805E+04, 0.98977E+04,
      0.10220E+05, 0.10547E+05, 0.10878E+05, 0.11215E+05, 0.11556E+05,
      0.11903E+05, 0.12254E+05, 0.12611E+05, 0.12972E+05, 0.13338E+05,
      0.13710E+05, 0.14086E+05, 0.14468E+05, 0.14855E+05, 0.15247E+05,
      0.15644E+05, 0.16046E+05, 0.16453E+05, 0.16866E+05, 0.17283E+05,
      0.17706E+05, 0.18135E+05, 0.18568E+05, 0.19007E+05, 0.19452E+05,
      0.19901E+05, 0.20356E+05, 0.20817E+05, 0.21283E+05, 0.21754E+05,
      0.22231E+05, 0.22713E+05, 0.23201E+05, 0.23695E+05, 0.24194E+05,
      0.24699E+05, 0.25209E+05, 0.25725E+05, 0.26247E+05, 0.26775E+05,
      0.27308E+05, 0.27847E+05, 0.28393E+05, 0.28944E+05, 0.29500E+05,
      0.30063E+05, 0.30632E+05, 0.31207E+05, 0.31788E+05, 0.32375E+05,
      0.32968E+05, 0.33568E+05, 0.34173E+05, 0.34785E+05, 0.35403E+05,
      0.36028E+05, 0.36659E+05, 0.37296E+05, 0.37939E+05, 0.38590E+05,
      0.39246E+05, 0.39909E+05, 0.40579E+05, 0.41256E+05, 0.41939E+05,
      0.42629E+05, 0.43325E+05, 0.44029E+05, 0.44739E+05, 0.45456E+05,
      0.46180E+05, 0.46911E+05, 0.47649E+05, 0.48394E+05, 0.49146E+05,
      0.49905E+05, 0.50671E+05, 0.51445E+05, 0.52226E+05, 0.53014E+05,
      0.53809E+05])


#  --------------- NO 46: M = 8, I = 1 ---------------------
M = 8
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(3.)
TIPS_ISO_HASH[(M,I)] = float32([0.15840E+03, 0.23971E+03, 0.33080E+03,
      0.42907E+03, 0.53251E+03, 0.63972E+03, 0.74975E+03, 0.86195E+03,
      0.97582E+03, 0.10911E+04, 0.12074E+04, 0.13248E+04, 0.14430E+04,
      0.15621E+04, 0.16820E+04, 0.18027E+04, 0.19243E+04, 0.20468E+04,
      0.21703E+04, 0.22948E+04, 0.24204E+04, 0.25472E+04, 0.26753E+04,
      0.28046E+04, 0.29354E+04, 0.30676E+04, 0.32013E+04, 0.33365E+04,
      0.34734E+04, 0.36120E+04, 0.37522E+04, 0.38942E+04, 0.40379E+04,
      0.41835E+04, 0.43310E+04, 0.44803E+04, 0.46316E+04, 0.47849E+04,
      0.49400E+04, 0.50972E+04, 0.52564E+04, 0.54176E+04, 0.55809E+04,
      0.57462E+04, 0.59137E+04, 0.60832E+04, 0.62548E+04, 0.64286E+04,
      0.66045E+04, 0.67825E+04, 0.69628E+04, 0.71451E+04, 0.73297E+04,
      0.75164E+04, 0.77053E+04, 0.78964E+04, 0.80897E+04, 0.82853E+04,
      0.84830E+04, 0.86830E+04, 0.88852E+04, 0.90896E+04, 0.92963E+04,
      0.95052E+04, 0.97164E+04, 0.99297E+04, 0.10145E+05, 0.10363E+05,
      0.10583E+05, 0.10806E+05, 0.11031E+05, 0.11258E+05, 0.11487E+05,
      0.11718E+05, 0.11952E+05, 0.12188E+05, 0.12426E+05, 0.12667E+05,
      0.12910E+05, 0.13155E+05, 0.13403E+05, 0.13652E+05, 0.13905E+05,
      0.14159E+05, 0.14416E+05, 0.14675E+05, 0.14936E+05, 0.15199E+05,
      0.15465E+05, 0.15733E+05, 0.16004E+05, 0.16277E+05, 0.16552E+05,
      0.16829E+05, 0.17109E+05, 0.17391E+05, 0.17675E+05, 0.17962E+05,
      0.18251E+05, 0.18542E+05, 0.18836E+05, 0.19131E+05, 0.19430E+05,
      0.19730E+05, 0.20033E+05, 0.20338E+05, 0.20646E+05, 0.20955E+05,
      0.21268E+05, 0.21582E+05, 0.21899E+05, 0.22218E+05, 0.22539E+05,
      0.22863E+05, 0.23189E+05, 0.23518E+05, 0.23848E+05, 0.24181E+05,
      0.24517E+05])


#  --------------- NO 56: M = 8, I = 2 ---------------------
M = 8
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.10942E+03, 0.16560E+03, 0.22856E+03,
      0.29647E+03, 0.36795E+03, 0.44204E+03, 0.51808E+03, 0.59561E+03,
      0.67432E+03, 0.75396E+03, 0.83439E+03, 0.91551E+03, 0.99725E+03,
      0.10796E+04, 0.11625E+04, 0.12460E+04, 0.13302E+04, 0.14150E+04,
      0.15005E+04, 0.15868E+04, 0.16739E+04, 0.17618E+04, 0.18506E+04,
      0.19404E+04, 0.20311E+04, 0.21229E+04, 0.22158E+04, 0.23098E+04,
      0.24050E+04, 0.25013E+04, 0.25989E+04, 0.26976E+04, 0.27977E+04,
      0.28991E+04, 0.30018E+04, 0.31058E+04, 0.32112E+04, 0.33180E+04,
      0.34262E+04, 0.35358E+04, 0.36468E+04, 0.37593E+04, 0.38732E+04,
      0.39885E+04, 0.41054E+04, 0.42237E+04, 0.43436E+04, 0.44649E+04,
      0.45877E+04, 0.47121E+04, 0.48379E+04, 0.49654E+04, 0.50943E+04,
      0.52248E+04, 0.53568E+04, 0.54904E+04, 0.56255E+04, 0.57622E+04,
      0.59004E+04, 0.60403E+04, 0.61816E+04, 0.63246E+04, 0.64692E+04,
      0.66152E+04, 0.67630E+04, 0.69123E+04, 0.70631E+04, 0.72156E+04,
      0.73696E+04, 0.75253E+04, 0.76825E+04, 0.78414E+04, 0.80018E+04,
      0.81638E+04, 0.83275E+04, 0.84927E+04, 0.86596E+04, 0.88280E+04,
      0.89981E+04, 0.91698E+04, 0.93430E+04, 0.95180E+04, 0.96945E+04,
      0.98726E+04, 0.10052E+05, 0.10234E+05, 0.10417E+05, 0.10601E+05,
      0.10788E+05, 0.10975E+05, 0.11165E+05, 0.11356E+05, 0.11549E+05,
      0.11743E+05, 0.11939E+05, 0.12137E+05, 0.12336E+05, 0.12537E+05,
      0.12739E+05, 0.12943E+05, 0.13149E+05, 0.13356E+05, 0.13565E+05,
      0.13776E+05, 0.13988E+05, 0.14202E+05, 0.14418E+05, 0.14635E+05,
      0.14853E+05, 0.15074E+05, 0.15296E+05, 0.15520E+05, 0.15745E+05,
      0.15972E+05, 0.16200E+05, 0.16431E+05, 0.16663E+05, 0.16896E+05,
      0.17131E+05])


#  --------------- NO 48: M = 8, I = 3 ---------------------
M = 8
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(3.)
TIPS_ISO_HASH[(M,I)] = float32([0.16695E+03, 0.25269E+03, 0.34876E+03,
      0.45239E+03, 0.56148E+03, 0.67455E+03, 0.79059E+03, 0.90891E+03,
      0.10290E+04, 0.11506E+04, 0.12733E+04, 0.13971E+04, 0.15219E+04,
      0.16476E+04, 0.17742E+04, 0.19017E+04, 0.20302E+04, 0.21598E+04,
      0.22904E+04, 0.24223E+04, 0.25553E+04, 0.26897E+04, 0.28255E+04,
      0.29628E+04, 0.31016E+04, 0.32420E+04, 0.33842E+04, 0.35280E+04,
      0.36736E+04, 0.38211E+04, 0.39704E+04, 0.41217E+04, 0.42750E+04,
      0.44302E+04, 0.45876E+04, 0.47469E+04, 0.49084E+04, 0.50720E+04,
      0.52378E+04, 0.54058E+04, 0.55759E+04, 0.57483E+04, 0.59230E+04,
      0.60999E+04, 0.62791E+04, 0.64605E+04, 0.66443E+04, 0.68304E+04,
      0.70187E+04, 0.72095E+04, 0.74026E+04, 0.75980E+04, 0.77958E+04,
      0.79960E+04, 0.81986E+04, 0.84036E+04, 0.86109E+04, 0.88207E+04,
      0.90328E+04, 0.92474E+04, 0.94644E+04, 0.96839E+04, 0.99057E+04,
      0.10130E+05, 0.10357E+05, 0.10586E+05, 0.10817E+05, 0.11052E+05,
      0.11288E+05, 0.11527E+05, 0.11768E+05, 0.12012E+05, 0.12259E+05,
      0.12507E+05, 0.12759E+05, 0.13012E+05, 0.13269E+05, 0.13527E+05,
      0.13788E+05, 0.14052E+05, 0.14318E+05, 0.14587E+05, 0.14858E+05,
      0.15131E+05, 0.15408E+05, 0.15686E+05, 0.15967E+05, 0.16251E+05,
      0.16537E+05, 0.16825E+05, 0.17116E+05, 0.17410E+05, 0.17706E+05,
      0.18004E+05, 0.18305E+05, 0.18609E+05, 0.18915E+05, 0.19224E+05,
      0.19535E+05, 0.19848E+05, 0.20164E+05, 0.20483E+05, 0.20804E+05,
      0.21127E+05, 0.21453E+05, 0.21782E+05, 0.22113E+05, 0.22447E+05,
      0.22783E+05, 0.23122E+05, 0.23463E+05, 0.23807E+05, 0.24153E+05,
      0.24502E+05, 0.24853E+05, 0.25207E+05, 0.25563E+05, 0.25922E+05,
      0.26283E+05])


#  --------------- SO2 626: M = 9, I = 1 ---------------------
M = 9
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.52899E+03, 0.89171E+03, 0.13139E+04,
      0.17915E+04, 0.23246E+04, 0.29155E+04, 0.35675E+04, 0.42848E+04,
      0.50723E+04, 0.59352E+04, 0.68794E+04, 0.79109E+04, 0.90366E+04,
      0.10264E+05, 0.11599E+05, 0.13052E+05, 0.14629E+05, 0.16340E+05,
      0.18193E+05, 0.20199E+05, 0.22366E+05, 0.24704E+05, 0.27225E+05,
      0.29938E+05, 0.32855E+05, 0.35987E+05, 0.39346E+05, 0.42944E+05,
      0.46794E+05, 0.50909E+05, 0.55302E+05, 0.59986E+05, 0.64977E+05,
      0.70288E+05, 0.75934E+05, 0.81931E+05, 0.88294E+05, 0.95040E+05,
      0.10219E+06, 0.10975E+06, 0.11774E+06, 0.12619E+06, 0.13511E+06,
      0.14452E+06, 0.15443E+06, 0.16487E+06, 0.17586E+06, 0.18742E+06,
      0.19957E+06, 0.21234E+06, 0.22573E+06, 0.23978E+06, 0.25451E+06,
      0.26995E+06, 0.28611E+06, 0.30302E+06, 0.32071E+06, 0.33920E+06,
      0.35852E+06, 0.37869E+06, 0.39974E+06, 0.42171E+06, 0.44461E+06,
      0.46848E+06, 0.49334E+06, 0.51922E+06, 0.54617E+06, 0.57419E+06,
      0.60334E+06, 0.63363E+06, 0.66511E+06, 0.69780E+06, 0.73174E+06,
      0.76696E+06, 0.80349E+06, 0.84138E+06, 0.88066E+06, 0.92136E+06,
      0.96352E+06, 0.10072E+07, 0.10524E+07, 0.10992E+07, 0.11475E+07,
      0.11976E+07, 0.12493E+07, 0.13028E+07, 0.13580E+07, 0.14151E+07,
      0.14741E+07, 0.15349E+07, 0.15977E+07, 0.16625E+07, 0.17293E+07,
      0.17982E+07, 0.18693E+07, 0.19425E+07, 0.20180E+07, 0.20958E+07,
      0.21758E+07, 0.22583E+07, 0.23432E+07, 0.24305E+07, 0.25204E+07,
      0.26129E+07, 0.27080E+07, 0.28058E+07, 0.29064E+07, 0.30097E+07,
      0.31159E+07, 0.32250E+07, 0.33371E+07, 0.34522E+07, 0.35705E+07,
      0.36918E+07, 0.38164E+07, 0.39442E+07, 0.40754E+07, 0.42099E+07,
      0.43479E+07])


#  --------------- SO2 646: M = 9, I = 2 ---------------------
M = 9
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.53140E+03, 0.89578E+03, 0.13199E+04,
      0.17997E+04, 0.23353E+04, 0.29288E+04, 0.35837E+04, 0.43043E+04,
      0.50953E+04, 0.59621E+04, 0.69104E+04, 0.79465E+04, 0.90772E+04,
      0.10310E+05, 0.11651E+05, 0.13110E+05, 0.14694E+05, 0.16413E+05,
      0.18274E+05, 0.20289E+05, 0.22465E+05, 0.24814E+05, 0.27345E+05,
      0.30070E+05, 0.33000E+05, 0.36145E+05, 0.39519E+05, 0.43133E+05,
      0.46999E+05, 0.51132E+05, 0.55544E+05, 0.60248E+05, 0.65260E+05,
      0.70594E+05, 0.76264E+05, 0.82287E+05, 0.88678E+05, 0.95453E+05,
      0.10263E+06, 0.11022E+06, 0.11825E+06, 0.12674E+06, 0.13569E+06,
      0.14514E+06, 0.15510E+06, 0.16558E+06, 0.17662E+06, 0.18823E+06,
      0.20043E+06, 0.21325E+06, 0.22670E+06, 0.24081E+06, 0.25561E+06,
      0.27111E+06, 0.28733E+06, 0.30432E+06, 0.32208E+06, 0.34065E+06,
      0.36005E+06, 0.38031E+06, 0.40145E+06, 0.42351E+06, 0.44651E+06,
      0.47047E+06, 0.49544E+06, 0.52144E+06, 0.54849E+06, 0.57664E+06,
      0.60591E+06, 0.63633E+06, 0.66794E+06, 0.70077E+06, 0.73485E+06,
      0.77022E+06, 0.80691E+06, 0.84496E+06, 0.88440E+06, 0.92527E+06,
      0.96761E+06, 0.10115E+07, 0.10568E+07, 0.11038E+07, 0.11524E+07,
      0.12027E+07, 0.12546E+07, 0.13083E+07, 0.13638E+07, 0.14211E+07,
      0.14803E+07, 0.15414E+07, 0.16045E+07, 0.16695E+07, 0.17366E+07,
      0.18059E+07, 0.18772E+07, 0.19507E+07, 0.20265E+07, 0.21046E+07,
      0.21850E+07, 0.22678E+07, 0.23531E+07, 0.24408E+07, 0.25310E+07,
      0.26239E+07, 0.27194E+07, 0.28176E+07, 0.29186E+07, 0.30224E+07,
      0.31290E+07, 0.32386E+07, 0.33512E+07, 0.34668E+07, 0.35855E+07,
      0.37074E+07, 0.38324E+07, 0.39608E+07, 0.40925E+07, 0.42276E+07,
      0.43662E+07])

#  --------------- NO2 646: M = 10, I = 1 ---------------------
M = 10
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(3.)
TIPS_ISO_HASH[(M,I)] = float32([0.12046E+04, 0.20297E+04, 0.29875E+04,
      0.40626E+04, 0.52463E+04, 0.65350E+04, 0.79286E+04, 0.94298E+04,
      0.11043E+05, 0.12776E+05, 0.14634E+05, 0.16627E+05, 0.18765E+05,
      0.21056E+05, 0.23511E+05, 0.26143E+05, 0.28961E+05, 0.31979E+05,
      0.35209E+05, 0.38663E+05, 0.42355E+05, 0.46300E+05, 0.50510E+05,
      0.55001E+05, 0.59787E+05, 0.64884E+05, 0.70308E+05, 0.76075E+05,
      0.82201E+05, 0.88704E+05, 0.95602E+05, 0.10291E+06, 0.11065E+06,
      0.11884E+06, 0.12750E+06, 0.13665E+06, 0.14631E+06, 0.15650E+06,
      0.16724E+06, 0.17856E+06, 0.19047E+06, 0.20301E+06, 0.21618E+06,
      0.23002E+06, 0.24456E+06, 0.25981E+06, 0.27580E+06, 0.29256E+06,
      0.31012E+06, 0.32850E+06, 0.34773E+06, 0.36784E+06, 0.38886E+06,
      0.41082E+06, 0.43374E+06, 0.45766E+06, 0.48262E+06, 0.50863E+06,
      0.53574E+06, 0.56398E+06, 0.59339E+06, 0.62398E+06, 0.65581E+06,
      0.68891E+06, 0.72331E+06, 0.75905E+06, 0.79617E+06, 0.83470E+06,
      0.87469E+06, 0.91617E+06, 0.95919E+06, 0.10038E+07, 0.10500E+07,
      0.10979E+07, 0.11474E+07, 0.11988E+07, 0.12519E+07, 0.13068E+07,
      0.13636E+07, 0.14224E+07, 0.14831E+07, 0.15459E+07, 0.16107E+07,
      0.16776E+07, 0.17467E+07, 0.18180E+07, 0.18916E+07, 0.19675E+07,
      0.20458E+07, 0.21265E+07, 0.22097E+07, 0.22954E+07, 0.23837E+07,
      0.24747E+07, 0.25684E+07, 0.26648E+07, 0.27641E+07, 0.28662E+07,
      0.29713E+07, 0.30794E+07, 0.31905E+07, 0.33048E+07, 0.34223E+07,
      0.35430E+07, 0.36670E+07, 0.37944E+07, 0.39253E+07, 0.40597E+07,
      0.41976E+07, 0.43393E+07, 0.44846E+07, 0.46337E+07, 0.47867E+07,
      0.49437E+07, 0.51046E+07, 0.52696E+07, 0.54388E+07, 0.56122E+07,
      0.57900E+07])


#  --------------- NH3 4111: M = 11, I = 1 ---------------------
M = 11
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(3.)
TIPS_ISO_HASH[(M,I)] = float32([0.16013E+03, 0.26692E+03, 0.39067E+03,
      0.52933E+03, 0.68153E+03, 0.84641E+03, 0.10234E+04, 0.12125E+04,
      0.14136E+04, 0.16272E+04, 0.18537E+04, 0.20937E+04, 0.23481E+04,
      0.26177E+04, 0.29035E+04, 0.32065E+04, 0.35279E+04, 0.38688E+04,
      0.42304E+04, 0.46141E+04, 0.50212E+04, 0.54531E+04, 0.59114E+04,
      0.63976E+04, 0.69133E+04, 0.74602E+04, 0.80401E+04, 0.86549E+04,
      0.93066E+04, 0.99971E+04, 0.10729E+05, 0.11504E+05, 0.12324E+05,
      0.13193E+05, 0.14112E+05, 0.15085E+05, 0.16114E+05, 0.17201E+05,
      0.18352E+05, 0.19567E+05, 0.20851E+05, 0.22208E+05, 0.23640E+05,
      0.25152E+05, 0.26747E+05, 0.28430E+05, 0.30205E+05, 0.32077E+05,
      0.34050E+05, 0.36128E+05, 0.38317E+05, 0.40623E+05, 0.43050E+05,
      0.45605E+05, 0.48292E+05, 0.51119E+05, 0.54091E+05, 0.57215E+05,
      0.60498E+05, 0.63947E+05, 0.67569E+05, 0.71372E+05, 0.75364E+05,
      0.79552E+05, 0.83946E+05, 0.88553E+05, 0.93384E+05, 0.98447E+05,
      0.10375E+06, 0.10931E+06, 0.11513E+06, 0.12122E+06, 0.12760E+06,
      0.13427E+06, 0.14125E+06, 0.14855E+06, 0.15619E+06, 0.16417E+06,
      0.17250E+06, 0.18121E+06, 0.19031E+06, 0.19981E+06, 0.20973E+06,
      0.22008E+06, 0.23088E+06, 0.24215E+06, 0.25390E+06, 0.26615E+06,
      0.27892E+06, 0.29223E+06, 0.30610E+06, 0.32055E+06, 0.33559E+06,
      0.35125E+06, 0.36756E+06, 0.38453E+06, 0.40219E+06, 0.42056E+06,
      0.43967E+06, 0.45953E+06, 0.48019E+06, 0.50165E+06, 0.52396E+06,
      0.54714E+06, 0.57122E+06, 0.59622E+06, 0.62218E+06, 0.64913E+06,
      0.67710E+06, 0.70613E+06, 0.73624E+06, 0.76748E+06, 0.79988E+06,
      0.83347E+06, 0.86829E+06, 0.90439E+06, 0.94180E+06, 0.98056E+06,
      0.10207E+07])


#  --------------- NH3 5111: M = 11, I = 2 ---------------------
M = 11
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.10697E+03, 0.17832E+03, 0.26100E+03,
      0.35364E+03, 0.45533E+03, 0.56549E+03, 0.68377E+03, 0.81007E+03,
      0.94447E+03, 0.10872E+04, 0.12385E+04, 0.13988E+04, 0.15688E+04,
      0.17490E+04, 0.19399E+04, 0.21424E+04, 0.23571E+04, 0.25848E+04,
      0.28264E+04, 0.30828E+04, 0.33548E+04, 0.36434E+04, 0.39496E+04,
      0.42745E+04, 0.46190E+04, 0.49845E+04, 0.53720E+04, 0.57828E+04,
      0.62182E+04, 0.66796E+04, 0.71684E+04, 0.76862E+04, 0.82344E+04,
      0.88149E+04, 0.94292E+04, 0.10079E+05, 0.10767E+05, 0.11494E+05,
      0.12262E+05, 0.13074E+05, 0.13932E+05, 0.14839E+05, 0.15796E+05,
      0.16806E+05, 0.17872E+05, 0.18997E+05, 0.20183E+05, 0.21434E+05,
      0.22752E+05, 0.24141E+05, 0.25604E+05, 0.27145E+05, 0.28767E+05,
      0.30475E+05, 0.32271E+05, 0.34160E+05, 0.36146E+05, 0.38234E+05,
      0.40428E+05, 0.42733E+05, 0.45154E+05, 0.47696E+05, 0.50364E+05,
      0.53163E+05, 0.56100E+05, 0.59180E+05, 0.62408E+05, 0.65792E+05,
      0.69339E+05, 0.73053E+05, 0.76943E+05, 0.81016E+05, 0.85279E+05,
      0.89740E+05, 0.94406E+05, 0.99287E+05, 0.10439E+06, 0.10972E+06,
      0.11530E+06, 0.12112E+06, 0.12720E+06, 0.13355E+06, 0.14018E+06,
      0.14711E+06, 0.15433E+06, 0.16186E+06, 0.16971E+06, 0.17791E+06,
      0.18645E+06, 0.19534E+06, 0.20462E+06, 0.21428E+06, 0.22434E+06,
      0.23481E+06, 0.24572E+06, 0.25706E+06, 0.26887E+06, 0.28116E+06,
      0.29393E+06, 0.30722E+06, 0.32103E+06, 0.33539E+06, 0.35031E+06,
      0.36581E+06, 0.38191E+06, 0.39864E+06, 0.41600E+06, 0.43403E+06,
      0.45274E+06, 0.47215E+06, 0.49230E+06, 0.51319E+06, 0.53487E+06,
      0.55734E+06, 0.58064E+06, 0.60478E+06, 0.62981E+06, 0.65574E+06,
      0.68260E+06])


#  --------------- HNO3 146: M = 12, I = 1 ---------------------
M = 12
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.15010E+05, 0.25316E+05, 0.37374E+05,
      0.51216E+05, 0.67105E+05, 0.85473E+05, 0.10688E+06, 0.13201E+06,
      0.16165E+06, 0.19671E+06, 0.23825E+06, 0.28749E+06, 0.34583E+06,
      0.41490E+06, 0.49657E+06, 0.59302E+06, 0.70673E+06, 0.84054E+06,
      0.99775E+06, 0.11821E+07, 0.13978E+07, 0.16498E+07, 0.19436E+07,
      0.22855E+07, 0.26825E+07, 0.31428E+07, 0.36753E+07, 0.42903E+07,
      0.49993E+07, 0.58151E+07, 0.67523E+07, 0.78269E+07, 0.90572E+07,
      0.10463E+08, 0.12067E+08, 0.13895E+08, 0.15973E+08, 0.18333E+08,
      0.21009E+08, 0.24039E+08, 0.27464E+08, 0.31331E+08, 0.35690E+08,
      0.40597E+08, 0.46115E+08, 0.52310E+08, 0.59257E+08, 0.67037E+08,
      0.75739E+08, 0.85461E+08, 0.96310E+08, 0.10840E+09, 0.12186E+09,
      0.13683E+09, 0.15346E+09, 0.17191E+09, 0.19236E+09, 0.21501E+09,
      0.24006E+09, 0.26774E+09, 0.29830E+09, 0.33200E+09, 0.36914E+09,
      0.41002E+09, 0.45498E+09, 0.50438E+09, 0.55862E+09, 0.61812E+09,
      0.68332E+09, 0.75473E+09, 0.83286E+09, 0.91828E+09, 0.10116E+10,
      0.11134E+10, 0.12245E+10, 0.13456E+10, 0.14775E+10, 0.16210E+10,
      0.17771E+10, 0.19467E+10, 0.21309E+10, 0.23309E+10, 0.25477E+10,
      0.27827E+10, 0.30372E+10, 0.33127E+10, 0.36107E+10, 0.39329E+10,
      0.42809E+10, 0.46567E+10, 0.50623E+10, 0.54997E+10, 0.59711E+10,
      0.64789E+10, 0.70257E+10, 0.76140E+10, 0.82468E+10, 0.89269E+10,
      0.96575E+10, 0.10442E+11, 0.11284E+11, 0.12187E+11, 0.13155E+11,
      0.14193E+11, 0.15304E+11, 0.16494E+11, 0.17767E+11, 0.19129E+11,
      0.20585E+11, 0.22140E+11, 0.23802E+11, 0.25576E+11, 0.27469E+11,
      0.29489E+11, 0.31642E+11, 0.33937E+11, 0.36382E+11, 0.38985E+11,
      0.41757E+11])


#  --------------- HNO3 156: M = 12, I = 2 --------------------- NOT IN TIPS-2011
M = 12
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(0.)
TIPS_ISO_HASH[(M,I)] = float32([0.])


#  --------------- OH 61: M = 13, I = 1 ---------------------
M = 13
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.20066E+02, 0.24774E+02, 0.30309E+02,
      0.36357E+02, 0.42745E+02, 0.49371E+02, 0.56168E+02, 0.63093E+02,
      0.70116E+02, 0.77217E+02, 0.84380E+02, 0.91594E+02, 0.98850E+02,
      0.10614E+03, 0.11346E+03, 0.12081E+03, 0.12818E+03, 0.13557E+03,
      0.14298E+03, 0.15041E+03, 0.15785E+03, 0.16531E+03, 0.17278E+03,
      0.18027E+03, 0.18778E+03, 0.19530E+03, 0.20284E+03, 0.21040E+03,
      0.21797E+03, 0.22556E+03, 0.23318E+03, 0.24082E+03, 0.24848E+03,
      0.25617E+03, 0.26389E+03, 0.27163E+03, 0.27941E+03, 0.28721E+03,
      0.29505E+03, 0.30292E+03, 0.31084E+03, 0.31878E+03, 0.32677E+03,
      0.33480E+03, 0.34287E+03, 0.35099E+03, 0.35915E+03, 0.36736E+03,
      0.37561E+03, 0.38391E+03, 0.39227E+03, 0.40067E+03, 0.40913E+03,
      0.41764E+03, 0.42620E+03, 0.43482E+03, 0.44350E+03, 0.45223E+03,
      0.46102E+03, 0.46987E+03, 0.47878E+03, 0.48775E+03, 0.49679E+03,
      0.50588E+03, 0.51503E+03, 0.52425E+03, 0.53354E+03, 0.54288E+03,
      0.55229E+03, 0.56177E+03, 0.57132E+03, 0.58092E+03, 0.59060E+03,
      0.60035E+03, 0.61016E+03, 0.62004E+03, 0.62999E+03, 0.64001E+03,
      0.65010E+03, 0.66025E+03, 0.67049E+03, 0.68078E+03, 0.69115E+03,
      0.70160E+03, 0.71211E+03, 0.72269E+03, 0.73335E+03, 0.74408E+03,
      0.75488E+03, 0.76576E+03, 0.77671E+03, 0.78773E+03, 0.79883E+03,
      0.81000E+03, 0.82124E+03, 0.83256E+03, 0.84396E+03, 0.85542E+03,
      0.86696E+03, 0.87858E+03, 0.89027E+03, 0.90204E+03, 0.91389E+03,
      0.92580E+03, 0.93781E+03, 0.94988E+03, 0.96203E+03, 0.97425E+03,
      0.98656E+03, 0.99893E+03, 0.10114E+04, 0.10239E+04, 0.10365E+04,
      0.10492E+04, 0.10620E+04, 0.10748E+04, 0.10878E+04, 0.11007E+04,
      0.11138E+04])

#  --------------- OH 81: M = 13, I = 2 ---------------------
M = 13
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.20124E+02, 0.24876E+02, 0.30457E+02,
      0.36553E+02, 0.42991E+02, 0.49666E+02, 0.56513E+02, 0.63489E+02,
      0.70563E+02, 0.77715E+02, 0.84929E+02, 0.92195E+02, 0.99504E+02,
      0.10685E+03, 0.11423E+03, 0.12164E+03, 0.12907E+03, 0.13654E+03,
      0.14403E+03, 0.15154E+03, 0.15909E+03, 0.16666E+03, 0.17427E+03,
      0.18191E+03, 0.18959E+03, 0.19731E+03, 0.20507E+03, 0.21287E+03,
      0.22073E+03, 0.22863E+03, 0.23658E+03, 0.24459E+03, 0.25266E+03,
      0.26078E+03, 0.26897E+03, 0.27722E+03, 0.28554E+03, 0.29393E+03,
      0.30238E+03, 0.31091E+03, 0.31952E+03, 0.32820E+03, 0.33696E+03,
      0.34579E+03, 0.35471E+03, 0.36371E+03, 0.37279E+03, 0.38196E+03,
      0.39121E+03, 0.40055E+03, 0.40998E+03, 0.41949E+03, 0.42910E+03,
      0.43879E+03, 0.44858E+03, 0.45845E+03, 0.46843E+03, 0.47849E+03,
      0.48865E+03, 0.49890E+03, 0.50924E+03, 0.51969E+03, 0.53022E+03,
      0.54086E+03, 0.55159E+03, 0.56242E+03, 0.57335E+03, 0.58437E+03,
      0.59550E+03, 0.60673E+03, 0.61805E+03, 0.62947E+03, 0.64100E+03,
      0.65263E+03, 0.66435E+03, 0.67618E+03, 0.68811E+03, 0.70014E+03,
      0.71228E+03, 0.72451E+03, 0.73685E+03, 0.74929E+03, 0.76184E+03,
      0.77449E+03, 0.78724E+03, 0.80009E+03, 0.81306E+03, 0.82612E+03,
      0.83929E+03, 0.85256E+03, 0.86594E+03, 0.87942E+03, 0.89301E+03,
      0.90670E+03, 0.92050E+03, 0.93440E+03, 0.94841E+03, 0.96253E+03,
      0.97675E+03, 0.99108E+03, 0.10055E+04, 0.10201E+04, 0.10347E+04,
      0.10495E+04, 0.10643E+04, 0.10793E+04, 0.10944E+04, 0.11096E+04,
      0.11248E+04, 0.11402E+04, 0.11558E+04, 0.11714E+04, 0.11871E+04,
      0.12029E+04, 0.12189E+04, 0.12349E+04, 0.12511E+04, 0.12673E+04,
      0.12837E+04])

#  --------------- OH 62: M = 13, I = 3 ---------------------
M = 13
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(3.)
TIPS_ISO_HASH[(M,I)] = float32([0.41032E+02, 0.54704E+02, 0.70201E+02,
      0.86985E+02, 0.10469E+03, 0.12306E+03, 0.14194E+03, 0.16119E+03,
      0.18075E+03, 0.20054E+03, 0.22053E+03, 0.24068E+03, 0.26096E+03,
      0.28135E+03, 0.30183E+03, 0.32241E+03, 0.34305E+03, 0.36376E+03,
      0.38453E+03, 0.40535E+03, 0.42622E+03, 0.44714E+03, 0.46811E+03,
      0.48913E+03, 0.51019E+03, 0.53131E+03, 0.55246E+03, 0.57368E+03,
      0.59495E+03, 0.61627E+03, 0.63766E+03, 0.65912E+03, 0.68064E+03,
      0.70223E+03, 0.72390E+03, 0.74565E+03, 0.76749E+03, 0.78941E+03,
      0.81143E+03, 0.83355E+03, 0.85578E+03, 0.87810E+03, 0.90054E+03,
      0.92310E+03, 0.94577E+03, 0.96857E+03, 0.99149E+03, 0.10145E+04,
      0.10377E+04, 0.10611E+04, 0.10845E+04, 0.11081E+04, 0.11319E+04,
      0.11558E+04, 0.11798E+04, 0.12040E+04, 0.12284E+04, 0.12529E+04,
      0.12776E+04, 0.13025E+04, 0.13275E+04, 0.13527E+04, 0.13781E+04,
      0.14036E+04, 0.14293E+04, 0.14552E+04, 0.14813E+04, 0.15076E+04,
      0.15340E+04, 0.15606E+04, 0.15874E+04, 0.16144E+04, 0.16416E+04,
      0.16690E+04, 0.16965E+04, 0.17243E+04, 0.17522E+04, 0.17804E+04,
      0.18087E+04, 0.18373E+04, 0.18660E+04, 0.18949E+04, 0.19241E+04,
      0.19534E+04, 0.19829E+04, 0.20127E+04, 0.20426E+04, 0.20727E+04,
      0.21031E+04, 0.21336E+04, 0.21644E+04, 0.21954E+04, 0.22266E+04,
      0.22579E+04, 0.22895E+04, 0.23213E+04, 0.23534E+04, 0.23856E+04,
      0.24180E+04, 0.24506E+04, 0.24835E+04, 0.25166E+04, 0.25499E+04,
      0.25834E+04, 0.26171E+04, 0.26510E+04, 0.26852E+04, 0.27195E+04,
      0.27541E+04, 0.27889E+04, 0.28239E+04, 0.28592E+04, 0.28946E+04,
      0.29303E+04, 0.29661E+04, 0.30023E+04, 0.30386E+04, 0.30751E+04,
      0.31119E+04])


#  --------------- HF 19: M = 14, I = 1 ---------------------
M = 14
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(4.)
TIPS_ISO_HASH[(M,I)] = float32([0.95958E+01, 0.12933E+02, 0.16295E+02,
      0.19666E+02, 0.23043E+02, 0.26425E+02, 0.29809E+02, 0.33195E+02,
      0.36584E+02, 0.39974E+02, 0.43366E+02, 0.46759E+02, 0.50154E+02,
      0.53550E+02, 0.56947E+02, 0.60346E+02, 0.63746E+02, 0.67148E+02,
      0.70550E+02, 0.73955E+02, 0.77361E+02, 0.80769E+02, 0.84179E+02,
      0.87591E+02, 0.91006E+02, 0.94424E+02, 0.97846E+02, 0.10127E+03,
      0.10470E+03, 0.10813E+03, 0.11157E+03, 0.11502E+03, 0.11847E+03,
      0.12193E+03, 0.12540E+03, 0.12888E+03, 0.13236E+03, 0.13586E+03,
      0.13936E+03, 0.14288E+03, 0.14641E+03, 0.14995E+03, 0.15351E+03,
      0.15708E+03, 0.16066E+03, 0.16426E+03, 0.16788E+03, 0.17151E+03,
      0.17516E+03, 0.17882E+03, 0.18251E+03, 0.18621E+03, 0.18994E+03,
      0.19368E+03, 0.19745E+03, 0.20123E+03, 0.20504E+03, 0.20887E+03,
      0.21272E+03, 0.21659E+03, 0.22049E+03, 0.22441E+03, 0.22836E+03,
      0.23233E+03, 0.23632E+03, 0.24034E+03, 0.24439E+03, 0.24846E+03,
      0.25255E+03, 0.25668E+03, 0.26083E+03, 0.26501E+03, 0.26921E+03,
      0.27344E+03, 0.27770E+03, 0.28199E+03, 0.28631E+03, 0.29066E+03,
      0.29503E+03, 0.29944E+03, 0.30387E+03, 0.30833E+03, 0.31282E+03,
      0.31735E+03, 0.32190E+03, 0.32648E+03, 0.33110E+03, 0.33574E+03,
      0.34042E+03, 0.34512E+03, 0.34986E+03, 0.35463E+03, 0.35943E+03,
      0.36426E+03, 0.36913E+03, 0.37402E+03, 0.37895E+03, 0.38391E+03,
      0.38891E+03, 0.39393E+03, 0.39899E+03, 0.40408E+03, 0.40921E+03,
      0.41436E+03, 0.41955E+03, 0.42478E+03, 0.43004E+03, 0.43533E+03,
      0.44065E+03, 0.44601E+03, 0.45140E+03, 0.45683E+03, 0.46229E+03,
      0.46779E+03, 0.47332E+03, 0.47888E+03, 0.48448E+03, 0.49011E+03,
      0.49578E+03])

#  --------------- HF 29: M = 14, I = 2 --------------------- not in TIPS-2011
M = 14
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(0.)
TIPS_ISO_HASH[(M,I)] = float32([0.])


#  --------------- HСl 15: M = 15, I = 1 --------------------
M = 15
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(8.)
TIPS_ISO_HASH[(M,I)] = float32([0.34775E+02, 0.48060E+02, 0.61370E+02,
      0.74692E+02, 0.88024E+02, 0.10136E+03, 0.11471E+03, 0.12806E+03,
      0.14141E+03, 0.15478E+03, 0.16814E+03, 0.18151E+03, 0.19489E+03,
      0.20827E+03, 0.22166E+03, 0.23506E+03, 0.24847E+03, 0.26189E+03,
      0.27533E+03, 0.28878E+03, 0.30225E+03, 0.31575E+03, 0.32928E+03,
      0.34284E+03, 0.35645E+03, 0.37009E+03, 0.38378E+03, 0.39753E+03,
      0.41134E+03, 0.42521E+03, 0.43914E+03, 0.45316E+03, 0.46725E+03,
      0.48142E+03, 0.49568E+03, 0.51003E+03, 0.52448E+03, 0.53902E+03,
      0.55368E+03, 0.56843E+03, 0.58330E+03, 0.59829E+03, 0.61339E+03,
      0.62862E+03, 0.64396E+03, 0.65944E+03, 0.67504E+03, 0.69078E+03,
      0.70665E+03, 0.72265E+03, 0.73880E+03, 0.75508E+03, 0.77151E+03,
      0.78809E+03, 0.80481E+03, 0.82168E+03, 0.83870E+03, 0.85587E+03,
      0.87320E+03, 0.89068E+03, 0.90832E+03, 0.92611E+03, 0.94407E+03,
      0.96218E+03, 0.98046E+03, 0.99889E+03, 0.10175E+04, 0.10363E+04,
      0.10552E+04, 0.10743E+04, 0.10936E+04, 0.11130E+04, 0.11326E+04,
      0.11524E+04, 0.11723E+04, 0.11924E+04, 0.12127E+04, 0.12332E+04,
      0.12538E+04, 0.12746E+04, 0.12956E+04, 0.13168E+04, 0.13381E+04,
      0.13597E+04, 0.13814E+04, 0.14032E+04, 0.14253E+04, 0.14475E+04,
      0.14700E+04, 0.14926E+04, 0.15153E+04, 0.15383E+04, 0.15615E+04,
      0.15848E+04, 0.16083E+04, 0.16320E+04, 0.16559E+04, 0.16800E+04,
      0.17043E+04, 0.17287E+04, 0.17533E+04, 0.17782E+04, 0.18032E+04,
      0.18284E+04, 0.18538E+04, 0.18794E+04, 0.19051E+04, 0.19311E+04,
      0.19573E+04, 0.19836E+04, 0.20102E+04, 0.20369E+04, 0.20638E+04,
      0.20910E+04, 0.21183E+04, 0.21458E+04, 0.21735E+04, 0.22014E+04,
      0.22295E+04])


#  --------------- HСl 17: M = 15, I = 2 ---------------------
M = 15
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(8.)
TIPS_ISO_HASH[(M,I)] = float32([0.34823E+02, 0.48128E+02, 0.61458E+02,
      0.74801E+02, 0.88152E+02, 0.10151E+03, 0.11488E+03, 0.12825E+03,
      0.14162E+03, 0.15500E+03, 0.16839E+03, 0.18178E+03, 0.19518E+03,
      0.20858E+03, 0.22199E+03, 0.23541E+03, 0.24884E+03, 0.26228E+03,
      0.27574E+03, 0.28921E+03, 0.30270E+03, 0.31622E+03, 0.32977E+03,
      0.34336E+03, 0.35698E+03, 0.37065E+03, 0.38436E+03, 0.39813E+03,
      0.41196E+03, 0.42585E+03, 0.43981E+03, 0.45384E+03, 0.46796E+03,
      0.48215E+03, 0.49644E+03, 0.51081E+03, 0.52528E+03, 0.53986E+03,
      0.55453E+03, 0.56932E+03, 0.58421E+03, 0.59922E+03, 0.61435E+03,
      0.62960E+03, 0.64498E+03, 0.66048E+03, 0.67611E+03, 0.69187E+03,
      0.70777E+03, 0.72381E+03, 0.73998E+03, 0.75630E+03, 0.77276E+03,
      0.78936E+03, 0.80612E+03, 0.82302E+03, 0.84007E+03, 0.85727E+03,
      0.87463E+03, 0.89215E+03, 0.90982E+03, 0.92765E+03, 0.94563E+03,
      0.96378E+03, 0.98209E+03, 0.10006E+04, 0.10192E+04, 0.10380E+04,
      0.10570E+04, 0.10761E+04, 0.10954E+04, 0.11149E+04, 0.11345E+04,
      0.11543E+04, 0.11743E+04, 0.11945E+04, 0.12148E+04, 0.12353E+04,
      0.12560E+04, 0.12768E+04, 0.12979E+04, 0.13191E+04, 0.13405E+04,
      0.13620E+04, 0.13838E+04, 0.14057E+04, 0.14278E+04, 0.14501E+04,
      0.14726E+04, 0.14952E+04, 0.15180E+04, 0.15410E+04, 0.15642E+04,
      0.15876E+04, 0.16112E+04, 0.16349E+04, 0.16589E+04, 0.16830E+04,
      0.17073E+04, 0.17318E+04, 0.17565E+04, 0.17814E+04, 0.18064E+04,
      0.18317E+04, 0.18572E+04, 0.18828E+04, 0.19086E+04, 0.19346E+04,
      0.19609E+04, 0.19873E+04, 0.20139E+04, 0.20406E+04, 0.20676E+04,
      0.20948E+04, 0.21222E+04, 0.21498E+04, 0.21775E+04, 0.22055E+04,
      0.22337E+04])


#  --------------- HСl 25: M = 15, I = 3 --------------------- not in TIPS-2011
M = 15
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(0.)
TIPS_ISO_HASH[(M,I)] = float32([0.])


#  --------------- HСl 27: M = 15, I = 4 --------------------- not in TIPS-2011
M = 15
I = 4
TIPS_GSI_HASH[(M,I)] = __FloatType__(0.)
TIPS_ISO_HASH[(M,I)] = float32([0.])


#  --------------- HBr 19: M = 16, I = 1 ---------------------
M = 16
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(8.)
TIPS_ISO_HASH[(M,I)] = float32([0.42744E+02, 0.59373E+02, 0.76023E+02,
      0.92685E+02, 0.10936E+03, 0.12604E+03, 0.14272E+03, 0.15942E+03,
      0.17612E+03, 0.19282E+03, 0.20954E+03, 0.22626E+03, 0.24299E+03,
      0.25973E+03, 0.27648E+03, 0.29325E+03, 0.31004E+03, 0.32686E+03,
      0.34371E+03, 0.36060E+03, 0.37753E+03, 0.39451E+03, 0.41156E+03,
      0.42868E+03, 0.44587E+03, 0.46314E+03, 0.48051E+03, 0.49798E+03,
      0.51556E+03, 0.53325E+03, 0.55106E+03, 0.56900E+03, 0.58708E+03,
      0.60530E+03, 0.62367E+03, 0.64219E+03, 0.66088E+03, 0.67972E+03,
      0.69874E+03, 0.71793E+03, 0.73730E+03, 0.75685E+03, 0.77659E+03,
      0.79652E+03, 0.81664E+03, 0.83696E+03, 0.85748E+03, 0.87820E+03,
      0.89914E+03, 0.92028E+03, 0.94163E+03, 0.96319E+03, 0.98498E+03,
      0.10070E+04, 0.10292E+04, 0.10516E+04, 0.10743E+04, 0.10972E+04,
      0.11203E+04, 0.11437E+04, 0.11673E+04, 0.11911E+04, 0.12151E+04,
      0.12394E+04, 0.12640E+04, 0.12887E+04, 0.13137E+04, 0.13390E+04,
      0.13645E+04, 0.13902E+04, 0.14162E+04, 0.14424E+04, 0.14689E+04,
      0.14956E+04, 0.15226E+04, 0.15498E+04, 0.15773E+04, 0.16050E+04,
      0.16330E+04, 0.16612E+04, 0.16897E+04, 0.17185E+04, 0.17475E+04,
      0.17767E+04, 0.18062E+04, 0.18360E+04, 0.18660E+04, 0.18963E+04,
      0.19269E+04, 0.19577E+04, 0.19888E+04, 0.20202E+04, 0.20518E+04,
      0.20837E+04, 0.21158E+04, 0.21482E+04, 0.21809E+04, 0.22139E+04,
      0.22471E+04, 0.22806E+04, 0.23143E+04, 0.23484E+04, 0.23827E+04,
      0.24173E+04, 0.24521E+04, 0.24873E+04, 0.25227E+04, 0.25584E+04,
      0.25943E+04, 0.26306E+04, 0.26671E+04, 0.27039E+04, 0.27409E+04,
      0.27783E+04, 0.28159E+04, 0.28538E+04, 0.28920E+04, 0.29305E+04,
      0.29693E+04])


#  --------------- HBr 11: M = 16, I = 2 ---------------------
M = 16
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(8.)
TIPS_ISO_HASH[(M,I)] = float32([0.42756E+02, 0.59390E+02, 0.76045E+02,
      0.92713E+02, 0.10939E+03, 0.12607E+03, 0.14277E+03, 0.15947E+03,
      0.17617E+03, 0.19288E+03, 0.20960E+03, 0.22633E+03, 0.24306E+03,
      0.25981E+03, 0.27656E+03, 0.29334E+03, 0.31014E+03, 0.32696E+03,
      0.34381E+03, 0.36071E+03, 0.37764E+03, 0.39464E+03, 0.41169E+03,
      0.42881E+03, 0.44601E+03, 0.46329E+03, 0.48066E+03, 0.49813E+03,
      0.51572E+03, 0.53341E+03, 0.55123E+03, 0.56918E+03, 0.58727E+03,
      0.60549E+03, 0.62387E+03, 0.64240E+03, 0.66109E+03, 0.67994E+03,
      0.69896E+03, 0.71816E+03, 0.73754E+03, 0.75710E+03, 0.77684E+03,
      0.79678E+03, 0.81691E+03, 0.83724E+03, 0.85776E+03, 0.87850E+03,
      0.89943E+03, 0.92058E+03, 0.94194E+03, 0.96352E+03, 0.98531E+03,
      0.10073E+04, 0.10295E+04, 0.10520E+04, 0.10747E+04, 0.10976E+04,
      0.11207E+04, 0.11441E+04, 0.11677E+04, 0.11915E+04, 0.12156E+04,
      0.12399E+04, 0.12644E+04, 0.12892E+04, 0.13142E+04, 0.13395E+04,
      0.13650E+04, 0.13907E+04, 0.14167E+04, 0.14429E+04, 0.14694E+04,
      0.14961E+04, 0.15231E+04, 0.15504E+04, 0.15778E+04, 0.16056E+04,
      0.16336E+04, 0.16618E+04, 0.16903E+04, 0.17191E+04, 0.17481E+04,
      0.17773E+04, 0.18069E+04, 0.18367E+04, 0.18667E+04, 0.18970E+04,
      0.19276E+04, 0.19584E+04, 0.19895E+04, 0.20209E+04, 0.20525E+04,
      0.20844E+04, 0.21166E+04, 0.21490E+04, 0.21817E+04, 0.22147E+04,
      0.22479E+04, 0.22814E+04, 0.23152E+04, 0.23492E+04, 0.23835E+04,
      0.24181E+04, 0.24530E+04, 0.24882E+04, 0.25236E+04, 0.25593E+04,
      0.25952E+04, 0.26315E+04, 0.26680E+04, 0.27048E+04, 0.27419E+04,
      0.27793E+04, 0.28169E+04, 0.28549E+04, 0.28931E+04, 0.29316E+04,
      0.29703E+04])


#  --------------- HBr 29: M = 16, I = 3 --------------------- not in TIPS-2011
M = 16
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(0.)
TIPS_ISO_HASH[(M,I)] = float32([0.])


#  --------------- HBr 21: M = 16, I = 4 --------------------- not in TIPS-2011
M = 16
I = 4
TIPS_GSI_HASH[(M,I)] = __FloatType__(0.)
TIPS_ISO_HASH[(M,I)] = float32([0.])


#  --------------- HI 17: M = 17, I = 1 ---------------------
M = 17
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(12.)
TIPS_ISO_HASH[(M,I)] = float32([0.82031E+02, 0.11447E+03, 0.14694E+03,
      0.17943E+03, 0.21194E+03, 0.24445E+03, 0.27699E+03, 0.30953E+03,
      0.34209E+03, 0.37466E+03, 0.40725E+03, 0.43986E+03, 0.47249E+03,
      0.50517E+03, 0.53789E+03, 0.57068E+03, 0.60354E+03, 0.63650E+03,
      0.66957E+03, 0.70278E+03, 0.73614E+03, 0.76967E+03, 0.80340E+03,
      0.83735E+03, 0.87153E+03, 0.90596E+03, 0.94067E+03, 0.97566E+03,
      0.10110E+04, 0.10466E+04, 0.10826E+04, 0.11189E+04, 0.11555E+04,
      0.11926E+04, 0.12300E+04, 0.12679E+04, 0.13061E+04, 0.13448E+04,
      0.13839E+04, 0.14235E+04, 0.14635E+04, 0.15039E+04, 0.15448E+04,
      0.15862E+04, 0.16280E+04, 0.16704E+04, 0.17132E+04, 0.17565E+04,
      0.18003E+04, 0.18446E+04, 0.18894E+04, 0.19347E+04, 0.19806E+04,
      0.20269E+04, 0.20738E+04, 0.21212E+04, 0.21691E+04, 0.22176E+04,
      0.22666E+04, 0.23162E+04, 0.23662E+04, 0.24169E+04, 0.24680E+04,
      0.25198E+04, 0.25720E+04, 0.26249E+04, 0.26783E+04, 0.27322E+04,
      0.27867E+04, 0.28418E+04, 0.28975E+04, 0.29537E+04, 0.30105E+04,
      0.30678E+04, 0.31258E+04, 0.31843E+04, 0.32434E+04, 0.33031E+04,
      0.33633E+04, 0.34242E+04, 0.34856E+04, 0.35477E+04, 0.36103E+04,
      0.36735E+04, 0.37373E+04, 0.38018E+04, 0.38668E+04, 0.39324E+04,
      0.39986E+04, 0.40654E+04, 0.41329E+04, 0.42009E+04, 0.42696E+04,
      0.43388E+04, 0.44087E+04, 0.44792E+04, 0.45503E+04, 0.46221E+04,
      0.46944E+04, 0.47674E+04, 0.48410E+04, 0.49152E+04, 0.49901E+04,
      0.50656E+04, 0.51417E+04, 0.52185E+04, 0.52959E+04, 0.53739E+04,
      0.54526E+04, 0.55319E+04, 0.56118E+04, 0.56924E+04, 0.57736E+04,
      0.58555E+04, 0.59380E+04, 0.60212E+04, 0.61050E+04, 0.61895E+04,
      0.62746E+04])


#  --------------- HI 27: M = 17, I = 2 --------------------- not in TIPS-2011
M = 17
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(0.)
TIPS_ISO_HASH[(M,I)] = float32([0.])


#  --------------- ClO 56: M = 18, I = 1 ---------------------
M = 18
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(4.)
TIPS_ISO_HASH[(M,I)] = float32([0.53847E+03, 0.76580E+03, 0.10017E+04,
      0.12511E+04, 0.15168E+04, 0.18001E+04, 0.21014E+04, 0.24206E+04,
      0.27577E+04, 0.31127E+04, 0.34857E+04, 0.38765E+04, 0.42854E+04,
      0.47124E+04, 0.51575E+04, 0.56208E+04, 0.61025E+04, 0.66026E+04,
      0.71211E+04, 0.76582E+04, 0.82138E+04, 0.87882E+04, 0.93813E+04,
      0.99932E+04, 0.10624E+05, 0.11273E+05, 0.11942E+05, 0.12629E+05,
      0.13336E+05, 0.14061E+05, 0.14806E+05, 0.15570E+05, 0.16353E+05,
      0.17155E+05, 0.17976E+05, 0.18816E+05, 0.19676E+05, 0.20555E+05,
      0.21453E+05, 0.22371E+05, 0.23308E+05, 0.24264E+05, 0.25240E+05,
      0.26236E+05, 0.27250E+05, 0.28284E+05, 0.29338E+05, 0.30412E+05,
      0.31505E+05, 0.32617E+05, 0.33749E+05, 0.34901E+05, 0.36072E+05,
      0.37263E+05, 0.38474E+05, 0.39705E+05, 0.40955E+05, 0.42225E+05,
      0.43515E+05, 0.44825E+05, 0.46154E+05, 0.47504E+05, 0.48873E+05,
      0.50262E+05, 0.51672E+05, 0.53101E+05, 0.54549E+05, 0.56019E+05,
      0.57508E+05, 0.59017E+05, 0.60546E+05, 0.62095E+05, 0.63665E+05,
      0.65254E+05, 0.66864E+05, 0.68494E+05, 0.70144E+05, 0.71814E+05,
      0.73504E+05, 0.75215E+05, 0.76946E+05, 0.78698E+05, 0.80470E+05,
      0.82261E+05, 0.84074E+05, 0.85907E+05, 0.87760E+05, 0.89633E+05,
      0.91527E+05, 0.93442E+05, 0.95377E+05, 0.97333E+05, 0.99309E+05,
      0.10131E+06, 0.10332E+06, 0.10536E+06, 0.10742E+06, 0.10950E+06,
      0.11160E+06, 0.11372E+06, 0.11586E+06, 0.11802E+06, 0.12020E+06,
      0.12241E+06, 0.12463E+06, 0.12688E+06, 0.12914E+06, 0.13143E+06,
      0.13374E+06, 0.13607E+06, 0.13842E+06, 0.14079E+06, 0.14318E+06,
      0.14559E+06, 0.14802E+06, 0.15048E+06, 0.15295E+06, 0.15545E+06,
      0.15797E+06])


#  --------------- ClO 76: M = 18, I = 2 ---------------------
M = 18
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(4.)
TIPS_ISO_HASH[(M,I)] = float32([0.54775E+03, 0.77899E+03, 0.10189E+04,
      0.12726E+04, 0.15430E+04, 0.18313E+04, 0.21378E+04, 0.24627E+04,
      0.28059E+04, 0.31674E+04, 0.35472E+04, 0.39454E+04, 0.43621E+04,
      0.47972E+04, 0.52508E+04, 0.57232E+04, 0.62143E+04, 0.67242E+04,
      0.72531E+04, 0.78010E+04, 0.83678E+04, 0.89537E+04, 0.95589E+04,
      0.10183E+05, 0.10827E+05, 0.11490E+05, 0.12172E+05, 0.12874E+05,
      0.13595E+05, 0.14335E+05, 0.15095E+05, 0.15875E+05, 0.16674E+05,
      0.17493E+05, 0.18332E+05, 0.19190E+05, 0.20068E+05, 0.20965E+05,
      0.21882E+05, 0.22820E+05, 0.23776E+05, 0.24753E+05, 0.25750E+05,
      0.26766E+05, 0.27803E+05, 0.28859E+05, 0.29935E+05, 0.31032E+05,
      0.32148E+05, 0.33284E+05, 0.34441E+05, 0.35617E+05, 0.36814E+05,
      0.38031E+05, 0.39267E+05, 0.40524E+05, 0.41802E+05, 0.43099E+05,
      0.44417E+05, 0.45755E+05, 0.47113E+05, 0.48492E+05, 0.49891E+05,
      0.51310E+05, 0.52750E+05, 0.54210E+05, 0.55690E+05, 0.57191E+05,
      0.58713E+05, 0.60255E+05, 0.61817E+05, 0.63400E+05, 0.65004E+05,
      0.66628E+05, 0.68272E+05, 0.69938E+05, 0.71624E+05, 0.73331E+05,
      0.75058E+05, 0.76806E+05, 0.78575E+05, 0.80364E+05, 0.82175E+05,
      0.84006E+05, 0.85858E+05, 0.87731E+05, 0.89625E+05, 0.91539E+05,
      0.93475E+05, 0.95431E+05, 0.97409E+05, 0.99407E+05, 0.10143E+06,
      0.10347E+06, 0.10553E+06, 0.10761E+06, 0.10972E+06, 0.11184E+06,
      0.11399E+06, 0.11615E+06, 0.11834E+06, 0.12055E+06, 0.12278E+06,
      0.12503E+06, 0.12731E+06, 0.12960E+06, 0.13192E+06, 0.13425E+06,
      0.13661E+06, 0.13899E+06, 0.14139E+06, 0.14382E+06, 0.14626E+06,
      0.14873E+06, 0.15121E+06, 0.15372E+06, 0.15625E+06, 0.15880E+06,
      0.16138E+06])


#  --------------- OCS 622: M = 19, I = 1 ---------------------
M = 19
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.20609E+03, 0.29199E+03, 0.37861E+03,
      0.46737E+03, 0.56024E+03, 0.65929E+03, 0.76649E+03, 0.88361E+03,
      0.10123E+04, 0.11541E+04, 0.13105E+04, 0.14829E+04, 0.16728E+04,
      0.18818E+04, 0.21113E+04, 0.23629E+04, 0.26383E+04, 0.29391E+04,
      0.32672E+04, 0.36245E+04, 0.40128E+04, 0.44343E+04, 0.48911E+04,
      0.53853E+04, 0.59193E+04, 0.64956E+04, 0.71166E+04, 0.77849E+04,
      0.85033E+04, 0.92746E+04, 0.10102E+05, 0.10988E+05, 0.11936E+05,
      0.12949E+05, 0.14032E+05, 0.15186E+05, 0.16416E+05, 0.17726E+05,
      0.19120E+05, 0.20601E+05, 0.22173E+05, 0.23842E+05, 0.25611E+05,
      0.27484E+05, 0.29468E+05, 0.31566E+05, 0.33783E+05, 0.36124E+05,
      0.38595E+05, 0.41202E+05, 0.43949E+05, 0.46842E+05, 0.49888E+05,
      0.53092E+05, 0.56460E+05, 0.59999E+05, 0.63716E+05, 0.67616E+05,
      0.71708E+05, 0.75997E+05, 0.80491E+05, 0.85197E+05, 0.90124E+05,
      0.95278E+05, 0.10067E+06, 0.10630E+06, 0.11219E+06, 0.11833E+06,
      0.12475E+06, 0.13144E+06, 0.13842E+06, 0.14570E+06, 0.15328E+06,
      0.16117E+06, 0.16940E+06, 0.17795E+06, 0.18686E+06, 0.19611E+06,
      0.20574E+06, 0.21574E+06, 0.22613E+06, 0.23692E+06, 0.24813E+06,
      0.25975E+06, 0.27182E+06, 0.28433E+06, 0.29730E+06, 0.31074E+06,
      0.32467E+06, 0.33909E+06, 0.35403E+06, 0.36950E+06, 0.38551E+06,
      0.40207E+06, 0.41920E+06, 0.43691E+06, 0.45522E+06, 0.47415E+06,
      0.49370E+06, 0.51390E+06, 0.53476E+06, 0.55629E+06, 0.57852E+06,
      0.60146E+06, 0.62513E+06, 0.64954E+06, 0.67471E+06, 0.70067E+06,
      0.72742E+06, 0.75499E+06, 0.78339E+06, 0.81265E+06, 0.84279E+06,
      0.87381E+06, 0.90576E+06, 0.93863E+06, 0.97246E+06, 0.10073E+07,
      0.10431E+07])


#  --------------- OCS 624: M = 19, I = 2 ---------------------
M = 19
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.21125E+03, 0.29930E+03, 0.38809E+03,
      0.47911E+03, 0.57437E+03, 0.67603E+03, 0.78610E+03, 0.90643E+03,
      0.10387E+04, 0.11846E+04, 0.13456E+04, 0.15231E+04, 0.17188E+04,
      0.19342E+04, 0.21709E+04, 0.24304E+04, 0.27145E+04, 0.30250E+04,
      0.33638E+04, 0.37328E+04, 0.41339E+04, 0.45694E+04, 0.50415E+04,
      0.55524E+04, 0.61045E+04, 0.67004E+04, 0.73427E+04, 0.80340E+04,
      0.87773E+04, 0.95755E+04, 0.10432E+05, 0.11349E+05, 0.12330E+05,
      0.13380E+05, 0.14500E+05, 0.15696E+05, 0.16970E+05, 0.18327E+05,
      0.19770E+05, 0.21305E+05, 0.22934E+05, 0.24663E+05, 0.26497E+05,
      0.28439E+05, 0.30495E+05, 0.32669E+05, 0.34968E+05, 0.37396E+05,
      0.39958E+05, 0.42661E+05, 0.45510E+05, 0.48511E+05, 0.51669E+05,
      0.54993E+05, 0.58487E+05, 0.62159E+05, 0.66014E+05, 0.70061E+05,
      0.74306E+05, 0.78757E+05, 0.83421E+05, 0.88305E+05, 0.93418E+05,
      0.98767E+05, 0.10436E+06, 0.11021E+06, 0.11632E+06, 0.12270E+06,
      0.12936E+06, 0.13631E+06, 0.14355E+06, 0.15111E+06, 0.15898E+06,
      0.16718E+06, 0.17572E+06, 0.18460E+06, 0.19385E+06, 0.20346E+06,
      0.21346E+06, 0.22385E+06, 0.23464E+06, 0.24585E+06, 0.25748E+06,
      0.26956E+06, 0.28209E+06, 0.29509E+06, 0.30856E+06, 0.32252E+06,
      0.33699E+06, 0.35198E+06, 0.36750E+06, 0.38357E+06, 0.40020E+06,
      0.41741E+06, 0.43521E+06, 0.45362E+06, 0.47264E+06, 0.49231E+06,
      0.51263E+06, 0.53362E+06, 0.55529E+06, 0.57768E+06, 0.60078E+06,
      0.62462E+06, 0.64922E+06, 0.67459E+06, 0.70075E+06, 0.72773E+06,
      0.75554E+06, 0.78419E+06, 0.81372E+06, 0.84413E+06, 0.87546E+06,
      0.90771E+06, 0.94092E+06, 0.97509E+06, 0.10103E+07, 0.10464E+07,
      0.10837E+07])


#  --------------- OCS 632: M = 19, I = 3 ---------------------
M = 19
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.41351E+03, 0.58591E+03, 0.76004E+03,
      0.93907E+03, 0.11273E+04, 0.13289E+04, 0.15481E+04, 0.17884E+04,
      0.20533E+04, 0.23459E+04, 0.26692E+04, 0.30264E+04, 0.34205E+04,
      0.38547E+04, 0.43323E+04, 0.48565E+04, 0.54309E+04, 0.60592E+04,
      0.67451E+04, 0.74928E+04, 0.83064E+04, 0.91903E+04, 0.10149E+05,
      0.11187E+05, 0.12310E+05, 0.13523E+05, 0.14831E+05, 0.16240E+05,
      0.17756E+05, 0.19384E+05, 0.21132E+05, 0.23005E+05, 0.25011E+05,
      0.27157E+05, 0.29449E+05, 0.31896E+05, 0.34506E+05, 0.37286E+05,
      0.40245E+05, 0.43392E+05, 0.46735E+05, 0.50284E+05, 0.54048E+05,
      0.58038E+05, 0.62263E+05, 0.66733E+05, 0.71460E+05, 0.76455E+05,
      0.81728E+05, 0.87292E+05, 0.93159E+05, 0.99341E+05, 0.10585E+06,
      0.11270E+06, 0.11991E+06, 0.12748E+06, 0.13543E+06, 0.14378E+06,
      0.15255E+06, 0.16174E+06, 0.17137E+06, 0.18146E+06, 0.19202E+06,
      0.20308E+06, 0.21465E+06, 0.22674E+06, 0.23937E+06, 0.25257E+06,
      0.26635E+06, 0.28073E+06, 0.29573E+06, 0.31137E+06, 0.32767E+06,
      0.34466E+06, 0.36235E+06, 0.38076E+06, 0.39992E+06, 0.41985E+06,
      0.44057E+06, 0.46211E+06, 0.48450E+06, 0.50775E+06, 0.53189E+06,
      0.55695E+06, 0.58295E+06, 0.60992E+06, 0.63789E+06, 0.66688E+06,
      0.69693E+06, 0.72806E+06, 0.76030E+06, 0.79368E+06, 0.82823E+06,
      0.86399E+06, 0.90097E+06, 0.93923E+06, 0.97878E+06, 0.10197E+07,
      0.10619E+07, 0.11056E+07, 0.11506E+07, 0.11972E+07, 0.12453E+07,
      0.12949E+07, 0.13460E+07, 0.13988E+07, 0.14533E+07, 0.15094E+07,
      0.15673E+07, 0.16270E+07, 0.16884E+07, 0.17518E+07, 0.18170E+07,
      0.18842E+07, 0.19533E+07, 0.20245E+07, 0.20978E+07, 0.21732E+07,
      0.22507E+07])


#  --------------- OCS 623: M = 19, I = 4 ---------------------
M = 19
I = 4
TIPS_GSI_HASH[(M,I)] = __FloatType__(4.)
TIPS_ISO_HASH[(M,I)] = float32([0.83485E+03, 0.11828E+04, 0.15337E+04,
      0.18934E+04, 0.22697E+04, 0.26712E+04, 0.31059E+04, 0.35809E+04,
      0.41030E+04, 0.46785E+04, 0.53133E+04, 0.60135E+04, 0.67850E+04,
      0.76338E+04, 0.85663E+04, 0.95888E+04, 0.10708E+05, 0.11931E+05,
      0.13265E+05, 0.14718E+05, 0.16298E+05, 0.18012E+05, 0.19870E+05,
      0.21881E+05, 0.24054E+05, 0.26399E+05, 0.28926E+05, 0.31646E+05,
      0.34570E+05, 0.37710E+05, 0.41077E+05, 0.44685E+05, 0.48545E+05,
      0.52672E+05, 0.57078E+05, 0.61780E+05, 0.66790E+05, 0.72125E+05,
      0.77801E+05, 0.83833E+05, 0.90239E+05, 0.97036E+05, 0.10424E+06,
      0.11188E+06, 0.11996E+06, 0.12850E+06, 0.13754E+06, 0.14708E+06,
      0.15715E+06, 0.16777E+06, 0.17896E+06, 0.19076E+06, 0.20317E+06,
      0.21623E+06, 0.22996E+06, 0.24438E+06, 0.25953E+06, 0.27543E+06,
      0.29211E+06, 0.30959E+06, 0.32791E+06, 0.34710E+06, 0.36718E+06,
      0.38820E+06, 0.41017E+06, 0.43314E+06, 0.45713E+06, 0.48219E+06,
      0.50835E+06, 0.53564E+06, 0.56409E+06, 0.59376E+06, 0.62468E+06,
      0.65688E+06, 0.69041E+06, 0.72530E+06, 0.76161E+06, 0.79937E+06,
      0.83862E+06, 0.87941E+06, 0.92179E+06, 0.96581E+06, 0.10115E+07,
      0.10589E+07, 0.11081E+07, 0.11591E+07, 0.12120E+07, 0.12669E+07,
      0.13237E+07, 0.13825E+07, 0.14435E+07, 0.15066E+07, 0.15718E+07,
      0.16394E+07, 0.17093E+07, 0.17815E+07, 0.18562E+07, 0.19334E+07,
      0.20132E+07, 0.20956E+07, 0.21807E+07, 0.22685E+07, 0.23592E+07,
      0.24528E+07, 0.25494E+07, 0.26490E+07, 0.27517E+07, 0.28576E+07,
      0.29667E+07, 0.30792E+07, 0.31951E+07, 0.33145E+07, 0.34374E+07,
      0.35640E+07, 0.36943E+07, 0.38285E+07, 0.39665E+07, 0.41085E+07,
      0.42546E+07])


#  --------------- OCS 822: M = 19, I = 5 ---------------------
M = 19
I = 5
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.21967E+03, 0.31126E+03, 0.40370E+03,
      0.49862E+03, 0.59823E+03, 0.70481E+03, 0.82050E+03, 0.94724E+03,
      0.10868E+04, 0.12409E+04, 0.14112E+04, 0.15993E+04, 0.18067E+04,
      0.20353E+04, 0.22866E+04, 0.25624E+04, 0.28645E+04, 0.31950E+04,
      0.35558E+04, 0.39490E+04, 0.43767E+04, 0.48413E+04, 0.53452E+04,
      0.58909E+04, 0.64810E+04, 0.71182E+04, 0.78053E+04, 0.85454E+04,
      0.93413E+04, 0.10196E+05, 0.11114E+05, 0.12098E+05, 0.13151E+05,
      0.14277E+05, 0.15480E+05, 0.16764E+05, 0.18133E+05, 0.19592E+05,
      0.21144E+05, 0.22794E+05, 0.24548E+05, 0.26409E+05, 0.28383E+05,
      0.30475E+05, 0.32689E+05, 0.35033E+05, 0.37511E+05, 0.40128E+05,
      0.42892E+05, 0.45808E+05, 0.48882E+05, 0.52121E+05, 0.55532E+05,
      0.59121E+05, 0.62895E+05, 0.66861E+05, 0.71028E+05, 0.75402E+05,
      0.79991E+05, 0.84803E+05, 0.89847E+05, 0.95130E+05, 0.10066E+06,
      0.10645E+06, 0.11251E+06, 0.11883E+06, 0.12545E+06, 0.13236E+06,
      0.13957E+06, 0.14710E+06, 0.15495E+06, 0.16313E+06, 0.17166E+06,
      0.18055E+06, 0.18980E+06, 0.19944E+06, 0.20946E+06, 0.21989E+06,
      0.23073E+06, 0.24200E+06, 0.25371E+06, 0.26587E+06, 0.27850E+06,
      0.29161E+06, 0.30521E+06, 0.31931E+06, 0.33394E+06, 0.34910E+06,
      0.36482E+06, 0.38109E+06, 0.39795E+06, 0.41541E+06, 0.43348E+06,
      0.45217E+06, 0.47151E+06, 0.49151E+06, 0.51219E+06, 0.53356E+06,
      0.55565E+06, 0.57847E+06, 0.60204E+06, 0.62637E+06, 0.65149E+06,
      0.67742E+06, 0.70417E+06, 0.73176E+06, 0.76023E+06, 0.78957E+06,
      0.81982E+06, 0.85100E+06, 0.88313E+06, 0.91622E+06, 0.95031E+06,
      0.98541E+06, 0.10216E+07, 0.10587E+07, 0.10970E+07, 0.11364E+07,
      0.11769E+07])


#  --------------- H2CO 126: M = 20, I = 2 ---------------------
M = 20
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.25934E+03, 0.43623E+03, 0.64143E+03,
      0.87152E+03, 0.11241E+04, 0.13975E+04, 0.16906E+04, 0.20029E+04,
      0.23344E+04, 0.26857E+04, 0.30577E+04, 0.34518E+04, 0.38698E+04,
      0.43138E+04, 0.47860E+04, 0.52890E+04, 0.58256E+04, 0.63985E+04,
      0.70109E+04, 0.76660E+04, 0.83673E+04, 0.91184E+04, 0.99230E+04,
      0.10785E+05, 0.11710E+05, 0.12700E+05, 0.13762E+05, 0.14900E+05,
      0.16119E+05, 0.17425E+05, 0.18823E+05, 0.20320E+05, 0.21923E+05,
      0.23637E+05, 0.25471E+05, 0.27432E+05, 0.29527E+05, 0.31765E+05,
      0.34155E+05, 0.36706E+05, 0.39428E+05, 0.42330E+05, 0.45424E+05,
      0.48720E+05, 0.52231E+05, 0.55968E+05, 0.59945E+05, 0.64175E+05,
      0.68672E+05, 0.73450E+05, 0.78526E+05, 0.83915E+05, 0.89634E+05,
      0.95701E+05, 0.10213E+06, 0.10895E+06, 0.11618E+06, 0.12383E+06,
      0.13193E+06, 0.14049E+06, 0.14956E+06, 0.15914E+06, 0.16927E+06,
      0.17997E+06, 0.19127E+06, 0.20320E+06, 0.21578E+06, 0.22906E+06,
      0.24306E+06, 0.25782E+06, 0.27336E+06, 0.28974E+06, 0.30698E+06,
      0.32513E+06, 0.34422E+06, 0.36430E+06, 0.38542E+06, 0.40761E+06,
      0.43093E+06, 0.45542E+06, 0.48114E+06, 0.50813E+06, 0.53646E+06,
      0.56617E+06, 0.59733E+06, 0.63000E+06, 0.66423E+06, 0.70010E+06,
      0.73767E+06, 0.77701E+06, 0.81818E+06, 0.86127E+06, 0.90635E+06,
      0.95349E+06, 0.10028E+07, 0.10543E+07, 0.11082E+07, 0.11644E+07,
      0.12232E+07, 0.12845E+07, 0.13485E+07, 0.14154E+07, 0.14851E+07,
      0.15578E+07, 0.16337E+07, 0.17127E+07, 0.17952E+07, 0.18810E+07,
      0.19705E+07, 0.20637E+07, 0.21607E+07, 0.22617E+07, 0.23669E+07,
      0.24763E+07, 0.25901E+07, 0.27085E+07, 0.28316E+07, 0.29596E+07,
      0.30926E+07])


#  --------------- H2CO 136: M = 20, I = 2 ---------------------
M = 20
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.53173E+03, 0.89447E+03, 0.13153E+04,
      0.17871E+04, 0.23051E+04, 0.28658E+04, 0.34669E+04, 0.41073E+04,
      0.47872E+04, 0.55074E+04, 0.62702E+04, 0.70785E+04, 0.79357E+04,
      0.88462E+04, 0.98147E+04, 0.10846E+05, 0.11946E+05, 0.13121E+05,
      0.14377E+05, 0.15721E+05, 0.17159E+05, 0.18699E+05, 0.20349E+05,
      0.22118E+05, 0.24013E+05, 0.26045E+05, 0.28222E+05, 0.30555E+05,
      0.33055E+05, 0.35733E+05, 0.38601E+05, 0.41671E+05, 0.44958E+05,
      0.48474E+05, 0.52235E+05, 0.56255E+05, 0.60552E+05, 0.65142E+05,
      0.70043E+05, 0.75275E+05, 0.80856E+05, 0.86808E+05, 0.93152E+05,
      0.99913E+05, 0.10711E+06, 0.11478E+06, 0.12293E+06, 0.13161E+06,
      0.14083E+06, 0.15063E+06, 0.16104E+06, 0.17209E+06, 0.18382E+06,
      0.19626E+06, 0.20945E+06, 0.22343E+06, 0.23825E+06, 0.25394E+06,
      0.27054E+06, 0.28812E+06, 0.30671E+06, 0.32636E+06, 0.34713E+06,
      0.36907E+06, 0.39224E+06, 0.41671E+06, 0.44252E+06, 0.46975E+06,
      0.49845E+06, 0.52872E+06, 0.56060E+06, 0.59418E+06, 0.62954E+06,
      0.66676E+06, 0.70591E+06, 0.74710E+06, 0.79040E+06, 0.83591E+06,
      0.88373E+06, 0.93395E+06, 0.98669E+06, 0.10421E+07, 0.11001E+07,
      0.11611E+07, 0.12250E+07, 0.12920E+07, 0.13622E+07, 0.14357E+07,
      0.15128E+07, 0.15934E+07, 0.16779E+07, 0.17662E+07, 0.18587E+07,
      0.19554E+07, 0.20565E+07, 0.21621E+07, 0.22725E+07, 0.23879E+07,
      0.25084E+07, 0.26342E+07, 0.27655E+07, 0.29026E+07, 0.30456E+07,
      0.31947E+07, 0.33502E+07, 0.35124E+07, 0.36814E+07, 0.38575E+07,
      0.40410E+07, 0.42321E+07, 0.44311E+07, 0.46382E+07, 0.48538E+07,
      0.50782E+07, 0.53116E+07, 0.55544E+07, 0.58068E+07, 0.60693E+07,
      0.63421E+07])


#  --------------- H2CO 128: M = 20, I = 3 ---------------------
M = 20
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.27198E+03, 0.45755E+03, 0.67282E+03,
      0.91421E+03, 0.11792E+04, 0.14660E+04, 0.17735E+04, 0.21012E+04,
      0.24490E+04, 0.28175E+04, 0.32077E+04, 0.36212E+04, 0.40598E+04,
      0.45256E+04, 0.50211E+04, 0.55488E+04, 0.61116E+04, 0.67127E+04,
      0.73552E+04, 0.80426E+04, 0.87783E+04, 0.95663E+04, 0.10410E+05,
      0.11315E+05, 0.12285E+05, 0.13324E+05, 0.14438E+05, 0.15632E+05,
      0.16911E+05, 0.18281E+05, 0.19748E+05, 0.21319E+05, 0.23000E+05,
      0.24799E+05, 0.26723E+05, 0.28780E+05, 0.30978E+05, 0.33326E+05,
      0.35834E+05, 0.38510E+05, 0.41365E+05, 0.44410E+05, 0.47656E+05,
      0.51115E+05, 0.54798E+05, 0.58719E+05, 0.62891E+05, 0.67329E+05,
      0.72047E+05, 0.77060E+05, 0.82385E+05, 0.88039E+05, 0.94039E+05,
      0.10040E+06, 0.10715E+06, 0.11431E+06, 0.12189E+06, 0.12991E+06,
      0.13841E+06, 0.14740E+06, 0.15691E+06, 0.16696E+06, 0.17759E+06,
      0.18882E+06, 0.20067E+06, 0.21318E+06, 0.22639E+06, 0.24032E+06,
      0.25501E+06, 0.27049E+06, 0.28680E+06, 0.30398E+06, 0.32207E+06,
      0.34111E+06, 0.36114E+06, 0.38221E+06, 0.40436E+06, 0.42765E+06,
      0.45211E+06, 0.47781E+06, 0.50479E+06, 0.53311E+06, 0.56283E+06,
      0.59400E+06, 0.62669E+06, 0.66097E+06, 0.69688E+06, 0.73451E+06,
      0.77393E+06, 0.81520E+06, 0.85840E+06, 0.90360E+06, 0.95090E+06,
      0.10004E+07, 0.10521E+07, 0.11061E+07, 0.11626E+07, 0.12216E+07,
      0.12833E+07, 0.13476E+07, 0.14148E+07, 0.14849E+07, 0.15581E+07,
      0.16344E+07, 0.17140E+07, 0.17969E+07, 0.18834E+07, 0.19735E+07,
      0.20674E+07, 0.21651E+07, 0.22669E+07, 0.23729E+07, 0.24832E+07,
      0.25980E+07, 0.27174E+07, 0.28416E+07, 0.29708E+07, 0.31050E+07,
      0.32446E+07])


#  --------------- HOCl 165: M = 21, I = 1 ---------------------
M = 21
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(8.)
TIPS_ISO_HASH[(M,I)] = float32([0.17041E+04, 0.28708E+04, 0.42250E+04,
      0.57456E+04, 0.74211E+04, 0.92470E+04, 0.11225E+05, 0.13359E+05,
      0.15657E+05, 0.18129E+05, 0.20785E+05, 0.23637E+05, 0.26696E+05,
      0.29974E+05, 0.33484E+05, 0.37239E+05, 0.41252E+05, 0.45536E+05,
      0.50105E+05, 0.54973E+05, 0.60152E+05, 0.65659E+05, 0.71507E+05,
      0.77711E+05, 0.84286E+05, 0.91249E+05, 0.98614E+05, 0.10640E+06,
      0.11462E+06, 0.12330E+06, 0.13244E+06, 0.14208E+06, 0.15222E+06,
      0.16289E+06, 0.17411E+06, 0.18589E+06, 0.19825E+06, 0.21123E+06,
      0.22483E+06, 0.23908E+06, 0.25400E+06, 0.26962E+06, 0.28596E+06,
      0.30303E+06, 0.32087E+06, 0.33950E+06, 0.35895E+06, 0.37923E+06,
      0.40038E+06, 0.42243E+06, 0.44539E+06, 0.46930E+06, 0.49419E+06,
      0.52008E+06, 0.54700E+06, 0.57498E+06, 0.60406E+06, 0.63426E+06,
      0.66562E+06, 0.69816E+06, 0.73192E+06, 0.76692E+06, 0.80322E+06,
      0.84083E+06, 0.87979E+06, 0.92014E+06, 0.96192E+06, 0.10052E+07,
      0.10499E+07, 0.10961E+07, 0.11440E+07, 0.11934E+07, 0.12445E+07,
      0.12973E+07, 0.13518E+07, 0.14081E+07, 0.14661E+07, 0.15261E+07,
      0.15879E+07, 0.16516E+07, 0.17174E+07, 0.17851E+07, 0.18550E+07,
      0.19269E+07, 0.20010E+07, 0.20773E+07, 0.21559E+07, 0.22367E+07,
      0.23200E+07, 0.24056E+07, 0.24936E+07, 0.25842E+07, 0.26773E+07,
      0.27730E+07, 0.28714E+07, 0.29724E+07, 0.30763E+07, 0.31829E+07,
      0.32924E+07, 0.34049E+07, 0.35203E+07, 0.36387E+07, 0.37603E+07,
      0.38850E+07, 0.40129E+07, 0.41441E+07, 0.42786E+07, 0.44165E+07,
      0.45579E+07, 0.47028E+07, 0.48512E+07, 0.50033E+07, 0.51592E+07,
      0.53187E+07, 0.54822E+07, 0.56495E+07, 0.58208E+07, 0.59961E+07,
      0.61755E+07])


#  --------------- HOCl 167: M = 21, I = 2 ---------------------
M = 21
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(8.)
TIPS_ISO_HASH[(M,I)] = float32([0.17342E+04, 0.29215E+04, 0.42998E+04,
      0.58473E+04, 0.75524E+04, 0.94107E+04, 0.11423E+05, 0.13595E+05,
      0.15935E+05, 0.18450E+05, 0.21154E+05, 0.24056E+05, 0.27168E+05,
      0.30505E+05, 0.34077E+05, 0.37899E+05, 0.41983E+05, 0.46343E+05,
      0.50993E+05, 0.55947E+05, 0.61218E+05, 0.66822E+05, 0.72774E+05,
      0.79088E+05, 0.85780E+05, 0.92866E+05, 0.10036E+06, 0.10829E+06,
      0.11665E+06, 0.12548E+06, 0.13479E+06, 0.14460E+06, 0.15492E+06,
      0.16578E+06, 0.17719E+06, 0.18918E+06, 0.20177E+06, 0.21497E+06,
      0.22881E+06, 0.24332E+06, 0.25851E+06, 0.27440E+06, 0.29102E+06,
      0.30840E+06, 0.32656E+06, 0.34552E+06, 0.36531E+06, 0.38595E+06,
      0.40748E+06, 0.42991E+06, 0.45328E+06, 0.47762E+06, 0.50295E+06,
      0.52929E+06, 0.55669E+06, 0.58517E+06, 0.61477E+06, 0.64550E+06,
      0.67741E+06, 0.71053E+06, 0.74489E+06, 0.78052E+06, 0.81745E+06,
      0.85573E+06, 0.89539E+06, 0.93645E+06, 0.97897E+06, 0.10230E+07,
      0.10685E+07, 0.11156E+07, 0.11643E+07, 0.12146E+07, 0.12666E+07,
      0.13203E+07, 0.13757E+07, 0.14330E+07, 0.14921E+07, 0.15531E+07,
      0.16160E+07, 0.16809E+07, 0.17478E+07, 0.18168E+07, 0.18878E+07,
      0.19611E+07, 0.20365E+07, 0.21141E+07, 0.21941E+07, 0.22764E+07,
      0.23611E+07, 0.24482E+07, 0.25378E+07, 0.26300E+07, 0.27248E+07,
      0.28222E+07, 0.29223E+07, 0.30251E+07, 0.31308E+07, 0.32393E+07,
      0.33508E+07, 0.34652E+07, 0.35827E+07, 0.37032E+07, 0.38269E+07,
      0.39539E+07, 0.40840E+07, 0.42176E+07, 0.43545E+07, 0.44948E+07,
      0.46387E+07, 0.47861E+07, 0.49372E+07, 0.50920E+07, 0.52506E+07,
      0.54130E+07, 0.55793E+07, 0.57496E+07, 0.59239E+07, 0.61024E+07,
      0.62850E+07])


#  --------------- N2 44: M = 22, I = 1 ---------------------
M = 22
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.95487E+02, 0.13466E+03, 0.17386E+03,
      0.21307E+03, 0.25230E+03, 0.29154E+03, 0.33080E+03, 0.37008E+03,
      0.40937E+03, 0.44868E+03, 0.48800E+03, 0.52736E+03, 0.56674E+03,
      0.60616E+03, 0.64562E+03, 0.68515E+03, 0.72475E+03, 0.76445E+03,
      0.80426E+03, 0.84420E+03, 0.88430E+03, 0.92457E+03, 0.96505E+03,
      0.10057E+04, 0.10467E+04, 0.10879E+04, 0.11293E+04, 0.11711E+04,
      0.12132E+04, 0.12556E+04, 0.12984E+04, 0.13416E+04, 0.13851E+04,
      0.14291E+04, 0.14734E+04, 0.15182E+04, 0.15635E+04, 0.16091E+04,
      0.16553E+04, 0.17019E+04, 0.17490E+04, 0.17965E+04, 0.18446E+04,
      0.18932E+04, 0.19422E+04, 0.19918E+04, 0.20419E+04, 0.20926E+04,
      0.21437E+04, 0.21954E+04, 0.22477E+04, 0.23004E+04, 0.23538E+04,
      0.24077E+04, 0.24621E+04, 0.25171E+04, 0.25727E+04, 0.26288E+04,
      0.26856E+04, 0.27428E+04, 0.28007E+04, 0.28591E+04, 0.29181E+04,
      0.29777E+04, 0.30379E+04, 0.30986E+04, 0.31600E+04, 0.32219E+04,
      0.32844E+04, 0.33475E+04, 0.34112E+04, 0.34755E+04, 0.35404E+04,
      0.36059E+04, 0.36720E+04, 0.37387E+04, 0.38060E+04, 0.38739E+04,
      0.39424E+04, 0.40115E+04, 0.40812E+04, 0.41515E+04, 0.42224E+04,
      0.42939E+04, 0.43661E+04, 0.44388E+04, 0.45122E+04, 0.45861E+04,
      0.46607E+04, 0.47359E+04, 0.48117E+04, 0.48882E+04, 0.49652E+04,
      0.50428E+04, 0.51211E+04, 0.52000E+04, 0.52795E+04, 0.53596E+04,
      0.54404E+04, 0.55217E+04, 0.56037E+04, 0.56863E+04, 0.57695E+04,
      0.58533E+04, 0.59378E+04, 0.60229E+04, 0.61086E+04, 0.61950E+04,
      0.62819E+04, 0.63695E+04, 0.64577E+04, 0.65465E+04, 0.66360E+04,
      0.67261E+04, 0.68168E+04, 0.69081E+04, 0.70001E+04, 0.70927E+04,
      0.71859E+04])


#  --------------- N2 45: M = 22, I = 2 --------------------- not in TIPS-2011
M = 22
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(0.)
TIPS_ISO_HASH[(M,I)] = float32([0.])


#  --------------- HCN 124: M = 23, I = 1 ---------------------
M = 23
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.17143E+03, 0.24209E+03, 0.31285E+03,
      0.38392E+03, 0.45582E+03, 0.52929E+03, 0.60515E+03, 0.68424E+03,
      0.76731E+03, 0.85505E+03, 0.94805E+03, 0.10468E+04, 0.11519E+04,
      0.12637E+04, 0.13826E+04, 0.15090E+04, 0.16435E+04, 0.17863E+04,
      0.19378E+04, 0.20985E+04, 0.22689E+04, 0.24492E+04, 0.26401E+04,
      0.28418E+04, 0.30550E+04, 0.32801E+04, 0.35176E+04, 0.37680E+04,
      0.40318E+04, 0.43097E+04, 0.46021E+04, 0.49097E+04, 0.52330E+04,
      0.55727E+04, 0.59294E+04, 0.63038E+04, 0.66964E+04, 0.71081E+04,
      0.75396E+04, 0.79915E+04, 0.84646E+04, 0.89596E+04, 0.94774E+04,
      0.10019E+05, 0.10585E+05, 0.11176E+05, 0.11793E+05, 0.12437E+05,
      0.13108E+05, 0.13809E+05, 0.14540E+05, 0.15301E+05, 0.16094E+05,
      0.16919E+05, 0.17779E+05, 0.18673E+05, 0.19603E+05, 0.20570E+05,
      0.21575E+05, 0.22619E+05, 0.23704E+05, 0.24831E+05, 0.26000E+05,
      0.27213E+05, 0.28472E+05, 0.29778E+05, 0.31131E+05, 0.32534E+05,
      0.33987E+05, 0.35493E+05, 0.37052E+05, 0.38666E+05, 0.40336E+05,
      0.42064E+05, 0.43852E+05, 0.45701E+05, 0.47612E+05, 0.49587E+05,
      0.51629E+05, 0.53738E+05, 0.55916E+05, 0.58165E+05, 0.60486E+05,
      0.62883E+05, 0.65355E+05, 0.67905E+05, 0.70536E+05, 0.73249E+05,
      0.76045E+05, 0.78927E+05, 0.81897E+05, 0.84957E+05, 0.88108E+05,
      0.91354E+05, 0.94696E+05, 0.98136E+05, 0.10168E+06, 0.10532E+06,
      0.10907E+06, 0.11292E+06, 0.11689E+06, 0.12096E+06, 0.12516E+06,
      0.12946E+06, 0.13389E+06, 0.13844E+06, 0.14311E+06, 0.14791E+06,
      0.15284E+06, 0.15790E+06, 0.16310E+06, 0.16843E+06, 0.17391E+06,
      0.17953E+06, 0.18529E+06, 0.19120E+06, 0.19726E+06, 0.20348E+06,
      0.20986E+06])


#  --------------- HCN 134: M = 23, I = 2 ---------------------
M = 23
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(12.)
TIPS_ISO_HASH[(M,I)] = float32([0.35186E+03, 0.49693E+03, 0.64221E+03,
      0.78815E+03, 0.93585E+03, 0.10868E+04, 0.12428E+04, 0.14056E+04,
      0.15766E+04, 0.17574E+04, 0.19491E+04, 0.21528E+04, 0.23695E+04,
      0.26002E+04, 0.28457E+04, 0.31068E+04, 0.33845E+04, 0.36795E+04,
      0.39926E+04, 0.43249E+04, 0.46770E+04, 0.50500E+04, 0.54447E+04,
      0.58621E+04, 0.63032E+04, 0.67690E+04, 0.72606E+04, 0.77789E+04,
      0.83252E+04, 0.89005E+04, 0.95062E+04, 0.10143E+05, 0.10813E+05,
      0.11517E+05, 0.12256E+05, 0.13032E+05, 0.13846E+05, 0.14699E+05,
      0.15593E+05, 0.16530E+05, 0.17511E+05, 0.18538E+05, 0.19612E+05,
      0.20734E+05, 0.21908E+05, 0.23134E+05, 0.24414E+05, 0.25750E+05,
      0.27145E+05, 0.28599E+05, 0.30115E+05, 0.31694E+05, 0.33340E+05,
      0.35054E+05, 0.36838E+05, 0.38694E+05, 0.40625E+05, 0.42633E+05,
      0.44720E+05, 0.46889E+05, 0.49142E+05, 0.51481E+05, 0.53910E+05,
      0.56430E+05, 0.59045E+05, 0.61757E+05, 0.64568E+05, 0.67482E+05,
      0.70502E+05, 0.73630E+05, 0.76869E+05, 0.80223E+05, 0.83694E+05,
      0.87285E+05, 0.91000E+05, 0.94843E+05, 0.98815E+05, 0.10292E+06,
      0.10716E+06, 0.11155E+06, 0.11608E+06, 0.12075E+06, 0.12558E+06,
      0.13056E+06, 0.13570E+06, 0.14100E+06, 0.14647E+06, 0.15211E+06,
      0.15793E+06, 0.16392E+06, 0.17009E+06, 0.17646E+06, 0.18301E+06,
      0.18976E+06, 0.19671E+06, 0.20387E+06, 0.21123E+06, 0.21881E+06,
      0.22660E+06, 0.23462E+06, 0.24287E+06, 0.25135E+06, 0.26007E+06,
      0.26903E+06, 0.27824E+06, 0.28771E+06, 0.29743E+06, 0.30742E+06,
      0.31767E+06, 0.32820E+06, 0.33901E+06, 0.35011E+06, 0.36150E+06,
      0.37319E+06, 0.38518E+06, 0.39749E+06, 0.41010E+06, 0.42304E+06,
      0.43631E+06])


#  --------------- HCN 135: M = 23, I = 3 ---------------------
M = 23
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(4.)
TIPS_ISO_HASH[(M,I)] = float32([0.11863E+03, 0.16755E+03, 0.21653E+03,
      0.26576E+03, 0.31559E+03, 0.36656E+03, 0.41926E+03, 0.47428E+03,
      0.53214E+03, 0.59333E+03, 0.65824E+03, 0.72727E+03, 0.80074E+03,
      0.87898E+03, 0.96227E+03, 0.10509E+04, 0.11452E+04, 0.12454E+04,
      0.13518E+04, 0.14647E+04, 0.15844E+04, 0.17112E+04, 0.18455E+04,
      0.19875E+04, 0.21377E+04, 0.22962E+04, 0.24636E+04, 0.26402E+04,
      0.28263E+04, 0.30224E+04, 0.32289E+04, 0.34461E+04, 0.36745E+04,
      0.39145E+04, 0.41667E+04, 0.44314E+04, 0.47092E+04, 0.50005E+04,
      0.53059E+04, 0.56259E+04, 0.59609E+04, 0.63116E+04, 0.66785E+04,
      0.70622E+04, 0.74633E+04, 0.78823E+04, 0.83200E+04, 0.87769E+04,
      0.92536E+04, 0.97509E+04, 0.10269E+05, 0.10810E+05, 0.11373E+05,
      0.11959E+05, 0.12570E+05, 0.13205E+05, 0.13866E+05, 0.14554E+05,
      0.15268E+05, 0.16011E+05, 0.16782E+05, 0.17583E+05, 0.18415E+05,
      0.19279E+05, 0.20174E+05, 0.21103E+05, 0.22067E+05, 0.23065E+05,
      0.24100E+05, 0.25172E+05, 0.26282E+05, 0.27432E+05, 0.28622E+05,
      0.29853E+05, 0.31127E+05, 0.32445E+05, 0.33807E+05, 0.35215E+05,
      0.36670E+05, 0.38174E+05, 0.39727E+05, 0.41330E+05, 0.42986E+05,
      0.44695E+05, 0.46459E+05, 0.48278E+05, 0.50155E+05, 0.52091E+05,
      0.54086E+05, 0.56143E+05, 0.58263E+05, 0.60447E+05, 0.62696E+05,
      0.65013E+05, 0.67399E+05, 0.69856E+05, 0.72384E+05, 0.74986E+05,
      0.77663E+05, 0.80416E+05, 0.83249E+05, 0.86161E+05, 0.89156E+05,
      0.92233E+05, 0.95397E+05, 0.98648E+05, 0.10199E+06, 0.10542E+06,
      0.10894E+06, 0.11256E+06, 0.11627E+06, 0.12009E+06, 0.12400E+06,
      0.12802E+06, 0.13214E+06, 0.13636E+06, 0.14070E+06, 0.14515E+06,
      0.14971E+06])


#  --------------- CH3Cl 215: M = 24, I = 1 ---------------------
M = 24
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(4.)
TIPS_ISO_HASH[(M,I)] = float32([0.50529E+04, 0.85123E+04, 0.12528E+05,
      0.17036E+05, 0.22005E+05, 0.27429E+05, 0.33325E+05, 0.39734E+05,
      0.46713E+05, 0.54336E+05, 0.62690E+05, 0.71876E+05, 0.82006E+05,
      0.93204E+05, 0.10560E+06, 0.11936E+06, 0.13463E+06, 0.15158E+06,
      0.17043E+06, 0.19137E+06, 0.21464E+06, 0.24049E+06, 0.26920E+06,
      0.30107E+06, 0.33642E+06, 0.37563E+06, 0.41907E+06, 0.46719E+06,
      0.52045E+06, 0.57936E+06, 0.64448E+06, 0.71641E+06, 0.79582E+06,
      0.88341E+06, 0.97997E+06, 0.10863E+07, 0.12034E+07, 0.13323E+07,
      0.14739E+07, 0.16295E+07, 0.18003E+07, 0.19877E+07, 0.21932E+07,
      0.24183E+07, 0.26649E+07, 0.29346E+07, 0.32296E+07, 0.35519E+07,
      0.39039E+07, 0.42881E+07, 0.47072E+07, 0.51639E+07, 0.56615E+07,
      0.62032E+07, 0.67926E+07, 0.74335E+07, 0.81299E+07, 0.88862E+07,
      0.97071E+07, 0.10598E+08, 0.11563E+08, 0.12609E+08, 0.13742E+08,
      0.14968E+08, 0.16294E+08, 0.17728E+08, 0.19277E+08, 0.20950E+08,
      0.22756E+08, 0.24704E+08, 0.26805E+08, 0.29069E+08, 0.31507E+08,
      0.34132E+08, 0.36957E+08, 0.39995E+08, 0.43260E+08, 0.46769E+08,
      0.50538E+08, 0.54583E+08, 0.58923E+08, 0.63578E+08, 0.68568E+08,
      0.73914E+08, 0.79640E+08, 0.85770E+08, 0.92329E+08, 0.99345E+08,
      0.10685E+09, 0.11486E+09, 0.12342E+09, 0.13257E+09, 0.14233E+09,
      0.15274E+09, 0.16384E+09, 0.17568E+09, 0.18829E+09, 0.20173E+09,
      0.21604E+09, 0.23127E+09, 0.24748E+09, 0.26471E+09, 0.28304E+09,
      0.30252E+09, 0.32322E+09, 0.34520E+09, 0.36853E+09, 0.39330E+09,
      0.41958E+09, 0.44745E+09, 0.47701E+09, 0.50833E+09, 0.54151E+09,
      0.57667E+09, 0.61389E+09, 0.65329E+09, 0.69498E+09, 0.73909E+09,
      0.78573E+09])


#  --------------- CH3Cl 217: M = 24, I = 2 ---------------------
M = 24
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(4.)
TIPS_ISO_HASH[(M,I)] = float32([0.51327E+04, 0.86469E+04, 0.12726E+05,
      0.17306E+05, 0.22354E+05, 0.27863E+05, 0.33853E+05, 0.40364E+05,
      0.47453E+05, 0.55197E+05, 0.63684E+05, 0.73016E+05, 0.83306E+05,
      0.94681E+05, 0.10728E+06, 0.12125E+06, 0.13676E+06, 0.15399E+06,
      0.17313E+06, 0.19441E+06, 0.21804E+06, 0.24430E+06, 0.27347E+06,
      0.30584E+06, 0.34176E+06, 0.38158E+06, 0.42572E+06, 0.47460E+06,
      0.52871E+06, 0.58855E+06, 0.65471E+06, 0.72778E+06, 0.80844E+06,
      0.89743E+06, 0.99552E+06, 0.11036E+07, 0.12225E+07, 0.13534E+07,
      0.14973E+07, 0.16553E+07, 0.18289E+07, 0.20193E+07, 0.22280E+07,
      0.24567E+07, 0.27072E+07, 0.29812E+07, 0.32808E+07, 0.36083E+07,
      0.39659E+07, 0.43562E+07, 0.47819E+07, 0.52459E+07, 0.57514E+07,
      0.63017E+07, 0.69005E+07, 0.75515E+07, 0.82590E+07, 0.90273E+07,
      0.98613E+07, 0.10766E+08, 0.11747E+08, 0.12809E+08, 0.13960E+08,
      0.15206E+08, 0.16553E+08, 0.18010E+08, 0.19584E+08, 0.21283E+08,
      0.23118E+08, 0.25097E+08, 0.27231E+08, 0.29531E+08, 0.32008E+08,
      0.34674E+08, 0.37544E+08, 0.40630E+08, 0.43948E+08, 0.47513E+08,
      0.51341E+08, 0.55451E+08, 0.59860E+08, 0.64589E+08, 0.69658E+08,
      0.75089E+08, 0.80906E+08, 0.87134E+08, 0.93797E+08, 0.10092E+09,
      0.10854E+09, 0.11669E+09, 0.12539E+09, 0.13467E+09, 0.14459E+09,
      0.15517E+09, 0.16645E+09, 0.17847E+09, 0.19129E+09, 0.20494E+09,
      0.21948E+09, 0.23495E+09, 0.25141E+09, 0.26893E+09, 0.28754E+09,
      0.30733E+09, 0.32836E+09, 0.35069E+09, 0.37440E+09, 0.39956E+09,
      0.42626E+09, 0.45457E+09, 0.48460E+09, 0.51642E+09, 0.55013E+09,
      0.58585E+09, 0.62366E+09, 0.66369E+09, 0.70605E+09, 0.75085E+09,
      0.79824E+09])


#  --------------- H2O2 1661: M = 25, I = 1 ---------------------
M = 25
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.62392E+03, 0.10958E+04, 0.16692E+04,
      0.23492E+04, 0.31427E+04, 0.40574E+04, 0.51014E+04, 0.62840E+04,
      0.76157E+04, 0.91085E+04, 0.10776E+05, 0.12633E+05, 0.14696E+05,
      0.16983E+05, 0.19515E+05, 0.22312E+05, 0.25396E+05, 0.28792E+05,
      0.32526E+05, 0.36625E+05, 0.41118E+05, 0.46036E+05, 0.51410E+05,
      0.57275E+05, 0.63667E+05, 0.70623E+05, 0.78185E+05, 0.86394E+05,
      0.95295E+05, 0.10493E+06, 0.11536E+06, 0.12662E+06, 0.13878E+06,
      0.15188E+06, 0.16600E+06, 0.18118E+06, 0.19750E+06, 0.21503E+06,
      0.23383E+06, 0.25398E+06, 0.27556E+06, 0.29864E+06, 0.32333E+06,
      0.34970E+06, 0.37784E+06, 0.40786E+06, 0.43985E+06, 0.47392E+06,
      0.51018E+06, 0.54874E+06, 0.58972E+06, 0.63324E+06, 0.67943E+06,
      0.72843E+06, 0.78037E+06, 0.83540E+06, 0.89366E+06, 0.95530E+06,
      0.10205E+07, 0.10894E+07, 0.11622E+07, 0.12391E+07, 0.13202E+07,
      0.14057E+07, 0.14959E+07, 0.15909E+07, 0.16910E+07, 0.17963E+07,
      0.19072E+07, 0.20237E+07, 0.21463E+07, 0.22750E+07, 0.24102E+07,
      0.25522E+07, 0.27012E+07, 0.28575E+07, 0.30213E+07, 0.31931E+07,
      0.33730E+07, 0.35615E+07, 0.37588E+07, 0.39653E+07, 0.41813E+07,
      0.44072E+07, 0.46433E+07, 0.48901E+07, 0.51479E+07, 0.54171E+07,
      0.56982E+07, 0.59915E+07, 0.62976E+07, 0.66167E+07, 0.69495E+07,
      0.72963E+07, 0.76577E+07, 0.80342E+07, 0.84262E+07, 0.88343E+07,
      0.92591E+07, 0.97011E+07, 0.10161E+08, 0.10639E+08, 0.11136E+08,
      0.11652E+08, 0.12189E+08, 0.12746E+08, 0.13325E+08, 0.13926E+08,
      0.14550E+08, 0.15198E+08, 0.15870E+08, 0.16566E+08, 0.17289E+08,
      0.18038E+08, 0.18814E+08, 0.19619E+08, 0.20452E+08, 0.21315E+08,
      0.22209E+08])


#  --------------- C2H2 1221: M = 26, I = 1 ---------------------
M = 26
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.71617E+02, 0.10121E+03, 0.13092E+03,
      0.16104E+03, 0.19218E+03, 0.22509E+03, 0.26062E+03, 0.29959E+03,
      0.34281E+03, 0.39103E+03, 0.44503E+03, 0.50558E+03, 0.57346E+03,
      0.64950E+03, 0.73457E+03, 0.82960E+03, 0.93557E+03, 0.10535E+04,
      0.11846E+04, 0.13301E+04, 0.14911E+04, 0.16692E+04, 0.18658E+04,
      0.20825E+04, 0.23211E+04, 0.25833E+04, 0.28711E+04, 0.31867E+04,
      0.35323E+04, 0.39102E+04, 0.43230E+04, 0.47735E+04, 0.52645E+04,
      0.57991E+04, 0.63807E+04, 0.70127E+04, 0.76988E+04, 0.84430E+04,
      0.92495E+04, 0.10123E+05, 0.11067E+05, 0.12088E+05, 0.13191E+05,
      0.14381E+05, 0.15664E+05, 0.17047E+05, 0.18536E+05, 0.20137E+05,
      0.21859E+05, 0.23710E+05, 0.25696E+05, 0.27827E+05, 0.30112E+05,
      0.32561E+05, 0.35183E+05, 0.37990E+05, 0.40991E+05, 0.44199E+05,
      0.47626E+05, 0.51285E+05, 0.55189E+05, 0.59353E+05, 0.63791E+05,
      0.68518E+05, 0.73551E+05, 0.78908E+05, 0.84604E+05, 0.90661E+05,
      0.97095E+05, 0.10393E+06, 0.11118E+06, 0.11888E+06, 0.12704E+06,
      0.13569E+06, 0.14486E+06, 0.15457E+06, 0.16485E+06, 0.17572E+06,
      0.18722E+06, 0.19938E+06, 0.21223E+06, 0.22581E+06, 0.24014E+06,
      0.25527E+06, 0.27123E+06, 0.28807E+06, 0.30582E+06, 0.32452E+06,
      0.34423E+06, 0.36498E+06, 0.38683E+06, 0.40982E+06, 0.43401E+06,
      0.45944E+06, 0.48618E+06, 0.51428E+06, 0.54380E+06, 0.57480E+06,
      0.60735E+06, 0.64151E+06, 0.67735E+06, 0.71494E+06, 0.75436E+06,
      0.79568E+06, 0.83898E+06, 0.88434E+06, 0.93184E+06, 0.98158E+06,
      0.10336E+07, 0.10881E+07, 0.11451E+07, 0.12047E+07, 0.12670E+07,
      0.13321E+07, 0.14002E+07, 0.14713E+07, 0.15455E+07, 0.16231E+07,
      0.17040E+07])


#  --------------- C2H2 1231: M = 26, I = 2 ---------------------
M = 26
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(8.)
TIPS_ISO_HASH[(M,I)] = float32([0.28647E+03, 0.40486E+03, 0.52369E+03,
      0.64419E+03, 0.76874E+03, 0.90040E+03, 0.10425E+04, 0.11984E+04,
      0.13713E+04, 0.15642E+04, 0.17802E+04, 0.20223E+04, 0.22939E+04,
      0.25981E+04, 0.29384E+04, 0.33185E+04, 0.37424E+04, 0.42142E+04,
      0.47386E+04, 0.53203E+04, 0.59646E+04, 0.66769E+04, 0.74633E+04,
      0.83302E+04, 0.92845E+04, 0.10333E+05, 0.11485E+05, 0.12747E+05,
      0.14129E+05, 0.15641E+05, 0.17292E+05, 0.19094E+05, 0.21058E+05,
      0.23197E+05, 0.25523E+05, 0.28051E+05, 0.30796E+05, 0.33773E+05,
      0.36999E+05, 0.40492E+05, 0.44270E+05, 0.48354E+05, 0.52765E+05,
      0.57525E+05, 0.62658E+05, 0.68189E+05, 0.74144E+05, 0.80551E+05,
      0.87439E+05, 0.94840E+05, 0.10279E+06, 0.11131E+06, 0.12045E+06,
      0.13025E+06, 0.14074E+06, 0.15196E+06, 0.16397E+06, 0.17680E+06,
      0.19051E+06, 0.20514E+06, 0.22076E+06, 0.23742E+06, 0.25517E+06,
      0.27408E+06, 0.29421E+06, 0.31564E+06, 0.33842E+06, 0.36265E+06,
      0.38839E+06, 0.41572E+06, 0.44474E+06, 0.47553E+06, 0.50818E+06,
      0.54278E+06, 0.57945E+06, 0.61829E+06, 0.65940E+06, 0.70289E+06,
      0.74890E+06, 0.79754E+06, 0.84894E+06, 0.90324E+06, 0.96057E+06,
      0.10211E+07, 0.10849E+07, 0.11523E+07, 0.12233E+07, 0.12981E+07,
      0.13769E+07, 0.14599E+07, 0.15473E+07, 0.16393E+07, 0.17361E+07,
      0.18378E+07, 0.19447E+07, 0.20571E+07, 0.21752E+07, 0.22992E+07,
      0.24294E+07, 0.25661E+07, 0.27094E+07, 0.28598E+07, 0.30175E+07,
      0.31828E+07, 0.33560E+07, 0.35374E+07, 0.37274E+07, 0.39264E+07,
      0.41346E+07, 0.43525E+07, 0.45805E+07, 0.48188E+07, 0.50681E+07,
      0.53286E+07, 0.56008E+07, 0.58852E+07, 0.61823E+07, 0.64924E+07,
      0.68162E+07])


#  --------------- C2H2 1222: M = 26, I = 3 ---------------------
M = 26
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.24843E+03, 0.35373E+03, 0.45997E+03,
      0.56930E+03, 0.68497E+03, 0.81065E+03, 0.94999E+03, 0.11065E+04,
      0.12837E+04, 0.14848E+04, 0.17135E+04, 0.19731E+04, 0.22675E+04,
      0.26205E+04, 0.29999E+04, 0.34276E+04, 0.39086E+04, 0.44486E+04,
      0.50533E+04, 0.57294E+04, 0.64837E+04, 0.73237E+04, 0.82576E+04,
      0.92941E+04, 0.10443E+05, 0.11714E+05, 0.13117E+05, 0.14666E+05,
      0.16373E+05, 0.18250E+05, 0.20313E+05, 0.22578E+05, 0.25060E+05,
      0.27777E+05, 0.30750E+05, 0.33997E+05, 0.37541E+05, 0.41405E+05,
      0.45614E+05, 0.50192E+05, 0.55170E+05, 0.60576E+05, 0.66441E+05,
      0.72799E+05, 0.79686E+05, 0.87140E+05, 0.95199E+05, 0.10391E+06,
      0.11331E+06, 0.12345E+06, 0.13438E+06, 0.14615E+06, 0.15882E+06,
      0.17245E+06, 0.18710E+06, 0.20283E+06, 0.21972E+06, 0.23783E+06,
      0.25724E+06, 0.27804E+06, 0.30030E+06, 0.32411E+06, 0.34958E+06,
      0.37679E+06, 0.40585E+06, 0.43686E+06, 0.46994E+06, 0.50521E+06,
      0.54280E+06, 0.58282E+06, 0.62542E+06, 0.67074E+06, 0.71892E+06,
      0.77013E+06, 0.82453E+06, 0.88228E+06, 0.94356E+06, 0.10086E+07,
      0.10775E+07, 0.11505E+07, 0.12279E+07, 0.13098E+07, 0.13964E+07,
      0.14881E+07, 0.15850E+07, 0.16875E+07, 0.17957E+07, 0.19100E+07,
      0.20307E+07, 0.21580E+07, 0.22923E+07, 0.24339E+07, 0.25831E+07,
      0.27404E+07, 0.29060E+07, 0.30803E+07, 0.32638E+07, 0.34568E+07,
      0.36598E+07, 0.38733E+07, 0.40976E+07, 0.43332E+07, 0.45807E+07,
      0.48406E+07, 0.51133E+07, 0.53995E+07, 0.56997E+07, 0.60144E+07,
      0.63444E+07, 0.66901E+07, 0.70524E+07, 0.74317E+07, 0.78289E+07,
      0.82447E+07, 0.86797E+07, 0.91348E+07, 0.96108E+07, 0.10108E+08,
      0.10629E+08])


#  --------------- C2H6 1221: M = 27, I = 1 ---------------------
M = 27
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.47267E+04, 0.80011E+04, 0.11928E+05,
      0.16564E+05, 0.21985E+05, 0.28287E+05, 0.35590E+05, 0.44049E+05,
      0.53862E+05, 0.65277E+05, 0.78597E+05, 0.94191E+05, 0.11250E+06,
      0.13407E+06, 0.15952E+06, 0.18962E+06, 0.22526E+06, 0.26751E+06,
      0.31763E+06, 0.37714E+06, 0.44780E+06, 0.53174E+06, 0.63145E+06,
      0.74989E+06, 0.89056E+06, 0.10576E+07, 0.12559E+07, 0.14912E+07,
      0.17704E+07, 0.21013E+07, 0.24936E+07, 0.29582E+07, 0.35083E+07,
      0.41591E+07, 0.49286E+07, 0.58379E+07, 0.69116E+07, 0.81787E+07,
      0.96728E+07, 0.11433E+08, 0.13506E+08, 0.15945E+08, 0.18812E+08,
      0.22180E+08, 0.26134E+08, 0.30770E+08, 0.36204E+08, 0.42565E+08,
      0.50008E+08, 0.58708E+08, 0.68868E+08, 0.80725E+08, 0.94548E+08,
      0.11065E+09, 0.12940E+09, 0.15119E+09, 0.17652E+09, 0.20593E+09,
      0.24003E+09, 0.27956E+09, 0.32533E+09, 0.37829E+09, 0.43951E+09,
      0.51021E+09, 0.59180E+09, 0.68588E+09, 0.79427E+09, 0.91904E+09,
      0.10625E+10, 0.12275E+10, 0.14168E+10, 0.16341E+10, 0.18831E+10,
      0.21684E+10, 0.24949E+10, 0.28684E+10, 0.32951E+10, 0.37823E+10,
      0.43382E+10, 0.49719E+10, 0.56938E+10, 0.65156E+10, 0.74502E+10,
      0.85125E+10, 0.97190E+10, 0.11088E+11, 0.12641E+11, 0.14401E+11,
      0.16393E+11, 0.18648E+11, 0.21198E+11, 0.24079E+11, 0.27332E+11,
      0.31003E+11, 0.35142E+11, 0.39807E+11, 0.45060E+11, 0.50972E+11,
      0.57620E+11, 0.65091E+11, 0.73483E+11, 0.82902E+11, 0.93467E+11,
      0.10531E+12, 0.11858E+12, 0.13343E+12, 0.15005E+12, 0.16864E+12,
      0.18941E+12, 0.21260E+12, 0.23849E+12, 0.26737E+12, 0.29957E+12,
      0.33545E+12, 0.37541E+12, 0.41987E+12, 0.46934E+12, 0.52432E+12,
      0.58542E+12])


#  --------------- C2H6 1231: M = 27, I = 2 ---------------------
M = 27
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.24128E+04, 0.40845E+04, 0.60896E+04,
      0.84564E+04, 0.11224E+05, 0.14442E+05, 0.18170E+05, 0.22490E+05,
      0.27501E+05, 0.33329E+05, 0.40131E+05, 0.48094E+05, 0.57446E+05,
      0.68459E+05, 0.81458E+05, 0.96828E+05, 0.11503E+06, 0.13661E+06,
      0.16221E+06, 0.19260E+06, 0.22869E+06, 0.27156E+06, 0.32249E+06,
      0.38298E+06, 0.45483E+06, 0.54015E+06, 0.64144E+06, 0.76164E+06,
      0.90423E+06, 0.10733E+07, 0.12737E+07, 0.15110E+07, 0.17920E+07,
      0.21245E+07, 0.25176E+07, 0.29821E+07, 0.35307E+07, 0.41780E+07,
      0.49414E+07, 0.58408E+07, 0.68999E+07, 0.81461E+07, 0.96110E+07,
      0.11332E+08, 0.13352E+08, 0.15721E+08, 0.18497E+08, 0.21748E+08,
      0.25551E+08, 0.29997E+08, 0.35189E+08, 0.41248E+08, 0.48313E+08,
      0.56542E+08, 0.66122E+08, 0.77262E+08, 0.90206E+08, 0.10523E+09,
      0.12267E+09, 0.14287E+09, 0.16626E+09, 0.19333E+09, 0.22462E+09,
      0.26076E+09, 0.30247E+09, 0.35056E+09, 0.40596E+09, 0.46974E+09,
      0.54310E+09, 0.62740E+09, 0.72420E+09, 0.83527E+09, 0.96260E+09,
      0.11084E+10, 0.12754E+10, 0.14663E+10, 0.16845E+10, 0.19336E+10,
      0.22178E+10, 0.25418E+10, 0.29109E+10, 0.33311E+10, 0.38090E+10,
      0.43522E+10, 0.49691E+10, 0.56693E+10, 0.64633E+10, 0.73631E+10,
      0.83821E+10, 0.95352E+10, 0.10839E+11, 0.12312E+11, 0.13976E+11,
      0.15854E+11, 0.17971E+11, 0.20357E+11, 0.23043E+11, 0.26067E+11,
      0.29467E+11, 0.33289E+11, 0.37581E+11, 0.42399E+11, 0.47804E+11,
      0.53862E+11, 0.60649E+11, 0.68247E+11, 0.76750E+11, 0.86257E+11,
      0.96882E+11, 0.10875E+12, 0.12199E+12, 0.13677E+12, 0.15325E+12,
      0.17160E+12, 0.19204E+12, 0.21480E+12, 0.24010E+12, 0.26824E+12,
      0.29950E+12])


#  --------------- PH3 1111: M = 28, I = 1 ---------------------
M = 28
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.29652E+03, 0.49643E+03, 0.72810E+03,
      0.98777E+03, 0.12729E+04, 0.15820E+04, 0.19145E+04, 0.22708E+04,
      0.26520E+04, 0.30600E+04, 0.34971E+04, 0.39662E+04, 0.44702E+04,
      0.50126E+04, 0.55970E+04, 0.62273E+04, 0.69075E+04, 0.76421E+04,
      0.84357E+04, 0.92933E+04, 0.10220E+05, 0.11222E+05, 0.12304E+05,
      0.13473E+05, 0.14736E+05, 0.16099E+05, 0.17571E+05, 0.19160E+05,
      0.20873E+05, 0.22720E+05, 0.24710E+05, 0.26854E+05, 0.29162E+05,
      0.31646E+05, 0.34317E+05, 0.37188E+05, 0.40273E+05, 0.43585E+05,
      0.47140E+05, 0.50953E+05, 0.55040E+05, 0.59419E+05, 0.64108E+05,
      0.69127E+05, 0.74496E+05, 0.80236E+05, 0.86369E+05, 0.92918E+05,
      0.99909E+05, 0.10737E+06, 0.11532E+06, 0.12380E+06, 0.13282E+06,
      0.14244E+06, 0.15266E+06, 0.16354E+06, 0.17511E+06, 0.18739E+06,
      0.20044E+06, 0.21430E+06, 0.22900E+06, 0.24459E+06, 0.26111E+06,
      0.27862E+06, 0.29716E+06, 0.31680E+06, 0.33757E+06, 0.35954E+06,
      0.38277E+06, 0.40733E+06, 0.43326E+06, 0.46065E+06, 0.48955E+06,
      0.52005E+06, 0.55222E+06, 0.58614E+06, 0.62188E+06, 0.65953E+06,
      0.69917E+06, 0.74091E+06, 0.78483E+06, 0.83103E+06, 0.87960E+06,
      0.93067E+06, 0.98432E+06, 0.10407E+07, 0.10999E+07, 0.11620E+07,
      0.12272E+07, 0.12956E+07, 0.13673E+07, 0.14425E+07, 0.15212E+07,
      0.16038E+07, 0.16902E+07, 0.17808E+07, 0.18755E+07, 0.19746E+07,
      0.20784E+07, 0.21868E+07, 0.23002E+07, 0.24187E+07, 0.25425E+07,
      0.26719E+07, 0.28070E+07, 0.29480E+07, 0.30952E+07, 0.32488E+07,
      0.34091E+07, 0.35762E+07, 0.37504E+07, 0.39320E+07, 0.41213E+07,
      0.43185E+07, 0.45239E+07, 0.47378E+07, 0.49605E+07, 0.51923E+07,
      0.54335E+07])


#  --------------- COF2 269: M = 29, I = 1 ---------------------
M = 29
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.54999E+04, 0.92749E+04, 0.13668E+05,
      0.18643E+05, 0.24224E+05, 0.30487E+05, 0.37547E+05, 0.45543E+05,
      0.54639E+05, 0.65019E+05, 0.76886E+05, 0.90462E+05, 0.10600E+06,
      0.12377E+06, 0.14407E+06, 0.16723E+06, 0.19363E+06, 0.22367E+06,
      0.25780E+06, 0.29650E+06, 0.34031E+06, 0.38982E+06, 0.44568E+06,
      0.50859E+06, 0.57932E+06, 0.65872E+06, 0.74770E+06, 0.84724E+06,
      0.95844E+06, 0.10825E+07, 0.12205E+07, 0.13741E+07, 0.15446E+07,
      0.17336E+07, 0.19428E+07, 0.21742E+07, 0.24296E+07, 0.27113E+07,
      0.30214E+07, 0.33626E+07, 0.37373E+07, 0.41484E+07, 0.45989E+07,
      0.50921E+07, 0.56313E+07, 0.62202E+07, 0.68626E+07, 0.75628E+07,
      0.83251E+07, 0.91542E+07, 0.10055E+08, 0.11033E+08, 0.12093E+08,
      0.13242E+08, 0.14486E+08, 0.15831E+08, 0.17284E+08, 0.18853E+08,
      0.20546E+08, 0.22371E+08, 0.24335E+08, 0.26450E+08, 0.28724E+08,
      0.31167E+08, 0.33790E+08, 0.36605E+08, 0.39623E+08, 0.42856E+08,
      0.46318E+08, 0.50022E+08, 0.53983E+08, 0.58215E+08, 0.62735E+08,
      0.67558E+08, 0.72702E+08, 0.78186E+08, 0.84028E+08, 0.90247E+08,
      0.96865E+08, 0.10390E+09, 0.11138E+09, 0.11933E+09, 0.12777E+09,
      0.13672E+09, 0.14622E+09, 0.15629E+09, 0.16695E+09, 0.17825E+09,
      0.19021E+09, 0.20287E+09, 0.21625E+09, 0.23039E+09, 0.24534E+09,
      0.26113E+09, 0.27779E+09, 0.29538E+09, 0.31392E+09, 0.33348E+09,
      0.35409E+09, 0.37580E+09, 0.39867E+09, 0.42274E+09, 0.44806E+09,
      0.47470E+09, 0.50271E+09, 0.53215E+09, 0.56308E+09, 0.59557E+09,
      0.62968E+09, 0.66548E+09, 0.70304E+09, 0.74243E+09, 0.78374E+09,
      0.82703E+09, 0.87240E+09, 0.91992E+09, 0.96967E+09, 0.10218E+10,
      0.10763E+10])


#  --------------- COF2 369: M = 29, I = 2 --------------------- not in TIPS-2011
M = 29
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(0.)
TIPS_ISO_HASH[(M,I)] = float32([0.])


#  --------------- SF6 29: M = 30, I = 1 ---------------------
M = 30
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.46373E+05, 0.78844E+05, 0.11939E+06,
      0.17183E+06, 0.24247E+06, 0.34059E+06, 0.47963E+06, 0.67906E+06,
      0.96713E+06, 0.13848E+07, 0.19911E+07, 0.28714E+07, 0.41481E+07,
      0.59956E+07, 0.86617E+07, 0.12496E+08, 0.17991E+08, 0.25832E+08,
      0.36971E+08, 0.52724E+08, 0.74895E+08, 0.10595E+09, 0.14923E+09,
      0.20925E+09, 0.29208E+09, 0.40582E+09, 0.56124E+09, 0.77259E+09,
      0.10586E+10, 0.14439E+10, 0.19605E+10, 0.26500E+10, 0.35662E+10,
      0.47781E+10, 0.63747E+10, 0.84689E+10, 0.11205E+11, 0.14765E+11,
      0.19378E+11, 0.25336E+11, 0.32998E+11, 0.42819E+11, 0.55361E+11,
      0.71323E+11, 0.91569E+11, 0.11716E+12, 0.14941E+12, 0.18992E+12,
      0.24065E+12, 0.30398E+12, 0.38283E+12, 0.48069E+12, 0.60182E+12,
      0.75136E+12, 0.93546E+12, 0.11615E+13, 0.14384E+13, 0.17767E+13,
      0.21890E+13, 0.26903E+13, 0.32984E+13, 0.40344E+13, 0.49232E+13,
      0.59942E+13, 0.72819E+13, 0.88272E+13, 0.10678E+14, 0.12889E+14,
      0.15527E+14, 0.18666E+14, 0.22397E+14, 0.26823E+14, 0.32062E+14,
      0.38253E+14, 0.45558E+14, 0.54161E+14, 0.64277E+14, 0.76153E+14,
      0.90072E+14, 0.10636E+15, 0.12539E+15, 0.14759E+15, 0.17345E+15,
      0.20354E+15, 0.23848E+15, 0.27902E+15, 0.32597E+15, 0.38028E+15,
      0.44303E+15, 0.51542E+15, 0.59883E+15, 0.69482E+15, 0.80516E+15,
      0.93182E+15, 0.10770E+16, 0.12434E+16, 0.14336E+16, 0.16511E+16,
      0.18992E+16, 0.21821E+16, 0.25043E+16, 0.28709E+16, 0.32875E+16,
      0.37604E+16, 0.42968E+16, 0.49046E+16, 0.55925E+16, 0.63704E+16,
      0.72492E+16, 0.82411E+16, 0.93596E+16, 0.10620E+17, 0.12038E+17,
      0.13633E+17, 0.15425E+17, 0.17438E+17, 0.19694E+17, 0.22224E+17,
      0.25057E+17])


#  --------------- H2S 121: M = 31, I = 1 ---------------------
M = 31
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.47192E+02, 0.78671E+02, 0.11510E+03,
      0.15589E+03, 0.20061E+03, 0.24896E+03, 0.30070E+03, 0.35571E+03,
      0.41386E+03, 0.47513E+03, 0.53951E+03, 0.60703E+03, 0.67772E+03,
      0.75167E+03, 0.82896E+03, 0.90969E+03, 0.99396E+03, 0.10819E+04,
      0.11736E+04, 0.12692E+04, 0.13689E+04, 0.14727E+04, 0.15809E+04,
      0.16937E+04, 0.18111E+04, 0.19333E+04, 0.20606E+04, 0.21931E+04,
      0.23309E+04, 0.24744E+04, 0.26236E+04, 0.27788E+04, 0.29403E+04,
      0.31081E+04, 0.32825E+04, 0.34638E+04, 0.36522E+04, 0.38478E+04,
      0.40510E+04, 0.42619E+04, 0.44808E+04, 0.47080E+04, 0.49437E+04,
      0.51881E+04, 0.54415E+04, 0.57042E+04, 0.59764E+04, 0.62584E+04,
      0.65505E+04, 0.68529E+04, 0.71660E+04, 0.74899E+04, 0.78251E+04,
      0.81718E+04, 0.85303E+04, 0.89008E+04, 0.92838E+04, 0.96795E+04,
      0.10088E+05, 0.10510E+05, 0.10946E+05, 0.11396E+05, 0.11860E+05,
      0.12339E+05, 0.12833E+05, 0.13342E+05, 0.13867E+05, 0.14408E+05,
      0.14966E+05, 0.15540E+05, 0.16132E+05, 0.16741E+05, 0.17368E+05,
      0.18013E+05, 0.18677E+05, 0.19361E+05, 0.20064E+05, 0.20786E+05,
      0.21529E+05, 0.22293E+05, 0.23078E+05, 0.23885E+05, 0.24714E+05,
      0.25565E+05, 0.26439E+05, 0.27337E+05, 0.28258E+05, 0.29204E+05,
      0.30174E+05, 0.31170E+05, 0.32191E+05, 0.33239E+05, 0.34313E+05,
      0.35414E+05, 0.36543E+05, 0.37700E+05, 0.38886E+05, 0.40101E+05,
      0.41346E+05, 0.42621E+05, 0.43926E+05, 0.45263E+05, 0.46631E+05,
      0.48033E+05, 0.49466E+05, 0.50934E+05, 0.52435E+05, 0.53971E+05,
      0.55542E+05, 0.57149E+05, 0.58792E+05, 0.60472E+05, 0.62190E+05,
      0.63946E+05, 0.65740E+05, 0.67574E+05, 0.69448E+05, 0.71362E+05,
      0.73318E+05])


#  --------------- H2S 141: M = 31, I = 2 ---------------------
M = 31
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.47310E+02, 0.78869E+02, 0.11539E+03,
      0.15628E+03, 0.20112E+03, 0.24959E+03, 0.30147E+03, 0.35661E+03,
      0.41491E+03, 0.47634E+03, 0.54088E+03, 0.60857E+03, 0.67945E+03,
      0.75359E+03, 0.83107E+03, 0.91201E+03, 0.99649E+03, 0.10846E+04,
      0.11766E+04, 0.12724E+04, 0.13724E+04, 0.14765E+04, 0.15850E+04,
      0.16980E+04, 0.18157E+04, 0.19382E+04, 0.20658E+04, 0.21987E+04,
      0.23369E+04, 0.24807E+04, 0.26303E+04, 0.27859E+04, 0.29478E+04,
      0.31160E+04, 0.32909E+04, 0.34727E+04, 0.36615E+04, 0.38576E+04,
      0.40613E+04, 0.42728E+04, 0.44923E+04, 0.47200E+04, 0.49563E+04,
      0.52013E+04, 0.54554E+04, 0.57188E+04, 0.59917E+04, 0.62744E+04,
      0.65672E+04, 0.68704E+04, 0.71843E+04, 0.75090E+04, 0.78451E+04,
      0.81926E+04, 0.85520E+04, 0.89236E+04, 0.93075E+04, 0.97042E+04,
      0.10114E+05, 0.10537E+05, 0.10974E+05, 0.11425E+05, 0.11890E+05,
      0.12370E+05, 0.12866E+05, 0.13376E+05, 0.13903E+05, 0.14445E+05,
      0.15004E+05, 0.15580E+05, 0.16173E+05, 0.16784E+05, 0.17412E+05,
      0.18059E+05, 0.18725E+05, 0.19410E+05, 0.20115E+05, 0.20839E+05,
      0.21584E+05, 0.22350E+05, 0.23137E+05, 0.23946E+05, 0.24777E+05,
      0.25630E+05, 0.26507E+05, 0.27407E+05, 0.28330E+05, 0.29278E+05,
      0.30251E+05, 0.31249E+05, 0.32273E+05, 0.33324E+05, 0.34401E+05,
      0.35505E+05, 0.36637E+05, 0.37797E+05, 0.38985E+05, 0.40204E+05,
      0.41451E+05, 0.42729E+05, 0.44038E+05, 0.45379E+05, 0.46751E+05,
      0.48155E+05, 0.49593E+05, 0.51064E+05, 0.52569E+05, 0.54109E+05,
      0.55684E+05, 0.57295E+05, 0.58943E+05, 0.60627E+05, 0.62349E+05,
      0.64109E+05, 0.65908E+05, 0.67747E+05, 0.69625E+05, 0.71544E+05,
      0.73505E+05])


#  --------------- H2S 131: M = 30, I = 3 ---------------------
M = 31
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(4.)
TIPS_ISO_HASH[(M,I)] = float32([0.18901E+03, 0.31509E+03, 0.46102E+03,
      0.62437E+03, 0.80349E+03, 0.99713E+03, 0.12044E+04, 0.14247E+04,
      0.16576E+04, 0.19030E+04, 0.21609E+04, 0.24313E+04, 0.27145E+04,
      0.30106E+04, 0.33202E+04, 0.36436E+04, 0.39811E+04, 0.43332E+04,
      0.47005E+04, 0.50835E+04, 0.54827E+04, 0.58987E+04, 0.63321E+04,
      0.67836E+04, 0.72538E+04, 0.77434E+04, 0.82532E+04, 0.87838E+04,
      0.93360E+04, 0.99106E+04, 0.10508E+05, 0.11130E+05, 0.11777E+05,
      0.12449E+05, 0.13147E+05, 0.13874E+05, 0.14628E+05, 0.15412E+05,
      0.16225E+05, 0.17070E+05, 0.17947E+05, 0.18857E+05, 0.19801E+05,
      0.20780E+05, 0.21795E+05, 0.22847E+05, 0.23937E+05, 0.25067E+05,
      0.26236E+05, 0.27448E+05, 0.28702E+05, 0.29999E+05, 0.31342E+05,
      0.32730E+05, 0.34166E+05, 0.35650E+05, 0.37184E+05, 0.38769E+05,
      0.40406E+05, 0.42097E+05, 0.43842E+05, 0.45644E+05, 0.47503E+05,
      0.49421E+05, 0.51399E+05, 0.53439E+05, 0.55542E+05, 0.57709E+05,
      0.59942E+05, 0.62242E+05, 0.64611E+05, 0.67051E+05, 0.69563E+05,
      0.72148E+05, 0.74808E+05, 0.77545E+05, 0.80360E+05, 0.83255E+05,
      0.86232E+05, 0.89291E+05, 0.92435E+05, 0.95667E+05, 0.98986E+05,
      0.10240E+06, 0.10590E+06, 0.10949E+06, 0.11318E+06, 0.11697E+06,
      0.12086E+06, 0.12484E+06, 0.12893E+06, 0.13313E+06, 0.13743E+06,
      0.14184E+06, 0.14637E+06, 0.15100E+06, 0.15575E+06, 0.16062E+06,
      0.16560E+06, 0.17071E+06, 0.17594E+06, 0.18129E+06, 0.18677E+06,
      0.19238E+06, 0.19813E+06, 0.20400E+06, 0.21002E+06, 0.21617E+06,
      0.22246E+06, 0.22890E+06, 0.23548E+06, 0.24221E+06, 0.24909E+06,
      0.25612E+06, 0.26331E+06, 0.27065E+06, 0.27816E+06, 0.28583E+06,
      0.29366E+06])


#  --------------- HCOOH 126: M = 32, I = 1 ---------------------
M = 32
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(4.)
TIPS_ISO_HASH[(M,I)] = float32([0.31899E+04, 0.53773E+04, 0.79205E+04,
      0.10792E+05, 0.13993E+05, 0.17550E+05, 0.21509E+05, 0.25930E+05,
      0.30885E+05, 0.36460E+05, 0.42750E+05, 0.49864E+05, 0.57926E+05,
      0.67071E+05, 0.77453E+05, 0.89243E+05, 0.10263E+06, 0.11783E+06,
      0.13507E+06, 0.15462E+06, 0.17676E+06, 0.20183E+06, 0.23018E+06,
      0.26221E+06, 0.29836E+06, 0.33911E+06, 0.38501E+06, 0.43664E+06,
      0.49467E+06, 0.55981E+06, 0.63286E+06, 0.71470E+06, 0.80628E+06,
      0.90865E+06, 0.10230E+07, 0.11505E+07, 0.12927E+07, 0.14509E+07,
      0.16269E+07, 0.18225E+07, 0.20396E+07, 0.22804E+07, 0.25472E+07,
      0.28425E+07, 0.31692E+07, 0.35301E+07, 0.39285E+07, 0.43681E+07,
      0.48525E+07, 0.53858E+07, 0.59727E+07, 0.66178E+07, 0.73265E+07,
      0.81042E+07, 0.89571E+07, 0.98918E+07, 0.10915E+08, 0.12035E+08,
      0.13259E+08, 0.14597E+08, 0.16057E+08, 0.17650E+08, 0.19387E+08,
      0.21279E+08, 0.23339E+08, 0.25579E+08, 0.28016E+08, 0.30663E+08,
      0.33536E+08, 0.36655E+08, 0.40037E+08, 0.43701E+08, 0.47671E+08,
      0.51967E+08, 0.56614E+08, 0.61639E+08, 0.67068E+08, 0.72930E+08,
      0.79257E+08, 0.86082E+08, 0.93439E+08, 0.10137E+09, 0.10990E+09,
      0.11909E+09, 0.12898E+09, 0.13960E+09, 0.15102E+09, 0.16329E+09,
      0.17646E+09, 0.19059E+09, 0.20575E+09, 0.22200E+09, 0.23941E+09,
      0.25806E+09, 0.27802E+09, 0.29938E+09, 0.32223E+09, 0.34666E+09,
      0.37276E+09, 0.40064E+09, 0.43041E+09, 0.46218E+09, 0.49607E+09,
      0.53221E+09, 0.57074E+09, 0.61179E+09, 0.65551E+09, 0.70206E+09,
      0.75159E+09, 0.80430E+09, 0.86034E+09, 0.91992E+09, 0.98324E+09,
      0.10505E+10, 0.11219E+10, 0.11977E+10, 0.12782E+10, 0.13635E+10,
      0.14540E+10])


#  --------------- HO2 166: M = 33, I = 1 ---------------------
M = 33
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.39277E+03, 0.66062E+03, 0.97123E+03,
      0.13194E+04, 0.17014E+04, 0.21148E+04, 0.25578E+04, 0.30296E+04,
      0.35297E+04, 0.40585E+04, 0.46167E+04, 0.52055E+04, 0.58264E+04,
      0.64809E+04, 0.71707E+04, 0.78978E+04, 0.86641E+04, 0.94715E+04,
      0.10322E+05, 0.11218E+05, 0.12161E+05, 0.13154E+05, 0.14198E+05,
      0.15296E+05, 0.16449E+05, 0.17661E+05, 0.18933E+05, 0.20267E+05,
      0.21666E+05, 0.23133E+05, 0.24669E+05, 0.26277E+05, 0.27960E+05,
      0.29720E+05, 0.31560E+05, 0.33482E+05, 0.35489E+05, 0.37584E+05,
      0.39769E+05, 0.42048E+05, 0.44423E+05, 0.46898E+05, 0.49475E+05,
      0.52157E+05, 0.54948E+05, 0.57850E+05, 0.60868E+05, 0.64003E+05,
      0.67261E+05, 0.70643E+05, 0.74154E+05, 0.77797E+05, 0.81575E+05,
      0.85492E+05, 0.89553E+05, 0.93760E+05, 0.98118E+05, 0.10263E+06,
      0.10730E+06, 0.11213E+06, 0.11713E+06, 0.12230E+06, 0.12765E+06,
      0.13317E+06, 0.13888E+06, 0.14478E+06, 0.15086E+06, 0.15715E+06,
      0.16363E+06, 0.17032E+06, 0.17723E+06, 0.18434E+06, 0.19168E+06,
      0.19924E+06, 0.20704E+06, 0.21506E+06, 0.22333E+06, 0.23185E+06,
      0.24061E+06, 0.24963E+06, 0.25891E+06, 0.26846E+06, 0.27828E+06,
      0.28838E+06, 0.29876E+06, 0.30943E+06, 0.32039E+06, 0.33166E+06,
      0.34323E+06, 0.35512E+06, 0.36732E+06, 0.37985E+06, 0.39271E+06,
      0.40590E+06, 0.41944E+06, 0.43333E+06, 0.44758E+06, 0.46219E+06,
      0.47717E+06, 0.49252E+06, 0.50826E+06, 0.52439E+06, 0.54091E+06,
      0.55784E+06, 0.57518E+06, 0.59293E+06, 0.61112E+06, 0.62973E+06,
      0.64878E+06, 0.66828E+06, 0.68824E+06, 0.70866E+06, 0.72955E+06,
      0.75091E+06, 0.77276E+06, 0.79511E+06, 0.81795E+06, 0.84131E+06,
      0.86518E+06])


#  --------------- O 6: M = 34, I = 1 --------------------- not in TIPS-2011
M = 34
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(0.)
TIPS_ISO_HASH[(M,I)] = float32([0.])


#  --------------- ClONO2 5646: M = 35, I = 1 ---------------------
M = 35
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(12.)
TIPS_ISO_HASH[(M,I)] = float32([0.11444E+06, 0.21121E+06, 0.34858E+06,
      0.53934E+06, 0.80041E+06, 0.11539E+07, 0.16286E+07, 0.22614E+07,
      0.30992E+07, 0.42015E+07, 0.56426E+07, 0.75152E+07, 0.99344E+07,
      0.13042E+08, 0.17012E+08, 0.22058E+08, 0.28437E+08, 0.36463E+08,
      0.46514E+08, 0.59042E+08, 0.74589E+08, 0.93801E+08, 0.11744E+09,
      0.14643E+09, 0.18181E+09, 0.22486E+09, 0.27705E+09, 0.34009E+09,
      0.41598E+09, 0.50705E+09, 0.61599E+09, 0.74590E+09, 0.90037E+09,
      0.10835E+10, 0.13001E+10, 0.15554E+10, 0.18556E+10, 0.22079E+10,
      0.26200E+10, 0.31012E+10, 0.36615E+10, 0.43126E+10, 0.50675E+10,
      0.59409E+10, 0.69492E+10, 0.81110E+10, 0.94469E+10, 0.10980E+11,
      0.12736E+11, 0.14745E+11, 0.17037E+11, 0.19649E+11, 0.22620E+11,
      0.25994E+11, 0.29819E+11, 0.34150E+11, 0.39044E+11, 0.44568E+11,
      0.50794E+11, 0.57799E+11, 0.65672E+11, 0.74506E+11, 0.84408E+11,
      0.95490E+11, 0.10788E+12, 0.12171E+12, 0.13713E+12, 0.15431E+12,
      0.17342E+12, 0.19465E+12, 0.21822E+12, 0.24435E+12, 0.27329E+12,
      0.30530E+12, 0.34069E+12, 0.37976E+12, 0.42286E+12, 0.47034E+12,
      0.52262E+12, 0.58012E+12, 0.64330E+12, 0.71267E+12, 0.78875E+12,
      0.87214E+12, 0.96344E+12, 0.10633E+13, 0.11725E+13, 0.12918E+13,
      0.14220E+13, 0.15640E+13, 0.17188E+13, 0.18873E+13, 0.20706E+13,
      0.22700E+13, 0.24866E+13, 0.27218E+13, 0.29771E+13, 0.32538E+13,
      0.35537E+13, 0.38784E+13, 0.42299E+13, 0.46100E+13, 0.50208E+13,
      0.54645E+13, 0.59435E+13, 0.64603E+13, 0.70175E+13, 0.76180E+13,
      0.82647E+13, 0.89608E+13, 0.97097E+13, 0.10515E+14, 0.11380E+14,
      0.12310E+14, 0.13307E+14, 0.14378E+14, 0.15526E+14, 0.16756E+14,
      0.18075E+14])


#  --------------- ClONO2 7646: M = 35, I = 2 ---------------------
M = 35
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(12.)
TIPS_ISO_HASH[(M,I)] = float32([0.11735E+06, 0.21659E+06, 0.35745E+06,
      0.55307E+06, 0.82078E+06, 0.11833E+07, 0.16700E+07, 0.23189E+07,
      0.31781E+07, 0.43084E+07, 0.57862E+07, 0.77065E+07, 0.10187E+08,
      0.13374E+08, 0.17445E+08, 0.22619E+08, 0.29161E+08, 0.37391E+08,
      0.47698E+08, 0.60545E+08, 0.76487E+08, 0.96188E+08, 0.12043E+09,
      0.15015E+09, 0.18644E+09, 0.23059E+09, 0.28410E+09, 0.34874E+09,
      0.42657E+09, 0.51995E+09, 0.63167E+09, 0.76489E+09, 0.92329E+09,
      0.11111E+10, 0.13331E+10, 0.15950E+10, 0.19029E+10, 0.22641E+10,
      0.26867E+10, 0.31801E+10, 0.37547E+10, 0.44224E+10, 0.51965E+10,
      0.60921E+10, 0.71261E+10, 0.83174E+10, 0.96873E+10, 0.11260E+11,
      0.13061E+11, 0.15120E+11, 0.17471E+11, 0.20149E+11, 0.23196E+11,
      0.26656E+11, 0.30578E+11, 0.35019E+11, 0.40038E+11, 0.45703E+11,
      0.52087E+11, 0.59270E+11, 0.67343E+11, 0.76403E+11, 0.86556E+11,
      0.97921E+11, 0.11062E+12, 0.12481E+12, 0.14062E+12, 0.15824E+12,
      0.17783E+12, 0.19961E+12, 0.22377E+12, 0.25057E+12, 0.28024E+12,
      0.31308E+12, 0.34936E+12, 0.38943E+12, 0.43362E+12, 0.48232E+12,
      0.53593E+12, 0.59489E+12, 0.65968E+12, 0.73081E+12, 0.80883E+12,
      0.89434E+12, 0.98797E+12, 0.10904E+13, 0.12024E+13, 0.13247E+13,
      0.14582E+13, 0.16038E+13, 0.17625E+13, 0.19353E+13, 0.21233E+13,
      0.23278E+13, 0.25499E+13, 0.27911E+13, 0.30528E+13, 0.33366E+13,
      0.36442E+13, 0.39772E+13, 0.43376E+13, 0.47273E+13, 0.51486E+13,
      0.56036E+13, 0.60948E+13, 0.66248E+13, 0.71962E+13, 0.78119E+13,
      0.84751E+13, 0.91889E+13, 0.99569E+13, 0.10783E+14, 0.11670E+14,
      0.12623E+14, 0.13646E+14, 0.14744E+14, 0.15921E+14, 0.17183E+14,
      0.18535E+14])



#  --------------- NOp 46: M = 36, I = 1 ---------------------
M = 36
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(3.)
TIPS_ISO_HASH[(M,I)] = float32([0.63956E+02, 0.90185E+02, 0.11642E+03,
      0.14265E+03, 0.16889E+03, 0.19513E+03, 0.22138E+03, 0.24763E+03,
      0.27388E+03, 0.30013E+03, 0.32639E+03, 0.35266E+03, 0.37894E+03,
      0.40523E+03, 0.43155E+03, 0.45790E+03, 0.48429E+03, 0.51074E+03,
      0.53725E+03, 0.56383E+03, 0.59052E+03, 0.61731E+03, 0.64422E+03,
      0.67127E+03, 0.69846E+03, 0.72582E+03, 0.75335E+03, 0.78108E+03,
      0.80901E+03, 0.83715E+03, 0.86552E+03, 0.89413E+03, 0.92298E+03,
      0.95208E+03, 0.98144E+03, 0.10111E+04, 0.10410E+04, 0.10712E+04,
      0.11017E+04, 0.11325E+04, 0.11636E+04, 0.11950E+04, 0.12268E+04,
      0.12588E+04, 0.12912E+04, 0.13239E+04, 0.13570E+04, 0.13903E+04,
      0.14241E+04, 0.14581E+04, 0.14926E+04, 0.15273E+04, 0.15624E+04,
      0.15979E+04, 0.16337E+04, 0.16699E+04, 0.17065E+04, 0.17434E+04,
      0.17806E+04, 0.18183E+04, 0.18563E+04, 0.18947E+04, 0.19334E+04,
      0.19725E+04, 0.20120E+04, 0.20519E+04, 0.20921E+04, 0.21327E+04,
      0.21737E+04, 0.22151E+04, 0.22568E+04, 0.22990E+04, 0.23415E+04,
      0.23844E+04, 0.24276E+04, 0.24713E+04, 0.25153E+04, 0.25598E+04,
      0.26046E+04, 0.26497E+04, 0.26953E+04, 0.27413E+04, 0.27876E+04,
      0.28343E+04, 0.28815E+04, 0.29290E+04, 0.29769E+04, 0.30251E+04,
      0.30738E+04, 0.31229E+04, 0.31723E+04, 0.32222E+04, 0.32724E+04,
      0.33230E+04, 0.33740E+04, 0.34254E+04, 0.34772E+04, 0.35294E+04,
      0.35819E+04, 0.36349E+04, 0.36883E+04, 0.37420E+04, 0.37961E+04,
      0.38507E+04, 0.39056E+04, 0.39609E+04, 0.40166E+04, 0.40727E+04,
      0.41292E+04, 0.41861E+04, 0.42434E+04, 0.43010E+04, 0.43591E+04,
      0.44176E+04, 0.44764E+04, 0.45357E+04, 0.45953E+04, 0.46554E+04,
      0.47158E+04])


#  --------------- HOBr 169: M = 37, I = 1 ---------------------
M = 37
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(8.)
TIPS_ISO_HASH[(M,I)] = float32([0.24445E+04, 0.41206E+04, 0.60683E+04,
      0.82610E+04, 0.10689E+05, 0.13352E+05, 0.16261E+05, 0.19427E+05,
      0.22867E+05, 0.26600E+05, 0.30643E+05, 0.35018E+05, 0.39745E+05,
      0.44844E+05, 0.50338E+05, 0.56249E+05, 0.62599E+05, 0.69410E+05,
      0.76706E+05, 0.84509E+05, 0.92845E+05, 0.10174E+06, 0.11121E+06,
      0.12128E+06, 0.13199E+06, 0.14335E+06, 0.15540E+06, 0.16815E+06,
      0.18165E+06, 0.19591E+06, 0.21096E+06, 0.22684E+06, 0.24358E+06,
      0.26120E+06, 0.27974E+06, 0.29922E+06, 0.31969E+06, 0.34118E+06,
      0.36372E+06, 0.38735E+06, 0.41210E+06, 0.43800E+06, 0.46511E+06,
      0.49345E+06, 0.52307E+06, 0.55400E+06, 0.58628E+06, 0.61997E+06,
      0.65509E+06, 0.69170E+06, 0.72984E+06, 0.76954E+06, 0.81087E+06,
      0.85386E+06, 0.89856E+06, 0.94502E+06, 0.99329E+06, 0.10434E+07,
      0.10955E+07, 0.11495E+07, 0.12055E+07, 0.12636E+07, 0.13238E+07,
      0.13862E+07, 0.14508E+07, 0.15177E+07, 0.15870E+07, 0.16587E+07,
      0.17328E+07, 0.18095E+07, 0.18888E+07, 0.19707E+07, 0.20554E+07,
      0.21428E+07, 0.22331E+07, 0.23263E+07, 0.24225E+07, 0.25217E+07,
      0.26241E+07, 0.27296E+07, 0.28385E+07, 0.29506E+07, 0.30662E+07,
      0.31853E+07, 0.33079E+07, 0.34341E+07, 0.35641E+07, 0.36979E+07,
      0.38355E+07, 0.39771E+07, 0.41228E+07, 0.42725E+07, 0.44265E+07,
      0.45848E+07, 0.47474E+07, 0.49145E+07, 0.50862E+07, 0.52624E+07,
      0.54435E+07, 0.56293E+07, 0.58201E+07, 0.60159E+07, 0.62168E+07,
      0.64229E+07, 0.66343E+07, 0.68511E+07, 0.70734E+07, 0.73013E+07,
      0.75349E+07, 0.77742E+07, 0.80196E+07, 0.82709E+07, 0.85283E+07,
      0.87920E+07, 0.90620E+07, 0.93385E+07, 0.96215E+07, 0.99112E+07,
      0.10208E+08])


#  --------------- HOBr 161: M = 37, I = 2 ---------------------
M = 37
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(8.)
TIPS_ISO_HASH[(M,I)] = float32([0.24350E+04, 0.41047E+04, 0.60448E+04,
      0.82291E+04, 0.10648E+05, 0.13301E+05, 0.16200E+05, 0.19355E+05,
      0.22784E+05, 0.26504E+05, 0.30534E+05, 0.34895E+05, 0.39607E+05,
      0.44691E+05, 0.50169E+05, 0.56063E+05, 0.62394E+05, 0.69186E+05,
      0.76461E+05, 0.84243E+05, 0.92555E+05, 0.10142E+06, 0.11087E+06,
      0.12091E+06, 0.13159E+06, 0.14292E+06, 0.15494E+06, 0.16766E+06,
      0.18112E+06, 0.19534E+06, 0.21036E+06, 0.22620E+06, 0.24289E+06,
      0.26047E+06, 0.27896E+06, 0.29840E+06, 0.31882E+06, 0.34025E+06,
      0.36274E+06, 0.38630E+06, 0.41099E+06, 0.43683E+06, 0.46387E+06,
      0.49215E+06, 0.52169E+06, 0.55255E+06, 0.58475E+06, 0.61836E+06,
      0.65340E+06, 0.68992E+06, 0.72796E+06, 0.76757E+06, 0.80880E+06,
      0.85169E+06, 0.89628E+06, 0.94263E+06, 0.99079E+06, 0.10408E+07,
      0.10927E+07, 0.11466E+07, 0.12025E+07, 0.12605E+07, 0.13205E+07,
      0.13828E+07, 0.14472E+07, 0.15140E+07, 0.15831E+07, 0.16546E+07,
      0.17286E+07, 0.18051E+07, 0.18842E+07, 0.19660E+07, 0.20504E+07,
      0.21377E+07, 0.22277E+07, 0.23207E+07, 0.24167E+07, 0.25157E+07,
      0.26178E+07, 0.27231E+07, 0.28317E+07, 0.29436E+07, 0.30589E+07,
      0.31777E+07, 0.33001E+07, 0.34260E+07, 0.35557E+07, 0.36892E+07,
      0.38265E+07, 0.39678E+07, 0.41131E+07, 0.42626E+07, 0.44162E+07,
      0.45741E+07, 0.47364E+07, 0.49031E+07, 0.50744E+07, 0.52503E+07,
      0.54309E+07, 0.56164E+07, 0.58067E+07, 0.60021E+07, 0.62025E+07,
      0.64081E+07, 0.66191E+07, 0.68354E+07, 0.70572E+07, 0.72846E+07,
      0.75177E+07, 0.77565E+07, 0.80013E+07, 0.82521E+07, 0.85090E+07,
      0.87721E+07, 0.90415E+07, 0.93173E+07, 0.95997E+07, 0.98888E+07,
      0.10185E+08])


#  --------------- C2H4 221: M = 38, I = 1 ---------------------
M = 38
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.95843E+03, 0.16137E+04, 0.23744E+04,
      0.32285E+04, 0.41694E+04, 0.51963E+04, 0.63143E+04, 0.75337E+04,
      0.88702E+04, 0.10344E+05, 0.11978E+05, 0.13802E+05, 0.15846E+05,
      0.18145E+05, 0.20740E+05, 0.23675E+05, 0.27000E+05, 0.30770E+05,
      0.35048E+05, 0.39905E+05, 0.45420E+05, 0.51680E+05, 0.58786E+05,
      0.66850E+05, 0.75997E+05, 0.86369E+05, 0.98123E+05, 0.11144E+06,
      0.12651E+06, 0.14356E+06, 0.16284E+06, 0.18463E+06, 0.20923E+06,
      0.23699E+06, 0.26831E+06, 0.30360E+06, 0.34334E+06, 0.38808E+06,
      0.43840E+06, 0.49495E+06, 0.55847E+06, 0.62976E+06, 0.70973E+06,
      0.79935E+06, 0.89973E+06, 0.10121E+07, 0.11378E+07, 0.12782E+07,
      0.14351E+07, 0.16102E+07, 0.18055E+07, 0.20231E+07, 0.22656E+07,
      0.25354E+07, 0.28356E+07, 0.31692E+07, 0.35398E+07, 0.39511E+07,
      0.44074E+07, 0.49132E+07, 0.54736E+07, 0.60940E+07, 0.67803E+07,
      0.75392E+07, 0.83776E+07, 0.93035E+07, 0.10325E+08, 0.11452E+08,
      0.12694E+08, 0.14062E+08, 0.15567E+08, 0.17224E+08, 0.19045E+08,
      0.21046E+08, 0.23243E+08, 0.25655E+08, 0.28300E+08, 0.31200E+08,
      0.34377E+08, 0.37856E+08, 0.41662E+08, 0.45826E+08, 0.50378E+08,
      0.55351E+08, 0.60781E+08, 0.66707E+08, 0.73172E+08, 0.80219E+08,
      0.87899E+08, 0.96262E+08, 0.10537E+09, 0.11527E+09, 0.12604E+09,
      0.13775E+09, 0.15047E+09, 0.16428E+09, 0.17927E+09, 0.19553E+09,
      0.21316E+09, 0.23226E+09, 0.25296E+09, 0.27537E+09, 0.29963E+09,
      0.32587E+09, 0.35425E+09, 0.38492E+09, 0.41805E+09, 0.45383E+09,
      0.49246E+09, 0.53413E+09, 0.57908E+09, 0.62754E+09, 0.67977E+09,
      0.73602E+09, 0.79660E+09, 0.86179E+09, 0.93194E+09, 0.10074E+10,
      0.10885E+10])


#  --------------- C2H4 231: M = 38, I = 2 ---------------------
M = 38
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.39228E+04, 0.66051E+04, 0.97190E+04,
      0.13215E+05, 0.17066E+05, 0.21270E+05, 0.25846E+05, 0.30838E+05,
      0.36309E+05, 0.42341E+05, 0.49032E+05, 0.56496E+05, 0.64862E+05,
      0.74275E+05, 0.84897E+05, 0.96912E+05, 0.11052E+06, 0.12595E+06,
      0.14347E+06, 0.16335E+06, 0.18592E+06, 0.21155E+06, 0.24064E+06,
      0.27365E+06, 0.31109E+06, 0.35354E+06, 0.40166E+06, 0.45615E+06,
      0.51785E+06, 0.58765E+06, 0.66657E+06, 0.75575E+06, 0.85646E+06,
      0.97011E+06, 0.10983E+07, 0.12428E+07, 0.14055E+07, 0.15886E+07,
      0.17945E+07, 0.20260E+07, 0.22861E+07, 0.25779E+07, 0.29052E+07,
      0.32721E+07, 0.36830E+07, 0.41429E+07, 0.46573E+07, 0.52323E+07,
      0.58744E+07, 0.65912E+07, 0.73906E+07, 0.82816E+07, 0.92740E+07,
      0.10379E+08, 0.11607E+08, 0.12973E+08, 0.14490E+08, 0.16174E+08,
      0.18042E+08, 0.20112E+08, 0.22406E+08, 0.24945E+08, 0.27755E+08,
      0.30861E+08, 0.34293E+08, 0.38083E+08, 0.42266E+08, 0.46878E+08,
      0.51961E+08, 0.57560E+08, 0.63724E+08, 0.70504E+08, 0.77959E+08,
      0.86150E+08, 0.95145E+08, 0.10502E+09, 0.11585E+09, 0.12772E+09,
      0.14072E+09, 0.15496E+09, 0.17054E+09, 0.18759E+09, 0.20622E+09,
      0.22658E+09, 0.24880E+09, 0.27306E+09, 0.29952E+09, 0.32837E+09,
      0.35981E+09, 0.39404E+09, 0.43131E+09, 0.47186E+09, 0.51595E+09,
      0.56387E+09, 0.61594E+09, 0.67247E+09, 0.73382E+09, 0.80038E+09,
      0.87255E+09, 0.95076E+09, 0.10355E+10, 0.11272E+10, 0.12265E+10,
      0.13339E+10, 0.14501E+10, 0.15756E+10, 0.17113E+10, 0.18577E+10,
      0.20159E+10, 0.21865E+10, 0.23705E+10, 0.25688E+10, 0.27826E+10,
      0.30129E+10, 0.32608E+10, 0.35277E+10, 0.38149E+10, 0.41237E+10,
      0.44557E+10])


#  --------------- CH3OH 2161: M = 39, I = 1 --------------------- not in TIPS-2011
M = 39
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(0.)
TIPS_ISO_HASH[(M,I)] = float32([0.])


#  --------------- CH3Br 219: M = 40, I = 1 ---------------------
M = 40
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(4.)
TIPS_ISO_HASH[(M,I)] = float32([0.70299E+04, 0.11847E+05, 0.17442E+05,
      0.23741E+05, 0.30723E+05, 0.38408E+05, 0.46851E+05, 0.56138E+05,
      0.66375E+05, 0.77692E+05, 0.90239E+05, 0.10418E+06, 0.11972E+06,
      0.13704E+06, 0.15639E+06, 0.17801E+06, 0.20218E+06, 0.22920E+06,
      0.25940E+06, 0.29316E+06, 0.33087E+06, 0.37296E+06, 0.41992E+06,
      0.47229E+06, 0.53062E+06, 0.59557E+06, 0.66781E+06, 0.74812E+06,
      0.83731E+06, 0.93629E+06, 0.10461E+07, 0.11677E+07, 0.13023E+07,
      0.14513E+07, 0.16159E+07, 0.17978E+07, 0.19985E+07, 0.22199E+07,
      0.24638E+07, 0.27324E+07, 0.30280E+07, 0.33529E+07, 0.37099E+07,
      0.41019E+07, 0.45319E+07, 0.50034E+07, 0.55199E+07, 0.60853E+07,
      0.67039E+07, 0.73801E+07, 0.81189E+07, 0.89255E+07, 0.98056E+07,
      0.10765E+08, 0.11811E+08, 0.12949E+08, 0.14188E+08, 0.15535E+08,
      0.17000E+08, 0.18590E+08, 0.20317E+08, 0.22190E+08, 0.24220E+08,
      0.26421E+08, 0.28804E+08, 0.31383E+08, 0.34173E+08, 0.37189E+08,
      0.40448E+08, 0.43967E+08, 0.47765E+08, 0.51862E+08, 0.56280E+08,
      0.61040E+08, 0.66167E+08, 0.71686E+08, 0.77624E+08, 0.84009E+08,
      0.90873E+08, 0.98247E+08, 0.10616E+09, 0.11466E+09, 0.12378E+09,
      0.13356E+09, 0.14403E+09, 0.15526E+09, 0.16728E+09, 0.18014E+09,
      0.19391E+09, 0.20863E+09, 0.22436E+09, 0.24117E+09, 0.25913E+09,
      0.27830E+09, 0.29875E+09, 0.32057E+09, 0.34384E+09, 0.36864E+09,
      0.39506E+09, 0.42320E+09, 0.45316E+09, 0.48504E+09, 0.51896E+09,
      0.55502E+09, 0.59336E+09, 0.63410E+09, 0.67738E+09, 0.72334E+09,
      0.77212E+09, 0.82388E+09, 0.87879E+09, 0.93701E+09, 0.99873E+09,
      0.10641E+10, 0.11334E+10, 0.12068E+10, 0.12845E+10, 0.13667E+10,
      0.14536E+10])


#  --------------- CH3Br 211: M = 40, I = 2 ---------------------
M = 40
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(4.)
TIPS_ISO_HASH[(M,I)] = float32([0.70566E+04, 0.11892E+05, 0.17508E+05,
      0.23832E+05, 0.30841E+05, 0.38557E+05, 0.47036E+05, 0.56362E+05,
      0.66644E+05, 0.78011E+05, 0.90615E+05, 0.10462E+06, 0.12023E+06,
      0.13763E+06, 0.15707E+06, 0.17880E+06, 0.20308E+06, 0.23023E+06,
      0.26059E+06, 0.29451E+06, 0.33240E+06, 0.37471E+06, 0.42191E+06,
      0.47453E+06, 0.53316E+06, 0.59843E+06, 0.67104E+06, 0.75176E+06,
      0.84141E+06, 0.94090E+06, 0.10512E+07, 0.11735E+07, 0.13088E+07,
      0.14585E+07, 0.16241E+07, 0.18069E+07, 0.20086E+07, 0.22312E+07,
      0.24764E+07, 0.27464E+07, 0.30435E+07, 0.33702E+07, 0.37291E+07,
      0.41231E+07, 0.45554E+07, 0.50294E+07, 0.55486E+07, 0.61171E+07,
      0.67389E+07, 0.74188E+07, 0.81616E+07, 0.89725E+07, 0.98573E+07,
      0.10822E+08, 0.11873E+08, 0.13018E+08, 0.14263E+08, 0.15618E+08,
      0.17090E+08, 0.18689E+08, 0.20425E+08, 0.22308E+08, 0.24350E+08,
      0.26563E+08, 0.28959E+08, 0.31552E+08, 0.34357E+08, 0.37389E+08,
      0.40666E+08, 0.44204E+08, 0.48023E+08, 0.52143E+08, 0.56585E+08,
      0.61371E+08, 0.66526E+08, 0.72076E+08, 0.78046E+08, 0.84467E+08,
      0.91369E+08, 0.98783E+08, 0.10674E+09, 0.11529E+09, 0.12446E+09,
      0.13429E+09, 0.14482E+09, 0.15611E+09, 0.16820E+09, 0.18113E+09,
      0.19497E+09, 0.20978E+09, 0.22560E+09, 0.24250E+09, 0.26056E+09,
      0.27983E+09, 0.30040E+09, 0.32234E+09, 0.34574E+09, 0.37068E+09,
      0.39725E+09, 0.42555E+09, 0.45567E+09, 0.48773E+09, 0.52184E+09,
      0.55811E+09, 0.59666E+09, 0.63763E+09, 0.68115E+09, 0.72736E+09,
      0.77642E+09, 0.82847E+09, 0.88368E+09, 0.94223E+09, 0.10043E+10,
      0.10701E+10, 0.11397E+10, 0.12135E+10, 0.12916E+10, 0.13743E+10,
      0.14618E+10])


#  --------------- CH3CN 2124: M = 41, I = 1 ---------------------
M = 41
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(3.)
TIPS_ISO_HASH[(M,I)] = float32([0.54361E+04, 0.91953E+04, 0.13708E+05,
      0.19097E+05, 0.25531E+05, 0.33206E+05, 0.42337E+05, 0.53173E+05,
      0.66002E+05, 0.81163E+05, 0.99053E+05, 0.12014E+06, 0.14496E+06,
      0.17414E+06, 0.20843E+06, 0.24866E+06, 0.29580E+06, 0.35099E+06,
      0.41551E+06, 0.49085E+06, 0.57871E+06, 0.68104E+06, 0.80008E+06,
      0.93836E+06, 0.10988E+07, 0.12848E+07, 0.14999E+07, 0.17487E+07,
      0.20359E+07, 0.23670E+07, 0.27484E+07, 0.31871E+07, 0.36912E+07,
      0.42697E+07, 0.49328E+07, 0.56921E+07, 0.65605E+07, 0.75526E+07,
      0.86847E+07, 0.99753E+07, 0.11445E+08, 0.13116E+08, 0.15016E+08,
      0.17172E+08, 0.19617E+08, 0.22386E+08, 0.25520E+08, 0.29063E+08,
      0.33064E+08, 0.37578E+08, 0.42667E+08, 0.48397E+08, 0.54844E+08,
      0.62090E+08, 0.70228E+08, 0.79358E+08, 0.89592E+08, 0.10105E+09,
      0.11388E+09, 0.12822E+09, 0.14424E+09, 0.16212E+09, 0.18205E+09,
      0.20427E+09, 0.22900E+09, 0.25652E+09, 0.28710E+09, 0.32107E+09,
      0.35877E+09, 0.40059E+09, 0.44692E+09, 0.49822E+09, 0.55500E+09,
      0.61777E+09, 0.68712E+09, 0.76370E+09, 0.84819E+09, 0.94135E+09,
      0.10440E+10, 0.11570E+10, 0.12814E+10, 0.14181E+10, 0.15684E+10,
      0.17334E+10, 0.19145E+10, 0.21131E+10, 0.23308E+10, 0.25693E+10,
      0.28304E+10, 0.31161E+10, 0.34285E+10, 0.37698E+10, 0.41426E+10,
      0.45496E+10, 0.49935E+10, 0.54776E+10, 0.60051E+10, 0.65796E+10,
      0.72049E+10, 0.78853E+10, 0.86251E+10, 0.94291E+10, 0.10303E+11,
      0.11251E+11, 0.12280E+11, 0.13396E+11, 0.14606E+11, 0.15916E+11,
      0.17336E+11, 0.18873E+11, 0.20536E+11, 0.22334E+11, 0.24278E+11,
      0.26379E+11, 0.28647E+11, 0.31096E+11, 0.33739E+11, 0.36589E+11,
      0.39661E+11])


#  --------------- CH3CN 2134: M = 41, I = 2 --------------------- not in HITRAN-2012
M = 41
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.10906E+05, 0.18458E+05, 0.27552E+05,
      0.38455E+05, 0.51523E+05, 0.67161E+05, 0.85818E+05, 0.10801E+06,
      0.13434E+06, 0.16550E+06, 0.20234E+06, 0.24581E+06, 0.29705E+06,
      0.35737E+06, 0.42831E+06, 0.51162E+06, 0.60936E+06, 0.72387E+06,
      0.85786E+06, 0.10145E+07, 0.11972E+07, 0.14102E+07, 0.16582E+07,
      0.19465E+07, 0.22813E+07, 0.26695E+07, 0.31190E+07, 0.36390E+07,
      0.42397E+07, 0.49328E+07, 0.57314E+07, 0.66507E+07, 0.77076E+07,
      0.89211E+07, 0.10313E+08, 0.11907E+08, 0.13732E+08, 0.15817E+08,
      0.18198E+08, 0.20914E+08, 0.24007E+08, 0.27527E+08, 0.31529E+08,
      0.36073E+08, 0.41228E+08, 0.47070E+08, 0.53683E+08, 0.61162E+08,
      0.69612E+08, 0.79149E+08, 0.89903E+08, 0.10202E+09, 0.11565E+09,
      0.13098E+09, 0.14820E+09, 0.16753E+09, 0.18921E+09, 0.21349E+09,
      0.24066E+09, 0.27106E+09, 0.30502E+09, 0.34293E+09, 0.38523E+09,
      0.43237E+09, 0.48486E+09, 0.54328E+09, 0.60823E+09, 0.68039E+09,
      0.76049E+09, 0.84935E+09, 0.94784E+09, 0.10569E+10, 0.11777E+10,
      0.13112E+10, 0.14588E+10, 0.16217E+10, 0.18016E+10, 0.19999E+10,
      0.22185E+10, 0.24592E+10, 0.27241E+10, 0.30155E+10, 0.33357E+10,
      0.36875E+10, 0.40736E+10, 0.44971E+10, 0.49615E+10, 0.54702E+10,
      0.60273E+10, 0.66369E+10, 0.73035E+10, 0.80322E+10, 0.88282E+10,
      0.96972E+10, 0.10645E+11, 0.11679E+11, 0.12806E+11, 0.14034E+11,
      0.15370E+11, 0.16824E+11, 0.18406E+11, 0.20125E+11, 0.21992E+11,
      0.24020E+11, 0.26221E+11, 0.28608E+11, 0.31197E+11, 0.34002E+11,
      0.37040E+11, 0.40330E+11, 0.43889E+11, 0.47739E+11, 0.51902E+11,
      0.56400E+11, 0.61259E+11, 0.66504E+11, 0.72165E+11, 0.78272E+11,
      0.84856E+11])


#  --------------- CH3CN 3124: M = 41, I = 3 --------------------- not in HITRAN-2012
M = 41
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.11223E+05, 0.18985E+05, 0.28307E+05,
      0.39441E+05, 0.52744E+05, 0.68620E+05, 0.87523E+05, 0.10997E+06,
      0.13658E+06, 0.16806E+06, 0.20524E+06, 0.24910E+06, 0.30080E+06,
      0.36165E+06, 0.43319E+06, 0.51722E+06, 0.61579E+06, 0.73127E+06,
      0.86640E+06, 0.10243E+07, 0.12086E+07, 0.14234E+07, 0.16735E+07,
      0.19642E+07, 0.23017E+07, 0.26931E+07, 0.31464E+07, 0.36706E+07,
      0.42762E+07, 0.49749E+07, 0.57801E+07, 0.67069E+07, 0.77722E+07,
      0.89955E+07, 0.10398E+08, 0.12006E+08, 0.13845E+08, 0.15947E+08,
      0.18346E+08, 0.21083E+08, 0.24201E+08, 0.27748E+08, 0.31781E+08,
      0.36361E+08, 0.41556E+08, 0.47442E+08, 0.54106E+08, 0.61643E+08,
      0.70157E+08, 0.79767E+08, 0.90604E+08, 0.10281E+09, 0.11655E+09,
      0.13199E+09, 0.14935E+09, 0.16882E+09, 0.19065E+09, 0.21512E+09,
      0.24250E+09, 0.27312E+09, 0.30733E+09, 0.34553E+09, 0.38814E+09,
      0.43562E+09, 0.48851E+09, 0.54736E+09, 0.61279E+09, 0.68548E+09,
      0.76617E+09, 0.85568E+09, 0.95489E+09, 0.10648E+10, 0.11864E+10,
      0.13209E+10, 0.14695E+10, 0.16337E+10, 0.18148E+10, 0.20146E+10,
      0.22348E+10, 0.24772E+10, 0.27441E+10, 0.30375E+10, 0.33601E+10,
      0.37143E+10, 0.41032E+10, 0.45298E+10, 0.49975E+10, 0.55099E+10,
      0.60709E+10, 0.66849E+10, 0.73563E+10, 0.80902E+10, 0.88918E+10,
      0.97670E+10, 0.10722E+11, 0.11763E+11, 0.12898E+11, 0.14134E+11,
      0.15480E+11, 0.16945E+11, 0.18537E+11, 0.20269E+11, 0.22149E+11,
      0.24191E+11, 0.26408E+11, 0.28812E+11, 0.31419E+11, 0.34244E+11,
      0.37303E+11, 0.40616E+11, 0.44201E+11, 0.48078E+11, 0.52269E+11,
      0.56799E+11, 0.61692E+11, 0.66974E+11, 0.72675E+11, 0.78824E+11,
      0.85454E+11])


#  --------------- CH3CN 3134: M = 41, I = 4 --------------------- not in HITRAN-2012
M = 41
I = 4
TIPS_GSI_HASH[(M,I)] = __FloatType__(12.)
TIPS_ISO_HASH[(M,I)] = float32([0.22522E+05, 0.38117E+05, 0.56899E+05,
      0.79412E+05, 0.10640E+06, 0.13870E+06, 0.17726E+06, 0.22314E+06,
      0.27761E+06, 0.34214E+06, 0.41847E+06, 0.50862E+06, 0.61497E+06,
      0.74028E+06, 0.88774E+06, 0.10611E+07, 0.12646E+07, 0.15031E+07,
      0.17825E+07, 0.21092E+07, 0.24908E+07, 0.29358E+07, 0.34541E+07,
      0.40571E+07, 0.47576E+07, 0.55703E+07, 0.65120E+07, 0.76018E+07,
      0.88614E+07, 0.10315E+08, 0.11992E+08, 0.13922E+08, 0.16142E+08,
      0.18693E+08, 0.21619E+08, 0.24973E+08, 0.28812E+08, 0.33202E+08,
      0.38216E+08, 0.43936E+08, 0.50455E+08, 0.57876E+08, 0.66315E+08,
      0.75901E+08, 0.86779E+08, 0.99110E+08, 0.11307E+09, 0.12887E+09,
      0.14672E+09, 0.16688E+09, 0.18961E+09, 0.21523E+09, 0.24407E+09,
      0.27651E+09, 0.31295E+09, 0.35387E+09, 0.39975E+09, 0.45118E+09,
      0.50875E+09, 0.57315E+09, 0.64512E+09, 0.72549E+09, 0.81517E+09,
      0.91514E+09, 0.10265E+10, 0.11504E+10, 0.12883E+10, 0.14414E+10,
      0.16115E+10, 0.18001E+10, 0.20093E+10, 0.22410E+10, 0.24975E+10,
      0.27812E+10, 0.30948E+10, 0.34412E+10, 0.38235E+10, 0.42452E+10,
      0.47101E+10, 0.52220E+10, 0.57856E+10, 0.64055E+10, 0.70869E+10,
      0.78355E+10, 0.86574E+10, 0.95591E+10, 0.10548E+11, 0.11631E+11,
      0.12817E+11, 0.14116E+11, 0.15536E+11, 0.17088E+11, 0.18785E+11,
      0.20636E+11, 0.22657E+11, 0.24861E+11, 0.27264E+11, 0.29881E+11,
      0.32730E+11, 0.35832E+11, 0.39205E+11, 0.42871E+11, 0.46855E+11,
      0.51182E+11, 0.55878E+11, 0.60973E+11, 0.66497E+11, 0.72484E+11,
      0.78970E+11, 0.85992E+11, 0.93592E+11, 0.10181E+12, 0.11070E+12,
      0.12031E+12, 0.13069E+12, 0.14189E+12, 0.15398E+12, 0.16703E+12,
      0.18110E+12])


#  --------------- CF4 29: M = 42, I = 1 ---------------------
M = 42
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.76233E+04, 0.12867E+05, 0.19059E+05,
      0.26316E+05, 0.34895E+05, 0.45145E+05, 0.57461E+05, 0.72259E+05,
      0.89950E+05, 0.11092E+06, 0.13550E+06, 0.16399E+06, 0.19658E+06,
      0.23341E+06, 0.27457E+06, 0.32004E+06, 0.36978E+06, 0.42369E+06,
      0.48161E+06, 0.54338E+06, 0.60880E+06, 0.67764E+06, 0.55684E+07,
      0.71250E+07, 0.90615E+07, 0.11458E+08, 0.14407E+08, 0.18021E+08,
      0.22428E+08, 0.27778E+08, 0.34247E+08, 0.42038E+08, 0.51386E+08,
      0.62559E+08, 0.75869E+08, 0.91670E+08, 0.11037E+09, 0.13242E+09,
      0.15836E+09, 0.18878E+09, 0.22436E+09, 0.26584E+09, 0.31410E+09,
      0.37008E+09, 0.43488E+09, 0.50970E+09, 0.59589E+09, 0.69496E+09,
      0.80858E+09, 0.93863E+09, 0.10872E+10, 0.12565E+10, 0.14491E+10,
      0.16679E+10, 0.19159E+10, 0.21966E+10, 0.25136E+10, 0.28711E+10,
      0.32740E+10, 0.37260E+10, 0.42340E+10, 0.48030E+10, 0.54400E+10,
      0.61520E+10, 0.69470E+10, 0.78320E+10, 0.88170E+10, 0.99120E+10,
      0.11130E+11, 0.12470E+11, 0.13970E+11, 0.15620E+11, 0.17440E+11,
      0.19450E+11, 0.21670E+11, 0.24100E+11, 0.26790E+11, 0.29730E+11,
      0.33000E+11, 0.36500E+11, 0.40400E+11, 0.44600E+11, 0.49300E+11,
      0.54300E+11, 0.59800E+11, 0.65800E+11, 0.72400E+11, 0.79500E+11,
      0.87200E+11, 0.95500E+11, 0.10500E+12, 0.11400E+12, 0.12500E+12,
      0.13600E+12, 0.14900E+12, 0.16200E+12, 0.17700E+12, 0.19200E+12,
      0.21000E+12, 0.23000E+12, 0.25000E+12, 0.27000E+12, 0.29000E+12,
      0.31000E+12, 0.34000E+12, 0.36000E+12, 0.39000E+12, 0.42000E+12,
      0.46000E+12, 0.49000E+12, 0.53000E+12, 0.57000E+12, 0.61000E+12,
      0.66000E+12, 0.70000E+12, 0.75000E+12, 0.81000E+12, 0.86000E+12,
      0.93000E+12])


#  --------------- C4H2 1221: M = 43, I = 1 ---------------------
M = 43
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.57628E+03, 0.84874E+03, 0.11789E+04,
      0.15952E+04, 0.21317E+04, 0.28324E+04, 0.37543E+04, 0.49705E+04,
      0.65754E+04, 0.86894E+04, 0.11466E+05, 0.15099E+05, 0.19834E+05,
      0.25980E+05, 0.33920E+05, 0.44132E+05, 0.57210E+05, 0.73884E+05,
      0.95049E+05, 0.12180E+06, 0.15548E+06, 0.19771E+06, 0.25045E+06,
      0.31606E+06, 0.39739E+06, 0.49786E+06, 0.62152E+06, 0.77324E+06,
      0.95878E+06, 0.11850E+07, 0.14599E+07, 0.17930E+07, 0.21956E+07,
      0.26807E+07, 0.32637E+07, 0.39626E+07, 0.47983E+07, 0.57951E+07,
      0.69813E+07, 0.83896E+07, 0.10058E+08, 0.12030E+08, 0.14356E+08,
      0.17093E+08, 0.20309E+08, 0.24079E+08, 0.28491E+08, 0.33644E+08,
      0.39651E+08, 0.46642E+08, 0.54764E+08, 0.64184E+08, 0.75091E+08,
      0.87699E+08, 0.10225E+09, 0.11902E+09, 0.13832E+09, 0.16049E+09,
      0.18593E+09, 0.21507E+09, 0.24841E+09, 0.28650E+09, 0.32996E+09,
      0.37949E+09, 0.43586E+09, 0.49993E+09, 0.57266E+09, 0.65513E+09,
      0.74852E+09, 0.85418E+09, 0.97356E+09, 0.11083E+10, 0.12602E+10,
      0.14313E+10, 0.16238E+10, 0.18401E+10, 0.20829E+10, 0.23553E+10,
      0.26605E+10, 0.30021E+10, 0.33841E+10, 0.38109E+10, 0.42874E+10,
      0.48187E+10, 0.54107E+10, 0.60698E+10, 0.68029E+10, 0.76176E+10,
      0.85223E+10, 0.95260E+10, 0.10639E+11, 0.11871E+11, 0.13236E+11,
      0.14744E+11, 0.16412E+11, 0.18253E+11, 0.20285E+11, 0.22526E+11,
      0.24995E+11, 0.27714E+11, 0.30705E+11, 0.33995E+11, 0.37609E+11,
      0.41579E+11, 0.45934E+11, 0.50711E+11, 0.55947E+11, 0.61681E+11,
      0.67957E+11, 0.74824E+11, 0.82330E+11, 0.90532E+11, 0.99487E+11,
      0.10926E+12, 0.11992E+12, 0.13154E+12, 0.14420E+12, 0.15799E+12,
      0.17299E+12])


#  --------------- HC3N 12224: M = 44, I = 1 --------------------- 1224 in HITRAN, 12224 in TIPS
M = 44
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.16683E+04, 0.24538E+04, 0.33995E+04,
      0.45769E+04, 0.60637E+04, 0.79533E+04, 0.10360E+05, 0.13422E+05,
      0.17311E+05, 0.22232E+05, 0.28434E+05, 0.36215E+05, 0.45932E+05,
      0.58011E+05, 0.72958E+05, 0.91370E+05, 0.11395E+06, 0.14153E+06,
      0.17507E+06, 0.21570E+06, 0.26475E+06, 0.32372E+06, 0.39440E+06,
      0.47881E+06, 0.57930E+06, 0.69856E+06, 0.83968E+06, 0.10062E+07,
      0.12021E+07, 0.14320E+07, 0.17011E+07, 0.20153E+07, 0.23812E+07,
      0.28065E+07, 0.32996E+07, 0.38701E+07, 0.45287E+07, 0.52876E+07,
      0.61602E+07, 0.71616E+07, 0.83088E+07, 0.96206E+07, 0.11118E+08,
      0.12824E+08, 0.14765E+08, 0.16969E+08, 0.19469E+08, 0.22299E+08,
      0.25498E+08, 0.29110E+08, 0.33181E+08, 0.37763E+08, 0.42914E+08,
      0.48697E+08, 0.55180E+08, 0.62440E+08, 0.70558E+08, 0.79627E+08,
      0.89743E+08, 0.10102E+09, 0.11356E+09, 0.12752E+09, 0.14301E+09,
      0.16020E+09, 0.17925E+09, 0.20035E+09, 0.22367E+09, 0.24945E+09,
      0.27790E+09, 0.30928E+09, 0.34385E+09, 0.38191E+09, 0.42376E+09,
      0.46975E+09, 0.52023E+09, 0.57562E+09, 0.63632E+09, 0.70279E+09,
      0.77553E+09, 0.85506E+09, 0.94195E+09, 0.10368E+10, 0.11403E+10,
      0.12531E+10, 0.13759E+10, 0.15097E+10, 0.16552E+10, 0.18133E+10,
      0.19851E+10, 0.21715E+10, 0.23738E+10, 0.25931E+10, 0.28307E+10,
      0.30879E+10, 0.33662E+10, 0.36672E+10, 0.39926E+10, 0.43439E+10,
      0.47233E+10, 0.51325E+10, 0.55738E+10, 0.60493E+10, 0.65615E+10,
      0.71129E+10, 0.77061E+10, 0.83441E+10, 0.90298E+10, 0.97664E+10,
      0.10557E+11, 0.11406E+11, 0.12317E+11, 0.13293E+11, 0.14339E+11,
      0.15459E+11, 0.16659E+11, 0.17942E+11, 0.19316E+11, 0.20784E+11,
      0.22353E+11])


#  --------------- HC3N 12234: M = 44, I = 2 --------------------- see above
M = 44
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(12.)
TIPS_ISO_HASH[(M,I)] = float32([0.33507E+04, 0.49290E+04, 0.68293E+04,
      0.91959E+04, 0.12185E+05, 0.15986E+05, 0.20828E+05, 0.26993E+05,
      0.34824E+05, 0.44739E+05, 0.57239E+05, 0.72931E+05, 0.92539E+05,
      0.11693E+06, 0.14713E+06, 0.18435E+06, 0.23004E+06, 0.28588E+06,
      0.35384E+06, 0.43625E+06, 0.53580E+06, 0.65562E+06, 0.79933E+06,
      0.97115E+06, 0.11759E+07, 0.14191E+07, 0.17073E+07, 0.20476E+07,
      0.24486E+07, 0.29196E+07, 0.34716E+07, 0.41169E+07, 0.48696E+07,
      0.57453E+07, 0.67621E+07, 0.79402E+07, 0.93022E+07, 0.10874E+08,
      0.12684E+08, 0.14764E+08, 0.17150E+08, 0.19884E+08, 0.23009E+08,
      0.26576E+08, 0.30641E+08, 0.35265E+08, 0.40518E+08, 0.46477E+08,
      0.53225E+08, 0.60856E+08, 0.69475E+08, 0.79195E+08, 0.90143E+08,
      0.10246E+09, 0.11629E+09, 0.13182E+09, 0.14921E+09, 0.16868E+09,
      0.19045E+09, 0.21477E+09, 0.24189E+09, 0.27211E+09, 0.30575E+09,
      0.34316E+09, 0.38471E+09, 0.43083E+09, 0.48196E+09, 0.53858E+09,
      0.60125E+09, 0.67052E+09, 0.74704E+09, 0.83148E+09, 0.92459E+09,
      0.10272E+10, 0.11401E+10, 0.12643E+10, 0.14007E+10, 0.15506E+10,
      0.17150E+10, 0.18953E+10, 0.20928E+10, 0.23090E+10, 0.25456E+10,
      0.28042E+10, 0.30867E+10, 0.33951E+10, 0.37316E+10, 0.40984E+10,
      0.44981E+10, 0.49332E+10, 0.54067E+10, 0.59216E+10, 0.64812E+10,
      0.70890E+10, 0.77488E+10, 0.84645E+10, 0.92405E+10, 0.10081E+11,
      0.10992E+11, 0.11978E+11, 0.13044E+11, 0.14197E+11, 0.15443E+11,
      0.16789E+11, 0.18243E+11, 0.19810E+11, 0.21501E+11, 0.23324E+11,
      0.25288E+11, 0.27403E+11, 0.29680E+11, 0.32130E+11, 0.34764E+11,
      0.37596E+11, 0.40639E+11, 0.43907E+11, 0.47416E+11, 0.51181E+11,
      0.55220E+11])


#  --------------- HC3N 12324: M = 44, I = 3 --------------------- see above
M = 44
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(12.)
TIPS_ISO_HASH[(M,I)] = float32([0.33506E+04, 0.49280E+04, 0.68267E+04,
      0.91901E+04, 0.12174E+05, 0.15966E+05, 0.20793E+05, 0.26936E+05,
      0.34734E+05, 0.44598E+05, 0.57026E+05, 0.72612E+05, 0.92071E+05,
      0.11625E+06, 0.14616E+06, 0.18298E+06, 0.22813E+06, 0.28323E+06,
      0.35022E+06, 0.43133E+06, 0.52918E+06, 0.64677E+06, 0.78761E+06,
      0.95571E+06, 0.11557E+07, 0.13929E+07, 0.16734E+07, 0.20041E+07,
      0.23929E+07, 0.28488E+07, 0.33820E+07, 0.40040E+07, 0.47280E+07,
      0.55686E+07, 0.65423E+07, 0.76678E+07, 0.89661E+07, 0.10460E+08,
      0.12177E+08, 0.14145E+08, 0.16397E+08, 0.18970E+08, 0.21903E+08,
      0.25242E+08, 0.29036E+08, 0.33339E+08, 0.38214E+08, 0.43726E+08,
      0.49949E+08, 0.56965E+08, 0.64864E+08, 0.73743E+08, 0.83711E+08,
      0.94886E+08, 0.10740E+09, 0.12139E+09, 0.13701E+09, 0.15443E+09,
      0.17384E+09, 0.19543E+09, 0.21943E+09, 0.24607E+09, 0.27561E+09,
      0.30832E+09, 0.34452E+09, 0.38453E+09, 0.42870E+09, 0.47742E+09,
      0.53110E+09, 0.59020E+09, 0.65518E+09, 0.72659E+09, 0.80496E+09,
      0.89092E+09, 0.98510E+09, 0.10882E+10, 0.12010E+10, 0.13242E+10,
      0.14588E+10, 0.16056E+10, 0.17657E+10, 0.19401E+10, 0.21299E+10,
      0.23363E+10, 0.25606E+10, 0.28043E+10, 0.30687E+10, 0.33553E+10,
      0.36660E+10, 0.40024E+10, 0.43665E+10, 0.47601E+10, 0.51856E+10,
      0.56450E+10, 0.61408E+10, 0.66756E+10, 0.72520E+10, 0.78729E+10,
      0.85413E+10, 0.92604E+10, 0.10034E+11, 0.10864E+11, 0.11757E+11,
      0.12714E+11, 0.13742E+11, 0.14843E+11, 0.16023E+11, 0.17287E+11,
      0.18640E+11, 0.20087E+11, 0.21634E+11, 0.23288E+11, 0.25054E+11,
      0.26939E+11, 0.28950E+11, 0.31096E+11, 0.33382E+11, 0.35819E+11,
      0.38413E+11])


#  --------------- HC3N 13224: M = 44, I = 4 --------------------- see above
M = 44
I = 4
TIPS_GSI_HASH[(M,I)] = __FloatType__(12.)
TIPS_ISO_HASH[(M,I)] = float32([0.34439E+04, 0.50672E+04, 0.70230E+04,
      0.94603E+04, 0.12542E+05, 0.16462E+05, 0.21461E+05, 0.27833E+05,
      0.35935E+05, 0.46204E+05, 0.59168E+05, 0.75463E+05, 0.95854E+05,
      0.12126E+06, 0.15276E+06, 0.19165E+06, 0.23947E+06, 0.29802E+06,
      0.36943E+06, 0.45619E+06, 0.56121E+06, 0.68789E+06, 0.84018E+06,
      0.10227E+07, 0.12407E+07, 0.15003E+07, 0.18086E+07, 0.21738E+07,
      0.26052E+07, 0.31134E+07, 0.37106E+07, 0.44109E+07, 0.52300E+07,
      0.61861E+07, 0.72996E+07, 0.85939E+07, 0.10095E+08, 0.11833E+08,
      0.13841E+08, 0.16158E+08, 0.18825E+08, 0.21890E+08, 0.25407E+08,
      0.29436E+08, 0.34045E+08, 0.39308E+08, 0.45309E+08, 0.52143E+08,
      0.59912E+08, 0.68734E+08, 0.78737E+08, 0.90065E+08, 0.10288E+09,
      0.11735E+09, 0.13367E+09, 0.15206E+09, 0.17277E+09, 0.19604E+09,
      0.22217E+09, 0.25148E+09, 0.28432E+09, 0.32108E+09, 0.36218E+09,
      0.40809E+09, 0.45932E+09, 0.51644E+09, 0.58004E+09, 0.65082E+09,
      0.72950E+09, 0.81690E+09, 0.91388E+09, 0.10214E+10, 0.11405E+10,
      0.12724E+10, 0.14182E+10, 0.15794E+10, 0.17573E+10, 0.19536E+10,
      0.21701E+10, 0.24086E+10, 0.26711E+10, 0.29599E+10, 0.32774E+10,
      0.36262E+10, 0.40090E+10, 0.44290E+10, 0.48895E+10, 0.53939E+10,
      0.59462E+10, 0.65504E+10, 0.72111E+10, 0.79332E+10, 0.87217E+10,
      0.95823E+10, 0.10521E+11, 0.11544E+11, 0.12659E+11, 0.13874E+11,
      0.15195E+11, 0.16632E+11, 0.18194E+11, 0.19892E+11, 0.21735E+11,
      0.23736E+11, 0.25907E+11, 0.28260E+11, 0.30810E+11, 0.33572E+11,
      0.36563E+11, 0.39799E+11, 0.43299E+11, 0.47083E+11, 0.51172E+11,
      0.55588E+11, 0.60355E+11, 0.65500E+11, 0.71049E+11, 0.77031E+11,
      0.83478E+11])


#  --------------- HC3N 12225: M = 44, I = 5 --------------------- see above
M = 44
I = 5
TIPS_GSI_HASH[(M,I)] = __FloatType__(4.)
TIPS_ISO_HASH[(M,I)] = float32([0.11455E+04, 0.16850E+04, 0.23345E+04,
      0.31432E+04, 0.41647E+04, 0.54630E+04, 0.71168E+04, 0.92219E+04,
      0.11895E+05, 0.15279E+05, 0.19545E+05, 0.24897E+05, 0.31584E+05,
      0.39899E+05, 0.50190E+05, 0.62871E+05, 0.78428E+05, 0.97434E+05,
      0.12056E+06, 0.14859E+06, 0.18243E+06, 0.22314E+06, 0.27194E+06,
      0.33026E+06, 0.39972E+06, 0.48219E+06, 0.57983E+06, 0.69509E+06,
      0.83077E+06, 0.99009E+06, 0.11767E+07, 0.13946E+07, 0.16487E+07,
      0.19441E+07, 0.22868E+07, 0.26836E+07, 0.31420E+07, 0.36704E+07,
      0.42786E+07, 0.49770E+07, 0.57776E+07, 0.66938E+07, 0.77404E+07,
      0.89339E+07, 0.10293E+08, 0.11837E+08, 0.13590E+08, 0.15576E+08,
      0.17823E+08, 0.20362E+08, 0.23227E+08, 0.26454E+08, 0.30085E+08,
      0.34166E+08, 0.38745E+08, 0.43877E+08, 0.49622E+08, 0.56046E+08,
      0.63219E+08, 0.71222E+08, 0.80138E+08, 0.90062E+08, 0.10110E+09,
      0.11335E+09, 0.12695E+09, 0.14202E+09, 0.15870E+09, 0.17716E+09,
      0.19756E+09, 0.22009E+09, 0.24493E+09, 0.27232E+09, 0.30247E+09,
      0.33565E+09, 0.37211E+09, 0.41217E+09, 0.45613E+09, 0.50433E+09,
      0.55714E+09, 0.61497E+09, 0.67823E+09, 0.74739E+09, 0.82293E+09,
      0.90540E+09, 0.99536E+09, 0.10934E+10, 0.12002E+10, 0.13165E+10,
      0.14430E+10, 0.15805E+10, 0.17299E+10, 0.18922E+10, 0.20682E+10,
      0.22591E+10, 0.24660E+10, 0.26901E+10, 0.29326E+10, 0.31951E+10,
      0.34788E+10, 0.37854E+10, 0.41166E+10, 0.44741E+10, 0.48598E+10,
      0.52758E+10, 0.57240E+10, 0.62069E+10, 0.67269E+10, 0.72864E+10,
      0.78882E+10, 0.85352E+10, 0.92305E+10, 0.99773E+10, 0.10779E+11,
      0.11639E+11, 0.12562E+11, 0.13552E+11, 0.14612E+11, 0.15748E+11,
      0.16964E+11])


#  --------------- HC3N 22224: M = 44, I = 6 --------------------- see above
M = 44
I = 6
TIPS_GSI_HASH[(M,I)] = __FloatType__(9.)
TIPS_ISO_HASH[(M,I)] = float32([0.27029E+04, 0.39999E+04, 0.55894E+04,
      0.76092E+04, 0.10219E+05, 0.13616E+05, 0.18042E+05, 0.23798E+05,
      0.31255E+05, 0.40867E+05, 0.53189E+05, 0.68897E+05, 0.88807E+05,
      0.11390E+06, 0.14537E+06, 0.18461E+06, 0.23330E+06, 0.29342E+06,
      0.36733E+06, 0.45779E+06, 0.56802E+06, 0.70182E+06, 0.86361E+06,
      0.10585E+07, 0.12925E+07, 0.15725E+07, 0.19064E+07, 0.23034E+07,
      0.27739E+07, 0.33302E+07, 0.39858E+07, 0.47566E+07, 0.56604E+07,
      0.67176E+07, 0.79511E+07, 0.93872E+07, 0.11055E+08, 0.12989E+08,
      0.15225E+08, 0.17806E+08, 0.20779E+08, 0.24197E+08, 0.28119E+08,
      0.32612E+08, 0.37749E+08, 0.43612E+08, 0.50294E+08, 0.57895E+08,
      0.66528E+08, 0.76318E+08, 0.87403E+08, 0.99937E+08, 0.11409E+09,
      0.13004E+09, 0.14800E+09, 0.16819E+09, 0.19086E+09, 0.21629E+09,
      0.24476E+09, 0.27661E+09, 0.31219E+09, 0.35189E+09, 0.39615E+09,
      0.44542E+09, 0.50021E+09, 0.56108E+09, 0.62862E+09, 0.70350E+09,
      0.78641E+09, 0.87814E+09, 0.97952E+09, 0.10915E+10, 0.12149E+10,
      0.13510E+10, 0.15008E+10, 0.16656E+10, 0.18468E+10, 0.20457E+10,
      0.22640E+10, 0.25032E+10, 0.27653E+10, 0.30522E+10, 0.33659E+10,
      0.37088E+10, 0.40832E+10, 0.44917E+10, 0.49371E+10, 0.54224E+10,
      0.59508E+10, 0.65256E+10, 0.71507E+10, 0.78298E+10, 0.85671E+10,
      0.93672E+10, 0.10235E+11, 0.11175E+11, 0.12193E+11, 0.13295E+11,
      0.14487E+11, 0.15776E+11, 0.17168E+11, 0.18671E+11, 0.20293E+11,
      0.22043E+11, 0.23929E+11, 0.25960E+11, 0.28148E+11, 0.30502E+11,
      0.33034E+11, 0.35756E+11, 0.38681E+11, 0.41823E+11, 0.45195E+11,
      0.48812E+11, 0.52692E+11, 0.56850E+11, 0.61306E+11, 0.66076E+11,
      0.71183E+11])


#  --------------- H2 11: M = 45, I = 1 --------------------- 
M = 45
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.15265E+01, 0.22243E+01, 0.29619E+01,
      0.36724E+01, 0.43456E+01, 0.49880E+01, 0.56090E+01, 0.62165E+01,
      0.68161E+01, 0.74113E+01, 0.80044E+01, 0.85966E+01, 0.91887E+01,
      0.97810E+01, 0.10374E+02, 0.10967E+02, 0.11561E+02, 0.12156E+02,
      0.12751E+02, 0.13347E+02, 0.13944E+02, 0.14541E+02, 0.15139E+02,
      0.15738E+02, 0.16337E+02, 0.16937E+02, 0.17538E+02, 0.18140E+02,
      0.18743E+02, 0.19346E+02, 0.19951E+02, 0.20556E+02, 0.21163E+02,
      0.21771E+02, 0.22379E+02, 0.22990E+02, 0.23601E+02, 0.24214E+02,
      0.24829E+02, 0.25445E+02, 0.26063E+02, 0.26683E+02, 0.27304E+02,
      0.27928E+02, 0.28553E+02, 0.29181E+02, 0.29811E+02, 0.30443E+02,
      0.31078E+02, 0.31715E+02, 0.32355E+02, 0.32997E+02, 0.33643E+02,
      0.34291E+02, 0.34942E+02, 0.35596E+02, 0.36253E+02, 0.36914E+02,
      0.37578E+02, 0.38245E+02, 0.38916E+02, 0.39590E+02, 0.40268E+02,
      0.40949E+02, 0.41635E+02, 0.42324E+02, 0.43017E+02, 0.43715E+02,
      0.44416E+02, 0.45122E+02, 0.45831E+02, 0.46546E+02, 0.47264E+02,
      0.47987E+02, 0.48714E+02, 0.49446E+02, 0.50183E+02, 0.50925E+02,
      0.51671E+02, 0.52422E+02, 0.53178E+02, 0.53939E+02, 0.54705E+02,
      0.55476E+02, 0.56252E+02, 0.57033E+02, 0.57820E+02, 0.58612E+02,
      0.59409E+02, 0.60212E+02, 0.61020E+02, 0.61833E+02, 0.62652E+02,
      0.63477E+02, 0.64308E+02, 0.65144E+02, 0.65986E+02, 0.66833E+02,
      0.67687E+02, 0.68546E+02, 0.69411E+02, 0.70283E+02, 0.71160E+02,
      0.72043E+02, 0.72933E+02, 0.73829E+02, 0.74730E+02, 0.75638E+02,
      0.76553E+02, 0.77473E+02, 0.78400E+02, 0.79333E+02, 0.80273E+02,
      0.81219E+02, 0.82172E+02, 0.83131E+02, 0.84097E+02, 0.85069E+02,
      0.86048E+02])


#  --------------- H2 12: M = 45, I = 2 --------------------- 
M = 45
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(6.)
TIPS_ISO_HASH[(M,I)] = float32([0.81692E+01, 0.10308E+02, 0.12557E+02,
      0.14848E+02, 0.17159E+02, 0.19482E+02, 0.21815E+02, 0.24153E+02,
      0.26497E+02, 0.28845E+02, 0.31197E+02, 0.33552E+02, 0.35910E+02,
      0.38272E+02, 0.40636E+02, 0.43002E+02, 0.45372E+02, 0.47744E+02,
      0.50119E+02, 0.52496E+02, 0.54877E+02, 0.57261E+02, 0.59649E+02,
      0.62040E+02, 0.64435E+02, 0.66835E+02, 0.69240E+02, 0.71650E+02,
      0.74066E+02, 0.76489E+02, 0.78918E+02, 0.81354E+02, 0.83799E+02,
      0.86252E+02, 0.88715E+02, 0.91187E+02, 0.93669E+02, 0.96163E+02,
      0.98668E+02, 0.10118E+03, 0.10371E+03, 0.10626E+03, 0.10881E+03,
      0.11138E+03, 0.11397E+03, 0.11657E+03, 0.11919E+03, 0.12182E+03,
      0.12447E+03, 0.12714E+03, 0.12982E+03, 0.13252E+03, 0.13524E+03,
      0.13798E+03, 0.14074E+03, 0.14352E+03, 0.14632E+03, 0.14914E+03,
      0.15198E+03, 0.15484E+03, 0.15772E+03, 0.16062E+03, 0.16355E+03,
      0.16649E+03, 0.16946E+03, 0.17246E+03, 0.17547E+03, 0.17851E+03,
      0.18157E+03, 0.18466E+03, 0.18777E+03, 0.19090E+03, 0.19406E+03,
      0.19725E+03, 0.20045E+03, 0.20369E+03, 0.20695E+03, 0.21023E+03,
      0.21354E+03, 0.21687E+03, 0.22024E+03, 0.22362E+03, 0.22704E+03,
      0.23048E+03, 0.23394E+03, 0.23744E+03, 0.24096E+03, 0.24451E+03,
      0.24808E+03, 0.25169E+03, 0.25532E+03, 0.25897E+03, 0.26266E+03,
      0.26638E+03, 0.27012E+03, 0.27389E+03, 0.27769E+03, 0.28152E+03,
      0.28537E+03, 0.28926E+03, 0.29317E+03, 0.29712E+03, 0.30109E+03,
      0.30509E+03, 0.30913E+03, 0.31319E+03, 0.31728E+03, 0.32140E+03,
      0.32555E+03, 0.32974E+03, 0.33395E+03, 0.33819E+03, 0.34246E+03,
      0.34677E+03, 0.35110E+03, 0.35547E+03, 0.35987E+03, 0.36429E+03,
      0.36875E+03])


#  --------------- CS 22: M = 46, I = 1 --------------------- 
M = 46
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.51416E+02, 0.72723E+02, 0.94044E+02,
      0.11538E+03, 0.13673E+03, 0.15810E+03, 0.17949E+03, 0.20093E+03,
      0.22245E+03, 0.24407E+03, 0.26582E+03, 0.28776E+03, 0.30992E+03,
      0.33233E+03, 0.35504E+03, 0.37807E+03, 0.40147E+03, 0.42525E+03,
      0.44944E+03, 0.47406E+03, 0.49914E+03, 0.52468E+03, 0.55071E+03,
      0.57723E+03, 0.60427E+03, 0.63183E+03, 0.65991E+03, 0.68854E+03,
      0.71771E+03, 0.74743E+03, 0.77771E+03, 0.80855E+03, 0.83996E+03,
      0.87193E+03, 0.90449E+03, 0.93762E+03, 0.97134E+03, 0.10056E+04,
      0.10405E+04, 0.10760E+04, 0.11121E+04, 0.11487E+04, 0.11860E+04,
      0.12239E+04, 0.12623E+04, 0.13014E+04, 0.13410E+04, 0.13813E+04,
      0.14222E+04, 0.14637E+04, 0.15057E+04, 0.15484E+04, 0.15917E+04,
      0.16357E+04, 0.16802E+04, 0.17253E+04, 0.17711E+04, 0.18175E+04,
      0.18645E+04, 0.19121E+04, 0.19603E+04, 0.20091E+04, 0.20586E+04,
      0.21087E+04, 0.21594E+04, 0.22107E+04, 0.22626E+04, 0.23152E+04,
      0.23684E+04, 0.24222E+04, 0.24767E+04, 0.25317E+04, 0.25874E+04,
      0.26438E+04, 0.27007E+04, 0.27583E+04, 0.28165E+04, 0.28754E+04,
      0.29348E+04, 0.29949E+04, 0.30557E+04, 0.31170E+04, 0.31790E+04,
      0.32417E+04, 0.33049E+04, 0.33688E+04, 0.34334E+04, 0.34986E+04,
      0.35644E+04, 0.36308E+04, 0.36979E+04, 0.37656E+04, 0.38340E+04,
      0.39030E+04, 0.39727E+04, 0.40430E+04, 0.41139E+04, 0.41855E+04,
      0.42577E+04, 0.43306E+04, 0.44041E+04, 0.44782E+04, 0.45530E+04,
      0.46284E+04, 0.47045E+04, 0.47813E+04, 0.48587E+04, 0.49367E+04,
      0.50154E+04, 0.50947E+04, 0.51747E+04, 0.52553E+04, 0.53366E+04,
      0.54185E+04, 0.55011E+04, 0.55844E+04, 0.56683E+04, 0.57528E+04,
      0.58380E+04])


#  --------------- CS 24: M = 46, I = 2 --------------------- 
M = 46
I = 2
TIPS_GSI_HASH[(M,I)] = __FloatType__(1.)
TIPS_ISO_HASH[(M,I)] = float32([0.52247E+02, 0.73900E+02, 0.95568E+02,
      0.11725E+03, 0.13895E+03, 0.16066E+03, 0.18241E+03, 0.20420E+03,
      0.22607E+03, 0.24805E+03, 0.27018E+03, 0.29249E+03, 0.31503E+03,
      0.33784E+03, 0.36096E+03, 0.38442E+03, 0.40824E+03, 0.43247E+03,
      0.45712E+03, 0.48221E+03, 0.50778E+03, 0.53382E+03, 0.56037E+03,
      0.58743E+03, 0.61501E+03, 0.64312E+03, 0.67179E+03, 0.70100E+03,
      0.73077E+03, 0.76111E+03, 0.79202E+03, 0.82351E+03, 0.85559E+03,
      0.88824E+03, 0.92149E+03, 0.95533E+03, 0.98977E+03, 0.10248E+04,
      0.10605E+04, 0.10967E+04, 0.11336E+04, 0.11710E+04, 0.12091E+04,
      0.12478E+04, 0.12871E+04, 0.13270E+04, 0.13675E+04, 0.14087E+04,
      0.14505E+04, 0.14929E+04, 0.15359E+04, 0.15795E+04, 0.16238E+04,
      0.16687E+04, 0.17142E+04, 0.17604E+04, 0.18071E+04, 0.18546E+04,
      0.19026E+04, 0.19513E+04, 0.20006E+04, 0.20505E+04, 0.21011E+04,
      0.21523E+04, 0.22042E+04, 0.22566E+04, 0.23098E+04, 0.23635E+04,
      0.24179E+04, 0.24730E+04, 0.25286E+04, 0.25850E+04, 0.26419E+04,
      0.26995E+04, 0.27578E+04, 0.28167E+04, 0.28762E+04, 0.29364E+04,
      0.29972E+04, 0.30587E+04, 0.31208E+04, 0.31836E+04, 0.32470E+04,
      0.33111E+04, 0.33758E+04, 0.34412E+04, 0.35072E+04, 0.35739E+04,
      0.36412E+04, 0.37092E+04, 0.37778E+04, 0.38471E+04, 0.39171E+04,
      0.39877E+04, 0.40589E+04, 0.41309E+04, 0.42034E+04, 0.42767E+04,
      0.43505E+04, 0.44251E+04, 0.45003E+04, 0.45762E+04, 0.46527E+04,
      0.47299E+04, 0.48077E+04, 0.48863E+04, 0.49654E+04, 0.50453E+04,
      0.51258E+04, 0.52070E+04, 0.52888E+04, 0.53713E+04, 0.54545E+04,
      0.55383E+04, 0.56229E+04, 0.57080E+04, 0.57939E+04, 0.58804E+04,
      0.59676E+04])


#  --------------- CS 32: M = 46, I = 3 --------------------- 
M = 46
I = 3
TIPS_GSI_HASH[(M,I)] = __FloatType__(2.)
TIPS_ISO_HASH[(M,I)] = float32([0.10889E+03, 0.15403E+03, 0.19920E+03,
      0.24440E+03, 0.28964E+03, 0.33491E+03, 0.38026E+03, 0.42571E+03,
      0.47134E+03, 0.51722E+03, 0.56342E+03, 0.61005E+03, 0.65719E+03,
      0.70493E+03, 0.75334E+03, 0.80249E+03, 0.85245E+03, 0.90329E+03,
      0.95504E+03, 0.10078E+04, 0.10615E+04, 0.11163E+04, 0.11721E+04,
      0.12291E+04, 0.12872E+04, 0.13464E+04, 0.14068E+04, 0.14684E+04,
      0.15311E+04, 0.15951E+04, 0.16604E+04, 0.17268E+04, 0.17945E+04,
      0.18635E+04, 0.19337E+04, 0.20051E+04, 0.20779E+04, 0.21519E+04,
      0.22272E+04, 0.23038E+04, 0.23817E+04, 0.24609E+04, 0.25414E+04,
      0.26232E+04, 0.27064E+04, 0.27908E+04, 0.28765E+04, 0.29636E+04,
      0.30520E+04, 0.31417E+04, 0.32327E+04, 0.33251E+04, 0.34188E+04,
      0.35138E+04, 0.36102E+04, 0.37079E+04, 0.38070E+04, 0.39074E+04,
      0.40091E+04, 0.41122E+04, 0.42166E+04, 0.43224E+04, 0.44295E+04,
      0.45380E+04, 0.46478E+04, 0.47590E+04, 0.48715E+04, 0.49854E+04,
      0.51007E+04, 0.52173E+04, 0.53353E+04, 0.54547E+04, 0.55754E+04,
      0.56975E+04, 0.58210E+04, 0.59458E+04, 0.60720E+04, 0.61996E+04,
      0.63285E+04, 0.64589E+04, 0.65906E+04, 0.67236E+04, 0.68581E+04,
      0.69940E+04, 0.71312E+04, 0.72698E+04, 0.74098E+04, 0.75512E+04,
      0.76940E+04, 0.78381E+04, 0.79837E+04, 0.81307E+04, 0.82790E+04,
      0.84287E+04, 0.85799E+04, 0.87324E+04, 0.88864E+04, 0.90417E+04,
      0.91984E+04, 0.93566E+04, 0.95161E+04, 0.96771E+04, 0.98394E+04,
      0.10003E+05, 0.10168E+05, 0.10335E+05, 0.10503E+05, 0.10672E+05,
      0.10843E+05, 0.11015E+05, 0.11189E+05, 0.11364E+05, 0.11541E+05,
      0.11719E+05, 0.11898E+05, 0.12079E+05, 0.12261E+05, 0.12444E+05,
      0.12630E+05])


#  --------------- CS 23: M = 46, I = 4 --------------------- 
M = 46
I = 4
TIPS_GSI_HASH[(M,I)] = __FloatType__(4.)
TIPS_ISO_HASH[(M,I)] = float32([0.20737E+03, 0.29330E+03, 0.37930E+03,
      0.46535E+03, 0.55145E+03, 0.63764E+03, 0.72394E+03, 0.81043E+03,
      0.89722E+03, 0.98443E+03, 0.10722E+04, 0.11607E+04, 0.12501E+04,
      0.13406E+04, 0.14323E+04, 0.15253E+04, 0.16197E+04, 0.17158E+04,
      0.18135E+04, 0.19129E+04, 0.20142E+04, 0.21174E+04, 0.22226E+04,
      0.23298E+04, 0.24391E+04, 0.25504E+04, 0.26639E+04, 0.27796E+04,
      0.28976E+04, 0.30177E+04, 0.31401E+04, 0.32648E+04, 0.33918E+04,
      0.35211E+04, 0.36527E+04, 0.37867E+04, 0.39231E+04, 0.40618E+04,
      0.42029E+04, 0.43463E+04, 0.44922E+04, 0.46405E+04, 0.47912E+04,
      0.49443E+04, 0.50999E+04, 0.52579E+04, 0.54183E+04, 0.55812E+04,
      0.57465E+04, 0.59143E+04, 0.60846E+04, 0.62573E+04, 0.64325E+04,
      0.66102E+04, 0.67903E+04, 0.69729E+04, 0.71581E+04, 0.73457E+04,
      0.75358E+04, 0.77284E+04, 0.79235E+04, 0.81211E+04, 0.83212E+04,
      0.85239E+04, 0.87290E+04, 0.89367E+04, 0.91469E+04, 0.93596E+04,
      0.95748E+04, 0.97926E+04, 0.10013E+05, 0.10236E+05, 0.10461E+05,
      0.10689E+05, 0.10920E+05, 0.11153E+05, 0.11388E+05, 0.11626E+05,
      0.11867E+05, 0.12110E+05, 0.12356E+05, 0.12604E+05, 0.12855E+05,
      0.13109E+05, 0.13365E+05, 0.13623E+05, 0.13884E+05, 0.14148E+05,
      0.14415E+05, 0.14683E+05, 0.14955E+05, 0.15229E+05, 0.15506E+05,
      0.15785E+05, 0.16067E+05, 0.16351E+05, 0.16638E+05, 0.16928E+05,
      0.17220E+05, 0.17515E+05, 0.17813E+05, 0.18113E+05, 0.18416E+05,
      0.18721E+05, 0.19029E+05, 0.19340E+05, 0.19653E+05, 0.19969E+05,
      0.20287E+05, 0.20608E+05, 0.20932E+05, 0.21258E+05, 0.21587E+05,
      0.21919E+05, 0.22253E+05, 0.22590E+05, 0.22930E+05, 0.23272E+05,
      0.23617E+05])


#  --------------- SO3 26: M = 46, I = 1 --------------------- not in TIPS-2011
M = 47
I = 1
TIPS_GSI_HASH[(M,I)] = __FloatType__(0.)
TIPS_ISO_HASH[(M,I)] = float32([0.])


# NOT IN HITRAN, BUT PRESENT IN TIPS-2011
#                                                        ... extracted from iso_comparison
#
# id       M    I     COMMENT          TIPS_M    TIPS_I        iso_name                 abundance          mass       mol_name
#101    1001    1    not in HITRAN      45                     H                        \N                 \N              H
#                
#102    1002    1    not in HITRAN      45                     He                       \N                 \N              He
#                
#104    1018    1    not in HITRAN      45                     Ar                       \N                 \N              Ar
#                
#                    not in HITRAN      45         4224                                                                    C2N2
#                    not in HITRAN      45         5225                                                                    C2N2
#
#                    not in HITRAN      48         26                                                                      SO
#                    not in HITRAN      48         46                                                                      SO
#                    not in HITRAN      48         28                                                                      SO
#
#                    not in HITRAN      49         1221                                                                    C3H4
#
#                    not in HITRAN      50         2111                                                                    CH3
#
#                    not in HITRAN      51         222                                                                     CS2
#                    not in HITRAN      51         224                                                                     CS2
#                    not in HITRAN      51         223                                                                     CS2
#                    not in HITRAN      51         232                                                                     CS2

def BD_TIPS_2011_PYTHON(M,I,T):

    # out of temperature range
    if T<70. or T>3000.:
        raise Exception('TIPS: T must be between 70K and 3000K.')
    
    try:
        # get statistical weight for specified isotopologue
        gi = TIPS_GSI_HASH[(M,I)]
        # interpolate partition sum for specified isotopologue
        Qt = AtoB(T,Tdat,TIPS_ISO_HASH[(M,I)],TIPS_NPT)
    except KeyError:
        raise Exception('TIPS: no data for M,I = %d,%d.' % (M,I))
    
    return gi,Qt

#  --------------- /TIPS-2011 IMPLEMENTATION ----------------------

#  --------------- TIPS-2017 IMPLEMENTATION ----------------------

#  --------------- TIPS-2017 GRID TABLE --------------------------
TIPS_2017_ISOT = {}
TIPS_2017_ISOT[0] = float64([
    1.0, 20.0, 40.0, 60.0, 80.0, 100.0, 120.0, 140.0, 160.0, 180.0,
    200.0, 220.0, 240.0, 260.0, 280.0, 300.0, 320.0, 340.0, 360.0, 380.0,
    400.0, 420.0, 440.0, 460.0, 480.0, 500.0, 520.0, 540.0, 560.0, 580.0,
    600.0, 620.0, 640.0, 660.0, 680.0, 700.0, 720.0, 740.0, 760.0, 780.0,
    800.0, 820.0, 840.0, 860.0, 880.0, 900.0, 920.0, 940.0, 960.0, 980.0,
    1000.0, 1020.0, 1040.0, 1060.0, 1080.0, 1100.0, 1120.0, 1140.0, 1160.0, 1180.0,
    1200.0, 1220.0, 1240.0, 1260.0, 1280.0, 1300.0, 1320.0, 1340.0, 1360.0, 1380.0,
    1400.0, 1420.0, 1440.0, 1460.0, 1480.0, 1500.0, 1520.0, 1540.0, 1560.0, 1580.0,
    1600.0, 1620.0, 1640.0, 1660.0, 1680.0, 1700.0, 1720.0, 1740.0, 1760.0, 1780.0,
    1800.0, 1820.0, 1840.0, 1860.0, 1880.0, 1900.0, 1920.0, 1940.0, 1960.0, 1980.0,
    2000.0, 2020.0, 2040.0, 2060.0, 2080.0, 2100.0, 2120.0, 2140.0, 2160.0, 2180.0,
    2200.0, 2220.0, 2240.0, 2260.0, 2280.0, 2300.0, 2320.0, 2340.0, 2360.0, 2380.0,
    2400.0, 2420.0, 2440.0, 2460.0, 2480.0, 2500.0, 2520.0, 2540.0, 2560.0, 2580.0,
    2600.0, 2620.0, 2640.0, 2660.0, 2680.0, 2700.0, 2720.0, 2740.0, 2760.0, 2780.0,
    2800.0, 2820.0, 2840.0, 2860.0, 2880.0, 2900.0, 2920.0, 2940.0, 2960.0, 2980.0,
    3000.0, 3020.0, 3040.0, 3060.0, 3080.0, 3100.0, 3120.0, 3140.0, 3160.0, 3180.0,
    3200.0, 3220.0, 3240.0, 3260.0, 3280.0, 3300.0, 3320.0, 3340.0, 3360.0, 3380.0,
    3400.0, 3420.0, 3440.0, 3460.0, 3480.0, 3500.0, 3520.0, 3540.0, 3560.0, 3580.0,
    3600.0, 3620.0, 3640.0, 3660.0, 3680.0, 3700.0, 3720.0, 3740.0, 3760.0, 3780.0,
    3800.0, 3820.0, 3840.0, 3860.0, 3880.0, 3900.0, 3920.0, 3940.0, 3960.0, 3980.0,
    4000.0, 4020.0, 4040.0, 4060.0, 4080.0, 4100.0, 4120.0, 4140.0, 4160.0, 4180.0,
    4200.0, 4220.0, 4240.0, 4260.0, 4280.0, 4300.0, 4320.0, 4340.0, 4360.0, 4380.0,
    4400.0, 4420.0, 4440.0, 4460.0, 4480.0, 4500.0, 4520.0, 4540.0, 4560.0, 4580.0,
    4600.0, 4620.0, 4640.0, 4660.0, 4680.0, 4700.0, 4720.0, 4740.0, 4760.0, 4780.0,
    4800.0, 4820.0, 4840.0, 4860.0, 4880.0, 4900.0, 4920.0, 4940.0, 4960.0, 4980.0,
    5000.0,
])
TIPS_2017_ISOT[1] = float64([
    1.0, 20.0, 40.0, 60.0, 80.0, 100.0, 120.0, 140.0, 160.0, 180.0,
    200.0, 220.0, 240.0, 260.0, 280.0, 300.0, 320.0, 340.0, 360.0, 380.0,
    400.0, 420.0, 440.0, 460.0, 480.0, 500.0, 520.0, 540.0, 560.0, 580.0,
    600.0, 620.0, 640.0, 660.0, 680.0, 700.0, 720.0, 740.0, 760.0, 780.0,
    800.0, 820.0, 840.0, 860.0, 880.0, 900.0, 920.0, 940.0, 960.0, 980.0,
    1000.0, 1020.0, 1040.0, 1060.0, 1080.0, 1100.0, 1120.0, 1140.0, 1160.0, 1180.0,
    1200.0, 1220.0, 1240.0, 1260.0, 1280.0, 1300.0, 1320.0, 1340.0, 1360.0, 1380.0,
    1400.0, 1420.0, 1440.0, 1460.0, 1480.0, 1500.0, 1520.0, 1540.0, 1560.0, 1580.0,
    1600.0, 1620.0, 1640.0, 1660.0, 1680.0, 1700.0, 1720.0, 1740.0, 1760.0, 1780.0,
    1800.0, 1820.0, 1840.0, 1860.0, 1880.0, 1900.0, 1920.0, 1940.0, 1960.0, 1980.0,
    2000.0, 2020.0, 2040.0, 2060.0, 2080.0, 2100.0, 2120.0, 2140.0, 2160.0, 2180.0,
    2200.0, 2220.0, 2240.0, 2260.0, 2280.0, 2300.0, 2320.0, 2340.0, 2360.0, 2380.0,
    2400.0, 2420.0, 2440.0, 2460.0, 2480.0, 2500.0, 2520.0, 2540.0, 2560.0, 2580.0,
    2600.0, 2620.0, 2640.0, 2660.0, 2680.0, 2700.0, 2720.0, 2740.0, 2760.0, 2780.0,
    2800.0, 2820.0, 2840.0, 2860.0, 2880.0, 2900.0, 2920.0, 2940.0, 2960.0, 2980.0,
    3000.0, 3020.0, 3040.0, 3060.0, 3080.0, 3100.0, 3120.0, 3140.0, 3160.0, 3180.0,
    3200.0, 3220.0, 3240.0, 3260.0, 3280.0, 3300.0, 3320.0, 3340.0, 3360.0, 3380.0,
    3400.0, 3420.0, 3440.0, 3460.0, 3480.0, 3500.0, 3520.0, 3540.0, 3560.0, 3580.0,
    3600.0, 3620.0, 3640.0, 3660.0, 3680.0, 3700.0, 3720.0, 3740.0, 3760.0, 3780.0,
    3800.0, 3820.0, 3840.0, 3860.0, 3880.0, 3900.0, 3920.0, 3940.0, 3960.0, 3980.0,
    4000.0, 4020.0, 4040.0, 4060.0, 4080.0, 4100.0, 4120.0, 4140.0, 4160.0, 4180.0,
    4200.0, 4220.0, 4240.0, 4260.0, 4280.0, 4300.0, 4320.0, 4340.0, 4360.0, 4380.0,
    4400.0, 4420.0, 4440.0, 4460.0, 4480.0, 4500.0, 4520.0, 4540.0, 4560.0, 4580.0,
    4600.0, 4620.0, 4640.0, 4660.0, 4680.0, 4700.0, 4720.0, 4740.0, 4760.0, 4780.0,
    4800.0, 4820.0, 4840.0, 4860.0, 4880.0, 4900.0, 4920.0, 4940.0, 4960.0, 4980.0,
    5000.0, 5020.0, 5040.0, 5060.0, 5080.0, 5100.0, 5120.0, 5140.0, 5160.0, 5180.0,
    5200.0, 5220.0, 5240.0, 5260.0, 5280.0, 5300.0, 5320.0, 5340.0, 5360.0, 5380.0,
    5400.0, 5420.0, 5440.0, 5460.0, 5480.0, 5500.0, 5520.0, 5540.0, 5560.0, 5580.0,
    5600.0, 5620.0, 5640.0, 5660.0, 5680.0, 5700.0, 5720.0, 5740.0, 5760.0, 5780.0,
    5800.0, 5820.0, 5840.0, 5860.0, 5880.0, 5900.0, 5920.0, 5940.0, 5960.0, 5980.0,
    6000.0,
])
TIPS_2017_ISOT[2] = float64([
    1.0, 20.0, 40.0, 60.0, 80.0, 100.0, 120.0, 140.0, 160.0, 180.0,
    200.0, 220.0, 240.0, 260.0, 280.0, 300.0, 320.0, 340.0, 360.0, 380.0,
    400.0, 420.0, 440.0, 460.0, 480.0, 500.0, 520.0, 540.0, 560.0, 580.0,
    600.0, 620.0, 640.0, 660.0, 680.0, 700.0, 720.0, 740.0, 760.0, 780.0,
    800.0, 820.0, 840.0, 860.0, 880.0, 900.0, 920.0, 940.0, 960.0, 980.0,
    1000.0, 1020.0, 1040.0, 1060.0, 1080.0, 1100.0, 1120.0, 1140.0, 1160.0, 1180.0,
    1200.0, 1220.0, 1240.0, 1260.0, 1280.0, 1300.0, 1320.0, 1340.0, 1360.0, 1380.0,
    1400.0, 1420.0, 1440.0, 1460.0, 1480.0, 1500.0, 1520.0, 1540.0, 1560.0, 1580.0,
    1600.0, 1620.0, 1640.0, 1660.0, 1680.0, 1700.0, 1720.0, 1740.0, 1760.0, 1780.0,
    1800.0, 1820.0, 1840.0, 1860.0, 1880.0, 1900.0, 1920.0, 1940.0, 1960.0, 1980.0,
    2000.0, 2020.0, 2040.0, 2060.0, 2080.0, 2100.0, 2120.0, 2140.0, 2160.0, 2180.0,
    2200.0, 2220.0, 2240.0, 2260.0, 2280.0, 2300.0, 2320.0, 2340.0, 2360.0, 2380.0,
    2400.0, 2420.0, 2440.0, 2460.0, 2480.0, 2500.0, 2520.0, 2540.0, 2560.0, 2580.0,
    2600.0, 2620.0, 2640.0, 2660.0, 2680.0, 2700.0, 2720.0, 2740.0, 2760.0, 2780.0,
    2800.0, 2820.0, 2840.0, 2860.0, 2880.0, 2900.0, 2920.0, 2940.0, 2960.0, 2980.0,
    3000.0, 3020.0, 3040.0, 3060.0, 3080.0, 3100.0, 3120.0, 3140.0, 3160.0, 3180.0,
    3200.0, 3220.0, 3240.0, 3260.0, 3280.0, 3300.0, 3320.0, 3340.0, 3360.0, 3380.0,
    3400.0, 3420.0, 3440.0, 3460.0, 3480.0, 3500.0,
])
TIPS_2017_ISOT[3] = float64([
    1.0, 20.0, 40.0, 60.0, 80.0, 100.0, 120.0, 140.0, 160.0, 180.0,
    200.0, 220.0, 240.0, 260.0, 280.0, 300.0, 320.0, 340.0, 360.0, 380.0,
    400.0, 420.0, 440.0, 460.0, 480.0, 500.0, 520.0, 540.0, 560.0, 580.0,
    600.0, 620.0, 640.0, 660.0, 680.0, 700.0, 720.0, 740.0, 760.0, 780.0,
    800.0, 820.0, 840.0, 860.0, 880.0, 900.0, 920.0, 940.0, 960.0, 980.0,
    1000.0, 1020.0, 1040.0, 1060.0, 1080.0, 1100.0, 1120.0, 1140.0, 1160.0, 1180.0,
    1200.0, 1220.0, 1240.0, 1260.0, 1280.0, 1300.0, 1320.0, 1340.0, 1360.0, 1380.0,
    1400.0, 1420.0, 1440.0, 1460.0, 1480.0, 1500.0, 1520.0, 1540.0, 1560.0, 1580.0,
    1600.0, 1620.0, 1640.0, 1660.0, 1680.0, 1700.0, 1720.0, 1740.0, 1760.0, 1780.0,
    1800.0, 1820.0, 1840.0, 1860.0, 1880.0, 1900.0, 1920.0, 1940.0, 1960.0, 1980.0,
    2000.0, 2020.0, 2040.0, 2060.0, 2080.0, 2100.0, 2120.0, 2140.0, 2160.0, 2180.0,
    2200.0, 2220.0, 2240.0, 2260.0, 2280.0, 2300.0, 2320.0, 2340.0, 2360.0, 2380.0,
    2400.0, 2420.0, 2440.0, 2460.0, 2480.0, 2500.0, 2520.0, 2540.0, 2560.0, 2580.0,
    2600.0, 2620.0, 2640.0, 2660.0, 2680.0, 2700.0, 2720.0, 2740.0, 2760.0, 2780.0,
    2800.0, 2820.0, 2840.0, 2860.0, 2880.0, 2900.0, 2920.0, 2940.0, 2960.0, 2980.0,
    3000.0, 3020.0, 3040.0, 3060.0, 3080.0, 3100.0, 3120.0, 3140.0, 3160.0, 3180.0,
    3200.0, 3220.0, 3240.0, 3260.0, 3280.0, 3300.0, 3320.0, 3340.0, 3360.0, 3380.0,
    3400.0, 3420.0, 3440.0, 3460.0, 3480.0, 3500.0, 3520.0, 3540.0, 3560.0, 3580.0,
    3600.0, 3620.0, 3640.0, 3660.0, 3680.0, 3700.0, 3720.0, 3740.0, 3760.0, 3780.0,
    3800.0, 3820.0, 3840.0, 3860.0, 3880.0, 3900.0, 3920.0, 3940.0, 3960.0, 3980.0,
    4000.0, 4020.0, 4040.0, 4060.0, 4080.0, 4100.0, 4120.0, 4140.0, 4160.0, 4180.0,
    4200.0, 4220.0, 4240.0, 4260.0, 4280.0, 4300.0, 4320.0, 4340.0, 4360.0, 4380.0,
    4400.0, 4420.0, 4440.0, 4460.0, 4480.0, 4500.0, 4520.0, 4540.0, 4560.0, 4580.0,
    4600.0, 4620.0, 4640.0, 4660.0, 4680.0, 4700.0, 4720.0, 4740.0, 4760.0, 4780.0,
    4800.0, 4820.0, 4840.0, 4860.0, 4880.0, 4900.0, 4920.0, 4940.0, 4960.0, 4980.0,
    5000.0, 5020.0, 5040.0, 5060.0, 5080.0, 5100.0, 5120.0, 5140.0, 5160.0, 5180.0,
    5200.0, 5220.0, 5240.0, 5260.0, 5280.0, 5300.0, 5320.0, 5340.0, 5360.0, 5380.0,
    5400.0, 5420.0, 5440.0, 5460.0, 5480.0, 5500.0, 5520.0, 5540.0, 5560.0, 5580.0,
    5600.0, 5620.0, 5640.0, 5660.0, 5680.0, 5700.0, 5720.0, 5740.0, 5760.0, 5780.0,
    5800.0, 5820.0, 5840.0, 5860.0, 5880.0, 5900.0, 5920.0, 5940.0, 5960.0, 5980.0,
    6000.0, 6020.0, 6040.0, 6060.0, 6080.0, 6100.0, 6120.0, 6140.0, 6160.0, 6180.0,
    6200.0, 6220.0, 6240.0, 6260.0, 6280.0, 6300.0, 6320.0, 6340.0, 6360.0, 6380.0,
    6400.0, 6420.0, 6440.0, 6460.0, 6480.0, 6500.0, 6520.0, 6540.0, 6560.0, 6580.0,
    6600.0, 6620.0, 6640.0, 6660.0, 6680.0, 6700.0, 6720.0, 6740.0, 6760.0, 6780.0,
    6800.0, 6820.0, 6840.0, 6860.0, 6880.0, 6900.0, 6920.0, 6940.0, 6960.0, 6980.0,
    7000.0, 7020.0, 7040.0, 7060.0, 7080.0, 7100.0, 7120.0, 7140.0, 7160.0, 7180.0,
    7200.0, 7220.0, 7240.0, 7260.0, 7280.0, 7300.0, 7320.0, 7340.0, 7360.0, 7380.0,
    7400.0, 7420.0, 7440.0, 7460.0, 7480.0, 7500.0, 7520.0, 7540.0, 7560.0, 7580.0,
    7600.0, 7620.0, 7640.0, 7660.0, 7680.0, 7700.0, 7720.0, 7740.0, 7760.0, 7780.0,
    7800.0, 7820.0, 7840.0, 7860.0, 7880.0, 7900.0, 7920.0, 7940.0, 7960.0, 7980.0,
    8000.0, 8020.0, 8040.0, 8060.0, 8080.0, 8100.0, 8120.0, 8140.0, 8160.0, 8180.0,
    8200.0, 8220.0, 8240.0, 8260.0, 8280.0, 8300.0, 8320.0, 8340.0, 8360.0, 8380.0,
    8400.0, 8420.0, 8440.0, 8460.0, 8480.0, 8500.0, 8520.0, 8540.0, 8560.0, 8580.0,
    8600.0, 8620.0, 8640.0, 8660.0, 8680.0, 8700.0, 8720.0, 8740.0, 8760.0, 8780.0,
    8800.0, 8820.0, 8840.0, 8860.0, 8880.0, 8900.0, 8920.0, 8940.0, 8960.0, 8980.0,
    9000.0,
])
TIPS_2017_ISOT[4] = float64([
    1.0, 20.0, 40.0, 60.0, 80.0, 100.0, 120.0, 140.0, 160.0, 180.0,
    200.0, 220.0, 240.0, 260.0, 280.0, 300.0, 320.0, 340.0, 360.0, 380.0,
    400.0, 420.0, 440.0, 460.0, 480.0, 500.0, 520.0, 540.0, 560.0, 580.0,
    600.0, 620.0, 640.0, 660.0, 680.0, 700.0, 720.0, 740.0, 760.0, 780.0,
    800.0, 820.0, 840.0, 860.0, 880.0, 900.0, 920.0, 940.0, 960.0, 980.0,
    1000.0, 1020.0, 1040.0, 1060.0, 1080.0, 1100.0, 1120.0, 1140.0, 1160.0, 1180.0,
    1200.0, 1220.0, 1240.0, 1260.0, 1280.0, 1300.0, 1320.0, 1340.0, 1360.0, 1380.0,
    1400.0, 1420.0, 1440.0, 1460.0, 1480.0, 1500.0, 1520.0, 1540.0, 1560.0, 1580.0,
    1600.0, 1620.0, 1640.0, 1660.0, 1680.0, 1700.0, 1720.0, 1740.0, 1760.0, 1780.0,
    1800.0, 1820.0, 1840.0, 1860.0, 1880.0, 1900.0, 1920.0, 1940.0, 1960.0, 1980.0,
    2000.0, 2020.0, 2040.0, 2060.0, 2080.0, 2100.0, 2120.0, 2140.0, 2160.0, 2180.0,
    2200.0, 2220.0, 2240.0, 2260.0, 2280.0, 2300.0, 2320.0, 2340.0, 2360.0, 2380.0,
    2400.0, 2420.0, 2440.0, 2460.0, 2480.0, 2500.0, 2520.0, 2540.0, 2560.0, 2580.0,
    2600.0, 2620.0, 2640.0, 2660.0, 2680.0, 2700.0, 2720.0, 2740.0, 2760.0, 2780.0,
    2800.0, 2820.0, 2840.0, 2860.0, 2880.0, 2900.0, 2920.0, 2940.0, 2960.0, 2980.0,
    3000.0, 3020.0, 3040.0, 3060.0, 3080.0, 3100.0, 3120.0, 3140.0, 3160.0, 3180.0,
    3200.0, 3220.0, 3240.0, 3260.0, 3280.0, 3300.0, 3320.0, 3340.0, 3360.0, 3380.0,
    3400.0, 3420.0, 3440.0, 3460.0, 3480.0, 3500.0, 3520.0, 3540.0, 3560.0, 3580.0,
    3600.0, 3620.0, 3640.0, 3660.0, 3680.0, 3700.0, 3720.0, 3740.0, 3760.0, 3780.0,
    3800.0, 3820.0, 3840.0, 3860.0, 3880.0, 3900.0, 3920.0, 3940.0, 3960.0, 3980.0,
    4000.0, 4020.0, 4040.0, 4060.0, 4080.0, 4100.0, 4120.0, 4140.0, 4160.0, 4180.0,
    4200.0, 4220.0, 4240.0, 4260.0, 4280.0, 4300.0, 4320.0, 4340.0, 4360.0, 4380.0,
    4400.0, 4420.0, 4440.0, 4460.0, 4480.0, 4500.0,
])
TIPS_2017_ISOT[5] = float64([
    1.0, 20.0, 40.0, 60.0, 80.0, 100.0, 120.0, 140.0, 160.0, 180.0,
    200.0, 220.0, 240.0, 260.0, 280.0, 300.0, 320.0, 340.0, 360.0, 380.0,
    400.0, 420.0, 440.0, 460.0, 480.0, 500.0, 520.0, 540.0, 560.0, 580.0,
    600.0, 620.0, 640.0, 660.0, 680.0, 700.0, 720.0, 740.0, 760.0, 780.0,
    800.0, 820.0, 840.0, 860.0, 880.0, 900.0, 920.0, 940.0, 960.0, 980.0,
    1000.0, 1020.0, 1040.0, 1060.0, 1080.0, 1100.0, 1120.0, 1140.0, 1160.0, 1180.0,
    1200.0, 1220.0, 1240.0, 1260.0, 1280.0, 1300.0, 1320.0, 1340.0, 1360.0, 1380.0,
    1400.0, 1420.0, 1440.0, 1460.0, 1480.0, 1500.0, 1520.0, 1540.0, 1560.0, 1580.0,
    1600.0, 1620.0, 1640.0, 1660.0, 1680.0, 1700.0, 1720.0, 1740.0, 1760.0, 1780.0,
    1800.0, 1820.0, 1840.0, 1860.0, 1880.0, 1900.0, 1920.0, 1940.0, 1960.0, 1980.0,
    2000.0, 2020.0, 2040.0, 2060.0, 2080.0, 2100.0, 2120.0, 2140.0, 2160.0, 2180.0,
    2200.0, 2220.0, 2240.0, 2260.0, 2280.0, 2300.0, 2320.0, 2340.0, 2360.0, 2380.0,
    2400.0, 2420.0, 2440.0, 2460.0, 2480.0, 2500.0, 2520.0, 2540.0, 2560.0, 2580.0,
    2600.0, 2620.0, 2640.0, 2660.0, 2680.0, 2700.0, 2720.0, 2740.0, 2760.0, 2780.0,
    2800.0, 2820.0, 2840.0, 2860.0, 2880.0, 2900.0, 2920.0, 2940.0, 2960.0, 2980.0,
    3000.0, 3020.0, 3040.0, 3060.0, 3080.0, 3100.0, 3120.0, 3140.0, 3160.0, 3180.0,
    3200.0, 3220.0, 3240.0, 3260.0, 3280.0, 3300.0, 3320.0, 3340.0, 3360.0, 3380.0,
    3400.0, 3420.0, 3440.0, 3460.0, 3480.0, 3500.0, 3520.0, 3540.0, 3560.0, 3580.0,
    3600.0, 3620.0, 3640.0, 3660.0, 3680.0, 3700.0, 3720.0, 3740.0, 3760.0, 3780.0,
    3800.0, 3820.0, 3840.0, 3860.0, 3880.0, 3900.0, 3920.0, 3940.0, 3960.0, 3980.0,
    4000.0, 4020.0, 4040.0, 4060.0, 4080.0, 4100.0, 4120.0, 4140.0, 4160.0, 4180.0,
    4200.0, 4220.0, 4240.0, 4260.0, 4280.0, 4300.0, 4320.0, 4340.0, 4360.0, 4380.0,
    4400.0, 4420.0, 4440.0, 4460.0, 4480.0, 4500.0, 4520.0, 4540.0, 4560.0, 4580.0,
    4600.0, 4620.0, 4640.0, 4660.0, 4680.0, 4700.0, 4720.0, 4740.0, 4760.0, 4780.0,
    4800.0, 4820.0, 4840.0, 4860.0, 4880.0, 4900.0, 4920.0, 4940.0, 4960.0, 4980.0,
    5000.0, 5020.0, 5040.0, 5060.0, 5080.0, 5100.0, 5120.0, 5140.0, 5160.0, 5180.0,
    5200.0, 5220.0, 5240.0, 5260.0, 5280.0, 5300.0, 5320.0, 5340.0, 5360.0, 5380.0,
    5400.0, 5420.0, 5440.0, 5460.0, 5480.0, 5500.0, 5520.0, 5540.0, 5560.0, 5580.0,
    5600.0, 5620.0, 5640.0, 5660.0, 5680.0, 5700.0, 5720.0, 5740.0, 5760.0, 5780.0,
    5800.0, 5820.0, 5840.0, 5860.0, 5880.0, 5900.0, 5920.0, 5940.0, 5960.0, 5980.0,
    6000.0, 6020.0, 6040.0, 6060.0, 6080.0, 6100.0, 6120.0, 6140.0, 6160.0, 6180.0,
    6200.0, 6220.0, 6240.0, 6260.0, 6280.0, 6300.0, 6320.0, 6340.0, 6360.0, 6380.0,
    6400.0, 6420.0, 6440.0, 6460.0, 6480.0, 6500.0, 6520.0, 6540.0, 6560.0, 6580.0,
    6600.0, 6620.0, 6640.0, 6660.0, 6680.0, 6700.0, 6720.0, 6740.0, 6760.0, 6780.0,
    6800.0, 6820.0, 6840.0, 6860.0, 6880.0, 6900.0, 6920.0, 6940.0, 6960.0, 6980.0,
    7000.0, 7020.0, 7040.0, 7060.0, 7080.0, 7100.0, 7120.0, 7140.0, 7160.0, 7180.0,
    7200.0, 7220.0, 7240.0, 7260.0, 7280.0, 7300.0, 7320.0, 7340.0, 7360.0, 7380.0,
    7400.0, 7420.0, 7440.0, 7460.0, 7480.0, 7500.0,
])
TIPS_2017_ISOT[6] = float64([
    1.0, 20.0, 40.0, 60.0, 80.0, 100.0, 120.0, 140.0, 160.0, 180.0,
    200.0, 220.0, 240.0, 260.0, 280.0, 300.0, 320.0, 340.0, 360.0, 380.0,
    400.0, 420.0, 440.0, 460.0, 480.0, 500.0, 520.0, 540.0, 560.0, 580.0,
    600.0, 620.0, 640.0, 660.0, 680.0, 700.0, 720.0, 740.0, 760.0, 780.0,
    800.0, 820.0, 840.0, 860.0, 880.0, 900.0, 920.0, 940.0, 960.0, 980.0,
    1000.0, 1020.0, 1040.0, 1060.0, 1080.0, 1100.0, 1120.0, 1140.0, 1160.0, 1180.0,
    1200.0, 1220.0, 1240.0, 1260.0, 1280.0, 1300.0, 1320.0, 1340.0, 1360.0, 1380.0,
    1400.0, 1420.0, 1440.0, 1460.0, 1480.0, 1500.0, 1520.0, 1540.0, 1560.0, 1580.0,
    1600.0, 1620.0, 1640.0, 1660.0, 1680.0, 1700.0, 1720.0, 1740.0, 1760.0, 1780.0,
    1800.0, 1820.0, 1840.0, 1860.0, 1880.0, 1900.0, 1920.0, 1940.0, 1960.0, 1980.0,
    2000.0, 2020.0, 2040.0, 2060.0, 2080.0, 2100.0, 2120.0, 2140.0, 2160.0, 2180.0,
    2200.0, 2220.0, 2240.0, 2260.0, 2280.0, 2300.0, 2320.0, 2340.0, 2360.0, 2380.0,
    2400.0, 2420.0, 2440.0, 2460.0, 2480.0, 2500.0, 2520.0, 2540.0, 2560.0, 2580.0,
    2600.0, 2620.0, 2640.0, 2660.0, 2680.0, 2700.0, 2720.0, 2740.0, 2760.0, 2780.0,
    2800.0, 2820.0, 2840.0, 2860.0, 2880.0, 2900.0, 2920.0, 2940.0, 2960.0, 2980.0,
    3000.0, 3020.0, 3040.0, 3060.0, 3080.0, 3100.0, 3120.0, 3140.0, 3160.0, 3180.0,
    3200.0, 3220.0, 3240.0, 3260.0, 3280.0, 3300.0, 3320.0, 3340.0, 3360.0, 3380.0,
    3400.0, 3420.0, 3440.0, 3460.0, 3480.0, 3500.0, 3520.0, 3540.0, 3560.0, 3580.0,
    3600.0, 3620.0, 3640.0, 3660.0, 3680.0, 3700.0, 3720.0, 3740.0, 3760.0, 3780.0,
    3800.0, 3820.0, 3840.0, 3860.0, 3880.0, 3900.0, 3920.0, 3940.0, 3960.0, 3980.0,
    4000.0,
])

#  --------------- TIPS-2017 PARTITION SUMS ----------------------
TIPS_2017_ISOT_HASH = {}
TIPS_2017_ISOQ_HASH = {}

#  ---------------------- M = 1, I = 1 ---------------------------
M = 1
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.000000E+00, 3.348920E+00, 9.416960E+00, 1.681863E+01, 2.543182E+01, 3.515313E+01,
    4.587953E+01, 5.752827E+01, 7.003401E+01, 8.334408E+01, 9.741515E+01, 1.122112E+02,
    1.277022E+02, 1.438633E+02, 1.606747E+02, 1.781206E+02, 1.961892E+02, 2.148721E+02,
    2.341643E+02, 2.540633E+02, 2.745691E+02, 2.956837E+02, 3.174110E+02, 3.397563E+02,
    3.627261E+02, 3.863281E+02, 4.105710E+02, 4.354642E+02, 4.610180E+02, 4.872432E+02,
    5.141514E+02, 5.417546E+02, 5.700656E+02, 5.990973E+02, 6.288636E+02, 6.593787E+02,
    6.906572E+02, 7.227143E+02, 7.555656E+02, 7.892274E+02, 8.237162E+02, 8.590491E+02,
    8.952437E+02, 9.323179E+02, 9.702901E+02, 1.009179E+03, 1.049005E+03, 1.089786E+03,
    1.131543E+03, 1.174297E+03, 1.218069E+03, 1.262880E+03, 1.308752E+03, 1.355706E+03,
    1.403766E+03, 1.452954E+03, 1.503294E+03, 1.554809E+03, 1.607523E+03, 1.661461E+03,
    1.716647E+03, 1.773106E+03, 1.830864E+03, 1.889947E+03, 1.950381E+03, 2.012191E+03,
    2.075405E+03, 2.140051E+03, 2.206155E+03, 2.273745E+03, 2.342850E+03, 2.413498E+03,
    2.485717E+03, 2.559538E+03, 2.634989E+03, 2.712101E+03, 2.790904E+03, 2.871427E+03,
    2.953704E+03, 3.037763E+03, 3.123638E+03, 3.211360E+03, 3.300961E+03, 3.392475E+03,
    3.485933E+03, 3.581371E+03, 3.678821E+03, 3.778318E+03, 3.879895E+03, 3.983589E+03,
    4.089435E+03, 4.197467E+03, 4.307722E+03, 4.420237E+03, 4.535049E+03, 4.652193E+03,
    4.771709E+03, 4.893634E+03, 5.018006E+03, 5.144865E+03, 5.274249E+03, 5.406197E+03,
    5.540751E+03, 5.677950E+03, 5.817835E+03, 5.960448E+03, 6.105829E+03, 6.254022E+03,
    6.405068E+03, 6.559010E+03, 6.715891E+03, 6.875757E+03, 7.038649E+03, 7.204614E+03,
    7.373696E+03, 7.545940E+03, 7.721393E+03, 7.900101E+03, 8.082111E+03, 8.267469E+03,
    8.456223E+03, 8.648422E+03, 8.844114E+03, 9.043348E+03, 9.246173E+03, 9.452640E+03,
    9.662798E+03, 9.876698E+03, 1.009439E+04, 1.031593E+04, 1.054137E+04, 1.077075E+04,
    1.100414E+04, 1.124158E+04, 1.148314E+04, 1.172886E+04, 1.197879E+04, 1.223300E+04,
    1.249154E+04, 1.275447E+04, 1.302184E+04, 1.329370E+04, 1.357012E+04, 1.385116E+04,
    1.413687E+04, 1.442730E+04, 1.472253E+04, 1.502260E+04, 1.532759E+04, 1.563754E+04,
    1.595251E+04, 1.627258E+04, 1.659780E+04, 1.692823E+04, 1.726394E+04, 1.760498E+04,
    1.795142E+04, 1.830332E+04, 1.866075E+04, 1.902377E+04, 1.939243E+04, 1.976682E+04,
    2.014699E+04, 2.053301E+04, 2.092493E+04, 2.132284E+04, 2.172678E+04, 2.213684E+04,
    2.255307E+04, 2.297555E+04, 2.340433E+04, 2.383949E+04, 2.428109E+04, 2.472920E+04,
    2.518388E+04, 2.564522E+04, 2.611326E+04, 2.658809E+04, 2.706977E+04, 2.755836E+04,
    2.805395E+04, 2.855658E+04, 2.906634E+04, 2.958330E+04, 3.010751E+04, 3.063906E+04,
    3.117801E+04, 3.172442E+04, 3.227838E+04, 3.283994E+04, 3.340917E+04, 3.398616E+04,
    3.457096E+04, 3.516364E+04, 3.576428E+04, 3.637294E+04, 3.698969E+04, 3.761461E+04,
    3.824775E+04, 3.888919E+04, 3.953900E+04, 4.019725E+04, 4.086401E+04, 4.153933E+04,
    4.222330E+04, 4.291598E+04, 4.361744E+04, 4.432774E+04, 4.504696E+04, 4.577516E+04,
    4.651241E+04, 4.725878E+04, 4.801433E+04, 4.877912E+04, 4.955323E+04, 5.033673E+04,
    5.112967E+04, 5.193213E+04, 5.274416E+04, 5.356584E+04, 5.439722E+04, 5.523838E+04,
    5.608937E+04, 5.695027E+04, 5.782113E+04, 5.870201E+04, 5.959298E+04, 6.049411E+04,
    6.140545E+04, 6.232706E+04, 6.325900E+04, 6.420135E+04, 6.515415E+04, 6.611746E+04,
    6.709136E+04, 6.807588E+04, 6.907110E+04, 7.007707E+04, 7.109385E+04, 7.212149E+04,
    7.316006E+04, 7.420960E+04, 7.527018E+04, 7.634185E+04, 7.742466E+04, 7.851867E+04,
    7.962393E+04, 8.074050E+04, 8.186842E+04, 8.300775E+04, 8.415854E+04,
])

#  ---------------------- M = 1, I = 2 ---------------------------
M = 1
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.000000E+00, 3.372790E+00, 9.492280E+00, 1.695557E+01, 2.564104E+01, 3.544414E+01,
    4.626091E+01, 5.800788E+01, 7.061916E+01, 8.404163E+01, 9.823160E+01, 1.131528E+02,
    1.287749E+02, 1.450731E+02, 1.620274E+02, 1.796219E+02, 1.978450E+02, 2.166883E+02,
    2.361468E+02, 2.562183E+02, 2.769027E+02, 2.982024E+02, 3.201213E+02, 3.426649E+02,
    3.658398E+02, 3.896541E+02, 4.141165E+02, 4.392366E+02, 4.650249E+02, 4.914924E+02,
    5.186508E+02, 5.465124E+02, 5.750899E+02, 6.043966E+02, 6.344463E+02, 6.652532E+02,
    6.968322E+02, 7.291983E+02, 7.623672E+02, 7.963551E+02, 8.311783E+02, 8.668538E+02,
    9.033990E+02, 9.408315E+02, 9.791695E+02, 1.018432E+03, 1.058637E+03, 1.099804E+03,
    1.141952E+03, 1.185103E+03, 1.229276E+03, 1.274491E+03, 1.320770E+03, 1.368133E+03,
    1.416603E+03, 1.466202E+03, 1.516951E+03, 1.568873E+03, 1.621992E+03, 1.676329E+03,
    1.731908E+03, 1.788753E+03, 1.846888E+03, 1.906337E+03, 1.967124E+03, 2.029275E+03,
    2.092814E+03, 2.157766E+03, 2.224157E+03, 2.292012E+03, 2.361359E+03, 2.432222E+03,
    2.504629E+03, 2.578606E+03, 2.654181E+03, 2.731380E+03, 2.810232E+03, 2.890764E+03,
    2.973005E+03, 3.056982E+03, 3.142725E+03, 3.230262E+03, 3.319623E+03, 3.410837E+03,
    3.503933E+03, 3.598943E+03, 3.695895E+03, 3.794822E+03, 3.895752E+03, 3.998719E+03,
    4.103752E+03, 4.210885E+03, 4.320147E+03, 4.431573E+03, 4.545194E+03, 4.661044E+03,
    4.779155E+03, 4.899561E+03, 5.022295E+03, 5.147392E+03, 5.274886E+03, 5.404811E+03,
    5.537202E+03, 5.672095E+03, 5.809525E+03, 5.949527E+03, 6.092139E+03, 6.237395E+03,
    6.385334E+03, 6.535992E+03, 6.689406E+03, 6.845614E+03, 7.004655E+03, 7.166565E+03,
    7.331385E+03, 7.499153E+03, 7.669907E+03, 7.843689E+03, 8.020537E+03, 8.200491E+03,
    8.383594E+03, 8.569884E+03, 8.759404E+03, 8.952195E+03, 9.148299E+03, 9.347757E+03,
    9.550614E+03, 9.756911E+03, 9.966691E+03, 1.018000E+04, 1.039688E+04, 1.061737E+04,
    1.084152E+04, 1.106938E+04, 1.130099E+04, 1.153639E+04, 1.177564E+04, 1.201877E+04,
    1.226583E+04, 1.251688E+04, 1.277195E+04, 1.303110E+04, 1.329438E+04, 1.356182E+04,
    1.383348E+04, 1.410942E+04, 1.438967E+04, 1.467429E+04, 1.496333E+04, 1.525683E+04,
    1.555486E+04, 1.585745E+04, 1.616467E+04, 1.647655E+04, 1.679316E+04, 1.711455E+04,
    1.744077E+04, 1.777186E+04, 1.810790E+04, 1.844892E+04, 1.879498E+04, 1.914614E+04,
    1.950245E+04, 1.986396E+04, 2.023074E+04, 2.060283E+04, 2.098029E+04, 2.136318E+04,
    2.175156E+04, 2.214547E+04, 2.254498E+04, 2.295015E+04, 2.336103E+04, 2.377768E+04,
    2.420015E+04, 2.462851E+04, 2.506282E+04, 2.550313E+04, 2.594950E+04, 2.640200E+04,
    2.686067E+04, 2.732559E+04, 2.779681E+04, 2.827439E+04, 2.875840E+04, 2.924889E+04,
    2.974592E+04, 3.024956E+04, 3.075987E+04, 3.127691E+04, 3.180074E+04, 3.233143E+04,
    3.286903E+04, 3.341361E+04, 3.396524E+04, 3.452397E+04, 3.508988E+04, 3.566301E+04,
    3.624345E+04, 3.683124E+04, 3.742646E+04, 3.802917E+04, 3.863943E+04, 3.925731E+04,
    3.988288E+04, 4.051619E+04, 4.115732E+04, 4.180633E+04, 4.246328E+04, 4.312824E+04,
    4.380128E+04, 4.448245E+04, 4.517184E+04, 4.586950E+04, 4.657550E+04, 4.728991E+04,
    4.801279E+04, 4.874421E+04, 4.948423E+04, 5.023293E+04, 5.099037E+04, 5.175661E+04,
    5.253172E+04, 5.331578E+04, 5.410884E+04, 5.491098E+04, 5.572226E+04, 5.654275E+04,
    5.737251E+04, 5.821161E+04, 5.906013E+04, 5.991812E+04, 6.078566E+04, 6.166281E+04,
    6.254964E+04, 6.344622E+04, 6.435261E+04, 6.526889E+04, 6.619511E+04, 6.713135E+04,
    6.807767E+04, 6.903414E+04, 7.000083E+04, 7.097781E+04, 7.196513E+04, 7.296288E+04,
    7.397111E+04, 7.498990E+04, 7.601930E+04, 7.705940E+04, 7.811024E+04,
])

#  ---------------------- M = 1, I = 3 ---------------------------
M = 1
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    6.000000E+00, 2.016901E+01, 5.674024E+01, 1.013453E+02, 1.532533E+02, 2.118401E+02,
    2.764846E+02, 3.466880E+02, 4.220566E+02, 5.022729E+02, 5.870757E+02, 6.762480E+02,
    7.696091E+02, 8.670100E+02, 9.683305E+02, 1.073476E+03, 1.182375E+03, 1.294979E+03,
    1.411255E+03, 1.531189E+03, 1.654781E+03, 1.782042E+03, 1.912994E+03, 2.047669E+03,
    2.186104E+03, 2.328346E+03, 2.474446E+03, 2.624459E+03, 2.778445E+03, 2.936469E+03,
    3.098600E+03, 3.264908E+03, 3.435470E+03, 3.610362E+03, 3.789666E+03, 3.973467E+03,
    4.161852E+03, 4.354909E+03, 4.552733E+03, 4.755418E+03, 4.963062E+03, 5.175767E+03,
    5.393635E+03, 5.616772E+03, 5.845286E+03, 6.079288E+03, 6.318891E+03, 6.564209E+03,
    6.815361E+03, 7.072464E+03, 7.335642E+03, 7.605018E+03, 7.880717E+03, 8.162866E+03,
    8.451596E+03, 8.747038E+03, 9.049324E+03, 9.358591E+03, 9.674973E+03, 9.998611E+03,
    1.032964E+04, 1.066821E+04, 1.101447E+04, 1.136854E+04, 1.173059E+04, 1.210077E+04,
    1.247921E+04, 1.286608E+04, 1.326152E+04, 1.366570E+04, 1.407876E+04, 1.450087E+04,
    1.493219E+04, 1.537288E+04, 1.582310E+04, 1.628301E+04, 1.675279E+04, 1.723259E+04,
    1.772260E+04, 1.822297E+04, 1.873389E+04, 1.925553E+04, 1.978806E+04, 2.033166E+04,
    2.088651E+04, 2.145280E+04, 2.203069E+04, 2.262039E+04, 2.322207E+04, 2.383593E+04,
    2.446215E+04, 2.510092E+04, 2.575244E+04, 2.641690E+04, 2.709449E+04, 2.778543E+04,
    2.848989E+04, 2.920810E+04, 2.994025E+04, 3.068655E+04, 3.144720E+04, 3.222241E+04,
    3.301240E+04, 3.381738E+04, 3.463755E+04, 3.547315E+04, 3.632438E+04, 3.719147E+04,
    3.807464E+04, 3.897411E+04, 3.989012E+04, 4.082287E+04, 4.177262E+04, 4.273959E+04,
    4.372400E+04, 4.472611E+04, 4.574614E+04, 4.678433E+04, 4.784094E+04, 4.891619E+04,
    5.001034E+04, 5.112363E+04, 5.225631E+04, 5.340864E+04, 5.458086E+04, 5.577324E+04,
    5.698602E+04, 5.821948E+04, 5.947386E+04, 6.074944E+04, 6.204648E+04, 6.336525E+04,
    6.470602E+04, 6.606905E+04, 6.745462E+04, 6.886301E+04, 7.029450E+04, 7.174936E+04,
    7.322787E+04, 7.473032E+04, 7.625700E+04, 7.780820E+04, 7.938419E+04, 8.098528E+04,
    8.261176E+04, 8.426392E+04, 8.594206E+04, 8.764648E+04, 8.937748E+04, 9.113537E+04,
    9.292045E+04, 9.473303E+04, 9.657342E+04, 9.844193E+04, 1.003389E+05, 1.022646E+05,
    1.042193E+05, 1.062035E+05, 1.082173E+05, 1.102612E+05, 1.123354E+05, 1.144403E+05,
    1.165763E+05, 1.187435E+05, 1.209425E+05, 1.231734E+05, 1.254367E+05, 1.277326E+05,
    1.300616E+05, 1.324240E+05, 1.348200E+05, 1.372501E+05, 1.397146E+05, 1.422138E+05,
    1.447482E+05, 1.473179E+05, 1.499235E+05, 1.525652E+05, 1.552435E+05, 1.579586E+05,
    1.607109E+05, 1.635008E+05, 1.663287E+05, 1.691948E+05, 1.720997E+05, 1.750436E+05,
    1.780269E+05, 1.810500E+05, 1.841132E+05, 1.872170E+05, 1.903617E+05, 1.935477E+05,
    1.967753E+05, 2.000449E+05, 2.033570E+05, 2.067118E+05, 2.101098E+05, 2.135514E+05,
    2.170369E+05, 2.205667E+05, 2.241412E+05, 2.277608E+05, 2.314260E+05, 2.351369E+05,
    2.388942E+05, 2.426981E+05, 2.465490E+05, 2.504474E+05, 2.543936E+05, 2.583880E+05,
    2.624310E+05, 2.665231E+05, 2.706645E+05, 2.748558E+05, 2.790973E+05, 2.833893E+05,
    2.877324E+05, 2.921269E+05, 2.965732E+05, 3.010717E+05, 3.056228E+05, 3.102268E+05,
    3.148843E+05, 3.195956E+05, 3.243612E+05, 3.291813E+05, 3.340565E+05, 3.389870E+05,
    3.439734E+05, 3.490161E+05, 3.541153E+05, 3.592716E+05, 3.644854E+05, 3.697570E+05,
    3.750868E+05, 3.804753E+05, 3.859229E+05, 3.914300E+05, 3.969969E+05, 4.026240E+05,
    4.083119E+05, 4.140609E+05, 4.198713E+05, 4.257436E+05, 4.316783E+05, 4.376756E+05,
    4.437360E+05, 4.498599E+05, 4.560478E+05, 4.622999E+05, 4.686168E+05,
])

#  ---------------------- M = 1, I = 4 ---------------------------
M = 1
I = 4
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    6.000000E+00, 1.766833E+01, 4.570241E+01, 8.177074E+01, 1.242917E+02, 1.724141E+02,
    2.255614E+02, 2.833095E+02, 3.453297E+02, 4.113595E+02, 4.811860E+02, 5.546366E+02,
    6.315735E+02, 7.118902E+02, 7.955081E+02, 8.823735E+02, 9.724548E+02, 1.065740E+03,
    1.162232E+03, 1.261951E+03, 1.364927E+03, 1.471199E+03, 1.580818E+03, 1.693839E+03,
    1.810326E+03, 1.930347E+03, 2.053977E+03, 2.181293E+03, 2.312379E+03, 2.447320E+03,
    2.586208E+03, 2.729136E+03, 2.876201E+03, 3.027504E+03, 3.183148E+03, 3.343240E+03,
    3.507890E+03, 3.677211E+03, 3.851318E+03, 4.030329E+03, 4.214365E+03, 4.403551E+03,
    4.598012E+03, 4.797878E+03, 5.003279E+03, 5.214351E+03, 5.431230E+03, 5.654054E+03,
    5.882966E+03, 6.118108E+03, 6.359628E+03, 6.607674E+03, 6.862397E+03, 7.123950E+03,
    7.392489E+03, 7.668172E+03, 7.951159E+03, 8.241613E+03, 8.539699E+03, 8.845584E+03,
    9.159437E+03, 9.481431E+03, 9.811739E+03, 1.015054E+04, 1.049801E+04, 1.085433E+04,
    1.121969E+04, 1.159426E+04, 1.197825E+04, 1.237183E+04, 1.277521E+04, 1.318858E+04,
    1.361213E+04, 1.404607E+04, 1.449060E+04, 1.494593E+04, 1.541225E+04, 1.588979E+04,
    1.637875E+04, 1.687936E+04, 1.739182E+04, 1.791635E+04, 1.845319E+04, 1.900254E+04,
    1.956465E+04, 2.013974E+04, 2.072804E+04, 2.132979E+04, 2.194522E+04, 2.257457E+04,
    2.321808E+04, 2.387600E+04, 2.454858E+04, 2.523606E+04, 2.593868E+04, 2.665672E+04,
    2.739041E+04, 2.814002E+04, 2.890581E+04, 2.968803E+04, 3.048695E+04, 3.130284E+04,
    3.213596E+04, 3.298659E+04, 3.385498E+04, 3.474142E+04, 3.564619E+04, 3.656955E+04,
    3.751178E+04, 3.847316E+04, 3.945398E+04, 4.045452E+04, 4.147505E+04, 4.251587E+04,
    4.357726E+04, 4.465950E+04, 4.576289E+04, 4.688772E+04, 4.803426E+04, 4.920283E+04,
    5.039369E+04, 5.160716E+04, 5.284352E+04, 5.410306E+04, 5.538609E+04, 5.669288E+04,
    5.802374E+04, 5.937896E+04, 6.075884E+04, 6.216367E+04, 6.359374E+04, 6.504935E+04,
    6.653080E+04, 6.803837E+04, 6.957236E+04, 7.113307E+04, 7.272078E+04, 7.433578E+04,
    7.597838E+04, 7.764885E+04, 7.934749E+04, 8.107459E+04, 8.283042E+04, 8.461529E+04,
    8.642947E+04, 8.827324E+04, 9.014689E+04, 9.205070E+04, 9.398494E+04, 9.594990E+04,
    9.794585E+04, 9.997307E+04, 1.020318E+05, 1.041224E+05, 1.062450E+05, 1.084000E+05,
    1.105876E+05, 1.128080E+05, 1.150616E+05, 1.173485E+05, 1.196691E+05, 1.220236E+05,
    1.244122E+05, 1.268353E+05, 1.292929E+05, 1.317854E+05, 1.343131E+05, 1.368761E+05,
    1.394747E+05, 1.421091E+05, 1.447796E+05, 1.474863E+05, 1.502295E+05, 1.530095E+05,
    1.558264E+05, 1.586804E+05, 1.615718E+05, 1.645008E+05, 1.674676E+05, 1.704723E+05,
    1.735152E+05, 1.765964E+05, 1.797163E+05, 1.828748E+05, 1.860723E+05, 1.893090E+05,
    1.925849E+05, 1.959003E+05, 1.992553E+05, 2.026501E+05, 2.060849E+05, 2.095599E+05,
    2.130751E+05, 2.166308E+05, 2.202271E+05, 2.238641E+05, 2.275421E+05, 2.312610E+05,
    2.350212E+05, 2.388226E+05, 2.426655E+05, 2.465499E+05, 2.504760E+05, 2.544439E+05,
    2.584538E+05, 2.625057E+05, 2.665997E+05, 2.707359E+05, 2.749145E+05, 2.791356E+05,
    2.833992E+05, 2.877055E+05, 2.920544E+05, 2.964462E+05, 3.008809E+05, 3.053586E+05,
    3.098792E+05, 3.144431E+05, 3.190500E+05, 3.237003E+05, 3.283938E+05, 3.331308E+05,
    3.379111E+05, 3.427350E+05, 3.476023E+05, 3.525132E+05, 3.574677E+05, 3.624659E+05,
    3.675078E+05, 3.725934E+05, 3.777227E+05, 3.828958E+05, 3.881127E+05, 3.933734E+05,
    3.986779E+05, 4.040263E+05, 4.094186E+05, 4.148547E+05, 4.203346E+05, 4.258585E+05,
    4.314262E+05, 4.370377E+05, 4.426932E+05, 4.483925E+05, 4.541356E+05, 4.599225E+05,
    4.657532E+05, 4.716277E+05, 4.775460E+05, 4.835080E+05, 4.895137E+05,
])

#  ---------------------- M = 1, I = 5 ---------------------------
M = 1
I = 5
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    6.000000E+00, 1.786086E+01, 4.625151E+01, 8.276956E+01, 1.258219E+02, 1.745467E+02,
    2.283599E+02, 2.868320E+02, 3.496303E+02, 4.164889E+02, 4.871921E+02, 5.615653E+02,
    6.394690E+02, 7.207956E+02, 8.054658E+02, 8.934261E+02, 9.846456E+02, 1.079113E+03,
    1.176835E+03, 1.277832E+03, 1.382138E+03, 1.489795E+03, 1.600858E+03, 1.715385E+03,
    1.833446E+03, 1.955112E+03, 2.080462E+03, 2.209579E+03, 2.342549E+03, 2.479463E+03,
    2.620417E+03, 2.765507E+03, 2.914836E+03, 3.068509E+03, 3.226632E+03, 3.389316E+03,
    3.556674E+03, 3.728824E+03, 3.905883E+03, 4.087973E+03, 4.275219E+03, 4.467747E+03,
    4.665685E+03, 4.869166E+03, 5.078323E+03, 5.293293E+03, 5.514213E+03, 5.741224E+03,
    5.974468E+03, 6.214091E+03, 6.460239E+03, 6.713060E+03, 6.972705E+03, 7.239327E+03,
    7.513080E+03, 7.794120E+03, 8.082606E+03, 8.378697E+03, 8.682555E+03, 8.994343E+03,
    9.314226E+03, 9.642370E+03, 9.978944E+03, 1.032412E+04, 1.067806E+04, 1.104095E+04,
    1.141296E+04, 1.179427E+04, 1.218504E+04, 1.258547E+04, 1.299573E+04, 1.341601E+04,
    1.384648E+04, 1.428733E+04, 1.473876E+04, 1.520094E+04, 1.567407E+04, 1.615834E+04,
    1.665393E+04, 1.716105E+04, 1.767988E+04, 1.821063E+04, 1.875349E+04, 1.930866E+04,
    1.987633E+04, 2.045672E+04, 2.105002E+04, 2.165643E+04, 2.227617E+04, 2.290944E+04,
    2.355644E+04, 2.421738E+04, 2.489248E+04, 2.558195E+04, 2.628600E+04, 2.700484E+04,
    2.773869E+04, 2.848776E+04, 2.925227E+04, 3.003245E+04, 3.082850E+04, 3.164066E+04,
    3.246914E+04, 3.331417E+04, 3.417596E+04, 3.505476E+04, 3.595078E+04, 3.686425E+04,
    3.779540E+04, 3.874447E+04, 3.971168E+04, 4.069726E+04, 4.170145E+04, 4.272449E+04,
    4.376661E+04, 4.482804E+04, 4.590903E+04, 4.700981E+04, 4.813063E+04, 4.927172E+04,
    5.043333E+04, 5.161570E+04, 5.281908E+04, 5.404371E+04, 5.528984E+04, 5.655772E+04,
    5.784759E+04, 5.915970E+04, 6.049431E+04, 6.185167E+04, 6.323202E+04, 6.463563E+04,
    6.606275E+04, 6.751363E+04, 6.898853E+04, 7.048770E+04, 7.201142E+04, 7.355993E+04,
    7.513350E+04, 7.673238E+04, 7.835685E+04, 8.000716E+04, 8.168358E+04, 8.338637E+04,
    8.511579E+04, 8.687212E+04, 8.865563E+04, 9.046658E+04, 9.230523E+04, 9.417187E+04,
    9.606675E+04, 9.799016E+04, 9.994237E+04, 1.019236E+05, 1.039343E+05, 1.059745E+05,
    1.080446E+05, 1.101449E+05, 1.122756E+05, 1.144371E+05, 1.166295E+05, 1.188533E+05,
    1.211086E+05, 1.233957E+05, 1.257149E+05, 1.280666E+05, 1.304509E+05, 1.328682E+05,
    1.353187E+05, 1.378027E+05, 1.403206E+05, 1.428725E+05, 1.454589E+05, 1.480799E+05,
    1.507358E+05, 1.534269E+05, 1.561536E+05, 1.589161E+05, 1.617147E+05, 1.645496E+05,
    1.674212E+05, 1.703297E+05, 1.732755E+05, 1.762588E+05, 1.792799E+05, 1.823391E+05,
    1.854366E+05, 1.885729E+05, 1.917480E+05, 1.949625E+05, 1.982164E+05, 2.015102E+05,
    2.048440E+05, 2.082182E+05, 2.116331E+05, 2.150890E+05, 2.185861E+05, 2.221247E+05,
    2.257051E+05, 2.293277E+05, 2.329926E+05, 2.367001E+05, 2.404506E+05, 2.442444E+05,
    2.480816E+05, 2.519626E+05, 2.558877E+05, 2.598572E+05, 2.638713E+05, 2.679303E+05,
    2.720345E+05, 2.761841E+05, 2.803795E+05, 2.846209E+05, 2.889086E+05, 2.932429E+05,
    2.976240E+05, 3.020522E+05, 3.065279E+05, 3.110511E+05, 3.156223E+05, 3.202417E+05,
    3.249096E+05, 3.296262E+05, 3.343917E+05, 3.392066E+05, 3.440709E+05, 3.489851E+05,
    3.539492E+05, 3.589637E+05, 3.640287E+05, 3.691446E+05, 3.743115E+05, 3.795298E+05,
    3.847996E+05, 3.901213E+05, 3.954950E+05, 4.009211E+05, 4.063997E+05, 4.119311E+05,
    4.175156E+05, 4.231535E+05, 4.288448E+05, 4.345899E+05, 4.403891E+05, 4.462425E+05,
    4.521504E+05, 4.581130E+05, 4.641305E+05, 4.702033E+05, 4.763314E+05,
])

#  ---------------------- M = 1, I = 6 ---------------------------
M = 1
I = 6
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    3.600000E+01, 1.066180E+02, 2.759495E+02, 4.937802E+02, 7.505847E+02, 1.041223E+03,
    1.362217E+03, 1.711020E+03, 2.085674E+03, 2.484623E+03, 2.906617E+03, 3.350642E+03,
    3.815891E+03, 4.301738E+03, 4.807723E+03, 5.333532E+03, 5.878982E+03, 6.444007E+03,
    7.028640E+03, 7.633001E+03, 8.257283E+03, 8.901742E+03, 9.566686E+03, 1.025247E+04,
    1.095949E+04, 1.168818E+04, 1.243899E+04, 1.321241E+04, 1.400896E+04, 1.482918E+04,
    1.567361E+04, 1.654285E+04, 1.743748E+04, 1.835812E+04, 1.930541E+04, 2.027999E+04,
    2.128254E+04, 2.231374E+04, 2.337427E+04, 2.446487E+04, 2.558624E+04, 2.673914E+04,
    2.792432E+04, 2.914254E+04, 3.039459E+04, 3.168126E+04, 3.300335E+04, 3.436168E+04,
    3.575707E+04, 3.719038E+04, 3.866244E+04, 4.017412E+04, 4.172629E+04, 4.331984E+04,
    4.495565E+04, 4.663462E+04, 4.835768E+04, 5.012574E+04, 5.193973E+04, 5.380059E+04,
    5.570927E+04, 5.766673E+04, 5.967393E+04, 6.173186E+04, 6.384149E+04, 6.600381E+04,
    6.821984E+04, 7.049057E+04, 7.281703E+04, 7.520023E+04, 7.764122E+04, 8.014104E+04,
    8.270072E+04, 8.532134E+04, 8.800395E+04, 9.074963E+04, 9.355945E+04, 9.643451E+04,
    9.937589E+04, 1.023847E+05, 1.054620E+05, 1.086090E+05, 1.118268E+05, 1.151165E+05,
    1.184792E+05, 1.219162E+05, 1.254284E+05, 1.290172E+05, 1.326836E+05, 1.364289E+05,
    1.402542E+05, 1.441607E+05, 1.481496E+05, 1.522221E+05, 1.563793E+05, 1.606226E+05,
    1.649532E+05, 1.693722E+05, 1.738809E+05, 1.784805E+05, 1.831723E+05, 1.879575E+05,
    1.928374E+05, 1.978133E+05, 2.028864E+05, 2.080579E+05, 2.133293E+05, 2.187018E+05,
    2.241766E+05, 2.297550E+05, 2.354385E+05, 2.412283E+05, 2.471256E+05, 2.531319E+05,
    2.592485E+05, 2.654767E+05, 2.718178E+05, 2.782733E+05, 2.848444E+05, 2.915325E+05,
    2.983391E+05, 3.052654E+05, 3.123128E+05, 3.194828E+05, 3.267767E+05, 3.341959E+05,
    3.417418E+05, 3.494159E+05, 3.572195E+05, 3.651541E+05, 3.732210E+05, 3.814218E+05,
    3.897578E+05, 3.982305E+05, 4.068414E+05, 4.155918E+05, 4.244833E+05, 4.335172E+05,
    4.426952E+05, 4.520186E+05, 4.614889E+05, 4.711076E+05, 4.808762E+05, 4.907962E+05,
    5.008691E+05, 5.110963E+05, 5.214794E+05, 5.320199E+05, 5.427193E+05, 5.535792E+05,
    5.646009E+05, 5.757862E+05, 5.871364E+05, 5.986532E+05, 6.103381E+05, 6.221926E+05,
    6.342182E+05, 6.464166E+05, 6.587892E+05, 6.713377E+05, 6.840635E+05, 6.969684E+05,
    7.100537E+05, 7.233211E+05, 7.367722E+05, 7.504086E+05, 7.642317E+05, 7.782433E+05,
    7.924449E+05, 8.068380E+05, 8.214243E+05, 8.362053E+05, 8.511827E+05, 8.663581E+05,
    8.817330E+05, 8.973090E+05, 9.130878E+05, 9.290709E+05, 9.452599E+05, 9.616565E+05,
    9.782622E+05, 9.950787E+05, 1.012108E+06, 1.029350E+06, 1.046809E+06, 1.064484E+06,
    1.082379E+06, 1.100493E+06, 1.118830E+06, 1.137390E+06, 1.156176E+06, 1.175188E+06,
    1.194429E+06, 1.213900E+06, 1.233602E+06, 1.253538E+06, 1.273708E+06, 1.294115E+06,
    1.314760E+06, 1.335644E+06, 1.356770E+06, 1.378138E+06, 1.399750E+06, 1.421609E+06,
    1.443715E+06, 1.466070E+06, 1.488676E+06, 1.511534E+06, 1.534646E+06, 1.558013E+06,
    1.581636E+06, 1.605519E+06, 1.629661E+06, 1.654065E+06, 1.678732E+06, 1.703663E+06,
    1.728861E+06, 1.754326E+06, 1.780061E+06, 1.806066E+06, 1.832343E+06, 1.858894E+06,
    1.885720E+06, 1.912822E+06, 1.940203E+06, 1.967863E+06, 1.995805E+06, 2.024028E+06,
    2.052536E+06, 2.081329E+06, 2.110410E+06, 2.139778E+06, 2.169436E+06, 2.199385E+06,
    2.229627E+06, 2.260162E+06, 2.290993E+06, 2.322120E+06, 2.353546E+06, 2.385271E+06,
    2.417296E+06, 2.449624E+06, 2.482255E+06, 2.515191E+06, 2.548433E+06, 2.581982E+06,
    2.615840E+06, 2.650008E+06, 2.684487E+06, 2.719279E+06, 2.754384E+06,
])

#  ---------------------- M = 1, I = 7 ---------------------------
M = 1
I = 7
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    6.000000E+00, 2.020566E+01, 5.328177E+01, 9.597421E+01, 1.463666E+02, 2.034368E+02,
    2.664953E+02, 3.350360E+02, 4.086708E+02, 4.870968E+02, 5.700805E+02, 6.574474E+02,
    7.490750E+02, 8.448862E+02, 9.448428E+02, 1.048939E+03, 1.157194E+03, 1.269651E+03,
    1.386368E+03, 1.507418E+03, 1.632887E+03, 1.762868E+03, 1.897464E+03, 2.036785E+03,
    2.180948E+03, 2.330078E+03, 2.484305E+03, 2.643764E+03, 2.808599E+03, 2.978957E+03,
    3.154991E+03, 3.336863E+03, 3.524736E+03, 3.718782E+03, 3.919176E+03, 4.126099E+03,
    4.339738E+03, 4.560283E+03, 4.787929E+03, 5.022879E+03, 5.265336E+03, 5.515511E+03,
    5.773617E+03, 6.039873E+03, 6.314501E+03, 6.597728E+03, 6.889787E+03, 7.190910E+03,
    7.501338E+03, 7.821315E+03, 8.151087E+03, 8.490906E+03, 8.841027E+03, 9.201709E+03,
    9.573217E+03, 9.955817E+03, 1.034978E+04, 1.075538E+04, 1.117291E+04, 1.160263E+04,
    1.204485E+04, 1.249984E+04, 1.296792E+04, 1.344937E+04, 1.394451E+04, 1.445364E+04,
    1.497708E+04, 1.551513E+04, 1.606813E+04, 1.663640E+04, 1.722027E+04, 1.782008E+04,
    1.843615E+04, 1.906885E+04, 1.971852E+04, 2.038550E+04, 2.107016E+04, 2.177286E+04,
    2.249397E+04, 2.323386E+04, 2.399290E+04, 2.477148E+04, 2.556999E+04, 2.638881E+04,
    2.722834E+04, 2.808898E+04, 2.897114E+04, 2.987523E+04, 3.080167E+04, 3.175088E+04,
    3.272328E+04, 3.371931E+04, 3.473940E+04, 3.578400E+04, 3.685355E+04, 3.794852E+04,
    3.906935E+04, 4.021651E+04, 4.139047E+04, 4.259171E+04, 4.382070E+04, 4.507794E+04,
    4.636392E+04, 4.767913E+04, 4.902407E+04, 5.039927E+04, 5.180522E+04, 5.324247E+04,
    5.471152E+04, 5.621291E+04, 5.774719E+04, 5.931490E+04, 6.091658E+04, 6.255280E+04,
    6.422412E+04, 6.593111E+04, 6.767435E+04, 6.945442E+04, 7.127190E+04, 7.312739E+04,
    7.502150E+04, 7.695482E+04, 7.892798E+04, 8.094160E+04, 8.299630E+04, 8.509271E+04,
    8.723149E+04, 8.941326E+04, 9.163870E+04, 9.390846E+04, 9.622320E+04, 9.858361E+04,
    1.009904E+05, 1.034441E+05, 1.059457E+05, 1.084956E+05, 1.110947E+05, 1.137436E+05,
    1.164431E+05, 1.191939E+05, 1.219968E+05, 1.248525E+05, 1.277617E+05, 1.307252E+05,
    1.337438E+05, 1.368183E+05, 1.399493E+05, 1.431378E+05, 1.463845E+05, 1.496903E+05,
    1.530558E+05, 1.564820E+05, 1.599697E+05, 1.635197E+05, 1.671329E+05, 1.708101E+05,
    1.745521E+05, 1.783599E+05, 1.822343E+05, 1.861762E+05, 1.901864E+05, 1.942659E+05,
    1.984156E+05, 2.026364E+05, 2.069291E+05, 2.112949E+05, 2.157345E+05, 2.202489E+05,
    2.248391E+05, 2.295060E+05, 2.342506E+05, 2.390739E+05, 2.439768E+05, 2.489604E+05,
    2.540256E+05, 2.591735E+05, 2.644051E+05, 2.697213E+05, 2.751233E+05, 2.806120E+05,
    2.861886E+05, 2.918540E+05, 2.976093E+05, 3.034557E+05, 3.093941E+05, 3.154257E+05,
    3.215515E+05, 3.277727E+05, 3.340904E+05, 3.405057E+05, 3.470196E+05, 3.536334E+05,
    3.603482E+05, 3.671652E+05, 3.740854E+05, 3.811100E+05, 3.882403E+05, 3.954774E+05,
    4.028224E+05, 4.102766E+05, 4.178412E+05, 4.255174E+05, 4.333064E+05, 4.412094E+05,
    4.492276E+05, 4.573623E+05, 4.656147E+05, 4.739861E+05, 4.824777E+05, 4.910907E+05,
    4.998266E+05, 5.086864E+05, 5.176716E+05, 5.267833E+05, 5.360229E+05, 5.453917E+05,
    5.548910E+05, 5.645222E+05, 5.742864E+05, 5.841851E+05, 5.942195E+05, 6.043911E+05,
    6.147011E+05, 6.251509E+05, 6.357419E+05, 6.464753E+05, 6.573526E+05, 6.683752E+05,
    6.795444E+05, 6.908615E+05, 7.023280E+05, 7.139453E+05, 7.257147E+05, 7.376376E+05,
    7.497155E+05, 7.619497E+05, 7.743416E+05, 7.868928E+05, 7.996044E+05, 8.124781E+05,
    8.255152E+05, 8.387171E+05, 8.520852E+05, 8.656211E+05, 8.793261E+05, 8.932017E+05,
    9.072493E+05, 9.214703E+05, 9.358662E+05, 9.504385E+05, 9.651886E+05, 9.801180E+05,
    9.952280E+05, 1.010520E+06, 1.025996E+06, 1.041657E+06, 1.057504E+06, 1.073540E+06,
    1.089765E+06, 1.106181E+06, 1.122789E+06, 1.139591E+06, 1.156589E+06, 1.173783E+06,
    1.191175E+06, 1.208768E+06, 1.226561E+06, 1.244557E+06, 1.262758E+06, 1.281164E+06,
    1.299777E+06, 1.318598E+06, 1.337630E+06, 1.356873E+06, 1.376329E+06, 1.396000E+06,
    1.415886E+06, 1.435990E+06, 1.456312E+06, 1.476855E+06, 1.497619E+06, 1.518607E+06,
    1.539819E+06, 1.561257E+06, 1.582923E+06, 1.604818E+06, 1.626943E+06, 1.649300E+06,
    1.671891E+06, 1.694716E+06, 1.717777E+06, 1.741076E+06, 1.764614E+06, 1.788392E+06,
    1.812412E+06, 1.836675E+06, 1.861184E+06, 1.885938E+06, 1.910939E+06, 1.936190E+06,
    1.961691E+06,
])

#  ---------------------- M = 1, I = 8 ---------------------------
M = 1
I = 8
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    6.000000E+00, 2.048931E+01, 5.406267E+01, 9.739573E+01, 1.485457E+02, 2.064750E+02,
    2.704833E+02, 3.400571E+02, 4.148027E+02, 4.944135E+02, 5.786535E+02, 6.673477E+02,
    7.603741E+02, 8.576570E+02, 9.591602E+02, 1.064880E+03, 1.174840E+03, 1.289086E+03,
    1.407678E+03, 1.530695E+03, 1.658222E+03, 1.790360E+03, 1.927213E+03, 2.068896E+03,
    2.215529E+03, 2.367241E+03, 2.524165E+03, 2.686441E+03, 2.854217E+03, 3.027645E+03,
    3.206883E+03, 3.392094E+03, 3.583450E+03, 3.781124E+03, 3.985298E+03, 4.196157E+03,
    4.413893E+03, 4.638702E+03, 4.870784E+03, 5.110345E+03, 5.357596E+03, 5.612751E+03,
    5.876031E+03, 6.147658E+03, 6.427862E+03, 6.716875E+03, 7.014934E+03, 7.322280E+03,
    7.639158E+03, 7.965818E+03, 8.302513E+03, 8.649501E+03, 9.007044E+03, 9.375407E+03,
    9.754862E+03, 1.014568E+04, 1.054814E+04, 1.096253E+04, 1.138913E+04, 1.182824E+04,
    1.228015E+04, 1.274515E+04, 1.322355E+04, 1.371567E+04, 1.422181E+04, 1.474229E+04,
    1.527743E+04, 1.582755E+04, 1.639299E+04, 1.697409E+04, 1.757117E+04, 1.818460E+04,
    1.881470E+04, 1.946184E+04, 2.012638E+04, 2.080867E+04, 2.150909E+04, 2.222801E+04,
    2.296581E+04, 2.372286E+04, 2.449955E+04, 2.529629E+04, 2.611346E+04, 2.695146E+04,
    2.781071E+04, 2.869162E+04, 2.959460E+04, 3.052008E+04, 3.146849E+04, 3.244026E+04,
    3.343582E+04, 3.445564E+04, 3.550014E+04, 3.656980E+04, 3.766507E+04, 3.878642E+04,
    3.993432E+04, 4.110925E+04, 4.231169E+04, 4.354213E+04, 4.480107E+04, 4.608901E+04,
    4.740645E+04, 4.875391E+04, 5.013191E+04, 5.154097E+04, 5.298163E+04, 5.445442E+04,
    5.595988E+04, 5.749856E+04, 5.907103E+04, 6.067783E+04, 6.231955E+04, 6.399675E+04,
    6.571001E+04, 6.745993E+04, 6.924709E+04, 7.107210E+04, 7.293556E+04, 7.483809E+04,
    7.678030E+04, 7.876284E+04, 8.078632E+04, 8.285139E+04, 8.495869E+04, 8.710888E+04,
    8.930263E+04, 9.154060E+04, 9.382346E+04, 9.615189E+04, 9.852659E+04, 1.009483E+05,
    1.034176E+05, 1.059353E+05, 1.085021E+05, 1.111187E+05, 1.137859E+05, 1.165043E+05,
    1.192748E+05, 1.220981E+05, 1.249750E+05, 1.279061E+05, 1.308924E+05, 1.339345E+05,
    1.370334E+05, 1.401897E+05, 1.434043E+05, 1.466780E+05, 1.500116E+05, 1.534060E+05,
    1.568620E+05, 1.603804E+05, 1.639621E+05, 1.676080E+05, 1.713189E+05, 1.750957E+05,
    1.789393E+05, 1.828506E+05, 1.868305E+05, 1.908799E+05, 1.949997E+05, 1.991908E+05,
    2.034543E+05, 2.077909E+05, 2.122018E+05, 2.166877E+05, 2.212498E+05, 2.258889E+05,
    2.306061E+05, 2.354024E+05, 2.402787E+05, 2.452361E+05, 2.502755E+05, 2.553980E+05,
    2.606047E+05, 2.658966E+05, 2.712746E+05, 2.767400E+05, 2.822937E+05, 2.879368E+05,
    2.936704E+05, 2.994957E+05, 3.054136E+05, 3.114253E+05, 3.175319E+05, 3.237345E+05,
    3.300344E+05, 3.364325E+05, 3.429300E+05, 3.495282E+05, 3.562281E+05, 3.630310E+05,
    3.699380E+05, 3.769503E+05, 3.840690E+05, 3.912955E+05, 3.986309E+05, 4.060763E+05,
    4.136332E+05, 4.213026E+05, 4.290858E+05, 4.369840E+05, 4.449986E+05, 4.531307E+05,
    4.613817E+05, 4.697528E+05, 4.782453E+05, 4.868605E+05, 4.955996E+05, 5.044641E+05,
    5.134551E+05, 5.225741E+05, 5.318222E+05, 5.412010E+05, 5.507116E+05, 5.603555E+05,
    5.701339E+05, 5.800483E+05, 5.901000E+05, 6.002904E+05, 6.106208E+05, 6.210926E+05,
    6.317072E+05, 6.424660E+05, 6.533704E+05, 6.644218E+05, 6.756215E+05, 6.869711E+05,
    6.984719E+05, 7.101253E+05, 7.219327E+05, 7.338957E+05, 7.460156E+05, 7.582938E+05,
    7.707319E+05, 7.833312E+05, 7.960932E+05, 8.090193E+05, 8.221111E+05, 8.353700E+05,
    8.487974E+05, 8.623948E+05, 8.761637E+05, 8.901056E+05, 9.042219E+05, 9.185141E+05,
    9.329838E+05, 9.476323E+05, 9.624612E+05, 9.774720E+05, 9.926662E+05, 1.008045E+06,
    1.023611E+06, 1.039364E+06, 1.055307E+06, 1.071440E+06, 1.087766E+06, 1.104286E+06,
    1.121001E+06, 1.137913E+06, 1.155024E+06, 1.172334E+06, 1.189846E+06, 1.207561E+06,
    1.225480E+06, 1.243606E+06, 1.261938E+06, 1.280480E+06, 1.299233E+06, 1.318197E+06,
    1.337375E+06, 1.356767E+06, 1.376377E+06, 1.396204E+06, 1.416251E+06, 1.436519E+06,
    1.457010E+06, 1.477725E+06, 1.498665E+06, 1.519833E+06, 1.541229E+06, 1.562855E+06,
    1.584712E+06, 1.606803E+06, 1.629128E+06, 1.651689E+06, 1.674488E+06, 1.697526E+06,
    1.720804E+06, 1.744324E+06, 1.768088E+06, 1.792097E+06, 1.816351E+06, 1.840854E+06,
    1.865606E+06, 1.890609E+06, 1.915864E+06, 1.941372E+06, 1.967136E+06, 1.993156E+06,
    2.019434E+06,
])

#  ---------------------- M = 1, I = 9 ---------------------------
M = 1
I = 9
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    3.600000E+01, 1.221318E+02, 3.221607E+02, 5.803382E+02, 8.850833E+02, 1.230215E+03,
    1.611562E+03, 2.026065E+03, 2.471377E+03, 2.945671E+03, 3.447538E+03, 3.975930E+03,
    4.530112E+03, 5.109629E+03, 5.714256E+03, 6.343965E+03, 6.998887E+03, 7.679283E+03,
    8.385518E+03, 9.118039E+03, 9.877365E+03, 1.066407E+04, 1.147879E+04, 1.232218E+04,
    1.319497E+04, 1.409792E+04, 1.503180E+04, 1.599746E+04, 1.699576E+04, 1.802760E+04,
    1.909392E+04, 2.019568E+04, 2.133390E+04, 2.250960E+04, 2.372386E+04, 2.497778E+04,
    2.627249E+04, 2.760915E+04, 2.898894E+04, 3.041308E+04, 3.188283E+04, 3.339945E+04,
    3.496424E+04, 3.657853E+04, 3.824366E+04, 3.996102E+04, 4.173201E+04, 4.355806E+04,
    4.544061E+04, 4.738114E+04, 4.938116E+04, 5.144219E+04, 5.356578E+04, 5.575351E+04,
    5.800696E+04, 6.032778E+04, 6.271759E+04, 6.517809E+04, 6.771096E+04, 7.031792E+04,
    7.300073E+04, 7.576115E+04, 7.860099E+04, 8.152207E+04, 8.452624E+04, 8.761536E+04,
    9.079135E+04, 9.405614E+04, 9.741167E+04, 1.008599E+05, 1.044029E+05, 1.080427E+05,
    1.117813E+05, 1.156208E+05, 1.195634E+05, 1.236111E+05, 1.277662E+05, 1.320309E+05,
    1.364074E+05, 1.408980E+05, 1.455049E+05, 1.502304E+05, 1.550770E+05, 1.600471E+05,
    1.651429E+05, 1.703670E+05, 1.757218E+05, 1.812099E+05, 1.868337E+05, 1.925958E+05,
    1.984989E+05, 2.045456E+05, 2.107385E+05, 2.170803E+05, 2.235737E+05, 2.302216E+05,
    2.370267E+05, 2.439918E+05, 2.511198E+05, 2.584135E+05, 2.658760E+05, 2.735102E+05,
    2.813191E+05, 2.893056E+05, 2.974730E+05, 3.058242E+05, 3.143624E+05, 3.230909E+05,
    3.320128E+05, 3.411313E+05, 3.504498E+05, 3.599715E+05, 3.696999E+05, 3.796383E+05,
    3.897902E+05, 4.001590E+05, 4.107482E+05, 4.215615E+05, 4.326023E+05, 4.438744E+05,
    4.553813E+05, 4.671268E+05, 4.791146E+05, 4.913485E+05, 5.038324E+05, 5.165701E+05,
    5.295655E+05, 5.428226E+05, 5.563453E+05, 5.701377E+05, 5.842039E+05, 5.985480E+05,
    6.131740E+05, 6.280863E+05, 6.432891E+05, 6.587866E+05, 6.745832E+05, 6.906833E+05,
    7.070913E+05, 7.238115E+05, 7.408486E+05, 7.582071E+05, 7.758916E+05, 7.939067E+05,
    8.122571E+05, 8.309475E+05, 8.499827E+05, 8.693675E+05, 8.891068E+05, 9.092054E+05,
    9.296684E+05, 9.505008E+05, 9.717075E+05, 9.932938E+05, 1.015265E+06, 1.037625E+06,
    1.060381E+06, 1.083537E+06, 1.107099E+06, 1.131072E+06, 1.155461E+06, 1.180272E+06,
    1.205511E+06, 1.231183E+06, 1.257293E+06, 1.283848E+06, 1.310853E+06, 1.338313E+06,
    1.366236E+06, 1.394625E+06, 1.423488E+06, 1.452831E+06, 1.482659E+06, 1.512978E+06,
    1.543794E+06, 1.575115E+06, 1.606945E+06, 1.639291E+06, 1.672160E+06, 1.705557E+06,
    1.739489E+06, 1.773963E+06, 1.808986E+06, 1.844562E+06, 1.880700E+06, 1.917406E+06,
    1.954686E+06, 1.992547E+06, 2.030996E+06, 2.070041E+06, 2.109686E+06, 2.149941E+06,
    2.190810E+06, 2.232303E+06, 2.274425E+06, 2.317183E+06, 2.360586E+06, 2.404639E+06,
    2.449351E+06, 2.494728E+06, 2.540778E+06, 2.587508E+06, 2.634926E+06, 2.683039E+06,
    2.731855E+06, 2.781380E+06, 2.831623E+06, 2.882592E+06, 2.934293E+06, 2.986735E+06,
    3.039925E+06, 3.093872E+06, 3.148582E+06, 3.204065E+06, 3.260327E+06, 3.317376E+06,
    3.375221E+06, 3.433870E+06, 3.493331E+06, 3.553611E+06, 3.614719E+06, 3.676663E+06,
    3.739451E+06, 3.803092E+06, 3.867593E+06, 3.932963E+06, 3.999210E+06, 4.066343E+06,
    4.134370E+06, 4.203299E+06, 4.273139E+06, 4.343898E+06, 4.415584E+06, 4.488207E+06,
    4.561774E+06, 4.636295E+06, 4.711778E+06, 4.788230E+06, 4.865663E+06, 4.944082E+06,
    5.023498E+06, 5.103920E+06, 5.185355E+06, 5.267812E+06, 5.351301E+06, 5.435830E+06,
    5.521408E+06, 5.608043E+06, 5.695745E+06, 5.784521E+06, 5.874383E+06, 5.965337E+06,
    6.057392E+06, 6.150559E+06, 6.244845E+06, 6.340260E+06, 6.436812E+06, 6.534510E+06,
    6.633364E+06, 6.733382E+06, 6.834573E+06, 6.936945E+06, 7.040509E+06, 7.145273E+06,
    7.251246E+06, 7.358436E+06, 7.466853E+06, 7.576506E+06, 7.687404E+06, 7.799555E+06,
    7.912969E+06, 8.027654E+06, 8.143620E+06, 8.260875E+06, 8.379428E+06, 8.499289E+06,
    8.620465E+06, 8.742967E+06, 8.866803E+06, 8.991981E+06, 9.118511E+06, 9.246401E+06,
    9.375661E+06, 9.506299E+06, 9.638324E+06, 9.771745E+06, 9.906570E+06, 1.004281E+07,
    1.018047E+07, 1.031956E+07, 1.046009E+07, 1.060207E+07, 1.074551E+07, 1.089041E+07,
    1.103679E+07, 1.118465E+07, 1.133400E+07, 1.148485E+07, 1.163721E+07, 1.179108E+07,
    1.194648E+07,
])

#  ---------------------- M = 2, I = 1 ---------------------------
M = 2
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.172300E+00, 1.797937E+01, 3.579152E+01, 5.360487E+01, 7.142001E+01, 8.924629E+01,
    1.071216E+02, 1.251286E+02, 1.433912E+02, 1.620592E+02, 1.812909E+02, 2.012420E+02,
    2.220599E+02, 2.438818E+02, 2.668355E+02, 2.910405E+02, 3.166103E+02, 3.436540E+02,
    3.722775E+02, 4.025853E+02, 4.346809E+02, 4.686683E+02, 5.046524E+02, 5.427394E+02,
    5.830375E+02, 6.256573E+02, 6.707119E+02, 7.183172E+02, 7.685924E+02, 8.216598E+02,
    8.776450E+02, 9.366774E+02, 9.988899E+02, 1.064419E+03, 1.133406E+03, 1.205995E+03,
    1.282335E+03, 1.362579E+03, 1.446883E+03, 1.535408E+03, 1.628322E+03, 1.725793E+03,
    1.827996E+03, 1.935111E+03, 2.047321E+03, 2.164814E+03, 2.287785E+03, 2.416430E+03,
    2.550954E+03, 2.691564E+03, 2.838473E+03, 2.991899E+03, 3.152065E+03, 3.319200E+03,
    3.493538E+03, 3.675317E+03, 3.864782E+03, 4.062182E+03, 4.267774E+03, 4.481819E+03,
    4.704582E+03, 4.936336E+03, 5.177359E+03, 5.427935E+03, 5.688354E+03, 5.958911E+03,
    6.239909E+03, 6.531654E+03, 6.834460E+03, 7.148647E+03, 7.474541E+03, 7.812474E+03,
    8.162786E+03, 8.525820E+03, 8.901928E+03, 9.291468E+03, 9.694803E+03, 1.011231E+04,
    1.054435E+04, 1.099132E+04, 1.145362E+04, 1.193162E+04, 1.242575E+04, 1.293641E+04,
    1.346401E+04, 1.400899E+04, 1.457177E+04, 1.515280E+04, 1.575251E+04, 1.637136E+04,
    1.700982E+04, 1.766834E+04, 1.834740E+04, 1.904748E+04, 1.976907E+04, 2.051267E+04,
    2.127878E+04, 2.206790E+04, 2.288057E+04, 2.371729E+04, 2.457861E+04, 2.546506E+04,
    2.637719E+04, 2.731556E+04, 2.828073E+04, 2.927327E+04, 3.029376E+04, 3.134277E+04,
    3.242091E+04, 3.352878E+04, 3.466698E+04, 3.583614E+04, 3.703686E+04, 3.826979E+04,
    3.953557E+04, 4.083483E+04, 4.216825E+04, 4.353647E+04, 4.494018E+04, 4.638004E+04,
    4.785674E+04, 4.937098E+04, 5.092347E+04, 5.251490E+04, 5.414600E+04, 5.581749E+04,
    5.753010E+04, 5.928458E+04, 6.108167E+04, 6.292213E+04, 6.480672E+04, 6.673621E+04,
    6.871139E+04, 7.073304E+04, 7.280195E+04, 7.491893E+04, 7.708478E+04, 7.930032E+04,
    8.156638E+04, 8.388379E+04, 8.625338E+04, 8.867600E+04, 9.115251E+04, 9.368376E+04,
    9.627062E+04, 9.891397E+04, 1.016147E+05, 1.043737E+05, 1.071918E+05, 1.100700E+05,
    1.130091E+05, 1.160101E+05, 1.190739E+05, 1.222014E+05, 1.253936E+05, 1.286514E+05,
    1.319757E+05, 1.353675E+05, 1.388278E+05, 1.423574E+05, 1.459575E+05, 1.496289E+05,
    1.533726E+05, 1.571897E+05, 1.610810E+05, 1.650477E+05, 1.690906E+05, 1.732109E+05,
    1.774094E+05, 1.816873E+05, 1.860455E+05, 1.904851E+05, 1.950070E+05, 1.996124E+05,
    2.043022E+05, 2.090774E+05, 2.139392E+05, 2.188885E+05, 2.239265E+05, 2.290540E+05,
    2.342723E+05, 2.395823E+05, 2.449850E+05, 2.504817E+05, 2.560732E+05, 2.617607E+05,
    2.675453E+05, 2.734279E+05, 2.794097E+05, 2.854917E+05, 2.916749E+05, 2.979606E+05,
    3.043496E+05, 3.108431E+05, 3.174422E+05, 3.241479E+05, 3.309613E+05, 3.378834E+05,
    3.449153E+05, 3.520581E+05, 3.593129E+05, 3.666806E+05, 3.741624E+05, 3.817594E+05,
    3.894725E+05, 3.973029E+05, 4.052515E+05, 4.133195E+05, 4.215079E+05, 4.298178E+05,
    4.382501E+05, 4.468060E+05, 4.554865E+05, 4.642925E+05, 4.732252E+05, 4.822856E+05,
    4.914747E+05, 5.007935E+05, 5.102430E+05, 5.198243E+05, 5.295383E+05, 5.393861E+05,
    5.493687E+05, 5.594870E+05, 5.697421E+05, 5.801349E+05, 5.906665E+05, 6.013377E+05,
    6.121497E+05, 6.231032E+05, 6.341994E+05, 6.454391E+05, 6.568232E+05, 6.683529E+05,
    6.800289E+05, 6.918522E+05, 7.038237E+05, 7.159444E+05, 7.282151E+05, 7.406368E+05,
    7.532103E+05, 7.659366E+05, 7.788165E+05, 7.918509E+05, 8.050406E+05, 8.183866E+05,
    8.318897E+05, 8.455507E+05, 8.593704E+05, 8.733498E+05, 8.874895E+05,
])

#  ---------------------- M = 2, I = 2 ---------------------------
M = 2
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.344550E+00, 3.595709E+01, 7.157976E+01, 1.072048E+02, 1.428342E+02, 1.784919E+02,
    2.142697E+02, 2.503579E+02, 2.870319E+02, 3.246141E+02, 3.634386E+02, 4.038283E+02,
    4.460854E+02, 4.904894E+02, 5.372995E+02, 5.867589E+02, 6.390984E+02, 6.945407E+02,
    7.533034E+02, 8.156015E+02, 8.816500E+02, 9.516650E+02, 1.025865E+03, 1.104474E+03,
    1.187719E+03, 1.275832E+03, 1.369052E+03, 1.467626E+03, 1.571805E+03, 1.681848E+03,
    1.798023E+03, 1.920605E+03, 2.049877E+03, 2.186128E+03, 2.329660E+03, 2.480780E+03,
    2.639806E+03, 2.807062E+03, 2.982884E+03, 3.167617E+03, 3.361614E+03, 3.565239E+03,
    3.778863E+03, 4.002871E+03, 4.237655E+03, 4.483619E+03, 4.741175E+03, 5.010747E+03,
    5.292769E+03, 5.587687E+03, 5.895957E+03, 6.218045E+03, 6.554429E+03, 6.905599E+03,
    7.272055E+03, 7.654309E+03, 8.052885E+03, 8.468319E+03, 8.901159E+03, 9.351963E+03,
    9.821303E+03, 1.030976E+04, 1.081794E+04, 1.134645E+04, 1.189590E+04, 1.246693E+04,
    1.306020E+04, 1.367635E+04, 1.431607E+04, 1.498004E+04, 1.566896E+04, 1.638355E+04,
    1.712453E+04, 1.789264E+04, 1.868864E+04, 1.951329E+04, 2.036739E+04, 2.125172E+04,
    2.216710E+04, 2.311436E+04, 2.409433E+04, 2.510787E+04, 2.615584E+04, 2.723914E+04,
    2.835866E+04, 2.951531E+04, 3.071003E+04, 3.194376E+04, 3.321745E+04, 3.453208E+04,
    3.588864E+04, 3.728814E+04, 3.873160E+04, 4.022005E+04, 4.175454E+04, 4.333615E+04,
    4.496596E+04, 4.664506E+04, 4.837457E+04, 5.015563E+04, 5.198938E+04, 5.387698E+04,
    5.581962E+04, 5.781850E+04, 5.987482E+04, 6.198981E+04, 6.416473E+04, 6.640083E+04,
    6.869940E+04, 7.106172E+04, 7.348912E+04, 7.598292E+04, 7.854447E+04, 8.117512E+04,
    8.387627E+04, 8.664930E+04, 8.949564E+04, 9.241670E+04, 9.541394E+04, 9.848883E+04,
    1.016428E+05, 1.048775E+05, 1.081942E+05, 1.115947E+05, 1.150803E+05, 1.186528E+05,
    1.223136E+05, 1.260644E+05, 1.299067E+05, 1.338423E+05, 1.378728E+05, 1.419997E+05,
    1.462249E+05, 1.505501E+05, 1.549768E+05, 1.595069E+05, 1.641422E+05, 1.688843E+05,
    1.737350E+05, 1.786963E+05, 1.837698E+05, 1.889574E+05, 1.942609E+05, 1.996823E+05,
    2.052233E+05, 2.108859E+05, 2.166720E+05, 2.225835E+05, 2.286223E+05, 2.347904E+05,
    2.410897E+05, 2.475223E+05, 2.540900E+05, 2.607950E+05, 2.676393E+05, 2.746248E+05,
    2.817536E+05, 2.890278E+05, 2.964495E+05, 3.040207E+05, 3.117435E+05, 3.196200E+05,
    3.276524E+05, 3.358427E+05, 3.441932E+05, 3.527060E+05, 3.613832E+05, 3.702270E+05,
    3.792395E+05, 3.884231E+05, 3.977798E+05, 4.073119E+05, 4.170216E+05, 4.269111E+05,
    4.369827E+05, 4.472386E+05, 4.576810E+05, 4.683123E+05, 4.791346E+05, 4.901502E+05,
    5.013614E+05, 5.127705E+05, 5.243797E+05, 5.361914E+05, 5.482078E+05, 5.604312E+05,
    5.728640E+05, 5.855084E+05, 5.983667E+05, 6.114413E+05, 6.247344E+05, 6.382484E+05,
    6.519855E+05, 6.659481E+05, 6.801385E+05, 6.945590E+05, 7.092119E+05, 7.240996E+05,
    7.392243E+05, 7.545883E+05, 7.701940E+05, 7.860436E+05, 8.021395E+05, 8.184840E+05,
    8.350793E+05, 8.519277E+05, 8.690316E+05, 8.863932E+05, 9.040148E+05, 9.218986E+05,
    9.400470E+05, 9.584622E+05, 9.771464E+05, 9.961019E+05, 1.015331E+06, 1.034836E+06,
    1.054619E+06, 1.074682E+06, 1.095027E+06, 1.115657E+06, 1.136574E+06, 1.157780E+06,
    1.179277E+06, 1.201067E+06, 1.223153E+06, 1.245537E+06, 1.268220E+06, 1.291205E+06,
    1.314494E+06, 1.338089E+06, 1.361993E+06, 1.386206E+06, 1.410732E+06, 1.435573E+06,
    1.460729E+06, 1.486204E+06, 1.512000E+06, 1.538118E+06, 1.564560E+06, 1.591328E+06,
    1.618425E+06, 1.645852E+06, 1.673611E+06, 1.701705E+06, 1.730134E+06, 1.758901E+06,
    1.788007E+06, 1.817455E+06, 1.847247E+06, 1.877383E+06, 1.907867E+06,
])

#  ---------------------- M = 2, I = 3 ---------------------------
M = 2
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.260550E+00, 3.809072E+01, 7.584727E+01, 1.136063E+02, 1.513692E+02, 1.891574E+02,
    2.270552E+02, 2.652444E+02, 3.039959E+02, 3.436341E+02, 3.845016E+02, 4.269345E+02,
    4.712510E+02, 5.177476E+02, 5.667010E+02, 6.183711E+02, 6.730046E+02, 7.308391E+02,
    7.921059E+02, 8.570332E+02, 9.258477E+02, 9.987770E+02, 1.076051E+03, 1.157902E+03,
    1.244567E+03, 1.336290E+03, 1.433318E+03, 1.535906E+03, 1.644315E+03, 1.758814E+03,
    1.879680E+03, 2.007196E+03, 2.141654E+03, 2.283356E+03, 2.432611E+03, 2.589737E+03,
    2.755060E+03, 2.928918E+03, 3.111655E+03, 3.303627E+03, 3.505199E+03, 3.716746E+03,
    3.938651E+03, 4.171310E+03, 4.415128E+03, 4.670521E+03, 4.937913E+03, 5.217743E+03,
    5.510457E+03, 5.816514E+03, 6.136385E+03, 6.470548E+03, 6.819498E+03, 7.183738E+03,
    7.563783E+03, 7.960160E+03, 8.373410E+03, 8.804082E+03, 9.252740E+03, 9.719961E+03,
    1.020633E+04, 1.071245E+04, 1.123894E+04, 1.178641E+04, 1.235551E+04, 1.294689E+04,
    1.356122E+04, 1.419917E+04, 1.486143E+04, 1.554872E+04, 1.626173E+04, 1.700122E+04,
    1.776792E+04, 1.856258E+04, 1.938600E+04, 2.023894E+04, 2.112221E+04, 2.203662E+04,
    2.298300E+04, 2.396219E+04, 2.497505E+04, 2.602245E+04, 2.710526E+04, 2.822438E+04,
    2.938072E+04, 3.057522E+04, 3.180880E+04, 3.308241E+04, 3.439702E+04, 3.575361E+04,
    3.715316E+04, 3.859669E+04, 4.008520E+04, 4.161973E+04, 4.320132E+04, 4.483101E+04,
    4.650988E+04, 4.823901E+04, 5.001948E+04, 5.185240E+04, 5.373887E+04, 5.568001E+04,
    5.767697E+04, 5.973088E+04, 6.184289E+04, 6.401416E+04, 6.624588E+04, 6.853920E+04,
    7.089533E+04, 7.331546E+04, 7.580078E+04, 7.835251E+04, 8.097186E+04, 8.366006E+04,
    8.641834E+04, 8.924792E+04, 9.215004E+04, 9.512594E+04, 9.817687E+04, 1.013041E+05,
    1.045088E+05, 1.077923E+05, 1.111559E+05, 1.146007E+05, 1.181281E+05, 1.217393E+05,
    1.254356E+05, 1.292182E+05, 1.330883E+05, 1.370473E+05, 1.410964E+05, 1.452368E+05,
    1.494697E+05, 1.537965E+05, 1.582183E+05, 1.627364E+05, 1.673520E+05, 1.720663E+05,
    1.768805E+05, 1.817959E+05, 1.868137E+05, 1.919349E+05, 1.971610E+05, 2.024929E+05,
    2.079318E+05, 2.134791E+05, 2.191356E+05, 2.249027E+05, 2.307815E+05, 2.367730E+05,
    2.428784E+05, 2.490987E+05, 2.554351E+05, 2.618886E+05, 2.684603E+05, 2.751512E+05,
    2.819624E+05, 2.888948E+05, 2.959496E+05, 3.031276E+05, 3.104298E+05, 3.178573E+05,
    3.254109E+05, 3.330916E+05, 3.409004E+05, 3.488380E+05, 3.569055E+05, 3.651036E+05,
    3.734333E+05, 3.818953E+05, 3.904905E+05, 3.992197E+05, 4.080837E+05, 4.170833E+05,
    4.262192E+05, 4.354922E+05,
])

#  ---------------------- M = 2, I = 4 ---------------------------
M = 2
I = 4
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.325894E+01, 2.223057E+02, 4.426057E+02, 6.629202E+02, 8.832573E+02, 1.103737E+03,
    1.324841E+03, 1.547611E+03, 1.773607E+03, 2.004700E+03, 2.242871E+03, 2.490065E+03,
    2.748119E+03, 3.018750E+03, 3.303554E+03, 3.604030E+03, 3.921599E+03, 4.257630E+03,
    4.613454E+03, 4.990381E+03, 5.389715E+03, 5.812764E+03, 6.260847E+03, 6.735300E+03,
    7.237487E+03, 7.768800E+03, 8.330662E+03, 8.924535E+03, 9.551919E+03, 1.021436E+04,
    1.091343E+04, 1.165077E+04, 1.242805E+04, 1.324700E+04, 1.410938E+04, 1.501703E+04,
    1.597182E+04, 1.697567E+04, 1.803057E+04, 1.913855E+04, 2.030170E+04, 2.152217E+04,
    2.280215E+04, 2.414392E+04, 2.554979E+04, 2.702215E+04, 2.856342E+04, 3.017611E+04,
    3.186279E+04, 3.362608E+04, 3.546867E+04, 3.739332E+04, 3.940283E+04, 4.150010E+04,
    4.368808E+04, 4.596979E+04, 4.834831E+04, 5.082679E+04, 5.340848E+04, 5.609665E+04,
    5.889468E+04, 6.180600E+04, 6.483413E+04, 6.798265E+04, 7.125521E+04, 7.465556E+04,
    7.818749E+04, 8.185488E+04, 8.566170E+04, 8.961199E+04, 9.370984E+04, 9.795946E+04,
    1.023651E+05, 1.069311E+05, 1.116619E+05, 1.165621E+05, 1.216361E+05, 1.268886E+05,
    1.323244E+05, 1.379484E+05, 1.437653E+05, 1.497802E+05, 1.559982E+05, 1.624244E+05,
    1.690639E+05, 1.759222E+05, 1.830045E+05, 1.903164E+05, 1.978633E+05, 2.056509E+05,
    2.136848E+05, 2.219708E+05, 2.305147E+05, 2.393225E+05, 2.484001E+05, 2.577535E+05,
    2.673889E+05, 2.773125E+05, 2.875304E+05, 2.980491E+05, 3.088749E+05, 3.200142E+05,
    3.314736E+05, 3.432596E+05, 3.553788E+05, 3.678379E+05, 3.806436E+05, 3.938026E+05,
    4.073219E+05, 4.212082E+05, 4.354685E+05, 4.501096E+05, 4.651387E+05, 4.805627E+05,
    4.963887E+05, 5.126236E+05, 5.292748E+05, 5.463492E+05, 5.638540E+05, 5.817965E+05,
    6.001837E+05, 6.190230E+05, 6.383214E+05, 6.580863E+05, 6.783248E+05, 6.990442E+05,
    7.202517E+05, 7.419545E+05, 7.641599E+05, 7.868749E+05, 8.101069E+05, 8.338629E+05,
    8.581502E+05, 8.829758E+05, 9.083470E+05, 9.342706E+05, 9.607539E+05, 9.878038E+05,
    1.015427E+06, 1.043631E+06, 1.072423E+06, 1.101809E+06, 1.131795E+06, 1.162390E+06,
    1.193600E+06, 1.225430E+06, 1.257889E+06, 1.290982E+06, 1.324715E+06, 1.359096E+06,
    1.394131E+06, 1.429826E+06, 1.466187E+06, 1.503220E+06, 1.540932E+06, 1.579328E+06,
    1.618415E+06, 1.658198E+06, 1.698683E+06, 1.739876E+06, 1.781782E+06, 1.824407E+06,
    1.867756E+06, 1.911835E+06, 1.956650E+06, 2.002204E+06, 2.048503E+06, 2.095553E+06,
    2.143358E+06, 2.191923E+06, 2.241253E+06, 2.291353E+06, 2.342226E+06, 2.393878E+06,
    2.446312E+06, 2.499534E+06,
])

#  ---------------------- M = 2, I = 5 ---------------------------
M = 2
I = 5
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    4.521080E+00, 7.618089E+01, 1.516935E+02, 2.272111E+02, 3.027382E+02, 3.783295E+02,
    4.541900E+02, 5.307384E+02, 6.085758E+02, 6.884015E+02, 7.709370E+02, 8.568792E+02,
    9.468802E+02, 1.041544E+03, 1.141433E+03, 1.247075E+03, 1.358972E+03, 1.477611E+03,
    1.603466E+03, 1.737009E+03, 1.878710E+03, 2.029043E+03, 2.188489E+03, 2.357536E+03,
    2.536684E+03, 2.726445E+03, 2.927343E+03, 3.139917E+03, 3.364724E+03, 3.602332E+03,
    3.853330E+03, 4.118322E+03, 4.397931E+03, 4.692798E+03, 5.003582E+03, 5.330962E+03,
    5.675636E+03, 6.038323E+03, 6.419760E+03, 6.820707E+03, 7.241943E+03, 7.684269E+03,
    8.148509E+03, 8.635506E+03, 9.146128E+03, 9.681264E+03, 1.024183E+04, 1.082875E+04,
    1.144299E+04, 1.208554E+04, 1.275739E+04, 1.345959E+04, 1.419317E+04, 1.495924E+04,
    1.575889E+04, 1.659324E+04, 1.746347E+04, 1.837075E+04, 1.931629E+04, 2.030133E+04,
    2.132713E+04, 2.239497E+04, 2.350618E+04, 2.466211E+04, 2.586412E+04, 2.711361E+04,
    2.841201E+04, 2.976079E+04, 3.116142E+04, 3.261543E+04, 3.412435E+04, 3.568975E+04,
    3.731326E+04, 3.899648E+04, 4.074109E+04, 4.254878E+04, 4.442127E+04, 4.636031E+04,
    4.836769E+04, 5.044520E+04, 5.259471E+04, 5.481807E+04, 5.711720E+04, 5.949401E+04,
    6.195048E+04, 6.448859E+04, 6.711037E+04, 6.981786E+04, 7.261315E+04, 7.549834E+04,
    7.847557E+04, 8.154701E+04, 8.471484E+04, 8.798130E+04, 9.134862E+04, 9.481909E+04,
    9.839500E+04, 1.020787E+05, 1.058725E+05, 1.097788E+05, 1.138000E+05, 1.179385E+05,
    1.221969E+05, 1.265774E+05, 1.310827E+05, 1.357152E+05, 1.404776E+05, 1.453722E+05,
    1.504017E+05, 1.555688E+05, 1.608759E+05, 1.663257E+05, 1.719208E+05, 1.776640E+05,
    1.835577E+05, 1.896048E+05, 1.958079E+05, 2.021696E+05, 2.086927E+05, 2.153799E+05,
    2.222339E+05, 2.292574E+05, 2.364532E+05, 2.438239E+05, 2.513722E+05, 2.591010E+05,
    2.670130E+05, 2.751108E+05, 2.833972E+05, 2.918749E+05, 3.005467E+05, 3.094152E+05,
    3.184831E+05, 3.277532E+05, 3.372281E+05, 3.469105E+05, 3.568031E+05, 3.669086E+05,
    3.772295E+05, 3.877685E+05, 3.985282E+05, 4.095113E+05, 4.207202E+05, 4.321576E+05,
    4.438260E+05, 4.557279E+05, 4.678659E+05, 4.802424E+05, 4.928599E+05, 5.057207E+05,
    5.188275E+05, 5.321824E+05, 5.457879E+05, 5.596464E+05, 5.737601E+05, 5.881313E+05,
    6.027624E+05, 6.176554E+05, 6.328126E+05, 6.482363E+05, 6.639284E+05, 6.798912E+05,
    6.961267E+05, 7.126369E+05, 7.294239E+05, 7.464896E+05, 7.638361E+05, 7.814651E+05,
    7.993786E+05, 8.175784E+05, 8.360663E+05, 8.548442E+05, 8.739137E+05, 8.932765E+05,
    9.129343E+05, 9.328888E+05,
])

#  ---------------------- M = 2, I = 6 ---------------------------
M = 2
I = 6
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.651606E+01, 4.445925E+02, 8.851739E+02, 1.325785E+03, 1.766449E+03, 2.207477E+03,
    2.650035E+03, 3.096522E+03, 3.550398E+03, 4.015699E+03, 4.496597E+03, 4.997122E+03,
    5.521046E+03, 6.071862E+03, 6.652812E+03, 7.266940E+03, 7.917143E+03, 8.606215E+03,
    9.336891E+03, 1.011188E+04, 1.093387E+04, 1.180560E+04, 1.272982E+04, 1.370933E+04,
    1.474700E+04, 1.584577E+04, 1.700865E+04, 1.823873E+04, 1.953919E+04, 2.091330E+04,
    2.236443E+04, 2.389604E+04, 2.551171E+04, 2.721511E+04, 2.901001E+04, 3.090030E+04,
    3.288998E+04, 3.498317E+04, 3.718410E+04, 3.949710E+04, 4.192665E+04, 4.447733E+04,
    4.715386E+04, 4.996106E+04, 5.290390E+04, 5.598748E+04, 5.921700E+04, 6.259783E+04,
    6.613544E+04, 6.983547E+04, 7.370367E+04, 7.774594E+04, 8.196832E+04, 8.637699E+04,
    9.097828E+04, 9.577865E+04, 1.007847E+05, 1.060033E+05, 1.114413E+05, 1.171057E+05,
    1.230038E+05, 1.291430E+05, 1.355307E+05, 1.421748E+05, 1.490831E+05, 1.562635E+05,
    1.637242E+05, 1.714737E+05, 1.795203E+05, 1.878728E+05, 1.965400E+05, 2.055309E+05,
    2.148547E+05, 2.245208E+05, 2.345385E+05, 2.449177E+05, 2.556682E+05, 2.668000E+05,
    2.783233E+05, 2.902486E+05, 3.025862E+05, 3.153471E+05, 3.285420E+05, 3.421821E+05,
    3.562785E+05, 3.708427E+05, 3.858862E+05, 4.014209E+05, 4.174586E+05, 4.340114E+05,
    4.510916E+05, 4.687115E+05, 4.868838E+05, 5.056212E+05, 5.249365E+05, 5.448429E+05,
    5.653535E+05, 5.864817E+05, 6.082409E+05, 6.306448E+05, 6.537072E+05, 6.774420E+05,
    7.018632E+05, 7.269850E+05, 7.528217E+05, 7.793877E+05, 8.066975E+05, 8.347658E+05,
    8.636073E+05, 8.932368E+05, 9.236693E+05, 9.549198E+05, 9.870035E+05, 1.019935E+06,
    1.053731E+06, 1.088405E+06, 1.123974E+06, 1.160452E+06, 1.197856E+06, 1.236200E+06,
    1.275501E+06, 1.315774E+06, 1.357034E+06, 1.399297E+06, 1.442580E+06, 1.486897E+06,
    1.532264E+06, 1.578697E+06, 1.626211E+06, 1.674823E+06, 1.724548E+06, 1.775400E+06,
    1.827397E+06, 1.880553E+06, 1.934884E+06, 1.990405E+06, 2.047132E+06, 2.105079E+06,
    2.164262E+06, 2.224697E+06, 2.286397E+06, 2.349379E+06, 2.413656E+06, 2.479244E+06,
    2.546157E+06, 2.614410E+06, 2.684016E+06, 2.754992E+06, 2.827349E+06, 2.901104E+06,
    2.976268E+06, 3.052857E+06, 3.130883E+06, 3.210361E+06, 3.291303E+06, 3.373723E+06,
    3.457633E+06, 3.543047E+06, 3.629976E+06, 3.718434E+06, 3.808433E+06, 3.899985E+06,
    3.993102E+06, 4.087795E+06, 4.184076E+06, 4.281956E+06, 4.381448E+06, 4.482561E+06,
    4.585306E+06, 4.689695E+06, 4.795737E+06, 4.903442E+06, 5.012821E+06, 5.123884E+06,
    5.236639E+06, 5.351097E+06,
])

#  ---------------------- M = 2, I = 7 ---------------------------
M = 2
I = 7
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.250890E+00, 2.020899E+01, 4.025104E+01, 6.029436E+01, 8.033986E+01, 1.003996E+02,
    1.205209E+02, 1.408040E+02, 1.613971E+02, 1.824767E+02, 2.042283E+02, 2.268340E+02,
    2.504658E+02, 2.752848E+02, 3.014409E+02, 3.290757E+02, 3.583237E+02, 3.893145E+02,
    4.221748E+02, 4.570293E+02, 4.940023E+02, 5.332187E+02, 5.748044E+02, 6.188873E+02,
    6.655979E+02, 7.150694E+02, 7.674382E+02, 8.228445E+02, 8.814321E+02, 9.433489E+02,
    1.008747E+03, 1.077783E+03, 1.150618E+03, 1.227417E+03, 1.308352E+03, 1.393596E+03,
    1.483332E+03, 1.577744E+03, 1.677024E+03, 1.781366E+03, 1.890973E+03, 2.006051E+03,
    2.126813E+03, 2.253477E+03, 2.386266E+03, 2.525410E+03, 2.671145E+03, 2.823711E+03,
    2.983356E+03, 3.150333E+03, 3.324902E+03, 3.507329E+03, 3.697886E+03, 3.896851E+03,
    4.104509E+03, 4.321152E+03, 4.547077E+03, 4.782590E+03, 5.028001E+03, 5.283630E+03,
    5.549800E+03, 5.826843E+03, 6.115099E+03, 6.414913E+03, 6.726639E+03, 7.050636E+03,
    7.387272E+03, 7.736920E+03, 8.099964E+03, 8.476792E+03, 8.867801E+03, 9.273394E+03,
    9.693984E+03, 1.012999E+04, 1.058184E+04, 1.104996E+04, 1.153481E+04, 1.203682E+04,
    1.255646E+04, 1.309419E+04, 1.365049E+04, 1.422584E+04, 1.482072E+04, 1.543563E+04,
    1.607109E+04, 1.672760E+04, 1.740569E+04, 1.810588E+04, 1.882871E+04, 1.957473E+04,
    2.034450E+04, 2.113858E+04, 2.195754E+04, 2.280195E+04, 2.367242E+04, 2.456953E+04,
    2.549389E+04, 2.644612E+04, 2.742683E+04, 2.843666E+04, 2.947625E+04, 3.054624E+04,
    3.164730E+04, 3.278009E+04, 3.394529E+04, 3.514357E+04, 3.637564E+04, 3.764218E+04,
    3.894392E+04, 4.028157E+04, 4.165585E+04, 4.306751E+04, 4.451729E+04, 4.600595E+04,
    4.753424E+04, 4.910294E+04, 5.071284E+04, 5.236471E+04, 5.405937E+04, 5.579761E+04,
    5.758027E+04, 5.940815E+04, 6.128210E+04, 6.320296E+04, 6.517158E+04, 6.718883E+04,
    6.925557E+04, 7.137269E+04, 7.354107E+04, 7.576161E+04, 7.803522E+04, 8.036280E+04,
    8.274529E+04, 8.518362E+04, 8.767872E+04, 9.023154E+04, 9.284305E+04, 9.551420E+04,
    9.824598E+04, 1.010394E+05, 1.038953E+05, 1.068149E+05, 1.097991E+05, 1.128489E+05,
    1.159653E+05, 1.191494E+05, 1.224023E+05, 1.257249E+05, 1.291182E+05, 1.325835E+05,
    1.361217E+05, 1.397340E+05, 1.434213E+05, 1.471849E+05, 1.510257E+05, 1.549449E+05,
    1.589437E+05, 1.630231E+05, 1.671842E+05, 1.714282E+05, 1.757562E+05, 1.801694E+05,
    1.846689E+05, 1.892559E+05, 1.939315E+05, 1.986969E+05, 2.035533E+05, 2.085017E+05,
    2.135436E+05, 2.186799E+05, 2.239119E+05, 2.292408E+05, 2.346678E+05, 2.401941E+05,
    2.458208E+05, 2.515493E+05, 2.573807E+05, 2.633163E+05, 2.693572E+05, 2.755047E+05,
    2.817600E+05, 2.881243E+05, 2.945989E+05, 3.011850E+05, 3.078838E+05, 3.146966E+05,
    3.216246E+05, 3.286691E+05, 3.358312E+05, 3.431123E+05, 3.505135E+05, 3.580362E+05,
    3.656815E+05, 3.734508E+05, 3.813451E+05, 3.893659E+05, 3.975143E+05, 4.057916E+05,
    4.141989E+05, 4.227377E+05, 4.314090E+05, 4.402142E+05, 4.491544E+05, 4.582309E+05,
    4.674450E+05, 4.767978E+05, 4.862906E+05, 4.959246E+05, 5.057011E+05, 5.156212E+05,
    5.256861E+05, 5.358972E+05, 5.462555E+05, 5.567623E+05, 5.674188E+05, 5.782262E+05,
    5.891856E+05, 6.002984E+05, 6.115655E+05, 6.229883E+05, 6.345679E+05, 6.463054E+05,
    6.582021E+05, 6.702590E+05, 6.824774E+05, 6.948584E+05, 7.074030E+05, 7.201125E+05,
    7.329880E+05, 7.460305E+05, 7.592412E+05, 7.726213E+05, 7.861717E+05, 7.998937E+05,
    8.137882E+05, 8.278563E+05, 8.420992E+05, 8.565179E+05, 8.711134E+05, 8.858868E+05,
    9.008391E+05, 9.159714E+05, 9.312846E+05, 9.467798E+05, 9.624580E+05, 9.783202E+05,
    9.943673E+05, 1.010600E+06, 1.027020E+06, 1.043628E+06, 1.060425E+06,
])

#  ---------------------- M = 2, I = 8 ---------------------------
M = 2
I = 8
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.391189E+01, 2.356850E+02, 4.693664E+02, 7.030628E+02, 9.367842E+02, 1.170667E+03,
    1.405248E+03, 1.641677E+03, 1.881655E+03, 2.127217E+03, 2.380506E+03, 2.643624E+03,
    2.918557E+03, 3.207165E+03, 3.511177E+03, 3.832224E+03, 4.171854E+03, 4.531559E+03,
    4.912793E+03, 5.316991E+03, 5.745580E+03, 6.199992E+03, 6.681674E+03, 7.192093E+03,
    7.732745E+03, 8.305157E+03, 8.910895E+03, 9.551564E+03, 1.022881E+04, 1.094434E+04,
    1.169988E+04, 1.249723E+04, 1.333824E+04, 1.422480E+04, 1.515886E+04, 1.614244E+04,
    1.717760E+04, 1.826646E+04, 1.941121E+04, 2.061408E+04, 2.187739E+04, 2.320350E+04,
    2.459484E+04, 2.605391E+04, 2.758326E+04, 2.918551E+04, 3.086338E+04, 3.261960E+04,
    3.445702E+04, 3.637852E+04, 3.838709E+04, 4.048575E+04, 4.267762E+04, 4.496589E+04,
    4.735382E+04, 4.984473E+04, 5.244204E+04, 5.514923E+04, 5.796986E+04, 6.090757E+04,
    6.396608E+04, 6.714918E+04, 7.046075E+04, 7.390475E+04, 7.748521E+04, 8.120625E+04,
    8.507207E+04, 8.908696E+04, 9.325528E+04, 9.758148E+04, 1.020701E+05, 1.067258E+05,
    1.115532E+05, 1.165571E+05, 1.217425E+05, 1.271142E+05, 1.326774E+05, 1.384371E+05,
    1.443986E+05, 1.505672E+05, 1.569483E+05, 1.635474E+05, 1.703700E+05, 1.774218E+05,
    1.847086E+05, 1.922361E+05, 2.000103E+05, 2.080372E+05, 2.163229E+05, 2.248735E+05,
    2.336952E+05, 2.427945E+05, 2.521776E+05, 2.618512E+05, 2.718217E+05, 2.820959E+05,
    2.926803E+05, 3.035819E+05, 3.148074E+05, 3.263639E+05, 3.382582E+05, 3.504976E+05,
    3.630890E+05, 3.760398E+05, 3.893571E+05, 4.030483E+05, 4.171208E+05, 4.315819E+05,
    4.464392E+05, 4.617002E+05, 4.773724E+05, 4.934635E+05, 5.099811E+05, 5.269329E+05,
    5.443266E+05, 5.621700E+05, 5.804710E+05, 5.992372E+05, 6.184765E+05, 6.381968E+05,
    6.584059E+05, 6.791119E+05, 7.003224E+05, 7.220455E+05, 7.442890E+05, 7.670610E+05,
    7.903691E+05, 8.142215E+05, 8.386259E+05, 8.635903E+05, 8.891225E+05, 9.152303E+05,
    9.419216E+05, 9.692043E+05, 9.970860E+05, 1.025575E+06, 1.054678E+06, 1.084403E+06,
    1.114758E+06, 1.145751E+06, 1.177388E+06, 1.209678E+06, 1.242628E+06, 1.276246E+06,
    1.310538E+06, 1.345512E+06, 1.381175E+06, 1.417535E+06, 1.454598E+06, 1.492372E+06,
    1.530863E+06, 1.570079E+06, 1.610026E+06, 1.650710E+06, 1.692140E+06, 1.734320E+06,
    1.777258E+06, 1.820960E+06, 1.865433E+06, 1.910682E+06, 1.956714E+06, 2.003535E+06,
    2.051150E+06, 2.099566E+06, 2.148789E+06, 2.198824E+06, 2.249676E+06, 2.301352E+06,
    2.353856E+06, 2.407194E+06, 2.461371E+06, 2.516392E+06, 2.572262E+06, 2.628986E+06,
    2.686569E+06, 2.745015E+06,
])

#  ---------------------- M = 2, I = 9 ---------------------------
M = 2
I = 9
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    4.124397E+01, 6.874652E+02, 1.368916E+03, 2.050410E+03, 2.731976E+03, 3.413998E+03,
    4.098007E+03, 4.787289E+03, 5.486743E+03, 6.202233E+03, 6.939947E+03, 7.705959E+03,
    8.506016E+03, 9.345477E+03, 1.022934E+04, 1.116230E+04, 1.214882E+04, 1.319320E+04,
    1.429960E+04, 1.547216E+04, 1.671497E+04, 1.803216E+04, 1.942786E+04, 2.090629E+04,
    2.247174E+04, 2.412860E+04, 2.588134E+04, 2.773459E+04, 2.969305E+04, 3.176160E+04,
    3.394522E+04, 3.624906E+04, 3.867840E+04, 4.123867E+04, 4.393547E+04, 4.677456E+04,
    4.976184E+04, 5.290340E+04, 5.620549E+04, 5.967453E+04, 6.331712E+04, 6.714003E+04,
    7.115023E+04, 7.535485E+04, 7.976121E+04, 8.437683E+04, 8.920942E+04, 9.426687E+04,
    9.955728E+04, 1.050889E+05, 1.108703E+05, 1.169102E+05, 1.232174E+05, 1.298010E+05,
    1.366705E+05, 1.438352E+05, 1.513051E+05, 1.590899E+05, 1.672000E+05, 1.756458E+05,
    1.844378E+05, 1.935870E+05, 2.031044E+05, 2.130014E+05, 2.232894E+05, 2.339804E+05,
    2.450863E+05, 2.566195E+05, 2.685923E+05, 2.810176E+05, 2.939083E+05, 3.072777E+05,
    3.211394E+05, 3.355069E+05, 3.503944E+05, 3.658161E+05, 3.817865E+05, 3.983203E+05,
    4.154326E+05, 4.331386E+05, 4.514539E+05, 4.703944E+05, 4.899760E+05, 5.102151E+05,
    5.311284E+05, 5.527328E+05, 5.750453E+05, 5.980834E+05, 6.218648E+05, 6.464075E+05,
    6.717298E+05, 6.978501E+05, 7.247874E+05, 7.525606E+05, 7.811893E+05, 8.106929E+05,
    8.410916E+05, 8.724056E+05, 9.046553E+05, 9.378615E+05, 9.720455E+05, 1.007229E+06,
    1.043432E+06, 1.080679E+06, 1.118991E+06, 1.158390E+06, 1.198899E+06, 1.240543E+06,
    1.283343E+06, 1.327324E+06, 1.372511E+06, 1.418926E+06, 1.466595E+06, 1.515544E+06,
    1.565796E+06, 1.617378E+06, 1.670316E+06, 1.724635E+06, 1.780363E+06, 1.837526E+06,
    1.896150E+06, 1.956265E+06, 2.017896E+06, 2.081072E+06, 2.145822E+06, 2.212173E+06,
    2.280156E+06, 2.349798E+06, 2.421130E+06, 2.494182E+06, 2.568982E+06, 2.645562E+06,
    2.723953E+06, 2.804185E+06, 2.886290E+06, 2.970298E+06, 3.056243E+06, 3.144156E+06,
    3.234069E+06, 3.326015E+06, 3.420028E+06, 3.516140E+06, 3.614384E+06, 3.714796E+06,
    3.817408E+06, 3.922256E+06, 4.029373E+06, 4.138795E+06, 4.250556E+06, 4.364693E+06,
    4.481240E+06, 4.600234E+06, 4.721710E+06, 4.845706E+06, 4.972258E+06, 5.101402E+06,
    5.233175E+06, 5.367616E+06, 5.504761E+06, 5.644648E+06, 5.787315E+06, 5.932800E+06,
    6.081142E+06, 6.232379E+06, 6.386550E+06, 6.543694E+06, 6.703849E+06, 6.867056E+06,
    7.033353E+06, 7.202781E+06, 7.375378E+06, 7.551186E+06, 7.730243E+06, 7.912590E+06,
    8.098268E+06, 8.287317E+06, 8.479777E+06, 8.675690E+06, 8.875095E+06, 9.078035E+06,
    9.284549E+06, 9.494679E+06, 9.708467E+06, 9.925953E+06, 1.014718E+07, 1.037219E+07,
    1.060102E+07, 1.083371E+07, 1.107031E+07, 1.131086E+07, 1.155539E+07, 1.180396E+07,
    1.205660E+07, 1.231335E+07, 1.257426E+07, 1.283936E+07, 1.310870E+07, 1.338233E+07,
    1.366027E+07, 1.394258E+07, 1.422929E+07, 1.452045E+07, 1.481610E+07, 1.511628E+07,
    1.542103E+07, 1.573039E+07, 1.604440E+07, 1.636310E+07, 1.668655E+07, 1.701476E+07,
    1.734779E+07, 1.768569E+07, 1.802847E+07, 1.837620E+07, 1.872890E+07, 1.908663E+07,
    1.944941E+07, 1.981729E+07, 2.019030E+07, 2.056850E+07, 2.095191E+07, 2.134058E+07,
    2.173454E+07, 2.213383E+07, 2.253850E+07, 2.294858E+07, 2.336411E+07, 2.378512E+07,
    2.421166E+07, 2.464377E+07, 2.508147E+07, 2.552481E+07, 2.597382E+07, 2.642854E+07,
    2.688901E+07, 2.735527E+07, 2.782734E+07, 2.830527E+07, 2.878909E+07, 2.927883E+07,
    2.977453E+07, 3.027623E+07, 3.078396E+07, 3.129774E+07, 3.181763E+07, 3.234365E+07,
    3.287583E+07, 3.341421E+07, 3.395881E+07, 3.450968E+07, 3.506684E+07,
])

#  ---------------------- M = 2, I = 10 ---------------------------
M = 2
I = 10
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.501700E+00, 4.041606E+01, 8.049824E+01, 1.205830E+02, 1.606730E+02, 2.007995E+02,
    2.410771E+02, 2.817371E+02, 3.231082E+02, 3.655703E+02, 4.095143E+02, 4.553170E+02,
    5.033312E+02, 5.538845E+02, 6.072820E+02, 6.638110E+02, 7.237459E+02, 7.873522E+02,
    8.548902E+02, 9.266178E+02, 1.002793E+03, 1.083677E+03, 1.169533E+03, 1.260630E+03,
    1.357243E+03, 1.459653E+03, 1.568150E+03, 1.683030E+03, 1.804598E+03, 1.933171E+03,
    2.069070E+03, 2.212630E+03, 2.364195E+03, 2.524117E+03, 2.692761E+03, 2.870501E+03,
    3.057723E+03, 3.254823E+03, 3.462209E+03, 3.680301E+03, 3.909530E+03, 4.150338E+03,
    4.403179E+03, 4.668522E+03, 4.946845E+03, 5.238639E+03, 5.544410E+03, 5.864675E+03,
    6.199964E+03, 6.550821E+03, 6.917802E+03, 7.301478E+03, 7.702433E+03, 8.121265E+03,
    8.558585E+03, 9.015020E+03, 9.491209E+03, 9.987807E+03, 1.050548E+04, 1.104492E+04,
    1.160682E+04, 1.219189E+04, 1.280087E+04, 1.343449E+04, 1.409352E+04, 1.477873E+04,
    1.549091E+04, 1.623087E+04, 1.699943E+04, 1.779742E+04, 1.862571E+04, 1.948515E+04,
    2.037665E+04, 2.130110E+04, 2.225942E+04, 2.325254E+04, 2.428143E+04, 2.534705E+04,
    2.645038E+04, 2.759244E+04, 2.877424E+04, 2.999682E+04, 3.126123E+04, 3.256856E+04,
    3.391989E+04, 3.531632E+04, 3.675900E+04, 3.824905E+04, 3.978765E+04, 4.137597E+04,
    4.301520E+04, 4.470658E+04, 4.645132E+04, 4.825069E+04, 5.010596E+04, 5.201841E+04,
    5.398935E+04, 5.602011E+04, 5.811204E+04, 6.026650E+04, 6.248487E+04, 6.476855E+04,
    6.711897E+04, 6.953757E+04, 7.202580E+04, 7.458515E+04, 7.721711E+04, 7.992319E+04,
    8.270494E+04, 8.556390E+04, 8.850166E+04, 9.151980E+04, 9.461994E+04, 9.780372E+04,
    1.010728E+05, 1.044288E+05, 1.078734E+05, 1.114085E+05, 1.150356E+05, 1.187565E+05,
    1.225730E+05, 1.264870E+05, 1.305001E+05, 1.346143E+05, 1.388313E+05, 1.431531E+05,
    1.475815E+05, 1.521185E+05, 1.567659E+05, 1.615257E+05, 1.663998E+05, 1.713903E+05,
    1.764992E+05, 1.817284E+05, 1.870800E+05, 1.925560E+05, 1.981586E+05, 2.038898E+05,
    2.097518E+05, 2.157466E+05, 2.218764E+05, 2.281434E+05, 2.345498E+05, 2.410978E+05,
    2.477895E+05, 2.546272E+05, 2.616132E+05, 2.687498E+05, 2.760392E+05, 2.834837E+05,
    2.910857E+05, 2.988474E+05, 3.067713E+05, 3.148597E+05, 3.231150E+05, 3.315396E+05,
    3.401358E+05, 3.489062E+05, 3.578531E+05, 3.669790E+05, 3.762864E+05, 3.857777E+05,
    3.954555E+05, 4.053222E+05, 4.153804E+05, 4.256326E+05, 4.360813E+05, 4.467291E+05,
    4.575786E+05, 4.686322E+05, 4.798927E+05, 4.913626E+05, 5.030444E+05, 5.149409E+05,
    5.270546E+05, 5.393881E+05, 5.519441E+05, 5.647252E+05, 5.777342E+05, 5.909735E+05,
    6.044459E+05, 6.181541E+05, 6.321007E+05, 6.462885E+05, 6.607200E+05, 6.753979E+05,
    6.903251E+05, 7.055041E+05, 7.209376E+05, 7.366284E+05, 7.525791E+05, 7.687924E+05,
    7.852711E+05, 8.020178E+05, 8.190353E+05, 8.363262E+05, 8.538932E+05, 8.717391E+05,
    8.898665E+05, 9.082781E+05, 9.269767E+05, 9.459648E+05, 9.652453E+05, 9.848207E+05,
    1.004694E+06, 1.024867E+06, 1.045343E+06, 1.066125E+06, 1.087216E+06, 1.108617E+06,
    1.130332E+06, 1.152363E+06, 1.174712E+06, 1.197384E+06, 1.220379E+06, 1.243701E+06,
    1.267352E+06, 1.291335E+06, 1.315653E+06, 1.340307E+06, 1.365301E+06, 1.390637E+06,
    1.416318E+06, 1.442345E+06, 1.468723E+06, 1.495452E+06, 1.522536E+06, 1.549977E+06,
    1.577777E+06, 1.605939E+06, 1.634466E+06, 1.663359E+06, 1.692621E+06, 1.722255E+06,
    1.752263E+06, 1.782647E+06, 1.813409E+06, 1.844552E+06, 1.876079E+06, 1.907990E+06,
    1.940290E+06, 1.972979E+06, 2.006060E+06, 2.039536E+06, 2.073408E+06, 2.107678E+06,
    2.142350E+06, 2.177424E+06, 2.212904E+06, 2.248791E+06, 2.285086E+06,
])

#  --------------- CO2 838: M = 2, I = 0 ALIAS-----------------
TIPS_2017_ISOT_HASH[(M,0)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,0)] = TIPS_2017_ISOQ_HASH[(M,I)]

#  ---------------------- M = 2, I = 11 ---------------------------
M = 2
I = 11
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.782335E+01, 4.713557E+02, 9.387029E+02, 1.406080E+03, 1.873518E+03, 2.341368E+03,
    2.810932E+03, 3.284859E+03, 3.766925E+03, 4.261513E+03, 4.773136E+03, 5.306147E+03,
    5.864623E+03, 6.452344E+03, 7.072828E+03, 7.729387E+03, 8.425175E+03, 9.163245E+03,
    9.946589E+03, 1.077817E+04, 1.166095E+04, 1.259791E+04, 1.359209E+04, 1.464656E+04,
    1.576448E+04, 1.694906E+04, 1.820363E+04, 1.953159E+04, 2.093643E+04, 2.242176E+04,
    2.399128E+04, 2.564882E+04, 2.739831E+04, 2.924378E+04, 3.118942E+04, 3.323950E+04,
    3.539843E+04, 3.767076E+04, 4.006115E+04, 4.257440E+04, 4.521543E+04, 4.798931E+04,
    5.090124E+04, 5.395655E+04, 5.716074E+04, 6.051942E+04, 6.403836E+04, 6.772347E+04,
    7.158083E+04, 7.561663E+04, 7.983725E+04, 8.424921E+04, 8.885918E+04, 9.367400E+04,
    9.870066E+04, 1.039463E+05, 1.094183E+05, 1.151240E+05, 1.210712E+05, 1.272677E+05,
    1.337214E+05, 1.404405E+05, 1.474333E+05, 1.547084E+05, 1.622744E+05, 1.701402E+05,
    1.783147E+05, 1.868073E+05, 1.956274E+05, 2.047844E+05, 2.142882E+05, 2.241486E+05,
    2.343760E+05, 2.449804E+05, 2.559725E+05, 2.673630E+05, 2.791627E+05, 2.913827E+05,
    3.040342E+05, 3.171288E+05, 3.306779E+05, 3.446935E+05, 3.591876E+05, 3.741723E+05,
    3.896601E+05, 4.056634E+05, 4.221951E+05, 4.392682E+05, 4.568957E+05, 4.750909E+05,
    4.938673E+05, 5.132387E+05, 5.332187E+05, 5.538215E+05, 5.750612E+05, 5.969522E+05,
    6.195090E+05, 6.427462E+05, 6.666788E+05, 6.913216E+05, 7.166898E+05, 7.427988E+05,
    7.696639E+05, 7.973007E+05, 8.257250E+05, 8.549525E+05, 8.849992E+05, 9.158811E+05,
    9.476145E+05, 9.802157E+05, 1.013701E+06, 1.048087E+06, 1.083390E+06, 1.119627E+06,
    1.156815E+06, 1.194970E+06, 1.234110E+06, 1.274250E+06, 1.315409E+06, 1.357604E+06,
    1.400851E+06, 1.445167E+06, 1.490570E+06, 1.537077E+06, 1.584704E+06, 1.633470E+06,
    1.683392E+06, 1.734485E+06, 1.786769E+06, 1.840259E+06, 1.894973E+06, 1.950928E+06,
    2.008141E+06, 2.066630E+06, 2.126410E+06, 2.187498E+06, 2.249913E+06, 2.313669E+06,
    2.378785E+06, 2.445276E+06, 2.513159E+06, 2.582450E+06, 2.653165E+06, 2.725322E+06,
    2.798935E+06, 2.874020E+06, 2.950594E+06, 3.028672E+06, 3.108269E+06, 3.189401E+06,
    3.272083E+06, 3.356330E+06, 3.442157E+06, 3.529579E+06, 3.618610E+06, 3.709264E+06,
    3.801557E+06, 3.895501E+06, 3.991111E+06, 4.088400E+06, 4.187382E+06, 4.288070E+06,
    4.390478E+06, 4.494617E+06, 4.600501E+06, 4.708142E+06, 4.817553E+06, 4.928745E+06,
    5.041730E+06, 5.156520E+06, 5.273126E+06, 5.391560E+06, 5.511832E+06, 5.633953E+06,
    5.757934E+06, 5.883784E+06,
])

#  ---------------------- M = 2, I = 12 ---------------------------
M = 2
I = 12
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    8.248682E+01, 1.374885E+03, 2.737741E+03, 4.100686E+03, 5.463802E+03, 6.828082E+03,
    8.197234E+03, 9.578835E+03, 1.098375E+04, 1.242462E+04, 1.391448E+04, 1.546591E+04,
    1.709070E+04, 1.879977E+04, 2.060326E+04, 2.251071E+04, 2.453122E+04, 2.667355E+04,
    2.894631E+04, 3.135799E+04, 3.391712E+04, 3.663224E+04, 3.951206E+04, 4.256541E+04,
    4.580131E+04, 4.922903E+04, 5.285804E+04, 5.669811E+04, 6.075926E+04, 6.505180E+04,
    6.958637E+04, 7.437390E+04, 7.942564E+04, 8.475318E+04, 9.036846E+04, 9.628373E+04,
    1.025116E+05, 1.090652E+05, 1.159576E+05, 1.232028E+05, 1.308148E+05, 1.388081E+05,
    1.471975E+05, 1.559983E+05, 1.652263E+05, 1.748975E+05, 1.850284E+05, 1.956358E+05,
    2.067372E+05, 2.183504E+05, 2.304934E+05, 2.431851E+05, 2.564443E+05, 2.702908E+05,
    2.847445E+05, 2.998258E+05, 3.155556E+05, 3.319555E+05, 3.490472E+05, 3.668531E+05,
    3.853960E+05, 4.046992E+05, 4.247867E+05, 4.456827E+05, 4.674120E+05, 4.900000E+05,
    5.134725E+05, 5.378560E+05, 5.631772E+05, 5.894637E+05, 6.167433E+05, 6.450447E+05,
    6.743967E+05, 7.048290E+05, 7.363717E+05, 7.690554E+05, 8.029115E+05, 8.379716E+05,
    8.742681E+05, 9.118340E+05, 9.507027E+05, 9.909083E+05, 1.032485E+06, 1.075469E+06,
    1.119896E+06, 1.165801E+06, 1.213222E+06, 1.262197E+06, 1.312763E+06, 1.364960E+06,
    1.418826E+06, 1.474402E+06, 1.531729E+06, 1.590847E+06, 1.651799E+06, 1.714626E+06,
    1.779372E+06, 1.846081E+06, 1.914797E+06, 1.985564E+06, 2.058429E+06, 2.133437E+06,
    2.210636E+06, 2.290073E+06, 2.371795E+06, 2.455852E+06, 2.542294E+06, 2.631169E+06,
    2.722529E+06, 2.816425E+06, 2.912910E+06, 3.012035E+06, 3.113855E+06, 3.218422E+06,
    3.325792E+06, 3.436020E+06, 3.549161E+06, 3.665273E+06, 3.784412E+06, 3.906637E+06,
    4.032006E+06, 4.160578E+06, 4.292413E+06, 4.427572E+06, 4.566115E+06, 4.708105E+06,
    4.853604E+06, 5.002674E+06, 5.155381E+06, 5.311788E+06, 5.471961E+06, 5.635964E+06,
    5.803865E+06, 5.975731E+06, 6.151628E+06, 6.331626E+06, 6.515793E+06, 6.704199E+06,
    6.896914E+06, 7.094009E+06, 7.295554E+06, 7.501623E+06, 7.712287E+06, 7.927620E+06,
    8.147695E+06, 8.372588E+06, 8.602372E+06, 8.837123E+06, 9.076918E+06, 9.321833E+06,
    9.571945E+06, 9.827332E+06, 1.008807E+07, 1.035424E+07, 1.062593E+07, 1.090320E+07,
    1.118615E+07, 1.147484E+07, 1.176937E+07, 1.206982E+07, 1.237626E+07, 1.268879E+07,
    1.300747E+07, 1.333241E+07, 1.366367E+07, 1.400135E+07, 1.434553E+07, 1.469629E+07,
    1.505372E+07, 1.541791E+07, 1.578894E+07, 1.616690E+07, 1.655188E+07, 1.694395E+07,
    1.734322E+07, 1.774977E+07, 1.816368E+07, 1.858505E+07, 1.901396E+07, 1.945050E+07,
    1.989476E+07, 2.034683E+07, 2.080679E+07, 2.127475E+07, 2.175079E+07, 2.223499E+07,
    2.272745E+07, 2.322825E+07, 2.373750E+07, 2.425527E+07, 2.478166E+07, 2.531676E+07,
    2.586066E+07, 2.641344E+07, 2.697521E+07, 2.754605E+07, 2.812604E+07, 2.871529E+07,
    2.931387E+07, 2.992189E+07, 3.053943E+07, 3.116658E+07, 3.180343E+07, 3.245007E+07,
    3.310659E+07, 3.377308E+07, 3.444963E+07, 3.513633E+07, 3.583326E+07, 3.654053E+07,
    3.725820E+07, 3.798638E+07, 3.872515E+07, 3.947459E+07, 4.023480E+07, 4.100587E+07,
    4.178787E+07, 4.258090E+07, 4.338505E+07, 4.420039E+07, 4.502701E+07, 4.586501E+07,
    4.671445E+07, 4.757544E+07, 4.844805E+07, 4.933236E+07, 5.022846E+07, 5.113643E+07,
    5.205636E+07, 5.298832E+07, 5.393239E+07, 5.488867E+07, 5.585722E+07, 5.683812E+07,
    5.783147E+07, 5.883732E+07, 5.985577E+07, 6.088689E+07, 6.193076E+07, 6.298745E+07,
    6.405704E+07, 6.513961E+07, 6.623523E+07, 6.734397E+07, 6.846590E+07, 6.960111E+07,
    7.074966E+07, 7.191162E+07, 7.308707E+07, 7.427607E+07, 7.547870E+07,
])

#  ---------------------- M = 2, I = 13 ---------------------------
M = 2
I = 13
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.172250E+00, 1.797782E+01, 3.578844E+01, 5.360026E+01, 7.141461E+01, 8.924655E+01,
    1.071502E+02, 1.252330E+02, 1.436455E+02, 1.625593E+02, 1.821490E+02, 2.025809E+02,
    2.240094E+02, 2.465762E+02, 2.704128E+02, 2.956425E+02, 3.223827E+02, 3.507470E+02,
    3.808470E+02, 4.127935E+02, 4.466974E+02, 4.826712E+02, 5.208288E+02, 5.612866E+02,
    6.041637E+02, 6.495824E+02, 6.976684E+02, 7.485507E+02, 8.023624E+02, 8.592405E+02,
    9.193259E+02, 9.827640E+02, 1.049704E+03, 1.120301E+03, 1.194713E+03, 1.273103E+03,
    1.355638E+03, 1.442493E+03, 1.533844E+03, 1.629875E+03, 1.730772E+03, 1.836729E+03,
    1.947943E+03, 2.064618E+03, 2.186963E+03, 2.315191E+03, 2.449522E+03, 2.590182E+03,
    2.737401E+03, 2.891416E+03, 3.052469E+03, 3.220808E+03, 3.396688E+03, 3.580369E+03,
    3.772118E+03, 3.972206E+03, 4.180913E+03, 4.398524E+03, 4.625331E+03, 4.861631E+03,
    5.107729E+03, 5.363936E+03, 5.630570E+03, 5.907955E+03, 6.196423E+03, 6.496312E+03,
    6.807968E+03, 7.131742E+03, 7.467993E+03, 7.817088E+03, 8.179401E+03, 8.555312E+03,
    8.945209E+03, 9.349488E+03, 9.768552E+03, 1.020281E+04, 1.065268E+04, 1.111859E+04,
    1.160098E+04, 1.210027E+04, 1.261693E+04, 1.315140E+04, 1.370416E+04, 1.427567E+04,
    1.486641E+04, 1.547688E+04, 1.610757E+04, 1.675898E+04, 1.743163E+04, 1.812603E+04,
    1.884273E+04, 1.958224E+04, 2.034513E+04, 2.113194E+04, 2.194324E+04, 2.277960E+04,
    2.364159E+04, 2.452981E+04, 2.544485E+04, 2.638733E+04, 2.735784E+04, 2.835702E+04,
    2.938550E+04, 3.044392E+04, 3.153292E+04, 3.265317E+04, 3.380534E+04, 3.499009E+04,
    3.620811E+04, 3.746011E+04, 3.874677E+04, 4.006882E+04, 4.142697E+04, 4.282196E+04,
    4.425451E+04, 4.572539E+04, 4.723535E+04, 4.878515E+04, 5.037557E+04, 5.200740E+04,
    5.368142E+04, 5.539845E+04, 5.715929E+04, 5.896477E+04, 6.081571E+04, 6.271296E+04,
    6.465735E+04, 6.664976E+04, 6.869105E+04, 7.078209E+04, 7.292377E+04, 7.511697E+04,
    7.736262E+04, 7.966160E+04, 8.201486E+04, 8.442330E+04, 8.688789E+04, 8.940955E+04,
    9.198924E+04, 9.462794E+04, 9.732660E+04, 1.000862E+05, 1.029078E+05, 1.057923E+05,
    1.087407E+05, 1.117541E+05, 1.148335E+05, 1.179799E+05, 1.211944E+05, 1.244779E+05,
    1.278316E+05, 1.312566E+05, 1.347538E+05, 1.383243E+05, 1.419693E+05, 1.456898E+05,
    1.494870E+05, 1.533619E+05, 1.573156E+05, 1.613494E+05, 1.654642E+05, 1.696612E+05,
    1.739416E+05, 1.783064E+05, 1.827570E+05, 1.872943E+05, 1.919196E+05, 1.966340E+05,
    2.014388E+05, 2.063350E+05, 2.113239E+05, 2.164066E+05, 2.215843E+05, 2.268583E+05,
    2.322298E+05, 2.376998E+05, 2.432697E+05, 2.489407E+05, 2.547139E+05, 2.605906E+05,
    2.665720E+05, 2.726593E+05, 2.788538E+05, 2.851567E+05, 2.915692E+05, 2.980925E+05,
    3.047279E+05, 3.114766E+05, 3.183399E+05, 3.253190E+05, 3.324151E+05, 3.396295E+05,
    3.469634E+05, 3.544181E+05, 3.619948E+05, 3.696947E+05, 3.775191E+05, 3.854693E+05,
    3.935464E+05, 4.017517E+05, 4.100865E+05, 4.185520E+05, 4.271494E+05, 4.358800E+05,
    4.447450E+05, 4.537456E+05, 4.628831E+05, 4.721587E+05, 4.815736E+05, 4.911291E+05,
    5.008263E+05, 5.106664E+05, 5.206508E+05, 5.307806E+05, 5.410569E+05, 5.514811E+05,
    5.620543E+05, 5.727776E+05, 5.836524E+05, 5.946797E+05, 6.058608E+05, 6.171968E+05,
    6.286889E+05, 6.403382E+05, 6.521460E+05, 6.641133E+05, 6.762413E+05, 6.885312E+05,
    7.009841E+05, 7.136011E+05, 7.263834E+05, 7.393320E+05, 7.524481E+05, 7.657327E+05,
    7.791870E+05, 7.928120E+05, 8.066089E+05, 8.205787E+05, 8.347224E+05, 8.490412E+05,
    8.635361E+05, 8.782080E+05, 8.930581E+05, 9.080874E+05, 9.232969E+05, 9.386875E+05,
    9.542604E+05, 9.700165E+05, 9.859567E+05, 1.002082E+06, 1.018394E+06,
])

#  ---------------------- M = 3, I = 1 ---------------------------
M = 3
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    7.847400E-01, 5.870075E+01, 1.653093E+02, 3.033348E+02, 4.668337E+02, 6.523999E+02,
    8.578395E+02, 1.081788E+03, 1.323572E+03, 1.583129E+03, 1.860885E+03, 2.157688E+03,
    2.474797E+03, 2.813601E+03, 3.175846E+03, 3.563351E+03, 3.978042E+03, 4.422045E+03,
    4.897528E+03, 5.406676E+03, 5.951807E+03, 6.535268E+03, 7.159589E+03, 7.827197E+03,
    8.540615E+03, 9.302558E+03, 1.011573E+04, 1.098290E+04, 1.190686E+04, 1.289056E+04,
    1.393708E+04, 1.504939E+04, 1.623081E+04, 1.748441E+04, 1.881364E+04, 2.022192E+04,
    2.171273E+04, 2.328963E+04, 2.495636E+04, 2.671676E+04, 2.857462E+04, 3.053392E+04,
    3.259874E+04, 3.477325E+04, 3.706184E+04, 3.946872E+04, 4.199829E+04, 4.465523E+04,
    4.744417E+04, 5.037000E+04, 5.343750E+04, 5.665155E+04, 6.001736E+04, 6.354022E+04,
    6.722531E+04, 7.107790E+04, 7.510385E+04, 7.930836E+04, 8.369781E+04, 8.827733E+04,
    9.305350E+04, 9.803196E+04, 1.032190E+05, 1.086212E+05, 1.142448E+05, 1.200965E+05,
    1.261830E+05, 1.325109E+05, 1.390870E+05, 1.459185E+05, 1.530127E+05, 1.603767E+05,
    1.680182E+05, 1.759443E+05, 1.841627E+05, 1.926817E+05, 2.015087E+05, 2.106519E+05,
    2.201195E+05, 2.299199E+05, 2.400610E+05, 2.505524E+05, 2.614014E+05, 2.726184E+05,
    2.842108E+05, 2.961892E+05, 3.085615E+05, 3.213380E+05, 3.345278E+05, 3.481407E+05,
    3.621862E+05, 3.766745E+05, 3.916156E+05, 4.070192E+05, 4.228968E+05, 4.392584E+05,
    4.561139E+05, 4.734750E+05, 4.913523E+05, 5.097566E+05, 5.286996E+05, 5.481927E+05,
    5.682468E+05, 5.888737E+05, 6.100856E+05, 6.318949E+05, 6.543128E+05, 6.773516E+05,
    7.010245E+05, 7.253433E+05, 7.503210E+05, 7.759709E+05, 8.023056E+05, 8.293382E+05,
    8.570820E+05, 8.855514E+05, 9.147587E+05, 9.447191E+05, 9.754458E+05, 1.006954E+06,
    1.039256E+06, 1.072368E+06, 1.106304E+06, 1.141080E+06, 1.176708E+06, 1.213207E+06,
    1.250589E+06, 1.288871E+06, 1.328070E+06, 1.368200E+06, 1.409276E+06, 1.451317E+06,
    1.494338E+06, 1.538354E+06, 1.583385E+06, 1.629444E+06, 1.676552E+06, 1.724723E+06,
    1.773976E+06, 1.824329E+06, 1.875798E+06, 1.928402E+06, 1.982159E+06, 2.037088E+06,
    2.093206E+06, 2.150533E+06, 2.209086E+06, 2.268887E+06, 2.329953E+06, 2.392303E+06,
    2.455959E+06, 2.520938E+06, 2.587261E+06, 2.654949E+06, 2.724021E+06, 2.794498E+06,
    2.866401E+06, 2.939752E+06, 3.014570E+06, 3.090877E+06, 3.168695E+06, 3.248044E+06,
    3.328948E+06, 3.411429E+06, 3.495507E+06, 3.581206E+06, 3.668550E+06, 3.757559E+06,
    3.848259E+06, 3.940671E+06, 4.034818E+06, 4.130727E+06, 4.228419E+06, 4.327918E+06,
    4.429248E+06, 4.532437E+06,
])

#  ---------------------- M = 3, I = 2 ---------------------------
M = 3
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.659650E+00, 1.252867E+02, 3.529131E+02, 6.476329E+02, 9.967517E+02, 1.393009E+03,
    1.831769E+03, 2.310206E+03, 2.827051E+03, 3.382309E+03, 3.977121E+03, 4.613505E+03,
    5.294212E+03, 6.022501E+03, 6.802194E+03, 7.637375E+03, 8.532274E+03, 9.491628E+03,
    1.052016E+04, 1.162272E+04, 1.280444E+04, 1.407051E+04, 1.542641E+04, 1.687762E+04,
    1.842974E+04, 2.008860E+04, 2.186018E+04, 2.375072E+04, 2.576662E+04, 2.791400E+04,
    3.019968E+04, 3.263083E+04, 3.521391E+04, 3.795663E+04, 4.086590E+04, 4.394976E+04,
    4.721594E+04, 5.067202E+04, 5.432657E+04, 5.818808E+04, 6.226493E+04, 6.656593E+04,
    7.110012E+04, 7.587705E+04, 8.090577E+04, 8.619627E+04, 9.175872E+04, 9.760273E+04,
    1.037389E+05, 1.101777E+05, 1.169305E+05, 1.240078E+05, 1.314214E+05, 1.391823E+05,
    1.473028E+05, 1.557947E+05, 1.646703E+05, 1.739424E+05, 1.836239E+05, 1.937268E+05,
    2.042656E+05, 2.152535E+05, 2.267043E+05, 2.386323E+05, 2.510512E+05, 2.639756E+05,
    2.774213E+05, 2.914026E+05, 3.059358E+05, 3.210359E+05, 3.367181E+05, 3.530002E+05,
    3.698980E+05, 3.874283E+05, 4.056074E+05, 4.244543E+05, 4.439858E+05, 4.642198E+05,
    4.851738E+05, 5.068680E+05, 5.293193E+05, 5.525478E+05, 5.765737E+05, 6.014149E+05,
    6.270935E+05, 6.536272E+05, 6.810391E+05, 7.093488E+05, 7.385776E+05, 7.687471E+05,
    7.998788E+05, 8.319959E+05, 8.651198E+05, 8.992744E+05, 9.344815E+05, 9.707647E+05,
    1.008148E+06, 1.046657E+06, 1.086313E+06, 1.127143E+06, 1.169172E+06, 1.212425E+06,
    1.256927E+06, 1.302705E+06, 1.349784E+06, 1.398192E+06, 1.447956E+06, 1.499104E+06,
    1.551662E+06, 1.605658E+06, 1.661122E+06, 1.718084E+06, 1.776569E+06, 1.836610E+06,
    1.898235E+06, 1.961475E+06, 2.026362E+06, 2.092923E+06, 2.161193E+06, 2.231202E+06,
    2.302981E+06, 2.376565E+06, 2.451985E+06, 2.529274E+06, 2.608467E+06, 2.689597E+06,
    2.772695E+06, 2.857802E+06, 2.944948E+06, 3.034171E+06, 3.125505E+06, 3.218987E+06,
    3.314654E+06, 3.412542E+06, 3.512687E+06, 3.615129E+06, 3.719906E+06, 3.827055E+06,
    3.936617E+06, 4.048628E+06, 4.163130E+06, 4.280164E+06, 4.399769E+06, 4.521984E+06,
    4.646853E+06, 4.774418E+06, 4.904720E+06, 5.037801E+06, 5.173705E+06, 5.312474E+06,
    5.454153E+06, 5.598785E+06, 5.746414E+06, 5.897090E+06, 6.050851E+06, 6.207749E+06,
    6.367826E+06, 6.531132E+06, 6.697714E+06, 6.867614E+06, 7.040889E+06, 7.217578E+06,
    7.397739E+06, 7.581418E+06, 7.768659E+06, 7.959519E+06, 8.154048E+06, 8.352294E+06,
    8.554311E+06, 8.760149E+06, 8.969865E+06, 9.183502E+06, 9.401127E+06, 9.622785E+06,
    9.848527E+06, 1.007842E+07,
])

#  ---------------------- M = 3, I = 3 ---------------------------
M = 3
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    8.185900E-01, 6.125610E+01, 1.725127E+02, 3.165570E+02, 4.871861E+02, 6.808519E+02,
    8.952625E+02, 1.129039E+03, 1.381537E+03, 1.652726E+03, 1.943193E+03, 2.253943E+03,
    2.586326E+03, 2.942026E+03, 3.322888E+03, 3.730960E+03, 4.168424E+03, 4.637562E+03,
    5.140769E+03, 5.680414E+03, 6.259100E+03, 6.879368E+03, 7.543867E+03, 8.255453E+03,
    9.016771E+03, 9.830870E+03, 1.070057E+04, 1.162904E+04, 1.261929E+04, 1.367462E+04,
    1.479833E+04, 1.599377E+04, 1.726436E+04, 1.861379E+04, 2.004575E+04, 2.156398E+04,
    2.317217E+04, 2.487462E+04, 2.667503E+04, 2.857789E+04, 3.058721E+04, 3.270766E+04,
    3.494358E+04, 3.729938E+04, 3.978003E+04, 4.239034E+04, 4.513515E+04, 4.801943E+04,
    5.104863E+04, 5.422773E+04, 5.756221E+04, 6.105763E+04, 6.471951E+04, 6.855364E+04,
    7.256596E+04, 7.676252E+04, 8.114927E+04, 8.573247E+04, 9.051853E+04, 9.551399E+04,
    1.007252E+05, 1.061593E+05, 1.118231E+05, 1.177230E+05, 1.238669E+05, 1.302620E+05,
    1.369151E+05, 1.438344E+05, 1.510269E+05, 1.585011E+05, 1.662646E+05, 1.743254E+05,
    1.826918E+05, 1.913721E+05, 2.003749E+05, 2.097086E+05, 2.193821E+05, 2.294048E+05,
    2.397852E+05, 2.505325E+05, 2.616565E+05, 2.731663E+05, 2.850718E+05, 2.973825E+05,
    3.101087E+05, 3.232600E+05, 3.368470E+05, 3.508807E+05, 3.653706E+05, 3.803279E+05,
    3.957632E+05, 4.116883E+05, 4.281136E+05, 4.450504E+05, 4.625108E+05, 4.805059E+05,
    4.990478E+05, 5.181485E+05, 5.378202E+05, 5.580750E+05, 5.789261E+05, 6.003846E+05,
    6.224646E+05, 6.451784E+05, 6.685396E+05, 6.925614E+05, 7.172569E+05, 7.426405E+05,
    7.687248E+05, 7.955251E+05, 8.230548E+05, 8.513281E+05, 8.803602E+05, 9.101655E+05,
    9.407590E+05, 9.721548E+05, 1.004370E+06, 1.037418E+06, 1.071315E+06, 1.106078E+06,
    1.141721E+06, 1.178261E+06, 1.215715E+06, 1.254098E+06, 1.293428E+06, 1.333722E+06,
    1.374996E+06, 1.417268E+06, 1.460554E+06, 1.504874E+06, 1.550244E+06, 1.596682E+06,
    1.644207E+06, 1.692838E+06, 1.742592E+06, 1.793488E+06, 1.845546E+06, 1.898785E+06,
    1.953223E+06, 2.008881E+06, 2.065779E+06, 2.123935E+06, 2.183371E+06, 2.244106E+06,
    2.306163E+06, 2.369560E+06, 2.434319E+06, 2.500462E+06, 2.568009E+06, 2.636982E+06,
    2.707404E+06, 2.779295E+06, 2.852679E+06, 2.927578E+06, 3.004013E+06, 3.082010E+06,
    3.161589E+06, 3.242776E+06, 3.325591E+06, 3.410061E+06, 3.496209E+06, 3.584059E+06,
    3.673634E+06, 3.764960E+06, 3.858061E+06, 3.952965E+06, 4.049693E+06, 4.148272E+06,
    4.248730E+06, 4.351088E+06, 4.455377E+06, 4.561620E+06, 4.669848E+06, 4.780082E+06,
    4.892352E+06, 5.006686E+06,
])

#  ---------------------- M = 3, I = 4 ---------------------------
M = 3
I = 4
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    9.693860E+00, 7.286110E+02, 2.052138E+03, 3.765742E+03, 5.795622E+03, 8.099566E+03,
    1.065031E+04, 1.343144E+04, 1.643497E+04, 1.966034E+04, 2.311386E+04, 2.680652E+04,
    3.075398E+04, 3.497494E+04, 3.949041E+04, 4.432421E+04, 4.950089E+04, 5.504705E+04,
    6.098959E+04, 6.735674E+04, 7.417719E+04, 8.148200E+04, 8.930109E+04, 9.766575E+04,
    1.066095E+05, 1.161652E+05, 1.263658E+05, 1.372482E+05, 1.488478E+05, 1.612019E+05,
    1.743472E+05, 1.883247E+05, 2.031740E+05, 2.189350E+05, 2.356501E+05, 2.533639E+05,
    2.721204E+05, 2.919662E+05, 3.129454E+05, 3.351083E+05, 3.585027E+05, 3.831788E+05,
    4.091905E+05, 4.365882E+05, 4.654258E+05, 4.957624E+05, 5.276496E+05, 5.611468E+05,
    5.963170E+05, 6.332148E+05, 6.719059E+05, 7.124520E+05, 7.549189E+05, 7.993710E+05,
    8.458764E+05, 8.945049E+05, 9.453249E+05, 9.984083E+05, 1.053826E+06, 1.111656E+06,
    1.171972E+06, 1.234856E+06, 1.300379E+06, 1.368623E+06, 1.439674E+06, 1.513611E+06,
    1.590524E+06, 1.670492E+06, 1.753610E+06, 1.839962E+06, 1.929642E+06, 2.022739E+06,
    2.119351E+06, 2.219570E+06, 2.323499E+06, 2.431227E+06, 2.542864E+06, 2.658513E+06,
    2.778268E+06, 2.902239E+06, 3.030540E+06, 3.163267E+06, 3.300539E+06, 3.442465E+06,
    3.589168E+06, 3.740750E+06, 3.897328E+06, 4.059035E+06, 4.225983E+06, 4.398292E+06,
    4.576090E+06, 4.759502E+06, 4.948660E+06, 5.143690E+06, 5.344725E+06, 5.551892E+06,
    5.765331E+06, 5.985180E+06, 6.211579E+06, 6.444666E+06, 6.684586E+06, 6.931482E+06,
    7.185500E+06, 7.446782E+06, 7.715490E+06, 7.991764E+06, 8.275772E+06, 8.567664E+06,
    8.867593E+06, 9.175712E+06, 9.492203E+06, 9.817227E+06, 1.015093E+07, 1.049349E+07,
    1.084509E+07, 1.120589E+07, 1.157607E+07, 1.195579E+07, 1.234524E+07, 1.274460E+07,
    1.315405E+07, 1.357378E+07, 1.400397E+07, 1.444480E+07, 1.489648E+07, 1.535919E+07,
    1.583314E+07, 1.631850E+07, 1.681549E+07, 1.732430E+07, 1.784514E+07, 1.837821E+07,
    1.892372E+07, 1.948191E+07, 2.005293E+07, 2.063704E+07, 2.123444E+07, 2.184536E+07,
    2.247001E+07, 2.310862E+07, 2.376142E+07, 2.442864E+07, 2.511048E+07, 2.580720E+07,
    2.651904E+07, 2.724621E+07, 2.798898E+07, 2.874757E+07, 2.952224E+07, 3.031321E+07,
    3.112077E+07, 3.194514E+07, 3.278657E+07, 3.364534E+07, 3.452170E+07, 3.541590E+07,
    3.632821E+07, 3.725890E+07, 3.820824E+07, 3.917648E+07, 4.016391E+07, 4.117082E+07,
    4.219747E+07, 4.324413E+07, 4.431109E+07, 4.539866E+07, 4.650710E+07, 4.763672E+07,
    4.878780E+07, 4.996063E+07, 5.115550E+07, 5.237277E+07, 5.361268E+07, 5.487554E+07,
    5.616170E+07, 5.747145E+07,
])

#  ---------------------- M = 3, I = 5 ---------------------------
M = 3
I = 5
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    4.813030E+00, 3.600892E+02, 1.014082E+03, 1.860805E+03, 2.863799E+03, 4.002166E+03,
    5.262502E+03, 6.636539E+03, 8.120243E+03, 9.713429E+03, 1.141910E+04, 1.324287E+04,
    1.519237E+04, 1.727719E+04, 1.950780E+04, 2.189589E+04, 2.445388E+04, 2.719480E+04,
    3.013257E+04, 3.328074E+04, 3.665424E+04, 4.026795E+04, 4.413675E+04, 4.827698E+04,
    5.270423E+04, 5.743513E+04, 6.248716E+04, 6.787719E+04, 7.362388E+04, 7.974527E+04,
    8.626022E+04, 9.318860E+04, 1.005494E+05, 1.083638E+05, 1.166537E+05, 1.254391E+05,
    1.347427E+05, 1.445876E+05, 1.549967E+05, 1.659941E+05, 1.776035E+05, 1.898519E+05,
    2.027633E+05, 2.163637E+05, 2.306819E+05, 2.457443E+05, 2.615788E+05, 2.782148E+05,
    2.956814E+05, 3.140094E+05, 3.332292E+05, 3.533726E+05, 3.744711E+05, 3.965582E+05,
    4.196678E+05, 4.438332E+05, 4.690883E+05, 4.954722E+05, 5.230185E+05, 5.517647E+05,
    5.817489E+05, 6.130100E+05, 6.455856E+05, 6.795180E+05, 7.148483E+05, 7.516143E+05,
    7.898604E+05, 8.296320E+05, 8.709688E+05, 9.139178E+05, 9.585242E+05, 1.004832E+06,
    1.052892E+06, 1.102747E+06, 1.154451E+06, 1.208049E+06, 1.263593E+06, 1.321131E+06,
    1.380722E+06, 1.442410E+06, 1.506255E+06, 1.572304E+06, 1.640622E+06, 1.711256E+06,
    1.784265E+06, 1.859711E+06, 1.937648E+06, 2.018139E+06, 2.101238E+06, 2.187013E+06,
    2.275522E+06, 2.366832E+06, 2.461001E+06, 2.558097E+06, 2.658186E+06, 2.761333E+06,
    2.867604E+06, 2.977071E+06, 3.089804E+06, 3.205868E+06, 3.325339E+06, 3.448284E+06,
    3.574781E+06, 3.704902E+06, 3.838724E+06, 3.976317E+06, 4.117767E+06, 4.263141E+06,
    4.412525E+06, 4.565996E+06, 4.723636E+06, 4.885531E+06, 5.051753E+06, 5.222396E+06,
    5.397538E+06, 5.577272E+06, 5.761674E+06, 5.950846E+06, 6.144860E+06, 6.343822E+06,
    6.547813E+06, 6.756924E+06, 6.971254E+06, 7.190894E+06, 7.415940E+06, 7.646489E+06,
    7.882637E+06, 8.124482E+06, 8.372123E+06, 8.625660E+06, 8.885193E+06, 9.150831E+06,
    9.422673E+06, 9.700822E+06, 9.985389E+06, 1.027648E+07, 1.057420E+07, 1.087865E+07,
    1.118997E+07, 1.150823E+07, 1.183358E+07, 1.216611E+07, 1.250594E+07, 1.285320E+07,
    1.320799E+07, 1.357043E+07, 1.394065E+07, 1.431876E+07, 1.470489E+07, 1.509916E+07,
    1.550169E+07, 1.591261E+07, 1.633204E+07, 1.676013E+07, 1.719697E+07, 1.764274E+07,
    1.809752E+07, 1.856147E+07, 1.903474E+07, 1.951743E+07, 2.000969E+07, 2.051167E+07,
    2.102349E+07, 2.154530E+07, 2.207724E+07, 2.261945E+07, 2.317208E+07, 2.373528E+07,
    2.430918E+07, 2.489392E+07, 2.548968E+07, 2.609659E+07, 2.671481E+07, 2.734449E+07,
    2.798577E+07, 2.863883E+07,
])

#  ---------------------- M = 3, I = 6 ---------------------------
M = 3
I = 6
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.732560E+00, 1.308439E+02, 3.685813E+02, 6.763945E+02, 1.041035E+03, 1.454902E+03,
    1.913200E+03, 2.413032E+03, 2.953209E+03, 3.533915E+03, 4.156499E+03, 4.823380E+03,
    5.537674E+03, 6.303045E+03, 7.123765E+03, 8.004388E+03, 8.949564E+03, 9.964533E+03,
    1.105444E+04, 1.222466E+04, 1.348084E+04, 1.482862E+04, 1.627401E+04, 1.782304E+04,
    1.948187E+04, 2.125695E+04, 2.315481E+04, 2.518216E+04, 2.734599E+04, 2.965343E+04,
    3.211177E+04, 3.472865E+04, 3.751164E+04, 4.046894E+04, 4.360818E+04, 4.693842E+04,
    5.046789E+04, 5.420551E+04, 5.816009E+04, 6.234127E+04, 6.675826E+04, 7.142107E+04,
    7.633970E+04, 8.152423E+04, 8.698542E+04, 9.273349E+04, 9.877997E+04, 1.051355E+05,
    1.118121E+05, 1.188218E+05, 1.261758E+05, 1.338869E+05, 1.419674E+05, 1.504307E+05,
    1.592888E+05, 1.685564E+05, 1.782460E+05, 1.883721E+05, 1.989491E+05, 2.099906E+05,
    2.215123E+05, 2.335285E+05, 2.460551E+05, 2.591078E+05, 2.727019E+05, 2.868541E+05,
    3.015810E+05, 3.168993E+05, 3.328262E+05, 3.493782E+05, 3.665743E+05, 3.844320E+05,
    4.029702E+05, 4.222066E+05, 4.421602E+05, 4.628517E+05, 4.842990E+05, 5.065236E+05,
    5.295438E+05, 5.533829E+05, 5.780588E+05, 6.035954E+05, 6.300121E+05, 6.573328E+05,
    6.855785E+05, 7.147714E+05, 7.449354E+05, 7.760936E+05, 8.082695E+05, 8.414866E+05,
    8.757691E+05, 9.111436E+05, 9.476326E+05, 9.852625E+05, 1.024060E+06, 1.064049E+06,
    1.105257E+06, 1.147712E+06, 1.191440E+06, 1.236468E+06, 1.282825E+06, 1.330539E+06,
    1.379637E+06, 1.430151E+06, 1.482109E+06, 1.535539E+06, 1.590474E+06, 1.646944E+06,
    1.704977E+06, 1.764609E+06, 1.825868E+06, 1.888788E+06, 1.953399E+06, 2.019738E+06,
    2.087834E+06, 2.157723E+06, 2.229439E+06, 2.303016E+06, 2.378487E+06, 2.455892E+06,
    2.535263E+06, 2.616637E+06, 2.700050E+06, 2.785540E+06, 2.873144E+06, 2.962900E+06,
    3.054845E+06, 3.149018E+06, 3.245459E+06, 3.344205E+06, 3.445299E+06, 3.548780E+06,
    3.654689E+06, 3.763068E+06, 3.873955E+06, 3.987395E+06, 4.103430E+06, 4.222104E+06,
    4.343458E+06, 4.467538E+06, 4.594386E+06, 4.724049E+06, 4.856572E+06, 4.991999E+06,
    5.130375E+06, 5.271750E+06, 5.416171E+06, 5.563681E+06, 5.714331E+06, 5.868170E+06,
    6.025247E+06, 6.185609E+06, 6.349307E+06, 6.516392E+06, 6.686913E+06, 6.860922E+06,
    7.038471E+06, 7.219613E+06, 7.404399E+06, 7.592883E+06, 7.785118E+06, 7.981157E+06,
    8.181058E+06, 8.384875E+06, 8.592660E+06, 8.804475E+06, 9.020369E+06, 9.240407E+06,
    9.464641E+06, 9.693130E+06, 9.925935E+06, 1.016311E+07, 1.040473E+07, 1.065083E+07,
    1.090149E+07, 1.115676E+07,
])

#  ---------------------- M = 3, I = 7 ---------------------------
M = 3
I = 7
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    8.789300E-01, 6.694173E+01, 1.886102E+02, 3.461476E+02, 5.327717E+02, 7.445995E+02,
    9.791830E+02, 1.235085E+03, 1.511700E+03, 1.809143E+03, 2.128107E+03, 2.469769E+03,
    2.835738E+03, 3.227819E+03, 3.648133E+03, 4.098969E+03, 4.582699E+03, 5.101875E+03,
    5.659118E+03, 6.257169E+03, 6.898826E+03, 7.586970E+03, 8.324625E+03, 9.114762E+03,
    9.960542E+03, 1.086520E+04, 1.183213E+04, 1.286461E+04, 1.396617E+04, 1.514050E+04,
    1.639117E+04, 1.772203E+04, 1.913689E+04, 2.064001E+04, 2.223524E+04, 2.392689E+04,
    2.571922E+04, 2.761689E+04, 2.962419E+04, 3.174614E+04, 3.398705E+04, 3.635233E+04,
    3.884663E+04, 4.147543E+04, 4.424376E+04, 4.715716E+04, 5.022104E+04, 5.344119E+04,
    5.682326E+04, 6.037348E+04, 6.409755E+04, 6.800167E+04, 7.209225E+04, 7.637581E+04,
    8.085871E+04, 8.554797E+04, 9.045026E+04, 9.557287E+04, 1.009224E+05, 1.065065E+05,
    1.123325E+05, 1.184081E+05, 1.247407E+05, 1.313384E+05, 1.382093E+05, 1.453615E+05,
    1.528030E+05, 1.605426E+05, 1.685888E+05, 1.769504E+05, 1.856360E+05, 1.946552E+05,
    2.040167E+05, 2.137304E+05, 2.238055E+05, 2.342516E+05, 2.450786E+05, 2.562968E+05,
    2.679164E+05, 2.799473E+05, 2.924003E+05, 3.052861E+05, 3.186154E+05, 3.323992E+05,
    3.466491E+05, 3.613758E+05, 3.765914E+05, 3.923068E+05, 4.085344E+05, 4.252868E+05,
    4.425748E+05, 4.604120E+05, 4.788108E+05, 4.977832E+05, 5.173427E+05, 5.375024E+05,
    5.582752E+05, 5.796753E+05, 6.017155E+05, 6.244101E+05, 6.477726E+05, 6.718181E+05,
    6.965603E+05, 7.220136E+05, 7.481935E+05, 7.751140E+05, 8.027916E+05, 8.312397E+05,
    8.604757E+05, 8.905141E+05, 9.213715E+05, 9.530639E+05, 9.856067E+05, 1.019018E+06,
    1.053313E+06, 1.088509E+06, 1.124624E+06, 1.161674E+06, 1.199678E+06, 1.238652E+06,
    1.278615E+06, 1.319584E+06, 1.361580E+06, 1.404619E+06, 1.448720E+06, 1.493902E+06,
    1.540186E+06, 1.587589E+06, 1.636131E+06, 1.685832E+06, 1.736714E+06, 1.788795E+06,
    1.842095E+06, 1.896636E+06, 1.952439E+06, 2.009525E+06, 2.067914E+06, 2.127629E+06,
    2.188690E+06, 2.251121E+06, 2.314944E+06, 2.380180E+06, 2.446852E+06, 2.514985E+06,
    2.584598E+06, 2.655719E+06, 2.728370E+06, 2.802572E+06, 2.878352E+06, 2.955734E+06,
    3.034741E+06, 3.115400E+06, 3.197733E+06, 3.281769E+06, 3.367530E+06, 3.455043E+06,
    3.544334E+06, 3.635429E+06, 3.728356E+06, 3.823139E+06, 3.919806E+06, 4.018383E+06,
    4.118900E+06, 4.221384E+06, 4.325860E+06, 4.432360E+06, 4.540909E+06, 4.651539E+06,
    4.764276E+06, 4.879150E+06, 4.996191E+06, 5.115428E+06, 5.236890E+06, 5.360610E+06,
    5.486615E+06, 5.614939E+06,
])

#  ---------------------- M = 3, I = 8 ---------------------------
M = 3
I = 8
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.018317E+01, 7.688685E+02, 2.165826E+03, 3.974547E+03, 6.117122E+03, 8.549075E+03,
    1.124182E+04, 1.417849E+04, 1.735159E+04, 2.076146E+04, 2.441595E+04, 2.832833E+04,
    3.251593E+04, 3.699983E+04, 4.180437E+04, 4.695509E+04, 5.247930E+04, 5.840655E+04,
    6.476616E+04, 7.158997E+04, 7.890924E+04, 8.675751E+04, 9.516855E+04, 1.041762E+05,
    1.138172E+05, 1.241287E+05, 1.351464E+05, 1.469105E+05, 1.594614E+05, 1.728378E+05,
    1.870849E+05, 2.022417E+05, 2.183567E+05, 2.354731E+05, 2.536375E+05, 2.728988E+05,
    2.933058E+05, 3.149078E+05, 3.377591E+05, 3.619112E+05, 3.874208E+05, 4.143397E+05,
    4.427289E+05, 4.726444E+05, 5.041454E+05, 5.372972E+05, 5.721616E+05, 6.087995E+05,
    6.472799E+05, 6.876690E+05, 7.300362E+05, 7.744501E+05, 8.209831E+05, 8.697093E+05,
    9.207011E+05, 9.740386E+05, 1.029797E+06, 1.088057E+06, 1.148900E+06, 1.212405E+06,
    1.278663E+06, 1.347752E+06, 1.419769E+06, 1.494792E+06, 1.572920E+06, 1.654247E+06,
    1.738862E+06, 1.826861E+06, 1.918342E+06, 2.013409E+06, 2.112162E+06, 2.214696E+06,
    2.321129E+06, 2.431558E+06, 2.546090E+06, 2.664846E+06, 2.787926E+06, 2.915445E+06,
    3.047527E+06, 3.184286E+06, 3.325838E+06, 3.472305E+06, 3.623810E+06, 3.780482E+06,
    3.942443E+06, 4.109824E+06, 4.282759E+06, 4.461371E+06, 4.645800E+06, 4.836192E+06,
    5.032673E+06, 5.235386E+06, 5.444471E+06, 5.660087E+06, 5.882366E+06, 6.111462E+06,
    6.347516E+06, 6.590696E+06, 6.841155E+06, 7.099044E+06, 7.364517E+06, 7.637744E+06,
    7.918894E+06, 8.208116E+06, 8.505584E+06, 8.811474E+06, 9.125951E+06, 9.449192E+06,
    9.781375E+06, 1.012267E+07, 1.047327E+07, 1.083334E+07, 1.120309E+07, 1.158269E+07,
    1.197233E+07, 1.237221E+07, 1.278251E+07, 1.320344E+07, 1.363520E+07, 1.407798E+07,
    1.453199E+07, 1.499743E+07, 1.547452E+07, 1.596346E+07, 1.646446E+07, 1.697775E+07,
    1.750353E+07, 1.804203E+07, 1.859347E+07, 1.915807E+07, 1.973607E+07, 2.032768E+07,
    2.093316E+07, 2.155272E+07, 2.218661E+07, 2.283506E+07, 2.349832E+07, 2.417662E+07,
    2.487023E+07, 2.557938E+07, 2.630433E+07, 2.704534E+07, 2.780265E+07, 2.857654E+07,
    2.936727E+07, 3.017509E+07, 3.100027E+07, 3.184310E+07, 3.270384E+07, 3.358276E+07,
    3.448014E+07, 3.539626E+07, 3.633142E+07, 3.728588E+07, 3.825994E+07, 3.925390E+07,
    4.026805E+07, 4.130268E+07, 4.235810E+07, 4.343460E+07, 4.453249E+07, 4.565210E+07,
    4.679370E+07, 4.795764E+07, 4.914420E+07, 5.035375E+07, 5.158655E+07, 5.284297E+07,
    5.412334E+07, 5.542796E+07, 5.675719E+07, 5.811133E+07, 5.949078E+07, 6.089583E+07,
    6.232682E+07, 6.378415E+07,
])

#  ---------------------- M = 3, I = 9 ---------------------------
M = 3
I = 9
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.025939E+01, 7.780853E+02, 2.192024E+03, 4.022766E+03, 6.191507E+03, 8.653040E+03,
    1.137884E+04, 1.435181E+04, 1.756428E+04, 2.101726E+04, 2.471805E+04, 2.867999E+04,
    3.292087E+04, 3.746130E+04, 4.232569E+04, 4.753963E+04, 5.313097E+04, 5.912840E+04,
    6.556172E+04, 7.246219E+04, 7.986305E+04, 8.779593E+04, 9.629532E+04, 1.053959E+05,
    1.151338E+05, 1.255461E+05, 1.366706E+05, 1.485447E+05, 1.612110E+05, 1.747082E+05,
    1.890807E+05, 2.043701E+05, 2.206199E+05, 2.378786E+05, 2.561919E+05, 2.756073E+05,
    2.961737E+05, 3.179441E+05, 3.409693E+05, 3.653014E+05, 3.909969E+05, 4.181098E+05,
    4.467016E+05, 4.768262E+05, 5.085454E+05, 5.419201E+05, 5.770169E+05, 6.138970E+05,
    6.526250E+05, 6.932718E+05, 7.359051E+05, 7.805932E+05, 8.274118E+05, 8.764327E+05,
    9.277281E+05, 9.813788E+05, 1.037460E+06, 1.096053E+06, 1.157238E+06, 1.221100E+06,
    1.287723E+06, 1.357190E+06, 1.429589E+06, 1.505014E+06, 1.583553E+06, 1.665298E+06,
    1.750344E+06, 1.838791E+06, 1.930734E+06, 2.026272E+06, 2.125506E+06, 2.228539E+06,
    2.335477E+06, 2.446431E+06, 2.561504E+06, 2.680806E+06, 2.804454E+06, 2.932553E+06,
    3.065232E+06, 3.202599E+06, 3.344773E+06, 3.491880E+06, 3.644044E+06, 3.801383E+06,
    3.964036E+06, 4.132119E+06, 4.305769E+06, 4.485124E+06, 4.670306E+06, 4.861464E+06,
    5.058736E+06, 5.262256E+06, 5.472173E+06, 5.688624E+06, 5.911762E+06, 6.141735E+06,
    6.378700E+06, 6.622796E+06, 6.874190E+06, 7.133033E+06, 7.399489E+06, 7.673712E+06,
    7.955873E+06, 8.246134E+06, 8.544672E+06, 8.851639E+06, 9.167217E+06, 9.491582E+06,
    9.824914E+06, 1.016738E+07, 1.051917E+07, 1.088046E+07, 1.125145E+07, 1.163231E+07,
    1.202323E+07, 1.242442E+07, 1.283606E+07, 1.325835E+07, 1.369150E+07, 1.413568E+07,
    1.459113E+07, 1.505804E+07, 1.553661E+07, 1.602706E+07, 1.652961E+07, 1.704446E+07,
    1.757184E+07, 1.811196E+07, 1.866504E+07, 1.923134E+07, 1.981104E+07, 2.040440E+07,
    2.101163E+07, 2.163299E+07, 2.226870E+07, 2.291900E+07, 2.358413E+07, 2.426437E+07,
    2.495991E+07, 2.567104E+07, 2.639800E+07, 2.714105E+07, 2.790045E+07, 2.867645E+07,
    2.946930E+07, 3.027929E+07, 3.110670E+07, 3.195175E+07, 3.281477E+07, 3.369600E+07,
    3.459571E+07, 3.551422E+07, 3.645178E+07, 3.740870E+07, 3.838525E+07, 3.938174E+07,
    4.039843E+07, 4.143568E+07, 4.249373E+07, 4.357290E+07, 4.467351E+07, 4.579585E+07,
    4.694025E+07, 4.810702E+07, 4.929646E+07, 5.050889E+07, 5.174464E+07, 5.300407E+07,
    5.428747E+07, 5.559516E+07, 5.692750E+07, 5.828480E+07, 5.966743E+07, 6.107572E+07,
    6.251002E+07, 6.397068E+07,
])

#  ---------------------- M = 3, I = 10 ---------------------------
M = 3
I = 10
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.011611E+01, 7.606432E+02, 2.142444E+03, 3.931508E+03, 6.050781E+03, 8.456192E+03,
    1.111961E+04, 1.402401E+04, 1.716179E+04, 2.053370E+04, 2.414710E+04, 2.801498E+04,
    3.215563E+04, 3.658950E+04, 4.134081E+04, 4.643537E+04, 5.190096E+04, 5.776617E+04,
    6.406074E+04, 7.081585E+04, 7.806331E+04, 8.583661E+04, 9.416875E+04, 1.030942E+05,
    1.126489E+05, 1.228698E+05, 1.337943E+05, 1.454598E+05, 1.579072E+05, 1.711776E+05,
    1.853113E+05, 2.003519E+05, 2.163455E+05, 2.333335E+05, 2.513666E+05, 2.704892E+05,
    2.907529E+05, 3.122071E+05, 3.349040E+05, 3.588951E+05, 3.842348E+05, 4.109803E+05,
    4.391888E+05, 4.689179E+05, 5.002271E+05, 5.331765E+05, 5.678301E+05, 6.042531E+05,
    6.425100E+05, 6.826686E+05, 7.247962E+05, 7.689631E+05, 8.152419E+05, 8.637038E+05,
    9.144236E+05, 9.674780E+05, 1.022945E+06, 1.080905E+06, 1.141440E+06, 1.204627E+06,
    1.270556E+06, 1.339309E+06, 1.410973E+06, 1.485641E+06, 1.563401E+06, 1.644344E+06,
    1.728569E+06, 1.816162E+06, 1.907235E+06, 2.001877E+06, 2.100190E+06, 2.202280E+06,
    2.308247E+06, 2.418202E+06, 2.532251E+06, 2.650506E+06, 2.773076E+06, 2.900076E+06,
    3.031614E+06, 3.167824E+06, 3.308806E+06, 3.454693E+06, 3.605607E+06, 3.761672E+06,
    3.923009E+06, 4.089756E+06, 4.262030E+06, 4.439977E+06, 4.623729E+06, 4.813416E+06,
    5.009182E+06, 5.211159E+06, 5.419501E+06, 5.634346E+06, 5.855846E+06, 6.084142E+06,
    6.319381E+06, 6.561730E+06, 6.811327E+06, 7.068344E+06, 7.332930E+06, 7.605253E+06,
    7.885465E+06, 8.173747E+06, 8.470249E+06, 8.775154E+06, 9.088627E+06, 9.410835E+06,
    9.741974E+06, 1.008220E+07, 1.043171E+07, 1.079068E+07, 1.115929E+07, 1.153774E+07,
    1.192620E+07, 1.232488E+07, 1.273397E+07, 1.315366E+07, 1.358416E+07, 1.402564E+07,
    1.447834E+07, 1.494244E+07, 1.541817E+07, 1.590572E+07, 1.640531E+07, 1.691716E+07,
    1.744149E+07, 1.797850E+07, 1.852842E+07, 1.909149E+07, 1.966793E+07, 2.025794E+07,
    2.086181E+07, 2.147971E+07, 2.211193E+07, 2.275867E+07, 2.342019E+07, 2.409675E+07,
    2.478856E+07, 2.549589E+07, 2.621900E+07, 2.695813E+07, 2.771354E+07, 2.848549E+07,
    2.927424E+07, 3.008006E+07, 3.090322E+07, 3.174399E+07, 3.260262E+07, 3.347942E+07,
    3.437463E+07, 3.528858E+07, 3.622149E+07, 3.717369E+07, 3.814547E+07, 3.913711E+07,
    4.014888E+07, 4.118112E+07, 4.223410E+07, 4.330813E+07, 4.440353E+07, 4.552057E+07,
    4.665960E+07, 4.782093E+07, 4.900486E+07, 5.021170E+07, 5.144178E+07, 5.269544E+07,
    5.397301E+07, 5.527477E+07, 5.660111E+07, 5.795235E+07, 5.932882E+07, 6.073086E+07,
    6.215883E+07, 6.361306E+07,
])

#  ---------------------- M = 3, I = 11 ---------------------------
M = 3
I = 11
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    5.946791E+01, 4.470503E+03, 1.259147E+04, 2.310592E+04, 3.556102E+04, 4.969775E+04,
    6.535012E+04, 8.241681E+04, 1.008519E+05, 1.206550E+05, 1.418675E+05, 1.645612E+05,
    1.888379E+05, 2.148151E+05, 2.426307E+05, 2.724293E+05, 3.043712E+05, 3.386193E+05,
    3.753466E+05, 4.147317E+05, 4.569565E+05, 5.022089E+05, 5.506812E+05, 6.025756E+05,
    6.580938E+05, 7.174414E+05, 7.808443E+05, 8.485166E+05, 9.206880E+05, 9.975858E+05,
    1.079455E+06, 1.166548E+06, 1.259104E+06, 1.357392E+06, 1.461671E+06, 1.572220E+06,
    1.689327E+06, 1.813266E+06, 1.944345E+06, 2.082848E+06, 2.229110E+06, 2.383425E+06,
    2.546143E+06, 2.717572E+06, 2.898074E+06, 3.087996E+06, 3.287693E+06, 3.497522E+06,
    3.717868E+06, 3.949102E+06, 4.191641E+06, 4.445858E+06, 4.712167E+06, 4.991009E+06,
    5.282761E+06, 5.587915E+06, 5.906867E+06, 6.240090E+06, 6.588050E+06, 6.951201E+06,
    7.330046E+06, 7.725058E+06, 8.136736E+06, 8.565590E+06, 9.012166E+06, 9.476920E+06,
    9.960460E+06, 1.046329E+07, 1.098601E+07, 1.152911E+07, 1.209326E+07, 1.267897E+07,
    1.328686E+07, 1.391754E+07, 1.457164E+07, 1.524976E+07, 1.595255E+07, 1.668068E+07,
    1.743475E+07, 1.821548E+07, 1.902353E+07, 1.985957E+07, 2.072435E+07, 2.161852E+07,
    2.254284E+07, 2.349807E+07, 2.448487E+07, 2.550405E+07, 2.655638E+07, 2.764263E+07,
    2.876356E+07, 2.992002E+07, 3.111277E+07, 3.234267E+07, 3.361055E+07, 3.491722E+07,
    3.626357E+07, 3.765045E+07, 3.907876E+07, 4.054937E+07, 4.206321E+07, 4.362119E+07,
    4.522422E+07, 4.687322E+07, 4.856921E+07, 5.031312E+07, 5.210590E+07, 5.394861E+07,
    5.584216E+07, 5.778765E+07, 5.978600E+07, 6.183837E+07, 6.394578E+07, 6.610924E+07,
    6.832982E+07, 7.060868E+07, 7.294696E+07, 7.534565E+07, 7.780593E+07, 8.032895E+07,
    8.291590E+07, 8.556789E+07, 8.828611E+07, 9.107179E+07, 9.392614E+07, 9.685037E+07,
    9.984571E+07, 1.029134E+08, 1.060547E+08, 1.092709E+08, 1.125633E+08, 1.159332E+08,
    1.193819E+08, 1.229108E+08, 1.265212E+08, 1.302143E+08, 1.339918E+08, 1.378548E+08,
    1.418050E+08, 1.458435E+08, 1.499719E+08, 1.541916E+08, 1.585041E+08, 1.629109E+08,
    1.674134E+08, 1.720131E+08, 1.767117E+08, 1.815105E+08, 1.864112E+08, 1.914154E+08,
    1.965245E+08, 2.017402E+08, 2.070642E+08, 2.124980E+08, 2.180433E+08, 2.237017E+08,
    2.294748E+08, 2.353645E+08, 2.413725E+08, 2.475002E+08, 2.537497E+08, 2.601226E+08,
    2.666206E+08, 2.732456E+08, 2.799992E+08, 2.868835E+08, 2.939002E+08, 3.010511E+08,
    3.083381E+08, 3.157630E+08, 3.233278E+08, 3.310344E+08, 3.388847E+08, 3.468806E+08,
    3.550242E+08, 3.633173E+08,
])

#  ---------------------- M = 3, I = 12 ---------------------------
M = 3
I = 12
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.995096E+01, 2.261799E+03, 6.371201E+03, 1.169187E+04, 1.799462E+04, 2.514837E+04,
    3.306945E+04, 4.170726E+04, 5.103813E+04, 6.106308E+04, 7.180236E+04, 8.329235E+04,
    9.558279E+04, 1.087346E+05, 1.228140E+05, 1.378944E+05, 1.540562E+05, 1.713824E+05,
    1.899567E+05, 2.098700E+05, 2.312133E+05, 2.540827E+05, 2.785728E+05, 3.047847E+05,
    3.328220E+05, 3.627880E+05, 3.947933E+05, 4.289450E+05, 4.653615E+05, 5.041570E+05,
    5.454511E+05, 5.893697E+05, 6.360411E+05, 6.855926E+05, 7.381579E+05, 7.938731E+05,
    8.528868E+05, 9.153318E+05, 9.813678E+05, 1.051138E+06, 1.124804E+06, 1.202521E+06,
    1.284451E+06, 1.370767E+06, 1.461638E+06, 1.557240E+06, 1.657755E+06, 1.763358E+06,
    1.874240E+06, 1.990604E+06, 2.112627E+06, 2.240529E+06, 2.374501E+06, 2.514755E+06,
    2.661508E+06, 2.814979E+06, 2.975376E+06, 3.142946E+06, 3.317913E+06, 3.500504E+06,
    3.690974E+06, 3.889547E+06, 4.096506E+06, 4.312075E+06, 4.536516E+06, 4.770115E+06,
    5.013131E+06, 5.265828E+06, 5.528484E+06, 5.801396E+06, 6.084839E+06, 6.379123E+06,
    6.684533E+06, 7.001385E+06, 7.329976E+06, 7.670618E+06, 8.023632E+06, 8.389355E+06,
    8.768103E+06, 9.160211E+06, 9.566016E+06, 9.985881E+06, 1.042016E+07, 1.086916E+07,
    1.133330E+07, 1.181291E+07, 1.230839E+07, 1.282011E+07, 1.334843E+07, 1.389376E+07,
    1.445651E+07, 1.503703E+07, 1.563578E+07, 1.625316E+07, 1.688956E+07, 1.754542E+07,
    1.822117E+07, 1.891726E+07, 1.963410E+07, 2.037217E+07, 2.113191E+07, 2.191375E+07,
    2.271818E+07, 2.354568E+07, 2.439673E+07, 2.527179E+07, 2.617137E+07, 2.709593E+07,
    2.804603E+07, 2.902211E+07, 3.002473E+07, 3.105443E+07, 3.211166E+07, 3.319703E+07,
    3.431104E+07, 3.545426E+07, 3.662721E+07, 3.783046E+07, 3.906458E+07, 4.033016E+07,
    4.162776E+07, 4.295797E+07, 4.432139E+07, 4.571860E+07, 4.715020E+07, 4.861682E+07,
    5.011909E+07, 5.165761E+07, 5.323299E+07, 5.484596E+07, 5.649708E+07, 5.818704E+07,
    5.991646E+07, 6.168607E+07, 6.349652E+07, 6.534847E+07, 6.724265E+07, 6.917967E+07,
    7.116036E+07, 7.318530E+07, 7.525531E+07, 7.737105E+07, 7.953329E+07, 8.174279E+07,
    8.400021E+07, 8.630635E+07, 8.866200E+07, 9.106790E+07, 9.352481E+07, 9.603355E+07,
    9.859486E+07, 1.012096E+08, 1.038786E+08, 1.066026E+08, 1.093824E+08, 1.122189E+08,
    1.151128E+08, 1.180652E+08, 1.210768E+08, 1.241484E+08, 1.272810E+08, 1.304754E+08,
    1.337324E+08, 1.370531E+08, 1.404382E+08, 1.438888E+08, 1.474056E+08, 1.509897E+08,
    1.546420E+08, 1.583634E+08, 1.621548E+08, 1.660172E+08, 1.699516E+08, 1.739590E+08,
    1.780403E+08, 1.821965E+08,
])

#  ---------------------- M = 3, I = 13 ---------------------------
M = 3
I = 13
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    9.181300E-01, 6.995503E+01, 1.971073E+02, 3.617463E+02, 5.567837E+02, 7.781615E+02,
    1.023352E+03, 1.290876E+03, 1.580177E+03, 1.891452E+03, 2.225587E+03, 2.583933E+03,
    2.968275E+03, 3.380727E+03, 3.823605E+03, 4.299458E+03, 4.810934E+03, 5.360874E+03,
    5.952091E+03, 6.587671E+03, 7.270592E+03, 8.004158E+03, 8.791511E+03, 9.636056E+03,
    1.054129E+04, 1.151069E+04, 1.254790E+04, 1.365681E+04, 1.484101E+04, 1.610476E+04,
    1.745189E+04, 1.888674E+04, 2.041352E+04, 2.203664E+04, 2.376087E+04, 2.559056E+04,
    2.753068E+04, 2.958620E+04, 3.176202E+04, 3.406331E+04, 3.649554E+04, 3.906410E+04,
    4.177453E+04, 4.463243E+04, 4.764399E+04, 5.081487E+04, 5.415128E+04, 5.765957E+04,
    6.134601E+04, 6.521766E+04, 6.928060E+04, 7.354192E+04, 7.800896E+04, 8.268836E+04,
    8.758771E+04, 9.271449E+04, 9.807632E+04, 1.036807E+05, 1.095361E+05, 1.156503E+05,
    1.220315E+05, 1.286881E+05, 1.356287E+05, 1.428623E+05, 1.503977E+05, 1.582439E+05,
    1.664102E+05, 1.749058E+05, 1.837406E+05, 1.929240E+05, 2.024661E+05, 2.123774E+05,
    2.226672E+05, 2.333470E+05, 2.444267E+05, 2.559177E+05, 2.678304E+05, 2.801759E+05,
    2.929662E+05, 3.062125E+05, 3.199267E+05, 3.341200E+05, 3.488054E+05, 3.639945E+05,
    3.797001E+05, 3.959348E+05, 4.127111E+05, 4.300426E+05, 4.479421E+05, 4.664231E+05,
    4.854994E+05, 5.051847E+05, 5.254935E+05, 5.464391E+05, 5.680363E+05, 5.902997E+05,
    6.132443E+05, 6.368844E+05, 6.612361E+05, 6.863149E+05, 7.121362E+05, 7.387153E+05,
    7.660688E+05, 7.942129E+05, 8.231639E+05, 8.529386E+05, 8.835533E+05, 9.150268E+05,
    9.473747E+05, 9.806152E+05, 1.014766E+06, 1.049845E+06, 1.085871E+06, 1.122862E+06,
    1.160837E+06, 1.199813E+06, 1.239812E+06, 1.280852E+06, 1.322952E+06, 1.366132E+06,
    1.410413E+06, 1.455814E+06, 1.502357E+06, 1.550060E+06, 1.598947E+06, 1.649037E+06,
    1.700354E+06, 1.752916E+06, 1.806748E+06, 1.861869E+06, 1.918305E+06, 1.976076E+06,
    2.035207E+06, 2.095720E+06, 2.157636E+06, 2.220983E+06, 2.285782E+06, 2.352058E+06,
    2.419834E+06, 2.489135E+06, 2.559988E+06, 2.632416E+06, 2.706446E+06, 2.782102E+06,
    2.859408E+06, 2.938395E+06, 3.019085E+06, 3.101507E+06, 3.185689E+06, 3.271654E+06,
    3.359434E+06, 3.449053E+06, 3.540540E+06, 3.633925E+06, 3.729233E+06, 3.826498E+06,
    3.925744E+06, 4.027002E+06, 4.130304E+06, 4.235674E+06, 4.343148E+06, 4.452752E+06,
    4.564520E+06, 4.678482E+06, 4.794667E+06, 4.913111E+06, 5.033839E+06, 5.156890E+06,
    5.282292E+06, 5.410079E+06, 5.540285E+06, 5.672941E+06, 5.808079E+06, 5.945739E+06,
    6.085949E+06, 6.228746E+06,
])

#  ---------------------- M = 3, I = 14 ---------------------------
M = 3
I = 14
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.070955E+01, 8.123365E+02, 2.288590E+03, 4.200027E+03, 6.464366E+03, 9.034498E+03,
    1.188072E+04, 1.498574E+04, 1.834233E+04, 2.195242E+04, 2.582553E+04, 2.997699E+04,
    3.442690E+04, 3.919918E+04, 4.432047E+04, 4.981933E+04, 5.572680E+04, 6.207426E+04,
    6.889543E+04, 7.622379E+04, 8.409527E+04, 9.254631E+04, 1.016143E+05, 1.113367E+05,
    1.217532E+05, 1.329047E+05, 1.448337E+05, 1.575806E+05, 1.711918E+05, 1.857116E+05,
    2.011872E+05, 2.176646E+05, 2.351958E+05, 2.538270E+05, 2.736142E+05, 2.946102E+05,
    3.168671E+05, 3.404431E+05, 3.653942E+05, 3.917795E+05, 4.196636E+05, 4.491018E+05,
    4.801635E+05, 5.129110E+05, 5.474123E+05, 5.837346E+05, 6.219462E+05, 6.621222E+05,
    7.043368E+05, 7.486602E+05, 7.951695E+05, 8.439459E+05, 8.950674E+05, 9.486189E+05,
    1.004678E+06, 1.063332E+06, 1.124673E+06, 1.188782E+06, 1.255752E+06, 1.325673E+06,
    1.398645E+06, 1.474761E+06, 1.554118E+06, 1.636818E+06, 1.722956E+06, 1.812642E+06,
    1.905980E+06, 2.003073E+06, 2.104038E+06, 2.208977E+06, 2.318006E+06, 2.431244E+06,
    2.548807E+06, 2.670808E+06, 2.797373E+06, 2.928627E+06, 3.064690E+06, 3.205688E+06,
    3.351759E+06, 3.503023E+06, 3.659621E+06, 3.821690E+06, 3.989358E+06, 4.162770E+06,
    4.342073E+06, 4.527402E+06, 4.718908E+06, 4.916745E+06, 5.121048E+06, 5.331982E+06,
    5.549699E+06, 5.774354E+06, 6.006112E+06, 6.245127E+06, 6.491574E+06, 6.745604E+06,
    7.007404E+06, 7.277128E+06, 7.554956E+06, 7.841067E+06, 8.135632E+06, 8.438833E+06,
    8.750860E+06, 9.071888E+06, 9.402110E+06, 9.741713E+06, 1.009090E+07, 1.044984E+07,
    1.081877E+07, 1.119784E+07, 1.158730E+07, 1.198732E+07, 1.239813E+07, 1.281992E+07,
    1.325293E+07, 1.369734E+07, 1.415340E+07, 1.462130E+07, 1.510128E+07, 1.559357E+07,
    1.609838E+07, 1.661596E+07, 1.714652E+07, 1.769032E+07, 1.824757E+07, 1.881854E+07,
    1.940346E+07, 2.000258E+07, 2.061614E+07, 2.124441E+07, 2.188760E+07, 2.254603E+07,
    2.321992E+07, 2.390954E+07, 2.461516E+07, 2.533705E+07, 2.607547E+07, 2.683070E+07,
    2.760302E+07, 2.839271E+07, 2.920004E+07, 3.002532E+07, 3.086881E+07, 3.173082E+07,
    3.261163E+07, 3.351154E+07, 3.443087E+07, 3.536991E+07, 3.632895E+07, 3.730831E+07,
    3.830832E+07, 3.932926E+07, 4.037147E+07, 4.143527E+07, 4.252097E+07, 4.362893E+07,
    4.475943E+07, 4.591285E+07, 4.708949E+07, 4.828972E+07, 4.951385E+07, 5.076226E+07,
    5.203526E+07, 5.333325E+07, 5.465654E+07, 5.600550E+07, 5.738049E+07, 5.878190E+07,
    6.021006E+07, 6.166536E+07, 6.314819E+07, 6.465888E+07, 6.619786E+07, 6.776550E+07,
    6.936214E+07, 7.098826E+07,
])

#  ---------------------- M = 3, I = 15 ---------------------------
M = 3
I = 15
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    5.394640E+00, 4.109442E+02, 1.157868E+03, 2.124994E+03, 3.270687E+03, 4.571107E+03,
    6.011346E+03, 7.582597E+03, 9.281391E+03, 1.110871E+04, 1.306931E+04, 1.517082E+04,
    1.742335E+04, 1.983895E+04, 2.243062E+04, 2.521306E+04, 2.820139E+04, 3.141182E+04,
    3.486102E+04, 3.856575E+04, 4.254401E+04, 4.681405E+04, 5.139459E+04, 5.630527E+04,
    6.156522E+04, 6.719486E+04, 7.321552E+04, 7.964854E+04, 8.651617E+04, 9.384115E+04,
    1.016460E+05, 1.099563E+05, 1.187949E+05, 1.281892E+05, 1.381630E+05, 1.487449E+05,
    1.599612E+05, 1.718402E+05, 1.844110E+05, 1.977042E+05, 2.117483E+05, 2.265752E+05,
    2.422176E+05, 2.587079E+05, 2.760779E+05, 2.943649E+05, 3.136008E+05, 3.338232E+05,
    3.550696E+05, 3.773763E+05, 4.007822E+05, 4.253252E+05, 4.510473E+05, 4.779882E+05,
    5.061912E+05, 5.356961E+05, 5.665499E+05, 5.987935E+05, 6.324759E+05, 6.676398E+05,
    7.043359E+05, 7.426104E+05, 7.825101E+05, 8.240889E+05, 8.673977E+05, 9.124835E+05,
    9.594033E+05, 1.008211E+06, 1.058958E+06, 1.111704E+06, 1.166502E+06, 1.223411E+06,
    1.282491E+06, 1.343799E+06, 1.407399E+06, 1.473350E+06, 1.541716E+06, 1.612559E+06,
    1.685944E+06, 1.761942E+06, 1.840612E+06, 1.922026E+06, 2.006251E+06, 2.093359E+06,
    2.183422E+06, 2.276511E+06, 2.372694E+06, 2.472055E+06, 2.574663E+06, 2.680597E+06,
    2.789932E+06, 2.902747E+06, 3.019125E+06, 3.139149E+06, 3.262894E+06, 3.390446E+06,
    3.521894E+06, 3.657318E+06, 3.796809E+06, 3.940448E+06, 4.088333E+06, 4.240552E+06,
    4.397193E+06, 4.558348E+06, 4.724115E+06, 4.894587E+06, 5.069865E+06, 5.250039E+06,
    5.435212E+06, 5.625483E+06, 5.820957E+06, 6.021730E+06, 6.227909E+06, 6.439603E+06,
    6.656912E+06, 6.879946E+06, 7.108816E+06, 7.343626E+06, 7.584494E+06, 7.831525E+06,
    8.084849E+06, 8.344561E+06, 8.610791E+06, 8.883657E+06, 9.163268E+06, 9.449758E+06,
    9.743239E+06, 1.004384E+07, 1.035168E+07, 1.066689E+07, 1.098959E+07, 1.131992E+07,
    1.165801E+07, 1.200399E+07, 1.235798E+07, 1.272013E+07, 1.309056E+07, 1.346943E+07,
    1.385686E+07, 1.425299E+07, 1.465797E+07, 1.507195E+07, 1.549505E+07, 1.592745E+07,
    1.636926E+07, 1.682065E+07, 1.728178E+07, 1.775276E+07, 1.823379E+07, 1.872501E+07,
    1.922656E+07, 1.973862E+07, 2.026134E+07, 2.079487E+07, 2.133938E+07, 2.189504E+07,
    2.246202E+07, 2.304046E+07, 2.363056E+07, 2.423247E+07, 2.484636E+07, 2.547242E+07,
    2.611081E+07, 2.676171E+07, 2.742530E+07, 2.810175E+07, 2.879126E+07, 2.949400E+07,
    3.021015E+07, 3.093989E+07, 3.168343E+07, 3.244094E+07, 3.321261E+07, 3.399866E+07,
    3.479924E+07, 3.561458E+07,
])

#  ---------------------- M = 3, I = 16 ---------------------------
M = 3
I = 16
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    6.295915E+01, 4.775780E+03, 1.345460E+04, 2.469181E+04, 3.800370E+04, 5.311278E+04,
    6.984465E+04, 8.809583E+04, 1.078215E+05, 1.290311E+05, 1.517738E+05, 1.761371E+05,
    2.022330E+05, 2.301970E+05, 2.601801E+05, 2.923500E+05, 3.268756E+05, 3.639441E+05,
    4.037454E+05, 4.464741E+05, 4.923320E+05, 5.415271E+05, 5.942775E+05, 6.508028E+05,
    7.113265E+05, 7.760820E+05, 8.453078E+05, 9.192467E+05, 9.981526E+05, 1.082292E+06,
    1.171923E+06, 1.267315E+06, 1.368758E+06, 1.476543E+06, 1.590956E+06, 1.712308E+06,
    1.840917E+06, 1.977087E+06, 2.121159E+06, 2.273473E+06, 2.434375E+06, 2.604205E+06,
    2.783338E+06, 2.972142E+06, 3.171006E+06, 3.380315E+06, 3.600476E+06, 3.831875E+06,
    4.074950E+06, 4.330119E+06, 4.597824E+06, 4.878515E+06, 5.172635E+06, 5.480655E+06,
    5.803069E+06, 6.140331E+06, 6.492956E+06, 6.861430E+06, 7.246294E+06, 7.648073E+06,
    8.067282E+06, 8.504471E+06, 8.960216E+06, 9.435065E+06, 9.929615E+06, 1.044443E+07,
    1.098014E+07, 1.153735E+07, 1.211666E+07, 1.271871E+07, 1.334416E+07, 1.399362E+07,
    1.466782E+07, 1.536741E+07, 1.609309E+07, 1.684551E+07, 1.762546E+07, 1.843360E+07,
    1.927066E+07, 2.013746E+07, 2.103470E+07, 2.196320E+07, 2.292366E+07, 2.391697E+07,
    2.494387E+07, 2.600520E+07, 2.710179E+07, 2.823452E+07, 2.940420E+07, 3.061173E+07,
    3.185798E+07, 3.314380E+07, 3.447017E+07, 3.583801E+07, 3.724820E+07, 3.870172E+07,
    4.019952E+07, 4.174256E+07, 4.333188E+07, 4.496840E+07, 4.665318E+07, 4.838728E+07,
    5.017165E+07, 5.200740E+07, 5.389561E+07, 5.583735E+07, 5.783364E+07, 5.988569E+07,
    6.199461E+07, 6.416146E+07, 6.638749E+07, 6.867377E+07, 7.102154E+07, 7.343199E+07,
    7.590634E+07, 7.844571E+07, 8.105146E+07, 8.372481E+07, 8.646697E+07, 8.927926E+07,
    9.216301E+07, 9.511942E+07, 9.814997E+07, 1.012559E+08, 1.044386E+08, 1.076993E+08,
    1.110396E+08, 1.144608E+08, 1.179644E+08, 1.215516E+08, 1.252241E+08, 1.289833E+08,
    1.328306E+08, 1.367675E+08, 1.407957E+08, 1.449164E+08, 1.491314E+08, 1.534422E+08,
    1.578503E+08, 1.623573E+08, 1.669650E+08, 1.716746E+08, 1.764882E+08, 1.814073E+08,
    1.864334E+08, 1.915684E+08, 1.968138E+08, 2.021716E+08, 2.076433E+08, 2.132308E+08,
    2.189357E+08, 2.247600E+08, 2.307054E+08, 2.367737E+08, 2.429668E+08, 2.492865E+08,
    2.557347E+08, 2.623134E+08, 2.690243E+08, 2.758695E+08, 2.828508E+08, 2.899701E+08,
    2.972298E+08, 3.046315E+08, 3.121773E+08, 3.198694E+08, 3.277097E+08, 3.357002E+08,
    3.438432E+08, 3.521405E+08, 3.605945E+08, 3.692073E+08, 3.779810E+08, 3.869178E+08,
    3.960200E+08, 4.052896E+08,
])

#  ---------------------- M = 3, I = 17 ---------------------------
M = 3
I = 17
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    3.126567E+01, 2.362010E+03, 6.653739E+03, 1.221051E+04, 1.879316E+04, 2.626447E+04,
    3.453787E+04, 4.356149E+04, 5.331364E+04, 6.379765E+04, 7.503938E+04, 8.708147E+04,
    9.998024E+04, 1.138041E+05, 1.286285E+05, 1.445357E+05, 1.616128E+05, 1.799519E+05,
    1.996462E+05, 2.207944E+05, 2.434973E+05, 2.678588E+05, 2.939855E+05, 3.219866E+05,
    3.519763E+05, 3.840710E+05, 4.183868E+05, 4.550447E+05, 4.941742E+05, 5.359053E+05,
    5.803658E+05, 6.276941E+05, 6.780312E+05, 7.315213E+05, 7.883119E+05, 8.485581E+05,
    9.124080E+05, 9.800237E+05, 1.051575E+06, 1.127220E+06, 1.207142E+06, 1.291512E+06,
    1.380510E+06, 1.474325E+06, 1.573144E+06, 1.677168E+06, 1.786591E+06, 1.901612E+06,
    2.022451E+06, 2.149307E+06, 2.282409E+06, 2.421973E+06, 2.568228E+06, 2.721418E+06,
    2.881762E+06, 3.049503E+06, 3.224910E+06, 3.408205E+06, 3.599666E+06, 3.799557E+06,
    4.008133E+06, 4.225669E+06, 4.452440E+06, 4.688742E+06, 4.934864E+06, 5.191084E+06,
    5.457716E+06, 5.735050E+06, 6.023406E+06, 6.323116E+06, 6.634468E+06, 6.957803E+06,
    7.293467E+06, 7.641780E+06, 8.003085E+06, 8.377742E+06, 8.766122E+06, 9.168553E+06,
    9.585412E+06, 1.001709E+07, 1.046394E+07, 1.092639E+07, 1.140478E+07, 1.189953E+07,
    1.241103E+07, 1.293971E+07, 1.348598E+07, 1.405025E+07, 1.463297E+07, 1.523455E+07,
    1.585545E+07, 1.649609E+07, 1.715694E+07, 1.783848E+07, 1.854115E+07, 1.926541E+07,
    2.001179E+07, 2.078071E+07, 2.157272E+07, 2.238827E+07, 2.322791E+07, 2.409212E+07,
    2.498146E+07, 2.589638E+07, 2.683750E+07, 2.780528E+07, 2.880035E+07, 2.982319E+07,
    3.087441E+07, 3.195455E+07, 3.306416E+07, 3.420390E+07, 3.537427E+07, 3.657594E+07,
    3.780946E+07, 3.907546E+07, 4.037457E+07, 4.170740E+07, 4.307458E+07, 4.447674E+07,
    4.591456E+07, 4.738867E+07, 4.889974E+07, 5.044841E+07, 5.203540E+07, 5.366135E+07,
    5.532700E+07, 5.703306E+07, 5.878013E+07, 6.056905E+07, 6.240048E+07, 6.427519E+07,
    6.619386E+07, 6.815728E+07, 7.016617E+07, 7.222130E+07, 7.432351E+07, 7.647348E+07,
    7.867203E+07, 8.091997E+07, 8.321813E+07, 8.556722E+07, 8.796816E+07, 9.042171E+07,
    9.292876E+07, 9.549011E+07, 9.810667E+07, 1.007792E+08, 1.035087E+08, 1.062959E+08,
    1.091418E+08, 1.120472E+08, 1.150131E+08, 1.180403E+08, 1.211298E+08, 1.242826E+08,
    1.274995E+08, 1.307815E+08, 1.341295E+08, 1.375446E+08, 1.410276E+08, 1.445796E+08,
    1.482015E+08, 1.518944E+08, 1.556593E+08, 1.594971E+08, 1.634089E+08, 1.673958E+08,
    1.714587E+08, 1.755987E+08, 1.798170E+08, 1.841145E+08, 1.884924E+08, 1.929516E+08,
    1.974934E+08, 2.021189E+08,
])

#  ---------------------- M = 3, I = 18 ---------------------------
M = 3
I = 18
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.837627E+02, 1.387967E+04, 3.909803E+04, 7.174973E+04, 1.104283E+05, 1.543309E+05,
    2.029434E+05, 2.559580E+05, 3.132406E+05, 3.748050E+05, 4.407836E+05, 5.114192E+05,
    5.870305E+05, 6.680003E+05, 7.547560E+05, 8.477726E+05, 9.475368E+05, 1.054582E+06,
    1.169449E+06, 1.292692E+06, 1.424898E+06, 1.566662E+06, 1.718594E+06, 1.881319E+06,
    2.055491E+06, 2.241760E+06, 2.440823E+06, 2.653357E+06, 2.880106E+06, 3.121812E+06,
    3.379209E+06, 3.653089E+06, 3.944236E+06, 4.253526E+06, 4.581753E+06, 4.929803E+06,
    5.298553E+06, 5.688930E+06, 6.101879E+06, 6.538337E+06, 6.999325E+06, 7.485814E+06,
    7.998834E+06, 8.539473E+06, 9.108804E+06, 9.707966E+06, 1.033803E+07, 1.100023E+07,
    1.169569E+07, 1.242568E+07, 1.319142E+07, 1.399419E+07, 1.483526E+07, 1.571596E+07,
    1.663761E+07, 1.760170E+07, 1.860954E+07, 1.966258E+07, 2.076236E+07, 2.191031E+07,
    2.310791E+07, 2.435683E+07, 2.565854E+07, 2.701471E+07, 2.842701E+07, 2.989710E+07,
    3.142661E+07, 3.301744E+07, 3.467120E+07, 3.638967E+07, 3.817485E+07, 4.002844E+07,
    4.195248E+07, 4.394874E+07, 4.601924E+07, 4.816607E+07, 5.039106E+07, 5.269646E+07,
    5.508422E+07, 5.755659E+07, 6.011559E+07, 6.276353E+07, 6.550260E+07, 6.833496E+07,
    7.126300E+07, 7.428912E+07, 7.741560E+07, 8.064479E+07, 8.397913E+07, 8.742121E+07,
    9.097347E+07, 9.463835E+07, 9.841865E+07, 1.023169E+08, 1.063354E+08, 1.104775E+08,
    1.147455E+08, 1.191421E+08, 1.236703E+08, 1.283330E+08, 1.331328E+08, 1.380728E+08,
    1.431559E+08, 1.483852E+08, 1.537637E+08, 1.592942E+08, 1.649801E+08, 1.708245E+08,
    1.768305E+08, 1.830015E+08, 1.893405E+08, 1.958510E+08, 2.025364E+08, 2.093999E+08,
    2.164450E+08, 2.236751E+08, 2.310939E+08, 2.387049E+08, 2.465116E+08, 2.545176E+08,
    2.627268E+08, 2.711426E+08, 2.797689E+08, 2.886098E+08, 2.976687E+08, 3.069497E+08,
    3.164567E+08, 3.261937E+08, 3.361646E+08, 3.463737E+08, 3.568248E+08, 3.675224E+08,
    3.784706E+08, 3.896733E+08, 4.011353E+08, 4.128607E+08, 4.248538E+08, 4.371189E+08,
    4.496609E+08, 4.624839E+08, 4.755926E+08, 4.889916E+08, 5.026858E+08, 5.166797E+08,
    5.309777E+08, 5.455850E+08, 5.605064E+08, 5.757467E+08, 5.913112E+08, 6.072043E+08,
    6.234311E+08, 6.399973E+08, 6.569072E+08, 6.741666E+08, 6.917803E+08, 7.097538E+08,
    7.280923E+08, 7.468014E+08, 7.658867E+08, 7.853528E+08, 8.052060E+08, 8.254517E+08,
    8.460953E+08, 8.671429E+08, 8.885996E+08, 9.104718E+08, 9.327650E+08, 9.554851E+08,
    9.786380E+08, 1.002230E+09, 1.026267E+09, 1.050754E+09, 1.075698E+09, 1.101107E+09,
    1.126984E+09, 1.153337E+09,
])

#  ---------------------- M = 4, I = 1 ---------------------------
M = 4
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.834054E+01, 3.016016E+02, 6.001924E+02, 8.988074E+02, 1.197498E+03, 1.496709E+03,
    1.797823E+03, 2.103285E+03, 2.416237E+03, 2.740046E+03, 3.077980E+03, 3.433067E+03,
    3.808070E+03, 4.205517E+03, 4.627758E+03, 5.077011E+03, 5.555411E+03, 6.065047E+03,
    6.607987E+03, 7.186308E+03, 7.802105E+03, 8.457512E+03, 9.154706E+03, 9.895923E+03,
    1.068346E+04, 1.151968E+04, 1.240701E+04, 1.334798E+04, 1.434516E+04, 1.540125E+04,
    1.651898E+04, 1.770123E+04, 1.895091E+04, 2.027106E+04, 2.166480E+04, 2.313534E+04,
    2.468602E+04, 2.632023E+04, 2.804150E+04, 2.985343E+04, 3.175977E+04, 3.376432E+04,
    3.587102E+04, 3.808392E+04, 4.040717E+04, 4.284502E+04, 4.540186E+04, 4.808216E+04,
    5.089054E+04, 5.383169E+04, 5.691045E+04, 6.013176E+04, 6.350068E+04, 6.702236E+04,
    7.070211E+04, 7.454531E+04, 7.855746E+04, 8.274419E+04, 8.711122E+04, 9.166437E+04,
    9.640958E+04, 1.013529E+05, 1.065004E+05, 1.118584E+05, 1.174331E+05, 1.232310E+05,
    1.292585E+05, 1.355222E+05, 1.420287E+05, 1.487848E+05, 1.557971E+05, 1.630726E+05,
    1.706179E+05, 1.784402E+05, 1.865463E+05, 1.949432E+05, 2.036378E+05, 2.126373E+05,
    2.219487E+05, 2.315789E+05, 2.415351E+05, 2.518242E+05, 2.624534E+05, 2.734295E+05,
    2.847595E+05, 2.964505E+05, 3.085092E+05, 3.209425E+05, 3.337573E+05, 3.469602E+05,
    3.605579E+05, 3.745571E+05, 3.889642E+05, 4.037857E+05, 4.190280E+05, 4.346973E+05,
    4.507998E+05, 4.673415E+05, 4.843284E+05, 5.017665E+05, 5.196613E+05, 5.380185E+05,
    5.568437E+05, 5.761421E+05, 5.959191E+05, 6.161798E+05, 6.369291E+05, 6.581719E+05,
    6.799128E+05, 7.021566E+05, 7.249075E+05, 7.481698E+05, 7.719478E+05, 7.962454E+05,
    8.210663E+05, 8.464145E+05, 8.722932E+05, 8.987061E+05, 9.256562E+05, 9.531467E+05,
    9.811806E+05, 1.009761E+06, 1.038889E+06, 1.068569E+06, 1.098803E+06, 1.129593E+06,
    1.160940E+06, 1.192847E+06, 1.225316E+06, 1.258347E+06, 1.291944E+06, 1.326106E+06,
    1.360835E+06, 1.396133E+06, 1.431999E+06, 1.468436E+06, 1.505443E+06, 1.543020E+06,
    1.581169E+06, 1.619890E+06, 1.659183E+06, 1.699047E+06, 1.739482E+06, 1.780489E+06,
    1.822067E+06, 1.864215E+06, 1.906934E+06, 1.950221E+06, 1.994077E+06, 2.038500E+06,
    2.083489E+06, 2.129044E+06, 2.175162E+06, 2.221843E+06, 2.269086E+06, 2.316888E+06,
    2.365247E+06, 2.414164E+06, 2.463635E+06, 2.513658E+06, 2.564232E+06, 2.615355E+06,
    2.667024E+06, 2.719237E+06, 2.771992E+06, 2.825287E+06, 2.879119E+06, 2.933486E+06,
    2.988385E+06, 3.043814E+06, 3.099769E+06, 3.156248E+06, 3.213249E+06, 3.270768E+06,
    3.328803E+06, 3.387350E+06, 3.446406E+06, 3.505969E+06, 3.566036E+06, 3.626603E+06,
    3.687666E+06, 3.749224E+06, 3.811272E+06, 3.873807E+06, 3.936826E+06, 4.000325E+06,
    4.064302E+06, 4.128752E+06, 4.193672E+06, 4.259058E+06, 4.324908E+06, 4.391217E+06,
    4.457982E+06, 4.525200E+06, 4.592866E+06, 4.660977E+06, 4.729530E+06, 4.798520E+06,
    4.867944E+06, 4.937799E+06, 5.008080E+06, 5.078784E+06, 5.149907E+06, 5.221446E+06,
    5.293396E+06, 5.365753E+06, 5.438515E+06, 5.511677E+06, 5.585236E+06, 5.659187E+06,
    5.733527E+06, 5.808252E+06, 5.883358E+06, 5.958842E+06, 6.034699E+06, 6.110926E+06,
    6.187520E+06, 6.264475E+06, 6.341789E+06, 6.419458E+06, 6.497478E+06, 6.575844E+06,
    6.654554E+06, 6.733604E+06, 6.812990E+06, 6.892708E+06, 6.972754E+06, 7.053125E+06,
    7.133816E+06, 7.214826E+06, 7.296148E+06, 7.377781E+06, 7.459719E+06, 7.541961E+06,
    7.624501E+06, 7.707336E+06, 7.790463E+06, 7.873879E+06, 7.957578E+06, 8.041559E+06,
    8.125817E+06, 8.210349E+06, 8.295151E+06, 8.380220E+06, 8.465552E+06, 8.551144E+06,
    8.636993E+06, 8.723094E+06, 8.809445E+06, 8.896042E+06, 8.982882E+06,
])

#  ---------------------- M = 4, I = 2 ---------------------------
M = 4
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.222856E+01, 2.011405E+02, 4.003536E+02, 5.996615E+02, 7.991092E+02, 9.990614E+02,
    1.200554E+03, 1.405397E+03, 1.615831E+03, 1.834200E+03, 2.062784E+03, 2.303629E+03,
    2.558624E+03, 2.829471E+03, 3.117766E+03, 3.425050E+03, 3.752753E+03, 4.102350E+03,
    4.475234E+03, 4.872840E+03, 5.296650E+03, 5.748165E+03, 6.228913E+03, 6.740417E+03,
    7.284339E+03, 7.862305E+03, 8.476054E+03, 9.127358E+03, 9.818007E+03, 1.054988E+04,
    1.132500E+04, 1.214527E+04, 1.301286E+04, 1.392980E+04, 1.489831E+04, 1.592072E+04,
    1.699933E+04, 1.813650E+04, 1.933465E+04, 2.059648E+04, 2.192444E+04, 2.332135E+04,
    2.478986E+04, 2.633284E+04, 2.795313E+04, 2.965383E+04, 3.143800E+04, 3.330869E+04,
    3.526918E+04, 3.732275E+04, 3.947275E+04, 4.172267E+04, 4.407610E+04, 4.653671E+04,
    4.910823E+04, 5.179425E+04, 5.459898E+04, 5.752616E+04, 6.058007E+04, 6.376470E+04,
    6.708449E+04, 7.054360E+04, 7.414663E+04, 7.789815E+04, 8.180281E+04, 8.586522E+04,
    9.009033E+04, 9.448303E+04, 9.904846E+04, 1.037917E+05, 1.087180E+05, 1.138327E+05,
    1.191414E+05, 1.246497E+05, 1.303630E+05, 1.362872E+05, 1.424284E+05, 1.487924E+05,
    1.553853E+05, 1.622135E+05, 1.692834E+05, 1.766012E+05, 1.841737E+05, 1.920075E+05,
    2.001097E+05, 2.084867E+05, 2.171461E+05, 2.260948E+05, 2.353401E+05, 2.448897E+05,
    2.547509E+05, 2.649314E+05, 2.754390E+05, 2.862820E+05, 2.974681E+05, 3.090057E+05,
    3.209029E+05, 3.331687E+05, 3.458111E+05, 3.588390E+05, 3.722616E+05, 3.860877E+05,
    4.003264E+05, 4.149873E+05, 4.300793E+05, 4.456126E+05, 4.615966E+05, 4.780413E+05,
    4.949566E+05, 5.123527E+05, 5.302399E+05, 5.486289E+05, 5.675301E+05, 5.869545E+05,
    6.069124E+05, 6.274157E+05, 6.484752E+05, 6.701022E+05, 6.923088E+05, 7.151059E+05,
    7.385061E+05, 7.625212E+05, 7.871631E+05, 8.124445E+05, 8.383776E+05, 8.649755E+05,
    8.922511E+05, 9.202168E+05, 9.488864E+05, 9.782731E+05, 1.008390E+06, 1.039252E+06,
    1.070871E+06, 1.103263E+06, 1.136442E+06, 1.170421E+06, 1.205216E+06, 1.240841E+06,
    1.277312E+06, 1.314642E+06, 1.352849E+06, 1.391947E+06, 1.431951E+06, 1.472878E+06,
    1.514745E+06, 1.557566E+06, 1.601359E+06, 1.646140E+06, 1.691926E+06, 1.738734E+06,
    1.786582E+06, 1.835486E+06, 1.885465E+06, 1.936536E+06, 1.988717E+06, 2.042026E+06,
    2.096482E+06, 2.152104E+06, 2.208910E+06, 2.266919E+06, 2.326151E+06, 2.386625E+06,
    2.448361E+06, 2.511378E+06, 2.575697E+06, 2.641338E+06, 2.708322E+06, 2.776668E+06,
    2.846400E+06, 2.917536E+06, 2.990100E+06, 3.064111E+06, 3.139593E+06, 3.216567E+06,
    3.295056E+06, 3.375082E+06,
])

#  ---------------------- M = 4, I = 3 ---------------------------
M = 4
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.256526E+01, 2.080869E+02, 4.142500E+02, 6.205111E+02, 8.269083E+02, 1.033772E+03,
    1.242093E+03, 1.453602E+03, 1.670525E+03, 1.895214E+03, 2.129981E+03, 2.376914E+03,
    2.637982E+03, 2.914901E+03, 3.209368E+03, 3.522887E+03, 3.857001E+03, 4.213185E+03,
    4.592889E+03, 4.997581E+03, 5.428733E+03, 5.887874E+03, 6.376560E+03, 6.896343E+03,
    7.448865E+03, 8.035829E+03, 8.658960E+03, 9.319999E+03, 1.002082E+04, 1.076333E+04,
    1.154942E+04, 1.238119E+04, 1.326072E+04, 1.419008E+04, 1.517143E+04, 1.620721E+04,
    1.729969E+04, 1.845127E+04, 1.966438E+04, 2.094161E+04, 2.228561E+04, 2.369906E+04,
    2.518473E+04, 2.674545E+04, 2.838410E+04, 3.010380E+04, 3.190749E+04, 3.379832E+04,
    3.577964E+04, 3.785476E+04, 4.002697E+04, 4.229982E+04, 4.467683E+04, 4.716170E+04,
    4.975823E+04, 5.247004E+04, 5.530117E+04, 5.825564E+04, 6.133755E+04, 6.455106E+04,
    6.790052E+04, 7.139005E+04, 7.502442E+04, 7.880813E+04, 8.274562E+04, 8.684195E+04,
    9.110174E+04, 9.553009E+04, 1.001320E+05, 1.049128E+05, 1.098775E+05, 1.150316E+05,
    1.203805E+05, 1.259299E+05, 1.316857E+05, 1.376532E+05, 1.438385E+05, 1.502479E+05,
    1.568873E+05, 1.637628E+05, 1.708812E+05, 1.782486E+05, 1.858719E+05, 1.937574E+05,
    2.019124E+05, 2.103435E+05, 2.190582E+05, 2.280632E+05, 2.373661E+05, 2.469743E+05,
    2.568955E+05, 2.671372E+05, 2.777073E+05, 2.886138E+05, 2.998648E+05, 3.114686E+05,
    3.234334E+05, 3.357677E+05, 3.484802E+05, 3.615796E+05, 3.750749E+05, 3.889752E+05,
    4.032894E+05, 4.180269E+05, 4.331973E+05, 4.488102E+05, 4.648751E+05, 4.814022E+05,
    4.984014E+05, 5.158830E+05, 5.338571E+05, 5.523341E+05, 5.713253E+05, 5.908406E+05,
    6.108916E+05, 6.314891E+05, 6.526444E+05, 6.743689E+05, 6.966742E+05, 7.195719E+05,
    7.430743E+05, 7.671928E+05, 7.919402E+05, 8.173284E+05, 8.433701E+05, 8.700783E+05,
    8.974653E+05, 9.255449E+05, 9.543294E+05, 9.838329E+05, 1.014069E+06, 1.045050E+06,
    1.076792E+06, 1.109308E+06, 1.142611E+06, 1.176718E+06, 1.211642E+06, 1.247397E+06,
    1.284000E+06, 1.321465E+06, 1.359807E+06, 1.399043E+06, 1.439187E+06, 1.480256E+06,
    1.522265E+06, 1.565232E+06, 1.609171E+06, 1.654101E+06, 1.700038E+06, 1.746998E+06,
    1.795000E+06, 1.844060E+06, 1.894197E+06, 1.945428E+06, 1.997770E+06, 2.051243E+06,
    2.105865E+06, 2.161654E+06, 2.218630E+06, 2.276810E+06, 2.336216E+06, 2.396864E+06,
    2.458777E+06, 2.521974E+06, 2.586474E+06, 2.652298E+06, 2.719466E+06, 2.788000E+06,
    2.857920E+06, 2.929247E+06, 3.002003E+06, 3.076209E+06, 3.151888E+06, 3.229061E+06,
    3.307750E+06, 3.387980E+06,
])

#  ---------------------- M = 4, I = 4 ---------------------------
M = 4
I = 4
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.920009E+01, 3.193816E+02, 6.358818E+02, 9.525320E+02, 1.269392E+03, 1.586983E+03,
    1.906819E+03, 2.231621E+03, 2.564811E+03, 2.910018E+03, 3.270780E+03, 3.650417E+03,
    4.051873E+03, 4.477876E+03, 4.931017E+03, 5.413766E+03, 5.928387E+03, 6.477164E+03,
    7.062454E+03, 7.686493E+03, 8.351633E+03, 9.060190E+03, 9.814583E+03, 1.061722E+04,
    1.147081E+04, 1.237778E+04, 1.334087E+04, 1.436287E+04, 1.544662E+04, 1.659506E+04,
    1.781131E+04, 1.909836E+04, 2.045953E+04, 2.189821E+04, 2.341770E+04, 2.502151E+04,
    2.671353E+04, 2.849723E+04, 3.037651E+04, 3.235551E+04, 3.443819E+04, 3.662862E+04,
    3.893132E+04, 4.135055E+04, 4.389090E+04, 4.655709E+04, 4.935379E+04, 5.228607E+04,
    5.535866E+04, 5.857697E+04, 6.194628E+04, 6.547190E+04, 6.915948E+04, 7.301461E+04,
    7.704295E+04, 8.125083E+04, 8.564401E+04, 9.022885E+04, 9.501173E+04, 9.999909E+04,
    1.051977E+05, 1.106143E+05, 1.162557E+05, 1.221292E+05, 1.282420E+05, 1.346015E+05,
    1.412153E+05, 1.480910E+05, 1.552365E+05, 1.626599E+05, 1.703693E+05, 1.783731E+05,
    1.866800E+05, 1.952985E+05, 2.042375E+05, 2.135062E+05, 2.231135E+05, 2.330688E+05,
    2.433819E+05, 2.540625E+05, 2.651203E+05, 2.765653E+05, 2.884082E+05, 3.006592E+05,
    3.133289E+05, 3.264282E+05, 3.399683E+05, 3.539598E+05, 3.684149E+05, 3.833445E+05,
    3.987610E+05, 4.146759E+05, 4.311017E+05, 4.480505E+05, 4.655352E+05, 4.835685E+05,
    5.021632E+05, 5.213328E+05, 5.410906E+05, 5.614503E+05, 5.824256E+05, 6.040307E+05,
    6.262797E+05, 6.491874E+05, 6.727680E+05, 6.970374E+05, 7.220099E+05, 7.477012E+05,
    7.741267E+05, 8.013024E+05, 8.292446E+05, 8.579696E+05, 8.874938E+05, 9.178335E+05,
    9.490066E+05, 9.810301E+05, 1.013921E+06, 1.047698E+06, 1.082378E+06, 1.117980E+06,
    1.154522E+06, 1.192024E+06, 1.230503E+06, 1.269980E+06, 1.310473E+06, 1.352004E+06,
    1.394590E+06, 1.438254E+06, 1.483015E+06, 1.528894E+06, 1.575913E+06, 1.624092E+06,
    1.673454E+06, 1.724020E+06, 1.775813E+06, 1.828855E+06, 1.883168E+06, 1.938776E+06,
    1.995702E+06, 2.053969E+06, 2.113602E+06, 2.174624E+06, 2.237060E+06, 2.300936E+06,
    2.366274E+06, 2.433103E+06, 2.501445E+06, 2.571329E+06, 2.642780E+06, 2.715823E+06,
    2.790487E+06, 2.866799E+06, 2.944785E+06, 3.024473E+06, 3.105892E+06, 3.189071E+06,
    3.274037E+06, 3.360820E+06, 3.449449E+06, 3.539953E+06, 3.632364E+06, 3.726710E+06,
    3.823023E+06, 3.921334E+06, 4.021673E+06, 4.124073E+06, 4.228565E+06, 4.335183E+06,
    4.443957E+06, 4.554922E+06, 4.668111E+06, 4.783556E+06, 4.901294E+06, 5.021357E+06,
    5.143780E+06, 5.268599E+06,
])

#  ---------------------- M = 4, I = 5 ---------------------------
M = 4
I = 5
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.126859E+02, 1.864509E+03, 3.711701E+03, 5.559770E+03, 7.409060E+03, 9.262436E+03,
    1.112884E+04, 1.302350E+04, 1.496642E+04, 1.697860E+04, 1.908053E+04, 2.129100E+04,
    2.362782E+04, 2.610612E+04, 2.874132E+04, 3.154718E+04, 3.453696E+04, 3.772415E+04,
    4.112140E+04, 4.474253E+04, 4.860004E+04, 5.270834E+04, 5.708061E+04, 6.173070E+04,
    6.667417E+04, 7.192540E+04, 7.749930E+04, 8.341287E+04, 8.968139E+04, 9.632295E+04,
    1.033539E+05, 1.107929E+05, 1.186581E+05, 1.269691E+05, 1.357447E+05, 1.450058E+05,
    1.547736E+05, 1.650686E+05, 1.759141E+05, 1.873318E+05, 1.993451E+05, 2.119782E+05,
    2.252566E+05, 2.392045E+05, 2.538486E+05, 2.692152E+05, 2.853310E+05, 3.022263E+05,
    3.199273E+05, 3.384653E+05, 3.578704E+05, 3.781728E+05, 3.994046E+05, 4.215990E+05,
    4.447877E+05, 4.690061E+05, 4.942895E+05, 5.206713E+05, 5.481903E+05, 5.768826E+05,
    6.067872E+05, 6.379417E+05, 6.703877E+05, 7.041640E+05, 7.393128E+05, 7.758767E+05,
    8.138982E+05, 8.534227E+05, 8.944954E+05, 9.371609E+05, 9.814658E+05, 1.027460E+06,
    1.075192E+06, 1.124709E+06, 1.176064E+06, 1.229308E+06, 1.284495E+06, 1.341676E+06,
    1.400908E+06, 1.462245E+06, 1.525743E+06, 1.591463E+06, 1.659462E+06, 1.729799E+06,
    1.802536E+06, 1.877734E+06, 1.955456E+06, 2.035768E+06, 2.118733E+06, 2.204417E+06,
    2.292890E+06, 2.384219E+06, 2.478473E+06, 2.575724E+06, 2.676044E+06, 2.779507E+06,
    2.886184E+06, 2.996153E+06, 3.109493E+06, 3.226277E+06, 3.346586E+06, 3.470502E+06,
    3.598108E+06, 3.729483E+06, 3.864712E+06, 4.003881E+06, 4.147080E+06, 4.294392E+06,
    4.445907E+06, 4.601719E+06, 4.761916E+06, 4.926595E+06, 5.095849E+06, 5.269773E+06,
    5.448463E+06, 5.632024E+06, 5.820548E+06, 6.014144E+06, 6.212908E+06, 6.416949E+06,
    6.626371E+06, 6.841280E+06, 7.061789E+06, 7.288003E+06, 7.520035E+06, 7.758000E+06,
    8.002011E+06, 8.252184E+06, 8.508638E+06, 8.771489E+06, 9.040859E+06, 9.316869E+06,
    9.599643E+06, 9.889311E+06, 1.018599E+07, 1.048982E+07, 1.080092E+07, 1.111943E+07,
    1.144548E+07, 1.177920E+07, 1.212074E+07, 1.247022E+07, 1.282778E+07, 1.319359E+07,
    1.356776E+07, 1.395045E+07, 1.434180E+07, 1.474197E+07, 1.515110E+07, 1.556933E+07,
    1.599684E+07, 1.643377E+07, 1.688027E+07, 1.733652E+07, 1.780266E+07, 1.827886E+07,
    1.876528E+07, 1.926210E+07, 1.976946E+07, 2.028756E+07, 2.081655E+07, 2.135661E+07,
    2.190792E+07, 2.247066E+07, 2.304498E+07, 2.363109E+07, 2.422917E+07, 2.483940E+07,
    2.546196E+07, 2.609704E+07, 2.674484E+07, 2.740554E+07, 2.807935E+07, 2.876645E+07,
    2.946704E+07, 3.018133E+07,
])

#  ---------------------- M = 5, I = 1 ---------------------------
M = 5
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[3]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.011870E+00, 7.573520E+00, 1.480009E+01, 2.203056E+01, 2.926249E+01, 3.649540E+01,
    4.372913E+01, 5.096362E+01, 5.819883E+01, 6.543475E+01, 7.267137E+01, 7.990872E+01,
    8.714687E+01, 9.438597E+01, 1.016263E+02, 1.088684E+02, 1.161130E+02, 1.233610E+02,
    1.306138E+02, 1.378730E+02, 1.451405E+02, 1.524185E+02, 1.597093E+02, 1.670157E+02,
    1.743404E+02, 1.816863E+02, 1.890565E+02, 1.964539E+02, 2.038817E+02, 2.113428E+02,
    2.188405E+02, 2.263775E+02, 2.339567E+02, 2.415811E+02, 2.492532E+02, 2.569757E+02,
    2.647511E+02, 2.725817E+02, 2.804698E+02, 2.884176E+02, 2.964271E+02, 3.045004E+02,
    3.126391E+02, 3.208450E+02, 3.291199E+02, 3.374652E+02, 3.458824E+02, 3.543729E+02,
    3.629381E+02, 3.715790E+02, 3.802970E+02, 3.890930E+02, 3.979682E+02, 4.069234E+02,
    4.159597E+02, 4.250779E+02, 4.342787E+02, 4.435630E+02, 4.529316E+02, 4.623850E+02,
    4.719240E+02, 4.815492E+02, 4.912611E+02, 5.010603E+02, 5.109473E+02, 5.209227E+02,
    5.309869E+02, 5.411404E+02, 5.513836E+02, 5.617168E+02, 5.721405E+02, 5.826552E+02,
    5.932609E+02, 6.039583E+02, 6.147475E+02, 6.256289E+02, 6.366028E+02, 6.476694E+02,
    6.588291E+02, 6.700820E+02, 6.814285E+02, 6.928687E+02, 7.044030E+02, 7.160315E+02,
    7.277544E+02, 7.395719E+02, 7.514843E+02, 7.634917E+02, 7.755944E+02, 7.877924E+02,
    8.000860E+02, 8.124754E+02, 8.249606E+02, 8.375419E+02, 8.502194E+02, 8.629933E+02,
    8.758637E+02, 8.888307E+02, 9.018945E+02, 9.150553E+02, 9.283131E+02, 9.416680E+02,
    9.551203E+02, 9.686700E+02, 9.823173E+02, 9.960622E+02, 1.009905E+03, 1.023845E+03,
    1.037884E+03, 1.052021E+03, 1.066256E+03, 1.080589E+03, 1.095020E+03, 1.109551E+03,
    1.124179E+03, 1.138907E+03, 1.153733E+03, 1.168658E+03, 1.183682E+03, 1.198806E+03,
    1.214028E+03, 1.229350E+03, 1.244771E+03, 1.260292E+03, 1.275912E+03, 1.291632E+03,
    1.307451E+03, 1.323370E+03, 1.339390E+03, 1.355509E+03, 1.371728E+03, 1.388047E+03,
    1.404467E+03, 1.420987E+03, 1.437607E+03, 1.454327E+03, 1.471149E+03, 1.488070E+03,
    1.505093E+03, 1.522216E+03, 1.539440E+03, 1.556765E+03, 1.574191E+03, 1.591718E+03,
    1.609346E+03, 1.627076E+03, 1.644907E+03, 1.662839E+03, 1.680872E+03, 1.699007E+03,
    1.717244E+03, 1.735582E+03, 1.754022E+03, 1.772564E+03, 1.791208E+03, 1.809954E+03,
    1.828801E+03, 1.847751E+03, 1.866803E+03, 1.885957E+03, 1.905214E+03, 1.924573E+03,
    1.944034E+03, 1.963598E+03, 1.983264E+03, 2.003034E+03, 2.022906E+03, 2.042880E+03,
    2.062958E+03, 2.083138E+03, 2.103422E+03, 2.123809E+03, 2.144299E+03, 2.164892E+03,
    2.185588E+03, 2.206388E+03, 2.227291E+03, 2.248298E+03, 2.269408E+03, 2.290622E+03,
    2.311939E+03, 2.333361E+03, 2.354886E+03, 2.376515E+03, 2.398249E+03, 2.420086E+03,
    2.442027E+03, 2.464073E+03, 2.486223E+03, 2.508477E+03, 2.530836E+03, 2.553299E+03,
    2.575867E+03, 2.598539E+03, 2.621316E+03, 2.644198E+03, 2.667185E+03, 2.690276E+03,
    2.713473E+03, 2.736775E+03, 2.760181E+03, 2.783693E+03, 2.807310E+03, 2.831033E+03,
    2.854861E+03, 2.878794E+03, 2.902833E+03, 2.926977E+03, 2.951227E+03, 2.975583E+03,
    3.000045E+03, 3.024612E+03, 3.049286E+03, 3.074065E+03, 3.098951E+03, 3.123943E+03,
    3.149041E+03, 3.174245E+03, 3.199555E+03, 3.224973E+03, 3.250496E+03, 3.276126E+03,
    3.301863E+03, 3.327706E+03, 3.353657E+03, 3.379714E+03, 3.405878E+03, 3.432149E+03,
    3.458527E+03, 3.485012E+03, 3.511605E+03, 3.538304E+03, 3.565111E+03, 3.592026E+03,
    3.619048E+03, 3.646177E+03, 3.673414E+03, 3.700759E+03, 3.728212E+03, 3.755772E+03,
    3.783440E+03, 3.811216E+03, 3.839101E+03, 3.867093E+03, 3.895193E+03, 3.923402E+03,
    3.951719E+03, 3.980145E+03, 4.008679E+03, 4.037321E+03, 4.066072E+03, 4.094932E+03,
    4.123900E+03, 4.152977E+03, 4.182163E+03, 4.211458E+03, 4.240862E+03, 4.270375E+03,
    4.299997E+03, 4.329729E+03, 4.359569E+03, 4.389519E+03, 4.419579E+03, 4.449747E+03,
    4.480026E+03, 4.510414E+03, 4.540912E+03, 4.571519E+03, 4.602236E+03, 4.633064E+03,
    4.664001E+03, 4.695048E+03, 4.726205E+03, 4.757472E+03, 4.788850E+03, 4.820338E+03,
    4.851936E+03, 4.883645E+03, 4.915464E+03, 4.947394E+03, 4.979434E+03, 5.011585E+03,
    5.043847E+03, 5.076220E+03, 5.108703E+03, 5.141298E+03, 5.174003E+03, 5.206820E+03,
    5.239748E+03, 5.272787E+03, 5.305937E+03, 5.339199E+03, 5.372572E+03, 5.406056E+03,
    5.439652E+03, 5.473360E+03, 5.507179E+03, 5.541110E+03, 5.575153E+03, 5.609308E+03,
    5.643574E+03, 5.677953E+03, 5.712444E+03, 5.747047E+03, 5.781762E+03, 5.816589E+03,
    5.851528E+03, 5.886580E+03, 5.921745E+03, 5.957021E+03, 5.992411E+03, 6.027913E+03,
    6.063527E+03, 6.099255E+03, 6.135095E+03, 6.171048E+03, 6.207114E+03, 6.243293E+03,
    6.279585E+03, 6.315990E+03, 6.352508E+03, 6.389140E+03, 6.425885E+03, 6.462743E+03,
    6.499714E+03, 6.536799E+03, 6.573998E+03, 6.611310E+03, 6.648735E+03, 6.686275E+03,
    6.723928E+03, 6.761694E+03, 6.799575E+03, 6.837570E+03, 6.875678E+03, 6.913901E+03,
    6.952238E+03, 6.990688E+03, 7.029253E+03, 7.067933E+03, 7.106726E+03, 7.145634E+03,
    7.184656E+03, 7.223793E+03, 7.263044E+03, 7.302410E+03, 7.341891E+03, 7.381486E+03,
    7.421195E+03, 7.461020E+03, 7.500959E+03, 7.541014E+03, 7.581183E+03, 7.621467E+03,
    7.661866E+03, 7.702380E+03, 7.743010E+03, 7.783754E+03, 7.824614E+03, 7.865589E+03,
    7.906680E+03, 7.947885E+03, 7.989206E+03, 8.030643E+03, 8.072195E+03, 8.113863E+03,
    8.155646E+03, 8.197545E+03, 8.239560E+03, 8.281690E+03, 8.323936E+03, 8.366298E+03,
    8.408776E+03, 8.451369E+03, 8.494079E+03, 8.536905E+03, 8.579847E+03, 8.622904E+03,
    8.666078E+03, 8.709368E+03, 8.752775E+03, 8.796297E+03, 8.839936E+03, 8.883691E+03,
    8.927563E+03, 8.971551E+03, 9.015655E+03, 9.059876E+03, 9.104213E+03, 9.148667E+03,
    9.193238E+03, 9.237925E+03, 9.282729E+03, 9.327650E+03, 9.372687E+03, 9.417841E+03,
    9.463112E+03, 9.508500E+03, 9.554005E+03, 9.599626E+03, 9.645365E+03, 9.691221E+03,
    9.737193E+03, 9.783283E+03, 9.829489E+03, 9.875813E+03, 9.922254E+03, 9.968812E+03,
    1.001549E+04, 1.006228E+04, 1.010919E+04, 1.015622E+04, 1.020336E+04, 1.025062E+04,
    1.029800E+04, 1.034550E+04, 1.039311E+04, 1.044085E+04, 1.048869E+04, 1.053666E+04,
    1.058475E+04, 1.063295E+04, 1.068127E+04, 1.072970E+04, 1.077826E+04, 1.082693E+04,
    1.087572E+04, 1.092463E+04, 1.097365E+04, 1.102280E+04, 1.107206E+04, 1.112144E+04,
    1.117093E+04, 1.122055E+04, 1.127028E+04, 1.132013E+04, 1.137010E+04, 1.142018E+04,
    1.147039E+04, 1.152071E+04, 1.157115E+04, 1.162170E+04, 1.167238E+04, 1.172317E+04,
    1.177409E+04, 1.182512E+04, 1.187626E+04, 1.192753E+04, 1.197891E+04, 1.203042E+04,
    1.208204E+04,
])

#  ---------------------- M = 5, I = 2 ---------------------------
M = 5
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[3]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.030290E+00, 1.581148E+01, 3.093044E+01, 4.605695E+01, 6.118637E+01, 7.631778E+01,
    9.145088E+01, 1.065855E+02, 1.217217E+02, 1.368594E+02, 1.519985E+02, 1.671392E+02,
    1.822816E+02, 1.974261E+02, 2.125736E+02, 2.277251E+02, 2.428826E+02, 2.580484E+02,
    2.732257E+02, 2.884181E+02, 3.036302E+02, 3.188669E+02, 3.341338E+02, 3.494367E+02,
    3.647819E+02, 3.801760E+02, 3.956256E+02, 4.111372E+02, 4.267178E+02, 4.423738E+02,
    4.581119E+02, 4.739383E+02, 4.898593E+02, 5.058809E+02, 5.220087E+02, 5.382483E+02,
    5.546050E+02, 5.710838E+02, 5.876893E+02, 6.044262E+02, 6.212987E+02, 6.383109E+02,
    6.554665E+02, 6.727693E+02, 6.902226E+02, 7.078295E+02, 7.255932E+02, 7.435165E+02,
    7.616020E+02, 7.798524E+02, 7.982699E+02, 8.168568E+02, 8.356152E+02, 8.545471E+02,
    8.736544E+02, 8.929388E+02, 9.124020E+02, 9.320457E+02, 9.518712E+02, 9.718800E+02,
    9.920733E+02, 1.012453E+03, 1.033019E+03, 1.053773E+03, 1.074717E+03, 1.095851E+03,
    1.117176E+03, 1.138693E+03, 1.160404E+03, 1.182308E+03, 1.204406E+03, 1.226700E+03,
    1.249191E+03, 1.271877E+03, 1.294762E+03, 1.317844E+03, 1.341125E+03, 1.364605E+03,
    1.388285E+03, 1.412165E+03, 1.436246E+03, 1.460528E+03, 1.485012E+03, 1.509698E+03,
    1.534586E+03, 1.559678E+03, 1.584973E+03, 1.610472E+03, 1.636175E+03, 1.662083E+03,
    1.688195E+03, 1.714513E+03, 1.741037E+03, 1.767766E+03, 1.794701E+03, 1.821843E+03,
    1.849192E+03, 1.876748E+03, 1.904512E+03, 1.932483E+03, 1.960662E+03, 1.989049E+03,
    2.017644E+03, 2.046449E+03, 2.075462E+03, 2.104685E+03, 2.134116E+03, 2.163758E+03,
    2.193609E+03, 2.223671E+03, 2.253943E+03, 2.284425E+03, 2.315119E+03, 2.346023E+03,
    2.377138E+03, 2.408465E+03, 2.440003E+03, 2.471753E+03, 2.503715E+03, 2.535889E+03,
    2.568276E+03, 2.600875E+03, 2.633687E+03, 2.666711E+03, 2.699949E+03, 2.733400E+03,
    2.767065E+03, 2.800943E+03, 2.835035E+03, 2.869341E+03, 2.903861E+03, 2.938595E+03,
    2.973544E+03, 3.008708E+03, 3.044086E+03, 3.079680E+03, 3.115488E+03, 3.151512E+03,
    3.187752E+03, 3.224207E+03, 3.260878E+03, 3.297764E+03, 3.334867E+03, 3.372187E+03,
    3.409722E+03, 3.447475E+03, 3.485444E+03, 3.523630E+03, 3.562033E+03, 3.600653E+03,
    3.639491E+03, 3.678547E+03, 3.717820E+03, 3.757310E+03, 3.797019E+03, 3.836946E+03,
    3.877092E+03, 3.917456E+03, 3.958038E+03, 3.998839E+03, 4.039860E+03, 4.081099E+03,
    4.122557E+03, 4.164235E+03, 4.206133E+03, 4.248250E+03, 4.290587E+03, 4.333143E+03,
    4.375921E+03, 4.418918E+03, 4.462136E+03, 4.505574E+03, 4.549233E+03, 4.593113E+03,
    4.637214E+03, 4.681536E+03, 4.726079E+03, 4.770844E+03, 4.815831E+03, 4.861039E+03,
    4.906469E+03, 4.952121E+03, 4.997996E+03, 5.044093E+03, 5.090412E+03, 5.136954E+03,
    5.183719E+03, 5.230707E+03, 5.277917E+03, 5.325351E+03, 5.373009E+03, 5.420890E+03,
    5.468995E+03, 5.517323E+03, 5.565876E+03, 5.614652E+03, 5.663653E+03, 5.712879E+03,
    5.762329E+03, 5.812003E+03, 5.861903E+03, 5.912027E+03, 5.962377E+03, 6.012952E+03,
    6.063752E+03, 6.114778E+03, 6.166030E+03, 6.217508E+03, 6.269212E+03, 6.321142E+03,
    6.373298E+03, 6.425681E+03, 6.478290E+03, 6.531126E+03, 6.584189E+03, 6.637479E+03,
    6.690997E+03, 6.744742E+03, 6.798714E+03, 6.852914E+03, 6.907342E+03, 6.961997E+03,
    7.016881E+03, 7.071993E+03, 7.127334E+03, 7.182903E+03, 7.238700E+03, 7.294727E+03,
    7.350982E+03, 7.407467E+03, 7.464181E+03, 7.521124E+03, 7.578297E+03, 7.635699E+03,
    7.693332E+03, 7.751194E+03, 7.809287E+03, 7.867610E+03, 7.926163E+03, 7.984947E+03,
    8.043962E+03, 8.103207E+03, 8.162684E+03, 8.222391E+03, 8.282330E+03, 8.342501E+03,
    8.402903E+03, 8.463537E+03, 8.524402E+03, 8.585500E+03, 8.646830E+03, 8.708392E+03,
    8.770187E+03, 8.832214E+03, 8.894474E+03, 8.956967E+03, 9.019693E+03, 9.082652E+03,
    9.145844E+03, 9.209270E+03, 9.272930E+03, 9.336823E+03, 9.400950E+03, 9.465311E+03,
    9.529906E+03, 9.594736E+03, 9.659800E+03, 9.725098E+03, 9.790631E+03, 9.856400E+03,
    9.922403E+03, 9.988641E+03, 1.005511E+04, 1.012182E+04, 1.018877E+04, 1.025595E+04,
    1.032336E+04, 1.039102E+04, 1.045890E+04, 1.052703E+04, 1.059539E+04, 1.066399E+04,
    1.073282E+04, 1.080189E+04, 1.087120E+04, 1.094074E+04, 1.101052E+04, 1.108054E+04,
    1.115080E+04, 1.122129E+04, 1.129202E+04, 1.136299E+04, 1.143420E+04, 1.150565E+04,
    1.157733E+04, 1.164925E+04, 1.172141E+04, 1.179381E+04, 1.186645E+04, 1.193932E+04,
    1.201244E+04, 1.208579E+04, 1.215939E+04, 1.223322E+04, 1.230729E+04, 1.238161E+04,
    1.245616E+04, 1.253095E+04, 1.260598E+04, 1.268125E+04, 1.275676E+04, 1.283252E+04,
    1.290851E+04, 1.298474E+04, 1.306122E+04, 1.313793E+04, 1.321489E+04, 1.329208E+04,
    1.336952E+04, 1.344720E+04, 1.352512E+04, 1.360328E+04, 1.368169E+04, 1.376033E+04,
    1.383922E+04, 1.391835E+04, 1.399772E+04, 1.407733E+04, 1.415719E+04, 1.423729E+04,
    1.431763E+04, 1.439821E+04, 1.447903E+04, 1.456010E+04, 1.464141E+04, 1.472297E+04,
    1.480477E+04, 1.488681E+04, 1.496909E+04, 1.505162E+04, 1.513439E+04, 1.521740E+04,
    1.530066E+04, 1.538416E+04, 1.546791E+04, 1.555190E+04, 1.563614E+04, 1.572061E+04,
    1.580534E+04, 1.589030E+04, 1.597551E+04, 1.606097E+04, 1.614667E+04, 1.623262E+04,
    1.631881E+04, 1.640524E+04, 1.649192E+04, 1.657885E+04, 1.666602E+04, 1.675344E+04,
    1.684110E+04, 1.692901E+04, 1.701716E+04, 1.710556E+04, 1.719420E+04, 1.728309E+04,
    1.737222E+04, 1.746161E+04, 1.755123E+04, 1.764111E+04, 1.773123E+04, 1.782159E+04,
    1.791221E+04, 1.800306E+04, 1.809417E+04, 1.818552E+04, 1.827712E+04, 1.836897E+04,
    1.846106E+04, 1.855340E+04, 1.864598E+04, 1.873882E+04, 1.883190E+04, 1.892522E+04,
    1.901880E+04, 1.911262E+04, 1.920669E+04, 1.930100E+04, 1.939557E+04, 1.949038E+04,
    1.958544E+04, 1.968074E+04, 1.977629E+04, 1.987210E+04, 1.996815E+04, 2.006444E+04,
    2.016099E+04, 2.025778E+04, 2.035482E+04, 2.045211E+04, 2.054965E+04, 2.064743E+04,
    2.074547E+04, 2.084375E+04, 2.094228E+04, 2.104106E+04, 2.114008E+04, 2.123936E+04,
    2.133888E+04, 2.143865E+04, 2.153867E+04, 2.163894E+04, 2.173946E+04, 2.184022E+04,
    2.194124E+04, 2.204250E+04, 2.214402E+04, 2.224578E+04, 2.234779E+04, 2.245005E+04,
    2.255255E+04, 2.265531E+04, 2.275832E+04, 2.286157E+04, 2.296507E+04, 2.306883E+04,
    2.317283E+04, 2.327708E+04, 2.338158E+04, 2.348633E+04, 2.359133E+04, 2.369657E+04,
    2.380207E+04, 2.390782E+04, 2.401381E+04, 2.412006E+04, 2.422655E+04, 2.433329E+04,
    2.444028E+04, 2.454752E+04, 2.465502E+04, 2.476276E+04, 2.487074E+04, 2.497898E+04,
    2.508747E+04, 2.519621E+04, 2.530519E+04, 2.541443E+04, 2.552392E+04, 2.563365E+04,
    2.574363E+04,
])

#  ---------------------- M = 5, I = 3 ---------------------------
M = 5
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[3]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.015450E+00, 7.934590E+00, 1.552296E+01, 2.311511E+01, 3.070871E+01, 3.830330E+01,
    4.589875E+01, 5.349498E+01, 6.109197E+01, 6.868969E+01, 7.628815E+01, 8.388739E+01,
    9.148750E+01, 9.908871E+01, 1.066914E+02, 1.142961E+02, 1.219039E+02, 1.295159E+02,
    1.371337E+02, 1.447592E+02, 1.523947E+02, 1.600426E+02, 1.677058E+02, 1.753873E+02,
    1.830902E+02, 1.908178E+02, 1.985735E+02, 2.063605E+02, 2.141824E+02, 2.220424E+02,
    2.299438E+02, 2.378898E+02, 2.458836E+02, 2.539281E+02, 2.620262E+02, 2.701807E+02,
    2.783942E+02, 2.866693E+02, 2.950083E+02, 3.034135E+02, 3.118870E+02, 3.204310E+02,
    3.290472E+02, 3.377375E+02, 3.465036E+02, 3.553472E+02, 3.642697E+02, 3.732725E+02,
    3.823570E+02, 3.915246E+02, 4.007762E+02, 4.101132E+02, 4.195365E+02, 4.290471E+02,
    4.386461E+02, 4.483341E+02, 4.581122E+02, 4.679810E+02, 4.779414E+02, 4.879940E+02,
    4.981395E+02, 5.083786E+02, 5.187117E+02, 5.291395E+02, 5.396626E+02, 5.502813E+02,
    5.609963E+02, 5.718079E+02, 5.827166E+02, 5.937229E+02, 6.048270E+02, 6.160294E+02,
    6.273304E+02, 6.387304E+02, 6.502297E+02, 6.618287E+02, 6.735275E+02, 6.853265E+02,
    6.972260E+02, 7.092263E+02, 7.213275E+02, 7.335299E+02, 7.458338E+02, 7.582394E+02,
    7.707469E+02, 7.833566E+02, 7.960685E+02, 8.088830E+02, 8.218001E+02, 8.348201E+02,
    8.479432E+02, 8.611695E+02, 8.744993E+02, 8.879325E+02, 9.014695E+02, 9.151104E+02,
    9.288553E+02, 9.427043E+02, 9.566576E+02, 9.707154E+02, 9.848778E+02, 9.991449E+02,
    1.013517E+03, 1.027994E+03, 1.042576E+03, 1.057263E+03, 1.072055E+03, 1.086953E+03,
    1.101957E+03, 1.117066E+03, 1.132281E+03, 1.147601E+03, 1.163028E+03, 1.178561E+03,
    1.194200E+03, 1.209945E+03, 1.225797E+03, 1.241756E+03, 1.257820E+03, 1.273992E+03,
    1.290270E+03, 1.306656E+03, 1.323148E+03, 1.339747E+03, 1.356454E+03, 1.373267E+03,
    1.390188E+03, 1.407217E+03, 1.424353E+03, 1.441596E+03, 1.458948E+03, 1.476407E+03,
    1.493974E+03, 1.511649E+03, 1.529431E+03, 1.547322E+03, 1.565322E+03, 1.583429E+03,
    1.601645E+03, 1.619969E+03, 1.638402E+03, 1.656944E+03, 1.675594E+03, 1.694353E+03,
    1.713221E+03, 1.732197E+03, 1.751283E+03, 1.770478E+03, 1.789782E+03, 1.809195E+03,
    1.828718E+03, 1.848350E+03, 1.868091E+03, 1.887942E+03, 1.907903E+03, 1.927973E+03,
    1.948153E+03, 1.968443E+03, 1.988843E+03, 2.009353E+03, 2.029973E+03, 2.050703E+03,
    2.071543E+03, 2.092494E+03, 2.113555E+03, 2.134727E+03, 2.156009E+03, 2.177402E+03,
    2.198905E+03, 2.220519E+03, 2.242244E+03, 2.264080E+03, 2.286027E+03, 2.308085E+03,
    2.330255E+03, 2.352535E+03, 2.374927E+03, 2.397430E+03, 2.420044E+03, 2.442771E+03,
    2.465608E+03, 2.488557E+03, 2.511619E+03, 2.534791E+03, 2.558076E+03, 2.581473E+03,
    2.604982E+03, 2.628603E+03, 2.652336E+03, 2.676181E+03, 2.700139E+03, 2.724209E+03,
    2.748391E+03, 2.772686E+03, 2.797094E+03, 2.821615E+03, 2.846248E+03, 2.870994E+03,
    2.895853E+03, 2.920825E+03, 2.945910E+03, 2.971108E+03, 2.996420E+03, 3.021844E+03,
    3.047382E+03, 3.073034E+03, 3.098799E+03, 3.124678E+03, 3.150670E+03, 3.176776E+03,
    3.202996E+03, 3.229330E+03, 3.255778E+03, 3.282340E+03, 3.309015E+03, 3.335806E+03,
    3.362710E+03, 3.389729E+03, 3.416862E+03, 3.444109E+03, 3.471471E+03, 3.498948E+03,
    3.526540E+03, 3.554246E+03, 3.582067E+03, 3.610003E+03, 3.638054E+03, 3.666220E+03,
    3.694501E+03, 3.722898E+03, 3.751409E+03, 3.780036E+03, 3.808779E+03, 3.837637E+03,
    3.866611E+03, 3.895700E+03, 3.924905E+03, 3.954226E+03, 3.983662E+03, 4.013215E+03,
    4.042883E+03, 4.072668E+03, 4.102569E+03, 4.132586E+03, 4.162720E+03, 4.192969E+03,
    4.223336E+03, 4.253818E+03, 4.284418E+03, 4.315134E+03, 4.345967E+03, 4.376916E+03,
    4.407983E+03, 4.439166E+03, 4.470467E+03, 4.501884E+03, 4.533419E+03, 4.565071E+03,
    4.596841E+03, 4.628727E+03, 4.660731E+03, 4.692853E+03, 4.725092E+03, 4.757449E+03,
    4.789924E+03, 4.822517E+03, 4.855227E+03, 4.888055E+03, 4.921002E+03, 4.954066E+03,
    4.987249E+03, 5.020550E+03, 5.053969E+03, 5.087506E+03, 5.121162E+03, 5.154936E+03,
    5.188829E+03, 5.222841E+03, 5.256971E+03, 5.291221E+03, 5.325588E+03, 5.360075E+03,
    5.394681E+03, 5.429406E+03, 5.464250E+03, 5.499213E+03, 5.534296E+03, 5.569497E+03,
    5.604818E+03, 5.640259E+03, 5.675819E+03, 5.711498E+03, 5.747298E+03, 5.783217E+03,
    5.819255E+03, 5.855414E+03, 5.891692E+03, 5.928090E+03, 5.964609E+03, 6.001247E+03,
    6.038006E+03, 6.074885E+03, 6.111884E+03, 6.149003E+03, 6.186243E+03, 6.223603E+03,
    6.261084E+03, 6.298685E+03, 6.336407E+03, 6.374250E+03, 6.412213E+03, 6.450298E+03,
    6.488503E+03, 6.526829E+03, 6.565276E+03, 6.603844E+03, 6.642533E+03, 6.681344E+03,
    6.720275E+03, 6.759328E+03, 6.798502E+03, 6.837798E+03, 6.877215E+03, 6.916754E+03,
    6.956414E+03, 6.996196E+03, 7.036099E+03, 7.076125E+03, 7.116272E+03, 7.156541E+03,
    7.196931E+03, 7.237444E+03, 7.278079E+03, 7.318835E+03, 7.359714E+03, 7.400715E+03,
    7.441839E+03, 7.483084E+03, 7.524452E+03, 7.565942E+03, 7.607555E+03, 7.649290E+03,
    7.691147E+03, 7.733127E+03, 7.775230E+03, 7.817455E+03, 7.859804E+03, 7.902274E+03,
    7.944868E+03, 7.987585E+03, 8.030424E+03, 8.073386E+03, 8.116471E+03, 8.159680E+03,
    8.203011E+03, 8.246466E+03, 8.290043E+03, 8.333744E+03, 8.377568E+03, 8.421516E+03,
    8.465586E+03, 8.509781E+03, 8.554098E+03, 8.598539E+03, 8.643104E+03, 8.687792E+03,
    8.732603E+03, 8.777538E+03, 8.822597E+03, 8.867780E+03, 8.913086E+03, 8.958516E+03,
    9.004070E+03, 9.049748E+03, 9.095549E+03, 9.141475E+03, 9.187524E+03, 9.233697E+03,
    9.279995E+03, 9.326416E+03, 9.372962E+03, 9.419631E+03, 9.466425E+03, 9.513343E+03,
    9.560385E+03, 9.607552E+03, 9.654843E+03, 9.702258E+03, 9.749797E+03, 9.797461E+03,
    9.845249E+03, 9.893161E+03, 9.941198E+03, 9.989360E+03, 1.003765E+04, 1.008606E+04,
    1.013459E+04, 1.018325E+04, 1.023204E+04, 1.028094E+04, 1.032998E+04, 1.037914E+04,
    1.042842E+04, 1.047783E+04, 1.052736E+04, 1.057702E+04, 1.062680E+04, 1.067670E+04,
    1.072674E+04, 1.077689E+04, 1.082717E+04, 1.087758E+04, 1.092811E+04, 1.097877E+04,
    1.102955E+04, 1.108045E+04, 1.113149E+04, 1.118264E+04, 1.123392E+04, 1.128533E+04,
    1.133686E+04, 1.138851E+04, 1.144030E+04, 1.149220E+04, 1.154423E+04, 1.159639E+04,
    1.164867E+04, 1.170108E+04, 1.175361E+04, 1.180627E+04, 1.185905E+04, 1.191196E+04,
    1.196499E+04, 1.201815E+04, 1.207143E+04, 1.212484E+04, 1.217837E+04, 1.223203E+04,
    1.228581E+04, 1.233972E+04, 1.239375E+04, 1.244791E+04, 1.250220E+04, 1.255661E+04,
    1.261114E+04, 1.266580E+04, 1.272059E+04, 1.277550E+04, 1.283053E+04, 1.288570E+04,
    1.294098E+04,
])

#  ---------------------- M = 5, I = 4 ---------------------------
M = 5
I = 4
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[3]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    6.081920E+00, 4.656398E+01, 9.104865E+01, 1.355563E+02, 1.800726E+02, 2.245949E+02,
    2.691223E+02, 3.136542E+02, 3.581906E+02, 4.027313E+02, 4.472763E+02, 4.918259E+02,
    5.363805E+02, 5.809412E+02, 6.255100E+02, 6.700904E+02, 7.146872E+02, 7.593070E+02,
    8.039587E+02, 8.486526E+02, 8.934013E+02, 9.382189E+02, 9.831211E+02, 1.028125E+03,
    1.073248E+03, 1.118509E+03, 1.163927E+03, 1.209521E+03, 1.255311E+03, 1.301316E+03,
    1.347555E+03, 1.394046E+03, 1.440807E+03, 1.487857E+03, 1.535211E+03, 1.582885E+03,
    1.630896E+03, 1.679258E+03, 1.727985E+03, 1.777089E+03, 1.826585E+03, 1.876483E+03,
    1.926795E+03, 1.977532E+03, 2.028704E+03, 2.080320E+03, 2.132389E+03, 2.184919E+03,
    2.237919E+03, 2.291396E+03, 2.345357E+03, 2.399809E+03, 2.454758E+03, 2.510210E+03,
    2.566171E+03, 2.622645E+03, 2.679638E+03, 2.737155E+03, 2.795199E+03, 2.853775E+03,
    2.912888E+03, 2.972540E+03, 3.032735E+03, 3.093477E+03, 3.154769E+03, 3.216614E+03,
    3.279014E+03, 3.341973E+03, 3.405493E+03, 3.469576E+03, 3.534226E+03, 3.599443E+03,
    3.665230E+03, 3.731590E+03, 3.798524E+03, 3.866034E+03, 3.934122E+03, 4.002789E+03,
    4.072038E+03, 4.141869E+03, 4.212285E+03, 4.283286E+03, 4.354874E+03, 4.427051E+03,
    4.499818E+03, 4.573175E+03, 4.647125E+03, 4.721668E+03, 4.796805E+03, 4.872538E+03,
    4.948867E+03, 5.025794E+03, 5.103320E+03, 5.181445E+03, 5.260171E+03, 5.339498E+03,
    5.419427E+03, 5.499959E+03, 5.581096E+03, 5.662837E+03, 5.745183E+03, 5.828137E+03,
    5.911697E+03, 5.995865E+03, 6.080641E+03, 6.166027E+03, 6.252023E+03, 6.338630E+03,
    6.425848E+03, 6.513677E+03, 6.602120E+03, 6.691176E+03, 6.780846E+03, 6.871130E+03,
    6.962030E+03, 7.053545E+03, 7.145676E+03, 7.238425E+03, 7.331791E+03, 7.425775E+03,
    7.520378E+03, 7.615600E+03, 7.711441E+03, 7.807903E+03, 7.904986E+03, 8.002690E+03,
    8.101016E+03, 8.199964E+03, 8.299536E+03, 8.399730E+03, 8.500549E+03, 8.601992E+03,
    8.704060E+03, 8.806753E+03, 8.910073E+03, 9.014018E+03, 9.118591E+03, 9.223791E+03,
    9.329619E+03, 9.436075E+03, 9.543160E+03, 9.650874E+03, 9.759219E+03, 9.868193E+03,
    9.977798E+03, 1.008803E+04, 1.019890E+04, 1.031040E+04, 1.042253E+04, 1.053530E+04,
    1.064870E+04, 1.076273E+04, 1.087740E+04, 1.099270E+04, 1.110864E+04, 1.122521E+04,
    1.134242E+04, 1.146026E+04, 1.157875E+04, 1.169787E+04, 1.181763E+04, 1.193802E+04,
    1.205906E+04, 1.218073E+04, 1.230305E+04, 1.242600E+04, 1.254960E+04, 1.267383E+04,
    1.279871E+04, 1.292423E+04, 1.305039E+04, 1.317719E+04, 1.330464E+04, 1.343273E+04,
    1.356147E+04, 1.369085E+04, 1.382087E+04, 1.395154E+04, 1.408286E+04, 1.421482E+04,
    1.434743E+04, 1.448068E+04, 1.461459E+04, 1.474914E+04, 1.488434E+04, 1.502019E+04,
    1.515669E+04, 1.529383E+04, 1.543163E+04, 1.557008E+04, 1.570918E+04, 1.584893E+04,
    1.598933E+04, 1.613039E+04, 1.627209E+04, 1.641445E+04, 1.655747E+04, 1.670114E+04,
    1.684546E+04, 1.699044E+04, 1.713607E+04, 1.728236E+04, 1.742930E+04, 1.757690E+04,
    1.772516E+04, 1.787408E+04, 1.802365E+04, 1.817388E+04, 1.832477E+04, 1.847632E+04,
    1.862853E+04, 1.878140E+04, 1.893493E+04, 1.908912E+04, 1.924398E+04, 1.939949E+04,
    1.955567E+04, 1.971250E+04, 1.987001E+04, 2.002817E+04, 2.018700E+04, 2.034649E+04,
    2.050665E+04, 2.066748E+04, 2.082897E+04, 2.099112E+04, 2.115394E+04, 2.131743E+04,
    2.148159E+04, 2.164641E+04, 2.181190E+04, 2.197807E+04, 2.214490E+04, 2.231239E+04,
    2.248056E+04, 2.264940E+04, 2.281891E+04, 2.298910E+04, 2.315995E+04, 2.333147E+04,
    2.350367E+04, 2.367654E+04, 2.385009E+04, 2.402431E+04, 2.419920E+04, 2.437476E+04,
    2.455101E+04, 2.472792E+04, 2.490552E+04, 2.508379E+04, 2.526273E+04, 2.544236E+04,
    2.562266E+04, 2.580364E+04, 2.598529E+04, 2.616763E+04, 2.635065E+04, 2.653434E+04,
    2.671872E+04, 2.690377E+04, 2.708951E+04, 2.727593E+04, 2.746303E+04, 2.765081E+04,
    2.783928E+04, 2.802843E+04, 2.821826E+04, 2.840877E+04, 2.859997E+04, 2.879186E+04,
    2.898443E+04, 2.917768E+04, 2.937162E+04, 2.956625E+04, 2.976157E+04, 2.995757E+04,
    3.015426E+04, 3.035163E+04, 3.054970E+04, 3.074845E+04, 3.094790E+04, 3.114803E+04,
    3.134885E+04, 3.155037E+04, 3.175257E+04, 3.195547E+04, 3.215905E+04, 3.236333E+04,
    3.256830E+04, 3.277397E+04, 3.298033E+04, 3.318738E+04, 3.339512E+04, 3.360356E+04,
    3.381270E+04, 3.402252E+04, 3.423305E+04, 3.444427E+04, 3.465619E+04, 3.486880E+04,
    3.508211E+04, 3.529612E+04, 3.551082E+04, 3.572623E+04, 3.594233E+04, 3.615913E+04,
    3.637663E+04, 3.659483E+04, 3.681373E+04, 3.703333E+04, 3.725363E+04, 3.747464E+04,
    3.769634E+04, 3.791875E+04, 3.814186E+04, 3.836567E+04, 3.859018E+04, 3.881540E+04,
    3.904132E+04, 3.926794E+04, 3.949527E+04, 3.972330E+04, 3.995204E+04, 4.018149E+04,
    4.041164E+04, 4.064249E+04, 4.087405E+04, 4.110632E+04, 4.133930E+04, 4.157298E+04,
    4.180737E+04, 4.204247E+04, 4.227828E+04, 4.251480E+04, 4.275202E+04, 4.298996E+04,
    4.322860E+04, 4.346796E+04, 4.370802E+04, 4.394880E+04, 4.419028E+04, 4.443248E+04,
    4.467539E+04, 4.491901E+04, 4.516335E+04, 4.540839E+04, 4.565415E+04, 4.590062E+04,
    4.614781E+04, 4.639571E+04, 4.664432E+04, 4.689365E+04, 4.714369E+04, 4.739445E+04,
    4.764592E+04, 4.789811E+04, 4.815101E+04, 4.840463E+04, 4.865897E+04, 4.891402E+04,
    4.916979E+04, 4.942627E+04, 4.968348E+04, 4.994140E+04, 5.020004E+04, 5.045939E+04,
    5.071947E+04, 5.098026E+04, 5.124178E+04, 5.150401E+04, 5.176696E+04, 5.203063E+04,
    5.229502E+04, 5.256014E+04, 5.282597E+04, 5.309252E+04, 5.335979E+04, 5.362779E+04,
    5.389650E+04, 5.416594E+04, 5.443610E+04, 5.470698E+04, 5.497858E+04, 5.525091E+04,
    5.552396E+04, 5.579773E+04, 5.607222E+04, 5.634744E+04, 5.662338E+04, 5.690004E+04,
    5.717743E+04, 5.745555E+04, 5.773438E+04, 5.801394E+04, 5.829423E+04, 5.857524E+04,
    5.885698E+04, 5.913944E+04, 5.942262E+04, 5.970654E+04, 5.999117E+04, 6.027654E+04,
    6.056263E+04, 6.084944E+04, 6.113698E+04, 6.142525E+04, 6.171425E+04, 6.200397E+04,
    6.229442E+04, 6.258560E+04, 6.287750E+04, 6.317013E+04, 6.346349E+04, 6.375758E+04,
    6.405239E+04, 6.434794E+04, 6.464421E+04, 6.494121E+04, 6.523894E+04, 6.553739E+04,
    6.583658E+04, 6.613649E+04, 6.643713E+04, 6.673850E+04, 6.704060E+04, 6.734343E+04,
    6.764699E+04, 6.795128E+04, 6.825630E+04, 6.856205E+04, 6.886853E+04, 6.917573E+04,
    6.948367E+04, 6.979234E+04, 7.010173E+04, 7.041186E+04, 7.072272E+04, 7.103431E+04,
    7.134663E+04, 7.165967E+04, 7.197345E+04, 7.228796E+04, 7.260320E+04, 7.291917E+04,
    7.323588E+04, 7.355331E+04, 7.387147E+04, 7.419036E+04, 7.450999E+04, 7.483035E+04,
    7.515143E+04,
])

#  ---------------------- M = 5, I = 5 ---------------------------
M = 5
I = 5
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[3]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.039420E+00, 1.660365E+01, 3.251624E+01, 4.843614E+01, 6.435894E+01, 8.028378E+01,
    9.621037E+01, 1.121386E+02, 1.280684E+02, 1.439997E+02, 1.599326E+02, 1.758672E+02,
    1.918036E+02, 2.077426E+02, 2.236851E+02, 2.396326E+02, 2.555873E+02, 2.715523E+02,
    2.875312E+02, 3.035287E+02, 3.195498E+02, 3.356005E+02, 3.516871E+02, 3.678163E+02,
    3.839953E+02, 4.002313E+02, 4.165317E+02, 4.329040E+02, 4.493555E+02, 4.658935E+02,
    4.825251E+02, 4.992572E+02, 5.160964E+02, 5.330491E+02, 5.501215E+02, 5.673195E+02,
    5.846486E+02, 6.021141E+02, 6.197210E+02, 6.374741E+02, 6.553778E+02, 6.734364E+02,
    6.916539E+02, 7.100339E+02, 7.285802E+02, 7.472958E+02, 7.661840E+02, 7.852478E+02,
    8.044898E+02, 8.239127E+02, 8.435189E+02, 8.633107E+02, 8.832903E+02, 9.034597E+02,
    9.238208E+02, 9.443754E+02, 9.651252E+02, 9.860718E+02, 1.007217E+03, 1.028562E+03,
    1.050107E+03, 1.071856E+03, 1.093807E+03, 1.115964E+03, 1.138326E+03, 1.160895E+03,
    1.183672E+03, 1.206658E+03, 1.229853E+03, 1.253259E+03, 1.276876E+03, 1.300705E+03,
    1.324747E+03, 1.349002E+03, 1.373471E+03, 1.398155E+03, 1.423054E+03, 1.448169E+03,
    1.473501E+03, 1.499050E+03, 1.524816E+03, 1.550800E+03, 1.577002E+03, 1.603424E+03,
    1.630065E+03, 1.656925E+03, 1.684006E+03, 1.711308E+03, 1.738830E+03, 1.766574E+03,
    1.794539E+03, 1.822727E+03, 1.851137E+03, 1.879770E+03, 1.908625E+03, 1.937705E+03,
    1.967007E+03, 1.996534E+03, 2.026285E+03, 2.056261E+03, 2.086461E+03, 2.116887E+03,
    2.147537E+03, 2.178414E+03, 2.209516E+03, 2.240845E+03, 2.272400E+03, 2.304181E+03,
    2.336189E+03, 2.368425E+03, 2.400887E+03, 2.433577E+03, 2.466495E+03, 2.499641E+03,
    2.533015E+03, 2.566618E+03, 2.600449E+03, 2.634509E+03, 2.668798E+03, 2.703316E+03,
    2.738063E+03, 2.773040E+03, 2.808247E+03, 2.843684E+03, 2.879351E+03, 2.915249E+03,
    2.951377E+03, 2.987736E+03, 3.024325E+03, 3.061146E+03, 3.098198E+03, 3.135482E+03,
    3.172998E+03, 3.210745E+03, 3.248724E+03, 3.286935E+03, 3.325379E+03, 3.364056E+03,
    3.402965E+03, 3.442107E+03, 3.481482E+03, 3.521090E+03, 3.560932E+03, 3.601007E+03,
    3.641316E+03, 3.681859E+03, 3.722636E+03, 3.763647E+03, 3.804893E+03, 3.846373E+03,
    3.888088E+03, 3.930038E+03, 3.972223E+03, 4.014644E+03, 4.057299E+03, 4.100190E+03,
    4.143317E+03, 4.186680E+03, 4.230279E+03, 4.274114E+03, 4.318186E+03, 4.362494E+03,
    4.407039E+03, 4.451821E+03, 4.496839E+03, 4.542095E+03, 4.587588E+03, 4.633319E+03,
    4.679287E+03, 4.725494E+03, 4.771938E+03, 4.818620E+03, 4.865541E+03, 4.912700E+03,
    4.960098E+03, 5.007734E+03, 5.055610E+03, 5.103724E+03, 5.152078E+03, 5.200672E+03,
    5.249504E+03, 5.298577E+03, 5.347889E+03, 5.397442E+03, 5.447235E+03, 5.497268E+03,
    5.547541E+03, 5.598056E+03, 5.648811E+03, 5.699807E+03, 5.751044E+03, 5.802523E+03,
    5.854243E+03, 5.906205E+03, 5.958408E+03, 6.010853E+03, 6.063541E+03, 6.116471E+03,
    6.169643E+03, 6.223058E+03, 6.276715E+03, 6.330616E+03, 6.384759E+03, 6.439146E+03,
    6.493776E+03, 6.548649E+03, 6.603766E+03, 6.659127E+03, 6.714732E+03, 6.770581E+03,
    6.826675E+03, 6.883013E+03, 6.939595E+03, 6.996422E+03, 7.053495E+03, 7.110812E+03,
    7.168374E+03, 7.226182E+03, 7.284235E+03, 7.342534E+03, 7.401079E+03, 7.459870E+03,
    7.518907E+03, 7.578191E+03, 7.637721E+03, 7.697497E+03, 7.757520E+03, 7.817791E+03,
    7.878308E+03, 7.939072E+03, 8.000084E+03, 8.061344E+03, 8.122851E+03, 8.184606E+03,
    8.246609E+03, 8.308860E+03, 8.371360E+03, 8.434108E+03, 8.497104E+03, 8.560349E+03,
    8.623844E+03, 8.687587E+03, 8.751580E+03, 8.815821E+03, 8.880313E+03, 8.945054E+03,
    9.010045E+03, 9.075286E+03, 9.140777E+03, 9.206518E+03, 9.272509E+03, 9.338752E+03,
    9.405244E+03, 9.471988E+03, 9.538983E+03, 9.606229E+03, 9.673726E+03, 9.741475E+03,
    9.809475E+03, 9.877727E+03, 9.946231E+03, 1.001499E+04, 1.008399E+04, 1.015325E+04,
    1.022277E+04, 1.029253E+04, 1.036255E+04, 1.043282E+04, 1.050335E+04, 1.057412E+04,
    1.064515E+04, 1.071644E+04, 1.078797E+04, 1.085976E+04, 1.093181E+04, 1.100411E+04,
    1.107666E+04, 1.114947E+04, 1.122253E+04, 1.129585E+04, 1.136942E+04, 1.144324E+04,
    1.151732E+04, 1.159166E+04, 1.166625E+04, 1.174110E+04, 1.181620E+04, 1.189155E+04,
    1.196717E+04, 1.204304E+04, 1.211916E+04, 1.219554E+04, 1.227218E+04, 1.234907E+04,
    1.242622E+04, 1.250363E+04, 1.258129E+04, 1.265921E+04, 1.273739E+04, 1.281582E+04,
    1.289451E+04, 1.297346E+04, 1.305267E+04, 1.313213E+04, 1.321185E+04, 1.329183E+04,
    1.337207E+04, 1.345256E+04, 1.353332E+04, 1.361433E+04, 1.369560E+04, 1.377713E+04,
    1.385891E+04, 1.394096E+04, 1.402326E+04, 1.410583E+04, 1.418865E+04, 1.427173E+04,
    1.435507E+04, 1.443867E+04, 1.452253E+04, 1.460665E+04, 1.469103E+04, 1.477567E+04,
    1.486057E+04, 1.494572E+04, 1.503114E+04, 1.511682E+04, 1.520276E+04, 1.528896E+04,
    1.537542E+04, 1.546214E+04, 1.554912E+04, 1.563636E+04, 1.572387E+04, 1.581163E+04,
    1.589966E+04, 1.598794E+04, 1.607649E+04, 1.616530E+04, 1.625437E+04, 1.634370E+04,
    1.643329E+04, 1.652315E+04, 1.661326E+04, 1.670364E+04, 1.679428E+04, 1.688518E+04,
    1.697635E+04, 1.706777E+04, 1.715946E+04, 1.725141E+04, 1.734363E+04, 1.743610E+04,
    1.752884E+04, 1.762184E+04, 1.771510E+04, 1.780863E+04, 1.790242E+04, 1.799647E+04,
    1.809078E+04, 1.818536E+04, 1.828020E+04, 1.837531E+04, 1.847067E+04, 1.856630E+04,
    1.866220E+04, 1.875836E+04, 1.885478E+04, 1.895146E+04, 1.904841E+04, 1.914562E+04,
    1.924309E+04, 1.934083E+04, 1.943883E+04, 1.953710E+04, 1.963563E+04, 1.973443E+04,
    1.983348E+04, 1.993281E+04, 2.003239E+04, 2.013224E+04, 2.023236E+04, 2.033273E+04,
    2.043338E+04, 2.053428E+04, 2.063546E+04, 2.073689E+04, 2.083859E+04, 2.094056E+04,
    2.104279E+04, 2.114528E+04, 2.124804E+04, 2.135106E+04, 2.145435E+04, 2.155790E+04,
    2.166172E+04, 2.176580E+04, 2.187014E+04, 2.197476E+04, 2.207963E+04, 2.218477E+04,
    2.229018E+04, 2.239585E+04, 2.250178E+04, 2.260798E+04, 2.271445E+04, 2.282118E+04,
    2.292817E+04, 2.303543E+04, 2.314296E+04, 2.325075E+04, 2.335880E+04, 2.346712E+04,
    2.357571E+04, 2.368455E+04, 2.379367E+04, 2.390305E+04, 2.401269E+04, 2.412260E+04,
    2.423278E+04, 2.434322E+04, 2.445392E+04, 2.456489E+04, 2.467613E+04, 2.478763E+04,
    2.489939E+04, 2.501142E+04, 2.512372E+04, 2.523628E+04, 2.534910E+04, 2.546220E+04,
    2.557555E+04, 2.568917E+04, 2.580306E+04, 2.591721E+04, 2.603162E+04, 2.614630E+04,
    2.626125E+04, 2.637646E+04, 2.649193E+04, 2.660767E+04, 2.672367E+04, 2.683994E+04,
    2.695648E+04, 2.707328E+04, 2.719034E+04, 2.730767E+04, 2.742526E+04, 2.754312E+04,
    2.766124E+04,
])

#  ---------------------- M = 5, I = 6 ---------------------------
M = 5
I = 6
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[3]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.220898E+01, 9.732970E+01, 1.905089E+02, 2.837327E+02, 3.769738E+02, 4.702271E+02,
    5.634907E+02, 6.567639E+02, 7.500463E+02, 8.433378E+02, 9.366383E+02, 1.029949E+03,
    1.123270E+03, 1.216605E+03, 1.309959E+03, 1.403341E+03, 1.496762E+03, 1.590239E+03,
    1.683792E+03, 1.777446E+03, 1.871230E+03, 1.965177E+03, 2.059322E+03, 2.153704E+03,
    2.248362E+03, 2.343338E+03, 2.438674E+03, 2.534412E+03, 2.630596E+03, 2.727266E+03,
    2.824463E+03, 2.922228E+03, 3.020599E+03, 3.119613E+03, 3.219306E+03, 3.319712E+03,
    3.420863E+03, 3.522791E+03, 3.625525E+03, 3.729092E+03, 3.833519E+03, 3.938831E+03,
    4.045052E+03, 4.152203E+03, 4.260305E+03, 4.369377E+03, 4.479439E+03, 4.590508E+03,
    4.702599E+03, 4.815729E+03, 4.929912E+03, 5.045161E+03, 5.161489E+03, 5.278909E+03,
    5.397431E+03, 5.517067E+03, 5.637826E+03, 5.759718E+03, 5.882752E+03, 6.006937E+03,
    6.132280E+03, 6.258790E+03, 6.386472E+03, 6.515335E+03, 6.645385E+03, 6.776627E+03,
    6.909068E+03, 7.042712E+03, 7.177566E+03, 7.313634E+03, 7.450922E+03, 7.589432E+03,
    7.729171E+03, 7.870141E+03, 8.012348E+03, 8.155794E+03, 8.300483E+03, 8.446420E+03,
    8.593606E+03, 8.742045E+03, 8.891741E+03, 9.042696E+03, 9.194913E+03, 9.348394E+03,
    9.503144E+03, 9.659162E+03, 9.816454E+03, 9.975020E+03, 1.013486E+04, 1.029599E+04,
    1.045839E+04, 1.062208E+04, 1.078705E+04, 1.095331E+04, 1.112086E+04, 1.128970E+04,
    1.145983E+04, 1.163126E+04, 1.180398E+04, 1.197801E+04, 1.215333E+04, 1.232995E+04,
    1.250788E+04, 1.268711E+04, 1.286765E+04, 1.304949E+04, 1.323264E+04, 1.341711E+04,
    1.360288E+04, 1.378997E+04, 1.397837E+04, 1.416809E+04, 1.435913E+04, 1.455148E+04,
    1.474515E+04, 1.494015E+04, 1.513646E+04, 1.533410E+04, 1.553307E+04, 1.573336E+04,
    1.593497E+04, 1.613792E+04, 1.634219E+04, 1.654779E+04, 1.675473E+04, 1.696300E+04,
    1.717260E+04, 1.738353E+04, 1.759580E+04, 1.780941E+04, 1.802436E+04, 1.824064E+04,
    1.845827E+04, 1.867723E+04, 1.889754E+04, 1.911919E+04, 1.934219E+04, 1.956653E+04,
    1.979221E+04, 2.001925E+04, 2.024763E+04, 2.047736E+04, 2.070844E+04, 2.094087E+04,
    2.117466E+04, 2.140979E+04, 2.164629E+04, 2.188413E+04, 2.212333E+04, 2.236389E+04,
    2.260581E+04, 2.284909E+04, 2.309373E+04, 2.333972E+04, 2.358708E+04, 2.383580E+04,
    2.408589E+04, 2.433734E+04, 2.459015E+04, 2.484434E+04, 2.509989E+04, 2.535680E+04,
    2.561509E+04, 2.587475E+04, 2.613578E+04, 2.639818E+04, 2.666195E+04, 2.692710E+04,
    2.719362E+04, 2.746152E+04, 2.773080E+04, 2.800145E+04, 2.827348E+04, 2.854690E+04,
    2.882169E+04, 2.909786E+04, 2.937542E+04, 2.965436E+04, 2.993468E+04, 3.021639E+04,
    3.049948E+04, 3.078396E+04, 3.106983E+04, 3.135709E+04, 3.164574E+04, 3.193578E+04,
    3.222721E+04, 3.252003E+04, 3.281424E+04, 3.310985E+04, 3.340686E+04, 3.370526E+04,
    3.400506E+04, 3.430625E+04, 3.460885E+04, 3.491284E+04, 3.521824E+04, 3.552504E+04,
    3.583324E+04, 3.614284E+04, 3.645385E+04, 3.676626E+04, 3.708008E+04, 3.739530E+04,
    3.771194E+04, 3.802998E+04, 3.834943E+04, 3.867030E+04, 3.899257E+04, 3.931626E+04,
    3.964136E+04, 3.996788E+04, 4.029581E+04, 4.062516E+04, 4.095592E+04, 4.128810E+04,
    4.162170E+04, 4.195673E+04, 4.229317E+04, 4.263103E+04, 4.297032E+04, 4.331103E+04,
    4.365316E+04, 4.399672E+04, 4.434171E+04, 4.468812E+04, 4.503596E+04, 4.538523E+04,
    4.573593E+04, 4.608806E+04, 4.644162E+04, 4.679662E+04, 4.715304E+04, 4.751091E+04,
    4.787020E+04, 4.823094E+04, 4.859311E+04, 4.895671E+04, 4.932176E+04, 4.968825E+04,
    5.005618E+04, 5.042554E+04, 5.079636E+04, 5.116861E+04, 5.154231E+04, 5.191745E+04,
    5.229404E+04, 5.267208E+04, 5.305156E+04, 5.343249E+04, 5.381488E+04, 5.419871E+04,
    5.458399E+04, 5.497073E+04, 5.535892E+04, 5.574856E+04, 5.613965E+04, 5.653221E+04,
    5.692621E+04, 5.732168E+04, 5.771860E+04, 5.811699E+04, 5.851683E+04, 5.891813E+04,
    5.932090E+04, 5.972512E+04, 6.013081E+04, 6.053797E+04, 6.094659E+04, 6.135667E+04,
    6.176822E+04, 6.218124E+04, 6.259573E+04, 6.301168E+04, 6.342911E+04, 6.384801E+04,
    6.426837E+04, 6.469022E+04, 6.511353E+04, 6.553832E+04, 6.596458E+04, 6.639232E+04,
    6.682153E+04, 6.725222E+04, 6.768439E+04, 6.811804E+04, 6.855317E+04, 6.898978E+04,
    6.942787E+04, 6.986744E+04, 7.030849E+04, 7.075103E+04, 7.119505E+04, 7.164056E+04,
    7.208755E+04, 7.253603E+04, 7.298599E+04, 7.343745E+04, 7.389039E+04, 7.434483E+04,
    7.480075E+04, 7.525816E+04, 7.571707E+04, 7.617747E+04, 7.663936E+04, 7.710275E+04,
    7.756763E+04, 7.803400E+04, 7.850187E+04, 7.897124E+04, 7.944211E+04, 7.991447E+04,
    8.038834E+04, 8.086370E+04, 8.134057E+04, 8.181893E+04, 8.229880E+04, 8.278016E+04,
    8.326304E+04, 8.374741E+04, 8.423329E+04, 8.472068E+04, 8.520957E+04, 8.569996E+04,
    8.619187E+04, 8.668528E+04, 8.718020E+04, 8.767663E+04, 8.817457E+04, 8.867401E+04,
    8.917497E+04, 8.967744E+04, 9.018143E+04, 9.068692E+04, 9.119393E+04, 9.170245E+04,
    9.221249E+04, 9.272404E+04, 9.323710E+04, 9.375169E+04, 9.426779E+04, 9.478540E+04,
    9.530454E+04, 9.582519E+04, 9.634736E+04, 9.687105E+04, 9.739626E+04, 9.792299E+04,
    9.845125E+04, 9.898102E+04, 9.951232E+04, 1.000451E+05, 1.005795E+05, 1.011153E+05,
    1.016527E+05, 1.021916E+05, 1.027321E+05, 1.032740E+05, 1.038175E+05, 1.043626E+05,
    1.049091E+05, 1.054572E+05, 1.060068E+05, 1.065579E+05, 1.071105E+05, 1.076647E+05,
    1.082204E+05, 1.087777E+05, 1.093365E+05, 1.098968E+05, 1.104586E+05, 1.110220E+05,
    1.115869E+05, 1.121533E+05, 1.127213E+05, 1.132908E+05, 1.138618E+05, 1.144344E+05,
    1.150085E+05, 1.155841E+05, 1.161613E+05, 1.167400E+05, 1.173202E+05, 1.179020E+05,
    1.184853E+05, 1.190701E+05, 1.196565E+05, 1.202444E+05, 1.208339E+05, 1.214249E+05,
    1.220174E+05, 1.226115E+05, 1.232071E+05, 1.238043E+05, 1.244030E+05, 1.250032E+05,
    1.256050E+05, 1.262083E+05, 1.268131E+05, 1.274195E+05, 1.280275E+05, 1.286369E+05,
    1.292479E+05, 1.298605E+05, 1.304746E+05, 1.310902E+05, 1.317074E+05, 1.323261E+05,
    1.329464E+05, 1.335682E+05, 1.341915E+05, 1.348164E+05, 1.354429E+05, 1.360708E+05,
    1.367004E+05, 1.373314E+05, 1.379640E+05, 1.385982E+05, 1.392339E+05, 1.398711E+05,
    1.405099E+05, 1.411502E+05, 1.417921E+05, 1.424355E+05, 1.430804E+05, 1.437269E+05,
    1.443750E+05, 1.450246E+05, 1.456757E+05, 1.463284E+05, 1.469826E+05, 1.476384E+05,
    1.482957E+05, 1.489545E+05, 1.496149E+05, 1.502769E+05, 1.509404E+05, 1.516054E+05,
    1.522720E+05, 1.529401E+05, 1.536098E+05, 1.542810E+05, 1.549538E+05, 1.556281E+05,
    1.563039E+05, 1.569813E+05, 1.576602E+05, 1.583407E+05, 1.590227E+05, 1.597063E+05,
    1.603914E+05,
])

#  ---------------------- M = 5, I = 7 ---------------------------
M = 5
I = 7
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[3]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.018640E+00, 8.214730E+00, 1.608377E+01, 2.395650E+01, 3.183067E+01, 3.970586E+01,
    4.758192E+01, 5.545878E+01, 6.333642E+01, 7.121483E+01, 7.909401E+01, 8.697401E+01,
    9.485494E+01, 1.027371E+02, 1.106209E+02, 1.185071E+02, 1.263968E+02, 1.342914E+02,
    1.421927E+02, 1.501028E+02, 1.580244E+02, 1.659601E+02, 1.739131E+02, 1.818867E+02,
    1.898844E+02, 1.979096E+02, 2.059660E+02, 2.140572E+02, 2.221869E+02, 2.303586E+02,
    2.385758E+02, 2.468418E+02, 2.551601E+02, 2.635336E+02, 2.719655E+02, 2.804586E+02,
    2.890157E+02, 2.976394E+02, 3.063322E+02, 3.150964E+02, 3.239342E+02, 3.328478E+02,
    3.418392E+02, 3.509100E+02, 3.600623E+02, 3.692975E+02, 3.786172E+02, 3.880229E+02,
    3.975159E+02, 4.070976E+02, 4.167692E+02, 4.265318E+02, 4.363864E+02, 4.463342E+02,
    4.563759E+02, 4.665126E+02, 4.767451E+02, 4.870742E+02, 4.975006E+02, 5.080251E+02,
    5.186483E+02, 5.293708E+02, 5.401933E+02, 5.511163E+02, 5.621404E+02, 5.732661E+02,
    5.844938E+02, 5.958240E+02, 6.072572E+02, 6.187938E+02, 6.304342E+02, 6.421787E+02,
    6.540277E+02, 6.659816E+02, 6.780407E+02, 6.902053E+02, 7.024757E+02, 7.148522E+02,
    7.273350E+02, 7.399245E+02, 7.526209E+02, 7.654244E+02, 7.783353E+02, 7.913538E+02,
    8.044801E+02, 8.177145E+02, 8.310571E+02, 8.445081E+02, 8.580677E+02, 8.717362E+02,
    8.855137E+02, 8.994003E+02, 9.133963E+02, 9.275017E+02, 9.417169E+02, 9.560418E+02,
    9.704767E+02, 9.850217E+02, 9.996770E+02, 1.014443E+03, 1.029319E+03, 1.044306E+03,
    1.059403E+03, 1.074612E+03, 1.089932E+03, 1.105363E+03, 1.120905E+03, 1.136558E+03,
    1.152323E+03, 1.168200E+03, 1.184189E+03, 1.200289E+03, 1.216502E+03, 1.232826E+03,
    1.249263E+03, 1.265812E+03, 1.282473E+03, 1.299247E+03, 1.316134E+03, 1.333133E+03,
    1.350245E+03, 1.367470E+03, 1.384808E+03, 1.402259E+03, 1.419823E+03, 1.437501E+03,
    1.455292E+03, 1.473196E+03, 1.491214E+03, 1.509346E+03, 1.527592E+03, 1.545951E+03,
    1.564424E+03, 1.583011E+03, 1.601713E+03, 1.620528E+03, 1.639458E+03, 1.658502E+03,
    1.677661E+03, 1.696934E+03, 1.716321E+03, 1.735824E+03, 1.755441E+03, 1.775173E+03,
    1.795020E+03, 1.814982E+03, 1.835060E+03, 1.855252E+03, 1.875560E+03, 1.895983E+03,
    1.916521E+03, 1.937175E+03, 1.957945E+03, 1.978830E+03, 1.999832E+03, 2.020949E+03,
    2.042182E+03, 2.063530E+03, 2.084995E+03, 2.106577E+03, 2.128274E+03, 2.150088E+03,
    2.172018E+03, 2.194065E+03, 2.216228E+03, 2.238508E+03, 2.260905E+03, 2.283418E+03,
    2.306048E+03, 2.328796E+03, 2.351660E+03, 2.374642E+03, 2.397740E+03, 2.420956E+03,
    2.444290E+03, 2.467740E+03, 2.491309E+03, 2.514994E+03, 2.538798E+03, 2.562719E+03,
    2.586758E+03, 2.610915E+03, 2.635190E+03, 2.659584E+03, 2.684095E+03, 2.708724E+03,
    2.733472E+03, 2.758338E+03, 2.783322E+03, 2.808426E+03, 2.833647E+03, 2.858988E+03,
    2.884447E+03, 2.910025E+03, 2.935722E+03, 2.961537E+03, 2.987472E+03, 3.013526E+03,
    3.039700E+03, 3.065992E+03, 3.092404E+03, 3.118936E+03, 3.145587E+03, 3.172357E+03,
    3.199248E+03, 3.226258E+03, 3.253388E+03, 3.280638E+03, 3.308007E+03, 3.335497E+03,
    3.363107E+03, 3.390838E+03, 3.418688E+03, 3.446659E+03, 3.474751E+03, 3.502963E+03,
    3.531296E+03, 3.559749E+03, 3.588323E+03, 3.617018E+03, 3.645834E+03, 3.674771E+03,
    3.703829E+03, 3.733008E+03, 3.762308E+03, 3.791730E+03, 3.821273E+03, 3.850937E+03,
    3.880723E+03, 3.910631E+03, 3.940660E+03, 3.970811E+03, 4.001084E+03, 4.031479E+03,
    4.061996E+03, 4.092635E+03, 4.123396E+03, 4.154279E+03, 4.185284E+03, 4.216412E+03,
    4.247663E+03, 4.279035E+03, 4.310531E+03, 4.342149E+03, 4.373890E+03, 4.405753E+03,
    4.437740E+03, 4.469850E+03, 4.502082E+03, 4.534438E+03, 4.566917E+03, 4.599519E+03,
    4.632244E+03, 4.665093E+03, 4.698065E+03, 4.731161E+03, 4.764381E+03, 4.797724E+03,
    4.831191E+03, 4.864782E+03, 4.898496E+03, 4.932335E+03, 4.966298E+03, 5.000385E+03,
    5.034596E+03, 5.068931E+03, 5.103391E+03, 5.137975E+03, 5.172683E+03, 5.207516E+03,
    5.242474E+03, 5.277557E+03, 5.312764E+03, 5.348096E+03, 5.383553E+03, 5.419135E+03,
    5.454842E+03, 5.490674E+03, 5.526631E+03, 5.562713E+03, 5.598921E+03, 5.635254E+03,
    5.671712E+03, 5.708296E+03, 5.745006E+03, 5.781841E+03, 5.818802E+03, 5.855889E+03,
    5.893102E+03, 5.930440E+03, 5.967904E+03, 6.005495E+03, 6.043211E+03, 6.081054E+03,
    6.119023E+03, 6.157118E+03, 6.195340E+03, 6.233688E+03, 6.272162E+03, 6.310763E+03,
    6.349490E+03, 6.388345E+03, 6.427325E+03, 6.466433E+03, 6.505668E+03, 6.545029E+03,
    6.584517E+03, 6.624133E+03, 6.663875E+03, 6.703745E+03, 6.743741E+03, 6.783865E+03,
    6.824117E+03, 6.864495E+03, 6.905001E+03, 6.945635E+03, 6.986396E+03, 7.027284E+03,
    7.068301E+03, 7.109445E+03, 7.150716E+03, 7.192116E+03, 7.233643E+03, 7.275298E+03,
    7.317081E+03, 7.358993E+03, 7.401032E+03, 7.443199E+03, 7.485495E+03, 7.527918E+03,
    7.570470E+03, 7.613151E+03, 7.655959E+03, 7.698896E+03, 7.741962E+03, 7.785156E+03,
    7.828478E+03, 7.871929E+03, 7.915509E+03, 7.959218E+03, 8.003055E+03, 8.047021E+03,
    8.091116E+03, 8.135339E+03, 8.179692E+03, 8.224173E+03, 8.268784E+03, 8.313523E+03,
    8.358392E+03, 8.403390E+03, 8.448517E+03, 8.493773E+03, 8.539158E+03, 8.584673E+03,
    8.630317E+03, 8.676090E+03, 8.721993E+03, 8.768025E+03, 8.814186E+03, 8.860478E+03,
    8.906898E+03, 8.953448E+03, 9.000128E+03, 9.046938E+03, 9.093877E+03, 9.140946E+03,
    9.188144E+03, 9.235473E+03, 9.282931E+03, 9.330519E+03, 9.378237E+03, 9.426085E+03,
    9.474063E+03, 9.522171E+03, 9.570408E+03, 9.618776E+03, 9.667274E+03, 9.715902E+03,
    9.764660E+03, 9.813548E+03, 9.862566E+03, 9.911715E+03, 9.960994E+03, 1.001040E+04,
    1.005994E+04, 1.010961E+04, 1.015941E+04, 1.020934E+04, 1.025940E+04, 1.030959E+04,
    1.035991E+04, 1.041037E+04, 1.046095E+04, 1.051166E+04, 1.056250E+04, 1.061348E+04,
    1.066458E+04, 1.071581E+04, 1.076718E+04, 1.081867E+04, 1.087030E+04, 1.092206E+04,
    1.097395E+04, 1.102596E+04, 1.107811E+04, 1.113039E+04, 1.118280E+04, 1.123534E+04,
    1.128801E+04, 1.134081E+04, 1.139375E+04, 1.144681E+04, 1.150000E+04, 1.155333E+04,
    1.160678E+04, 1.166037E+04, 1.171409E+04, 1.176793E+04, 1.182191E+04, 1.187602E+04,
    1.193026E+04, 1.198463E+04, 1.203913E+04, 1.209377E+04, 1.214853E+04, 1.220342E+04,
    1.225845E+04, 1.231360E+04, 1.236889E+04, 1.242431E+04, 1.247985E+04, 1.253553E+04,
    1.259134E+04, 1.264728E+04, 1.270335E+04, 1.275955E+04, 1.281589E+04, 1.287235E+04,
    1.292894E+04, 1.298567E+04, 1.304252E+04, 1.309951E+04, 1.315662E+04, 1.321387E+04,
    1.327125E+04, 1.332876E+04, 1.338640E+04, 1.344417E+04, 1.350207E+04, 1.356010E+04,
    1.361826E+04,
])

#  ---------------------- M = 5, I = 8 ---------------------------
M = 5
I = 8
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[3]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.024260E+00, 8.644900E+00, 1.694486E+01, 2.524838E+01, 3.355335E+01, 4.185935E+01,
    5.016626E+01, 5.847401E+01, 6.678257E+01, 7.509194E+01, 8.340213E+01, 9.171321E+01,
    1.000254E+02, 1.083389E+02, 1.166545E+02, 1.249731E+02, 1.332960E+02, 1.416249E+02,
    1.499621E+02, 1.583101E+02, 1.666720E+02, 1.750509E+02, 1.834505E+02, 1.918746E+02,
    2.003270E+02, 2.088118E+02, 2.173329E+02, 2.258944E+02, 2.345004E+02, 2.431546E+02,
    2.518609E+02, 2.606229E+02, 2.694442E+02, 2.783282E+02, 2.872781E+02, 2.962969E+02,
    3.053877E+02, 3.145531E+02, 3.237958E+02, 3.331182E+02, 3.425226E+02, 3.520113E+02,
    3.615863E+02, 3.712495E+02, 3.810028E+02, 3.908478E+02, 4.007862E+02, 4.108194E+02,
    4.209489E+02, 4.311760E+02, 4.415019E+02, 4.519278E+02, 4.624548E+02, 4.730840E+02,
    4.838163E+02, 4.946525E+02, 5.055937E+02, 5.166406E+02, 5.277939E+02, 5.390545E+02,
    5.504229E+02, 5.618998E+02, 5.734859E+02, 5.851816E+02, 5.969876E+02, 6.089044E+02,
    6.209324E+02, 6.330722E+02, 6.453241E+02, 6.576887E+02, 6.701662E+02, 6.827571E+02,
    6.954618E+02, 7.082806E+02, 7.212138E+02, 7.342617E+02, 7.474247E+02, 7.607031E+02,
    7.740971E+02, 7.876070E+02, 8.012330E+02, 8.149755E+02, 8.288346E+02, 8.428107E+02,
    8.569038E+02, 8.711142E+02, 8.854422E+02, 8.998879E+02, 9.144515E+02, 9.291333E+02,
    9.439334E+02, 9.588519E+02, 9.738891E+02, 9.890452E+02, 1.004320E+03, 1.019714E+03,
    1.035228E+03, 1.050861E+03, 1.066613E+03, 1.082486E+03, 1.098478E+03, 1.114590E+03,
    1.130822E+03, 1.147175E+03, 1.163648E+03, 1.180242E+03, 1.196956E+03, 1.213792E+03,
    1.230748E+03, 1.247825E+03, 1.265023E+03, 1.282343E+03, 1.299784E+03, 1.317346E+03,
    1.335031E+03, 1.352836E+03, 1.370764E+03, 1.388814E+03, 1.406986E+03, 1.425280E+03,
    1.443696E+03, 1.462234E+03, 1.480895E+03, 1.499679E+03, 1.518585E+03, 1.537614E+03,
    1.556766E+03, 1.576041E+03, 1.595439E+03, 1.614960E+03, 1.634605E+03, 1.654372E+03,
    1.674264E+03, 1.694278E+03, 1.714417E+03, 1.734679E+03, 1.755065E+03, 1.775574E+03,
    1.796208E+03, 1.816966E+03, 1.837848E+03, 1.858855E+03, 1.879985E+03, 1.901240E+03,
    1.922620E+03, 1.944124E+03, 1.965753E+03, 1.987507E+03, 2.009386E+03, 2.031390E+03,
    2.053519E+03, 2.075772E+03, 2.098152E+03, 2.120656E+03, 2.143286E+03, 2.166041E+03,
    2.188922E+03, 2.211929E+03, 2.235061E+03, 2.258320E+03, 2.281704E+03, 2.305214E+03,
    2.328850E+03, 2.352613E+03, 2.376502E+03, 2.400517E+03, 2.424658E+03, 2.448926E+03,
    2.473321E+03, 2.497842E+03, 2.522490E+03, 2.547265E+03, 2.572167E+03, 2.597196E+03,
    2.622352E+03, 2.647635E+03, 2.673046E+03, 2.698584E+03, 2.724249E+03, 2.750042E+03,
    2.775962E+03, 2.802011E+03, 2.828187E+03, 2.854490E+03, 2.880922E+03, 2.907482E+03,
    2.934170E+03, 2.960986E+03, 2.987930E+03, 3.015003E+03, 3.042204E+03, 3.069534E+03,
    3.096992E+03, 3.124579E+03, 3.152295E+03, 3.180140E+03, 3.208113E+03, 3.236216E+03,
    3.264448E+03, 3.292809E+03, 3.321299E+03, 3.349918E+03, 3.378667E+03, 3.407546E+03,
    3.436554E+03, 3.465692E+03, 3.494959E+03, 3.524357E+03, 3.553884E+03, 3.583542E+03,
    3.613329E+03, 3.643247E+03, 3.673295E+03, 3.703473E+03, 3.733781E+03, 3.764221E+03,
    3.794790E+03, 3.825491E+03, 3.856322E+03, 3.887284E+03, 3.918377E+03, 3.949600E+03,
    3.980955E+03, 4.012441E+03, 4.044059E+03, 4.075807E+03, 4.107687E+03, 4.139699E+03,
    4.171842E+03, 4.204116E+03, 4.236523E+03, 4.269061E+03, 4.301731E+03, 4.334532E+03,
    4.367466E+03, 4.400532E+03, 4.433731E+03, 4.467061E+03, 4.500524E+03, 4.534119E+03,
    4.567847E+03, 4.601707E+03, 4.635700E+03, 4.669825E+03, 4.704084E+03, 4.738475E+03,
    4.772999E+03, 4.807656E+03, 4.842447E+03, 4.877370E+03, 4.912427E+03, 4.947617E+03,
    4.982940E+03, 5.018397E+03, 5.053988E+03, 5.089712E+03, 5.125570E+03, 5.161561E+03,
    5.197686E+03, 5.233946E+03, 5.270339E+03, 5.306866E+03, 5.343528E+03, 5.380323E+03,
    5.417253E+03, 5.454318E+03, 5.491516E+03, 5.528849E+03, 5.566317E+03, 5.603919E+03,
    5.641656E+03, 5.679528E+03, 5.717535E+03, 5.755676E+03, 5.793953E+03, 5.832364E+03,
    5.870911E+03, 5.909592E+03, 5.948409E+03, 5.987362E+03, 6.026449E+03, 6.065672E+03,
    6.105031E+03, 6.144525E+03, 6.184155E+03, 6.223920E+03, 6.263821E+03, 6.303858E+03,
    6.344031E+03, 6.384340E+03, 6.424784E+03, 6.465365E+03, 6.506082E+03, 6.546935E+03,
    6.587924E+03, 6.629050E+03, 6.670312E+03, 6.711710E+03, 6.753245E+03, 6.794916E+03,
    6.836724E+03, 6.878669E+03, 6.920750E+03, 6.962968E+03, 7.005323E+03, 7.047814E+03,
    7.090443E+03, 7.133208E+03, 7.176111E+03, 7.219151E+03, 7.262327E+03, 7.305641E+03,
    7.349092E+03, 7.392681E+03, 7.436407E+03, 7.480270E+03, 7.524270E+03, 7.568408E+03,
    7.612684E+03, 7.657097E+03, 7.701647E+03, 7.746336E+03, 7.791162E+03, 7.836126E+03,
    7.881227E+03, 7.926467E+03, 7.971844E+03, 8.017359E+03, 8.063013E+03, 8.108804E+03,
    8.154733E+03, 8.200801E+03, 8.247006E+03, 8.293350E+03, 8.339832E+03, 8.386452E+03,
    8.433210E+03, 8.480107E+03, 8.527142E+03, 8.574316E+03, 8.621628E+03, 8.669078E+03,
    8.716667E+03, 8.764395E+03, 8.812261E+03, 8.860266E+03, 8.908409E+03, 8.956691E+03,
    9.005112E+03, 9.053671E+03, 9.102369E+03, 9.151206E+03, 9.200182E+03, 9.249297E+03,
    9.298551E+03, 9.347943E+03, 9.397474E+03, 9.447145E+03, 9.496954E+03, 9.546902E+03,
    9.596990E+03, 9.647216E+03, 9.697582E+03, 9.748087E+03, 9.798730E+03, 9.849513E+03,
    9.900435E+03, 9.951496E+03, 1.000270E+04, 1.005404E+04, 1.010552E+04, 1.015713E+04,
    1.020889E+04, 1.026079E+04, 1.031282E+04, 1.036500E+04, 1.041731E+04, 1.046977E+04,
    1.052236E+04, 1.057509E+04, 1.062797E+04, 1.068098E+04, 1.073413E+04, 1.078742E+04,
    1.084085E+04, 1.089442E+04, 1.094813E+04, 1.100198E+04, 1.105596E+04, 1.111009E+04,
    1.116436E+04, 1.121876E+04, 1.127331E+04, 1.132799E+04, 1.138282E+04, 1.143778E+04,
    1.149289E+04, 1.154813E+04, 1.160351E+04, 1.165903E+04, 1.171470E+04, 1.177050E+04,
    1.182644E+04, 1.188252E+04, 1.193874E+04, 1.199510E+04, 1.205159E+04, 1.210823E+04,
    1.216501E+04, 1.222193E+04, 1.227898E+04, 1.233618E+04, 1.239351E+04, 1.245099E+04,
    1.250860E+04, 1.256636E+04, 1.262425E+04, 1.268228E+04, 1.274045E+04, 1.279876E+04,
    1.285722E+04, 1.291581E+04, 1.297453E+04, 1.303340E+04, 1.309241E+04, 1.315156E+04,
    1.321084E+04, 1.327027E+04, 1.332983E+04, 1.338954E+04, 1.344938E+04, 1.350936E+04,
    1.356949E+04, 1.362975E+04, 1.369015E+04, 1.375069E+04, 1.381136E+04, 1.387218E+04,
    1.393314E+04, 1.399423E+04, 1.405547E+04, 1.411684E+04, 1.417835E+04, 1.424000E+04,
    1.430179E+04, 1.436372E+04, 1.442579E+04, 1.448799E+04, 1.455034E+04, 1.461282E+04,
    1.467545E+04,
])

#  ---------------------- M = 5, I = 9 ---------------------------
M = 5
I = 9
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[3]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    6.128630E+00, 5.062329E+01, 9.917483E+01, 1.477481E+02, 1.963300E+02, 2.449180E+02,
    2.935114E+02, 3.421098E+02, 3.907129E+02, 4.393208E+02, 4.879334E+02, 5.365511E+02,
    5.851749E+02, 6.338065E+02, 6.824492E+02, 7.311080E+02, 7.797902E+02, 8.285052E+02,
    8.772651E+02, 9.260840E+02, 9.749786E+02, 1.023967E+03, 1.073070E+03, 1.122307E+03,
    1.171702E+03, 1.221277E+03, 1.271055E+03, 1.321059E+03, 1.371312E+03, 1.421836E+03,
    1.472653E+03, 1.523784E+03, 1.575250E+03, 1.627071E+03, 1.679265E+03, 1.731850E+03,
    1.784843E+03, 1.838260E+03, 1.892117E+03, 1.946428E+03, 2.001207E+03, 2.056467E+03,
    2.112219E+03, 2.168475E+03, 2.225246E+03, 2.282542E+03, 2.340373E+03, 2.398746E+03,
    2.457671E+03, 2.517156E+03, 2.577207E+03, 2.637832E+03, 2.699037E+03, 2.760829E+03,
    2.823213E+03, 2.886195E+03, 2.949779E+03, 3.013971E+03, 3.078775E+03, 3.144196E+03,
    3.210237E+03, 3.276902E+03, 3.344196E+03, 3.412121E+03, 3.480680E+03, 3.549878E+03,
    3.619716E+03, 3.690198E+03, 3.761326E+03, 3.833102E+03, 3.905530E+03, 3.978611E+03,
    4.052348E+03, 4.126742E+03, 4.201796E+03, 4.277511E+03, 4.353890E+03, 4.430934E+03,
    4.508645E+03, 4.587024E+03, 4.666073E+03, 4.745793E+03, 4.826187E+03, 4.907254E+03,
    4.988997E+03, 5.071417E+03, 5.154515E+03, 5.238293E+03, 5.322751E+03, 5.407890E+03,
    5.493712E+03, 5.580218E+03, 5.667408E+03, 5.755285E+03, 5.843848E+03, 5.933099E+03,
    6.023038E+03, 6.113667E+03, 6.204986E+03, 6.296996E+03, 6.389699E+03, 6.483094E+03,
    6.577183E+03, 6.671967E+03, 6.767446E+03, 6.863621E+03, 6.960492E+03, 7.058061E+03,
    7.156329E+03, 7.255295E+03, 7.354961E+03, 7.455327E+03, 7.556394E+03, 7.658162E+03,
    7.760633E+03, 7.863807E+03, 7.967684E+03, 8.072266E+03, 8.177552E+03, 8.283544E+03,
    8.390242E+03, 8.497647E+03, 8.605758E+03, 8.714578E+03, 8.824106E+03, 8.934343E+03,
    9.045289E+03, 9.156946E+03, 9.269313E+03, 9.382392E+03, 9.496182E+03, 9.610685E+03,
    9.725901E+03, 9.841830E+03, 9.958473E+03, 1.007583E+04, 1.019390E+04, 1.031269E+04,
    1.043220E+04, 1.055242E+04, 1.067336E+04, 1.079501E+04, 1.091739E+04, 1.104048E+04,
    1.116429E+04, 1.128882E+04, 1.141408E+04, 1.154005E+04, 1.166674E+04, 1.179416E+04,
    1.192230E+04, 1.205116E+04, 1.218075E+04, 1.231106E+04, 1.244209E+04, 1.257385E+04,
    1.270633E+04, 1.283954E+04, 1.297348E+04, 1.310814E+04, 1.324353E+04, 1.337965E+04,
    1.351649E+04, 1.365407E+04, 1.379237E+04, 1.393140E+04, 1.407117E+04, 1.421166E+04,
    1.435289E+04, 1.449485E+04, 1.463754E+04, 1.478096E+04, 1.492512E+04, 1.507001E+04,
    1.521563E+04, 1.536199E+04, 1.550909E+04, 1.565692E+04, 1.580548E+04, 1.595479E+04,
    1.610483E+04, 1.625560E+04, 1.640712E+04, 1.655937E+04, 1.671237E+04, 1.686610E+04,
    1.702057E+04, 1.717579E+04, 1.733174E+04, 1.748844E+04, 1.764588E+04, 1.780406E+04,
    1.796298E+04, 1.812265E+04, 1.828306E+04, 1.844421E+04, 1.860611E+04, 1.876876E+04,
    1.893215E+04, 1.909629E+04, 1.926117E+04, 1.942680E+04, 1.959318E+04, 1.976031E+04,
    1.992819E+04, 2.009681E+04, 2.026619E+04, 2.043631E+04, 2.060719E+04, 2.077882E+04,
    2.095120E+04, 2.112433E+04, 2.129821E+04, 2.147285E+04, 2.164824E+04, 2.182438E+04,
    2.200128E+04, 2.217893E+04, 2.235734E+04, 2.253651E+04, 2.271643E+04, 2.289711E+04,
    2.307854E+04, 2.326073E+04, 2.344368E+04, 2.362739E+04, 2.381186E+04, 2.399709E+04,
    2.418308E+04, 2.436983E+04, 2.455734E+04, 2.474561E+04, 2.493464E+04, 2.512444E+04,
    2.531500E+04, 2.550632E+04, 2.569841E+04, 2.589126E+04, 2.608487E+04, 2.627925E+04,
    2.647440E+04, 2.667031E+04, 2.686699E+04, 2.706444E+04, 2.726265E+04, 2.746163E+04,
    2.766138E+04, 2.786190E+04, 2.806319E+04, 2.826524E+04, 2.846807E+04, 2.867167E+04,
    2.887604E+04, 2.908118E+04, 2.928709E+04, 2.949378E+04, 2.970124E+04, 2.990947E+04,
    3.011847E+04, 3.032825E+04, 3.053880E+04, 3.075013E+04, 3.096224E+04, 3.117512E+04,
    3.138877E+04, 3.160321E+04, 3.181842E+04, 3.203441E+04, 3.225117E+04, 3.246872E+04,
    3.268704E+04, 3.290614E+04, 3.312602E+04, 3.334669E+04, 3.356813E+04, 3.379035E+04,
    3.401336E+04, 3.423714E+04, 3.446171E+04, 3.468706E+04, 3.491320E+04, 3.514011E+04,
    3.536782E+04, 3.559630E+04, 3.582557E+04, 3.605563E+04, 3.628647E+04, 3.651809E+04,
    3.675050E+04, 3.698370E+04, 3.721769E+04, 3.745246E+04, 3.768802E+04, 3.792437E+04,
    3.816150E+04, 3.839943E+04, 3.863814E+04, 3.887764E+04, 3.911794E+04, 3.935902E+04,
    3.960089E+04, 3.984356E+04, 4.008701E+04, 4.033126E+04, 4.057630E+04, 4.082213E+04,
    4.106875E+04, 4.131617E+04, 4.156438E+04, 4.181338E+04, 4.206318E+04, 4.231377E+04,
    4.256515E+04, 4.281734E+04, 4.307031E+04, 4.332408E+04, 4.357865E+04, 4.383401E+04,
    4.409017E+04, 4.434713E+04, 4.460488E+04, 4.486344E+04, 4.512278E+04, 4.538293E+04,
    4.564388E+04, 4.590562E+04, 4.616816E+04, 4.643150E+04, 4.669564E+04, 4.696058E+04,
    4.722632E+04, 4.749286E+04, 4.776021E+04, 4.802835E+04, 4.829729E+04, 4.856703E+04,
    4.883758E+04, 4.910893E+04, 4.938108E+04, 4.965403E+04, 4.992778E+04, 5.020234E+04,
    5.047770E+04, 5.075386E+04, 5.103083E+04, 5.130860E+04, 5.158717E+04, 5.186655E+04,
    5.214674E+04, 5.242772E+04, 5.270951E+04, 5.299211E+04, 5.327551E+04, 5.355972E+04,
    5.384474E+04, 5.413056E+04, 5.441718E+04, 5.470461E+04, 5.499285E+04, 5.528189E+04,
    5.557175E+04, 5.586240E+04, 5.615387E+04, 5.644614E+04, 5.673922E+04, 5.703311E+04,
    5.732781E+04, 5.762331E+04, 5.791962E+04, 5.821674E+04, 5.851467E+04, 5.881341E+04,
    5.911295E+04, 5.941331E+04, 5.971447E+04, 6.001644E+04, 6.031922E+04, 6.062281E+04,
    6.092721E+04, 6.123242E+04, 6.153844E+04, 6.184527E+04, 6.215291E+04, 6.246136E+04,
    6.277062E+04, 6.308069E+04, 6.339157E+04, 6.370326E+04, 6.401576E+04, 6.432907E+04,
    6.464320E+04, 6.495813E+04, 6.527387E+04, 6.559043E+04, 6.590779E+04, 6.622597E+04,
    6.654496E+04, 6.686476E+04, 6.718537E+04, 6.750679E+04, 6.782902E+04, 6.815206E+04,
    6.847592E+04, 6.880058E+04, 6.912606E+04, 6.945235E+04, 6.977945E+04, 7.010736E+04,
    7.043609E+04, 7.076562E+04, 7.109597E+04, 7.142713E+04, 7.175910E+04, 7.209188E+04,
    7.242547E+04, 7.275988E+04, 7.309509E+04, 7.343112E+04, 7.376796E+04, 7.410561E+04,
    7.444407E+04, 7.478335E+04, 7.512343E+04, 7.546433E+04, 7.580603E+04, 7.614855E+04,
    7.649188E+04, 7.683602E+04, 7.718098E+04, 7.752674E+04, 7.787331E+04, 7.822070E+04,
    7.856889E+04, 7.891790E+04, 7.926772E+04, 7.961835E+04, 7.996979E+04, 8.032204E+04,
    8.067510E+04, 8.102897E+04, 8.138365E+04, 8.173914E+04, 8.209544E+04, 8.245255E+04,
    8.281047E+04, 8.316920E+04, 8.352874E+04, 8.388909E+04, 8.425025E+04, 8.461221E+04,
    8.497499E+04,
])

#  ---------------------- M = 6, I = 1 ---------------------------
M = 6
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    5.000000E+00, 1.273695E+01, 3.049472E+01, 5.478766E+01, 8.367753E+01, 1.164151E+02,
    1.525835E+02, 1.918873E+02, 2.341011E+02, 2.790524E+02, 3.266165E+02, 3.767179E+02,
    4.293326E+02, 4.844902E+02, 5.422734E+02, 6.028155E+02, 6.662963E+02, 7.329384E+02,
    8.030016E+02, 8.767796E+02, 9.545964E+02, 1.036804E+03, 1.123780E+03, 1.215929E+03,
    1.313679E+03, 1.417485E+03, 1.527828E+03, 1.645218E+03, 1.770193E+03, 1.903324E+03,
    2.045215E+03, 2.196504E+03, 2.357871E+03, 2.530032E+03, 2.713751E+03, 2.909836E+03,
    3.119144E+03, 3.342585E+03, 3.581125E+03, 3.835788E+03, 4.107662E+03, 4.397900E+03,
    4.707726E+03, 5.038440E+03, 5.391417E+03, 5.768118E+03, 6.170091E+03, 6.598978E+03,
    7.056516E+03, 7.544548E+03, 8.065026E+03, 8.620015E+03, 9.211704E+03, 9.842405E+03,
    1.051457E+04, 1.123079E+04, 1.199379E+04, 1.280649E+04, 1.367192E+04, 1.459334E+04,
    1.557414E+04, 1.661792E+04, 1.772850E+04, 1.890986E+04, 2.016624E+04, 2.150208E+04,
    2.292207E+04, 2.443115E+04, 2.603450E+04, 2.773758E+04, 2.954614E+04, 3.150651E+04,
    3.355181E+04, 3.572292E+04, 3.802703E+04, 4.047168E+04, 4.306485E+04, 4.581488E+04,
    4.873055E+04, 5.182111E+04, 5.509623E+04, 5.856610E+04, 6.224140E+04, 6.613336E+04,
    7.025373E+04, 7.461487E+04, 7.922970E+04, 8.411182E+04, 8.927543E+04, 9.473544E+04,
    1.005075E+05, 1.066079E+05, 1.130538E+05, 1.198631E+05, 1.270545E+05, 1.346478E+05,
    1.426634E+05, 1.511227E+05, 1.600483E+05, 1.694635E+05, 1.793929E+05, 1.898620E+05,
    2.008977E+05, 2.125277E+05, 2.247814E+05, 2.376890E+05, 2.512823E+05, 2.655945E+05,
    2.806600E+05, 2.965148E+05, 3.131965E+05, 3.307442E+05, 3.491987E+05, 3.686022E+05,
    3.889992E+05, 4.104356E+05, 4.329593E+05, 4.566203E+05, 4.814703E+05, 5.075634E+05,
    5.349557E+05, 5.637057E+05, 5.938740E+05, 6.255236E+05, 6.587202E+05, 6.935319E+05,
    7.300294E+05, 7.682862E+05, 8.083785E+05, 8.503856E+05, 8.943897E+05, 9.404760E+05,
    9.887330E+05, 1.039252E+06, 1.092129E+06, 1.147463E+06, 1.205355E+06, 1.265911E+06,
    1.329242E+06, 1.395461E+06, 1.464686E+06, 1.537038E+06, 1.612646E+06, 1.691638E+06,
    1.774151E+06, 1.860324E+06, 1.950303E+06, 2.044237E+06, 2.142281E+06, 2.244596E+06,
    2.351347E+06, 2.462706E+06, 2.578850E+06, 2.699960E+06, 2.826228E+06, 2.957846E+06,
    3.095017E+06, 3.237950E+06, 3.386857E+06, 3.541963E+06, 3.703494E+06, 3.871687E+06,
    4.046785E+06, 4.229040E+06, 4.418710E+06, 4.616063E+06, 4.821373E+06, 5.034924E+06,
    5.257009E+06, 5.487928E+06, 5.727992E+06, 5.977520E+06, 6.236841E+06, 6.506295E+06,
    6.786230E+06, 7.077006E+06,
])

#  ---------------------- M = 6, I = 2 ---------------------------
M = 6
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.000001E+01, 2.547411E+01, 6.098959E+01, 1.095756E+02, 1.673556E+02, 2.328309E+02,
    3.051677E+02, 3.837737E+02, 4.682005E+02, 5.581019E+02, 6.532270E+02, 7.534273E+02,
    8.586439E+02, 9.689291E+02, 1.084461E+03, 1.205482E+03, 1.332363E+03, 1.465530E+03,
    1.605503E+03, 1.752860E+03, 1.908229E+03, 2.072336E+03, 2.245903E+03, 2.429737E+03,
    2.624664E+03, 2.831592E+03, 3.051489E+03, 3.285303E+03, 3.534174E+03, 3.799126E+03,
    4.081393E+03, 4.382239E+03, 4.702959E+03, 5.044945E+03, 5.409694E+03, 5.798822E+03,
    6.213891E+03, 6.656804E+03, 7.129283E+03, 7.633410E+03, 8.171277E+03, 8.745069E+03,
    9.357183E+03, 1.001001E+04, 1.070636E+04, 1.144891E+04, 1.224067E+04, 1.308471E+04,
    1.398443E+04, 1.494334E+04, 1.596502E+04, 1.705351E+04, 1.821287E+04, 1.944758E+04,
    2.076216E+04, 2.216156E+04, 2.365078E+04, 2.523547E+04, 2.692124E+04, 2.871415E+04,
    3.062057E+04, 3.264725E+04, 3.480122E+04, 3.708988E+04, 3.952125E+04, 4.210358E+04,
    4.484543E+04, 4.775615E+04, 5.084522E+04, 5.412274E+04, 5.759952E+04, 6.128669E+04,
    6.519606E+04, 6.933999E+04, 7.373142E+04, 7.838393E+04, 8.331199E+04, 8.853045E+04,
    9.405527E+04, 9.990284E+04, 1.060906E+05, 1.126365E+05, 1.195598E+05, 1.268805E+05,
    1.346193E+05, 1.427982E+05, 1.514402E+05, 1.605692E+05, 1.702104E+05, 1.803902E+05,
    1.911358E+05, 2.024762E+05, 2.144415E+05, 2.270628E+05, 2.403733E+05, 2.544072E+05,
    2.692002E+05, 2.847901E+05, 3.012156E+05, 3.185180E+05, 3.367395E+05, 3.559247E+05,
    3.761201E+05, 3.973740E+05, 4.197368E+05, 4.432611E+05, 4.680017E+05, 4.940161E+05,
    5.213635E+05, 5.501059E+05, 5.803083E+05, 6.120376E+05, 6.453640E+05, 6.803606E+05,
    7.171028E+05, 7.556704E+05, 7.961449E+05, 8.386119E+05, 8.831604E+05, 9.298828E+05,
    9.788752E+05, 1.030238E+06, 1.084073E+06, 1.140491E+06, 1.199602E+06, 1.261522E+06,
    1.326373E+06, 1.394281E+06, 1.465374E+06, 1.539790E+06, 1.617667E+06, 1.699151E+06,
    1.784394E+06, 1.873552E+06, 1.966787E+06, 2.064268E+06, 2.166169E+06, 2.272673E+06,
    2.383965E+06, 2.500240E+06, 2.621701E+06, 2.748556E+06, 2.881020E+06, 3.019317E+06,
    3.163680E+06, 3.314348E+06, 3.471569E+06, 3.635602E+06, 3.806711E+06, 3.985172E+06,
    4.171270E+06, 4.365299E+06, 4.567564E+06, 4.778380E+06, 4.998073E+06, 5.226979E+06,
    5.465447E+06, 5.713837E+06, 5.972520E+06, 6.241880E+06, 6.522314E+06, 6.814234E+06,
    7.118060E+06, 7.434232E+06, 7.763201E+06, 8.105433E+06, 8.461408E+06, 8.831625E+06,
    9.216596E+06, 9.616850E+06, 1.003293E+07, 1.046541E+07, 1.091485E+07, 1.138187E+07,
    1.186708E+07, 1.237112E+07,
])

#  ---------------------- M = 6, I = 3 ---------------------------
M = 6
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[4]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    8.136790E+00, 9.190122E+01, 2.433209E+02, 4.407898E+02, 6.744479E+02, 9.391965E+02,
    1.231752E+03, 1.549735E+03, 1.891357E+03, 2.255315E+03, 2.640780E+03, 3.047413E+03,
    3.475380E+03, 3.925354E+03, 4.398488E+03, 4.896378E+03, 5.421024E+03, 5.974778E+03,
    6.560313E+03, 7.180591E+03, 7.838835E+03, 8.538523E+03, 9.283379E+03, 1.007737E+04,
    1.092472E+04, 1.182991E+04, 1.279770E+04, 1.383315E+04, 1.494161E+04, 1.612880E+04,
    1.740075E+04, 1.876392E+04, 2.022515E+04, 2.179171E+04, 2.347136E+04, 2.527233E+04,
    2.720338E+04, 2.927384E+04, 3.149364E+04, 3.387330E+04, 3.642407E+04, 3.915786E+04,
    4.208734E+04, 4.522600E+04, 4.858815E+04, 5.218897E+04, 5.604463E+04, 6.017224E+04,
    6.458999E+04, 6.931716E+04, 7.437420E+04, 7.978279E+04, 8.556591E+04, 9.174789E+04,
    9.835451E+04, 1.054131E+05, 1.129524E+05, 1.210032E+05, 1.295976E+05, 1.387699E+05,
    1.485562E+05, 1.589946E+05, 1.701255E+05, 1.819914E+05, 1.946373E+05, 2.081105E+05,
    2.224612E+05, 2.377420E+05, 2.540087E+05, 2.713198E+05, 2.897370E+05, 3.093255E+05,
    3.301538E+05, 3.522940E+05, 3.758220E+05, 4.008176E+05, 4.273649E+05, 4.555521E+05,
    4.854721E+05, 5.172225E+05, 5.509057E+05, 5.866294E+05, 6.245067E+05, 6.646560E+05,
    7.072021E+05, 7.522754E+05, 8.000130E+05, 8.505587E+05, 9.040628E+05, 9.606834E+05,
    1.020586E+06, 1.083943E+06, 1.150937E+06, 1.221758E+06, 1.296603E+06, 1.375682E+06,
    1.459212E+06, 1.547420E+06, 1.640545E+06, 1.738836E+06, 1.842553E+06, 1.951967E+06,
    2.067363E+06, 2.189037E+06, 2.317299E+06, 2.452473E+06, 2.594895E+06, 2.744919E+06,
    2.902912E+06, 3.069257E+06, 3.244354E+06, 3.428620E+06, 3.622489E+06, 3.826415E+06,
    4.040869E+06, 4.266343E+06, 4.503349E+06, 4.752420E+06, 5.014112E+06, 5.289002E+06,
    5.577692E+06, 5.880808E+06, 6.199001E+06, 6.532948E+06, 6.883354E+06, 7.250950E+06,
    7.636499E+06, 8.040792E+06, 8.464650E+06, 8.908929E+06, 9.374515E+06, 9.862331E+06,
    1.037333E+07, 1.090852E+07, 1.146891E+07, 1.205559E+07, 1.266966E+07, 1.331228E+07,
    1.398464E+07, 1.468798E+07, 1.542360E+07, 1.619281E+07, 1.699702E+07, 1.783764E+07,
    1.871616E+07, 1.963413E+07, 2.059313E+07, 2.159483E+07, 2.264092E+07, 2.373319E+07,
    2.487345E+07, 2.606363E+07, 2.730567E+07, 2.860161E+07, 2.995356E+07, 3.136369E+07,
    3.283426E+07, 3.436760E+07, 3.596611E+07, 3.763229E+07, 3.936871E+07, 4.117804E+07,
    4.306303E+07, 4.502651E+07, 4.707144E+07, 4.920084E+07, 5.141786E+07, 5.372573E+07,
    5.612780E+07, 5.862752E+07, 6.122847E+07, 6.393432E+07, 6.674887E+07, 6.967606E+07,
    7.271992E+07, 7.588465E+07, 7.917453E+07, 8.259403E+07, 8.614773E+07, 8.984036E+07,
    9.367680E+07, 9.766207E+07, 1.018014E+08, 1.061000E+08, 1.105636E+08, 1.151977E+08,
    1.200082E+08, 1.250011E+08, 1.301826E+08, 1.355593E+08, 1.411375E+08, 1.469242E+08,
    1.529262E+08, 1.591509E+08, 1.656056E+08, 1.722979E+08, 1.792357E+08, 1.864271E+08,
    1.938803E+08, 2.016041E+08, 2.096070E+08, 2.178983E+08, 2.264872E+08, 2.353833E+08,
    2.445965E+08, 2.541369E+08, 2.640150E+08, 2.742414E+08, 2.848271E+08, 2.957836E+08,
    3.071224E+08, 3.188555E+08, 3.309952E+08, 3.435542E+08, 3.565454E+08, 3.699822E+08,
    3.838782E+08, 3.982476E+08, 4.131049E+08, 4.284647E+08, 4.443425E+08, 4.607539E+08,
    4.777149E+08, 4.952421E+08, 5.133524E+08, 5.320631E+08,
])

#  ---------------------- M = 6, I = 4 ---------------------------
M = 6
I = 4
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.626998E+01, 1.839479E+02, 4.870414E+02, 8.823115E+02, 1.350022E+03, 1.879967E+03,
    2.465569E+03, 3.102094E+03, 3.785903E+03, 4.514479E+03, 5.286106E+03, 6.100196E+03,
    6.957045E+03, 7.858055E+03, 8.805535E+03, 9.802766E+03, 1.085371E+04, 1.196313E+04,
    1.313628E+04, 1.437935E+04, 1.569859E+04, 1.710107E+04, 1.859418E+04, 2.018607E+04,
    2.188496E+04, 2.370019E+04, 2.564086E+04, 2.771745E+04, 2.994063E+04, 3.232161E+04,
    3.487298E+04, 3.760723E+04, 4.053859E+04, 4.368088E+04, 4.705047E+04, 5.066330E+04,
    5.453713E+04, 5.869100E+04, 6.314418E+04, 6.791804E+04, 7.303545E+04, 7.852005E+04,
    8.439702E+04, 9.069327E+04, 9.743844E+04, 1.046622E+05, 1.123971E+05, 1.206774E+05,
    1.295392E+05, 1.390219E+05, 1.491664E+05, 1.600155E+05, 1.716162E+05, 1.840160E+05,
    1.972679E+05, 2.114260E+05, 2.265476E+05, 2.426952E+05, 2.599326E+05, 2.783288E+05,
    2.979556E+05, 3.188895E+05, 3.412118E+05, 3.650082E+05, 3.903670E+05, 4.173853E+05,
    4.461617E+05, 4.768023E+05, 5.094192E+05, 5.441294E+05, 5.810569E+05, 6.203311E+05,
    6.620900E+05, 7.064783E+05, 7.536471E+05, 8.037577E+05, 8.569763E+05, 9.134819E+05,
    9.734592E+05, 1.037104E+06, 1.104621E+06, 1.176227E+06, 1.252147E+06, 1.332619E+06,
    1.417893E+06, 1.508228E+06, 1.603903E+06, 1.705201E+06, 1.812425E+06, 1.925893E+06,
    2.045933E+06, 2.172894E+06, 2.307139E+06, 2.449047E+06, 2.599017E+06, 2.757465E+06,
    2.924828E+06, 3.101559E+06, 3.288137E+06, 3.485058E+06, 3.692845E+06, 3.912041E+06,
    4.143216E+06, 4.386961E+06, 4.643897E+06, 4.914672E+06, 5.199959E+06, 5.500466E+06,
    5.816930E+06, 6.150113E+06, 6.500818E+06, 6.869878E+06, 7.258162E+06, 7.666580E+06,
    8.096072E+06, 8.547624E+06, 9.022260E+06, 9.521045E+06, 1.004509E+07, 1.059556E+07,
    1.117365E+07, 1.178061E+07, 1.241774E+07, 1.308641E+07, 1.378802E+07, 1.452403E+07,
    1.529597E+07, 1.610542E+07, 1.695402E+07, 1.784349E+07, 1.877560E+07, 1.975219E+07,
    2.077518E+07, 2.184655E+07, 2.296837E+07, 2.414278E+07, 2.537200E+07, 2.665835E+07,
    2.800420E+07, 2.941204E+07, 3.088444E+07, 3.242407E+07, 3.403370E+07, 3.571618E+07,
    3.747450E+07, 3.931172E+07, 4.123103E+07, 4.323575E+07, 4.532928E+07, 4.751517E+07,
    4.979709E+07, 5.217884E+07, 5.466433E+07, 5.725764E+07, 5.996298E+07, 6.278469E+07,
    6.572729E+07, 6.879543E+07, 7.199392E+07, 7.532775E+07, 7.880208E+07, 8.242220E+07,
    8.619364E+07, 9.012207E+07, 9.421338E+07, 9.847362E+07, 1.029091E+08, 1.075262E+08,
    1.123317E+08, 1.173325E+08, 1.225357E+08, 1.279487E+08, 1.335791E+08, 1.394346E+08,
    1.455235E+08, 1.518540E+08,
])

#  ---------------------- M = 7, I = 1 ---------------------------
M = 7
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[5]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.259272E+00, 1.541160E+01, 2.984283E+01, 4.432568E+01, 5.882297E+01, 7.332720E+01,
    8.783584E+01, 1.023478E+02, 1.168627E+02, 1.313804E+02, 1.459015E+02, 1.604275E+02,
    1.749609E+02, 1.895058E+02, 2.040679E+02, 2.186540E+02, 2.332724E+02, 2.479320E+02,
    2.626424E+02, 2.774132E+02, 2.922540E+02, 3.071739E+02, 3.221816E+02, 3.372850E+02,
    3.524912E+02, 3.678063E+02, 3.832358E+02, 3.987841E+02, 4.144551E+02, 4.302515E+02,
    4.461757E+02, 4.622292E+02, 4.784130E+02, 4.947276E+02, 5.111729E+02, 5.277485E+02,
    5.444538E+02, 5.612875E+02, 5.782484E+02, 5.953349E+02, 6.125453E+02, 6.298776E+02,
    6.473299E+02, 6.649000E+02, 6.825859E+02, 7.003851E+02, 7.182956E+02, 7.363149E+02,
    7.544408E+02, 7.726710E+02, 7.910032E+02, 8.094352E+02, 8.279648E+02, 8.465897E+02,
    8.653078E+02, 8.841171E+02, 9.030155E+02, 9.220009E+02, 9.410715E+02, 9.602254E+02,
    9.794607E+02, 9.987757E+02, 1.018169E+03, 1.037638E+03, 1.057182E+03, 1.076799E+03,
    1.096488E+03, 1.116247E+03, 1.136075E+03, 1.155971E+03, 1.175934E+03, 1.195961E+03,
    1.216053E+03, 1.236207E+03, 1.256424E+03, 1.276701E+03, 1.297039E+03, 1.317435E+03,
    1.337890E+03, 1.358403E+03, 1.378972E+03, 1.399598E+03, 1.420279E+03, 1.441014E+03,
    1.461804E+03, 1.482647E+03, 1.503544E+03, 1.524493E+03, 1.545494E+03, 1.566547E+03,
    1.587651E+03, 1.608806E+03, 1.630012E+03, 1.651268E+03, 1.672575E+03, 1.693931E+03,
    1.715336E+03, 1.736791E+03, 1.758295E+03, 1.779848E+03, 1.801450E+03, 1.823100E+03,
    1.844800E+03, 1.866547E+03, 1.888343E+03, 1.910188E+03, 1.932080E+03, 1.954021E+03,
    1.976010E+03, 1.998048E+03, 2.020133E+03, 2.042267E+03, 2.064449E+03, 2.086680E+03,
    2.108959E+03, 2.131286E+03, 2.153662E+03, 2.176086E+03, 2.198559E+03, 2.221081E+03,
    2.243651E+03, 2.266271E+03, 2.288939E+03, 2.311657E+03, 2.334424E+03, 2.357240E+03,
    2.380105E+03, 2.403021E+03, 2.425986E+03, 2.449001E+03, 2.472066E+03, 2.495181E+03,
    2.518346E+03, 2.541562E+03, 2.564828E+03, 2.588145E+03, 2.611513E+03, 2.634932E+03,
    2.658402E+03, 2.681924E+03, 2.705497E+03, 2.729122E+03, 2.752798E+03, 2.776526E+03,
    2.800307E+03, 2.824140E+03, 2.848025E+03, 2.871962E+03, 2.895953E+03, 2.919996E+03,
    2.944092E+03, 2.968242E+03, 2.992444E+03, 3.016701E+03, 3.041010E+03, 3.065374E+03,
    3.089791E+03, 3.114262E+03, 3.138788E+03, 3.163367E+03, 3.188002E+03, 3.212690E+03,
    3.237434E+03, 3.262232E+03, 3.287085E+03, 3.311993E+03, 3.336957E+03, 3.361975E+03,
    3.387050E+03, 3.412179E+03, 3.437365E+03, 3.462606E+03, 3.487903E+03, 3.513256E+03,
    3.538665E+03, 3.564130E+03, 3.589652E+03, 3.615230E+03, 3.640865E+03, 3.666556E+03,
    3.692304E+03, 3.718109E+03, 3.743970E+03, 3.769889E+03, 3.795864E+03, 3.821897E+03,
    3.847987E+03, 3.874134E+03, 3.900338E+03, 3.926600E+03, 3.952920E+03, 3.979297E+03,
    4.005731E+03, 4.032224E+03, 4.058774E+03, 4.085382E+03, 4.112047E+03, 4.138771E+03,
    4.165553E+03, 4.192392E+03, 4.219290E+03, 4.246246E+03, 4.273260E+03, 4.300332E+03,
    4.327463E+03, 4.354651E+03, 4.381898E+03, 4.409204E+03, 4.436567E+03, 4.463989E+03,
    4.491470E+03, 4.519009E+03, 4.546606E+03, 4.574262E+03, 4.601977E+03, 4.629750E+03,
    4.657581E+03, 4.685471E+03, 4.713419E+03, 4.741426E+03, 4.769492E+03, 4.797616E+03,
    4.825799E+03, 4.854040E+03, 4.882340E+03, 4.910698E+03, 4.939115E+03, 4.967591E+03,
    4.996125E+03, 5.024717E+03, 5.053368E+03, 5.082078E+03, 5.110846E+03, 5.139672E+03,
    5.168557E+03, 5.197500E+03, 5.226502E+03, 5.255562E+03, 5.284680E+03, 5.313857E+03,
    5.343091E+03, 5.372384E+03, 5.401736E+03, 5.431145E+03, 5.460613E+03, 5.490138E+03,
    5.519722E+03, 5.549364E+03, 5.579063E+03, 5.608821E+03, 5.638636E+03, 5.668509E+03,
    5.698440E+03, 5.728429E+03, 5.758475E+03, 5.788579E+03, 5.818741E+03, 5.848960E+03,
    5.879236E+03, 5.909570E+03, 5.939961E+03, 5.970409E+03, 6.000915E+03, 6.031477E+03,
    6.062097E+03, 6.092774E+03, 6.123507E+03, 6.154298E+03, 6.185145E+03, 6.216049E+03,
    6.247009E+03, 6.278026E+03, 6.309100E+03, 6.340229E+03, 6.371416E+03, 6.402658E+03,
    6.433957E+03, 6.465311E+03, 6.496722E+03, 6.528188E+03, 6.559710E+03, 6.591288E+03,
    6.622922E+03, 6.654611E+03, 6.686355E+03, 6.718155E+03, 6.750010E+03, 6.781920E+03,
    6.813886E+03, 6.845906E+03, 6.877981E+03, 6.910111E+03, 6.942296E+03, 6.974535E+03,
    7.006828E+03, 7.039176E+03, 7.071578E+03, 7.104035E+03, 7.136545E+03, 7.169109E+03,
    7.201728E+03, 7.234399E+03, 7.267125E+03, 7.299904E+03, 7.332736E+03, 7.365622E+03,
    7.398560E+03, 7.431552E+03, 7.464597E+03, 7.497694E+03, 7.530844E+03, 7.564047E+03,
    7.597302E+03, 7.630610E+03, 7.663969E+03, 7.697381E+03, 7.730845E+03, 7.764360E+03,
    7.797928E+03, 7.831547E+03, 7.865217E+03, 7.898939E+03, 7.932712E+03, 7.966535E+03,
    8.000410E+03, 8.034336E+03, 8.068313E+03, 8.102340E+03, 8.136417E+03, 8.170545E+03,
    8.204723E+03, 8.238951E+03, 8.273228E+03, 8.307556E+03, 8.341933E+03, 8.376360E+03,
    8.410836E+03, 8.445361E+03, 8.479935E+03, 8.514559E+03, 8.549231E+03, 8.583951E+03,
    8.618721E+03, 8.653538E+03, 8.688404E+03, 8.723318E+03, 8.758280E+03, 8.793289E+03,
    8.828347E+03, 8.863452E+03, 8.898604E+03, 8.933803E+03, 8.969050E+03, 9.004343E+03,
    9.039683E+03, 9.075070E+03, 9.110503E+03, 9.145983E+03, 9.181509E+03, 9.217080E+03,
    9.252698E+03, 9.288362E+03, 9.324071E+03, 9.359825E+03, 9.395625E+03, 9.431470E+03,
    9.467359E+03, 9.503294E+03, 9.539273E+03, 9.575297E+03, 9.611365E+03, 9.647478E+03,
    9.683634E+03, 9.719834E+03, 9.756078E+03, 9.792366E+03,
])

#  ---------------------- M = 7, I = 2 ---------------------------
M = 7
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[5]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    3.562445E+00, 3.092123E+01, 6.151168E+01, 9.218904E+01, 1.228922E+02, 1.536083E+02,
    1.843329E+02, 2.150641E+02, 2.458012E+02, 2.765444E+02, 3.072954E+02, 3.380581E+02,
    3.688395E+02, 3.996500E+02, 4.305039E+02, 4.614188E+02, 4.924152E+02, 5.235163E+02,
    5.547468E+02, 5.861323E+02, 6.176990E+02, 6.494730E+02, 6.814797E+02, 7.137437E+02,
    7.462888E+02, 7.791373E+02, 8.123102E+02, 8.458274E+02, 8.797073E+02, 9.139671E+02,
    9.486225E+02, 9.836883E+02, 1.019178E+03, 1.055104E+03, 1.091478E+03, 1.128311E+03,
    1.165612E+03, 1.203390E+03, 1.241654E+03, 1.280411E+03, 1.319669E+03, 1.359433E+03,
    1.399711E+03, 1.440506E+03, 1.481826E+03, 1.523674E+03, 1.566055E+03, 1.608973E+03,
    1.652432E+03, 1.696437E+03, 1.740989E+03, 1.786093E+03, 1.831752E+03, 1.877969E+03,
    1.924746E+03, 1.972085E+03, 2.019991E+03, 2.068465E+03, 2.117509E+03, 2.167125E+03,
    2.217317E+03, 2.268085E+03, 2.319433E+03, 2.371362E+03, 2.423874E+03, 2.476971E+03,
    2.530656E+03, 2.584929E+03, 2.639793E+03, 2.695250E+03, 2.751302E+03, 2.807950E+03,
    2.865197E+03, 2.923044E+03, 2.981493E+03, 3.040546E+03, 3.100206E+03, 3.160473E+03,
    3.221350E+03, 3.282838E+03, 3.344940E+03, 3.407658E+03, 3.470993E+03, 3.534948E+03,
    3.599524E+03, 3.664723E+03, 3.730548E+03, 3.797000E+03, 3.864082E+03, 3.931795E+03,
    4.000141E+03, 4.069124E+03, 4.138744E+03, 4.209003E+03, 4.279905E+03, 4.351450E+03,
    4.423642E+03, 4.496482E+03, 4.569972E+03, 4.644115E+03, 4.718912E+03, 4.794367E+03,
    4.870480E+03, 4.947254E+03, 5.024692E+03, 5.102796E+03, 5.181567E+03, 5.261008E+03,
    5.341122E+03, 5.421910E+03, 5.503374E+03, 5.585517E+03, 5.668341E+03, 5.751848E+03,
    5.836040E+03, 5.920920E+03, 6.006490E+03, 6.092751E+03, 6.179707E+03, 6.267359E+03,
    6.355709E+03, 6.444760E+03, 6.534513E+03, 6.624971E+03, 6.716136E+03, 6.808010E+03,
    6.900596E+03, 6.993894E+03, 7.087908E+03, 7.182639E+03, 7.278090E+03, 7.374263E+03,
    7.471159E+03, 7.568780E+03, 7.667129E+03, 7.766208E+03, 7.866019E+03, 7.966563E+03,
    8.067842E+03, 8.169859E+03, 8.272615E+03, 8.376113E+03, 8.480353E+03, 8.585339E+03,
    8.691071E+03, 8.797553E+03, 8.904784E+03, 9.012769E+03, 9.121507E+03, 9.231001E+03,
    9.341252E+03, 9.452263E+03, 9.564035E+03, 9.676570E+03, 9.789869E+03, 9.903935E+03,
    1.001877E+04, 1.013437E+04, 1.025074E+04, 1.036789E+04, 1.048581E+04, 1.060450E+04,
    1.072397E+04, 1.084422E+04, 1.096526E+04, 1.108707E+04, 1.120966E+04, 1.133304E+04,
    1.145721E+04, 1.158216E+04, 1.170790E+04, 1.183443E+04, 1.196175E+04, 1.208986E+04,
    1.221877E+04, 1.234847E+04, 1.247897E+04, 1.261026E+04, 1.274236E+04, 1.287525E+04,
    1.300894E+04, 1.314344E+04, 1.327874E+04, 1.341484E+04, 1.355175E+04, 1.368946E+04,
    1.382798E+04, 1.396731E+04, 1.410745E+04, 1.424840E+04, 1.439016E+04, 1.453273E+04,
    1.467611E+04, 1.482031E+04, 1.496532E+04, 1.511115E+04, 1.525780E+04, 1.540526E+04,
    1.555354E+04, 1.570264E+04, 1.585255E+04, 1.600329E+04, 1.615485E+04, 1.630723E+04,
    1.646043E+04, 1.661446E+04, 1.676931E+04, 1.692498E+04, 1.708148E+04, 1.723880E+04,
    1.739695E+04, 1.755593E+04, 1.771573E+04, 1.787636E+04, 1.803782E+04, 1.820010E+04,
    1.836322E+04, 1.852716E+04, 1.869193E+04, 1.885754E+04, 1.902397E+04, 1.919124E+04,
    1.935933E+04, 1.952826E+04, 1.969802E+04, 1.986861E+04, 2.004004E+04, 2.021229E+04,
    2.038538E+04, 2.055931E+04, 2.073406E+04, 2.090965E+04, 2.108607E+04, 2.126333E+04,
    2.144142E+04, 2.162034E+04, 2.180010E+04, 2.198070E+04, 2.216212E+04, 2.234438E+04,
    2.252748E+04, 2.271141E+04, 2.289618E+04, 2.308177E+04, 2.326821E+04, 2.345548E+04,
    2.364358E+04, 2.383251E+04, 2.402229E+04, 2.421289E+04, 2.440433E+04, 2.459660E+04,
    2.478971E+04, 2.498365E+04, 2.517842E+04, 2.537403E+04, 2.557046E+04, 2.576774E+04,
    2.596584E+04, 2.616477E+04, 2.636454E+04, 2.656514E+04, 2.676657E+04, 2.696883E+04,
    2.717193E+04, 2.737585E+04, 2.758060E+04, 2.778618E+04, 2.799260E+04, 2.819984E+04,
    2.840791E+04, 2.861680E+04, 2.882653E+04, 2.903708E+04, 2.924846E+04, 2.946066E+04,
    2.967369E+04, 2.988754E+04, 3.010222E+04, 3.031773E+04, 3.053405E+04, 3.075120E+04,
    3.096917E+04, 3.118797E+04, 3.140758E+04, 3.162802E+04, 3.184927E+04, 3.207135E+04,
    3.229424E+04, 3.251795E+04, 3.274248E+04, 3.296783E+04, 3.319399E+04, 3.342096E+04,
    3.364875E+04, 3.387736E+04, 3.410677E+04, 3.433700E+04, 3.456804E+04, 3.479989E+04,
    3.503255E+04, 3.526602E+04, 3.550030E+04, 3.573539E+04, 3.597128E+04, 3.620797E+04,
    3.644547E+04, 3.668378E+04, 3.692289E+04, 3.716280E+04, 3.740351E+04, 3.764502E+04,
    3.788733E+04, 3.813044E+04, 3.837434E+04, 3.861904E+04, 3.886454E+04, 3.911083E+04,
    3.935791E+04, 3.960579E+04, 3.985445E+04, 4.010391E+04, 4.035416E+04, 4.060519E+04,
    4.085701E+04, 4.110962E+04, 4.136301E+04, 4.161719E+04, 4.187214E+04, 4.212788E+04,
    4.238440E+04, 4.264170E+04, 4.289977E+04, 4.315863E+04, 4.341825E+04, 4.367866E+04,
    4.393983E+04, 4.420178E+04, 4.446450E+04, 4.472798E+04, 4.499224E+04, 4.525726E+04,
    4.552305E+04, 4.578960E+04, 4.605692E+04, 4.632499E+04, 4.659383E+04, 4.686343E+04,
    4.713378E+04, 4.740489E+04, 4.767676E+04, 4.794938E+04, 4.822276E+04, 4.849688E+04,
    4.877176E+04, 4.904738E+04, 4.932375E+04, 4.960087E+04, 4.987873E+04, 5.015733E+04,
    5.043668E+04, 5.071676E+04, 5.099758E+04, 5.127915E+04, 5.156144E+04, 5.184447E+04,
    5.212824E+04, 5.241273E+04, 5.269796E+04, 5.298391E+04, 5.327059E+04, 5.355800E+04,
    5.384613E+04, 5.413498E+04, 5.442456E+04, 5.471485E+04,
])

#  ---------------------- M = 7, I = 3 ---------------------------
M = 7
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[5]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.092314E+01, 1.808102E+02, 3.594239E+02, 5.385443E+02, 7.178153E+02, 8.971617E+02,
    1.076558E+03, 1.255992E+03, 1.435461E+03, 1.614965E+03, 1.794512E+03, 1.974123E+03,
    2.153834E+03, 2.333703E+03, 2.513805E+03, 2.694239E+03, 2.875119E+03, 3.056572E+03,
    3.238738E+03, 3.421762E+03, 3.605792E+03, 3.790976E+03, 3.977461E+03, 4.165389E+03,
    4.354895E+03, 4.546108E+03, 4.739152E+03, 4.934142E+03, 5.131184E+03, 5.330380E+03,
    5.531821E+03, 5.735595E+03, 5.941780E+03, 6.150451E+03, 6.361674E+03, 6.575513E+03,
    6.792026E+03, 7.011265E+03, 7.233281E+03, 7.458117E+03, 7.685816E+03, 7.916417E+03,
    8.149955E+03, 8.386465E+03, 8.625976E+03, 8.868517E+03, 9.114116E+03, 9.362797E+03,
    9.614584E+03, 9.869498E+03, 1.012756E+04, 1.038879E+04, 1.065321E+04, 1.092083E+04,
    1.119167E+04, 1.146574E+04, 1.174307E+04, 1.202366E+04, 1.230753E+04, 1.259470E+04,
    1.288517E+04, 1.317895E+04, 1.347607E+04, 1.377654E+04, 1.408035E+04, 1.438754E+04,
    1.469810E+04, 1.501205E+04, 1.532940E+04, 1.565016E+04, 1.597435E+04, 1.630197E+04,
    1.663303E+04, 1.696755E+04, 1.730554E+04, 1.764700E+04, 1.799195E+04, 1.834041E+04,
    1.869237E+04, 1.904786E+04, 1.940687E+04, 1.976944E+04, 2.013556E+04, 2.050525E+04,
    2.087851E+04, 2.125537E+04, 2.163583E+04, 2.201990E+04, 2.240760E+04, 2.279894E+04,
    2.319392E+04, 2.359257E+04, 2.399490E+04, 2.440090E+04, 2.481061E+04, 2.522403E+04,
    2.564117E+04, 2.606205E+04, 2.648668E+04, 2.691506E+04, 2.734723E+04, 2.778317E+04,
    2.822292E+04, 2.866648E+04, 2.911386E+04, 2.956508E+04, 3.002015E+04, 3.047908E+04,
    3.094189E+04, 3.140859E+04, 3.187919E+04, 3.235370E+04, 3.283214E+04, 3.331452E+04,
    3.380086E+04, 3.429116E+04, 3.478544E+04, 3.528371E+04, 3.578598E+04, 3.629228E+04,
    3.680260E+04, 3.731697E+04, 3.783539E+04, 3.835788E+04, 3.888445E+04, 3.941511E+04,
    3.994988E+04, 4.048877E+04, 4.103179E+04, 4.157894E+04, 4.213026E+04, 4.268574E+04,
    4.324540E+04, 4.380925E+04, 4.437731E+04, 4.494957E+04, 4.552607E+04, 4.610680E+04,
    4.669179E+04, 4.728103E+04, 4.787455E+04, 4.847235E+04, 4.907444E+04, 4.968084E+04,
    5.029156E+04, 5.090661E+04, 5.152600E+04, 5.214974E+04, 5.277783E+04, 5.341030E+04,
    5.404715E+04, 5.468839E+04, 5.533403E+04, 5.598409E+04, 5.663857E+04, 5.729748E+04,
    5.796083E+04, 5.862863E+04, 5.930089E+04, 5.997762E+04, 6.065884E+04, 6.134454E+04,
    6.203474E+04, 6.272945E+04, 6.342867E+04, 6.413242E+04, 6.484070E+04, 6.555352E+04,
    6.627089E+04, 6.699282E+04, 6.771932E+04, 6.845039E+04, 6.918604E+04, 6.992629E+04,
    7.067113E+04, 7.142058E+04, 7.217464E+04, 7.293332E+04, 7.369663E+04, 7.446457E+04,
    7.523716E+04, 7.601439E+04, 7.679628E+04, 7.758283E+04, 7.837405E+04, 7.916995E+04,
    7.997052E+04, 8.077579E+04, 8.158575E+04, 8.240040E+04, 8.321976E+04, 8.404384E+04,
    8.487263E+04, 8.570614E+04, 8.654437E+04, 8.738735E+04, 8.823506E+04, 8.908751E+04,
    8.994471E+04, 9.080666E+04, 9.167337E+04, 9.254484E+04, 9.342108E+04, 9.430209E+04,
    9.518787E+04, 9.607843E+04, 9.697378E+04, 9.787391E+04, 9.877883E+04, 9.968855E+04,
    1.006031E+05, 1.015224E+05, 1.024465E+05, 1.033754E+05, 1.043092E+05, 1.052477E+05,
    1.061911E+05, 1.071393E+05, 1.080923E+05, 1.090502E+05, 1.100128E+05, 1.109803E+05,
    1.119527E+05, 1.129299E+05, 1.139119E+05, 1.148987E+05, 1.158904E+05, 1.168869E+05,
    1.178883E+05, 1.188945E+05, 1.199056E+05, 1.209215E+05, 1.219423E+05, 1.229679E+05,
    1.239984E+05, 1.250338E+05, 1.260739E+05, 1.271190E+05, 1.281689E+05, 1.292237E+05,
    1.302833E+05, 1.313477E+05, 1.324171E+05, 1.334913E+05, 1.345703E+05, 1.356543E+05,
    1.367430E+05, 1.378367E+05, 1.389352E+05, 1.400385E+05, 1.411467E+05, 1.422598E+05,
    1.433777E+05, 1.445005E+05, 1.456282E+05, 1.467607E+05, 1.478980E+05, 1.490402E+05,
    1.501873E+05, 1.513392E+05, 1.524960E+05, 1.536576E+05, 1.548240E+05, 1.559954E+05,
    1.571715E+05, 1.583525E+05, 1.595383E+05, 1.607290E+05, 1.619245E+05, 1.631249E+05,
    1.643301E+05, 1.655401E+05, 1.667549E+05, 1.679746E+05, 1.691991E+05, 1.704284E+05,
    1.716626E+05, 1.729015E+05, 1.741453E+05, 1.753939E+05, 1.766473E+05, 1.779055E+05,
    1.791685E+05, 1.804363E+05, 1.817088E+05, 1.829862E+05, 1.842684E+05, 1.855554E+05,
    1.868471E+05, 1.881436E+05, 1.894449E+05, 1.907510E+05, 1.920619E+05, 1.933775E+05,
    1.946978E+05, 1.960229E+05, 1.973528E+05, 1.986874E+05, 2.000268E+05, 2.013709E+05,
    2.027197E+05, 2.040733E+05, 2.054316E+05, 2.067946E+05, 2.081624E+05, 2.095348E+05,
    2.109120E+05, 2.122938E+05, 2.136804E+05, 2.150716E+05, 2.164676E+05, 2.178682E+05,
    2.192735E+05, 2.206834E+05, 2.220981E+05, 2.235174E+05, 2.249413E+05, 2.263699E+05,
    2.278031E+05, 2.292410E+05, 2.306835E+05, 2.321307E+05, 2.335825E+05, 2.350388E+05,
    2.364998E+05, 2.379654E+05, 2.394356E+05, 2.409104E+05, 2.423898E+05, 2.438737E+05,
    2.453622E+05, 2.468553E+05, 2.483530E+05, 2.498552E+05, 2.513619E+05, 2.528732E+05,
    2.543890E+05, 2.559094E+05, 2.574342E+05, 2.589636E+05, 2.604975E+05, 2.620359E+05,
    2.635788E+05, 2.651261E+05, 2.666780E+05, 2.682343E+05, 2.697950E+05, 2.713603E+05,
    2.729299E+05, 2.745041E+05, 2.760826E+05, 2.776656E+05, 2.792530E+05, 2.808448E+05,
    2.824410E+05, 2.840416E+05, 2.856466E+05, 2.872560E+05, 2.888698E+05, 2.904879E+05,
    2.921104E+05, 2.937372E+05, 2.953684E+05, 2.970038E+05, 2.986437E+05, 3.002878E+05,
    3.019362E+05, 3.035890E+05, 3.052460E+05, 3.069073E+05, 3.085729E+05, 3.102428E+05,
    3.119169E+05, 3.135953E+05, 3.152779E+05, 3.169647E+05,
])

#  ---------------------- M = 7, I = 3 ---------------------------
M = 7
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[5]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.092314E+01, 1.808102E+02, 3.594239E+02, 5.385443E+02, 7.178153E+02, 8.971617E+02,
    1.076558E+03, 1.255992E+03, 1.435461E+03, 1.614965E+03, 1.794512E+03, 1.974123E+03,
    2.153834E+03, 2.333703E+03, 2.513805E+03, 2.694239E+03, 2.875119E+03, 3.056572E+03,
    3.238738E+03, 3.421762E+03, 3.605792E+03, 3.790976E+03, 3.977461E+03, 4.165389E+03,
    4.354895E+03, 4.546108E+03, 4.739152E+03, 4.934142E+03, 5.131184E+03, 5.330380E+03,
    5.531821E+03, 5.735595E+03, 5.941780E+03, 6.150451E+03, 6.361674E+03, 6.575513E+03,
    6.792026E+03, 7.011265E+03, 7.233281E+03, 7.458117E+03, 7.685816E+03, 7.916417E+03,
    8.149955E+03, 8.386465E+03, 8.625976E+03, 8.868517E+03, 9.114116E+03, 9.362797E+03,
    9.614584E+03, 9.869498E+03, 1.012756E+04, 1.038879E+04, 1.065321E+04, 1.092083E+04,
    1.119167E+04, 1.146574E+04, 1.174307E+04, 1.202366E+04, 1.230753E+04, 1.259470E+04,
    1.288517E+04, 1.317895E+04, 1.347607E+04, 1.377654E+04, 1.408035E+04, 1.438754E+04,
    1.469810E+04, 1.501205E+04, 1.532940E+04, 1.565016E+04, 1.597435E+04, 1.630197E+04,
    1.663303E+04, 1.696755E+04, 1.730554E+04, 1.764700E+04, 1.799195E+04, 1.834041E+04,
    1.869237E+04, 1.904786E+04, 1.940687E+04, 1.976944E+04, 2.013556E+04, 2.050525E+04,
    2.087851E+04, 2.125537E+04, 2.163583E+04, 2.201990E+04, 2.240760E+04, 2.279894E+04,
    2.319392E+04, 2.359257E+04, 2.399490E+04, 2.440090E+04, 2.481061E+04, 2.522403E+04,
    2.564117E+04, 2.606205E+04, 2.648668E+04, 2.691506E+04, 2.734723E+04, 2.778317E+04,
    2.822292E+04, 2.866648E+04, 2.911386E+04, 2.956508E+04, 3.002015E+04, 3.047908E+04,
    3.094189E+04, 3.140859E+04, 3.187919E+04, 3.235370E+04, 3.283214E+04, 3.331452E+04,
    3.380086E+04, 3.429116E+04, 3.478544E+04, 3.528371E+04, 3.578598E+04, 3.629228E+04,
    3.680260E+04, 3.731697E+04, 3.783539E+04, 3.835788E+04, 3.888445E+04, 3.941511E+04,
    3.994988E+04, 4.048877E+04, 4.103179E+04, 4.157894E+04, 4.213026E+04, 4.268574E+04,
    4.324540E+04, 4.380925E+04, 4.437731E+04, 4.494957E+04, 4.552607E+04, 4.610680E+04,
    4.669179E+04, 4.728103E+04, 4.787455E+04, 4.847235E+04, 4.907444E+04, 4.968084E+04,
    5.029156E+04, 5.090661E+04, 5.152600E+04, 5.214974E+04, 5.277783E+04, 5.341030E+04,
    5.404715E+04, 5.468839E+04, 5.533403E+04, 5.598409E+04, 5.663857E+04, 5.729748E+04,
    5.796083E+04, 5.862863E+04, 5.930089E+04, 5.997762E+04, 6.065884E+04, 6.134454E+04,
    6.203474E+04, 6.272945E+04, 6.342867E+04, 6.413242E+04, 6.484070E+04, 6.555352E+04,
    6.627089E+04, 6.699282E+04, 6.771932E+04, 6.845039E+04, 6.918604E+04, 6.992629E+04,
    7.067113E+04, 7.142058E+04, 7.217464E+04, 7.293332E+04, 7.369663E+04, 7.446457E+04,
    7.523716E+04, 7.601439E+04, 7.679628E+04, 7.758283E+04, 7.837405E+04, 7.916995E+04,
    7.997052E+04, 8.077579E+04, 8.158575E+04, 8.240040E+04, 8.321976E+04, 8.404384E+04,
    8.487263E+04, 8.570614E+04, 8.654437E+04, 8.738735E+04, 8.823506E+04, 8.908751E+04,
    8.994471E+04, 9.080666E+04, 9.167337E+04, 9.254484E+04, 9.342108E+04, 9.430209E+04,
    9.518787E+04, 9.607843E+04, 9.697378E+04, 9.787391E+04, 9.877883E+04, 9.968855E+04,
    1.006031E+05, 1.015224E+05, 1.024465E+05, 1.033754E+05, 1.043092E+05, 1.052477E+05,
    1.061911E+05, 1.071393E+05, 1.080923E+05, 1.090502E+05, 1.100128E+05, 1.109803E+05,
    1.119527E+05, 1.129299E+05, 1.139119E+05, 1.148987E+05, 1.158904E+05, 1.168869E+05,
    1.178883E+05, 1.188945E+05, 1.199056E+05, 1.209215E+05, 1.219423E+05, 1.229679E+05,
    1.239984E+05, 1.250338E+05, 1.260739E+05, 1.271190E+05, 1.281689E+05, 1.292237E+05,
    1.302833E+05, 1.313477E+05, 1.324171E+05, 1.334913E+05, 1.345703E+05, 1.356543E+05,
    1.367430E+05, 1.378367E+05, 1.389352E+05, 1.400385E+05, 1.411467E+05, 1.422598E+05,
    1.433777E+05, 1.445005E+05, 1.456282E+05, 1.467607E+05, 1.478980E+05, 1.490402E+05,
    1.501873E+05, 1.513392E+05, 1.524960E+05, 1.536576E+05, 1.548240E+05, 1.559954E+05,
    1.571715E+05, 1.583525E+05, 1.595383E+05, 1.607290E+05, 1.619245E+05, 1.631249E+05,
    1.643301E+05, 1.655401E+05, 1.667549E+05, 1.679746E+05, 1.691991E+05, 1.704284E+05,
    1.716626E+05, 1.729015E+05, 1.741453E+05, 1.753939E+05, 1.766473E+05, 1.779055E+05,
    1.791685E+05, 1.804363E+05, 1.817088E+05, 1.829862E+05, 1.842684E+05, 1.855554E+05,
    1.868471E+05, 1.881436E+05, 1.894449E+05, 1.907510E+05, 1.920619E+05, 1.933775E+05,
    1.946978E+05, 1.960229E+05, 1.973528E+05, 1.986874E+05, 2.000268E+05, 2.013709E+05,
    2.027197E+05, 2.040733E+05, 2.054316E+05, 2.067946E+05, 2.081624E+05, 2.095348E+05,
    2.109120E+05, 2.122938E+05, 2.136804E+05, 2.150716E+05, 2.164676E+05, 2.178682E+05,
    2.192735E+05, 2.206834E+05, 2.220981E+05, 2.235174E+05, 2.249413E+05, 2.263699E+05,
    2.278031E+05, 2.292410E+05, 2.306835E+05, 2.321307E+05, 2.335825E+05, 2.350388E+05,
    2.364998E+05, 2.379654E+05, 2.394356E+05, 2.409104E+05, 2.423898E+05, 2.438737E+05,
    2.453622E+05, 2.468553E+05, 2.483530E+05, 2.498552E+05, 2.513619E+05, 2.528732E+05,
    2.543890E+05, 2.559094E+05, 2.574342E+05, 2.589636E+05, 2.604975E+05, 2.620359E+05,
    2.635788E+05, 2.651261E+05, 2.666780E+05, 2.682343E+05, 2.697950E+05, 2.713603E+05,
    2.729299E+05, 2.745041E+05, 2.760826E+05, 2.776656E+05, 2.792530E+05, 2.808448E+05,
    2.824410E+05, 2.840416E+05, 2.856466E+05, 2.872560E+05, 2.888698E+05, 2.904879E+05,
    2.921104E+05, 2.937372E+05, 2.953684E+05, 2.970038E+05, 2.986437E+05, 3.002878E+05,
    3.019362E+05, 3.035890E+05, 3.052460E+05, 3.069073E+05, 3.085729E+05, 3.102428E+05,
    3.119169E+05, 3.135953E+05, 3.152779E+05, 3.169647E+05,
])

#  ---------------------- M = 7, I = 4 ---------------------------
M = 7
I = 4
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[5]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.270969E+00, 1.687773E+01, 3.312258E+01, 4.941808E+01, 6.572814E+01, 8.204535E+01,
    9.836721E+01, 1.146927E+02, 1.310214E+02, 1.473535E+02, 1.636904E+02, 1.800346E+02,
    1.963907E+02, 2.127654E+02, 2.291674E+02, 2.456073E+02, 2.620976E+02, 2.786516E+02,
    2.952836E+02, 3.120083E+02, 3.288405E+02, 3.457946E+02, 3.628847E+02, 3.801244E+02,
    3.975266E+02, 4.151032E+02, 4.328657E+02, 4.508247E+02, 4.689898E+02, 4.873702E+02,
    5.059742E+02, 5.248095E+02, 5.438832E+02, 5.632018E+02, 5.827712E+02, 6.025970E+02,
    6.226842E+02, 6.430374E+02, 6.636609E+02, 6.845586E+02, 7.057342E+02, 7.271909E+02,
    7.489318E+02, 7.709599E+02, 7.932777E+02, 8.158878E+02, 8.387923E+02, 8.619935E+02,
    8.854934E+02, 9.092938E+02, 9.333964E+02, 9.578031E+02, 9.825153E+02, 1.007535E+03,
    1.032862E+03, 1.058500E+03, 1.084448E+03, 1.110709E+03, 1.137284E+03, 1.164173E+03,
    1.191378E+03, 1.218900E+03, 1.246741E+03, 1.274900E+03, 1.303379E+03, 1.332180E+03,
    1.361303E+03, 1.390749E+03, 1.420519E+03, 1.450614E+03, 1.481036E+03, 1.511784E+03,
    1.542861E+03, 1.574267E+03, 1.606003E+03, 1.638069E+03, 1.670468E+03, 1.703200E+03,
    1.736266E+03, 1.769667E+03, 1.803403E+03, 1.837477E+03, 1.871889E+03, 1.906639E+03,
    1.941730E+03, 1.977162E+03, 2.012936E+03, 2.049053E+03, 2.085515E+03, 2.122321E+03,
    2.159474E+03, 2.196974E+03, 2.234823E+03, 2.273021E+03, 2.311570E+03, 2.350471E+03,
    2.389724E+03, 2.429331E+03, 2.469293E+03, 2.509612E+03, 2.550287E+03, 2.591320E+03,
    2.632713E+03, 2.674467E+03, 2.716582E+03, 2.759059E+03, 2.801900E+03, 2.845107E+03,
    2.888679E+03, 2.932618E+03, 2.976925E+03, 3.021602E+03, 3.066649E+03, 3.112067E+03,
    3.157858E+03, 3.204022E+03, 3.250562E+03, 3.297476E+03, 3.344768E+03, 3.392438E+03,
    3.440486E+03, 3.488915E+03, 3.537724E+03, 3.586916E+03, 3.636491E+03, 3.686450E+03,
    3.736794E+03, 3.787524E+03, 3.838641E+03, 3.890147E+03, 3.942042E+03, 3.994327E+03,
    4.047003E+03, 4.100071E+03, 4.153533E+03, 4.207388E+03, 4.261638E+03, 4.316284E+03,
    4.371327E+03, 4.426768E+03, 4.482608E+03, 4.538846E+03, 4.595486E+03, 4.652526E+03,
    4.709969E+03, 4.767815E+03, 4.826064E+03, 4.884718E+03, 4.943778E+03, 5.003244E+03,
    5.063117E+03, 5.123398E+03, 5.184088E+03, 5.245187E+03, 5.306697E+03, 5.368617E+03,
    5.430949E+03, 5.493693E+03, 5.556851E+03, 5.620423E+03, 5.684409E+03, 5.748810E+03,
    5.813627E+03, 5.878861E+03, 5.944512E+03, 6.010580E+03, 6.077068E+03, 6.143974E+03,
    6.211300E+03, 6.279047E+03, 6.347214E+03, 6.415803E+03, 6.484814E+03, 6.554247E+03,
    6.624104E+03, 6.694384E+03, 6.765089E+03, 6.836218E+03, 6.907772E+03, 6.979752E+03,
    7.052159E+03, 7.124992E+03, 7.198252E+03, 7.271939E+03, 7.346055E+03, 7.420599E+03,
    7.495571E+03, 7.570973E+03, 7.646805E+03, 7.723066E+03, 7.799758E+03, 7.876881E+03,
    7.954435E+03, 8.032420E+03, 8.110837E+03, 8.189686E+03, 8.268968E+03, 8.348682E+03,
    8.428830E+03, 8.509410E+03, 8.590424E+03, 8.671872E+03, 8.753754E+03, 8.836071E+03,
    8.918822E+03, 9.002007E+03, 9.085628E+03, 9.169684E+03, 9.254175E+03, 9.339102E+03,
    9.424465E+03, 9.510263E+03, 9.596498E+03, 9.683169E+03, 9.770276E+03, 9.857820E+03,
    9.945801E+03, 1.003422E+04, 1.012307E+04, 1.021236E+04, 1.030209E+04, 1.039226E+04,
    1.048286E+04, 1.057390E+04, 1.066537E+04, 1.075729E+04, 1.084964E+04, 1.094243E+04,
    1.103566E+04, 1.112932E+04, 1.122342E+04, 1.131796E+04, 1.141293E+04, 1.150835E+04,
    1.160420E+04, 1.170048E+04, 1.179721E+04, 1.189437E+04, 1.199197E+04, 1.209001E+04,
    1.218848E+04, 1.228739E+04, 1.238673E+04, 1.248652E+04, 1.258674E+04, 1.268739E+04,
    1.278848E+04, 1.289001E+04, 1.299198E+04, 1.309438E+04, 1.319721E+04, 1.330048E+04,
    1.340419E+04, 1.350833E+04, 1.361290E+04, 1.371791E+04, 1.382336E+04, 1.392924E+04,
    1.403555E+04, 1.414230E+04, 1.424948E+04, 1.435709E+04, 1.446514E+04, 1.457362E+04,
    1.468253E+04, 1.479187E+04, 1.490165E+04, 1.501185E+04, 1.512249E+04, 1.523356E+04,
    1.534506E+04, 1.545699E+04, 1.556935E+04, 1.568214E+04, 1.579536E+04, 1.590901E+04,
    1.602309E+04, 1.613759E+04, 1.625253E+04, 1.636789E+04, 1.648367E+04, 1.659989E+04,
    1.671653E+04, 1.683360E+04, 1.695109E+04, 1.706900E+04, 1.718735E+04, 1.730611E+04,
    1.742530E+04, 1.754492E+04, 1.766495E+04, 1.778541E+04, 1.790629E+04, 1.802759E+04,
    1.814932E+04, 1.827146E+04, 1.839402E+04, 1.851701E+04, 1.864041E+04, 1.876423E+04,
    1.888847E+04, 1.901313E+04, 1.913820E+04, 1.926369E+04, 1.938959E+04, 1.951592E+04,
    1.964265E+04, 1.976980E+04, 1.989737E+04, 2.002534E+04, 2.015373E+04, 2.028253E+04,
    2.041175E+04, 2.054137E+04, 2.067141E+04, 2.080185E+04, 2.093270E+04, 2.106396E+04,
    2.119563E+04, 2.132771E+04, 2.146019E+04, 2.159308E+04, 2.172638E+04, 2.186007E+04,
    2.199418E+04, 2.212868E+04, 2.226359E+04, 2.239890E+04, 2.253461E+04, 2.267073E+04,
    2.280724E+04, 2.294415E+04, 2.308146E+04, 2.321917E+04, 2.335728E+04, 2.349578E+04,
    2.363468E+04, 2.377397E+04, 2.391366E+04, 2.405374E+04, 2.419421E+04, 2.433508E+04,
    2.447634E+04, 2.461799E+04, 2.476002E+04, 2.490245E+04, 2.504527E+04, 2.518847E+04,
    2.533206E+04, 2.547604E+04, 2.562040E+04, 2.576515E+04, 2.591028E+04, 2.605580E+04,
    2.620169E+04, 2.634797E+04, 2.649463E+04, 2.664167E+04, 2.678908E+04, 2.693688E+04,
    2.708505E+04, 2.723360E+04, 2.738253E+04, 2.753183E+04, 2.768151E+04, 2.783155E+04,
    2.798198E+04, 2.813277E+04, 2.828393E+04, 2.843547E+04, 2.858737E+04, 2.873964E+04,
    2.889228E+04, 2.904528E+04, 2.919865E+04, 2.935239E+04,
])

#  ---------------------- M = 7, I = 5 ---------------------------
M = 7
I = 5
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[5]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.157469E+01, 1.910172E+02, 3.804688E+02, 5.704593E+02, 7.606097E+02, 9.508401E+02,
    1.141123E+03, 1.331447E+03, 1.521808E+03, 1.712208E+03, 1.902659E+03, 2.093190E+03,
    2.283847E+03, 2.474702E+03, 2.665851E+03, 2.857410E+03, 3.049515E+03, 3.242316E+03,
    3.435973E+03, 3.630652E+03, 3.826519E+03, 4.023741E+03, 4.222477E+03, 4.422884E+03,
    4.625110E+03, 4.829294E+03, 5.035569E+03, 5.244058E+03, 5.454873E+03, 5.668122E+03,
    5.883902E+03, 6.102304E+03, 6.323410E+03, 6.547298E+03, 6.774037E+03, 7.003694E+03,
    7.236326E+03, 7.471990E+03, 7.710735E+03, 7.952609E+03, 8.197654E+03, 8.445909E+03,
    8.697412E+03, 8.952197E+03, 9.210295E+03, 9.471735E+03, 9.736545E+03, 1.000475E+04,
    1.027637E+04, 1.055144E+04, 1.082997E+04, 1.111198E+04, 1.139749E+04, 1.168652E+04,
    1.197908E+04, 1.227520E+04, 1.257488E+04, 1.287815E+04, 1.318501E+04, 1.349548E+04,
    1.380958E+04, 1.412731E+04, 1.444869E+04, 1.477374E+04, 1.510246E+04, 1.543486E+04,
    1.577096E+04, 1.611078E+04, 1.645431E+04, 1.680158E+04, 1.715259E+04, 1.750736E+04,
    1.786589E+04, 1.822821E+04, 1.859431E+04, 1.896422E+04, 1.933794E+04, 1.971548E+04,
    2.009687E+04, 2.048210E+04, 2.087119E+04, 2.126415E+04, 2.166100E+04, 2.206174E+04,
    2.246639E+04, 2.287496E+04, 2.328746E+04, 2.370391E+04, 2.412431E+04, 2.454868E+04,
    2.497703E+04, 2.540938E+04, 2.584574E+04, 2.628611E+04, 2.673051E+04, 2.717896E+04,
    2.763147E+04, 2.808804E+04, 2.854870E+04, 2.901346E+04, 2.948232E+04, 2.995531E+04,
    3.043243E+04, 3.091370E+04, 3.139913E+04, 3.188873E+04, 3.238252E+04, 3.288051E+04,
    3.338272E+04, 3.388915E+04, 3.439982E+04, 3.491474E+04, 3.543392E+04, 3.595739E+04,
    3.648514E+04, 3.701720E+04, 3.755358E+04, 3.809428E+04, 3.863933E+04, 3.918873E+04,
    3.974250E+04, 4.030065E+04, 4.086319E+04, 4.143014E+04, 4.200151E+04, 4.257730E+04,
    4.315754E+04, 4.374223E+04, 4.433139E+04, 4.492502E+04, 4.552315E+04, 4.612578E+04,
    4.673292E+04, 4.734459E+04, 4.796079E+04, 4.858154E+04, 4.920686E+04, 4.983674E+04,
    5.047121E+04, 5.111027E+04, 5.175393E+04, 5.240221E+04, 5.305511E+04, 5.371265E+04,
    5.437484E+04, 5.504169E+04, 5.571320E+04, 5.638939E+04, 5.707027E+04, 5.775584E+04,
    5.844613E+04, 5.914113E+04, 5.984085E+04, 6.054532E+04, 6.125452E+04, 6.196849E+04,
    6.268721E+04, 6.341071E+04, 6.413899E+04, 6.487206E+04, 6.560993E+04, 6.635261E+04,
    6.710010E+04, 6.785242E+04, 6.860957E+04, 6.937155E+04, 7.013839E+04, 7.091008E+04,
    7.168663E+04, 7.246805E+04, 7.325435E+04, 7.404554E+04, 7.484161E+04, 7.564259E+04,
    7.644847E+04, 7.725926E+04, 7.807497E+04, 7.889561E+04, 7.972118E+04, 8.055169E+04,
    8.138714E+04, 8.222754E+04, 8.307289E+04, 8.392321E+04, 8.477849E+04, 8.563874E+04,
    8.650398E+04, 8.737419E+04, 8.824940E+04, 8.912959E+04, 9.001479E+04, 9.090498E+04,
    9.180018E+04, 9.270040E+04, 9.360563E+04, 9.451588E+04, 9.543115E+04, 9.635146E+04,
    9.727679E+04, 9.820716E+04, 9.914257E+04, 1.000830E+05, 1.010285E+05, 1.019791E+05,
    1.029347E+05, 1.038953E+05, 1.048611E+05, 1.058318E+05, 1.068077E+05, 1.077886E+05,
    1.087746E+05, 1.097656E+05, 1.107617E+05, 1.117630E+05, 1.127692E+05, 1.137806E+05,
    1.147971E+05, 1.158186E+05, 1.168452E+05, 1.178769E+05, 1.189137E+05, 1.199556E+05,
    1.210026E+05, 1.220546E+05, 1.231118E+05, 1.241741E+05, 1.252414E+05, 1.263139E+05,
    1.273914E+05, 1.284741E+05, 1.295618E+05, 1.306547E+05, 1.317526E+05, 1.328556E+05,
    1.339638E+05, 1.350770E+05, 1.361953E+05, 1.373188E+05, 1.384473E+05, 1.395809E+05,
    1.407197E+05, 1.418635E+05, 1.430124E+05, 1.441664E+05, 1.453255E+05, 1.464897E+05,
    1.476590E+05, 1.488334E+05, 1.500129E+05, 1.511974E+05, 1.523871E+05, 1.535818E+05,
    1.547816E+05, 1.559865E+05, 1.571964E+05, 1.584115E+05, 1.596316E+05, 1.608568E+05,
    1.620871E+05, 1.633224E+05, 1.645628E+05, 1.658083E+05, 1.670588E+05, 1.683144E+05,
    1.695750E+05, 1.708407E+05, 1.721114E+05, 1.733872E+05, 1.746680E+05, 1.759539E+05,
    1.772448E+05, 1.785407E+05, 1.798417E+05, 1.811477E+05, 1.824587E+05, 1.837748E+05,
    1.850958E+05, 1.864219E+05, 1.877530E+05, 1.890890E+05, 1.904301E+05, 1.917762E+05,
    1.931273E+05, 1.944833E+05, 1.958443E+05, 1.972104E+05, 1.985814E+05, 1.999573E+05,
    2.013382E+05, 2.027241E+05, 2.041150E+05, 2.055107E+05, 2.069115E+05, 2.083171E+05,
    2.097278E+05, 2.111433E+05, 2.125638E+05, 2.139891E+05, 2.154194E+05, 2.168546E+05,
    2.182947E+05, 2.197397E+05, 2.211896E+05, 2.226444E+05, 2.241041E+05, 2.255686E+05,
    2.270380E+05, 2.285122E+05, 2.299914E+05, 2.314753E+05, 2.329641E+05, 2.344578E+05,
    2.359562E+05, 2.374595E+05, 2.389676E+05, 2.404806E+05, 2.419983E+05, 2.435208E+05,
    2.450481E+05, 2.465802E+05, 2.481171E+05, 2.496587E+05, 2.512051E+05, 2.527563E+05,
    2.543122E+05, 2.558729E+05, 2.574382E+05, 2.590083E+05, 2.605832E+05, 2.621627E+05,
    2.637469E+05, 2.653359E+05, 2.669295E+05, 2.685278E+05, 2.701308E+05, 2.717384E+05,
    2.733507E+05, 2.749677E+05, 2.765893E+05, 2.782155E+05, 2.798463E+05, 2.814818E+05,
    2.831219E+05, 2.847665E+05, 2.864158E+05, 2.880697E+05, 2.897281E+05, 2.913911E+05,
    2.930586E+05, 2.947307E+05, 2.964074E+05, 2.980885E+05, 2.997742E+05, 3.014644E+05,
    3.031591E+05, 3.048584E+05, 3.065620E+05, 3.082702E+05, 3.099829E+05, 3.117000E+05,
    3.134215E+05, 3.151475E+05, 3.168780E+05, 3.186128E+05, 3.203521E+05, 3.220958E+05,
    3.238439E+05, 3.255963E+05, 3.273532E+05, 3.291144E+05, 3.308799E+05, 3.326499E+05,
    3.344241E+05, 3.362027E+05, 3.379856E+05, 3.397728E+05,
])

#  ---------------------- M = 7, I = 5 ---------------------------
M = 7
I = 5
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[5]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.157469E+01, 1.910172E+02, 3.804688E+02, 5.704593E+02, 7.606097E+02, 9.508401E+02,
    1.141123E+03, 1.331447E+03, 1.521808E+03, 1.712208E+03, 1.902659E+03, 2.093190E+03,
    2.283847E+03, 2.474702E+03, 2.665851E+03, 2.857410E+03, 3.049515E+03, 3.242316E+03,
    3.435973E+03, 3.630652E+03, 3.826519E+03, 4.023741E+03, 4.222477E+03, 4.422884E+03,
    4.625110E+03, 4.829294E+03, 5.035569E+03, 5.244058E+03, 5.454873E+03, 5.668122E+03,
    5.883902E+03, 6.102304E+03, 6.323410E+03, 6.547298E+03, 6.774037E+03, 7.003694E+03,
    7.236326E+03, 7.471990E+03, 7.710735E+03, 7.952609E+03, 8.197654E+03, 8.445909E+03,
    8.697412E+03, 8.952197E+03, 9.210295E+03, 9.471735E+03, 9.736545E+03, 1.000475E+04,
    1.027637E+04, 1.055144E+04, 1.082997E+04, 1.111198E+04, 1.139749E+04, 1.168652E+04,
    1.197908E+04, 1.227520E+04, 1.257488E+04, 1.287815E+04, 1.318501E+04, 1.349548E+04,
    1.380958E+04, 1.412731E+04, 1.444869E+04, 1.477374E+04, 1.510246E+04, 1.543486E+04,
    1.577096E+04, 1.611078E+04, 1.645431E+04, 1.680158E+04, 1.715259E+04, 1.750736E+04,
    1.786589E+04, 1.822821E+04, 1.859431E+04, 1.896422E+04, 1.933794E+04, 1.971548E+04,
    2.009687E+04, 2.048210E+04, 2.087119E+04, 2.126415E+04, 2.166100E+04, 2.206174E+04,
    2.246639E+04, 2.287496E+04, 2.328746E+04, 2.370391E+04, 2.412431E+04, 2.454868E+04,
    2.497703E+04, 2.540938E+04, 2.584574E+04, 2.628611E+04, 2.673051E+04, 2.717896E+04,
    2.763147E+04, 2.808804E+04, 2.854870E+04, 2.901346E+04, 2.948232E+04, 2.995531E+04,
    3.043243E+04, 3.091370E+04, 3.139913E+04, 3.188873E+04, 3.238252E+04, 3.288051E+04,
    3.338272E+04, 3.388915E+04, 3.439982E+04, 3.491474E+04, 3.543392E+04, 3.595739E+04,
    3.648514E+04, 3.701720E+04, 3.755358E+04, 3.809428E+04, 3.863933E+04, 3.918873E+04,
    3.974250E+04, 4.030065E+04, 4.086319E+04, 4.143014E+04, 4.200151E+04, 4.257730E+04,
    4.315754E+04, 4.374223E+04, 4.433139E+04, 4.492502E+04, 4.552315E+04, 4.612578E+04,
    4.673292E+04, 4.734459E+04, 4.796079E+04, 4.858154E+04, 4.920686E+04, 4.983674E+04,
    5.047121E+04, 5.111027E+04, 5.175393E+04, 5.240221E+04, 5.305511E+04, 5.371265E+04,
    5.437484E+04, 5.504169E+04, 5.571320E+04, 5.638939E+04, 5.707027E+04, 5.775584E+04,
    5.844613E+04, 5.914113E+04, 5.984085E+04, 6.054532E+04, 6.125452E+04, 6.196849E+04,
    6.268721E+04, 6.341071E+04, 6.413899E+04, 6.487206E+04, 6.560993E+04, 6.635261E+04,
    6.710010E+04, 6.785242E+04, 6.860957E+04, 6.937155E+04, 7.013839E+04, 7.091008E+04,
    7.168663E+04, 7.246805E+04, 7.325435E+04, 7.404554E+04, 7.484161E+04, 7.564259E+04,
    7.644847E+04, 7.725926E+04, 7.807497E+04, 7.889561E+04, 7.972118E+04, 8.055169E+04,
    8.138714E+04, 8.222754E+04, 8.307289E+04, 8.392321E+04, 8.477849E+04, 8.563874E+04,
    8.650398E+04, 8.737419E+04, 8.824940E+04, 8.912959E+04, 9.001479E+04, 9.090498E+04,
    9.180018E+04, 9.270040E+04, 9.360563E+04, 9.451588E+04, 9.543115E+04, 9.635146E+04,
    9.727679E+04, 9.820716E+04, 9.914257E+04, 1.000830E+05, 1.010285E+05, 1.019791E+05,
    1.029347E+05, 1.038953E+05, 1.048611E+05, 1.058318E+05, 1.068077E+05, 1.077886E+05,
    1.087746E+05, 1.097656E+05, 1.107617E+05, 1.117630E+05, 1.127692E+05, 1.137806E+05,
    1.147971E+05, 1.158186E+05, 1.168452E+05, 1.178769E+05, 1.189137E+05, 1.199556E+05,
    1.210026E+05, 1.220546E+05, 1.231118E+05, 1.241741E+05, 1.252414E+05, 1.263139E+05,
    1.273914E+05, 1.284741E+05, 1.295618E+05, 1.306547E+05, 1.317526E+05, 1.328556E+05,
    1.339638E+05, 1.350770E+05, 1.361953E+05, 1.373188E+05, 1.384473E+05, 1.395809E+05,
    1.407197E+05, 1.418635E+05, 1.430124E+05, 1.441664E+05, 1.453255E+05, 1.464897E+05,
    1.476590E+05, 1.488334E+05, 1.500129E+05, 1.511974E+05, 1.523871E+05, 1.535818E+05,
    1.547816E+05, 1.559865E+05, 1.571964E+05, 1.584115E+05, 1.596316E+05, 1.608568E+05,
    1.620871E+05, 1.633224E+05, 1.645628E+05, 1.658083E+05, 1.670588E+05, 1.683144E+05,
    1.695750E+05, 1.708407E+05, 1.721114E+05, 1.733872E+05, 1.746680E+05, 1.759539E+05,
    1.772448E+05, 1.785407E+05, 1.798417E+05, 1.811477E+05, 1.824587E+05, 1.837748E+05,
    1.850958E+05, 1.864219E+05, 1.877530E+05, 1.890890E+05, 1.904301E+05, 1.917762E+05,
    1.931273E+05, 1.944833E+05, 1.958443E+05, 1.972104E+05, 1.985814E+05, 1.999573E+05,
    2.013382E+05, 2.027241E+05, 2.041150E+05, 2.055107E+05, 2.069115E+05, 2.083171E+05,
    2.097278E+05, 2.111433E+05, 2.125638E+05, 2.139891E+05, 2.154194E+05, 2.168546E+05,
    2.182947E+05, 2.197397E+05, 2.211896E+05, 2.226444E+05, 2.241041E+05, 2.255686E+05,
    2.270380E+05, 2.285122E+05, 2.299914E+05, 2.314753E+05, 2.329641E+05, 2.344578E+05,
    2.359562E+05, 2.374595E+05, 2.389676E+05, 2.404806E+05, 2.419983E+05, 2.435208E+05,
    2.450481E+05, 2.465802E+05, 2.481171E+05, 2.496587E+05, 2.512051E+05, 2.527563E+05,
    2.543122E+05, 2.558729E+05, 2.574382E+05, 2.590083E+05, 2.605832E+05, 2.621627E+05,
    2.637469E+05, 2.653359E+05, 2.669295E+05, 2.685278E+05, 2.701308E+05, 2.717384E+05,
    2.733507E+05, 2.749677E+05, 2.765893E+05, 2.782155E+05, 2.798463E+05, 2.814818E+05,
    2.831219E+05, 2.847665E+05, 2.864158E+05, 2.880697E+05, 2.897281E+05, 2.913911E+05,
    2.930586E+05, 2.947307E+05, 2.964074E+05, 2.980885E+05, 2.997742E+05, 3.014644E+05,
    3.031591E+05, 3.048584E+05, 3.065620E+05, 3.082702E+05, 3.099829E+05, 3.117000E+05,
    3.134215E+05, 3.151475E+05, 3.168780E+05, 3.186128E+05, 3.203521E+05, 3.220958E+05,
    3.238439E+05, 3.255963E+05, 3.273532E+05, 3.291144E+05, 3.308799E+05, 3.326499E+05,
    3.344241E+05, 3.362027E+05, 3.379856E+05, 3.397728E+05,
])

#  ---------------------- M = 7, I = 6 ---------------------------
M = 7
I = 6
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[5]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    7.197400E+01, 5.586187E+02, 1.111229E+03, 1.665410E+03, 2.220057E+03, 2.774938E+03,
    3.329973E+03, 3.885127E+03, 4.440388E+03, 4.995759E+03, 5.551272E+03, 6.106999E+03,
    6.663066E+03, 7.219666E+03, 7.777057E+03, 8.335560E+03, 8.895550E+03, 9.457446E+03,
    1.002170E+04, 1.058877E+04, 1.115914E+04, 1.173327E+04, 1.231163E+04, 1.289467E+04,
    1.348281E+04, 1.407645E+04, 1.467598E+04, 1.528176E+04, 1.589412E+04, 1.651337E+04,
    1.713979E+04, 1.777365E+04, 1.841520E+04, 1.906465E+04, 1.972223E+04, 2.038811E+04,
    2.106247E+04, 2.174549E+04, 2.243730E+04, 2.313805E+04, 2.384786E+04, 2.456684E+04,
    2.529512E+04, 2.603279E+04, 2.677994E+04, 2.753666E+04, 2.830303E+04, 2.907913E+04,
    2.986503E+04, 3.066080E+04, 3.146649E+04, 3.228216E+04, 3.310788E+04, 3.394370E+04,
    3.478965E+04, 3.564581E+04, 3.651220E+04, 3.738887E+04, 3.827587E+04, 3.917323E+04,
    4.008100E+04, 4.099922E+04, 4.192792E+04, 4.286713E+04, 4.381690E+04, 4.477727E+04,
    4.574825E+04, 4.672990E+04, 4.772225E+04, 4.872532E+04, 4.973916E+04, 5.076379E+04,
    5.179925E+04, 5.284558E+04, 5.390280E+04, 5.497096E+04, 5.605008E+04, 5.714021E+04,
    5.824136E+04, 5.935359E+04, 6.047692E+04, 6.161139E+04, 6.275703E+04, 6.391387E+04,
    6.508197E+04, 6.626134E+04, 6.745202E+04, 6.865406E+04, 6.986749E+04, 7.109234E+04,
    7.232865E+04, 7.357646E+04, 7.483581E+04, 7.610672E+04, 7.738925E+04, 7.868343E+04,
    7.998930E+04, 8.130689E+04, 8.263624E+04, 8.397740E+04, 8.533040E+04, 8.669528E+04,
    8.807207E+04, 8.946082E+04, 9.086157E+04, 9.227436E+04, 9.369922E+04, 9.513619E+04,
    9.658532E+04, 9.804664E+04, 9.952019E+04, 1.010060E+05, 1.025041E+05, 1.040146E+05,
    1.055375E+05, 1.070728E+05, 1.086206E+05, 1.101808E+05, 1.117536E+05, 1.133390E+05,
    1.149371E+05, 1.165477E+05, 1.181711E+05, 1.198072E+05, 1.214561E+05, 1.231178E+05,
    1.247924E+05, 1.264798E+05, 1.281802E+05, 1.298935E+05, 1.316198E+05, 1.333592E+05,
    1.351116E+05, 1.368771E+05, 1.386558E+05, 1.404477E+05, 1.422528E+05, 1.440711E+05,
    1.459027E+05, 1.477476E+05, 1.496059E+05, 1.514775E+05, 1.533626E+05, 1.552611E+05,
    1.571732E+05, 1.590987E+05, 1.610378E+05, 1.629905E+05, 1.649567E+05, 1.669367E+05,
    1.689303E+05, 1.709376E+05, 1.729586E+05, 1.749935E+05, 1.770421E+05, 1.791045E+05,
    1.811808E+05, 1.832710E+05, 1.853751E+05, 1.874931E+05, 1.896251E+05, 1.917711E+05,
    1.939311E+05, 1.961052E+05, 1.982934E+05, 2.004956E+05, 2.027120E+05, 2.049425E+05,
    2.071872E+05, 2.094461E+05, 2.117192E+05, 2.140066E+05, 2.163082E+05, 2.186241E+05,
    2.209544E+05, 2.232989E+05, 2.256579E+05, 2.280312E+05, 2.304189E+05, 2.328211E+05,
    2.352376E+05, 2.376687E+05, 2.401142E+05, 2.425742E+05, 2.450488E+05, 2.475379E+05,
    2.500415E+05, 2.525597E+05, 2.550925E+05, 2.576400E+05, 2.602020E+05, 2.627787E+05,
    2.653700E+05, 2.679761E+05, 2.705968E+05, 2.732322E+05, 2.758823E+05, 2.785471E+05,
    2.812268E+05, 2.839211E+05, 2.866302E+05, 2.893542E+05, 2.920929E+05, 2.948464E+05,
    2.976148E+05, 3.003980E+05, 3.031960E+05, 3.060089E+05, 3.088366E+05, 3.116792E+05,
    3.145368E+05, 3.174092E+05, 3.202965E+05, 3.231987E+05, 3.261158E+05, 3.290479E+05,
    3.319949E+05, 3.349569E+05, 3.379338E+05, 3.409257E+05, 3.439325E+05, 3.469543E+05,
    3.499911E+05, 3.530429E+05, 3.561096E+05, 3.591914E+05, 3.622881E+05, 3.653999E+05,
    3.685267E+05, 3.716684E+05, 3.748252E+05, 3.779970E+05, 3.811839E+05, 3.843857E+05,
    3.876026E+05, 3.908345E+05, 3.940814E+05, 3.973434E+05, 4.006204E+05, 4.039124E+05,
    4.072195E+05, 4.105416E+05, 4.138787E+05, 4.172308E+05, 4.205980E+05, 4.239803E+05,
    4.273775E+05, 4.307898E+05, 4.342171E+05, 4.376594E+05, 4.411168E+05, 4.445892E+05,
    4.480765E+05, 4.515790E+05, 4.550964E+05, 4.586288E+05, 4.621762E+05, 4.657386E+05,
    4.693160E+05, 4.729085E+05, 4.765158E+05, 4.801382E+05, 4.837756E+05, 4.874279E+05,
    4.910951E+05, 4.947774E+05, 4.984745E+05, 5.021866E+05, 5.059137E+05, 5.096557E+05,
    5.134126E+05, 5.171844E+05, 5.209711E+05, 5.247727E+05, 5.285892E+05, 5.324206E+05,
    5.362668E+05, 5.401279E+05, 5.440038E+05, 5.478946E+05, 5.518002E+05, 5.557207E+05,
    5.596559E+05, 5.636060E+05, 5.675708E+05, 5.715504E+05, 5.755448E+05, 5.795539E+05,
    5.835777E+05, 5.876163E+05, 5.916696E+05, 5.957376E+05, 5.998203E+05, 6.039177E+05,
    6.080298E+05, 6.121564E+05, 6.162978E+05, 6.204537E+05, 6.246243E+05, 6.288094E+05,
    6.330091E+05, 6.372234E+05, 6.414523E+05, 6.456957E+05, 6.499535E+05, 6.542259E+05,
    6.585128E+05, 6.628142E+05, 6.671300E+05, 6.714602E+05, 6.758049E+05, 6.801639E+05,
    6.845374E+05, 6.889252E+05, 6.933274E+05, 6.977439E+05, 7.021747E+05, 7.066198E+05,
    7.110792E+05, 7.155528E+05, 7.200407E+05, 7.245429E+05, 7.290592E+05, 7.335897E+05,
    7.381343E+05, 7.426932E+05, 7.472661E+05, 7.518531E+05, 7.564543E+05, 7.610694E+05,
    7.656987E+05, 7.703419E+05, 7.749992E+05, 7.796704E+05, 7.843556E+05, 7.890547E+05,
    7.937678E+05, 7.984947E+05, 8.032355E+05, 8.079901E+05, 8.127586E+05, 8.175409E+05,
    8.223370E+05, 8.271468E+05, 8.319703E+05, 8.368076E+05, 8.416586E+05, 8.465232E+05,
    8.514015E+05, 8.562933E+05, 8.611988E+05, 8.661178E+05, 8.710504E+05, 8.759965E+05,
    8.809561E+05, 8.859292E+05, 8.909157E+05, 8.959156E+05, 9.009290E+05, 9.059557E+05,
    9.109957E+05, 9.160491E+05, 9.211157E+05, 9.261956E+05, 9.312888E+05, 9.363952E+05,
    9.415147E+05, 9.466474E+05, 9.517933E+05, 9.569522E+05, 9.621243E+05, 9.673094E+05,
    9.725075E+05, 9.777186E+05, 9.829427E+05, 9.881797E+05,
])

#  ---------------------- M = 8, I = 1 ---------------------------
M = 8
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.187882E+01, 5.438944E+01, 1.046755E+02, 1.605493E+02, 2.237848E+02, 2.936074E+02,
    3.693790E+02, 4.493987E+02, 5.325091E+02, 6.180282E+02, 7.054511E+02, 7.943992E+02,
    8.845954E+02, 9.758218E+02, 1.067921E+03, 1.160750E+03, 1.254256E+03, 1.348352E+03,
    1.443007E+03, 1.538179E+03, 1.633904E+03, 1.730134E+03, 1.826913E+03, 1.924263E+03,
    2.022194E+03, 2.120743E+03, 2.219926E+03, 2.319814E+03, 2.420412E+03, 2.521749E+03,
    2.623906E+03, 2.726868E+03, 2.830696E+03, 2.935398E+03, 3.041038E+03, 3.147623E+03,
    3.255221E+03, 3.363806E+03, 3.473413E+03, 3.584110E+03, 3.695900E+03, 3.808781E+03,
    3.922787E+03, 4.037949E+03, 4.154301E+03, 4.271797E+03, 4.390503E+03, 4.510451E+03,
    4.631630E+03, 4.754023E+03, 4.877700E+03, 5.002642E+03, 5.128876E+03, 5.256379E+03,
    5.385224E+03, 5.515334E+03, 5.646783E+03, 5.779540E+03, 5.913678E+03, 6.049113E+03,
    6.185916E+03, 6.324108E+03, 6.463595E+03, 6.604508E+03, 6.746748E+03, 6.890451E+03,
    7.035454E+03, 7.181892E+03, 7.329656E+03, 7.478888E+03, 7.629473E+03, 7.781490E+03,
    7.934955E+03, 8.089742E+03, 8.246004E+03, 8.403681E+03, 8.562787E+03, 8.723261E+03,
    8.885184E+03, 9.048571E+03, 9.213358E+03, 9.379626E+03, 9.547237E+03, 9.716351E+03,
    9.886903E+03, 1.005890E+04, 1.023227E+04, 1.040718E+04, 1.058348E+04, 1.076127E+04,
    1.094045E+04, 1.112113E+04, 1.130323E+04, 1.148684E+04, 1.167188E+04, 1.185836E+04,
    1.204629E+04, 1.223566E+04, 1.242649E+04, 1.261887E+04, 1.281263E+04, 1.300787E+04,
    1.320458E+04, 1.340278E+04, 1.360238E+04, 1.380347E+04, 1.400606E+04, 1.421006E+04,
    1.441557E+04, 1.462260E+04, 1.483105E+04, 1.504093E+04, 1.525234E+04, 1.546529E+04,
    1.567956E+04, 1.589539E+04, 1.611265E+04, 1.633148E+04, 1.655176E+04, 1.677349E+04,
    1.699667E+04, 1.722133E+04, 1.744744E+04, 1.767514E+04, 1.790419E+04, 1.813483E+04,
    1.836696E+04, 1.860044E+04, 1.883553E+04, 1.907199E+04, 1.931006E+04, 1.954950E+04,
    1.979055E+04, 2.003299E+04, 2.027692E+04, 2.052237E+04, 2.076931E+04, 2.101764E+04,
    2.126762E+04, 2.151898E+04, 2.177186E+04, 2.202612E+04, 2.228205E+04, 2.253936E+04,
    2.279821E+04, 2.305845E+04, 2.332023E+04, 2.358354E+04, 2.384840E+04, 2.411465E+04,
    2.438245E+04, 2.465165E+04, 2.492240E+04, 2.519456E+04, 2.546842E+04, 2.574354E+04,
    2.602037E+04, 2.629846E+04, 2.657826E+04, 2.685948E+04, 2.714211E+04, 2.742631E+04,
    2.771194E+04, 2.799913E+04, 2.828791E+04, 2.857810E+04, 2.886972E+04, 2.916292E+04,
    2.945755E+04, 2.975376E+04, 3.005141E+04, 3.035064E+04, 3.065130E+04, 3.095340E+04,
    3.125708E+04, 3.156221E+04,
])

#  ---------------------- M = 8, I = 2 ---------------------------
M = 8
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    7.941320E+00, 3.745917E+01, 7.221608E+01, 1.108280E+02, 1.545250E+02, 2.027736E+02,
    2.552148E+02, 3.105134E+02, 3.679482E+02, 4.270480E+02, 4.874637E+02, 5.489392E+02,
    6.112725E+02, 6.743180E+02, 7.379665E+02, 8.021363E+02, 8.667588E+02, 9.318076E+02,
    9.972455E+02, 1.063065E+03, 1.129260E+03, 1.195844E+03, 1.262803E+03, 1.330177E+03,
    1.397975E+03, 1.466208E+03, 1.534920E+03, 1.604131E+03, 1.673861E+03, 1.744132E+03,
    1.814968E+03, 1.886392E+03, 1.958449E+03, 2.031145E+03, 2.104487E+03, 2.178540E+03,
    2.253291E+03, 2.328765E+03, 2.404963E+03, 2.481933E+03, 2.559679E+03, 2.638222E+03,
    2.717563E+03, 2.797724E+03, 2.878728E+03, 2.960569E+03, 3.043241E+03, 3.126792E+03,
    3.211244E+03, 3.296556E+03, 3.382746E+03, 3.469862E+03, 3.557861E+03, 3.646792E+03,
    3.736671E+03, 3.827446E+03, 3.919168E+03, 4.011816E+03, 4.105405E+03, 4.199946E+03,
    4.295455E+03, 4.391908E+03, 4.489355E+03, 4.587730E+03, 4.687084E+03, 4.787429E+03,
    4.888735E+03, 4.991012E+03, 5.094314E+03, 5.198564E+03, 5.303815E+03, 5.410032E+03,
    5.517270E+03, 5.625490E+03, 5.734750E+03, 5.844962E+03, 5.956183E+03, 6.068418E+03,
    6.181628E+03, 6.295921E+03, 6.411150E+03, 6.527425E+03, 6.644701E+03, 6.762984E+03,
    6.882281E+03, 7.002598E+03, 7.123940E+03, 7.246257E+03, 7.369611E+03, 7.494009E+03,
    7.619456E+03, 7.745839E+03, 7.873343E+03, 8.001789E+03, 8.131307E+03, 8.261840E+03,
    8.393389E+03, 8.525963E+03, 8.659562E+03, 8.794193E+03, 8.929860E+03, 9.066566E+03,
    9.204314E+03, 9.343044E+03, 9.482892E+03, 9.623727E+03, 9.765553E+03, 9.908441E+03,
    1.005239E+04, 1.019735E+04, 1.034337E+04, 1.049040E+04, 1.063844E+04, 1.078756E+04,
    1.093769E+04, 1.108883E+04, 1.124107E+04, 1.139433E+04, 1.154862E+04, 1.170400E+04,
    1.186034E+04, 1.201778E+04, 1.217626E+04, 1.233577E+04, 1.249631E+04, 1.265790E+04,
    1.282053E+04, 1.298420E+04, 1.314892E+04, 1.331460E+04, 1.348141E+04, 1.364928E+04,
    1.381820E+04, 1.398809E+04, 1.415904E+04, 1.433114E+04, 1.450421E+04, 1.467825E+04,
    1.485345E+04, 1.502963E+04, 1.520687E+04, 1.538519E+04, 1.556458E+04, 1.574495E+04,
    1.592640E+04, 1.610883E+04, 1.629244E+04, 1.647703E+04, 1.666261E+04, 1.684928E+04,
    1.703703E+04, 1.722587E+04, 1.741570E+04, 1.760653E+04, 1.779854E+04, 1.799145E+04,
    1.818556E+04, 1.838056E+04, 1.857677E+04, 1.877397E+04, 1.897217E+04, 1.917148E+04,
    1.937178E+04, 1.957319E+04, 1.977572E+04, 1.997924E+04, 2.018376E+04, 2.038940E+04,
    2.059605E+04, 2.080370E+04, 2.101258E+04, 2.122235E+04, 2.143324E+04, 2.164514E+04,
    2.185817E+04, 2.207220E+04,
])

#  ---------------------- M = 8, I = 3 ---------------------------
M = 8
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.189184E+01, 5.706583E+01, 1.101111E+02, 1.690336E+02, 2.357137E+02, 3.093392E+02,
    3.894382E+02, 4.738273E+02, 5.614765E+02, 6.516668E+02, 7.438655E+02, 8.376816E+02,
    9.328071E+02, 1.029019E+03, 1.126152E+03, 1.224093E+03, 1.322726E+03, 1.422012E+03,
    1.521894E+03, 1.622376E+03, 1.723436E+03, 1.825094E+03, 1.927347E+03, 2.030239E+03,
    2.133784E+03, 2.238044E+03, 2.343022E+03, 2.448769E+03, 2.555318E+03, 2.662726E+03,
    2.771034E+03, 2.880252E+03, 2.990447E+03, 3.101631E+03, 3.213843E+03, 3.327092E+03,
    3.441482E+03, 3.556955E+03, 3.673617E+03, 3.791438E+03, 3.910456E+03, 4.030706E+03,
    4.152188E+03, 4.274977E+03, 4.399026E+03, 4.524369E+03, 4.651037E+03, 4.779063E+03,
    4.908435E+03, 5.039134E+03, 5.171234E+03, 5.304767E+03, 5.439664E+03, 5.575945E+03,
    5.713690E+03, 5.852870E+03, 5.993512E+03, 6.135528E+03, 6.279050E+03, 6.424044E+03,
    6.570531E+03, 6.718415E+03, 6.867831E+03, 7.018739E+03, 7.171158E+03, 7.325106E+03,
    7.480473E+03, 7.637402E+03, 7.795844E+03, 7.955817E+03, 8.117334E+03, 8.280340E+03,
    8.444852E+03, 8.610949E+03, 8.778504E+03, 8.947672E+03, 9.118396E+03, 9.290609E+03,
    9.464401E+03, 9.639785E+03, 9.816691E+03, 9.995128E+03, 1.017511E+04, 1.035664E+04,
    1.053982E+04, 1.072449E+04, 1.091073E+04, 1.109857E+04, 1.128800E+04, 1.147896E+04,
    1.167153E+04, 1.186564E+04, 1.206128E+04, 1.225857E+04, 1.245741E+04, 1.265782E+04,
    1.285979E+04, 1.306344E+04, 1.326857E+04, 1.347539E+04, 1.368371E+04, 1.389363E+04,
    1.410515E+04, 1.431819E+04, 1.453295E+04, 1.474923E+04, 1.496703E+04, 1.518647E+04,
    1.540755E+04, 1.563017E+04, 1.585444E+04, 1.608026E+04, 1.630775E+04, 1.653679E+04,
    1.676739E+04, 1.699955E+04, 1.723340E+04, 1.746882E+04, 1.770581E+04, 1.794450E+04,
    1.818465E+04, 1.842652E+04, 1.866997E+04, 1.891501E+04, 1.916153E+04, 1.940978E+04,
    1.965962E+04, 1.991107E+04, 2.016414E+04, 2.041882E+04, 2.067511E+04, 2.093290E+04,
    2.119244E+04, 2.145347E+04, 2.171627E+04, 2.198056E+04, 2.224635E+04, 2.251392E+04,
    2.278313E+04, 2.305384E+04, 2.332620E+04, 2.360007E+04, 2.387573E+04, 2.415291E+04,
    2.443174E+04, 2.471209E+04, 2.499410E+04, 2.527778E+04, 2.556313E+04, 2.584999E+04,
    2.613838E+04, 2.642845E+04, 2.672019E+04, 2.701361E+04, 2.730857E+04, 2.760506E+04,
    2.790322E+04, 2.820309E+04, 2.850449E+04, 2.880758E+04, 2.911222E+04, 2.941855E+04,
    2.972642E+04, 3.003600E+04, 3.034711E+04, 3.065978E+04, 3.097415E+04, 3.129023E+04,
    3.160787E+04, 3.192705E+04, 3.224796E+04, 3.257041E+04, 3.289459E+04, 3.322032E+04,
    3.354777E+04, 3.387661E+04,
])

#  ---------------------- M = 9, I = 1 ---------------------------
M = 9
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.454300E+00, 1.022319E+02, 2.882009E+02, 5.289590E+02, 8.141879E+02, 1.138246E+03,
    1.498364E+03, 1.893680E+03, 2.324649E+03, 2.792534E+03, 3.299086E+03, 3.846299E+03,
    4.436548E+03, 5.072346E+03, 5.756332E+03, 6.491539E+03, 7.281118E+03, 8.128285E+03,
    9.036571E+03, 1.000965E+04, 1.105141E+04, 1.216570E+04, 1.335683E+04, 1.462901E+04,
    1.598660E+04, 1.743436E+04, 1.897692E+04, 2.061904E+04, 2.236584E+04, 2.422246E+04,
    2.619414E+04, 2.828645E+04, 3.050468E+04, 3.285482E+04, 3.534284E+04, 3.797440E+04,
    4.075593E+04, 4.369377E+04, 4.679420E+04, 5.006431E+04, 5.351042E+04, 5.713968E+04,
    6.095944E+04, 6.497664E+04, 6.919892E+04, 7.363414E+04, 7.828963E+04, 8.317352E+04,
    8.829416E+04, 9.365966E+04, 9.927870E+04, 1.051597E+05, 1.113113E+05, 1.177434E+05,
    1.244642E+05, 1.314837E+05, 1.388114E+05, 1.464570E+05, 1.544299E+05, 1.627414E+05,
    1.714005E+05, 1.804188E+05, 1.898065E+05, 1.995743E+05, 2.097337E+05, 2.202957E+05,
    2.312719E+05, 2.426739E+05, 2.545138E+05, 2.668032E+05, 2.795549E+05, 2.927811E+05,
    3.064944E+05, 3.207079E+05, 3.354346E+05, 3.506881E+05, 3.664818E+05, 3.828290E+05,
    3.997436E+05, 4.172406E+05, 4.353337E+05, 4.540374E+05, 4.733667E+05, 4.933375E+05,
    5.139636E+05, 5.352603E+05, 5.572451E+05, 5.799322E+05, 6.033387E+05, 6.274802E+05,
    6.523742E+05, 6.780366E+05, 7.044845E+05, 7.317353E+05, 7.598073E+05, 7.887176E+05,
    8.184840E+05, 8.491238E+05, 8.806575E+05, 9.131028E+05, 9.464777E+05, 9.808023E+05,
    1.016096E+06, 1.052379E+06, 1.089669E+06, 1.127988E+06, 1.167357E+06, 1.207794E+06,
    1.249321E+06, 1.291961E+06, 1.335732E+06, 1.380659E+06, 1.426761E+06, 1.474061E+06,
    1.522583E+06, 1.572347E+06, 1.623377E+06, 1.675696E+06, 1.729329E+06, 1.784297E+06,
    1.840624E+06, 1.898336E+06, 1.957457E+06, 2.018012E+06, 2.080025E+06, 2.143521E+06,
    2.208524E+06, 2.275064E+06, 2.343163E+06, 2.412851E+06, 2.484152E+06, 2.557091E+06,
    2.631700E+06, 2.708003E+06, 2.786027E+06, 2.865802E+06, 2.947356E+06, 3.030718E+06,
    3.115914E+06, 3.202976E+06, 3.291932E+06, 3.382810E+06, 3.475643E+06, 3.570459E+06,
    3.667290E+06, 3.766164E+06, 3.867117E+06, 3.970174E+06, 4.075372E+06, 4.182739E+06,
    4.292312E+06, 4.404117E+06, 4.518192E+06, 4.634568E+06, 4.753277E+06, 4.874358E+06,
    4.997837E+06, 5.123754E+06, 5.252144E+06, 5.383038E+06, 5.516472E+06, 5.652484E+06,
    5.791109E+06, 5.932381E+06, 6.076339E+06, 6.223016E+06, 6.372454E+06, 6.524687E+06,
    6.679752E+06, 6.837689E+06, 6.998536E+06, 7.162331E+06, 7.329112E+06, 7.498922E+06,
    7.671795E+06, 7.847772E+06,
])

#  ---------------------- M = 9, I = 2 ---------------------------
M = 9
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.314830E+00, 1.027046E+02, 2.895361E+02, 5.313995E+02, 8.179047E+02, 1.143446E+03,
    1.505213E+03, 1.902320E+03, 2.335259E+03, 2.805284E+03, 3.314118E+03, 3.863828E+03,
    4.456728E+03, 5.095327E+03, 5.782419E+03, 6.520963E+03, 7.314000E+03, 8.164932E+03,
    9.077250E+03, 1.005464E+04, 1.110102E+04, 1.222023E+04, 1.341664E+04, 1.469432E+04,
    1.605802E+04, 1.751204E+04, 1.906127E+04, 2.071061E+04, 2.246511E+04, 2.432974E+04,
    2.631009E+04, 2.841143E+04, 3.063941E+04, 3.299970E+04, 3.549845E+04, 3.814136E+04,
    4.093511E+04, 4.388563E+04, 4.699944E+04, 5.028368E+04, 5.374467E+04, 5.738960E+04,
    6.122559E+04, 6.526014E+04, 6.950067E+04, 7.395475E+04, 7.863036E+04, 8.353538E+04,
    8.867784E+04, 9.406655E+04, 9.970955E+04, 1.056157E+05, 1.117939E+05, 1.182531E+05,
    1.250031E+05, 1.320525E+05, 1.394112E+05, 1.470895E+05, 1.550966E+05, 1.634432E+05,
    1.721394E+05, 1.811964E+05, 1.906239E+05, 2.004330E+05, 2.106356E+05, 2.212424E+05,
    2.322657E+05, 2.437162E+05, 2.556059E+05, 2.679482E+05, 2.807536E+05, 2.940361E+05,
    3.078079E+05, 3.220813E+05, 3.368710E+05, 3.521889E+05, 3.680495E+05, 3.844658E+05,
    4.014520E+05, 4.190232E+05, 4.371929E+05, 4.559761E+05, 4.753874E+05, 4.954424E+05,
    5.161552E+05, 5.375425E+05, 5.596200E+05, 5.824029E+05, 6.059083E+05, 6.301519E+05,
    6.551502E+05, 6.809212E+05, 7.074811E+05, 7.348473E+05, 7.630374E+05, 7.920695E+05,
    8.219613E+05, 8.527311E+05, 8.843978E+05, 9.169790E+05, 9.504949E+05, 9.849644E+05,
    1.020406E+06, 1.056841E+06, 1.094290E+06, 1.132770E+06, 1.172304E+06, 1.212911E+06,
    1.254614E+06, 1.297432E+06, 1.341388E+06, 1.386503E+06, 1.432799E+06, 1.480299E+06,
    1.529023E+06, 1.578997E+06, 1.630242E+06, 1.682781E+06, 1.736639E+06, 1.791837E+06,
    1.848402E+06, 1.906357E+06, 1.965726E+06, 2.026534E+06, 2.088807E+06, 2.152569E+06,
    2.217847E+06, 2.284665E+06, 2.353051E+06, 2.423030E+06, 2.494629E+06, 2.567875E+06,
    2.642796E+06, 2.719418E+06, 2.797771E+06, 2.877882E+06, 2.959776E+06, 3.043487E+06,
    3.129040E+06, 3.216467E+06, 3.305794E+06, 3.397055E+06, 3.490276E+06, 3.585490E+06,
    3.682726E+06, 3.782016E+06, 3.883388E+06, 3.986878E+06, 4.092517E+06, 4.200333E+06,
    4.310363E+06, 4.422637E+06, 4.537190E+06, 4.654052E+06, 4.773258E+06, 4.894845E+06,
    5.018842E+06, 5.145286E+06, 5.274211E+06, 5.405652E+06, 5.539647E+06, 5.676225E+06,
    5.815431E+06, 5.957293E+06, 6.101851E+06, 6.249144E+06, 6.399204E+06, 6.552073E+06,
    6.707787E+06, 6.866385E+06, 7.027904E+06, 7.192382E+06, 7.359862E+06, 7.530380E+06,
    7.703973E+06, 7.880689E+06,
])

#  ---------------------- M = 10, I = 1 ---------------------------
M = 10
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    3.067800E+00, 2.332671E+02, 6.566540E+02, 1.204608E+03, 1.853462E+03, 2.589584E+03,
    3.403999E+03, 4.290795E+03, 5.246348E+03, 6.268939E+03, 7.358458E+03, 8.515917E+03,
    9.743304E+03, 1.104335E+04, 1.241945E+04, 1.387519E+04, 1.541486E+04, 1.704306E+04,
    1.876472E+04, 2.058489E+04, 2.250873E+04, 2.454224E+04, 2.669106E+04, 2.896127E+04,
    3.135884E+04, 3.389046E+04, 3.656272E+04, 3.938207E+04, 4.235539E+04, 4.549001E+04,
    4.879273E+04, 5.227161E+04, 5.593368E+04, 5.978687E+04, 6.383887E+04, 6.809790E+04,
    7.257236E+04, 7.727028E+04, 8.220086E+04, 8.737241E+04, 9.279450E+04, 9.847505E+04,
    1.044246E+05, 1.106525E+05, 1.171684E+05, 1.239821E+05, 1.311035E+05, 1.385434E+05,
    1.463120E+05, 1.544204E+05, 1.628794E+05, 1.717004E+05, 1.808944E+05, 1.904724E+05,
    2.004476E+05, 2.108313E+05, 2.216352E+05, 2.328730E+05, 2.445559E+05, 2.566978E+05,
    2.693113E+05, 2.824111E+05, 2.960088E+05, 3.101196E+05, 3.247566E+05, 3.399344E+05,
    3.556676E+05, 3.719701E+05, 3.888595E+05, 4.063475E+05, 4.244521E+05, 4.431882E+05,
    4.625704E+05, 4.826163E+05, 5.033429E+05, 5.247666E+05, 5.469033E+05, 5.697703E+05,
    5.933851E+05, 6.177656E+05, 6.429297E+05, 6.688959E+05, 6.956828E+05, 7.233073E+05,
    7.517906E+05, 7.811517E+05, 8.114085E+05, 8.425809E+05, 8.746909E+05, 9.077571E+05,
    9.417998E+05, 9.768401E+05, 1.012899E+06, 1.050001E+06, 1.088162E+06, 1.127407E+06,
    1.167759E+06, 1.209239E+06, 1.251871E+06, 1.295677E+06, 1.340680E+06, 1.386904E+06,
    1.434373E+06, 1.483110E+06, 1.533142E+06, 1.584490E+06, 1.637182E+06, 1.691244E+06,
    1.746699E+06, 1.803572E+06, 1.861893E+06, 1.921685E+06, 1.982978E+06, 2.045797E+06,
    2.110167E+06, 2.176121E+06, 2.243683E+06, 2.312882E+06, 2.383749E+06, 2.456307E+06,
    2.530594E+06, 2.606631E+06, 2.684450E+06, 2.764086E+06, 2.845564E+06, 2.928916E+06,
    3.014177E+06, 3.101370E+06, 3.190535E+06, 3.281700E+06, 3.374897E+06, 3.470160E+06,
    3.567520E+06, 3.667012E+06, 3.768671E+06, 3.872526E+06, 3.978618E+06, 4.086977E+06,
    4.197639E+06, 4.310638E+06, 4.426011E+06, 4.543793E+06, 4.664020E+06, 4.786729E+06,
    4.911957E+06, 5.039746E+06, 5.170124E+06, 5.303134E+06, 5.438818E+06, 5.577205E+06,
    5.718344E+06, 5.862272E+06, 6.009019E+06, 6.158638E+06, 6.311162E+06, 6.466636E+06,
    6.625096E+06, 6.786586E+06, 6.951149E+06, 7.118822E+06, 7.289654E+06, 7.463683E+06,
    7.640954E+06, 7.821510E+06, 8.005394E+06, 8.192658E+06, 8.383334E+06, 8.577472E+06,
    8.775125E+06, 8.976324E+06, 9.181131E+06, 9.389576E+06, 9.601719E+06, 9.817605E+06,
    1.003727E+07, 1.026078E+07,
])

#  ---------------------- M = 11, I = 1 ---------------------------
M = 11
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    3.834090E+00, 3.544986E+01, 8.933083E+01, 1.601234E+02, 2.441067E+02, 3.392668E+02,
    4.443978E+02, 5.586602E+02, 6.814457E+02, 8.123287E+02, 9.510416E+02, 1.097456E+03,
    1.251567E+03, 1.413475E+03, 1.583372E+03, 1.761529E+03, 1.948287E+03, 2.144046E+03,
    2.349255E+03, 2.564410E+03, 2.790043E+03, 3.026722E+03, 3.275047E+03, 3.535644E+03,
    3.809167E+03, 4.096297E+03, 4.397737E+03, 4.714220E+03, 5.046499E+03, 5.395358E+03,
    5.761605E+03, 6.146077E+03, 6.549640E+03, 6.973192E+03, 7.417660E+03, 7.884006E+03,
    8.373224E+03, 8.886347E+03, 9.424443E+03, 9.988617E+03, 1.058002E+04, 1.119983E+04,
    1.184929E+04, 1.252967E+04, 1.324228E+04, 1.398851E+04, 1.476975E+04, 1.558747E+04,
    1.644319E+04, 1.733848E+04, 1.827495E+04, 1.925429E+04, 2.027822E+04, 2.134854E+04,
    2.246709E+04, 2.363579E+04, 2.485661E+04, 2.613158E+04, 2.746282E+04, 2.885248E+04,
    3.030281E+04, 3.181610E+04, 3.339475E+04, 3.504120E+04, 3.675797E+04, 3.854766E+04,
    4.041296E+04, 4.235662E+04, 4.438148E+04, 4.649045E+04, 4.868654E+04, 5.097282E+04,
    5.335248E+04, 5.582878E+04, 5.840505E+04, 6.108474E+04, 6.387137E+04, 6.676857E+04,
    6.978004E+04, 7.290961E+04, 7.616117E+04, 7.953873E+04, 8.304638E+04, 8.668833E+04,
    9.046886E+04, 9.439239E+04, 9.846341E+04, 1.026865E+05, 1.070664E+05, 1.116079E+05,
    1.163160E+05, 1.211955E+05, 1.262517E+05, 1.314898E+05, 1.369150E+05, 1.425329E+05,
    1.483489E+05, 1.543688E+05, 1.605982E+05, 1.670429E+05, 1.737090E+05, 1.806025E+05,
    1.877295E+05, 1.950963E+05, 2.027093E+05, 2.105748E+05, 2.186995E+05, 2.270900E+05,
    2.357531E+05, 2.446956E+05, 2.539243E+05, 2.634465E+05, 2.732691E+05, 2.833994E+05,
    2.938447E+05, 3.046123E+05, 3.157098E+05, 3.271446E+05, 3.389245E+05, 3.510570E+05,
    3.635501E+05, 3.764116E+05, 3.896493E+05, 4.032714E+05, 4.172859E+05, 4.317008E+05,
    4.465246E+05, 4.617653E+05, 4.774313E+05, 4.935310E+05, 5.100729E+05, 5.270654E+05,
    5.445171E+05, 5.624366E+05, 5.808324E+05, 5.997132E+05, 6.190879E+05, 6.389651E+05,
    6.593535E+05, 6.802621E+05, 7.016996E+05, 7.236749E+05, 7.461970E+05, 7.692747E+05,
    7.929169E+05, 8.171327E+05, 8.419309E+05, 8.673206E+05, 8.933106E+05, 9.199101E+05,
    9.471280E+05, 9.749732E+05, 1.003455E+06, 1.032582E+06, 1.062363E+06, 1.092807E+06,
    1.123924E+06, 1.155722E+06, 1.188210E+06, 1.221397E+06, 1.255292E+06, 1.289903E+06,
    1.325240E+06, 1.361311E+06, 1.398126E+06, 1.435692E+06, 1.474019E+06, 1.513114E+06,
    1.552988E+06, 1.593648E+06, 1.635103E+06, 1.677361E+06, 1.720431E+06, 1.764321E+06,
    1.809040E+06, 1.854596E+06, 1.900996E+06, 1.948250E+06, 1.996365E+06, 2.045349E+06,
    2.095211E+06, 2.145958E+06, 2.197599E+06, 2.250140E+06, 2.303590E+06, 2.357956E+06,
    2.413246E+06, 2.469467E+06, 2.526627E+06, 2.584734E+06, 2.643794E+06, 2.703815E+06,
    2.764803E+06, 2.826766E+06, 2.889712E+06, 2.953646E+06, 3.018575E+06, 3.084507E+06,
    3.151448E+06, 3.219404E+06, 3.288382E+06, 3.358388E+06, 3.429429E+06, 3.501511E+06,
    3.574640E+06, 3.648822E+06, 3.724062E+06, 3.800367E+06, 3.877743E+06, 3.956195E+06,
    4.035729E+06, 4.116350E+06, 4.198063E+06, 4.280875E+06, 4.364789E+06, 4.449812E+06,
    4.535948E+06, 4.623202E+06, 4.711579E+06, 4.801083E+06, 4.891720E+06, 4.983494E+06,
    5.076409E+06, 5.170470E+06, 5.265680E+06, 5.362045E+06, 5.459568E+06, 5.558252E+06,
    5.658103E+06, 5.759124E+06, 5.861317E+06, 5.964688E+06, 6.069240E+06, 6.174975E+06,
    6.281898E+06, 6.390011E+06, 6.499317E+06, 6.609820E+06, 6.721523E+06, 6.834428E+06,
    6.948538E+06, 7.063856E+06, 7.180384E+06, 7.298125E+06, 7.417081E+06, 7.537255E+06,
    7.658648E+06, 7.781264E+06, 7.905103E+06, 8.030168E+06, 8.156460E+06, 8.283982E+06,
    8.412735E+06, 8.542721E+06, 8.673941E+06, 8.806397E+06, 8.940089E+06, 9.075020E+06,
    9.211190E+06, 9.348600E+06, 9.487252E+06, 9.627147E+06, 9.768284E+06, 9.910666E+06,
    1.005429E+07, 1.019916E+07, 1.034528E+07, 1.049264E+07, 1.064125E+07, 1.079111E+07,
    1.094222E+07, 1.109457E+07, 1.124817E+07, 1.140301E+07, 1.155911E+07, 1.171645E+07,
    1.187503E+07, 1.203487E+07, 1.219595E+07, 1.235827E+07, 1.252184E+07, 1.268666E+07,
    1.285272E+07, 1.302002E+07, 1.318857E+07, 1.335836E+07, 1.352938E+07, 1.370165E+07,
    1.387516E+07, 1.404990E+07, 1.422589E+07, 1.440310E+07, 1.458155E+07, 1.476124E+07,
    1.494215E+07, 1.512430E+07, 1.530767E+07, 1.549227E+07, 1.567809E+07, 1.586514E+07,
    1.605341E+07,
])

#  ---------------------- M = 11, I = 2 ---------------------------
M = 11
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.689360E+00, 2.371743E+01, 5.973090E+01, 1.070476E+02, 1.631783E+02, 2.267778E+02,
    2.970405E+02, 3.734061E+02, 4.554693E+02, 5.429478E+02, 6.356653E+02, 7.335395E+02,
    8.365694E+02, 9.448250E+02, 1.058437E+03, 1.177588E+03, 1.302505E+03, 1.433451E+03,
    1.570721E+03, 1.714634E+03, 1.865531E+03, 2.023770E+03, 2.189723E+03, 2.363775E+03,
    2.546320E+03, 2.737759E+03, 2.938503E+03, 3.148970E+03, 3.369585E+03, 3.600778E+03,
    3.842989E+03, 4.096664E+03, 4.362256E+03, 4.640230E+03, 4.931055E+03, 5.235214E+03,
    5.553197E+03, 5.885507E+03, 6.232655E+03, 6.595167E+03, 6.973580E+03, 7.368444E+03,
    7.780322E+03, 8.209792E+03, 8.657444E+03, 9.123885E+03, 9.609736E+03, 1.011563E+04,
    1.064223E+04, 1.119020E+04, 1.176022E+04, 1.235300E+04, 1.296927E+04, 1.360974E+04,
    1.427520E+04, 1.496640E+04, 1.568415E+04, 1.642926E+04, 1.720255E+04, 1.800489E+04,
    1.883715E+04, 1.970021E+04, 2.059500E+04, 2.152244E+04, 2.248350E+04, 2.347915E+04,
    2.451040E+04, 2.557826E+04, 2.668378E+04, 2.782802E+04, 2.901208E+04, 3.023707E+04,
    3.150413E+04, 3.281442E+04, 3.416912E+04, 3.556945E+04, 3.701663E+04, 3.851194E+04,
    4.005664E+04, 4.165205E+04, 4.329951E+04, 4.500037E+04, 4.675603E+04, 4.856789E+04,
    5.043739E+04, 5.236600E+04, 5.435520E+04, 5.640652E+04, 5.852151E+04, 6.070172E+04,
    6.294876E+04, 6.526426E+04, 6.764987E+04, 7.010727E+04, 7.263815E+04, 7.524427E+04,
    7.792737E+04, 8.068925E+04, 8.353171E+04, 8.645661E+04, 8.946581E+04, 9.256120E+04,
    9.574471E+04, 9.901829E+04, 1.023839E+05, 1.058436E+05, 1.093993E+05, 1.130531E+05,
    1.168072E+05, 1.206635E+05, 1.246243E+05, 1.286917E+05, 1.328678E+05, 1.371549E+05,
    1.415551E+05, 1.460708E+05, 1.507042E+05, 1.554576E+05, 1.603333E+05, 1.653337E+05,
    1.704610E+05, 1.757178E+05, 1.811063E+05, 1.866290E+05, 1.922883E+05, 1.980868E+05,
    2.040268E+05, 2.101108E+05, 2.163414E+05, 2.227210E+05, 2.292523E+05, 2.359378E+05,
    2.427799E+05, 2.497814E+05, 2.569448E+05, 2.642727E+05, 2.717677E+05, 2.794324E+05,
    2.872695E+05, 2.952817E+05, 3.034715E+05, 3.118417E+05, 3.203948E+05, 3.291337E+05,
    3.380608E+05, 3.471791E+05, 3.564910E+05, 3.659994E+05, 3.757069E+05, 3.856162E+05,
    3.957300E+05, 4.060510E+05, 4.165819E+05, 4.273253E+05, 4.382841E+05, 4.494608E+05,
    4.608582E+05, 4.724789E+05, 4.843256E+05, 4.964010E+05, 5.087078E+05, 5.212486E+05,
    5.340260E+05, 5.470427E+05, 5.603014E+05, 5.738046E+05, 5.875549E+05, 6.015551E+05,
    6.158075E+05, 6.303150E+05, 6.450799E+05, 6.601048E+05, 6.753923E+05, 6.909449E+05,
    7.067651E+05, 7.228554E+05, 7.392182E+05, 7.558561E+05, 7.727714E+05, 7.899665E+05,
    8.074439E+05, 8.252060E+05, 8.432551E+05, 8.615934E+05, 8.802235E+05, 8.991475E+05,
    9.183677E+05, 9.378864E+05, 9.577058E+05, 9.778282E+05, 9.982556E+05, 1.018990E+06,
    1.040034E+06, 1.061390E+06, 1.083059E+06, 1.105044E+06, 1.127346E+06, 1.149969E+06,
    1.172913E+06, 1.196180E+06, 1.219773E+06, 1.243694E+06, 1.267944E+06, 1.292525E+06,
    1.317439E+06, 1.342688E+06, 1.368273E+06, 1.394197E+06, 1.420461E+06, 1.447066E+06,
    1.474015E+06, 1.501309E+06, 1.528949E+06, 1.556938E+06, 1.585276E+06, 1.613965E+06,
    1.643007E+06, 1.672403E+06, 1.702154E+06, 1.732262E+06, 1.762729E+06, 1.793555E+06,
    1.824742E+06, 1.856291E+06, 1.888203E+06, 1.920480E+06, 1.953123E+06, 1.986132E+06,
    2.019510E+06, 2.053256E+06, 2.087373E+06, 2.121861E+06, 2.156721E+06, 2.191954E+06,
    2.227562E+06, 2.263544E+06, 2.299902E+06, 2.336637E+06, 2.373749E+06, 2.411240E+06,
    2.449110E+06, 2.487359E+06, 2.525989E+06, 2.565001E+06, 2.604394E+06, 2.644169E+06,
    2.684328E+06, 2.724871E+06, 2.765797E+06, 2.807108E+06, 2.848805E+06, 2.890886E+06,
    2.933354E+06, 2.976209E+06, 3.019450E+06, 3.063078E+06, 3.107094E+06, 3.151498E+06,
    3.196289E+06, 3.241469E+06, 3.287037E+06, 3.332994E+06, 3.379340E+06, 3.426074E+06,
    3.473198E+06, 3.520711E+06, 3.568613E+06, 3.616905E+06, 3.665585E+06, 3.714655E+06,
    3.764114E+06, 3.813962E+06, 3.864200E+06, 3.914826E+06, 3.965841E+06, 4.017245E+06,
    4.069037E+06, 4.121218E+06, 4.173787E+06, 4.226743E+06, 4.280088E+06, 4.333819E+06,
    4.387938E+06, 4.442443E+06, 4.497335E+06, 4.552613E+06, 4.608276E+06, 4.664325E+06,
    4.720758E+06, 4.777576E+06, 4.834778E+06, 4.892363E+06, 4.950330E+06, 5.008681E+06,
    5.067413E+06, 5.126526E+06, 5.186021E+06, 5.245895E+06, 5.306149E+06, 5.366782E+06,
    5.427793E+06,
])

#  ---------------------- M = 12, I = 1 ---------------------------
M = 12
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    3.573524E+01, 2.896824E+03, 8.174539E+03, 1.500830E+04, 2.310835E+04, 3.233901E+04,
    4.267884E+04, 5.420733E+04, 6.708747E+04, 8.154826E+04, 9.787188E+04, 1.163871E+05,
    1.374678E+05, 1.615349E+05, 1.890610E+05, 2.205764E+05, 2.566761E+05, 2.980283E+05,
    3.453830E+05, 3.995815E+05, 4.615675E+05, 5.323980E+05, 6.132560E+05, 7.054636E+05,
    8.104973E+05, 9.300032E+05, 1.065814E+06, 1.219969E+06, 1.394732E+06, 1.592616E+06,
    1.816403E+06, 2.069175E+06, 2.354336E+06, 2.675644E+06, 3.037246E+06, 3.443706E+06,
    3.900048E+06, 4.411790E+06, 4.984997E+06, 5.626315E+06, 6.343026E+06, 7.143101E+06,
    8.035255E+06, 9.029012E+06, 1.013476E+07, 1.136382E+07, 1.272855E+07, 1.424236E+07,
    1.591986E+07, 1.777692E+07, 1.983076E+07, 2.210007E+07, 2.460508E+07, 2.736772E+07,
    3.041171E+07, 3.376270E+07, 3.744839E+07, 4.149872E+07, 4.594593E+07, 5.082484E+07,
    5.617291E+07, 6.203049E+07, 6.844098E+07, 7.545107E+07, 8.311086E+07, 9.147421E+07,
    1.005989E+08, 1.105468E+08, 1.213843E+08, 1.331825E+08, 1.460176E+08, 1.599709E+08,
    1.751294E+08, 1.915862E+08, 2.094406E+08, 2.287987E+08, 2.497734E+08, 2.724856E+08,
    2.970635E+08, 3.236443E+08, 3.523735E+08, 3.834064E+08, 4.169078E+08, 4.530531E+08,
    4.920286E+08, 5.340321E+08, 5.792739E+08, 6.279767E+08, 6.803770E+08, 7.367252E+08,
    7.972871E+08, 8.623440E+08, 9.321937E+08, 1.007151E+09, 1.087551E+09, 1.173744E+09,
    1.266104E+09, 1.365025E+09, 1.470923E+09, 1.584237E+09, 1.705430E+09, 1.834993E+09,
    1.973439E+09, 2.121315E+09, 2.279192E+09, 2.447675E+09, 2.627399E+09, 2.819035E+09,
    3.023286E+09, 3.240894E+09, 3.472638E+09, 3.719338E+09, 3.981857E+09, 4.261099E+09,
    4.558015E+09, 4.873603E+09, 5.208913E+09, 5.565043E+09, 5.943148E+09, 6.344438E+09,
    6.770183E+09, 7.221711E+09, 7.700418E+09, 8.207763E+09, 8.745275E+09, 9.314557E+09,
    9.917283E+09, 1.055521E+10, 1.123017E+10, 1.194408E+10, 1.269895E+10, 1.349688E+10,
    1.434006E+10, 1.523078E+10, 1.617143E+10, 1.716451E+10, 1.821263E+10, 1.931852E+10,
    2.048501E+10, 2.171506E+10, 2.301179E+10, 2.437840E+10, 2.581826E+10, 2.733487E+10,
    2.893189E+10, 3.061313E+10, 3.238254E+10, 3.424425E+10, 3.620258E+10, 3.826198E+10,
    4.042712E+10, 4.270285E+10, 4.509420E+10, 4.760643E+10, 5.024498E+10, 5.301553E+10,
    5.592397E+10, 5.897644E+10, 6.217929E+10, 6.553915E+10, 6.906289E+10, 7.275761E+10,
    7.663079E+10, 8.069007E+10, 8.494343E+10, 8.939920E+10, 9.406595E+10, 9.895261E+10,
    1.040684E+11, 1.094230E+11, 1.150262E+11, 1.208886E+11, 1.270205E+11, 1.334333E+11,
    1.401384E+11, 1.471476E+11,
])

#  ---------------------- M = 12, I = 2 ---------------------------
M = 12
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.382603E+01, 1.931437E+03, 5.450319E+03, 1.000669E+04, 1.540736E+04, 2.156205E+04,
    2.845720E+04, 3.614739E+04, 4.474377E+04, 5.440219E+04, 6.531440E+04, 7.770355E+04,
    9.182310E+04, 1.079584E+05, 1.264298E+05, 1.475967E+05, 1.718627E+05, 1.996812E+05,
    2.315613E+05, 2.680746E+05, 3.098624E+05, 3.576437E+05, 4.122233E+05, 4.745014E+05,
    5.454834E+05, 6.262912E+05, 7.181744E+05, 8.225232E+05, 9.408828E+05, 1.074968E+06,
    1.226680E+06, 1.398121E+06, 1.591620E+06, 1.809745E+06, 2.055330E+06, 2.331499E+06,
    2.641689E+06, 2.989677E+06, 3.379614E+06, 3.816053E+06, 4.303982E+06, 4.848864E+06,
    5.456671E+06, 6.133931E+06, 6.887769E+06, 7.725955E+06, 8.656957E+06, 9.689991E+06,
    1.083508E+07, 1.210311E+07, 1.350592E+07, 1.505634E+07, 1.676827E+07, 1.865678E+07,
    2.073817E+07, 2.303008E+07, 2.555154E+07, 2.832314E+07, 3.136707E+07, 3.470727E+07,
    3.836952E+07, 4.238158E+07, 4.677333E+07, 5.157688E+07, 5.682677E+07, 6.256005E+07,
    6.881650E+07, 7.563878E+07, 8.307263E+07, 9.116702E+07, 9.997437E+07, 1.095508E+08,
    1.199563E+08, 1.312550E+08, 1.435154E+08, 1.568106E+08, 1.712185E+08, 1.868225E+08,
    2.037110E+08, 2.219786E+08, 2.417258E+08, 2.630596E+08, 2.860939E+08, 3.109496E+08,
    3.377555E+08, 3.666480E+08, 3.977722E+08, 4.312820E+08, 4.673407E+08, 5.061213E+08,
    5.478072E+08, 5.925928E+08, 6.406837E+08, 6.922978E+08, 7.476655E+08, 8.070303E+08,
    8.706501E+08, 9.387968E+08, 1.011758E+09, 1.089838E+09, 1.173356E+09, 1.262651E+09,
    1.358080E+09, 1.460018E+09, 1.568862E+09, 1.685029E+09, 1.808960E+09, 1.941117E+09,
    2.081987E+09, 2.232084E+09, 2.391947E+09, 2.562142E+09, 2.743268E+09, 2.935948E+09,
    3.140842E+09, 3.358640E+09, 3.590069E+09, 3.835888E+09, 4.096897E+09, 4.373935E+09,
    4.667879E+09, 4.979648E+09, 5.310211E+09, 5.660576E+09, 6.031804E+09, 6.425001E+09,
    6.841331E+09, 7.282006E+09, 7.748297E+09, 8.241535E+09, 8.763108E+09, 9.314471E+09,
    9.897139E+09, 1.051270E+10, 1.116282E+10, 1.184921E+10, 1.257370E+10, 1.333817E+10,
    1.414458E+10, 1.499499E+10, 1.589154E+10, 1.683648E+10, 1.783212E+10, 1.888090E+10,
    1.998535E+10, 2.114810E+10, 2.237192E+10, 2.365965E+10, 2.501428E+10, 2.643891E+10,
    2.793676E+10, 2.951121E+10, 3.116574E+10, 3.290399E+10, 3.472976E+10, 3.664695E+10,
    3.865967E+10, 4.077217E+10, 4.298886E+10, 4.531432E+10, 4.775333E+10, 5.031083E+10,
    5.299197E+10, 5.580208E+10, 5.874669E+10, 6.183156E+10, 6.506265E+10, 6.844617E+10,
    7.198850E+10, 7.569632E+10, 7.957652E+10, 8.363629E+10, 8.788300E+10, 9.232438E+10,
    9.696836E+10, 1.018233E+11,
])

#  ---------------------- M = 13, I = 1 ---------------------------
M = 13
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[3]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.538496E+01, 1.602673E+01, 1.728778E+01, 2.005293E+01, 2.374009E+01, 2.800797E+01,
    3.266623E+01, 3.759941E+01, 4.273326E+01, 4.801797E+01, 5.341894E+01, 5.891142E+01,
    6.447729E+01, 7.010295E+01, 7.577806E+01, 8.149456E+01, 8.724613E+01, 9.302771E+01,
    9.883525E+01, 1.046654E+02, 1.105156E+02, 1.163834E+02, 1.222671E+02, 1.281652E+02,
    1.340764E+02, 1.399998E+02, 1.459345E+02, 1.518800E+02, 1.578359E+02, 1.638019E+02,
    1.697777E+02, 1.757636E+02, 1.817594E+02, 1.877655E+02, 1.937821E+02, 1.998096E+02,
    2.058484E+02, 2.118992E+02, 2.179624E+02, 2.240388E+02, 2.301290E+02, 2.362337E+02,
    2.423537E+02, 2.484898E+02, 2.546428E+02, 2.608134E+02, 2.670027E+02, 2.732114E+02,
    2.794403E+02, 2.856903E+02, 2.919624E+02, 2.982572E+02, 3.045757E+02, 3.109188E+02,
    3.172871E+02, 3.236817E+02, 3.301032E+02, 3.365525E+02, 3.430303E+02, 3.495374E+02,
    3.560745E+02, 3.626425E+02, 3.692419E+02, 3.758735E+02, 3.825380E+02, 3.892360E+02,
    3.959682E+02, 4.027353E+02, 4.095377E+02, 4.163762E+02, 4.232513E+02, 4.301635E+02,
    4.371135E+02, 4.441017E+02, 4.511287E+02, 4.581950E+02, 4.653011E+02, 4.724475E+02,
    4.796345E+02, 4.868628E+02, 4.941326E+02, 5.014445E+02, 5.087989E+02, 5.161961E+02,
    5.236366E+02, 5.311207E+02, 5.386489E+02, 5.462214E+02, 5.538387E+02, 5.615011E+02,
    5.692089E+02, 5.769624E+02, 5.847620E+02, 5.926080E+02, 6.005006E+02, 6.084403E+02,
    6.164272E+02, 6.244617E+02, 6.325440E+02, 6.406744E+02, 6.488532E+02, 6.570806E+02,
    6.653569E+02, 6.736823E+02, 6.820571E+02, 6.904815E+02, 6.989557E+02, 7.074801E+02,
    7.160547E+02, 7.246798E+02, 7.333557E+02, 7.420826E+02, 7.508606E+02, 7.596900E+02,
    7.685709E+02, 7.775036E+02, 7.864883E+02, 7.955252E+02, 8.046144E+02, 8.137561E+02,
    8.229506E+02, 8.321980E+02, 8.414985E+02, 8.508522E+02, 8.602594E+02, 8.697202E+02,
    8.792348E+02, 8.888034E+02, 8.984261E+02, 9.081031E+02, 9.178345E+02, 9.276206E+02,
    9.374615E+02, 9.473573E+02, 9.573083E+02, 9.673144E+02, 9.773761E+02, 9.874933E+02,
    9.976662E+02, 1.007895E+03, 1.018180E+03, 1.028521E+03, 1.038918E+03, 1.049372E+03,
    1.059883E+03, 1.070450E+03, 1.081075E+03, 1.091756E+03, 1.102495E+03, 1.113291E+03,
    1.124144E+03, 1.135055E+03, 1.146024E+03, 1.157051E+03, 1.168136E+03, 1.179280E+03,
    1.190482E+03, 1.201742E+03, 1.213061E+03, 1.224439E+03, 1.235875E+03, 1.247371E+03,
    1.258926E+03, 1.270541E+03, 1.282215E+03, 1.293948E+03, 1.305742E+03, 1.317595E+03,
    1.329509E+03, 1.341483E+03, 1.353517E+03, 1.365612E+03, 1.377767E+03, 1.389983E+03,
    1.402260E+03, 1.414599E+03, 1.426998E+03, 1.439459E+03, 1.451981E+03, 1.464566E+03,
    1.477211E+03, 1.489919E+03, 1.502689E+03, 1.515521E+03, 1.528416E+03, 1.541373E+03,
    1.554393E+03, 1.567476E+03, 1.580621E+03, 1.593830E+03, 1.607102E+03, 1.620437E+03,
    1.633836E+03, 1.647299E+03, 1.660825E+03, 1.674416E+03, 1.688070E+03, 1.701789E+03,
    1.715573E+03, 1.729421E+03, 1.743334E+03, 1.757311E+03, 1.771354E+03, 1.785462E+03,
    1.799635E+03, 1.813874E+03, 1.828179E+03, 1.842549E+03, 1.856985E+03, 1.871488E+03,
    1.886056E+03, 1.900691E+03, 1.915393E+03, 1.930162E+03, 1.944997E+03, 1.959899E+03,
    1.974869E+03, 1.989906E+03, 2.005010E+03, 2.020183E+03, 2.035423E+03, 2.050731E+03,
    2.066107E+03, 2.081552E+03, 2.097065E+03, 2.112647E+03, 2.128297E+03, 2.144017E+03,
    2.159805E+03, 2.175663E+03, 2.191591E+03, 2.207588E+03, 2.223654E+03, 2.239791E+03,
    2.255998E+03, 2.272275E+03, 2.288623E+03, 2.305041E+03, 2.321530E+03, 2.338090E+03,
    2.354721E+03, 2.371423E+03, 2.388197E+03, 2.405042E+03, 2.421959E+03, 2.438948E+03,
    2.456009E+03, 2.473143E+03, 2.490349E+03, 2.507627E+03, 2.524978E+03, 2.542402E+03,
    2.559899E+03, 2.577470E+03, 2.595114E+03, 2.612831E+03, 2.630622E+03, 2.648488E+03,
    2.666427E+03, 2.684441E+03, 2.702529E+03, 2.720691E+03, 2.738929E+03, 2.757241E+03,
    2.775629E+03, 2.794091E+03, 2.812630E+03, 2.831243E+03, 2.849933E+03, 2.868698E+03,
    2.887540E+03, 2.906458E+03, 2.925453E+03, 2.944524E+03, 2.963671E+03, 2.982896E+03,
    3.002198E+03, 3.021577E+03, 3.041034E+03, 3.060568E+03, 3.080180E+03, 3.099870E+03,
    3.119638E+03, 3.139484E+03, 3.159409E+03, 3.179412E+03, 3.199494E+03, 3.219655E+03,
    3.239895E+03, 3.260214E+03, 3.280613E+03, 3.301091E+03, 3.321649E+03, 3.342287E+03,
    3.363005E+03, 3.383803E+03, 3.404681E+03, 3.425640E+03, 3.446680E+03, 3.467800E+03,
    3.489001E+03, 3.510284E+03, 3.531647E+03, 3.553093E+03, 3.574619E+03, 3.596228E+03,
    3.617918E+03, 3.639691E+03, 3.661545E+03, 3.683482E+03, 3.705502E+03, 3.727604E+03,
    3.749788E+03, 3.772056E+03, 3.794407E+03, 3.816841E+03, 3.839358E+03, 3.861959E+03,
    3.884644E+03, 3.907412E+03, 3.930264E+03, 3.953201E+03, 3.976221E+03, 3.999326E+03,
    4.022515E+03, 4.045789E+03, 4.069147E+03, 4.092591E+03, 4.116119E+03, 4.139733E+03,
    4.163432E+03, 4.187216E+03, 4.211086E+03, 4.235041E+03, 4.259082E+03, 4.283209E+03,
    4.307422E+03, 4.331721E+03, 4.356107E+03, 4.380578E+03, 4.405137E+03, 4.429781E+03,
    4.454513E+03, 4.479331E+03, 4.504237E+03, 4.529229E+03, 4.554309E+03, 4.579476E+03,
    4.604730E+03, 4.630072E+03, 4.655502E+03, 4.681019E+03, 4.706624E+03, 4.732317E+03,
    4.758098E+03, 4.783967E+03, 4.809924E+03, 4.835970E+03, 4.862104E+03, 4.888326E+03,
    4.914637E+03, 4.941037E+03, 4.967526E+03, 4.994104E+03, 5.020770E+03, 5.047526E+03,
    5.074371E+03, 5.101305E+03, 5.128328E+03, 5.155441E+03, 5.182643E+03, 5.209935E+03,
    5.237316E+03, 5.264787E+03, 5.292348E+03, 5.319999E+03, 5.347739E+03, 5.375570E+03,
    5.403491E+03, 5.431502E+03, 5.459603E+03, 5.487794E+03, 5.516076E+03, 5.544448E+03,
    5.572910E+03, 5.601464E+03, 5.630107E+03, 5.658842E+03, 5.687667E+03, 5.716582E+03,
    5.745589E+03, 5.774686E+03, 5.803874E+03, 5.833154E+03, 5.862524E+03, 5.891985E+03,
    5.921537E+03, 5.951181E+03, 5.980916E+03, 6.010741E+03, 6.040658E+03, 6.070667E+03,
    6.100766E+03, 6.130957E+03, 6.161240E+03, 6.191614E+03, 6.222079E+03, 6.252635E+03,
    6.283284E+03, 6.314023E+03, 6.344855E+03, 6.375777E+03, 6.406792E+03, 6.437898E+03,
    6.469095E+03, 6.500385E+03, 6.531765E+03, 6.563238E+03, 6.594802E+03, 6.626458E+03,
    6.658206E+03, 6.690045E+03, 6.721976E+03, 6.753999E+03, 6.786113E+03, 6.818319E+03,
    6.850617E+03, 6.883007E+03, 6.915488E+03, 6.948061E+03, 6.980726E+03, 7.013482E+03,
    7.046330E+03, 7.079270E+03, 7.112302E+03, 7.145425E+03, 7.178640E+03, 7.211946E+03,
    7.245344E+03, 7.278834E+03, 7.312415E+03, 7.346088E+03, 7.379852E+03, 7.413708E+03,
    7.447655E+03, 7.481694E+03, 7.515824E+03, 7.550045E+03, 7.584358E+03, 7.618763E+03,
    7.653258E+03,
])

#  ---------------------- M = 13, I = 2 ---------------------------
M = 13
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.597389E+01, 1.605971E+01, 1.732986E+01, 2.012387E+01, 2.384501E+01, 2.814934E+01,
    3.284552E+01, 3.781754E+01, 4.299087E+01, 4.831550E+01, 5.375673E+01, 5.928972E+01,
    6.489630E+01, 7.056284E+01, 7.627897E+01, 8.203662E+01, 8.783033E+01, 9.365333E+01,
    9.950433E+01, 1.053783E+02, 1.112735E+02, 1.171880E+02, 1.231211E+02, 1.290729E+02,
    1.350413E+02, 1.410255E+02, 1.470291E+02, 1.530492E+02, 1.590887E+02, 1.651462E+02,
    1.712253E+02, 1.773248E+02, 1.834470E+02, 1.895945E+02, 1.957642E+02, 2.019629E+02,
    2.081875E+02, 2.144411E+02, 2.207268E+02, 2.270436E+02, 2.333928E+02, 2.397777E+02,
    2.461972E+02, 2.526550E+02, 2.591499E+02, 2.656857E+02, 2.722610E+02, 2.788770E+02,
    2.855378E+02, 2.922419E+02, 2.989905E+02, 3.057878E+02, 3.126291E+02, 3.195186E+02,
    3.264576E+02, 3.334441E+02, 3.404823E+02, 3.475737E+02, 3.547126E+02, 3.619069E+02,
    3.691508E+02, 3.764523E+02, 3.838055E+02, 3.912149E+02, 3.986779E+02, 4.061993E+02,
    4.137762E+02, 4.214096E+02, 4.290963E+02, 4.368453E+02, 4.446536E+02, 4.525179E+02,
    4.604431E+02, 4.684258E+02, 4.764670E+02, 4.845716E+02, 4.927317E+02, 5.009570E+02,
    5.092437E+02, 5.175879E+02, 5.259950E+02, 5.344656E+02, 5.429958E+02, 5.515910E+02,
    5.602469E+02, 5.689641E+02, 5.777484E+02, 5.865952E+02, 5.955001E+02, 6.044738E+02,
    6.135065E+02, 6.226095E+02, 6.317725E+02, 6.410014E+02, 6.502913E+02, 6.596482E+02,
    6.690726E+02, 6.785594E+02, 6.881090E+02, 6.977277E+02, 7.074100E+02, 7.171565E+02,
    7.269735E+02, 7.368494E+02, 7.467968E+02, 7.568101E+02, 7.668831E+02, 7.770291E+02,
    7.872422E+02, 7.975161E+02, 8.078577E+02, 8.182674E+02, 8.287457E+02, 8.392861E+02,
    8.498957E+02, 8.605749E+02, 8.713171E+02, 8.821296E+02, 8.930058E+02, 9.039529E+02,
    9.149714E+02, 9.260471E+02, 9.372022E+02, 9.484149E+02, 9.597003E+02, 9.710512E+02,
    9.824755E+02, 9.939582E+02, 1.005515E+03, 1.017146E+03, 1.028836E+03, 1.040601E+03,
    1.052425E+03, 1.064325E+03, 1.076292E+03, 1.088328E+03, 1.100432E+03, 1.112604E+03,
    1.124845E+03, 1.137155E+03, 1.149542E+03, 1.161991E+03, 1.174509E+03, 1.187097E+03,
    1.199746E+03, 1.212474E+03, 1.225273E+03, 1.238133E+03, 1.251065E+03, 1.264076E+03,
    1.277150E+03, 1.290286E+03, 1.303503E+03, 1.316783E+03, 1.330135E+03, 1.343559E+03,
    1.357046E+03, 1.370616E+03, 1.384249E+03, 1.397945E+03, 1.411724E+03, 1.425567E+03,
    1.439474E+03, 1.453465E+03, 1.467520E+03, 1.481639E+03, 1.495832E+03, 1.510100E+03,
    1.524443E+03, 1.538850E+03, 1.553323E+03, 1.567870E+03, 1.582494E+03, 1.597182E+03,
    1.611947E+03, 1.626776E+03,
])

#  ---------------------- M = 13, I = 3 ---------------------------
M = 13
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.398602E+01, 2.529396E+01, 3.190248E+01, 4.103232E+01, 5.180328E+01, 6.382165E+01,
    7.678389E+01, 9.046186E+01, 1.046887E+02, 1.193429E+02, 1.343350E+02, 1.495984E+02,
    1.650827E+02, 1.807493E+02, 1.965682E+02, 2.125155E+02, 2.285724E+02, 2.447236E+02,
    2.609566E+02, 2.772613E+02, 2.936290E+02, 3.100527E+02, 3.265296E+02, 3.430483E+02,
    3.596111E+02, 3.762111E+02, 3.928490E+02, 4.095183E+02, 4.262208E+02, 4.429551E+02,
    4.597196E+02, 4.765132E+02, 4.933399E+02, 5.101943E+02, 5.270811E+02, 5.439947E+02,
    5.609459E+02, 5.779297E+02, 5.949463E+02, 6.120082E+02, 6.290982E+02, 6.462353E+02,
    6.634146E+02, 6.806371E+02, 6.979040E+02, 7.152234E+02, 7.325899E+02, 7.500121E+02,
    7.674917E+02, 7.850226E+02, 8.026143E+02, 8.202684E+02, 8.379869E+02, 8.557801E+02,
    8.736330E+02, 8.915558E+02, 9.095506E+02, 9.276191E+02, 9.457727E+02, 9.639948E+02,
    9.823062E+02, 1.000699E+03, 1.019167E+03, 1.037730E+03, 1.056380E+03, 1.075110E+03,
    1.093943E+03, 1.112858E+03, 1.131880E+03, 1.150989E+03, 1.170197E+03, 1.189507E+03,
    1.208908E+03, 1.228415E+03, 1.248030E+03, 1.267742E+03, 1.287565E+03, 1.307488E+03,
    1.327527E+03, 1.347670E+03, 1.367918E+03, 1.388287E+03, 1.408751E+03, 1.429339E+03,
    1.450040E+03, 1.470854E+03, 1.491784E+03, 1.512831E+03, 1.533996E+03, 1.555282E+03,
    1.576689E+03, 1.598204E+03, 1.619844E+03, 1.641594E+03, 1.663488E+03, 1.685495E+03,
    1.707617E+03, 1.729870E+03, 1.752241E+03, 1.774729E+03, 1.797354E+03, 1.820116E+03,
    1.842984E+03, 1.865991E+03, 1.889123E+03, 1.912380E+03, 1.935764E+03, 1.959275E+03,
    1.982915E+03, 2.006703E+03, 2.030604E+03, 2.054636E+03, 2.078802E+03, 2.103083E+03,
    2.127519E+03, 2.152071E+03, 2.176760E+03, 2.201588E+03, 2.226555E+03, 2.251642E+03,
    2.276871E+03, 2.302222E+03, 2.327716E+03, 2.353353E+03, 2.379116E+03, 2.405003E+03,
    2.431038E+03, 2.457199E+03, 2.483508E+03, 2.509946E+03, 2.536534E+03, 2.563252E+03,
    2.590101E+03, 2.617102E+03, 2.644213E+03, 2.671478E+03, 2.698899E+03, 2.726432E+03,
    2.754121E+03, 2.781922E+03, 2.809883E+03, 2.837979E+03, 2.866213E+03, 2.894608E+03,
    2.923117E+03, 2.951765E+03, 2.980577E+03, 3.009504E+03, 3.038597E+03, 3.067806E+03,
    3.097157E+03, 3.126676E+03, 3.156312E+03, 3.186092E+03, 3.216016E+03, 3.246085E+03,
    3.276300E+03, 3.306660E+03, 3.337166E+03, 3.367793E+03, 3.398568E+03, 3.429490E+03,
    3.460562E+03, 3.491782E+03, 3.523124E+03, 3.554617E+03, 3.586261E+03, 3.618055E+03,
    3.649974E+03, 3.682073E+03, 3.714268E+03, 3.746645E+03, 3.779146E+03, 3.811803E+03,
    3.844614E+03, 3.877551E+03,
])

#  ---------------------- M = 14, I = 1 ---------------------------
M = 14
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    4.000000E+00, 4.626210E+00, 6.976320E+00, 9.595800E+00, 1.226348E+01, 1.494872E+01,
    1.764266E+01, 2.034174E+01, 2.304424E+01, 2.574926E+01, 2.845624E+01, 3.116485E+01,
    3.387487E+01, 3.658615E+01, 3.929857E+01, 4.201207E+01, 4.472657E+01, 4.744205E+01,
    5.015846E+01, 5.287579E+01, 5.559401E+01, 5.831312E+01, 6.103312E+01, 6.375401E+01,
    6.647583E+01, 6.919858E+01, 7.192233E+01, 7.464713E+01, 7.737305E+01, 8.010018E+01,
    8.282863E+01, 8.555853E+01, 8.829002E+01, 9.102328E+01, 9.375848E+01, 9.649584E+01,
    9.923557E+01, 1.019779E+02, 1.047231E+02, 1.074715E+02, 1.102232E+02, 1.129788E+02,
    1.157383E+02, 1.185022E+02, 1.212707E+02, 1.240443E+02, 1.268232E+02, 1.296079E+02,
    1.323986E+02, 1.351957E+02, 1.379995E+02, 1.408105E+02, 1.436289E+02, 1.464552E+02,
    1.492896E+02, 1.521325E+02, 1.549843E+02, 1.578453E+02, 1.607158E+02, 1.635962E+02,
    1.664868E+02, 1.693878E+02, 1.722998E+02, 1.752228E+02, 1.781573E+02, 1.811035E+02,
    1.840617E+02, 1.870323E+02, 1.900154E+02, 1.930114E+02, 1.960205E+02, 1.990429E+02,
    2.020790E+02, 2.051289E+02, 2.081930E+02, 2.112714E+02, 2.143644E+02, 2.174721E+02,
    2.205948E+02, 2.237328E+02, 2.268862E+02, 2.300551E+02, 2.332399E+02, 2.364407E+02,
    2.396577E+02, 2.428910E+02, 2.461408E+02, 2.494073E+02, 2.526907E+02, 2.559911E+02,
    2.593087E+02, 2.626436E+02, 2.659960E+02, 2.693660E+02, 2.727538E+02, 2.761594E+02,
    2.795831E+02, 2.830250E+02, 2.864851E+02, 2.899636E+02, 2.934607E+02, 2.969764E+02,
    3.005109E+02, 3.040643E+02, 3.076367E+02, 3.112281E+02, 3.148388E+02, 3.184687E+02,
    3.221181E+02, 3.257869E+02, 3.294754E+02, 3.331835E+02, 3.369115E+02, 3.406592E+02,
    3.444270E+02, 3.482148E+02, 3.520228E+02, 3.558509E+02, 3.596994E+02, 3.635682E+02,
    3.674576E+02, 3.713674E+02, 3.752979E+02, 3.792490E+02, 3.832209E+02, 3.872136E+02,
    3.912273E+02, 3.952619E+02, 3.993176E+02, 4.033944E+02, 4.074924E+02, 4.116116E+02,
    4.157521E+02, 4.199140E+02, 4.240974E+02, 4.283022E+02, 4.325286E+02, 4.367767E+02,
    4.410464E+02, 4.453379E+02, 4.496512E+02, 4.539863E+02, 4.583434E+02, 4.627224E+02,
    4.671235E+02, 4.715467E+02, 4.759920E+02, 4.804595E+02, 4.849493E+02, 4.894613E+02,
    4.939958E+02, 4.985526E+02, 5.031320E+02, 5.077338E+02, 5.123582E+02, 5.170052E+02,
    5.216749E+02, 5.263673E+02, 5.310825E+02, 5.358205E+02, 5.405813E+02, 5.453651E+02,
    5.501718E+02, 5.550016E+02, 5.598544E+02, 5.647303E+02, 5.696293E+02, 5.745516E+02,
    5.794971E+02, 5.844659E+02, 5.894580E+02, 5.944735E+02, 5.995124E+02, 6.045748E+02,
    6.096607E+02, 6.147701E+02, 6.199032E+02, 6.250599E+02, 6.302403E+02, 6.354444E+02,
    6.406723E+02, 6.459240E+02, 6.511995E+02, 6.564990E+02, 6.618224E+02, 6.671698E+02,
    6.725412E+02, 6.779366E+02, 6.833562E+02, 6.887999E+02, 6.942678E+02, 6.997600E+02,
    7.052764E+02, 7.108171E+02, 7.163821E+02, 7.219715E+02, 7.275854E+02, 7.332237E+02,
    7.388865E+02, 7.445739E+02, 7.502858E+02, 7.560223E+02, 7.617835E+02, 7.675694E+02,
    7.733800E+02, 7.792153E+02, 7.850755E+02, 7.909605E+02, 7.968704E+02, 8.028052E+02,
    8.087649E+02, 8.147496E+02, 8.207593E+02, 8.267940E+02, 8.328539E+02, 8.389388E+02,
    8.450490E+02, 8.511843E+02, 8.573448E+02, 8.635306E+02, 8.697416E+02, 8.759780E+02,
    8.822397E+02, 8.885269E+02, 8.948394E+02, 9.011774E+02, 9.075409E+02, 9.139299E+02,
    9.203444E+02, 9.267845E+02, 9.332502E+02, 9.397416E+02, 9.462586E+02, 9.528014E+02,
    9.593698E+02, 9.659640E+02, 9.725841E+02, 9.792299E+02, 9.859015E+02, 9.925991E+02,
    9.993225E+02, 1.006072E+03, 1.012847E+03, 1.019649E+03, 1.026476E+03, 1.033329E+03,
    1.040209E+03, 1.047114E+03, 1.054046E+03, 1.061004E+03, 1.067988E+03, 1.074998E+03,
    1.082034E+03, 1.089097E+03, 1.096186E+03, 1.103301E+03, 1.110443E+03, 1.117611E+03,
    1.124805E+03, 1.132026E+03, 1.139273E+03, 1.146546E+03, 1.153846E+03, 1.161173E+03,
    1.168525E+03, 1.175905E+03, 1.183311E+03, 1.190743E+03, 1.198203E+03, 1.205688E+03,
    1.213201E+03, 1.220740E+03, 1.228306E+03, 1.235898E+03, 1.243517E+03, 1.251163E+03,
    1.258835E+03, 1.266535E+03, 1.274261E+03, 1.282013E+03, 1.289793E+03, 1.297600E+03,
    1.305433E+03, 1.313293E+03, 1.321180E+03, 1.329094E+03, 1.337035E+03, 1.345003E+03,
    1.352997E+03, 1.361019E+03, 1.369067E+03, 1.377143E+03, 1.385245E+03, 1.393375E+03,
    1.401531E+03, 1.409715E+03, 1.417925E+03, 1.426163E+03, 1.434428E+03, 1.442719E+03,
    1.451038E+03,
])

#  ---------------------- M = 14, I = 2 ---------------------------
M = 14
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    6.000000E+00, 1.005366E+01, 1.753287E+01, 2.515838E+01, 3.281701E+01, 4.048954E+01,
    4.816994E+01, 5.585570E+01, 6.354558E+01, 7.123892E+01, 7.893531E+01, 8.663450E+01,
    9.433634E+01, 1.020407E+02, 1.097475E+02, 1.174568E+02, 1.251685E+02, 1.328827E+02,
    1.405995E+02, 1.483191E+02, 1.560419E+02, 1.637681E+02, 1.714984E+02, 1.792334E+02,
    1.869738E+02, 1.947206E+02, 2.024749E+02, 2.102378E+02, 2.180106E+02, 2.257947E+02,
    2.335917E+02, 2.414030E+02, 2.492305E+02, 2.570757E+02, 2.649405E+02, 2.728267E+02,
    2.807361E+02, 2.886704E+02, 2.966316E+02, 3.046215E+02, 3.126418E+02, 3.206944E+02,
    3.287810E+02, 3.369034E+02, 3.450632E+02, 3.532621E+02, 3.615017E+02, 3.697836E+02,
    3.781092E+02, 3.864802E+02, 3.948978E+02, 4.033635E+02, 4.118787E+02, 4.204445E+02,
    4.290623E+02, 4.377333E+02, 4.464585E+02, 4.552392E+02, 4.640764E+02, 4.729710E+02,
    4.819242E+02, 4.909369E+02, 5.000099E+02, 5.091442E+02, 5.183406E+02, 5.276000E+02,
    5.369230E+02, 5.463105E+02, 5.557632E+02, 5.652818E+02, 5.748670E+02, 5.845194E+02,
    5.942396E+02, 6.040283E+02, 6.138860E+02, 6.238133E+02, 6.338107E+02, 6.438787E+02,
    6.540179E+02, 6.642288E+02, 6.745117E+02, 6.848672E+02, 6.952957E+02, 7.057977E+02,
    7.163734E+02, 7.270235E+02, 7.377481E+02, 7.485477E+02, 7.594228E+02, 7.703735E+02,
    7.814004E+02, 7.925036E+02, 8.036835E+02, 8.149406E+02, 8.262749E+02, 8.376870E+02,
    8.491770E+02, 8.607452E+02, 8.723920E+02, 8.841176E+02, 8.959222E+02, 9.078062E+02,
    9.197698E+02, 9.318132E+02, 9.439367E+02, 9.561405E+02, 9.684249E+02, 9.807901E+02,
    9.932363E+02, 1.005764E+03, 1.018373E+03, 1.031063E+03, 1.043836E+03, 1.056690E+03,
    1.069627E+03, 1.082647E+03, 1.095749E+03, 1.108934E+03, 1.122202E+03, 1.135553E+03,
    1.148988E+03, 1.162506E+03, 1.176109E+03, 1.189795E+03, 1.203566E+03, 1.217421E+03,
    1.231360E+03, 1.245384E+03, 1.259493E+03, 1.273688E+03, 1.287967E+03, 1.302332E+03,
    1.316782E+03, 1.331318E+03, 1.345940E+03, 1.360648E+03, 1.375442E+03, 1.390322E+03,
    1.405289E+03, 1.420342E+03, 1.435482E+03, 1.450709E+03, 1.466023E+03, 1.481425E+03,
    1.496913E+03, 1.512490E+03, 1.528153E+03, 1.543905E+03, 1.559744E+03, 1.575672E+03,
    1.591687E+03, 1.607791E+03, 1.623983E+03, 1.640264E+03, 1.656633E+03, 1.673092E+03,
    1.689639E+03, 1.706275E+03, 1.723000E+03, 1.739815E+03, 1.756718E+03, 1.773712E+03,
    1.790795E+03, 1.807968E+03, 1.825230E+03, 1.842583E+03, 1.860025E+03, 1.877558E+03,
    1.895181E+03, 1.912894E+03, 1.930698E+03, 1.948593E+03, 1.966578E+03, 1.984654E+03,
    2.002820E+03, 2.021078E+03, 2.039427E+03, 2.057867E+03, 2.076398E+03, 2.095020E+03,
    2.113734E+03, 2.132540E+03, 2.151437E+03, 2.170425E+03, 2.189506E+03, 2.208678E+03,
    2.227942E+03, 2.247298E+03, 2.266747E+03, 2.286287E+03, 2.305920E+03, 2.325645E+03,
    2.345462E+03, 2.365371E+03, 2.385374E+03, 2.405468E+03, 2.425656E+03, 2.445936E+03,
    2.466309E+03, 2.486774E+03, 2.507333E+03, 2.527984E+03, 2.548729E+03, 2.569566E+03,
    2.590496E+03, 2.611520E+03, 2.632637E+03, 2.653847E+03, 2.675150E+03, 2.696547E+03,
    2.718037E+03, 2.739621E+03, 2.761298E+03, 2.783068E+03, 2.804932E+03, 2.826890E+03,
    2.848941E+03, 2.871086E+03, 2.893324E+03, 2.915656E+03, 2.938082E+03, 2.960602E+03,
    2.983215E+03, 3.005923E+03, 3.028724E+03, 3.051619E+03, 3.074608E+03, 3.097690E+03,
    3.120867E+03, 3.144138E+03, 3.167502E+03, 3.190961E+03, 3.214513E+03, 3.238160E+03,
    3.261900E+03, 3.285735E+03, 3.309663E+03, 3.333686E+03, 3.357802E+03, 3.382013E+03,
    3.406318E+03, 3.430716E+03, 3.455209E+03, 3.479796E+03, 3.504477E+03, 3.529252E+03,
    3.554120E+03, 3.579083E+03, 3.604140E+03, 3.629291E+03, 3.654536E+03, 3.679875E+03,
    3.705308E+03, 3.730835E+03, 3.756455E+03, 3.782170E+03, 3.807979E+03, 3.833881E+03,
    3.859877E+03, 3.885968E+03, 3.912152E+03, 3.938429E+03, 3.964801E+03, 3.991266E+03,
    4.017826E+03, 4.044478E+03, 4.071225E+03, 4.098065E+03, 4.124999E+03, 4.152026E+03,
    4.179147E+03, 4.206362E+03, 4.233670E+03, 4.261071E+03, 4.288566E+03, 4.316154E+03,
    4.343836E+03, 4.371610E+03, 4.399478E+03, 4.427440E+03, 4.455494E+03, 4.483642E+03,
    4.511883E+03, 4.540216E+03, 4.568643E+03, 4.597163E+03, 4.625776E+03, 4.654481E+03,
    4.683279E+03, 4.712171E+03, 4.741154E+03, 4.770231E+03, 4.799400E+03, 4.828661E+03,
    4.858015E+03, 4.887462E+03, 4.917001E+03, 4.946632E+03, 4.976356E+03, 5.006171E+03,
    5.036079E+03,
])

#  ---------------------- M = 15, I = 1 ---------------------------
M = 15
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    8.000000E+00, 1.379368E+01, 2.419193E+01, 3.477493E+01, 4.540023E+01, 5.604342E+01,
    6.669682E+01, 7.735721E+01, 8.802303E+01, 9.869341E+01, 1.093679E+02, 1.200460E+02,
    1.307278E+02, 1.414129E+02, 1.521013E+02, 1.627930E+02, 1.734880E+02, 1.841863E+02,
    1.948883E+02, 2.055941E+02, 2.163042E+02, 2.270192E+02, 2.377399E+02, 2.484672E+02,
    2.592022E+02, 2.699463E+02, 2.807009E+02, 2.914679E+02, 3.022490E+02, 3.130462E+02,
    3.238617E+02, 3.346978E+02, 3.455568E+02, 3.564412E+02, 3.673535E+02, 3.782961E+02,
    3.892717E+02, 4.002829E+02, 4.113322E+02, 4.224223E+02, 4.335555E+02, 4.447346E+02,
    4.559619E+02, 4.672398E+02, 4.785708E+02, 4.899571E+02, 5.014011E+02, 5.129049E+02,
    5.244706E+02, 5.361002E+02, 5.477959E+02, 5.595596E+02, 5.713930E+02, 5.832980E+02,
    5.952764E+02, 6.073297E+02, 6.194597E+02, 6.316679E+02, 6.439557E+02, 6.563247E+02,
    6.687761E+02, 6.813114E+02, 6.939317E+02, 7.066385E+02, 7.194327E+02, 7.323156E+02,
    7.452883E+02, 7.583518E+02, 7.715072E+02, 7.847553E+02, 7.980973E+02, 8.115339E+02,
    8.250660E+02, 8.386946E+02, 8.524203E+02, 8.662441E+02, 8.801667E+02, 8.941887E+02,
    9.083110E+02, 9.225341E+02, 9.368588E+02, 9.512857E+02, 9.658154E+02, 9.804486E+02,
    9.951858E+02, 1.010028E+03, 1.024974E+03, 1.040027E+03, 1.055186E+03, 1.070451E+03,
    1.085824E+03, 1.101304E+03, 1.116893E+03, 1.132590E+03, 1.148396E+03, 1.164312E+03,
    1.180338E+03, 1.196474E+03, 1.212720E+03, 1.229078E+03, 1.245547E+03, 1.262129E+03,
    1.278822E+03, 1.295628E+03, 1.312548E+03, 1.329580E+03, 1.346727E+03, 1.363987E+03,
    1.381362E+03, 1.398851E+03, 1.416456E+03, 1.434176E+03, 1.452011E+03, 1.469963E+03,
    1.488031E+03, 1.506216E+03, 1.524517E+03, 1.542936E+03, 1.561472E+03, 1.580126E+03,
    1.598899E+03, 1.617789E+03, 1.636798E+03, 1.655927E+03, 1.675174E+03, 1.694541E+03,
    1.714027E+03, 1.733634E+03, 1.753361E+03, 1.773209E+03, 1.793177E+03, 1.813266E+03,
    1.833477E+03, 1.853809E+03, 1.874263E+03, 1.894840E+03, 1.915538E+03, 1.936359E+03,
    1.957303E+03, 1.978370E+03, 1.999561E+03, 2.020875E+03, 2.042312E+03, 2.063874E+03,
    2.085560E+03, 2.107371E+03, 2.129306E+03, 2.151366E+03, 2.173551E+03, 2.195862E+03,
    2.218299E+03, 2.240861E+03, 2.263549E+03, 2.286364E+03, 2.309305E+03, 2.332374E+03,
    2.355569E+03, 2.378891E+03, 2.402340E+03, 2.425918E+03, 2.449623E+03, 2.473456E+03,
    2.497418E+03, 2.521507E+03, 2.545726E+03, 2.570074E+03, 2.594550E+03, 2.619156E+03,
    2.643891E+03, 2.668756E+03, 2.693751E+03, 2.718877E+03, 2.744132E+03, 2.769518E+03,
    2.795034E+03, 2.820682E+03, 2.846460E+03, 2.872370E+03, 2.898411E+03, 2.924584E+03,
    2.950888E+03, 2.977325E+03, 3.003893E+03, 3.030594E+03, 3.057428E+03, 3.084394E+03,
    3.111494E+03, 3.138726E+03, 3.166091E+03, 3.193590E+03, 3.221223E+03, 3.248989E+03,
    3.276889E+03, 3.304923E+03, 3.333091E+03, 3.361394E+03, 3.389831E+03, 3.418403E+03,
    3.447110E+03, 3.475952E+03, 3.504929E+03, 3.534041E+03, 3.563288E+03, 3.592672E+03,
    3.622191E+03, 3.651846E+03, 3.681637E+03, 3.711564E+03, 3.741627E+03, 3.771827E+03,
    3.802164E+03, 3.832637E+03, 3.863247E+03, 3.893994E+03, 3.924878E+03, 3.955899E+03,
    3.987057E+03, 4.018353E+03, 4.049787E+03, 4.081358E+03, 4.113067E+03, 4.144914E+03,
    4.176899E+03, 4.209023E+03, 4.241284E+03, 4.273684E+03, 4.306222E+03, 4.338898E+03,
    4.371714E+03, 4.404668E+03, 4.437761E+03, 4.470993E+03, 4.504364E+03, 4.537874E+03,
    4.571523E+03, 4.605311E+03, 4.639239E+03, 4.673306E+03, 4.707513E+03, 4.741860E+03,
    4.776346E+03, 4.810971E+03, 4.845737E+03, 4.880642E+03, 4.915688E+03, 4.950873E+03,
    4.986199E+03, 5.021664E+03, 5.057270E+03, 5.093016E+03, 5.128903E+03, 5.164930E+03,
    5.201097E+03, 5.237404E+03, 5.273852E+03, 5.310441E+03, 5.347170E+03, 5.384040E+03,
    5.421050E+03, 5.458201E+03, 5.495493E+03, 5.532926E+03, 5.570499E+03, 5.608213E+03,
    5.646068E+03, 5.684064E+03, 5.722201E+03, 5.760479E+03, 5.798897E+03, 5.837457E+03,
    5.876157E+03, 5.914998E+03, 5.953981E+03, 5.993104E+03, 6.032368E+03, 6.071773E+03,
    6.111319E+03, 6.151006E+03, 6.190834E+03, 6.230803E+03, 6.270913E+03, 6.311164E+03,
    6.351555E+03, 6.392088E+03, 6.432761E+03, 6.473575E+03, 6.514531E+03, 6.555626E+03,
    6.596863E+03, 6.638241E+03, 6.679759E+03, 6.721418E+03, 6.763217E+03, 6.805157E+03,
    6.847238E+03, 6.889459E+03, 6.931821E+03, 6.974323E+03, 7.016966E+03, 7.059749E+03,
    7.102672E+03,
])

#  ---------------------- M = 15, I = 2 ---------------------------
M = 15
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    8.000000E+00, 1.380884E+01, 2.422364E+01, 3.482280E+01, 4.546420E+01, 5.612347E+01,
    6.679294E+01, 7.746940E+01, 8.815129E+01, 9.883774E+01, 1.095283E+02, 1.202225E+02,
    1.309203E+02, 1.416215E+02, 1.523260E+02, 1.630338E+02, 1.737449E+02, 1.844594E+02,
    1.951774E+02, 2.058993E+02, 2.166256E+02, 2.273568E+02, 2.380936E+02, 2.488371E+02,
    2.595884E+02, 2.703487E+02, 2.811197E+02, 2.919031E+02, 3.027006E+02, 3.135144E+02,
    3.243465E+02, 3.351994E+02, 3.460752E+02, 3.569765E+02, 3.679058E+02, 3.788657E+02,
    3.898586E+02, 4.008872E+02, 4.119542E+02, 4.230620E+02, 4.342132E+02, 4.454103E+02,
    4.566559E+02, 4.679523E+02, 4.793019E+02, 4.907071E+02, 5.021701E+02, 5.136932E+02,
    5.252783E+02, 5.369277E+02, 5.486432E+02, 5.604270E+02, 5.722807E+02, 5.842062E+02,
    5.962054E+02, 6.082797E+02, 6.204309E+02, 6.326605E+02, 6.449701E+02, 6.573609E+02,
    6.698345E+02, 6.823922E+02, 6.950352E+02, 7.077648E+02, 7.205821E+02, 7.334884E+02,
    7.464847E+02, 7.595720E+02, 7.727515E+02, 7.860240E+02, 7.993905E+02, 8.128519E+02,
    8.264092E+02, 8.400631E+02, 8.538144E+02, 8.676641E+02, 8.816127E+02, 8.956611E+02,
    9.098099E+02, 9.240599E+02, 9.384118E+02, 9.528660E+02, 9.674234E+02, 9.820845E+02,
    9.968498E+02, 1.011720E+03, 1.026696E+03, 1.041777E+03, 1.056965E+03, 1.072260E+03,
    1.087662E+03, 1.103173E+03, 1.118791E+03, 1.134519E+03, 1.150356E+03, 1.166303E+03,
    1.182360E+03, 1.198527E+03, 1.214806E+03, 1.231195E+03, 1.247697E+03, 1.264311E+03,
    1.281037E+03, 1.297877E+03, 1.314829E+03, 1.331896E+03, 1.349076E+03, 1.366370E+03,
    1.383780E+03, 1.401304E+03, 1.418943E+03, 1.436699E+03, 1.454570E+03, 1.472557E+03,
    1.490661E+03, 1.508882E+03, 1.527220E+03, 1.545676E+03, 1.564249E+03, 1.582941E+03,
    1.601751E+03, 1.620679E+03, 1.639727E+03, 1.658894E+03, 1.678180E+03, 1.697586E+03,
    1.717112E+03, 1.736758E+03, 1.756525E+03, 1.776412E+03, 1.796421E+03, 1.816551E+03,
    1.836803E+03, 1.857176E+03, 1.877672E+03, 1.898290E+03, 1.919031E+03, 1.939894E+03,
    1.960881E+03, 1.981991E+03, 2.003225E+03, 2.024582E+03, 2.046063E+03, 2.067669E+03,
    2.089399E+03, 2.111255E+03, 2.133235E+03, 2.155340E+03, 2.177571E+03, 2.199928E+03,
    2.222410E+03, 2.245019E+03, 2.267754E+03, 2.290615E+03, 2.313604E+03, 2.336719E+03,
    2.359962E+03, 2.383332E+03, 2.406830E+03, 2.430456E+03, 2.454210E+03, 2.478092E+03,
    2.502103E+03, 2.526243E+03, 2.550512E+03, 2.574909E+03, 2.599436E+03, 2.624093E+03,
    2.648880E+03, 2.673796E+03, 2.698843E+03, 2.724020E+03, 2.749328E+03, 2.774766E+03,
    2.800335E+03, 2.826036E+03, 2.851868E+03, 2.877831E+03, 2.903926E+03, 2.930154E+03,
    2.956513E+03, 2.983004E+03, 3.009628E+03, 3.036385E+03, 3.063274E+03, 3.090296E+03,
    3.117452E+03, 3.144741E+03, 3.172163E+03, 3.199719E+03, 3.227409E+03, 3.255233E+03,
    3.283191E+03, 3.311284E+03, 3.339510E+03, 3.367872E+03, 3.396369E+03, 3.425000E+03,
    3.453767E+03, 3.482669E+03, 3.511706E+03, 3.540879E+03, 3.570188E+03, 3.599632E+03,
    3.629213E+03, 3.658929E+03, 3.688783E+03, 3.718772E+03, 3.748898E+03, 3.779161E+03,
    3.809561E+03, 3.840098E+03, 3.870771E+03, 3.901583E+03, 3.932531E+03, 3.963617E+03,
    3.994841E+03, 4.026202E+03, 4.057701E+03, 4.089338E+03, 4.121113E+03, 4.153027E+03,
    4.185079E+03, 4.217269E+03, 4.249598E+03, 4.282065E+03, 4.314671E+03, 4.347416E+03,
    4.380300E+03, 4.413323E+03, 4.446485E+03, 4.479786E+03, 4.513226E+03, 4.546806E+03,
    4.580526E+03, 4.614385E+03, 4.648383E+03, 4.682522E+03, 4.716800E+03, 4.751218E+03,
    4.785776E+03, 4.820474E+03, 4.855312E+03, 4.890290E+03, 4.925409E+03, 4.960667E+03,
    4.996067E+03, 5.031606E+03, 5.067286E+03, 5.103107E+03, 5.139068E+03, 5.175170E+03,
    5.211412E+03, 5.247795E+03, 5.284319E+03, 5.320984E+03, 5.357790E+03, 5.394736E+03,
    5.431824E+03, 5.469052E+03, 5.506421E+03, 5.543932E+03, 5.581583E+03, 5.619376E+03,
    5.657310E+03, 5.695385E+03, 5.733601E+03, 5.771958E+03, 5.810456E+03, 5.849096E+03,
    5.887876E+03, 5.926798E+03, 5.965862E+03, 6.005066E+03, 6.044412E+03, 6.083898E+03,
    6.123527E+03, 6.163296E+03, 6.203206E+03, 6.243258E+03, 6.283451E+03, 6.323785E+03,
    6.364261E+03, 6.404877E+03, 6.445635E+03, 6.486534E+03, 6.527574E+03, 6.568755E+03,
    6.610077E+03, 6.651540E+03, 6.693144E+03, 6.734889E+03, 6.776775E+03, 6.818802E+03,
    6.860969E+03, 6.903278E+03, 6.945727E+03, 6.988317E+03, 7.031048E+03, 7.073919E+03,
    7.116931E+03,
])

#  ---------------------- M = 15, I = 3 ---------------------------
M = 15
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.200001E+01, 3.527472E+01, 6.604716E+01, 9.694745E+01, 1.278841E+02, 1.588401E+02,
    1.898098E+02, 2.207909E+02, 2.517823E+02, 2.827832E+02, 3.137934E+02, 3.448129E+02,
    3.758418E+02, 4.068811E+02, 4.379322E+02, 4.689977E+02, 5.000812E+02, 5.311880E+02,
    5.623246E+02, 5.934988E+02, 6.247200E+02, 6.559986E+02, 6.873464E+02, 7.187757E+02,
    7.502996E+02, 7.819318E+02, 8.136862E+02, 8.455769E+02, 8.776178E+02, 9.098230E+02,
    9.422062E+02, 9.747808E+02, 1.007560E+03, 1.040555E+03, 1.073780E+03, 1.107246E+03,
    1.140963E+03, 1.174943E+03, 1.209195E+03, 1.243729E+03, 1.278555E+03, 1.313680E+03,
    1.349113E+03, 1.384862E+03, 1.420935E+03, 1.457337E+03, 1.494076E+03, 1.531157E+03,
    1.568588E+03, 1.606372E+03, 1.644516E+03, 1.683025E+03, 1.721902E+03, 1.761153E+03,
    1.800782E+03, 1.840792E+03, 1.881188E+03, 1.921974E+03, 1.963151E+03, 2.004725E+03,
    2.046697E+03, 2.089072E+03, 2.131851E+03, 2.175038E+03, 2.218634E+03, 2.262643E+03,
    2.307067E+03, 2.351908E+03, 2.397169E+03, 2.442851E+03, 2.488956E+03, 2.535486E+03,
    2.582444E+03, 2.629831E+03, 2.677648E+03, 2.725898E+03, 2.774582E+03, 2.823702E+03,
    2.873259E+03, 2.923254E+03, 2.973690E+03, 3.024567E+03, 3.075888E+03, 3.127652E+03,
    3.179862E+03, 3.232519E+03, 3.285625E+03, 3.339179E+03, 3.393184E+03, 3.447641E+03,
    3.502550E+03, 3.557914E+03, 3.613733E+03, 3.670008E+03, 3.726740E+03, 3.783930E+03,
    3.841580E+03, 3.899690E+03, 3.958261E+03, 4.017295E+03, 4.076791E+03, 4.136753E+03,
    4.197179E+03, 4.258071E+03, 4.319431E+03, 4.381258E+03, 4.443554E+03, 4.506320E+03,
    4.569557E+03, 4.633264E+03, 4.697444E+03, 4.762098E+03, 4.827225E+03, 4.892827E+03,
    4.958904E+03, 5.025457E+03, 5.092488E+03, 5.159997E+03, 5.227984E+03, 5.296451E+03,
    5.365397E+03, 5.434825E+03, 5.504734E+03, 5.575125E+03, 5.646000E+03, 5.717358E+03,
    5.789200E+03, 5.861527E+03, 5.934340E+03, 6.007640E+03, 6.081426E+03, 6.155700E+03,
    6.230462E+03, 6.305713E+03, 6.381453E+03, 6.457684E+03, 6.534405E+03, 6.611617E+03,
    6.689320E+03, 6.767516E+03, 6.846205E+03, 6.925387E+03, 7.005063E+03, 7.085233E+03,
    7.165898E+03, 7.247058E+03, 7.328713E+03, 7.410865E+03, 7.493514E+03, 7.576660E+03,
    7.660303E+03, 7.744444E+03, 7.829083E+03, 7.914221E+03, 7.999858E+03, 8.085994E+03,
    8.172630E+03, 8.259766E+03, 8.347402E+03, 8.435540E+03, 8.524178E+03, 8.613317E+03,
    8.702958E+03, 8.793101E+03, 8.883746E+03, 8.974893E+03, 9.066542E+03, 9.158695E+03,
    9.251350E+03, 9.344509E+03, 9.438170E+03, 9.532336E+03, 9.627005E+03, 9.722177E+03,
    9.817854E+03, 9.914034E+03, 1.001072E+04, 1.010791E+04, 1.020560E+04, 1.030380E+04,
    1.040250E+04, 1.050171E+04, 1.060142E+04, 1.070163E+04, 1.080235E+04, 1.090358E+04,
    1.100530E+04, 1.110754E+04, 1.121027E+04, 1.131352E+04, 1.141726E+04, 1.152151E+04,
    1.162627E+04, 1.173152E+04, 1.183728E+04, 1.194355E+04, 1.205032E+04, 1.215759E+04,
    1.226537E+04, 1.237364E+04, 1.248242E+04, 1.259171E+04, 1.270149E+04, 1.281178E+04,
    1.292257E+04, 1.303386E+04, 1.314566E+04, 1.325795E+04, 1.337075E+04, 1.348404E+04,
    1.359784E+04, 1.371213E+04, 1.382693E+04, 1.394222E+04, 1.405802E+04, 1.417431E+04,
    1.429110E+04, 1.440838E+04, 1.452617E+04, 1.464445E+04, 1.476323E+04, 1.488250E+04,
    1.500227E+04, 1.512254E+04, 1.524329E+04, 1.536455E+04, 1.548629E+04, 1.560853E+04,
    1.573127E+04, 1.585449E+04, 1.597821E+04, 1.610241E+04, 1.622711E+04, 1.635230E+04,
    1.647798E+04, 1.660414E+04, 1.673080E+04, 1.685794E+04, 1.698556E+04, 1.711368E+04,
    1.724228E+04, 1.737136E+04, 1.750093E+04, 1.763099E+04, 1.776152E+04, 1.789254E+04,
    1.802404E+04, 1.815602E+04, 1.828848E+04, 1.842142E+04, 1.855484E+04, 1.868874E+04,
    1.882311E+04, 1.895796E+04, 1.909329E+04, 1.922909E+04, 1.936536E+04, 1.950211E+04,
    1.963933E+04, 1.977703E+04, 1.991519E+04, 2.005382E+04, 2.019293E+04, 2.033250E+04,
    2.047254E+04, 2.061304E+04, 2.075401E+04, 2.089545E+04, 2.103735E+04, 2.117971E+04,
    2.132254E+04, 2.146582E+04, 2.160957E+04, 2.175378E+04, 2.189844E+04, 2.204356E+04,
    2.218914E+04, 2.233518E+04, 2.248166E+04, 2.262861E+04, 2.277600E+04, 2.292385E+04,
    2.307215E+04, 2.322090E+04, 2.337009E+04, 2.351974E+04, 2.366983E+04, 2.382036E+04,
    2.397134E+04, 2.412277E+04, 2.427464E+04, 2.442694E+04, 2.457969E+04, 2.473288E+04,
    2.488651E+04, 2.504057E+04, 2.519507E+04, 2.535001E+04, 2.550537E+04, 2.566118E+04,
    2.581741E+04,
])

#  ---------------------- M = 15, I = 4 ---------------------------
M = 15
I = 4
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.200001E+01, 3.536447E+01, 6.622832E+01, 9.721965E+01, 1.282473E+02, 1.592942E+02,
    1.903549E+02, 2.214270E+02, 2.525093E+02, 2.836013E+02, 3.147026E+02, 3.458131E+02,
    3.769332E+02, 4.080636E+02, 4.392059E+02, 4.703627E+02, 5.015378E+02, 5.327363E+02,
    5.639649E+02, 5.952315E+02, 6.265456E+02, 6.579177E+02, 6.893595E+02, 7.208836E+02,
    7.525032E+02, 7.842320E+02, 8.160840E+02, 8.480734E+02, 8.802143E+02, 9.125207E+02,
    9.450063E+02, 9.776848E+02, 1.010569E+03, 1.043672E+03, 1.077005E+03, 1.110580E+03,
    1.144409E+03, 1.178502E+03, 1.212869E+03, 1.247520E+03, 1.282464E+03, 1.317709E+03,
    1.353264E+03, 1.389137E+03, 1.425335E+03, 1.461864E+03, 1.498732E+03, 1.535944E+03,
    1.573507E+03, 1.611425E+03, 1.649705E+03, 1.688351E+03, 1.727367E+03, 1.766759E+03,
    1.806529E+03, 1.846683E+03, 1.887223E+03, 1.928154E+03, 1.969478E+03, 2.011199E+03,
    2.053320E+03, 2.095842E+03, 2.138770E+03, 2.182105E+03, 2.225850E+03, 2.270007E+03,
    2.314578E+03, 2.359564E+03, 2.404969E+03, 2.450792E+03, 2.497037E+03, 2.543704E+03,
    2.590795E+03, 2.638311E+03, 2.686254E+03, 2.734623E+03, 2.783422E+03, 2.832649E+03,
    2.882307E+03, 2.932396E+03, 2.982916E+03, 3.033868E+03, 3.085254E+03, 3.137072E+03,
    3.189325E+03, 3.242011E+03, 3.295131E+03, 3.348686E+03, 3.402676E+03, 3.457100E+03,
    3.511959E+03, 3.567253E+03, 3.622981E+03, 3.679143E+03, 3.735740E+03, 3.792771E+03,
    3.850235E+03, 3.908133E+03, 3.966464E+03, 4.025226E+03, 4.084421E+03, 4.144048E+03,
    4.204105E+03, 4.264592E+03, 4.325508E+03, 4.386854E+03, 4.448627E+03, 4.510828E+03,
    4.573454E+03, 4.636507E+03, 4.699984E+03, 4.763884E+03, 4.828207E+03, 4.892952E+03,
    4.958117E+03, 5.023701E+03, 5.089704E+03, 5.156124E+03, 5.222959E+03, 5.290210E+03,
    5.357874E+03, 5.425950E+03, 5.494437E+03, 5.563333E+03, 5.632638E+03, 5.702349E+03,
    5.772466E+03, 5.842987E+03, 5.913910E+03, 5.985234E+03, 6.056957E+03, 6.129079E+03,
    6.201596E+03, 6.274509E+03, 6.347814E+03, 6.421511E+03, 6.495598E+03, 6.570073E+03,
    6.644935E+03, 6.720182E+03, 6.795811E+03, 6.871822E+03, 6.948213E+03, 7.024981E+03,
    7.102125E+03, 7.179643E+03, 7.257534E+03, 7.335796E+03, 7.414426E+03, 7.493422E+03,
    7.572784E+03, 7.652509E+03, 7.732594E+03, 7.813039E+03, 7.893841E+03, 7.974999E+03,
    8.056509E+03, 8.138371E+03, 8.220583E+03, 8.303141E+03, 8.386045E+03, 8.469292E+03,
    8.552881E+03, 8.636808E+03, 8.721073E+03, 8.805673E+03, 8.890606E+03, 8.975870E+03,
    9.061463E+03, 9.147383E+03, 9.233628E+03, 9.320195E+03, 9.407082E+03, 9.494288E+03,
    9.581810E+03, 9.669647E+03, 9.757795E+03, 9.846253E+03, 9.935019E+03, 1.002409E+04,
    1.011347E+04, 1.020314E+04, 1.029312E+04, 1.038339E+04, 1.047396E+04, 1.056482E+04,
    1.065597E+04, 1.074740E+04, 1.083913E+04, 1.093114E+04, 1.102343E+04, 1.111600E+04,
    1.120884E+04, 1.130197E+04, 1.139536E+04, 1.148903E+04, 1.158296E+04, 1.167716E+04,
    1.177163E+04, 1.186635E+04, 1.196134E+04, 1.205658E+04, 1.215208E+04, 1.224784E+04,
    1.234384E+04, 1.244010E+04, 1.253660E+04, 1.263334E+04, 1.273033E+04, 1.282756E+04,
    1.292502E+04, 1.302272E+04, 1.312066E+04, 1.321883E+04, 1.331722E+04, 1.341585E+04,
    1.351469E+04, 1.361377E+04, 1.371306E+04, 1.381257E+04, 1.391230E+04, 1.401224E+04,
    1.411240E+04, 1.421276E+04, 1.431333E+04, 1.441411E+04, 1.451509E+04, 1.461628E+04,
    1.471766E+04, 1.481925E+04, 1.492102E+04, 1.502299E+04, 1.512516E+04, 1.522751E+04,
    1.533005E+04, 1.543277E+04, 1.553568E+04, 1.563877E+04, 1.574203E+04, 1.584548E+04,
    1.594910E+04, 1.605289E+04, 1.615685E+04, 1.626099E+04, 1.636528E+04, 1.646975E+04,
    1.657438E+04, 1.667916E+04, 1.678411E+04, 1.688922E+04, 1.699448E+04, 1.709989E+04,
    1.720545E+04, 1.731117E+04, 1.741703E+04, 1.752304E+04, 1.762919E+04, 1.773548E+04,
    1.784191E+04, 1.794848E+04, 1.805519E+04, 1.816203E+04, 1.826900E+04, 1.837611E+04,
    1.848334E+04, 1.859070E+04, 1.869818E+04, 1.880579E+04, 1.891352E+04, 1.902137E+04,
    1.912934E+04, 1.923742E+04, 1.934562E+04, 1.945393E+04, 1.956235E+04, 1.967088E+04,
    1.977952E+04, 1.988826E+04, 1.999711E+04, 2.010606E+04, 2.021511E+04, 2.032426E+04,
    2.043350E+04, 2.054284E+04, 2.065228E+04, 2.076181E+04, 2.087143E+04, 2.098113E+04,
    2.109093E+04, 2.120081E+04, 2.131078E+04, 2.142082E+04, 2.153095E+04, 2.164116E+04,
    2.175144E+04, 2.186181E+04, 2.197224E+04, 2.208275E+04, 2.219333E+04, 2.230399E+04,
    2.241471E+04,
])

#  ---------------------- M = 16, I = 1 ---------------------------
M = 16
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    8.000000E+00, 1.634928E+01, 2.947690E+01, 4.274338E+01, 5.604397E+01, 6.935994E+01,
    8.268533E+01, 9.601763E+01, 1.093556E+02, 1.226986E+02, 1.360462E+02, 1.493982E+02,
    1.627544E+02, 1.761147E+02, 1.894792E+02, 2.028480E+02, 2.162214E+02, 2.295997E+02,
    2.429839E+02, 2.563748E+02, 2.697738E+02, 2.831826E+02, 2.966031E+02, 3.100376E+02,
    3.234889E+02, 3.369598E+02, 3.504537E+02, 3.639739E+02, 3.775240E+02, 3.911080E+02,
    4.047298E+02, 4.183933E+02, 4.321026E+02, 4.458619E+02, 4.596752E+02, 4.735466E+02,
    4.874801E+02, 5.014796E+02, 5.155490E+02, 5.296922E+02, 5.439127E+02, 5.582142E+02,
    5.726001E+02, 5.870738E+02, 6.016384E+02, 6.162971E+02, 6.310528E+02, 6.459085E+02,
    6.608669E+02, 6.759306E+02, 6.911022E+02, 7.063841E+02, 7.217786E+02, 7.372879E+02,
    7.529142E+02, 7.686595E+02, 7.845258E+02, 8.005148E+02, 8.166285E+02, 8.328685E+02,
    8.492364E+02, 8.657339E+02, 8.823623E+02, 8.991233E+02, 9.160180E+02, 9.330480E+02,
    9.502144E+02, 9.675184E+02, 9.849613E+02, 1.002544E+03, 1.020268E+03, 1.038134E+03,
    1.056143E+03, 1.074296E+03, 1.092594E+03, 1.111039E+03, 1.129630E+03, 1.148368E+03,
    1.167255E+03, 1.186292E+03, 1.205478E+03, 1.224815E+03, 1.244304E+03, 1.263945E+03,
    1.283738E+03, 1.303686E+03, 1.323787E+03, 1.344044E+03, 1.364456E+03, 1.385024E+03,
    1.405748E+03, 1.426630E+03, 1.447670E+03, 1.468868E+03, 1.490225E+03, 1.511742E+03,
    1.533418E+03, 1.555255E+03, 1.577253E+03, 1.599413E+03, 1.621734E+03, 1.644218E+03,
    1.666865E+03, 1.689675E+03, 1.712649E+03, 1.735787E+03, 1.759090E+03, 1.782559E+03,
    1.806192E+03, 1.829993E+03, 1.853959E+03, 1.878093E+03, 1.902393E+03, 1.926862E+03,
    1.951499E+03, 1.976304E+03, 2.001279E+03, 2.026423E+03, 2.051737E+03, 2.077221E+03,
    2.102875E+03, 2.128701E+03, 2.154698E+03, 2.180867E+03, 2.207208E+03, 2.233722E+03,
    2.260408E+03, 2.287269E+03, 2.314302E+03, 2.341510E+03, 2.368893E+03, 2.396450E+03,
    2.424183E+03, 2.452091E+03, 2.480175E+03, 2.508436E+03, 2.536873E+03, 2.565488E+03,
    2.594280E+03, 2.623250E+03, 2.652398E+03, 2.681725E+03, 2.711231E+03, 2.740916E+03,
    2.770781E+03, 2.800826E+03, 2.831052E+03, 2.861458E+03, 2.892045E+03, 2.922814E+03,
    2.953765E+03, 2.984899E+03, 3.016214E+03, 3.047713E+03, 3.079395E+03, 3.111261E+03,
    3.143311E+03, 3.175545E+03, 3.207964E+03, 3.240568E+03, 3.273358E+03, 3.306334E+03,
    3.339495E+03, 3.372843E+03, 3.406378E+03, 3.440101E+03, 3.474010E+03, 3.508108E+03,
    3.542394E+03, 3.576869E+03, 3.611532E+03, 3.646385E+03, 3.681428E+03, 3.716660E+03,
    3.752083E+03, 3.787696E+03, 3.823500E+03, 3.859496E+03, 3.895683E+03, 3.932063E+03,
    3.968634E+03, 4.005398E+03, 4.042355E+03, 4.079506E+03, 4.116850E+03, 4.154388E+03,
    4.192120E+03, 4.230046E+03, 4.268168E+03, 4.306485E+03, 4.344997E+03, 4.383705E+03,
    4.422609E+03, 4.461709E+03, 4.501007E+03, 4.540501E+03, 4.580192E+03, 4.620081E+03,
    4.660168E+03, 4.700454E+03, 4.740937E+03, 4.781620E+03, 4.822501E+03, 4.863582E+03,
    4.904862E+03, 4.946342E+03, 4.988023E+03, 5.029903E+03, 5.071985E+03, 5.114267E+03,
    5.156751E+03, 5.199436E+03, 5.242323E+03, 5.285411E+03, 5.328702E+03, 5.372196E+03,
    5.415892E+03, 5.459791E+03, 5.503893E+03, 5.548199E+03, 5.592708E+03, 5.637422E+03,
    5.682339E+03, 5.727461E+03, 5.772787E+03, 5.818318E+03, 5.864053E+03, 5.909994E+03,
    5.956141E+03, 6.002493E+03, 6.049051E+03, 6.095814E+03, 6.142784E+03, 6.189960E+03,
    6.237343E+03, 6.284932E+03, 6.332728E+03, 6.380732E+03, 6.428942E+03, 6.477360E+03,
    6.525985E+03, 6.574818E+03, 6.623859E+03, 6.673107E+03, 6.722564E+03, 6.772229E+03,
    6.822102E+03, 6.872184E+03, 6.922475E+03, 6.972974E+03, 7.023682E+03, 7.074599E+03,
    7.125725E+03, 7.177060E+03, 7.228605E+03, 7.280359E+03, 7.332322E+03, 7.384495E+03,
    7.436877E+03, 7.489470E+03, 7.542272E+03, 7.595283E+03, 7.648505E+03, 7.701937E+03,
    7.755579E+03, 7.809431E+03, 7.863493E+03, 7.917765E+03, 7.972247E+03, 8.026940E+03,
    8.081842E+03, 8.136956E+03, 8.192279E+03, 8.247813E+03, 8.303557E+03, 8.359511E+03,
    8.415676E+03, 8.472051E+03, 8.528637E+03, 8.585433E+03, 8.642439E+03, 8.699655E+03,
    8.757082E+03, 8.814719E+03, 8.872566E+03, 8.930624E+03, 8.988891E+03, 9.047369E+03,
    9.106056E+03, 9.164954E+03, 9.224061E+03, 9.283378E+03, 9.342905E+03, 9.402642E+03,
    9.462588E+03, 9.522744E+03, 9.583109E+03, 9.643684E+03, 9.704468E+03, 9.765461E+03,
    9.826663E+03,
])

#  ---------------------- M = 16, I = 2 ---------------------------
M = 16
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    8.000000E+00, 1.635327E+01, 2.948508E+01, 4.275570E+01, 5.606042E+01, 6.938053E+01,
    8.271004E+01, 9.604648E+01, 1.093886E+02, 1.227357E+02, 1.360875E+02, 1.494436E+02,
    1.628039E+02, 1.761684E+02, 1.895370E+02, 2.029099E+02, 2.162874E+02, 2.296699E+02,
    2.430582E+02, 2.564533E+02, 2.698565E+02, 2.832694E+02, 2.966941E+02, 3.101328E+02,
    3.235883E+02, 3.370634E+02, 3.505615E+02, 3.640860E+02, 3.776405E+02, 3.912288E+02,
    4.048549E+02, 4.185229E+02, 4.322366E+02, 4.460004E+02, 4.598182E+02, 4.736942E+02,
    4.876323E+02, 5.016365E+02, 5.157107E+02, 5.298586E+02, 5.440840E+02, 5.583904E+02,
    5.727812E+02, 5.872599E+02, 6.018296E+02, 6.164934E+02, 6.312544E+02, 6.461153E+02,
    6.610791E+02, 6.761482E+02, 6.913252E+02, 7.066126E+02, 7.220127E+02, 7.375277E+02,
    7.531597E+02, 7.689108E+02, 7.847830E+02, 8.007780E+02, 8.168977E+02, 8.331437E+02,
    8.495178E+02, 8.660215E+02, 8.826562E+02, 8.994235E+02, 9.163248E+02, 9.333612E+02,
    9.505342E+02, 9.678449E+02, 9.852945E+02, 1.002884E+03, 1.020615E+03, 1.038488E+03,
    1.056504E+03, 1.074664E+03, 1.092970E+03, 1.111421E+03, 1.130019E+03, 1.148765E+03,
    1.167660E+03, 1.186704E+03, 1.205898E+03, 1.225242E+03, 1.244739E+03, 1.264388E+03,
    1.284189E+03, 1.304145E+03, 1.324254E+03, 1.344519E+03, 1.364939E+03, 1.385515E+03,
    1.406248E+03, 1.427139E+03, 1.448187E+03, 1.469393E+03, 1.490759E+03, 1.512284E+03,
    1.533969E+03, 1.555815E+03, 1.577822E+03, 1.599990E+03, 1.622320E+03, 1.644813E+03,
    1.667469E+03, 1.690288E+03, 1.713271E+03, 1.736419E+03, 1.759731E+03, 1.783208E+03,
    1.806851E+03, 1.830660E+03, 1.854636E+03, 1.878779E+03, 1.903089E+03, 1.927566E+03,
    1.952212E+03, 1.977026E+03, 2.002010E+03, 2.027162E+03, 2.052485E+03, 2.077977E+03,
    2.103640E+03, 2.129474E+03, 2.155479E+03, 2.181656E+03, 2.208005E+03, 2.234526E+03,
    2.261220E+03, 2.288087E+03, 2.315127E+03, 2.342341E+03, 2.369730E+03, 2.397293E+03,
    2.425030E+03, 2.452943E+03, 2.481032E+03, 2.509296E+03, 2.537737E+03, 2.566354E+03,
    2.595148E+03, 2.624119E+03, 2.653268E+03, 2.682595E+03, 2.712100E+03, 2.741783E+03,
    2.771646E+03, 2.801687E+03, 2.831908E+03, 2.862308E+03, 2.892889E+03, 2.923650E+03,
    2.954592E+03, 2.985715E+03, 3.017019E+03, 3.048505E+03, 3.080172E+03, 3.112022E+03,
    3.144054E+03, 3.176269E+03, 3.208667E+03, 3.241248E+03, 3.274013E+03, 3.306962E+03,
    3.340095E+03, 3.373412E+03, 3.406913E+03, 3.440600E+03, 3.474472E+03, 3.508529E+03,
    3.542772E+03, 3.577201E+03, 3.611816E+03, 3.646618E+03, 3.681606E+03, 3.716781E+03,
    3.752144E+03, 3.787693E+03, 3.823430E+03, 3.859355E+03, 3.895469E+03, 3.931770E+03,
    3.968260E+03, 4.004938E+03, 4.041806E+03, 4.078862E+03, 4.116108E+03, 4.153544E+03,
    4.191169E+03, 4.228984E+03, 4.266989E+03, 4.305184E+03, 4.343570E+03, 4.382146E+03,
    4.420914E+03, 4.459872E+03, 4.499021E+03, 4.538362E+03, 4.577894E+03, 4.617618E+03,
    4.657534E+03, 4.697641E+03, 4.737941E+03, 4.778433E+03, 4.819117E+03, 4.859994E+03,
    4.901064E+03, 4.942326E+03, 4.983781E+03, 5.025429E+03, 5.067270E+03, 5.109305E+03,
    5.151533E+03, 5.193954E+03, 5.236569E+03, 5.279378E+03, 5.322380E+03, 5.365577E+03,
    5.408967E+03, 5.452551E+03, 5.496329E+03, 5.540302E+03, 5.584469E+03, 5.628830E+03,
    5.673385E+03, 5.718135E+03, 5.763080E+03, 5.808219E+03, 5.853552E+03, 5.899081E+03,
    5.944804E+03, 5.990722E+03, 6.036834E+03, 6.083141E+03, 6.129644E+03, 6.176341E+03,
    6.223233E+03, 6.270319E+03, 6.317601E+03, 6.365078E+03, 6.412749E+03, 6.460616E+03,
    6.508677E+03, 6.556933E+03, 6.605384E+03, 6.654030E+03, 6.702871E+03, 6.751907E+03,
    6.801137E+03, 6.850563E+03, 6.900183E+03, 6.949997E+03, 7.000007E+03, 7.050211E+03,
    7.100609E+03, 7.151202E+03, 7.201990E+03, 7.252972E+03, 7.304148E+03, 7.355518E+03,
    7.407083E+03, 7.458842E+03, 7.510794E+03, 7.562941E+03, 7.615282E+03, 7.667816E+03,
    7.720544E+03, 7.773466E+03, 7.826581E+03, 7.879889E+03, 7.933391E+03, 7.987085E+03,
    8.040973E+03, 8.095054E+03, 8.149327E+03, 8.203793E+03, 8.258452E+03, 8.313303E+03,
    8.368346E+03, 8.423581E+03, 8.479008E+03, 8.534627E+03, 8.590438E+03, 8.646440E+03,
    8.702634E+03, 8.759019E+03, 8.815594E+03, 8.872361E+03, 8.929318E+03, 8.986466E+03,
    9.043803E+03, 9.101331E+03, 9.159049E+03, 9.216957E+03, 9.275054E+03, 9.333341E+03,
    9.391817E+03, 9.450481E+03, 9.509335E+03, 9.568377E+03, 9.627607E+03, 9.687025E+03,
    9.746631E+03,
])

#  ---------------------- M = 16, I = 3 ---------------------------
M = 16
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.200018E+01, 4.353101E+01, 8.267877E+01, 1.219284E+02, 1.612106E+02, 2.005120E+02,
    2.398284E+02, 2.791580E+02, 3.184998E+02, 3.578535E+02, 3.972192E+02, 4.365975E+02,
    4.759903E+02, 5.154010E+02, 5.548351E+02, 5.943007E+02, 6.338083E+02, 6.733713E+02,
    7.130053E+02, 7.527282E+02, 7.925598E+02, 8.325212E+02, 8.726345E+02, 9.129224E+02,
    9.534080E+02, 9.941141E+02, 1.035064E+03, 1.076278E+03, 1.117780E+03, 1.159590E+03,
    1.201727E+03, 1.244211E+03, 1.287059E+03, 1.330289E+03, 1.373916E+03, 1.417957E+03,
    1.462425E+03, 1.507334E+03, 1.552696E+03, 1.598524E+03, 1.644828E+03, 1.691619E+03,
    1.738907E+03, 1.786701E+03, 1.835009E+03, 1.883840E+03, 1.933201E+03, 1.983100E+03,
    2.033542E+03, 2.084536E+03, 2.136085E+03, 2.188197E+03, 2.240876E+03, 2.294127E+03,
    2.347956E+03, 2.402366E+03, 2.457362E+03, 2.512948E+03, 2.569127E+03, 2.625905E+03,
    2.683283E+03, 2.741265E+03, 2.799856E+03, 2.859056E+03, 2.918871E+03, 2.979301E+03,
    3.040351E+03, 3.102023E+03, 3.164318E+03, 3.227241E+03, 3.290792E+03, 3.354975E+03,
    3.419791E+03, 3.485243E+03, 3.551333E+03, 3.618062E+03, 3.685434E+03, 3.753448E+03,
    3.822109E+03, 3.891417E+03, 3.961373E+03, 4.031981E+03, 4.103242E+03, 4.175156E+03,
    4.247727E+03, 4.320955E+03, 4.394842E+03, 4.469390E+03, 4.544599E+03, 4.620473E+03,
    4.697012E+03, 4.774217E+03, 4.852090E+03, 4.930633E+03, 5.009846E+03, 5.089732E+03,
    5.170291E+03, 5.251524E+03, 5.333434E+03, 5.416021E+03, 5.499287E+03, 5.583232E+03,
    5.667858E+03, 5.753167E+03, 5.839159E+03, 5.925836E+03, 6.013198E+03, 6.101247E+03,
    6.189984E+03, 6.279409E+03, 6.369525E+03, 6.460332E+03, 6.551831E+03, 6.644024E+03,
    6.736910E+03, 6.830491E+03, 6.924769E+03, 7.019743E+03, 7.115415E+03, 7.211786E+03,
    7.308857E+03, 7.406628E+03, 7.505101E+03, 7.604275E+03, 7.704153E+03, 7.804734E+03,
    7.906020E+03, 8.008011E+03, 8.110707E+03, 8.214111E+03, 8.318221E+03, 8.423040E+03,
    8.528567E+03, 8.634803E+03, 8.741749E+03, 8.849406E+03, 8.957773E+03, 9.066852E+03,
    9.176642E+03, 9.287145E+03, 9.398361E+03, 9.510290E+03, 9.622933E+03, 9.736290E+03,
    9.850361E+03, 9.965148E+03, 1.008065E+04, 1.019687E+04, 1.031380E+04, 1.043145E+04,
    1.054981E+04, 1.066889E+04, 1.078869E+04, 1.090921E+04, 1.103044E+04, 1.115239E+04,
    1.127506E+04, 1.139844E+04, 1.152254E+04, 1.164736E+04, 1.177290E+04, 1.189916E+04,
    1.202613E+04, 1.215382E+04, 1.228223E+04, 1.241135E+04, 1.254120E+04, 1.267176E+04,
    1.280303E+04, 1.293503E+04, 1.306774E+04, 1.320117E+04, 1.333531E+04, 1.347018E+04,
    1.360575E+04, 1.374205E+04, 1.387906E+04, 1.401678E+04, 1.415522E+04, 1.429438E+04,
    1.443424E+04, 1.457483E+04, 1.471612E+04, 1.485813E+04, 1.500085E+04, 1.514429E+04,
    1.528843E+04, 1.543329E+04, 1.557886E+04, 1.572514E+04, 1.587213E+04, 1.601983E+04,
    1.616823E+04, 1.631735E+04, 1.646717E+04, 1.661770E+04, 1.676894E+04, 1.692088E+04,
    1.707353E+04, 1.722688E+04, 1.738094E+04, 1.753570E+04, 1.769116E+04, 1.784733E+04,
    1.800419E+04, 1.816176E+04, 1.832002E+04, 1.847899E+04, 1.863865E+04, 1.879901E+04,
    1.896006E+04, 1.912181E+04, 1.928426E+04, 1.944740E+04, 1.961123E+04, 1.977575E+04,
    1.994097E+04, 2.010688E+04, 2.027347E+04, 2.044076E+04, 2.060873E+04, 2.077739E+04,
    2.094673E+04, 2.111676E+04, 2.128747E+04, 2.145887E+04, 2.163095E+04, 2.180371E+04,
    2.197714E+04, 2.215126E+04, 2.232606E+04, 2.250153E+04, 2.267767E+04, 2.285449E+04,
    2.303199E+04, 2.321016E+04, 2.338899E+04, 2.356850E+04, 2.374868E+04, 2.392952E+04,
    2.411104E+04, 2.429321E+04, 2.447606E+04, 2.465956E+04, 2.484373E+04, 2.502856E+04,
    2.521404E+04, 2.540019E+04, 2.558699E+04, 2.577445E+04, 2.596257E+04, 2.615133E+04,
    2.634075E+04, 2.653083E+04, 2.672155E+04, 2.691292E+04, 2.710493E+04, 2.729760E+04,
    2.749091E+04, 2.768486E+04, 2.787945E+04, 2.807469E+04, 2.827056E+04, 2.846707E+04,
    2.866422E+04, 2.886201E+04, 2.906042E+04, 2.925948E+04, 2.945916E+04, 2.965947E+04,
    2.986042E+04, 3.006199E+04, 3.026418E+04, 3.046700E+04, 3.067045E+04, 3.087451E+04,
    3.107920E+04, 3.128450E+04, 3.149043E+04, 3.169697E+04, 3.190412E+04, 3.211189E+04,
    3.232027E+04, 3.252925E+04, 3.273885E+04, 3.294906E+04, 3.315987E+04, 3.337129E+04,
    3.358331E+04, 3.379593E+04, 3.400915E+04, 3.422297E+04, 3.443739E+04, 3.465240E+04,
    3.486801E+04, 3.508421E+04, 3.530100E+04, 3.551839E+04, 3.573636E+04, 3.595491E+04,
    3.617405E+04,
])

#  ---------------------- M = 16, I = 4 ---------------------------
M = 16
I = 4
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.200018E+01, 4.355487E+01, 8.272675E+01, 1.220005E+02, 1.613067E+02, 2.006322E+02,
    2.399726E+02, 2.793263E+02, 3.186922E+02, 3.580700E+02, 3.974597E+02, 4.368621E+02,
    4.762790E+02, 5.157139E+02, 5.551722E+02, 5.946621E+02, 6.341940E+02, 6.737815E+02,
    7.134401E+02, 7.531878E+02, 7.930444E+02, 8.330310E+02, 8.731698E+02, 9.134835E+02,
    9.539953E+02, 9.947280E+02, 1.035704E+03, 1.076947E+03, 1.118476E+03, 1.160314E+03,
    1.202480E+03, 1.244992E+03, 1.287870E+03, 1.331130E+03, 1.374788E+03, 1.418860E+03,
    1.463360E+03, 1.508301E+03, 1.553696E+03, 1.599557E+03, 1.645895E+03, 1.692720E+03,
    1.740043E+03, 1.787872E+03, 1.836217E+03, 1.885084E+03, 1.934483E+03, 1.984419E+03,
    2.034900E+03, 2.085932E+03, 2.137521E+03, 2.189673E+03, 2.242393E+03, 2.295685E+03,
    2.349556E+03, 2.404008E+03, 2.459047E+03, 2.514677E+03, 2.570901E+03, 2.627723E+03,
    2.685146E+03, 2.743175E+03, 2.801812E+03, 2.861060E+03, 2.920922E+03, 2.981401E+03,
    3.042499E+03, 3.104220E+03, 3.166566E+03, 3.229539E+03, 3.293142E+03, 3.357377E+03,
    3.422246E+03, 3.487751E+03, 3.553894E+03, 3.620678E+03, 3.688104E+03, 3.756175E+03,
    3.824891E+03, 3.894256E+03, 3.964270E+03, 4.034936E+03, 4.106255E+03, 4.178229E+03,
    4.250859E+03, 4.324148E+03, 4.398096E+03, 4.472705E+03, 4.547977E+03, 4.623914E+03,
    4.700516E+03, 4.777785E+03, 4.855723E+03, 4.934331E+03, 5.013611E+03, 5.093563E+03,
    5.174189E+03, 5.255491E+03, 5.337469E+03, 5.420125E+03, 5.503460E+03, 5.587476E+03,
    5.672173E+03, 5.757554E+03, 5.843618E+03, 5.930367E+03, 6.017803E+03, 6.105926E+03,
    6.194737E+03, 6.284239E+03, 6.374431E+03, 6.465314E+03, 6.556890E+03, 6.649160E+03,
    6.742125E+03, 6.835786E+03, 6.930143E+03, 7.025198E+03, 7.120951E+03, 7.217404E+03,
    7.314556E+03, 7.412410E+03, 7.510966E+03, 7.610225E+03, 7.710187E+03, 7.810854E+03,
    7.912226E+03, 8.014303E+03, 8.117087E+03, 8.220578E+03, 8.324778E+03, 8.429685E+03,
    8.535302E+03, 8.641629E+03, 8.748666E+03, 8.856414E+03, 8.964874E+03, 9.074045E+03,
    9.183929E+03, 9.294526E+03, 9.405837E+03, 9.517861E+03, 9.630600E+03, 9.744054E+03,
    9.858222E+03, 9.973107E+03, 1.008871E+04, 1.020502E+04, 1.032206E+04, 1.043980E+04,
    1.055827E+04, 1.067745E+04, 1.079735E+04, 1.091797E+04, 1.103931E+04, 1.116136E+04,
    1.128413E+04, 1.140762E+04, 1.153183E+04, 1.165676E+04, 1.178240E+04, 1.190877E+04,
    1.203585E+04, 1.216365E+04, 1.229217E+04, 1.242140E+04, 1.255136E+04, 1.268203E+04,
    1.281342E+04, 1.294553E+04, 1.307835E+04, 1.321189E+04, 1.334615E+04, 1.348113E+04,
    1.361683E+04, 1.375324E+04, 1.389036E+04, 1.402821E+04, 1.416676E+04, 1.430604E+04,
    1.444603E+04, 1.458673E+04, 1.472815E+04, 1.487028E+04, 1.501313E+04, 1.515669E+04,
    1.530096E+04, 1.544594E+04, 1.559163E+04, 1.573804E+04, 1.588516E+04, 1.603299E+04,
    1.618152E+04, 1.633077E+04, 1.648072E+04, 1.663138E+04, 1.678275E+04, 1.693483E+04,
    1.708761E+04, 1.724110E+04, 1.739530E+04, 1.755019E+04, 1.770579E+04, 1.786210E+04,
    1.801910E+04, 1.817681E+04, 1.833522E+04, 1.849432E+04, 1.865413E+04, 1.881464E+04,
    1.897584E+04, 1.913774E+04, 1.930033E+04, 1.946362E+04, 1.962761E+04, 1.979228E+04,
    1.995766E+04, 2.012372E+04, 2.029047E+04, 2.045791E+04, 2.062605E+04, 2.079487E+04,
    2.096438E+04, 2.113457E+04, 2.130545E+04, 2.147701E+04, 2.164926E+04, 2.182219E+04,
    2.199580E+04, 2.217009E+04, 2.234507E+04, 2.252072E+04, 2.269704E+04, 2.287405E+04,
    2.305173E+04, 2.323008E+04, 2.340911E+04, 2.358881E+04, 2.376918E+04, 2.395022E+04,
    2.413193E+04, 2.431431E+04, 2.449736E+04, 2.468107E+04, 2.486545E+04, 2.505049E+04,
    2.523619E+04, 2.542255E+04, 2.560958E+04, 2.579726E+04, 2.598560E+04, 2.617460E+04,
    2.636425E+04, 2.655456E+04, 2.674552E+04, 2.693713E+04, 2.712940E+04, 2.732231E+04,
    2.751587E+04, 2.771008E+04, 2.790494E+04, 2.810044E+04, 2.829658E+04, 2.849337E+04,
    2.869080E+04, 2.888886E+04, 2.908757E+04, 2.928691E+04, 2.948689E+04, 2.968751E+04,
    2.988876E+04, 3.009064E+04, 3.029315E+04, 3.049629E+04, 3.070006E+04, 3.090446E+04,
    3.110948E+04, 3.131513E+04, 3.152140E+04, 3.172830E+04, 3.193581E+04, 3.214394E+04,
    3.235270E+04, 3.256206E+04, 3.277205E+04, 3.298265E+04, 3.319386E+04, 3.340568E+04,
    3.361811E+04, 3.383115E+04, 3.404480E+04, 3.425905E+04, 3.447391E+04, 3.468937E+04,
    3.490544E+04, 3.512210E+04, 3.533937E+04, 3.555723E+04, 3.577568E+04, 3.599474E+04,
    3.621438E+04,
])

#  ---------------------- M = 17, I = 1 ---------------------------
M = 17
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.200000E+01, 3.036750E+01, 5.612178E+01, 8.203022E+01, 1.079799E+02, 1.339502E+02,
    1.599342E+02, 1.859291E+02, 2.119337E+02, 2.379470E+02, 2.639687E+02, 2.899986E+02,
    3.160366E+02, 3.420830E+02, 3.681386E+02, 3.942045E+02, 4.202826E+02, 4.463759E+02,
    4.724880E+02, 4.986238E+02, 5.247889E+02, 5.509901E+02, 5.772350E+02, 6.035318E+02,
    6.298897E+02, 6.563182E+02, 6.828272E+02, 7.094270E+02, 7.361281E+02, 7.629408E+02,
    7.898758E+02, 8.169433E+02, 8.441535E+02, 8.715165E+02, 8.990419E+02, 9.267392E+02,
    9.546175E+02, 9.826856E+02, 1.010952E+03, 1.039424E+03, 1.068111E+03, 1.097019E+03,
    1.126156E+03, 1.155527E+03, 1.185141E+03, 1.215002E+03, 1.245116E+03, 1.275490E+03,
    1.306128E+03, 1.337035E+03, 1.368216E+03, 1.399675E+03, 1.431417E+03, 1.463446E+03,
    1.495766E+03, 1.528380E+03, 1.561292E+03, 1.594505E+03, 1.628022E+03, 1.661847E+03,
    1.695982E+03, 1.730431E+03, 1.765195E+03, 1.800278E+03, 1.835682E+03, 1.871408E+03,
    1.907461E+03, 1.943841E+03, 1.980550E+03, 2.017592E+03, 2.054967E+03, 2.092677E+03,
    2.130725E+03, 2.169112E+03, 2.207840E+03, 2.246911E+03, 2.286325E+03, 2.326085E+03,
    2.366192E+03, 2.406647E+03, 2.447452E+03, 2.488609E+03, 2.530118E+03, 2.571981E+03,
    2.614199E+03, 2.656774E+03, 2.699706E+03, 2.742997E+03, 2.786647E+03, 2.830659E+03,
    2.875033E+03, 2.919770E+03, 2.964871E+03, 3.010338E+03, 3.056170E+03, 3.102370E+03,
    3.148938E+03, 3.195875E+03, 3.243182E+03, 3.290860E+03, 3.338910E+03, 3.387332E+03,
    3.436128E+03, 3.485298E+03, 3.534843E+03, 3.584764E+03, 3.635061E+03, 3.685735E+03,
    3.736788E+03, 3.788220E+03, 3.840030E+03, 3.892221E+03, 3.944793E+03, 3.997746E+03,
    4.051081E+03, 4.104799E+03, 4.158899E+03, 4.213384E+03, 4.268253E+03, 4.323507E+03,
    4.379146E+03, 4.435171E+03, 4.491582E+03, 4.548380E+03, 4.605566E+03, 4.663139E+03,
    4.721100E+03, 4.779450E+03, 4.838189E+03, 4.897317E+03, 4.956835E+03, 5.016743E+03,
    5.077042E+03, 5.137731E+03, 5.198811E+03, 5.260282E+03, 5.322144E+03, 5.384399E+03,
    5.447045E+03, 5.510084E+03, 5.573515E+03, 5.637339E+03, 5.701556E+03, 5.766165E+03,
    5.831168E+03, 5.896563E+03, 5.962352E+03, 6.028535E+03, 6.095111E+03, 6.162081E+03,
    6.229444E+03, 6.297201E+03, 6.365351E+03, 6.433896E+03, 6.502834E+03, 6.572165E+03,
    6.641891E+03, 6.712010E+03, 6.782522E+03, 6.853428E+03, 6.924728E+03, 6.996420E+03,
    7.068506E+03, 7.140985E+03, 7.213857E+03, 7.287122E+03, 7.360779E+03, 7.434829E+03,
    7.509271E+03, 7.584105E+03, 7.659331E+03, 7.734948E+03, 7.810957E+03, 7.887357E+03,
    7.964148E+03, 8.041329E+03, 8.118901E+03, 8.196862E+03, 8.275214E+03, 8.353954E+03,
    8.433084E+03, 8.512603E+03, 8.592509E+03, 8.672804E+03, 8.753486E+03, 8.834556E+03,
    8.916012E+03, 8.997855E+03, 9.080084E+03, 9.162699E+03, 9.245698E+03, 9.329082E+03,
    9.412851E+03, 9.497003E+03, 9.581539E+03, 9.666457E+03, 9.751757E+03, 9.837440E+03,
    9.923503E+03, 1.000995E+04, 1.009677E+04, 1.018398E+04, 1.027156E+04, 1.035952E+04,
    1.044786E+04, 1.053658E+04, 1.062567E+04, 1.071515E+04, 1.080499E+04, 1.089521E+04,
    1.098581E+04, 1.107678E+04, 1.116812E+04, 1.125984E+04, 1.135192E+04, 1.144438E+04,
    1.153721E+04, 1.163041E+04, 1.172398E+04, 1.181791E+04, 1.191222E+04, 1.200689E+04,
    1.210193E+04, 1.219733E+04, 1.229310E+04, 1.238923E+04, 1.248573E+04, 1.258259E+04,
    1.267981E+04, 1.277739E+04, 1.287533E+04, 1.297363E+04, 1.307230E+04, 1.317132E+04,
    1.327069E+04, 1.337043E+04, 1.347052E+04, 1.357096E+04, 1.367176E+04, 1.377291E+04,
    1.387441E+04, 1.397627E+04, 1.407848E+04, 1.418103E+04, 1.428394E+04, 1.438719E+04,
    1.449080E+04, 1.459474E+04, 1.469904E+04, 1.480368E+04, 1.490866E+04, 1.501399E+04,
    1.511966E+04, 1.522567E+04, 1.533202E+04, 1.543871E+04, 1.554573E+04, 1.565310E+04,
    1.576080E+04, 1.586884E+04, 1.597722E+04, 1.608592E+04, 1.619497E+04, 1.630434E+04,
    1.641404E+04, 1.652408E+04, 1.663444E+04, 1.674513E+04, 1.685615E+04, 1.696750E+04,
    1.707917E+04, 1.719117E+04, 1.730349E+04, 1.741613E+04, 1.752909E+04, 1.764238E+04,
    1.775598E+04, 1.786991E+04, 1.798415E+04, 1.809870E+04, 1.821358E+04, 1.832876E+04,
    1.844426E+04, 1.856008E+04, 1.867620E+04, 1.879264E+04, 1.890938E+04, 1.902644E+04,
    1.914380E+04, 1.926147E+04, 1.937944E+04, 1.949772E+04, 1.961630E+04, 1.973518E+04,
    1.985437E+04, 1.997385E+04, 2.009364E+04, 2.021372E+04, 2.033410E+04, 2.045477E+04,
    2.057574E+04,
])

#  ---------------------- M = 17, I = 2 ---------------------------
M = 17
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.800464E+01, 8.321075E+01, 1.599981E+02, 2.369086E+02, 3.138651E+02, 3.908526E+02,
    4.678662E+02, 5.449041E+02, 6.219653E+02, 6.990504E+02, 7.761621E+02, 8.533068E+02,
    9.304964E+02, 1.007750E+03, 1.085094E+03, 1.162563E+03, 1.240199E+03, 1.318049E+03,
    1.396167E+03, 1.474609E+03, 1.553433E+03, 1.632699E+03, 1.712467E+03, 1.792793E+03,
    1.873736E+03, 1.955349E+03, 2.037686E+03, 2.120794E+03, 2.204722E+03, 2.289512E+03,
    2.375206E+03, 2.461843E+03, 2.549457E+03, 2.638082E+03, 2.727749E+03, 2.818487E+03,
    2.910323E+03, 3.003280E+03, 3.097383E+03, 3.192652E+03, 3.289107E+03, 3.386767E+03,
    3.485650E+03, 3.585770E+03, 3.687143E+03, 3.789784E+03, 3.893705E+03, 3.998918E+03,
    4.105435E+03, 4.213267E+03, 4.322425E+03, 4.432917E+03, 4.544753E+03, 4.657942E+03,
    4.772491E+03, 4.888409E+03, 5.005702E+03, 5.124379E+03, 5.244445E+03, 5.365907E+03,
    5.488771E+03, 5.613043E+03, 5.738728E+03, 5.865832E+03, 5.994360E+03, 6.124317E+03,
    6.255707E+03, 6.388535E+03, 6.522805E+03, 6.658522E+03, 6.795690E+03, 6.934312E+03,
    7.074392E+03, 7.215933E+03, 7.358940E+03, 7.503415E+03, 7.649362E+03, 7.796783E+03,
    7.945681E+03, 8.096060E+03, 8.247921E+03, 8.401268E+03, 8.556102E+03, 8.712426E+03,
    8.870242E+03, 9.029552E+03, 9.190358E+03, 9.352662E+03, 9.516465E+03, 9.681769E+03,
    9.848576E+03, 1.001689E+04, 1.018670E+04, 1.035802E+04, 1.053085E+04, 1.070519E+04,
    1.088103E+04, 1.105839E+04, 1.123725E+04, 1.141763E+04, 1.159951E+04, 1.178291E+04,
    1.196781E+04, 1.215423E+04, 1.234216E+04, 1.253160E+04, 1.272255E+04, 1.291501E+04,
    1.310899E+04, 1.330447E+04, 1.350145E+04, 1.369995E+04, 1.389995E+04, 1.410146E+04,
    1.430448E+04, 1.450899E+04, 1.471501E+04, 1.492253E+04, 1.513156E+04, 1.534207E+04,
    1.555409E+04, 1.576760E+04, 1.598261E+04, 1.619910E+04, 1.641709E+04, 1.663656E+04,
    1.685752E+04, 1.707996E+04, 1.730388E+04, 1.752928E+04, 1.775616E+04, 1.798452E+04,
    1.821434E+04, 1.844564E+04, 1.867840E+04, 1.891262E+04, 1.914831E+04, 1.938545E+04,
    1.962405E+04, 1.986410E+04, 2.010561E+04, 2.034856E+04, 2.059295E+04, 2.083878E+04,
    2.108605E+04, 2.133476E+04, 2.158489E+04, 2.183646E+04, 2.208945E+04, 2.234386E+04,
    2.259968E+04, 2.285692E+04, 2.311558E+04, 2.337563E+04, 2.363710E+04, 2.389996E+04,
    2.416422E+04, 2.442987E+04, 2.469691E+04, 2.496533E+04, 2.523514E+04, 2.550632E+04,
    2.577888E+04, 2.605280E+04, 2.632810E+04, 2.660475E+04, 2.688276E+04, 2.716213E+04,
    2.744285E+04, 2.772491E+04, 2.800832E+04, 2.829306E+04, 2.857914E+04, 2.886655E+04,
    2.915528E+04, 2.944534E+04, 2.973671E+04, 3.002940E+04, 3.032340E+04, 3.061870E+04,
    3.091531E+04, 3.121321E+04, 3.151240E+04, 3.181289E+04, 3.211465E+04, 3.241770E+04,
    3.272203E+04, 3.302762E+04, 3.333449E+04, 3.364262E+04, 3.395200E+04, 3.426265E+04,
    3.457454E+04, 3.488768E+04, 3.520206E+04, 3.551768E+04, 3.583454E+04, 3.615262E+04,
    3.647193E+04, 3.679246E+04, 3.711421E+04, 3.743717E+04, 3.776134E+04, 3.808672E+04,
    3.841329E+04, 3.874106E+04, 3.907002E+04, 3.940017E+04, 3.973150E+04, 4.006401E+04,
    4.039770E+04, 4.073255E+04, 4.106857E+04, 4.140576E+04, 4.174410E+04, 4.208359E+04,
    4.242424E+04, 4.276602E+04, 4.310895E+04, 4.345302E+04, 4.379822E+04, 4.414455E+04,
    4.449200E+04, 4.484057E+04, 4.519026E+04, 4.554105E+04, 4.589296E+04, 4.624597E+04,
    4.660008E+04, 4.695528E+04, 4.731158E+04, 4.766896E+04, 4.802742E+04, 4.838697E+04,
    4.874758E+04, 4.910927E+04, 4.947202E+04, 4.983584E+04, 5.020071E+04, 5.056664E+04,
    5.093362E+04, 5.130164E+04, 5.167071E+04, 5.204081E+04, 5.241195E+04, 5.278412E+04,
    5.315731E+04, 5.353153E+04, 5.390676E+04, 5.428300E+04, 5.466026E+04, 5.503852E+04,
    5.541778E+04, 5.579805E+04, 5.617930E+04, 5.656155E+04, 5.694478E+04, 5.732899E+04,
    5.771418E+04, 5.810034E+04, 5.848748E+04, 5.887558E+04, 5.926464E+04, 5.965467E+04,
    6.004564E+04, 6.043757E+04, 6.083045E+04, 6.122426E+04, 6.161902E+04, 6.201471E+04,
    6.241133E+04, 6.280888E+04, 6.320736E+04, 6.360675E+04, 6.400706E+04, 6.440828E+04,
    6.481042E+04, 6.521345E+04, 6.561739E+04, 6.602222E+04, 6.642795E+04, 6.683456E+04,
    6.724206E+04, 6.765045E+04, 6.805971E+04, 6.846985E+04, 6.888085E+04, 6.929273E+04,
    6.970546E+04, 7.011906E+04, 7.053351E+04, 7.094882E+04, 7.136497E+04, 7.178197E+04,
    7.219981E+04, 7.261848E+04, 7.303799E+04, 7.345833E+04, 7.387950E+04, 7.430148E+04,
    7.472429E+04,
])

#  ---------------------- M = 18, I = 1 ---------------------------
M = 18
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    9.761590E+00, 1.797541E+02, 3.589432E+02, 5.384749E+02, 7.198461E+02, 9.059401E+02,
    1.099605E+03, 1.302867E+03, 1.516834E+03, 1.742050E+03, 1.978735E+03, 2.226901E+03,
    2.486580E+03, 2.757704E+03, 3.040307E+03, 3.334331E+03, 3.639872E+03, 3.956877E+03,
    4.285427E+03, 4.625509E+03, 4.977243E+03, 5.340628E+03, 5.715693E+03, 6.102457E+03,
    6.501039E+03, 6.911435E+03, 7.333681E+03, 7.767794E+03, 8.213845E+03, 8.671895E+03,
    9.141829E+03, 9.623827E+03, 1.011782E+04, 1.062389E+04, 1.114211E+04, 1.167236E+04,
    1.221467E+04, 1.276928E+04, 1.333592E+04, 1.391481E+04, 1.450597E+04, 1.510916E+04,
    1.572473E+04, 1.635254E+04, 1.699269E+04, 1.764503E+04, 1.830963E+04, 1.898657E+04,
    1.967596E+04, 2.037757E+04, 2.109164E+04, 2.181793E+04, 2.255682E+04, 2.330792E+04,
    2.407159E+04, 2.484758E+04, 2.563595E+04, 2.643692E+04, 2.725021E+04, 2.807621E+04,
    2.891445E+04, 2.976534E+04, 3.062873E+04, 3.150468E+04, 3.239304E+04, 3.329404E+04,
    3.420753E+04, 3.513376E+04, 3.607234E+04, 3.702375E+04, 3.798758E+04, 3.896408E+04,
    3.995331E+04, 4.095507E+04, 4.196962E+04, 4.299675E+04, 4.403651E+04, 4.508917E+04,
    4.615426E+04, 4.723232E+04, 4.832287E+04, 4.942645E+04, 5.054257E+04, 5.167152E+04,
    5.281306E+04, 5.396749E+04, 5.513482E+04, 5.631483E+04, 5.750780E+04, 5.871349E+04,
    5.993189E+04, 6.116335E+04, 6.240759E+04, 6.366463E+04, 6.493478E+04, 6.621748E+04,
    6.751335E+04, 6.882212E+04, 7.014378E+04, 7.147838E+04, 7.282593E+04, 7.418643E+04,
    7.555994E+04, 7.694646E+04, 7.834567E+04, 7.975826E+04, 8.118359E+04, 8.262236E+04,
    8.407389E+04, 8.553818E+04, 8.701598E+04, 8.850660E+04, 9.001039E+04, 9.152739E+04,
    9.305725E+04, 9.460073E+04, 9.615671E+04, 9.772636E+04, 9.930890E+04, 1.009044E+05,
    1.025132E+05, 1.041354E+05, 1.057705E+05, 1.074191E+05, 1.090806E+05, 1.107556E+05,
    1.124435E+05, 1.141450E+05, 1.158599E+05, 1.175875E+05, 1.193290E+05, 1.210832E+05,
    1.228509E+05, 1.246323E+05, 1.264267E+05, 1.282343E+05, 1.300555E+05, 1.318899E+05,
    1.337380E+05, 1.355993E+05, 1.374737E+05, 1.393619E+05, 1.412634E+05, 1.431781E+05,
    1.451065E+05, 1.470483E+05, 1.490034E+05, 1.509722E+05, 1.529544E+05, 1.549505E+05,
    1.569593E+05, 1.589821E+05, 1.610182E+05, 1.630683E+05, 1.651312E+05, 1.672086E+05,
    1.692989E+05, 1.714027E+05, 1.735204E+05, 1.756516E+05, 1.777969E+05, 1.799551E+05,
    1.821273E+05, 1.843131E+05, 1.865124E+05, 1.887258E+05, 1.909522E+05, 1.931928E+05,
    1.954469E+05, 1.977152E+05, 1.999966E+05, 2.022921E+05, 2.046013E+05, 2.069242E+05,
    2.092612E+05, 2.116114E+05,
])

#  ---------------------- M = 18, I = 2 ---------------------------
M = 18
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    9.912400E+00, 1.828410E+02, 3.651190E+02, 5.477455E+02, 7.322431E+02, 9.215448E+02,
    1.118548E+03, 1.325314E+03, 1.543000E+03, 1.772137E+03, 2.012972E+03, 2.265546E+03,
    2.529857E+03, 2.805894E+03, 3.093647E+03, 3.393126E+03, 3.704301E+03, 4.027259E+03,
    4.362060E+03, 4.708654E+03, 5.067118E+03, 5.437544E+03, 5.819917E+03, 6.214315E+03,
    6.620748E+03, 7.039331E+03, 7.469975E+03, 7.912762E+03, 8.367835E+03, 8.835036E+03,
    9.314552E+03, 9.806333E+03, 1.031039E+04, 1.082689E+04, 1.135564E+04, 1.189679E+04,
    1.245038E+04, 1.301635E+04, 1.359485E+04, 1.418577E+04, 1.478913E+04, 1.540504E+04,
    1.603337E+04, 1.667435E+04, 1.732783E+04, 1.799389E+04, 1.867251E+04, 1.936376E+04,
    2.006759E+04, 2.078409E+04, 2.151317E+04, 2.225490E+04, 2.300937E+04, 2.377649E+04,
    2.455631E+04, 2.534873E+04, 2.615398E+04, 2.697195E+04, 2.780269E+04, 2.864608E+04,
    2.950235E+04, 3.037136E+04, 3.125337E+04, 3.214803E+04, 3.305538E+04, 3.397565E+04,
    3.490891E+04, 3.585478E+04, 3.681371E+04, 3.778554E+04, 3.877029E+04, 3.976780E+04,
    4.077831E+04, 4.180164E+04, 4.283805E+04, 4.388733E+04, 4.494977E+04, 4.602491E+04,
    4.711326E+04, 4.821437E+04, 4.932877E+04, 5.045597E+04, 5.159626E+04, 5.274966E+04,
    5.391623E+04, 5.509570E+04, 5.628837E+04, 5.749399E+04, 5.871259E+04, 5.994477E+04,
    6.118969E+04, 6.244793E+04, 6.371925E+04, 6.500366E+04, 6.630117E+04, 6.761214E+04,
    6.893627E+04, 7.027357E+04, 7.162408E+04, 7.298748E+04, 7.436445E+04, 7.575468E+04,
    7.715820E+04, 7.857503E+04, 8.000484E+04, 8.144834E+04, 8.290486E+04, 8.437476E+04,
    8.585806E+04, 8.735480E+04, 8.886462E+04, 9.038828E+04, 9.192505E+04, 9.347494E+04,
    9.503836E+04, 9.661531E+04, 9.820585E+04, 9.980955E+04, 1.014269E+05, 1.030574E+05,
    1.047016E+05, 1.063594E+05, 1.080305E+05, 1.097153E+05, 1.114134E+05, 1.131248E+05,
    1.148503E+05, 1.165888E+05, 1.183415E+05, 1.201071E+05, 1.218866E+05, 1.236798E+05,
    1.254865E+05, 1.273067E+05, 1.291407E+05, 1.309881E+05, 1.328495E+05, 1.347244E+05,
    1.366128E+05, 1.385151E+05, 1.404309E+05, 1.423608E+05, 1.443038E+05, 1.462613E+05,
    1.482318E+05, 1.502165E+05, 1.522147E+05, 1.542271E+05, 1.562526E+05, 1.582928E+05,
    1.603461E+05, 1.624136E+05, 1.644948E+05, 1.665896E+05, 1.686987E+05, 1.708211E+05,
    1.729576E+05, 1.751085E+05, 1.772726E+05, 1.794510E+05, 1.816432E+05, 1.838497E+05,
    1.860695E+05, 1.883038E+05, 1.905518E+05, 1.928137E+05, 1.950900E+05, 1.973803E+05,
    1.996838E+05, 2.020024E+05, 2.043343E+05, 2.066802E+05, 2.090406E+05, 2.114150E+05,
    2.138033E+05, 2.162063E+05,
])

#  ---------------------- M = 19, I = 1 ---------------------------
M = 19
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    3.780860E+00, 6.888728E+01, 1.374712E+02, 2.060890E+02, 2.747808E+02, 3.438010E+02,
    4.137509E+02, 4.855483E+02, 5.602403E+02, 6.388943E+02, 7.225221E+02, 8.120553E+02,
    9.083745E+02, 1.012293E+03, 1.124615E+03, 1.246077E+03, 1.377438E+03, 1.519448E+03,
    1.672847E+03, 1.838414E+03, 2.016938E+03, 2.209189E+03, 2.416025E+03, 2.638267E+03,
    2.876824E+03, 3.132602E+03, 3.406543E+03, 3.699623E+03, 4.012839E+03, 4.347286E+03,
    4.703992E+03, 5.084126E+03, 5.488848E+03, 5.919326E+03, 6.376821E+03, 6.862632E+03,
    7.378065E+03, 7.924484E+03, 8.503314E+03, 9.115988E+03, 9.764033E+03, 1.044898E+04,
    1.117241E+04, 1.193599E+04, 1.274135E+04, 1.359026E+04, 1.448448E+04, 1.542590E+04,
    1.641632E+04, 1.745769E+04, 1.855200E+04, 1.970131E+04, 2.090764E+04, 2.217321E+04,
    2.350017E+04, 2.489078E+04, 2.634733E+04, 2.787217E+04, 2.946776E+04, 3.113658E+04,
    3.288109E+04, 3.470398E+04, 3.660778E+04, 3.859534E+04, 4.066933E+04, 4.283266E+04,
    4.508820E+04, 4.743892E+04, 4.988780E+04, 5.243797E+04, 5.509261E+04, 5.785493E+04,
    6.072821E+04, 6.371578E+04, 6.682109E+04, 7.004764E+04, 7.339906E+04, 7.687883E+04,
    8.049086E+04, 8.423872E+04, 8.812646E+04, 9.215786E+04, 9.633693E+04, 1.006678E+05,
    1.051547E+05, 1.098017E+05, 1.146132E+05, 1.195935E+05, 1.247472E+05, 1.300787E+05,
    1.355927E+05, 1.412939E+05, 1.471870E+05, 1.532770E+05, 1.595687E+05, 1.660673E+05,
    1.727778E+05, 1.797055E+05, 1.868555E+05, 1.942334E+05, 2.018446E+05, 2.096945E+05,
    2.177890E+05, 2.261336E+05, 2.347342E+05, 2.435967E+05, 2.527273E+05, 2.621318E+05,
    2.718165E+05, 2.817878E+05, 2.920519E+05, 3.026154E+05, 3.134849E+05, 3.246670E+05,
    3.361684E+05, 3.479962E+05, 3.601571E+05, 3.726584E+05, 3.855070E+05, 3.987105E+05,
    4.122760E+05, 4.262112E+05, 4.405236E+05, 4.552207E+05, 4.703105E+05, 4.858008E+05,
    5.016998E+05, 5.180154E+05, 5.347559E+05, 5.519297E+05, 5.695451E+05, 5.876108E+05,
    6.061353E+05, 6.251276E+05, 6.445965E+05, 6.645510E+05, 6.850001E+05, 7.059533E+05,
    7.274196E+05, 7.494089E+05, 7.719305E+05, 7.949943E+05, 8.186099E+05, 8.427874E+05,
    8.675368E+05, 8.928685E+05, 9.187926E+05, 9.453196E+05, 9.724601E+05, 1.000225E+06,
    1.028624E+06, 1.057670E+06, 1.087373E+06, 1.117744E+06, 1.148794E+06, 1.180535E+06,
    1.212979E+06, 1.246138E+06, 1.280022E+06, 1.314645E+06, 1.350018E+06, 1.386153E+06,
    1.423064E+06, 1.460762E+06, 1.499260E+06, 1.538571E+06, 1.578708E+06, 1.619685E+06,
    1.661513E+06, 1.704207E+06, 1.747780E+06, 1.792246E+06, 1.837618E+06, 1.883911E+06,
    1.931138E+06, 1.979315E+06,
])

#  ---------------------- M = 19, I = 2 ---------------------------
M = 19
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    3.866200E+00, 7.060513E+01, 1.409078E+02, 2.112453E+02, 2.816616E+02, 3.524124E+02,
    4.241291E+02, 4.977537E+02, 5.743694E+02, 6.550831E+02, 7.409406E+02, 8.329051E+02,
    9.318879E+02, 1.038741E+03, 1.154268E+03, 1.279272E+03, 1.414522E+03, 1.560780E+03,
    1.718840E+03, 1.889483E+03, 2.073538E+03, 2.271818E+03, 2.485183E+03, 2.714529E+03,
    2.960748E+03, 3.224797E+03, 3.507665E+03, 3.810366E+03, 4.133950E+03, 4.479493E+03,
    4.848138E+03, 5.241034E+03, 5.659402E+03, 6.104497E+03, 6.577577E+03, 7.080010E+03,
    7.613174E+03, 8.178454E+03, 8.777326E+03, 9.411328E+03, 1.008198E+04, 1.079091E+04,
    1.153976E+04, 1.233025E+04, 1.316410E+04, 1.404310E+04, 1.496915E+04, 1.594410E+04,
    1.696994E+04, 1.804866E+04, 1.918233E+04, 2.037303E+04, 2.162293E+04, 2.293432E+04,
    2.430941E+04, 2.575059E+04, 2.726019E+04, 2.884073E+04, 3.049469E+04, 3.222463E+04,
    3.403321E+04, 3.592313E+04, 3.789715E+04, 3.995806E+04, 4.210879E+04, 4.435223E+04,
    4.669141E+04, 4.912946E+04, 5.166950E+04, 5.431473E+04, 5.706844E+04, 5.993400E+04,
    6.291482E+04, 6.601437E+04, 6.923620E+04, 7.258402E+04, 7.606148E+04, 7.967242E+04,
    8.342064E+04, 8.731008E+04, 9.134474E+04, 9.552874E+04, 9.986620E+04, 1.043614E+05,
    1.090186E+05, 1.138423E+05, 1.188369E+05, 1.240069E+05, 1.293571E+05, 1.348921E+05,
    1.406168E+05, 1.465359E+05, 1.526546E+05, 1.589779E+05, 1.655109E+05, 1.722588E+05,
    1.792271E+05, 1.864210E+05, 1.938462E+05, 2.015082E+05, 2.094126E+05, 2.175653E+05,
    2.259721E+05, 2.346390E+05, 2.435721E+05, 2.527774E+05, 2.622614E+05, 2.720302E+05,
    2.820903E+05, 2.924483E+05, 3.031110E+05, 3.140847E+05, 3.253767E+05, 3.369937E+05,
    3.489427E+05, 3.612310E+05, 3.738659E+05, 3.868545E+05, 4.002045E+05, 4.139232E+05,
    4.280186E+05, 4.424983E+05, 4.573703E+05, 4.726424E+05, 4.883229E+05, 5.044200E+05,
    5.209419E+05, 5.378972E+05, 5.552945E+05, 5.731422E+05, 5.914493E+05, 6.102247E+05,
    6.294775E+05, 6.492166E+05, 6.694515E+05, 6.901914E+05, 7.114458E+05, 7.332246E+05,
    7.555370E+05, 7.783934E+05, 8.018036E+05, 8.257776E+05, 8.503257E+05, 8.754583E+05,
    9.011859E+05, 9.275190E+05, 9.544684E+05, 9.820450E+05, 1.010260E+06, 1.039124E+06,
    1.068649E+06, 1.098845E+06, 1.129726E+06, 1.161301E+06, 1.193583E+06, 1.226584E+06,
    1.260317E+06, 1.294792E+06, 1.330023E+06, 1.366022E+06, 1.402801E+06, 1.440374E+06,
    1.478753E+06, 1.517952E+06, 1.557983E+06, 1.598860E+06, 1.640596E+06, 1.683205E+06,
    1.726701E+06, 1.771098E+06, 1.816409E+06, 1.862649E+06, 1.909833E+06, 1.957974E+06,
    2.007088E+06, 2.057189E+06,
])

#  ---------------------- M = 19, I = 3 ---------------------------
M = 19
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    7.583710E+00, 1.382173E+02, 2.758282E+02, 4.135070E+02, 5.513676E+02, 6.900135E+02,
    8.308340E+02, 9.758329E+02, 1.127263E+03, 1.287312E+03, 1.458080E+03, 1.641493E+03,
    1.839335E+03, 2.053277E+03, 2.284972E+03, 2.535982E+03, 2.807868E+03, 3.102189E+03,
    3.420546E+03, 3.764540E+03, 4.135842E+03, 4.536154E+03, 4.967247E+03, 5.430907E+03,
    5.929040E+03, 6.463587E+03, 7.036596E+03, 7.650158E+03, 8.306416E+03, 9.007672E+03,
    9.756218E+03, 1.055447E+04, 1.140497E+04, 1.231027E+04, 1.327302E+04, 1.429603E+04,
    1.538213E+04, 1.653432E+04, 1.775555E+04, 1.904896E+04, 2.041788E+04, 2.186555E+04,
    2.339541E+04, 2.501105E+04, 2.671603E+04, 2.851411E+04, 3.040920E+04, 3.240513E+04,
    3.450607E+04, 3.671615E+04, 3.903959E+04, 4.148094E+04, 4.404455E+04, 4.673515E+04,
    4.955745E+04, 5.251625E+04, 5.561666E+04, 5.886377E+04, 6.226278E+04, 6.581899E+04,
    6.953795E+04, 7.342535E+04, 7.748685E+04, 8.172822E+04, 8.615570E+04, 9.077534E+04,
    9.559340E+04, 1.006162E+05, 1.058507E+05, 1.113031E+05, 1.169806E+05, 1.228899E+05,
    1.290384E+05, 1.354334E+05, 1.420821E+05, 1.489923E+05, 1.561716E+05, 1.636280E+05,
    1.713695E+05, 1.794043E+05, 1.877407E+05, 1.963875E+05, 2.053530E+05, 2.146464E+05,
    2.242765E+05, 2.342527E+05, 2.445840E+05, 2.552802E+05, 2.663510E+05, 2.778062E+05,
    2.896558E+05, 3.019102E+05, 3.145796E+05, 3.276747E+05, 3.412063E+05, 3.551851E+05,
    3.696226E+05, 3.845298E+05, 3.999184E+05, 4.158000E+05, 4.321865E+05, 4.490901E+05,
    4.665230E+05, 4.844975E+05, 5.030266E+05, 5.221230E+05, 5.417996E+05, 5.620701E+05,
    5.829476E+05, 6.044458E+05, 6.265791E+05, 6.493608E+05, 6.728060E+05, 6.969287E+05,
    7.217437E+05, 7.472664E+05, 7.735113E+05, 8.004945E+05, 8.282311E+05, 8.567373E+05,
    8.860289E+05, 9.161222E+05, 9.470339E+05, 9.787808E+05, 1.011380E+06, 1.044848E+06,
    1.079203E+06, 1.114462E+06, 1.150644E+06, 1.187766E+06, 1.225847E+06, 1.264906E+06,
    1.304961E+06, 1.346032E+06, 1.388138E+06, 1.431299E+06, 1.475534E+06, 1.520864E+06,
    1.567309E+06, 1.614891E+06, 1.663628E+06, 1.713544E+06, 1.764659E+06, 1.816995E+06,
    1.870575E+06, 1.925419E+06, 1.981552E+06, 2.038995E+06, 2.097772E+06, 2.157905E+06,
    2.219420E+06, 2.282339E+06, 2.346687E+06, 2.412488E+06, 2.479766E+06, 2.548548E+06,
    2.618858E+06, 2.690721E+06, 2.764165E+06, 2.839214E+06, 2.915895E+06, 2.994236E+06,
    3.074263E+06, 3.156003E+06, 3.239484E+06, 3.324735E+06, 3.411783E+06, 3.500657E+06,
    3.591387E+06, 3.683999E+06, 3.778527E+06, 3.874996E+06, 3.973440E+06, 4.073888E+06,
    4.176370E+06, 4.280919E+06,
])

#  ---------------------- M = 19, I = 4 ---------------------------
M = 19
I = 4
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.529716E+01, 2.790462E+02, 5.568808E+02, 8.348524E+02, 1.113124E+03, 1.392726E+03,
    1.676127E+03, 1.967024E+03, 2.269724E+03, 2.588528E+03, 2.927570E+03, 3.290640E+03,
    3.681356E+03, 4.103003E+03, 4.558813E+03, 5.051868E+03, 5.585175E+03, 6.161870E+03,
    6.784985E+03, 7.457593E+03, 8.182895E+03, 8.964139E+03, 9.804763E+03, 1.070818E+04,
    1.167802E+04, 1.271790E+04, 1.383181E+04, 1.502371E+04, 1.629759E+04, 1.765790E+04,
    1.910900E+04, 2.065548E+04, 2.230214E+04, 2.405374E+04, 2.591542E+04, 2.789238E+04,
    2.999010E+04, 3.221416E+04, 3.457015E+04, 3.706422E+04, 3.970237E+04, 4.249085E+04,
    4.543618E+04, 4.854516E+04, 5.182439E+04, 5.528126E+04, 5.892283E+04, 6.275655E+04,
    6.679011E+04, 7.103145E+04, 7.548857E+04, 8.016983E+04, 8.508371E+04, 9.023910E+04,
    9.564456E+04, 1.013096E+05, 1.072436E+05, 1.134560E+05, 1.199569E+05, 1.267562E+05,
    1.338645E+05, 1.412920E+05, 1.490498E+05, 1.571490E+05, 1.656006E+05, 1.744166E+05,
    1.836087E+05, 1.931887E+05, 2.031692E+05, 2.135629E+05, 2.243826E+05, 2.356413E+05,
    2.473527E+05, 2.595302E+05, 2.721881E+05, 2.853405E+05, 2.990019E+05, 3.131871E+05,
    3.279114E+05, 3.431903E+05, 3.590394E+05, 3.754746E+05, 3.925124E+05, 4.101694E+05,
    4.284625E+05, 4.474091E+05, 4.670263E+05, 4.873326E+05, 5.083460E+05, 5.300849E+05,
    5.525683E+05, 5.758153E+05, 5.998453E+05, 6.246787E+05, 6.503350E+05, 6.768352E+05,
    7.042000E+05, 7.324508E+05, 7.616091E+05, 7.916970E+05, 8.227364E+05, 8.547503E+05,
    8.877616E+05, 9.217938E+05, 9.568705E+05, 9.930163E+05, 1.030255E+06, 1.068612E+06,
    1.108113E+06, 1.148782E+06, 1.190646E+06, 1.233733E+06, 1.278068E+06, 1.323678E+06,
    1.370591E+06, 1.418836E+06, 1.468441E+06, 1.519434E+06, 1.571845E+06, 1.625704E+06,
    1.681041E+06, 1.737886E+06, 1.796270E+06, 1.856224E+06, 1.917781E+06, 1.980972E+06,
    2.045832E+06, 2.112391E+06, 2.180684E+06, 2.250746E+06, 2.322610E+06, 2.396312E+06,
    2.471886E+06, 2.549370E+06, 2.628798E+06, 2.710208E+06, 2.793637E+06, 2.879123E+06,
    2.966704E+06, 3.056419E+06, 3.148307E+06, 3.242406E+06, 3.338759E+06, 3.437405E+06,
    3.538386E+06, 3.641742E+06, 3.747517E+06, 3.855753E+06, 3.966492E+06, 4.079779E+06,
    4.195658E+06, 4.314174E+06, 4.435371E+06, 4.559296E+06, 4.685994E+06, 4.815512E+06,
    4.947899E+06, 5.083202E+06, 5.221468E+06, 5.362749E+06, 5.507091E+06, 5.654546E+06,
    5.805164E+06, 5.958997E+06, 6.116096E+06, 6.276514E+06, 6.440302E+06, 6.607517E+06,
    6.778209E+06, 6.952435E+06, 7.130249E+06, 7.311707E+06, 7.496867E+06, 7.685785E+06,
    7.878517E+06, 8.075122E+06,
])

#  ---------------------- M = 19, I = 5 ---------------------------
M = 19
I = 5
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    4.005750E+00, 7.341299E+01, 1.465248E+02, 2.196725E+02, 2.929080E+02, 3.665343E+02,
    4.412543E+02, 5.180980E+02, 5.982340E+02, 6.828353E+02, 7.730027E+02, 8.697668E+02,
    9.740684E+02, 1.086823E+03, 1.208879E+03, 1.341069E+03, 1.484232E+03, 1.639180E+03,
    1.806745E+03, 1.987801E+03, 2.183188E+03, 2.393832E+03, 2.620622E+03, 2.864546E+03,
    3.126558E+03, 3.407697E+03, 3.709043E+03, 4.031658E+03, 4.376687E+03, 4.745342E+03,
    5.138807E+03, 5.558356E+03, 6.005318E+03, 6.481015E+03, 6.986846E+03, 7.524300E+03,
    8.094828E+03, 8.699968E+03, 9.341331E+03, 1.002056E+04, 1.073933E+04, 1.149937E+04,
    1.230251E+04, 1.315057E+04, 1.404547E+04, 1.498914E+04, 1.598358E+04, 1.703092E+04,
    1.813322E+04, 1.929266E+04, 2.051152E+04, 2.179207E+04, 2.313669E+04, 2.454777E+04,
    2.602783E+04, 2.757939E+04, 2.920503E+04, 3.090744E+04, 3.268941E+04, 3.455366E+04,
    3.650309E+04, 3.854064E+04, 4.066930E+04, 4.289217E+04, 4.521234E+04, 4.763307E+04,
    5.015761E+04, 5.278935E+04, 5.553171E+04, 5.838817E+04, 6.136237E+04, 6.445786E+04,
    6.767842E+04, 7.102788E+04, 7.451010E+04, 7.812908E+04, 8.188878E+04, 8.579338E+04,
    8.984714E+04, 9.405423E+04, 9.841911E+04, 1.029462E+05, 1.076400E+05, 1.125052E+05,
    1.175464E+05, 1.227686E+05, 1.281765E+05, 1.337751E+05, 1.395696E+05, 1.455650E+05,
    1.517667E+05, 1.581798E+05, 1.648099E+05, 1.716626E+05, 1.787433E+05, 1.860579E+05,
    1.936120E+05, 2.014118E+05, 2.094632E+05, 2.177721E+05, 2.263450E+05, 2.351881E+05,
    2.443078E+05, 2.537105E+05, 2.634030E+05, 2.733918E+05, 2.836841E+05, 2.942864E+05,
    3.052060E+05, 3.164500E+05, 3.280255E+05, 3.399401E+05, 3.522011E+05, 3.648163E+05,
    3.777931E+05, 3.911396E+05, 4.048633E+05, 4.189728E+05, 4.334758E+05, 4.483808E+05,
    4.636960E+05, 4.794302E+05, 4.955917E+05, 5.121894E+05, 5.292323E+05, 5.467291E+05,
    5.646892E+05, 5.831215E+05, 6.020357E+05, 6.214410E+05, 6.413472E+05, 6.617640E+05,
    6.827011E+05, 7.041687E+05, 7.261769E+05, 7.487358E+05, 7.718559E+05, 7.955477E+05,
    8.198219E+05, 8.446893E+05, 8.701605E+05, 8.962469E+05, 9.229596E+05, 9.503098E+05,
    9.783092E+05, 1.006969E+06, 1.036302E+06, 1.066319E+06, 1.097032E+06, 1.128454E+06,
    1.160596E+06, 1.193472E+06, 1.227094E+06, 1.261474E+06, 1.296626E+06, 1.332563E+06,
    1.369297E+06, 1.406843E+06, 1.445213E+06, 1.484422E+06, 1.524483E+06, 1.565410E+06,
    1.607217E+06, 1.649919E+06, 1.693530E+06, 1.738064E+06, 1.783536E+06, 1.829962E+06,
    1.877355E+06, 1.925733E+06, 1.975109E+06, 2.025500E+06, 2.076920E+06, 2.129387E+06,
    2.182916E+06, 2.237523E+06,
])

#  ---------------------- M = 20, I = 1 ---------------------------
M = 20
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.091110E+00, 4.947081E+01, 1.416480E+02, 2.592507E+02, 3.983023E+02, 5.559709E+02,
    7.302920E+02, 9.198257E+02, 1.123477E+03, 1.340418E+03, 1.570058E+03, 1.812038E+03,
    2.066230E+03, 2.332728E+03, 2.611836E+03, 2.904051E+03, 3.210045E+03, 3.530639E+03,
    3.866789E+03, 4.219566E+03, 4.590146E+03, 4.979793E+03, 5.389860E+03, 5.821776E+03,
    6.277046E+03, 6.757251E+03, 7.264045E+03, 7.799159E+03, 8.364401E+03, 8.961659E+03,
    9.592904E+03, 1.026019E+04, 1.096512E+04, 1.170966E+04, 1.249659E+04, 1.332840E+04,
    1.420757E+04, 1.513680E+04, 1.611862E+04, 1.715616E+04, 1.825218E+04, 1.940974E+04,
    2.063218E+04, 2.192282E+04, 2.328522E+04, 2.472293E+04, 2.623979E+04, 2.783973E+04,
    2.952702E+04, 3.130573E+04, 3.318041E+04, 3.515575E+04, 3.723663E+04, 3.942790E+04,
    4.173473E+04, 4.416289E+04, 4.671773E+04, 4.940519E+04, 5.223118E+04, 5.520223E+04,
    5.832512E+04, 6.160614E+04, 6.505245E+04, 6.867173E+04, 7.247112E+04, 7.645903E+04,
    8.064278E+04, 8.503157E+04, 8.963402E+04, 9.445875E+04, 9.951574E+04, 1.048142E+05,
    1.103643E+05, 1.161765E+05, 1.222616E+05, 1.286309E+05, 1.352955E+05, 1.422674E+05,
    1.495587E+05, 1.571827E+05, 1.651519E+05, 1.734802E+05, 1.821813E+05, 1.912699E+05,
    2.007604E+05, 2.106690E+05, 2.210108E+05, 2.318027E+05, 2.430609E+05, 2.548036E+05,
    2.670483E+05, 2.798136E+05, 2.931184E+05, 3.069823E+05, 3.214256E+05, 3.364693E+05,
    3.521348E+05, 3.684439E+05, 3.854199E+05, 4.030855E+05, 4.214650E+05, 4.405834E+05,
    4.604660E+05, 4.811396E+05, 5.026303E+05, 5.249660E+05, 5.481756E+05, 5.722878E+05,
    5.973333E+05, 6.233430E+05, 6.503482E+05, 6.783817E+05, 7.074775E+05, 7.376698E+05,
    7.689940E+05, 8.014861E+05, 8.351831E+05, 8.701245E+05, 9.063479E+05, 9.438955E+05,
    9.828070E+05, 1.023125E+06, 1.064895E+06, 1.108158E+06, 1.152963E+06, 1.199355E+06,
    1.247383E+06, 1.297095E+06, 1.348543E+06, 1.401778E+06, 1.456853E+06, 1.513823E+06,
    1.572742E+06, 1.633669E+06, 1.696660E+06, 1.761777E+06, 1.829081E+06, 1.898632E+06,
    1.970497E+06, 2.044741E+06, 2.121430E+06, 2.200634E+06, 2.282422E+06, 2.366867E+06,
    2.454042E+06, 2.544023E+06, 2.636886E+06, 2.732710E+06, 2.831576E+06, 2.933566E+06,
    3.038763E+06, 3.147255E+06, 3.259128E+06, 3.374473E+06, 3.493381E+06, 3.615946E+06,
    3.742264E+06, 3.872434E+06, 4.006554E+06, 4.144726E+06, 4.287056E+06, 4.433650E+06,
    4.584616E+06, 4.740066E+06, 4.900110E+06, 5.064870E+06, 5.234458E+06, 5.408999E+06,
    5.588613E+06, 5.773427E+06, 5.963569E+06, 6.159169E+06, 6.360361E+06, 6.567283E+06,
    6.780068E+06, 6.998864E+06,
])

#  ---------------------- M = 20, I = 2 ---------------------------
M = 20
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.519830E+00, 1.041424E+02, 2.906766E+02, 5.317342E+02, 8.170004E+02, 1.140498E+03,
    1.498191E+03, 1.887134E+03, 2.305068E+03, 2.750313E+03, 3.221668E+03, 3.718356E+03,
    4.240095E+03, 4.787152E+03, 5.360055E+03, 5.959825E+03, 6.587858E+03, 7.245886E+03,
    7.935732E+03, 8.659593E+03, 9.420003E+03, 1.021942E+04, 1.106064E+04, 1.194646E+04,
    1.288013E+04, 1.386475E+04, 1.490370E+04, 1.600056E+04, 1.715893E+04, 1.838256E+04,
    1.967551E+04, 2.104202E+04, 2.248633E+04, 2.401319E+04, 2.562698E+04, 2.733280E+04,
    2.913575E+04, 3.104133E+04, 3.305479E+04, 3.518252E+04, 3.743016E+04, 3.980401E+04,
    4.231092E+04, 4.495767E+04, 4.775159E+04, 5.069997E+04, 5.381064E+04, 5.709169E+04,
    6.055190E+04, 6.419956E+04, 6.804403E+04, 7.209494E+04, 7.636229E+04, 8.085600E+04,
    8.558671E+04, 9.056622E+04, 9.580554E+04, 1.013168E+05, 1.071122E+05, 1.132050E+05,
    1.196093E+05, 1.263378E+05, 1.334053E+05, 1.408275E+05, 1.486191E+05, 1.567972E+05,
    1.653770E+05, 1.743773E+05, 1.838157E+05, 1.937100E+05, 2.040806E+05, 2.149463E+05,
    2.263282E+05, 2.382475E+05, 2.507265E+05, 2.637882E+05, 2.774557E+05, 2.917531E+05,
    3.067057E+05, 3.223406E+05, 3.386834E+05, 3.557626E+05, 3.736064E+05, 3.922448E+05,
    4.117074E+05, 4.320273E+05, 4.532358E+05, 4.753672E+05, 4.984549E+05, 5.225361E+05,
    5.476468E+05, 5.738251E+05, 6.011099E+05, 6.295411E+05, 6.591607E+05, 6.900115E+05,
    7.221373E+05, 7.555835E+05, 7.903966E+05, 8.266243E+05, 8.643161E+05, 9.035230E+05,
    9.442971E+05, 9.866935E+05, 1.030765E+06, 1.076570E+06, 1.124167E+06, 1.173615E+06,
    1.224977E+06, 1.278317E+06, 1.333697E+06, 1.391187E+06, 1.450855E+06, 1.512772E+06,
    1.577010E+06, 1.643643E+06, 1.712747E+06, 1.784403E+06, 1.858688E+06, 1.935688E+06,
    2.015486E+06, 2.098169E+06, 2.183827E+06, 2.272550E+06, 2.364432E+06, 2.459570E+06,
    2.558063E+06, 2.660012E+06, 2.765517E+06, 2.874690E+06, 2.987634E+06, 3.104464E+06,
    3.225293E+06, 3.350239E+06, 3.479418E+06, 3.612957E+06, 3.750979E+06, 3.893612E+06,
    4.040990E+06, 4.193245E+06, 4.350513E+06, 4.512941E+06, 4.680667E+06, 4.853844E+06,
    5.032618E+06, 5.217145E+06, 5.407584E+06, 5.604095E+06, 5.806845E+06, 6.016000E+06,
    6.231733E+06, 6.454222E+06, 6.683645E+06, 6.920189E+06, 7.164040E+06, 7.415389E+06,
    7.674437E+06, 7.941381E+06, 8.216427E+06, 8.499785E+06, 8.791667E+06, 9.092294E+06,
    9.401888E+06, 9.720676E+06, 1.004889E+07, 1.038677E+07, 1.073455E+07, 1.109249E+07,
    1.146083E+07, 1.183984E+07, 1.222977E+07, 1.263090E+07, 1.304350E+07, 1.346784E+07,
    1.390421E+07, 1.435290E+07,
])

#  ---------------------- M = 20, I = 3 ---------------------------
M = 20
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    7.773000E-01, 5.324867E+01, 1.486677E+02, 2.719834E+02, 4.179176E+02, 5.834117E+02,
    7.664008E+02, 9.653781E+02, 1.179188E+03, 1.406969E+03, 1.648109E+03, 1.902210E+03,
    2.169127E+03, 2.448996E+03, 2.742088E+03, 3.048926E+03, 3.370223E+03, 3.706865E+03,
    4.059785E+03, 4.430108E+03, 4.819128E+03, 5.228105E+03, 5.658465E+03, 6.111648E+03,
    6.589308E+03, 7.093032E+03, 7.624553E+03, 8.185700E+03, 8.778316E+03, 9.404317E+03,
    1.006578E+04, 1.076488E+04, 1.150378E+04, 1.228491E+04, 1.311052E+04, 1.398321E+04,
    1.490558E+04, 1.588047E+04, 1.691055E+04, 1.799908E+04, 1.914896E+04, 2.036341E+04,
    2.164594E+04, 2.300000E+04, 2.442935E+04, 2.593773E+04, 2.752913E+04, 2.920771E+04,
    3.097792E+04, 3.284406E+04, 3.481086E+04, 3.688330E+04, 3.906645E+04, 4.136541E+04,
    4.378561E+04, 4.633311E+04, 4.901353E+04, 5.183307E+04, 5.479795E+04, 5.791503E+04,
    6.119141E+04, 6.463369E+04, 6.824939E+04, 7.204656E+04, 7.603270E+04, 8.021661E+04,
    8.460600E+04, 8.921050E+04, 9.403916E+04, 9.910103E+04, 1.044066E+05, 1.099654E+05,
    1.157884E+05, 1.218862E+05, 1.282704E+05, 1.349528E+05, 1.419450E+05, 1.492595E+05,
    1.569092E+05, 1.649080E+05, 1.732688E+05, 1.820065E+05, 1.911353E+05, 2.006707E+05,
    2.106276E+05, 2.210232E+05, 2.318735E+05, 2.431958E+05, 2.550074E+05, 2.673273E+05,
    2.801738E+05, 2.935665E+05, 3.075254E+05, 3.220707E+05, 3.372240E+05, 3.530071E+05,
    3.694426E+05, 3.865535E+05, 4.043638E+05, 4.228977E+05, 4.421807E+05, 4.622389E+05,
    4.830988E+05, 5.047886E+05, 5.273357E+05, 5.507693E+05, 5.751197E+05, 6.004172E+05,
    6.266939E+05, 6.539820E+05, 6.823147E+05, 7.117262E+05, 7.422522E+05, 7.739285E+05,
    8.067927E+05, 8.408819E+05, 8.762352E+05, 9.128942E+05, 9.508983E+05, 9.902916E+05,
    1.031116E+06, 1.073416E+06, 1.117238E+06, 1.162629E+06, 1.209636E+06, 1.258308E+06,
    1.308697E+06, 1.360853E+06, 1.414829E+06, 1.470682E+06, 1.528464E+06, 1.588234E+06,
    1.650050E+06, 1.713972E+06, 1.780059E+06, 1.848378E+06, 1.918989E+06, 1.991960E+06,
    2.067358E+06, 2.145251E+06, 2.225709E+06, 2.308807E+06, 2.394615E+06, 2.483212E+06,
    2.574672E+06, 2.669076E+06, 2.766504E+06, 2.867038E+06, 2.970764E+06, 3.077768E+06,
    3.188136E+06, 3.301961E+06, 3.419333E+06, 3.540348E+06, 3.665102E+06, 3.793692E+06,
    3.926220E+06, 4.062788E+06, 4.203501E+06, 4.348466E+06, 4.497792E+06, 4.651592E+06,
    4.809980E+06, 4.973071E+06, 5.140983E+06, 5.313842E+06, 5.491767E+06, 5.674888E+06,
    5.863331E+06, 6.057231E+06, 6.256719E+06, 6.461935E+06, 6.673017E+06, 6.890111E+06,
    7.113357E+06, 7.342907E+06,
])

#  ---------------------- M = 21, I = 1 ---------------------------
M = 21
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    3.981460E+00, 3.304167E+02, 9.292313E+02, 1.704088E+03, 2.621559E+03, 3.662421E+03,
    4.814121E+03, 6.068541E+03, 7.421071E+03, 8.869781E+03, 1.041512E+04, 1.205911E+04,
    1.380519E+04, 1.565730E+04, 1.762056E+04, 1.970014E+04, 2.190181E+04, 2.423163E+04,
    2.669554E+04, 2.929991E+04, 3.205132E+04, 3.495586E+04, 3.802056E+04, 4.125226E+04,
    4.465708E+04, 4.824259E+04, 5.201564E+04, 5.598309E+04, 6.015219E+04, 6.453072E+04,
    6.912550E+04, 7.394438E+04, 7.899525E+04, 8.428610E+04, 8.982435E+04, 9.561869E+04,
    1.016773E+05, 1.080083E+05, 1.146215E+05, 1.215237E+05, 1.287266E+05, 1.362371E+05,
    1.440654E+05, 1.522203E+05, 1.607132E+05, 1.695523E+05, 1.787484E+05, 1.883127E+05,
    1.982535E+05, 2.085833E+05, 2.193115E+05, 2.304511E+05, 2.420107E+05, 2.540037E+05,
    2.664413E+05, 2.793348E+05, 2.926958E+05, 3.065372E+05, 3.208723E+05, 3.357116E+05,
    3.510703E+05, 3.669592E+05, 3.833922E+05, 4.003837E+05, 4.179467E+05, 4.360939E+05,
    4.548404E+05, 4.741996E+05, 4.941865E+05, 5.148171E+05, 5.361032E+05, 5.580627E+05,
    5.807100E+05, 6.040616E+05, 6.281301E+05, 6.529321E+05, 6.784872E+05, 7.048083E+05,
    7.319154E+05, 7.598219E+05, 7.885455E+05, 8.181023E+05, 8.485133E+05, 8.797949E+05,
    9.119639E+05, 9.450366E+05, 9.790353E+05, 1.013977E+06, 1.049882E+06, 1.086768E+06,
    1.124652E+06, 1.163556E+06, 1.203500E+06, 1.244505E+06, 1.286588E+06, 1.329770E+06,
    1.374075E+06, 1.419521E+06, 1.466130E+06, 1.513921E+06, 1.562920E+06, 1.613148E+06,
    1.664626E+06, 1.717375E+06, 1.771419E+06, 1.826780E+06, 1.883484E+06, 1.941552E+06,
    2.001009E+06, 2.061877E+06, 2.124181E+06, 2.187948E+06, 2.253196E+06, 2.319956E+06,
    2.388251E+06, 2.458107E+06, 2.529549E+06, 2.602607E+06, 2.677298E+06, 2.753658E+06,
    2.831706E+06, 2.911473E+06, 2.992987E+06, 3.076272E+06, 3.161364E+06, 3.248277E+06,
    3.337053E+06, 3.427717E+06, 3.520289E+06, 3.614807E+06, 3.711301E+06, 3.809798E+06,
    3.910326E+06, 4.012918E+06, 4.117603E+06, 4.224413E+06, 4.333378E+06, 4.444528E+06,
    4.557903E+06, 4.673519E+06, 4.791422E+06, 4.911642E+06, 5.034208E+06, 5.159155E+06,
    5.286513E+06, 5.416319E+06, 5.548609E+06, 5.683412E+06, 5.820768E+06, 5.960706E+06,
    6.103265E+06, 6.248481E+06, 6.396387E+06, 6.547017E+06, 6.700413E+06, 6.856604E+06,
    7.015634E+06, 7.177539E+06, 7.342350E+06, 7.510115E+06, 7.680862E+06, 7.854632E+06,
    8.031473E+06, 8.211412E+06, 8.394491E+06, 8.580756E+06, 8.770230E+06, 8.962977E+06,
    9.159022E+06, 9.358408E+06, 9.561181E+06, 9.767369E+06, 9.977034E+06, 1.019021E+07,
    1.040692E+07, 1.062723E+07,
])

#  ---------------------- M = 21, I = 2 ---------------------------
M = 21
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    4.058120E+00, 3.362054E+02, 9.456074E+02, 1.734179E+03, 2.667896E+03, 3.727193E+03,
    4.899294E+03, 6.175937E+03, 7.552431E+03, 9.026809E+03, 1.059953E+04, 1.227265E+04,
    1.404968E+04, 1.593461E+04, 1.793265E+04, 2.004908E+04, 2.228977E+04, 2.466088E+04,
    2.716846E+04, 2.981898E+04, 3.261916E+04, 3.557517E+04, 3.869418E+04, 4.198316E+04,
    4.544832E+04, 4.909737E+04, 5.293730E+04, 5.697506E+04, 6.121805E+04, 6.567419E+04,
    7.035040E+04, 7.525467E+04, 8.039507E+04, 8.577969E+04, 9.141609E+04, 9.731312E+04,
    1.034791E+05, 1.099223E+05, 1.166527E+05, 1.236773E+05, 1.310079E+05, 1.386514E+05,
    1.466185E+05, 1.549179E+05, 1.635614E+05, 1.725571E+05, 1.819162E+05, 1.916501E+05,
    2.017671E+05, 2.122799E+05, 2.231982E+05, 2.345353E+05, 2.462997E+05, 2.585053E+05,
    2.711634E+05, 2.842854E+05, 2.978833E+05, 3.119700E+05, 3.265592E+05, 3.416615E+05,
    3.572924E+05, 3.734629E+05, 3.901872E+05, 4.074799E+05, 4.253541E+05, 4.438230E+05,
    4.629018E+05, 4.826041E+05, 5.029453E+05, 5.239416E+05, 5.456049E+05, 5.679538E+05,
    5.910025E+05, 6.147679E+05, 6.392630E+05, 6.645047E+05, 6.905128E+05, 7.173004E+05,
    7.448879E+05, 7.732890E+05, 8.025217E+05, 8.326025E+05, 8.635525E+05, 8.953886E+05,
    9.281278E+05, 9.617867E+05, 9.963880E+05, 1.031949E+06, 1.068491E+06, 1.106030E+06,
    1.144586E+06, 1.184179E+06, 1.224831E+06, 1.266563E+06, 1.309392E+06, 1.353339E+06,
    1.398430E+06, 1.444682E+06, 1.492116E+06, 1.540754E+06, 1.590622E+06, 1.641741E+06,
    1.694130E+06, 1.747816E+06, 1.802817E+06, 1.859159E+06, 1.916869E+06, 1.975965E+06,
    2.036476E+06, 2.098423E+06, 2.161832E+06, 2.226729E+06, 2.293134E+06, 2.361077E+06,
    2.430583E+06, 2.501677E+06, 2.574385E+06, 2.648739E+06, 2.724753E+06, 2.802466E+06,
    2.881898E+06, 2.963079E+06, 3.046038E+06, 3.130799E+06, 3.217399E+06, 3.305853E+06,
    3.396203E+06, 3.488473E+06, 3.582687E+06, 3.678880E+06, 3.777084E+06, 3.877327E+06,
    3.979638E+06, 4.084047E+06, 4.190589E+06, 4.299291E+06, 4.410188E+06, 4.523309E+06,
    4.638693E+06, 4.756358E+06, 4.876351E+06, 4.998702E+06, 5.123440E+06, 5.250602E+06,
    5.380218E+06, 5.512325E+06, 5.646960E+06, 5.784152E+06, 5.923943E+06, 6.066361E+06,
    6.211446E+06, 6.359237E+06, 6.509766E+06, 6.663065E+06, 6.819180E+06, 6.978140E+06,
    7.139989E+06, 7.304764E+06, 7.472497E+06, 7.643236E+06, 7.817009E+06, 7.993858E+06,
    8.173833E+06, 8.356963E+06, 8.543287E+06, 8.732853E+06, 8.925687E+06, 9.121849E+06,
    9.321369E+06, 9.524290E+06, 9.730657E+06, 9.940501E+06, 1.015388E+07, 1.037083E+07,
    1.059139E+07, 1.081561E+07,
])

#  ---------------------- M = 22, I = 1 ---------------------------
M = 22
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[3]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    6.029370E+00, 3.298562E+01, 6.440722E+01, 9.584644E+01, 1.272919E+02, 1.587414E+02,
    1.901941E+02, 2.216498E+02, 2.531083E+02, 2.845695E+02, 3.160334E+02, 3.475000E+02,
    3.789694E+02, 4.104418E+02, 4.419180E+02, 4.733988E+02, 5.048860E+02, 5.363820E+02,
    5.678901E+02, 5.994148E+02, 6.309613E+02, 6.625362E+02, 6.941467E+02, 7.258012E+02,
    7.575086E+02, 7.892788E+02, 8.211221E+02, 8.530492E+02, 8.850710E+02, 9.171988E+02,
    9.494439E+02, 9.818175E+02, 1.014331E+03, 1.046995E+03, 1.079820E+03, 1.112818E+03,
    1.145998E+03, 1.179370E+03, 1.212943E+03, 1.246728E+03, 1.280732E+03, 1.314964E+03,
    1.349432E+03, 1.384144E+03, 1.419107E+03, 1.454327E+03, 1.489813E+03, 1.525569E+03,
    1.561602E+03, 1.597917E+03, 1.634520E+03, 1.671416E+03, 1.708610E+03, 1.746106E+03,
    1.783909E+03, 1.822022E+03, 1.860449E+03, 1.899195E+03, 1.938262E+03, 1.977654E+03,
    2.017374E+03, 2.057425E+03, 2.097810E+03, 2.138531E+03, 2.179590E+03, 2.220991E+03,
    2.262736E+03, 2.304826E+03, 2.347264E+03, 2.390051E+03, 2.433189E+03, 2.476681E+03,
    2.520527E+03, 2.564729E+03, 2.609289E+03, 2.654208E+03, 2.699487E+03, 2.745128E+03,
    2.791131E+03, 2.837498E+03, 2.884230E+03, 2.931327E+03, 2.978791E+03, 3.026622E+03,
    3.074822E+03, 3.123390E+03, 3.172328E+03, 3.221636E+03, 3.271315E+03, 3.321366E+03,
    3.371788E+03, 3.422582E+03, 3.473749E+03, 3.525290E+03, 3.577203E+03, 3.629490E+03,
    3.682151E+03, 3.735186E+03, 3.788596E+03, 3.842379E+03, 3.896538E+03, 3.951070E+03,
    4.005977E+03, 4.061259E+03, 4.116915E+03, 4.172946E+03, 4.229351E+03, 4.286130E+03,
    4.343283E+03, 4.400810E+03, 4.458710E+03, 4.516984E+03, 4.575631E+03, 4.634651E+03,
    4.694043E+03, 4.753807E+03, 4.813944E+03, 4.874451E+03, 4.935329E+03, 4.996578E+03,
    5.058196E+03, 5.120184E+03, 5.182541E+03, 5.245267E+03, 5.308360E+03, 5.371821E+03,
    5.435648E+03, 5.499841E+03, 5.564399E+03, 5.629323E+03, 5.694610E+03, 5.760261E+03,
    5.826274E+03, 5.892649E+03, 5.959385E+03, 6.026482E+03, 6.093938E+03, 6.161753E+03,
    6.229926E+03, 6.298457E+03, 6.367343E+03, 6.436585E+03, 6.506181E+03, 6.576131E+03,
    6.646434E+03, 6.717089E+03, 6.788095E+03, 6.859450E+03, 6.931155E+03, 7.003207E+03,
    7.075607E+03, 7.148353E+03, 7.221444E+03, 7.294879E+03, 7.368657E+03, 7.442778E+03,
    7.517239E+03, 7.592040E+03, 7.667180E+03, 7.742658E+03, 7.818473E+03, 7.894623E+03,
    7.971108E+03, 8.047926E+03, 8.125077E+03, 8.202560E+03, 8.280372E+03, 8.358513E+03,
    8.436983E+03, 8.515779E+03, 8.594900E+03, 8.674346E+03, 8.754116E+03, 8.834207E+03,
    8.914620E+03, 8.995352E+03, 9.076403E+03, 9.157772E+03, 9.239456E+03, 9.321456E+03,
    9.403770E+03, 9.486396E+03, 9.569334E+03, 9.652582E+03, 9.736138E+03, 9.820003E+03,
    9.904175E+03, 9.988651E+03, 1.007343E+04, 1.015852E+04, 1.024390E+04, 1.032959E+04,
    1.041557E+04, 1.050186E+04, 1.058844E+04, 1.067531E+04, 1.076248E+04, 1.084994E+04,
    1.093770E+04, 1.102574E+04, 1.111408E+04, 1.120270E+04, 1.129161E+04, 1.138080E+04,
    1.147028E+04, 1.156004E+04, 1.165009E+04, 1.174041E+04, 1.183101E+04, 1.192189E+04,
    1.201305E+04, 1.210448E+04, 1.219619E+04, 1.228817E+04, 1.238042E+04, 1.247295E+04,
    1.256574E+04, 1.265880E+04, 1.275212E+04, 1.284571E+04, 1.293957E+04, 1.303369E+04,
    1.312807E+04, 1.322271E+04, 1.331760E+04, 1.341276E+04, 1.350817E+04, 1.360384E+04,
    1.369976E+04, 1.379594E+04, 1.389237E+04, 1.398904E+04, 1.408597E+04, 1.418314E+04,
    1.428056E+04, 1.437823E+04, 1.447614E+04, 1.457429E+04, 1.467269E+04, 1.477132E+04,
    1.487019E+04, 1.496930E+04, 1.506865E+04, 1.516823E+04, 1.526805E+04, 1.536810E+04,
    1.546838E+04, 1.556890E+04, 1.566964E+04, 1.577061E+04, 1.587180E+04, 1.597322E+04,
    1.607487E+04, 1.617674E+04, 1.627883E+04, 1.638114E+04, 1.648367E+04, 1.658642E+04,
    1.668939E+04, 1.679257E+04, 1.689597E+04, 1.699958E+04, 1.710340E+04, 1.720744E+04,
    1.731168E+04, 1.741613E+04, 1.752079E+04, 1.762566E+04, 1.773073E+04, 1.783601E+04,
    1.794149E+04, 1.804717E+04, 1.815305E+04, 1.825913E+04, 1.836541E+04, 1.847189E+04,
    1.857856E+04, 1.868542E+04, 1.879248E+04, 1.889974E+04, 1.900718E+04, 1.911481E+04,
    1.922264E+04, 1.933065E+04, 1.943885E+04, 1.954723E+04, 1.965580E+04, 1.976455E+04,
    1.987348E+04, 1.998260E+04, 2.009189E+04, 2.020137E+04, 2.031102E+04, 2.042085E+04,
    2.053085E+04, 2.064103E+04, 2.075138E+04, 2.086191E+04, 2.097260E+04, 2.108347E+04,
    2.119450E+04, 2.130571E+04, 2.141708E+04, 2.152862E+04, 2.164032E+04, 2.175218E+04,
    2.186421E+04, 2.197640E+04, 2.208875E+04, 2.220126E+04, 2.231393E+04, 2.242675E+04,
    2.253974E+04, 2.265287E+04, 2.276617E+04, 2.287961E+04, 2.299321E+04, 2.310696E+04,
    2.322086E+04, 2.333491E+04, 2.344911E+04, 2.356345E+04, 2.367794E+04, 2.379258E+04,
    2.390736E+04, 2.402228E+04, 2.413735E+04, 2.425256E+04, 2.436791E+04, 2.448339E+04,
    2.459902E+04, 2.471478E+04, 2.483069E+04, 2.494672E+04, 2.506289E+04, 2.517920E+04,
    2.529563E+04, 2.541220E+04, 2.552890E+04, 2.564573E+04, 2.576269E+04, 2.587978E+04,
    2.599699E+04, 2.611433E+04, 2.623180E+04, 2.634939E+04, 2.646710E+04, 2.658493E+04,
    2.670289E+04, 2.682097E+04, 2.693917E+04, 2.705748E+04, 2.717592E+04, 2.729447E+04,
    2.741314E+04, 2.753192E+04, 2.765082E+04, 2.776984E+04, 2.788896E+04, 2.800820E+04,
    2.812754E+04, 2.824700E+04, 2.836657E+04, 2.848625E+04, 2.860603E+04, 2.872592E+04,
    2.884592E+04, 2.896602E+04, 2.908623E+04, 2.920654E+04, 2.932695E+04, 2.944746E+04,
    2.956808E+04, 2.968880E+04, 2.980961E+04, 2.993053E+04, 3.005154E+04, 3.017265E+04,
    3.029385E+04, 3.041515E+04, 3.053655E+04, 3.065804E+04, 3.077962E+04, 3.090129E+04,
    3.102306E+04, 3.114492E+04, 3.126686E+04, 3.138890E+04, 3.151103E+04, 3.163324E+04,
    3.175554E+04, 3.187793E+04, 3.200040E+04, 3.212295E+04, 3.224560E+04, 3.236832E+04,
    3.249113E+04, 3.261402E+04, 3.273698E+04, 3.286004E+04, 3.298317E+04, 3.310637E+04,
    3.322966E+04, 3.335303E+04, 3.347647E+04, 3.359999E+04, 3.372358E+04, 3.384725E+04,
    3.397099E+04, 3.409481E+04, 3.421870E+04, 3.434266E+04, 3.446669E+04, 3.459079E+04,
    3.471497E+04, 3.483921E+04, 3.496352E+04, 3.508790E+04, 3.521235E+04, 3.533686E+04,
    3.546144E+04, 3.558609E+04, 3.571080E+04, 3.583558E+04, 3.596041E+04, 3.608532E+04,
    3.621028E+04, 3.633531E+04, 3.646039E+04, 3.658554E+04, 3.671075E+04, 3.683601E+04,
    3.696134E+04, 3.708672E+04, 3.721216E+04, 3.733766E+04, 3.746321E+04, 3.758882E+04,
    3.771449E+04, 3.784021E+04, 3.796598E+04, 3.809181E+04, 3.821769E+04, 3.834362E+04,
    3.846960E+04, 3.859563E+04, 3.872172E+04, 3.884785E+04, 3.897404E+04, 3.910027E+04,
    3.922655E+04,
])

#  ---------------------- M = 22, I = 2 ---------------------------
M = 22
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[3]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    6.071020E+00, 4.541678E+01, 8.875137E+01, 1.321089E+02, 1.754746E+02, 2.188458E+02,
    2.622213E+02, 3.056010E+02, 3.489845E+02, 3.923717E+02, 4.357626E+02, 4.791572E+02,
    5.225558E+02, 5.659587E+02, 6.093669E+02, 6.527822E+02, 6.962070E+02, 7.396453E+02,
    7.831022E+02, 8.265842E+02, 8.700997E+02, 9.136581E+02, 9.572704E+02, 1.000949E+03,
    1.044707E+03, 1.088558E+03, 1.132518E+03, 1.176603E+03, 1.220827E+03, 1.265207E+03,
    1.309760E+03, 1.354500E+03, 1.399444E+03, 1.444608E+03, 1.490006E+03, 1.535654E+03,
    1.581565E+03, 1.627754E+03, 1.674233E+03, 1.721016E+03, 1.768115E+03, 1.815541E+03,
    1.863305E+03, 1.911418E+03, 1.959890E+03, 2.008730E+03, 2.057948E+03, 2.107552E+03,
    2.157550E+03, 2.207950E+03, 2.258760E+03, 2.309985E+03, 2.361634E+03, 2.413711E+03,
    2.466223E+03, 2.519175E+03, 2.572573E+03, 2.626421E+03, 2.680724E+03, 2.735487E+03,
    2.790713E+03, 2.846407E+03, 2.902573E+03, 2.959214E+03, 3.016333E+03, 3.073933E+03,
    3.132018E+03, 3.190590E+03, 3.249653E+03, 3.309208E+03, 3.369258E+03, 3.429805E+03,
    3.490852E+03, 3.552400E+03, 3.614451E+03, 3.677007E+03, 3.740070E+03, 3.803642E+03,
    3.867723E+03, 3.932316E+03, 3.997421E+03, 4.063040E+03, 4.129174E+03, 4.195823E+03,
    4.262990E+03, 4.330675E+03, 4.398878E+03, 4.467600E+03, 4.536843E+03, 4.606606E+03,
    4.676891E+03, 4.747698E+03, 4.819026E+03, 4.890878E+03, 4.963252E+03, 5.036150E+03,
    5.109571E+03, 5.183515E+03, 5.257983E+03, 5.332975E+03, 5.408491E+03, 5.484530E+03,
    5.561093E+03, 5.638179E+03, 5.715789E+03, 5.793921E+03, 5.872577E+03, 5.951754E+03,
    6.031454E+03, 6.111676E+03, 6.192419E+03, 6.273682E+03, 6.355466E+03, 6.437769E+03,
    6.520592E+03, 6.603933E+03, 6.687791E+03, 6.772167E+03, 6.857060E+03, 6.942467E+03,
    7.028390E+03, 7.114827E+03, 7.201776E+03, 7.289238E+03, 7.377212E+03, 7.465695E+03,
    7.554688E+03, 7.644190E+03, 7.734198E+03, 7.824713E+03, 7.915734E+03, 8.007258E+03,
    8.099285E+03, 8.191814E+03, 8.284844E+03, 8.378373E+03, 8.472401E+03, 8.566925E+03,
    8.661945E+03, 8.757460E+03, 8.853467E+03, 8.949967E+03, 9.046956E+03, 9.144435E+03,
    9.242402E+03, 9.340855E+03, 9.439793E+03, 9.539214E+03, 9.639117E+03, 9.739501E+03,
    9.840364E+03, 9.941705E+03, 1.004352E+04, 1.014581E+04, 1.024858E+04, 1.035181E+04,
    1.045552E+04, 1.055969E+04, 1.066433E+04, 1.076944E+04, 1.087501E+04, 1.098104E+04,
    1.108753E+04, 1.119448E+04, 1.130189E+04, 1.140975E+04, 1.151807E+04, 1.162684E+04,
    1.173606E+04, 1.184573E+04, 1.195584E+04, 1.206641E+04, 1.217741E+04, 1.228886E+04,
    1.240074E+04, 1.251307E+04, 1.262583E+04, 1.273903E+04, 1.285266E+04, 1.296672E+04,
    1.308121E+04, 1.319613E+04, 1.331148E+04, 1.342725E+04, 1.354344E+04, 1.366005E+04,
    1.377709E+04, 1.389454E+04, 1.401240E+04, 1.413069E+04, 1.424938E+04, 1.436848E+04,
    1.448799E+04, 1.460791E+04, 1.472823E+04, 1.484896E+04, 1.497009E+04, 1.509161E+04,
    1.521354E+04, 1.533586E+04, 1.545857E+04, 1.558168E+04, 1.570518E+04, 1.582906E+04,
    1.595333E+04, 1.607799E+04, 1.620303E+04, 1.632846E+04, 1.645426E+04, 1.658044E+04,
    1.670699E+04, 1.683392E+04, 1.696123E+04, 1.708890E+04, 1.721694E+04, 1.734535E+04,
    1.747412E+04, 1.760326E+04, 1.773276E+04, 1.786262E+04, 1.799284E+04, 1.812341E+04,
    1.825434E+04, 1.838562E+04, 1.851725E+04, 1.864923E+04, 1.878155E+04, 1.891423E+04,
    1.904724E+04, 1.918060E+04, 1.931430E+04, 1.944833E+04, 1.958271E+04, 1.971742E+04,
    1.985246E+04, 1.998783E+04, 2.012353E+04, 2.025956E+04, 2.039591E+04, 2.053259E+04,
    2.066960E+04, 2.080692E+04, 2.094456E+04, 2.108252E+04, 2.122079E+04, 2.135938E+04,
    2.149828E+04, 2.163749E+04, 2.177701E+04, 2.191683E+04, 2.205696E+04, 2.219740E+04,
    2.233813E+04, 2.247917E+04, 2.262050E+04, 2.276213E+04, 2.290406E+04, 2.304628E+04,
    2.318879E+04, 2.333159E+04, 2.347468E+04, 2.361805E+04, 2.376171E+04, 2.390565E+04,
    2.404987E+04, 2.419438E+04, 2.433916E+04, 2.448422E+04, 2.462955E+04, 2.477516E+04,
    2.492103E+04, 2.506718E+04, 2.521360E+04, 2.536028E+04, 2.550723E+04, 2.565444E+04,
    2.580191E+04, 2.594964E+04, 2.609763E+04, 2.624588E+04, 2.639439E+04, 2.654314E+04,
    2.669215E+04, 2.684141E+04, 2.699092E+04, 2.714068E+04, 2.729068E+04, 2.744093E+04,
    2.759142E+04, 2.774215E+04, 2.789312E+04, 2.804433E+04, 2.819578E+04, 2.834746E+04,
    2.849937E+04, 2.865152E+04, 2.880390E+04, 2.895650E+04, 2.910934E+04, 2.926240E+04,
    2.941568E+04, 2.956919E+04, 2.972292E+04, 2.987687E+04, 3.003104E+04, 3.018542E+04,
    3.034002E+04, 3.049484E+04, 3.064987E+04, 3.080511E+04, 3.096055E+04, 3.111621E+04,
    3.127208E+04, 3.142815E+04, 3.158442E+04, 3.174090E+04, 3.189758E+04, 3.205446E+04,
    3.221153E+04, 3.236881E+04, 3.252628E+04, 3.268394E+04, 3.284180E+04, 3.299985E+04,
    3.315809E+04, 3.331652E+04, 3.347514E+04, 3.363394E+04, 3.379293E+04, 3.395210E+04,
    3.411145E+04, 3.427098E+04, 3.443070E+04, 3.459059E+04, 3.475066E+04, 3.491090E+04,
    3.507132E+04, 3.523191E+04, 3.539268E+04, 3.555361E+04, 3.571471E+04, 3.587598E+04,
    3.603742E+04, 3.619902E+04, 3.636079E+04, 3.652272E+04, 3.668481E+04, 3.684706E+04,
    3.700947E+04, 3.717204E+04, 3.733477E+04, 3.749765E+04, 3.766069E+04, 3.782387E+04,
    3.798721E+04, 3.815070E+04, 3.831434E+04, 3.847813E+04, 3.864207E+04, 3.880615E+04,
    3.897037E+04, 3.913474E+04, 3.929926E+04, 3.946391E+04, 3.962870E+04, 3.979363E+04,
    3.995870E+04, 4.012391E+04, 4.028925E+04, 4.045473E+04, 4.062034E+04, 4.078608E+04,
    4.095195E+04, 4.111796E+04, 4.128409E+04, 4.145035E+04, 4.161673E+04, 4.178324E+04,
    4.194988E+04, 4.211664E+04, 4.228352E+04, 4.245053E+04, 4.261765E+04, 4.278489E+04,
    4.295225E+04, 4.311973E+04, 4.328733E+04, 4.345503E+04, 4.362286E+04, 4.379079E+04,
    4.395884E+04, 4.412700E+04, 4.429527E+04, 4.446365E+04, 4.463213E+04, 4.480073E+04,
    4.496943E+04, 4.513823E+04, 4.530714E+04, 4.547615E+04, 4.564526E+04, 4.581448E+04,
    4.598379E+04, 4.615320E+04, 4.632272E+04, 4.649233E+04, 4.666203E+04, 4.683183E+04,
    4.700173E+04, 4.717172E+04, 4.734180E+04, 4.751197E+04, 4.768224E+04, 4.785259E+04,
    4.802304E+04, 4.819357E+04, 4.836419E+04, 4.853489E+04, 4.870568E+04, 4.887656E+04,
    4.904752E+04, 4.921856E+04, 4.938969E+04, 4.956089E+04, 4.973218E+04, 4.990354E+04,
    5.007499E+04, 5.024651E+04, 5.041811E+04, 5.058978E+04, 5.076153E+04, 5.093336E+04,
    5.110526E+04, 5.127723E+04, 5.144927E+04, 5.162139E+04, 5.179357E+04, 5.196583E+04,
    5.213815E+04, 5.231054E+04, 5.248300E+04, 5.265553E+04, 5.282812E+04, 5.300078E+04,
    5.317350E+04, 5.334628E+04, 5.351913E+04, 5.369204E+04, 5.386501E+04, 5.403804E+04,
    5.421113E+04,
])

#  ---------------------- M = 22, I = 3 ---------------------------
M = 22
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[3]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.042930E+00, 1.565169E+01, 3.061038E+01, 4.557650E+01, 6.054536E+01, 7.551604E+01,
    9.048823E+01, 1.054618E+02, 1.204367E+02, 1.354129E+02, 1.503903E+02, 1.653691E+02,
    1.803492E+02, 1.953309E+02, 2.103145E+02, 2.253008E+02, 2.402907E+02, 2.552858E+02,
    2.702880E+02, 2.852999E+02, 3.003246E+02, 3.153656E+02, 3.304270E+02, 3.455134E+02,
    3.606296E+02, 3.757809E+02, 3.909726E+02, 4.062104E+02, 4.214999E+02, 4.368470E+02,
    4.522573E+02, 4.677366E+02, 4.832903E+02, 4.989241E+02, 5.146432E+02, 5.304529E+02,
    5.463581E+02, 5.623636E+02, 5.784740E+02, 5.946939E+02, 6.110273E+02, 6.274783E+02,
    6.440507E+02, 6.607482E+02, 6.775743E+02, 6.945321E+02, 7.116249E+02, 7.288554E+02,
    7.462266E+02, 7.637410E+02, 7.814012E+02, 7.992095E+02, 8.171680E+02, 8.352790E+02,
    8.535443E+02, 8.719660E+02, 8.905456E+02, 9.092849E+02, 9.281856E+02, 9.472490E+02,
    9.664766E+02, 9.858697E+02, 1.005430E+03, 1.025157E+03, 1.045054E+03, 1.065122E+03,
    1.085360E+03, 1.105770E+03, 1.126354E+03, 1.147111E+03, 1.168043E+03, 1.189150E+03,
    1.210433E+03, 1.231893E+03, 1.253530E+03, 1.275346E+03, 1.297340E+03, 1.319512E+03,
    1.341865E+03, 1.364397E+03, 1.387110E+03, 1.410003E+03, 1.433077E+03, 1.456333E+03,
    1.479771E+03, 1.503390E+03, 1.527192E+03, 1.551176E+03, 1.575343E+03, 1.599692E+03,
    1.624225E+03, 1.648941E+03, 1.673840E+03, 1.698922E+03, 1.724188E+03, 1.749637E+03,
    1.775269E+03, 1.801085E+03, 1.827085E+03, 1.853267E+03, 1.879634E+03, 1.906183E+03,
    1.932916E+03, 1.959832E+03, 1.986931E+03, 2.014212E+03, 2.041677E+03, 2.069324E+03,
    2.097154E+03, 2.125166E+03, 2.153359E+03, 2.181735E+03, 2.210292E+03, 2.239031E+03,
    2.267951E+03, 2.297051E+03, 2.326332E+03, 2.355794E+03, 2.385435E+03, 2.415256E+03,
    2.445256E+03, 2.475436E+03, 2.505794E+03, 2.536330E+03, 2.567044E+03, 2.597935E+03,
    2.629004E+03, 2.660250E+03, 2.691671E+03, 2.723269E+03, 2.755043E+03, 2.786991E+03,
    2.819114E+03, 2.851411E+03, 2.883882E+03, 2.916526E+03, 2.949343E+03, 2.982332E+03,
    3.015493E+03, 3.048826E+03, 3.082329E+03, 3.116002E+03, 3.149846E+03, 3.183859E+03,
    3.218040E+03, 3.252390E+03, 3.286907E+03, 3.321592E+03, 3.356443E+03, 3.391461E+03,
    3.426643E+03, 3.461991E+03, 3.497504E+03, 3.533180E+03, 3.569019E+03, 3.605021E+03,
    3.641186E+03, 3.677511E+03, 3.713998E+03, 3.750645E+03, 3.787452E+03, 3.824418E+03,
    3.861542E+03, 3.898825E+03, 3.936265E+03, 3.973861E+03, 4.011614E+03, 4.049522E+03,
    4.087585E+03, 4.125803E+03, 4.164174E+03, 4.202698E+03, 4.241375E+03, 4.280203E+03,
    4.319182E+03, 4.358312E+03, 4.397592E+03, 4.437021E+03, 4.476599E+03, 4.516325E+03,
    4.556198E+03, 4.596218E+03, 4.636384E+03, 4.676695E+03, 4.717151E+03, 4.757752E+03,
    4.798495E+03, 4.839382E+03, 4.880411E+03, 4.921581E+03, 4.962892E+03, 5.004344E+03,
    5.045935E+03, 5.087665E+03, 5.129534E+03, 5.171540E+03, 5.213683E+03, 5.255963E+03,
    5.298378E+03, 5.340928E+03, 5.383613E+03, 5.426432E+03, 5.469384E+03, 5.512468E+03,
    5.555684E+03, 5.599031E+03, 5.642509E+03, 5.686116E+03, 5.729853E+03, 5.773719E+03,
    5.817712E+03, 5.861833E+03, 5.906080E+03, 5.950453E+03, 5.994952E+03, 6.039576E+03,
    6.084323E+03, 6.129194E+03, 6.174188E+03, 6.219304E+03, 6.264541E+03, 6.309899E+03,
    6.355378E+03, 6.400976E+03, 6.446693E+03, 6.492528E+03, 6.538481E+03, 6.584551E+03,
    6.630738E+03, 6.677040E+03, 6.723457E+03, 6.769989E+03, 6.816635E+03, 6.863394E+03,
    6.910266E+03, 6.957250E+03, 7.004345E+03, 7.051551E+03, 7.098867E+03, 7.146292E+03,
    7.193827E+03, 7.241469E+03, 7.289220E+03, 7.337077E+03, 7.385041E+03, 7.433111E+03,
    7.481286E+03, 7.529565E+03, 7.577949E+03, 7.626436E+03, 7.675025E+03, 7.723717E+03,
    7.772511E+03, 7.821405E+03, 7.870400E+03, 7.919495E+03, 7.968689E+03, 8.017981E+03,
    8.067371E+03, 8.116859E+03, 8.166444E+03, 8.216125E+03, 8.265901E+03, 8.315773E+03,
    8.365739E+03, 8.415798E+03, 8.465952E+03, 8.516198E+03, 8.566536E+03, 8.616965E+03,
    8.667486E+03, 8.718097E+03, 8.768798E+03, 8.819588E+03, 8.870467E+03, 8.921434E+03,
    8.972489E+03, 9.023631E+03, 9.074859E+03, 9.126174E+03, 9.177573E+03, 9.229058E+03,
    9.280626E+03, 9.332279E+03, 9.384014E+03, 9.435833E+03, 9.487733E+03, 9.539715E+03,
    9.591778E+03, 9.643921E+03, 9.696144E+03, 9.748447E+03, 9.800828E+03, 9.853288E+03,
    9.905826E+03, 9.958440E+03, 1.001113E+04, 1.006390E+04, 1.011674E+04, 1.016966E+04,
    1.022266E+04, 1.027572E+04, 1.032886E+04, 1.038208E+04, 1.043536E+04, 1.048872E+04,
    1.054215E+04, 1.059566E+04, 1.064923E+04, 1.070287E+04, 1.075658E+04, 1.081036E+04,
    1.086421E+04, 1.091813E+04, 1.097212E+04, 1.102617E+04, 1.108029E+04, 1.113447E+04,
    1.118872E+04, 1.124304E+04, 1.129742E+04, 1.135187E+04, 1.140637E+04, 1.146095E+04,
    1.151558E+04, 1.157028E+04, 1.162504E+04, 1.167986E+04, 1.173474E+04, 1.178968E+04,
    1.184468E+04, 1.189975E+04, 1.195487E+04, 1.201005E+04, 1.206528E+04, 1.212058E+04,
    1.217593E+04, 1.223134E+04, 1.228681E+04, 1.234233E+04, 1.239791E+04, 1.245354E+04,
    1.250923E+04, 1.256497E+04, 1.262077E+04, 1.267661E+04, 1.273252E+04, 1.278847E+04,
    1.284448E+04, 1.290054E+04, 1.295665E+04, 1.301281E+04, 1.306902E+04, 1.312528E+04,
    1.318159E+04, 1.323795E+04, 1.329436E+04, 1.335081E+04, 1.340732E+04, 1.346387E+04,
    1.352047E+04, 1.357712E+04, 1.363381E+04, 1.369055E+04, 1.374734E+04, 1.380417E+04,
    1.386104E+04, 1.391796E+04, 1.397492E+04, 1.403193E+04, 1.408898E+04, 1.414607E+04,
    1.420321E+04, 1.426039E+04, 1.431761E+04, 1.437487E+04, 1.443217E+04, 1.448951E+04,
    1.454690E+04, 1.460432E+04, 1.466178E+04, 1.471928E+04, 1.477682E+04, 1.483440E+04,
    1.489202E+04, 1.494968E+04, 1.500737E+04, 1.506510E+04, 1.512287E+04, 1.518067E+04,
    1.523851E+04, 1.529638E+04, 1.535429E+04, 1.541224E+04, 1.547022E+04, 1.552823E+04,
    1.558628E+04, 1.564436E+04, 1.570248E+04, 1.576063E+04, 1.581881E+04, 1.587702E+04,
    1.593527E+04, 1.599354E+04, 1.605185E+04, 1.611019E+04, 1.616856E+04, 1.622697E+04,
    1.628540E+04, 1.634386E+04, 1.640235E+04, 1.646087E+04, 1.651942E+04, 1.657800E+04,
    1.663660E+04, 1.669524E+04, 1.675390E+04, 1.681259E+04, 1.687130E+04, 1.693005E+04,
    1.698882E+04, 1.704761E+04, 1.710643E+04, 1.716528E+04, 1.722415E+04, 1.728305E+04,
    1.734197E+04, 1.740092E+04, 1.745989E+04, 1.751889E+04, 1.757791E+04, 1.763695E+04,
    1.769602E+04, 1.775510E+04, 1.781422E+04, 1.787335E+04, 1.793250E+04, 1.799168E+04,
    1.805088E+04, 1.811010E+04, 1.816934E+04, 1.822860E+04, 1.828789E+04, 1.834719E+04,
    1.840651E+04, 1.846585E+04, 1.852522E+04, 1.858460E+04, 1.864400E+04, 1.870342E+04,
    1.876285E+04,
])

#  ---------------------- M = 23, I = 1 ---------------------------
M = 23
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    6.255910E+00, 5.846743E+01, 1.148737E+02, 1.712988E+02, 2.277330E+02, 2.841901E+02,
    3.407448E+02, 3.975806E+02, 4.549981E+02, 5.133834E+02, 5.731623E+02, 6.347625E+02,
    6.985902E+02, 7.650186E+02, 8.343861E+02, 9.069982E+02, 9.831325E+02, 1.063044E+03,
    1.146969E+03, 1.235132E+03, 1.327747E+03, 1.425022E+03, 1.527162E+03, 1.634371E+03,
    1.746850E+03, 1.864805E+03, 1.988442E+03, 2.117972E+03, 2.253608E+03, 2.395569E+03,
    2.544078E+03, 2.699363E+03, 2.861660E+03, 3.031206E+03, 3.208249E+03, 3.393039E+03,
    3.585834E+03, 3.786899E+03, 3.996503E+03, 4.214924E+03, 4.442444E+03, 4.679352E+03,
    4.925946E+03, 5.182528E+03, 5.449407E+03, 5.726899E+03, 6.015328E+03, 6.315021E+03,
    6.626316E+03, 6.949556E+03, 7.285090E+03, 7.644438E+03, 8.017771E+03, 8.405197E+03,
    8.807106E+03, 9.223898E+03, 9.655973E+03, 1.010375E+04, 1.056765E+04, 1.104809E+04,
    1.154553E+04, 1.206039E+04, 1.259313E+04, 1.314422E+04, 1.371412E+04, 1.430330E+04,
    1.491225E+04, 1.554147E+04, 1.619144E+04, 1.686268E+04, 1.755572E+04, 1.827106E+04,
    1.900924E+04, 1.977081E+04, 2.055631E+04, 2.136630E+04, 2.220136E+04, 2.306204E+04,
    2.394895E+04, 2.486265E+04, 2.580377E+04, 2.677291E+04, 2.777069E+04, 2.879773E+04,
    2.985468E+04, 3.094216E+04, 3.206084E+04, 3.321139E+04, 3.439446E+04, 3.561075E+04,
    3.686094E+04, 3.814573E+04, 3.946582E+04, 4.082193E+04, 4.221480E+04, 4.364516E+04,
    4.511375E+04, 4.662130E+04, 4.816861E+04, 4.975646E+04, 5.138558E+04, 5.305681E+04,
    5.477092E+04, 5.652873E+04, 5.833107E+04, 6.017876E+04, 6.207265E+04, 6.401357E+04,
    6.600238E+04, 6.803996E+04, 7.012718E+04, 7.226494E+04, 7.445411E+04, 7.669563E+04,
    7.899039E+04, 8.133932E+04, 8.374336E+04, 8.620347E+04, 8.872059E+04, 9.129568E+04,
    9.392974E+04, 9.662374E+04, 9.937866E+04, 1.021955E+05, 1.050753E+05, 1.080191E+05,
    1.110280E+05, 1.141028E+05, 1.172448E+05, 1.204550E+05, 1.237344E+05, 1.270842E+05,
    1.305054E+05, 1.339991E+05, 1.375665E+05, 1.412087E+05, 1.449268E+05, 1.487220E+05,
    1.525953E+05, 1.565481E+05, 1.605814E+05, 1.646964E+05, 1.688944E+05, 1.731765E+05,
    1.775439E+05, 1.819979E+05, 1.865398E+05, 1.911707E+05, 1.958919E+05, 2.007047E+05,
    2.056102E+05, 2.106100E+05, 2.157051E+05, 2.208970E+05, 2.261869E+05, 2.315762E+05,
    2.370662E+05, 2.426582E+05, 2.483535E+05, 2.541537E+05, 2.600599E+05, 2.660736E+05,
    2.721962E+05, 2.784291E+05, 2.847737E+05, 2.912314E+05, 2.978036E+05, 3.044918E+05,
    3.112975E+05, 3.182219E+05, 3.252667E+05, 3.324334E+05, 3.397233E+05, 3.471381E+05,
    3.546790E+05, 3.623478E+05,
])

#  ---------------------- M = 23, I = 2 ---------------------------
M = 23
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.256912E+01, 1.198179E+02, 2.355169E+02, 3.512532E+02, 4.670077E+02, 5.828098E+02,
    6.988156E+02, 8.154062E+02, 9.332049E+02, 1.053010E+03, 1.175699E+03, 1.302157E+03,
    1.433218E+03, 1.569655E+03, 1.712166E+03, 1.861382E+03, 2.017879E+03, 2.182186E+03,
    2.354797E+03, 2.536179E+03, 2.726782E+03, 2.927041E+03, 3.137388E+03, 3.358249E+03,
    3.590052E+03, 3.833228E+03, 4.088213E+03, 4.355451E+03, 4.635393E+03, 4.928501E+03,
    5.235245E+03, 5.556109E+03, 5.891584E+03, 6.242178E+03, 6.608407E+03, 6.990802E+03,
    7.389906E+03, 7.806274E+03, 8.240476E+03, 8.693093E+03, 9.164721E+03, 9.655968E+03,
    1.016746E+04, 1.069982E+04, 1.125372E+04, 1.182980E+04, 1.242875E+04, 1.305125E+04,
    1.369802E+04, 1.436977E+04, 1.506723E+04, 1.581858E+04, 1.659528E+04, 1.740150E+04,
    1.823806E+04, 1.910581E+04, 2.000560E+04, 2.093831E+04, 2.190483E+04, 2.290607E+04,
    2.394295E+04, 2.501642E+04, 2.612743E+04, 2.727695E+04, 2.846597E+04, 2.969552E+04,
    3.096661E+04, 3.228030E+04, 3.363762E+04, 3.503967E+04, 3.648755E+04, 3.798238E+04,
    3.952527E+04, 4.111738E+04, 4.275989E+04, 4.445397E+04, 4.620085E+04, 4.800173E+04,
    4.985785E+04, 5.177049E+04, 5.374092E+04, 5.577045E+04, 5.786038E+04, 6.001205E+04,
    6.222683E+04, 6.450609E+04, 6.685123E+04, 6.926364E+04, 7.174476E+04, 7.429607E+04,
    7.691903E+04, 7.961513E+04, 8.238587E+04, 8.523279E+04, 8.815748E+04, 9.116146E+04,
    9.424635E+04, 9.741373E+04, 1.006653E+05, 1.040027E+05, 1.074275E+05, 1.109415E+05,
    1.145465E+05, 1.182440E+05, 1.220360E+05, 1.259241E+05, 1.299102E+05, 1.339961E+05,
    1.381836E+05, 1.424746E+05, 1.468710E+05, 1.513747E+05, 1.559876E+05, 1.607117E+05,
    1.655489E+05, 1.705012E+05, 1.755707E+05, 1.807593E+05, 1.860692E+05, 1.915024E+05,
    1.970610E+05, 2.027471E+05, 2.085628E+05, 2.145104E+05, 2.205921E+05, 2.268099E+05,
    2.331663E+05, 2.396632E+05, 2.463032E+05, 2.530885E+05, 2.600213E+05, 2.671041E+05,
    2.743392E+05, 2.817289E+05, 2.892757E+05, 2.969820E+05, 3.048502E+05, 3.128829E+05,
    3.210824E+05, 3.294513E+05, 3.379922E+05, 3.467075E+05, 3.556000E+05, 3.646721E+05,
    3.739265E+05, 3.833658E+05, 3.929927E+05, 4.028098E+05, 4.128199E+05, 4.230257E+05,
    4.334300E+05, 4.440355E+05, 4.548450E+05, 4.658613E+05, 4.770871E+05, 4.885256E+05,
    5.001793E+05, 5.120512E+05, 5.241444E+05, 5.364618E+05, 5.490060E+05, 5.617803E+05,
    5.747876E+05, 5.880310E+05, 6.015134E+05, 6.152380E+05, 6.292077E+05, 6.434256E+05,
    6.578952E+05, 6.726192E+05, 6.876007E+05, 7.028432E+05, 7.183499E+05, 7.341236E+05,
    7.501680E+05, 7.664860E+05,
])

#  ---------------------- M = 23, I = 3 ---------------------------
M = 23
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    4.705510E+00, 4.011176E+01, 7.886627E+01, 1.176487E+02, 1.564527E+02, 1.952881E+02,
    2.342069E+02, 2.733368E+02, 3.128870E+02, 3.531270E+02, 3.943565E+02, 4.368808E+02,
    4.809954E+02, 5.269788E+02, 5.750913E+02, 6.255753E+02, 6.786581E+02, 7.345535E+02,
    7.934654E+02, 8.555889E+02, 9.211128E+02, 9.902210E+02, 1.063094E+03, 1.139909E+03,
    1.220842E+03, 1.306069E+03, 1.395765E+03, 1.490107E+03, 1.589273E+03, 1.693442E+03,
    1.802795E+03, 1.917518E+03, 2.037796E+03, 2.163821E+03, 2.295788E+03, 2.433892E+03,
    2.578337E+03, 2.729329E+03, 2.887077E+03, 3.051796E+03, 3.223707E+03, 3.403033E+03,
    3.590003E+03, 3.784852E+03, 3.987816E+03, 4.199140E+03, 4.419076E+03, 4.647874E+03,
    4.885797E+03, 5.133106E+03, 5.390076E+03, 5.656980E+03, 5.934099E+03, 6.221720E+03,
    6.520136E+03, 6.829645E+03, 7.150549E+03, 7.483158E+03, 7.827786E+03, 8.184755E+03,
    8.554390E+03, 8.937025E+03, 9.332999E+03, 9.742652E+03, 1.016634E+04, 1.060441E+04,
    1.105723E+04, 1.152517E+04, 1.200860E+04, 1.250790E+04, 1.302346E+04, 1.355567E+04,
    1.410492E+04, 1.467163E+04, 1.525621E+04, 1.585907E+04, 1.648063E+04, 1.712133E+04,
    1.778161E+04, 1.846190E+04, 1.916266E+04, 1.988433E+04, 2.062740E+04, 2.139232E+04,
    2.217957E+04, 2.298962E+04, 2.382298E+04, 2.468014E+04, 2.556160E+04, 2.646787E+04,
    2.739946E+04, 2.835689E+04, 2.934071E+04, 3.035144E+04, 3.138963E+04, 3.245582E+04,
    3.355059E+04, 3.467448E+04, 3.582807E+04, 3.701195E+04, 3.822668E+04, 3.947288E+04,
    4.075113E+04, 4.206204E+04, 4.340624E+04, 4.478434E+04, 4.619696E+04, 4.764476E+04,
    4.912835E+04, 5.064840E+04, 5.220557E+04, 5.380051E+04, 5.543391E+04, 5.710644E+04,
    5.881878E+04, 6.057165E+04, 6.236571E+04, 6.420171E+04, 6.608034E+04, 6.800233E+04,
    6.996841E+04, 7.197933E+04, 7.403584E+04, 7.613867E+04, 7.828858E+04, 8.048637E+04,
    8.273280E+04, 8.502866E+04, 8.737470E+04, 8.977178E+04, 9.222067E+04, 9.472220E+04,
    9.727719E+04, 9.988645E+04, 1.025508E+05, 1.052712E+05, 1.080484E+05, 1.108832E+05,
    1.137766E+05, 1.167294E+05, 1.197425E+05, 1.228168E+05, 1.259531E+05, 1.291525E+05,
    1.324157E+05, 1.357438E+05, 1.391376E+05, 1.425981E+05, 1.461262E+05, 1.497229E+05,
    1.533892E+05, 1.571259E+05, 1.609341E+05, 1.648147E+05, 1.687687E+05, 1.727973E+05,
    1.769011E+05, 1.810815E+05, 1.853392E+05, 1.896755E+05, 1.940913E+05, 1.985876E+05,
    2.031654E+05, 2.078260E+05, 2.125702E+05, 2.173992E+05, 2.223140E+05, 2.273157E+05,
    2.324055E+05, 2.375843E+05, 2.428533E+05, 2.482137E+05, 2.536665E+05, 2.592128E+05,
    2.648538E+05, 2.705906E+05,
])

#  ---------------------- M = 24, I = 1 ---------------------------
M = 24
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.292033E+01, 1.005064E+03, 2.758565E+03, 5.052901E+03, 7.773275E+03, 1.085963E+04,
    1.427439E+04, 1.799370E+04, 2.200520E+04, 2.630720E+04, 3.090790E+04, 3.582448E+04,
    4.108212E+04, 4.671306E+04, 5.275584E+04, 5.925467E+04, 6.625898E+04, 7.382313E+04,
    8.200629E+04, 9.087247E+04, 1.004906E+05, 1.109347E+05, 1.222845E+05, 1.346250E+05,
    1.480481E+05, 1.626518E+05, 1.785416E+05, 1.958306E+05, 2.146405E+05, 2.351016E+05,
    2.573542E+05, 2.815485E+05, 3.078462E+05, 3.364206E+05, 3.674579E+05, 4.011576E+05,
    4.377339E+05, 4.774164E+05, 5.204511E+05, 5.671017E+05, 6.176507E+05, 6.724001E+05,
    7.316734E+05, 7.958165E+05, 8.651993E+05, 9.402168E+05, 1.021291E+06, 1.108873E+06,
    1.203444E+06, 1.305515E+06, 1.415636E+06, 1.534387E+06, 1.662391E+06, 1.800308E+06,
    1.948842E+06, 2.108742E+06, 2.280804E+06, 2.465875E+06, 2.664853E+06, 2.878694E+06,
    3.108412E+06, 3.355083E+06, 3.619849E+06, 3.903920E+06, 4.208579E+06, 4.535186E+06,
    4.885181E+06, 5.260089E+06, 5.661522E+06, 6.091189E+06, 6.550894E+06, 7.042545E+06,
    7.568161E+06, 8.129872E+06, 8.729927E+06, 9.370703E+06, 1.005471E+07, 1.078458E+07,
    1.156312E+07, 1.239325E+07, 1.327809E+07, 1.422089E+07, 1.522509E+07, 1.629431E+07,
    1.743237E+07, 1.864327E+07, 1.993123E+07, 2.130068E+07, 2.275629E+07, 2.430295E+07,
    2.594581E+07, 2.769028E+07, 2.954203E+07, 3.150702E+07, 3.359151E+07, 3.580204E+07,
    3.814551E+07, 4.062912E+07, 4.326044E+07, 4.604738E+07, 4.899825E+07, 5.212174E+07,
    5.542695E+07, 5.892342E+07, 6.262112E+07, 6.653048E+07, 7.066241E+07, 7.502834E+07,
    7.964020E+07, 8.451045E+07, 8.965214E+07, 9.507889E+07, 1.008049E+08, 1.068451E+08,
    1.132149E+08, 1.199306E+08, 1.270089E+08, 1.344676E+08, 1.423251E+08, 1.506005E+08,
    1.593137E+08, 1.684857E+08, 1.781381E+08, 1.882936E+08, 1.989757E+08, 2.102091E+08,
    2.220191E+08, 2.344325E+08, 2.474770E+08, 2.611813E+08, 2.755755E+08, 2.906907E+08,
    3.065595E+08, 3.232155E+08, 3.406938E+08, 3.590310E+08, 3.782647E+08, 3.984346E+08,
    4.195814E+08, 4.417476E+08, 4.649774E+08, 4.893165E+08, 5.148126E+08, 5.415149E+08,
    5.694747E+08, 5.987452E+08, 6.293814E+08, 6.614407E+08, 6.949822E+08, 7.300674E+08,
    7.667602E+08, 8.051266E+08, 8.452350E+08, 8.871565E+08, 9.309645E+08, 9.767353E+08,
    1.024548E+09, 1.074483E+09, 1.126626E+09, 1.181065E+09, 1.237890E+09, 1.297194E+09,
    1.359076E+09, 1.423634E+09, 1.490973E+09, 1.561201E+09, 1.634429E+09, 1.710770E+09,
    1.790345E+09, 1.873276E+09, 1.959690E+09, 2.049717E+09, 2.143494E+09, 2.241160E+09,
    2.342859E+09, 2.448743E+09,
])

#  ---------------------- M = 24, I = 2 ---------------------------
M = 24
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.320610E+01, 1.020802E+03, 2.802064E+03, 5.132731E+03, 7.896193E+03, 1.103144E+04,
    1.450032E+04, 1.827857E+04, 2.235364E+04, 2.672383E+04, 3.139746E+04, 3.639199E+04,
    4.173298E+04, 4.745320E+04, 5.359179E+04, 6.019368E+04, 6.730905E+04, 7.499315E+04,
    8.330609E+04, 9.231288E+04, 1.020835E+05, 1.126934E+05, 1.242231E+05, 1.367594E+05,
    1.503953E+05, 1.652307E+05, 1.813726E+05, 1.989359E+05, 2.180442E+05, 2.388299E+05,
    2.614355E+05, 2.860137E+05, 3.127287E+05, 3.417565E+05, 3.732862E+05, 4.075207E+05,
    4.446774E+05, 4.849896E+05, 5.287073E+05, 5.760983E+05, 6.274494E+05, 6.830678E+05,
    7.432819E+05, 8.084431E+05, 8.789272E+05, 9.551356E+05, 1.037497E+06, 1.126469E+06,
    1.222541E+06, 1.326233E+06, 1.438102E+06, 1.558738E+06, 1.688774E+06, 1.828881E+06,
    1.979774E+06, 2.142213E+06, 2.317007E+06, 2.505017E+06, 2.707155E+06, 2.924392E+06,
    3.157758E+06, 3.408347E+06, 3.677317E+06, 3.965900E+06, 4.275399E+06, 4.607193E+06,
    4.962748E+06, 5.343611E+06, 5.751421E+06, 6.187913E+06, 6.654922E+06, 7.154384E+06,
    7.688351E+06, 8.258986E+06, 8.868575E+06, 9.519533E+06, 1.021440E+07, 1.095588E+07,
    1.174678E+07, 1.259011E+07, 1.348901E+07, 1.444679E+07, 1.546695E+07, 1.655317E+07,
    1.770932E+07, 1.893946E+07, 2.024790E+07, 2.163912E+07, 2.311786E+07, 2.468911E+07,
    2.635809E+07, 2.813029E+07, 3.001148E+07, 3.200771E+07, 3.412534E+07, 3.637102E+07,
    3.875175E+07, 4.127485E+07, 4.394801E+07, 4.677927E+07, 4.977706E+07, 5.295023E+07,
    5.630800E+07, 5.986008E+07, 6.361659E+07, 6.758813E+07, 7.178578E+07, 7.622116E+07,
    8.090637E+07, 8.585410E+07, 9.107759E+07, 9.659067E+07, 1.024078E+08, 1.085441E+08,
    1.150152E+08, 1.218377E+08, 1.290287E+08, 1.366061E+08, 1.445886E+08, 1.529956E+08,
    1.618476E+08, 1.711655E+08, 1.809715E+08, 1.912887E+08, 2.021408E+08, 2.135529E+08,
    2.255509E+08, 2.381619E+08, 2.514140E+08, 2.653365E+08, 2.799598E+08, 2.953157E+08,
    3.114371E+08, 3.283583E+08, 3.461149E+08, 3.647439E+08, 3.842840E+08, 4.047750E+08,
    4.262585E+08, 4.487777E+08, 4.723774E+08, 4.971041E+08, 5.230062E+08, 5.501338E+08,
    5.785389E+08, 6.082756E+08, 6.393998E+08, 6.719697E+08, 7.060455E+08, 7.416896E+08,
    7.789669E+08, 8.179445E+08, 8.586919E+08, 9.012813E+08, 9.457873E+08, 9.922872E+08,
    1.040861E+09, 1.091593E+09, 1.144567E+09, 1.199873E+09, 1.257603E+09, 1.317853E+09,
    1.380720E+09, 1.446308E+09, 1.514720E+09, 1.586067E+09, 1.660462E+09, 1.738021E+09,
    1.818864E+09, 1.903117E+09, 1.990908E+09, 2.082370E+09, 2.177642E+09, 2.276865E+09,
    2.380187E+09, 2.487758E+09,
])

#  ---------------------- M = 25, I = 1 ---------------------------
M = 25
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.770090E+00, 9.937734E+01, 3.215728E+02, 6.242169E+02, 9.940464E+02, 1.428783E+03,
    1.930438E+03, 2.502407E+03, 3.148468E+03, 3.872571E+03, 4.678893E+03, 5.571933E+03,
    6.556583E+03, 7.638169E+03, 8.822458E+03, 1.011566E+04, 1.152440E+04, 1.305573E+04,
    1.471706E+04, 1.651621E+04, 1.846133E+04, 2.056093E+04, 2.282389E+04, 2.525940E+04,
    2.787700E+04, 3.068660E+04, 3.369842E+04, 3.692304E+04, 4.037142E+04, 4.405485E+04,
    4.798501E+04, 5.217394E+04, 5.663409E+04, 6.137826E+04, 6.641968E+04, 7.177196E+04,
    7.744912E+04, 8.346559E+04, 8.983619E+04, 9.657618E+04, 1.037012E+05, 1.112273E+05,
    1.191709E+05, 1.275489E+05, 1.363785E+05, 1.456772E+05, 1.554631E+05, 1.657543E+05,
    1.765695E+05, 1.879275E+05, 1.998476E+05, 2.123492E+05, 2.254519E+05, 2.391757E+05,
    2.535406E+05, 2.685667E+05, 2.842744E+05, 3.006842E+05, 3.178165E+05, 3.356919E+05,
    3.543309E+05, 3.737539E+05, 3.939815E+05, 4.150340E+05, 4.369316E+05, 4.596945E+05,
    4.833425E+05, 5.078955E+05, 5.333729E+05, 5.597939E+05, 5.871776E+05, 6.155426E+05,
    6.449072E+05, 6.752896E+05, 7.067073E+05, 7.391776E+05, 7.727174E+05, 8.073431E+05,
    8.430708E+05, 8.799161E+05, 9.178940E+05, 9.570193E+05, 9.973060E+05, 1.038768E+06,
    1.081418E+06, 1.125270E+06, 1.170335E+06, 1.216625E+06, 1.264150E+06, 1.312923E+06,
    1.362953E+06, 1.414249E+06, 1.466821E+06, 1.520677E+06, 1.575826E+06, 1.632274E+06,
    1.690030E+06, 1.749098E+06, 1.809486E+06, 1.871200E+06, 1.934243E+06, 1.998620E+06,
    2.064337E+06, 2.131395E+06, 2.199799E+06, 2.269551E+06, 2.340654E+06, 2.413108E+06,
    2.486916E+06, 2.562078E+06, 2.638594E+06, 2.716466E+06, 2.795692E+06, 2.876272E+06,
    2.958205E+06, 3.041488E+06, 3.126122E+06, 3.212102E+06, 3.299428E+06, 3.388095E+06,
    3.478101E+06, 3.569442E+06, 3.662114E+06, 3.756113E+06, 3.851435E+06, 3.948076E+06,
    4.046029E+06, 4.145290E+06, 4.245854E+06, 4.347714E+06, 4.450865E+06, 4.555301E+06,
    4.661014E+06, 4.767999E+06, 4.876249E+06, 4.985756E+06, 5.096513E+06, 5.208514E+06,
    5.321749E+06, 5.436213E+06, 5.551896E+06, 5.668791E+06, 5.786890E+06, 5.906183E+06,
    6.026663E+06, 6.148322E+06, 6.271150E+06, 6.395138E+06, 6.520279E+06, 6.646561E+06,
    6.773978E+06, 6.902519E+06, 7.032175E+06, 7.162937E+06, 7.294795E+06, 7.427739E+06,
    7.561761E+06, 7.696851E+06, 7.832999E+06, 7.970195E+06, 8.108429E+06, 8.247692E+06,
    8.387974E+06, 8.529265E+06, 8.671555E+06, 8.814834E+06, 8.959093E+06, 9.104320E+06,
    9.250507E+06, 9.397644E+06, 9.545720E+06, 9.694725E+06, 9.844650E+06, 9.995484E+06,
    1.014722E+07, 1.029984E+07, 1.045334E+07, 1.060772E+07, 1.076295E+07, 1.091903E+07,
    1.107595E+07, 1.123370E+07, 1.139227E+07, 1.155166E+07, 1.171184E+07, 1.187281E+07,
    1.203457E+07, 1.219709E+07, 1.236038E+07, 1.252442E+07, 1.268921E+07, 1.285472E+07,
    1.302096E+07, 1.318792E+07, 1.335558E+07, 1.352393E+07, 1.369297E+07, 1.386269E+07,
    1.403307E+07, 1.420412E+07, 1.437581E+07, 1.454814E+07, 1.472111E+07, 1.489470E+07,
    1.506890E+07, 1.524370E+07, 1.541910E+07, 1.559509E+07, 1.577166E+07, 1.594880E+07,
    1.612649E+07, 1.630475E+07, 1.648354E+07, 1.666287E+07, 1.684273E+07, 1.702311E+07,
    1.720400E+07, 1.738539E+07, 1.756728E+07, 1.774966E+07, 1.793251E+07, 1.811584E+07,
    1.829962E+07, 1.848387E+07, 1.866856E+07, 1.885369E+07, 1.903926E+07, 1.922525E+07,
    1.941166E+07, 1.959847E+07, 1.978569E+07, 1.997331E+07, 2.016131E+07, 2.034970E+07,
    2.053846E+07, 2.072759E+07, 2.091708E+07, 2.110692E+07, 2.129710E+07, 2.148763E+07,
    2.167849E+07, 2.186967E+07, 2.206118E+07, 2.225300E+07, 2.244512E+07, 2.263755E+07,
    2.283027E+07, 2.302327E+07, 2.321656E+07, 2.341012E+07, 2.360396E+07, 2.379805E+07,
    2.399240E+07, 2.418700E+07, 2.438185E+07, 2.457694E+07, 2.477226E+07, 2.496781E+07,
    2.516358E+07, 2.535956E+07, 2.555576E+07, 2.575216E+07, 2.594876E+07, 2.614556E+07,
    2.634255E+07, 2.653972E+07, 2.673707E+07, 2.693459E+07, 2.713228E+07, 2.733013E+07,
    2.752815E+07, 2.772631E+07, 2.792462E+07, 2.812308E+07, 2.832168E+07, 2.852041E+07,
    2.871927E+07, 2.891825E+07, 2.911736E+07, 2.931658E+07, 2.951591E+07, 2.971535E+07,
    2.991488E+07, 3.011452E+07, 3.031425E+07, 3.051407E+07, 3.071398E+07, 3.091396E+07,
    3.111402E+07, 3.131416E+07, 3.151436E+07, 3.171463E+07, 3.191495E+07, 3.211534E+07,
    3.231577E+07, 3.251626E+07, 3.271679E+07, 3.291736E+07, 3.311797E+07, 3.331861E+07,
    3.351929E+07,
])

#  ---------------------- M = 26, I = 1 ---------------------------
M = 26
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[0]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.304830E+00, 2.430655E+01, 4.793073E+01, 7.156039E+01, 9.519594E+01, 1.188684E+02,
    1.426920E+02, 1.668995E+02, 1.918282E+02, 2.178787E+02, 2.454753E+02, 2.750411E+02,
    3.069871E+02, 3.417113E+02, 3.796013E+02, 4.210401E+02, 4.664113E+02, 5.161047E+02,
    5.705201E+02, 6.300721E+02, 6.951927E+02, 7.663344E+02, 8.439731E+02, 9.286097E+02,
    1.020773E+03, 1.121022E+03, 1.229946E+03, 1.348169E+03, 1.476351E+03, 1.615189E+03,
    1.765420E+03, 1.927822E+03, 2.103220E+03, 2.292482E+03, 2.496526E+03, 2.716320E+03,
    2.952886E+03, 3.207303E+03, 3.480706E+03, 3.774293E+03, 4.089326E+03, 4.427131E+03,
    4.789107E+03, 5.176724E+03, 5.591528E+03, 6.035144E+03, 6.509280E+03, 7.015728E+03,
    7.556371E+03, 8.133184E+03, 8.748238E+03, 9.403705E+03, 1.010186E+04, 1.084508E+04,
    1.163587E+04, 1.247683E+04, 1.337069E+04, 1.432031E+04, 1.532865E+04, 1.639884E+04,
    1.753411E+04, 1.873786E+04, 2.001359E+04, 2.136500E+04, 2.279588E+04, 2.431024E+04,
    2.591218E+04, 2.760600E+04, 2.939617E+04, 3.128728E+04, 3.328414E+04, 3.539170E+04,
    3.761510E+04, 3.995963E+04, 4.243078E+04, 4.503422E+04, 4.777578E+04, 5.066149E+04,
    5.369755E+04, 5.689035E+04, 6.024647E+04, 6.377264E+04, 6.747581E+04, 7.136309E+04,
    7.544178E+04, 7.971935E+04, 8.420347E+04, 8.890196E+04, 9.382283E+04, 9.897425E+04,
    1.043646E+05, 1.100024E+05, 1.158962E+05, 1.220551E+05, 1.284878E+05, 1.352037E+05,
    1.422121E+05, 1.495223E+05, 1.571439E+05, 1.650868E+05, 1.733608E+05, 1.819759E+05,
    1.909421E+05, 2.002699E+05, 2.099695E+05, 2.200513E+05, 2.305261E+05, 2.414044E+05,
    2.526970E+05, 2.644148E+05, 2.765688E+05, 2.891699E+05, 3.022293E+05, 3.157581E+05,
    3.297675E+05, 3.442689E+05, 3.592735E+05, 3.747927E+05, 3.908380E+05, 4.074208E+05,
    4.245525E+05, 4.422448E+05, 4.605090E+05, 4.793568E+05, 4.987996E+05, 5.188490E+05,
    5.395165E+05, 5.608137E+05, 5.827520E+05, 6.053430E+05, 6.285981E+05, 6.525288E+05,
    6.771464E+05, 7.024624E+05, 7.284880E+05, 7.552345E+05, 7.827132E+05, 8.109352E+05,
    8.399115E+05, 8.696534E+05, 9.001717E+05, 9.314773E+05, 9.635811E+05, 9.964938E+05,
    1.030226E+06, 1.064788E+06, 1.100191E+06, 1.136445E+06, 1.173561E+06, 1.211548E+06,
    1.250416E+06, 1.290176E+06, 1.330837E+06, 1.372410E+06, 1.414904E+06, 1.458328E+06,
    1.502692E+06, 1.548005E+06, 1.594276E+06, 1.641515E+06, 1.689730E+06, 1.738930E+06,
    1.789124E+06, 1.840321E+06, 1.892528E+06, 1.945754E+06, 2.000007E+06, 2.055296E+06,
    2.111627E+06, 2.169010E+06, 2.227450E+06, 2.286957E+06, 2.347537E+06, 2.409197E+06,
    2.471944E+06, 2.535786E+06, 2.600728E+06, 2.666778E+06, 2.733942E+06, 2.802227E+06,
    2.871638E+06, 2.942181E+06, 3.013863E+06, 3.086689E+06, 3.160664E+06, 3.235795E+06,
    3.312086E+06, 3.389542E+06, 3.468170E+06, 3.547972E+06, 3.628955E+06, 3.711122E+06,
    3.794479E+06, 3.879029E+06, 3.964776E+06, 4.051725E+06, 4.139880E+06, 4.229243E+06,
    4.319820E+06, 4.411612E+06, 4.504625E+06, 4.598859E+06, 4.694320E+06, 4.791009E+06,
    4.888930E+06, 4.988084E+06, 5.088475E+06, 5.190105E+06, 5.292976E+06, 5.397090E+06,
    5.502450E+06, 5.609057E+06, 5.716912E+06, 5.826018E+06, 5.936376E+06, 6.047987E+06,
    6.160852E+06, 6.274973E+06, 6.390351E+06, 6.506986E+06, 6.624880E+06, 6.744032E+06,
    6.864444E+06, 6.986115E+06, 7.109047E+06, 7.233239E+06, 7.358692E+06, 7.485405E+06,
    7.613378E+06, 7.742611E+06, 7.873104E+06, 8.004857E+06, 8.137868E+06, 8.272137E+06,
    8.407663E+06, 8.544446E+06, 8.682485E+06, 8.821778E+06, 8.962325E+06, 9.104124E+06,
    9.247174E+06, 9.391473E+06, 9.537021E+06, 9.683815E+06, 9.831853E+06, 9.981135E+06,
    1.013166E+07, 1.028342E+07, 1.043642E+07, 1.059066E+07, 1.074612E+07,
])

#  ---------------------- M = 26, I = 2 ---------------------------
M = 26
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    8.814510E+00, 9.724635E+01, 1.917632E+02, 2.864734E+02, 3.811700E+02, 4.760528E+02,
    5.715776E+02, 6.686900E+02, 7.687384E+02, 8.733388E+02, 9.841987E+02, 1.103046E+03,
    1.231520E+03, 1.371261E+03, 1.523833E+03, 1.690802E+03, 1.873730E+03, 2.074207E+03,
    2.293885E+03, 2.534441E+03, 2.797671E+03, 3.085414E+03, 3.399618E+03, 3.742372E+03,
    4.115810E+03, 4.522255E+03, 4.964139E+03, 5.444011E+03, 5.964565E+03, 6.528734E+03,
    7.139491E+03, 7.800087E+03, 8.513884E+03, 9.284496E+03, 1.011567E+04, 1.101143E+04,
    1.197594E+04, 1.301371E+04, 1.412933E+04, 1.532784E+04, 1.661437E+04, 1.799441E+04,
    1.947370E+04, 2.105830E+04, 2.275461E+04, 2.456929E+04, 2.650943E+04, 2.858230E+04,
    3.079576E+04, 3.315789E+04, 3.567723E+04, 3.836273E+04, 4.122372E+04, 4.427006E+04,
    4.751198E+04, 5.096026E+04, 5.462619E+04, 5.852141E+04, 6.265841E+04, 6.704991E+04,
    7.170941E+04, 7.665096E+04, 8.188914E+04, 8.743929E+04, 9.331736E+04, 9.953997E+04,
    1.061245E+05, 1.130890E+05, 1.204523E+05, 1.282340E+05, 1.364547E+05, 1.451355E+05,
    1.542986E+05, 1.639671E+05, 1.741648E+05, 1.849166E+05, 1.962486E+05, 2.081876E+05,
    2.207615E+05, 2.339994E+05, 2.479314E+05, 2.625888E+05, 2.780041E+05, 2.942110E+05,
    3.112445E+05, 3.291406E+05, 3.479371E+05, 3.676729E+05, 3.883884E+05, 4.101252E+05,
    4.329270E+05, 4.568381E+05, 4.819053E+05, 5.081766E+05, 5.357017E+05, 5.645321E+05,
    5.947207E+05, 6.263228E+05, 6.593952E+05, 6.939969E+05, 7.301883E+05, 7.680325E+05,
    8.075944E+05, 8.489406E+05, 8.921407E+05, 9.372660E+05, 9.843902E+05, 1.033589E+06,
    1.084942E+06, 1.138530E+06, 1.194435E+06, 1.252745E+06, 1.313548E+06, 1.376935E+06,
    1.443002E+06, 1.511846E+06, 1.583566E+06, 1.658265E+06, 1.736052E+06, 1.817034E+06,
    1.901325E+06, 1.989042E+06, 2.080302E+06, 2.175231E+06, 2.273953E+06, 2.376600E+06,
    2.483305E+06, 2.594207E+06, 2.709447E+06, 2.829171E+06, 2.953528E+06, 3.082674E+06,
    3.216765E+06, 3.355965E+06, 3.500442E+06, 3.650366E+06, 3.805915E+06, 3.967268E+06,
    4.134614E+06, 4.308144E+06, 4.488050E+06, 4.674537E+06, 4.867812E+06, 5.068086E+06,
    5.275577E+06, 5.490508E+06, 5.713108E+06, 5.943613E+06, 6.182262E+06, 6.429305E+06,
    6.684995E+06, 6.949588E+06, 7.223356E+06, 7.506568E+06, 7.799506E+06, 8.102456E+06,
    8.415711E+06, 8.739574E+06, 9.074353E+06, 9.420361E+06, 9.777928E+06, 1.014738E+07,
    1.052905E+07, 1.092330E+07, 1.133048E+07, 1.175095E+07, 1.218509E+07, 1.263327E+07,
    1.309589E+07, 1.357335E+07, 1.406605E+07, 1.457441E+07, 1.509887E+07, 1.563985E+07,
    1.619781E+07, 1.677320E+07,
])

#  ---------------------- M = 26, I = 3 ---------------------------
M = 26
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.044140E+00, 8.016384E+01, 1.642897E+02, 2.484331E+02, 3.326393E+02, 4.172555E+02,
    5.031646E+02, 5.918139E+02, 6.849749E+02, 7.845357E+02, 8.923827E+02, 1.010349E+03,
    1.140206E+03, 1.283685E+03, 1.442530E+03, 1.618488E+03, 1.813370E+03, 2.029067E+03,
    2.267536E+03, 2.550029E+03, 2.842636E+03, 3.164913E+03, 3.519348E+03, 3.908620E+03,
    4.335576E+03, 4.803251E+03, 5.314814E+03, 5.873741E+03, 6.483650E+03, 7.148457E+03,
    7.872235E+03, 8.659382E+03, 9.514548E+03, 1.044272E+04, 1.144906E+04, 1.253918E+04,
    1.371897E+04, 1.499462E+04, 1.637280E+04, 1.786041E+04, 1.946491E+04, 2.119405E+04,
    2.305609E+04, 2.505975E+04, 2.721420E+04, 2.952910E+04, 3.201468E+04, 3.468164E+04,
    3.754137E+04, 4.060572E+04, 4.388721E+04, 4.739912E+04, 5.115514E+04, 5.516994E+04,
    5.945871E+04, 6.403752E+04, 6.892318E+04, 7.413333E+04, 7.968642E+04, 8.560184E+04,
    9.189984E+04, 9.860168E+04, 1.057296E+05, 1.133067E+05, 1.213575E+05, 1.299074E+05,
    1.389828E+05, 1.486115E+05, 1.588224E+05, 1.696459E+05, 1.811134E+05, 1.932580E+05,
    2.061137E+05, 2.197165E+05, 2.341036E+05, 2.493138E+05, 2.653877E+05, 2.823672E+05,
    3.002963E+05, 3.192205E+05, 3.391872E+05, 3.602457E+05, 3.824474E+05, 4.058456E+05,
    4.304954E+05, 4.564546E+05, 4.837827E+05, 5.125418E+05, 5.427962E+05, 5.746128E+05,
    6.080607E+05, 6.432115E+05, 6.801401E+05, 7.189233E+05, 7.596412E+05, 8.023769E+05,
    8.472160E+05, 8.942474E+05, 9.435629E+05, 9.952583E+05, 1.049431E+06, 1.106185E+06,
    1.165624E+06, 1.227858E+06, 1.292999E+06, 1.361164E+06, 1.432473E+06, 1.507052E+06,
    1.585028E+06, 1.666535E+06, 1.751710E+06, 1.840694E+06, 1.933633E+06, 2.030680E+06,
    2.131987E+06, 2.237718E+06, 2.348038E+06, 2.463117E+06, 2.583132E+06, 2.708264E+06,
    2.838701E+06, 2.974635E+06, 3.116266E+06, 3.263798E+06, 3.417442E+06, 3.577415E+06,
    3.743943E+06, 3.917253E+06, 4.097585E+06, 4.285180E+06, 4.480292E+06, 4.683179E+06,
    4.894105E+06, 5.113345E+06, 5.341180E+06, 5.577898E+06, 5.823797E+06, 6.079183E+06,
    6.344370E+06, 6.619679E+06, 6.905443E+06, 7.202001E+06, 7.509705E+06, 7.828915E+06,
    8.159999E+06, 8.503333E+06, 8.859312E+06, 9.228330E+06, 9.610799E+06, 1.000714E+07,
    1.041778E+07, 1.084317E+07, 1.128377E+07, 1.174002E+07, 1.221242E+07, 1.270146E+07,
    1.320763E+07, 1.373146E+07, 1.427346E+07, 1.483419E+07, 1.541419E+07, 1.601404E+07,
    1.663432E+07, 1.727563E+07, 1.793856E+07, 1.862376E+07, 1.933187E+07, 2.006354E+07,
    2.081943E+07, 2.160025E+07, 2.240670E+07, 2.323950E+07, 2.409938E+07, 2.498711E+07,
    2.590344E+07, 2.684920E+07,
])

#  ---------------------- M = 27, I = 1 ---------------------------
M = 27
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.500304E+01, 9.183261E+02, 2.577299E+03, 4.726665E+03, 7.295077E+03, 1.027634E+04,
    1.369384E+04, 1.758296E+04, 2.198535E+04, 2.695083E+04, 3.253906E+04, 3.882423E+04,
    4.589599E+04, 5.386197E+04, 6.285216E+04, 7.301797E+04, 8.453841E+04, 9.762032E+04,
    1.125046E+05, 1.294682E+05, 1.488302E+05, 1.709580E+05, 1.962724E+05, 2.252593E+05,
    2.584720E+05, 2.965481E+05, 3.402164E+05, 3.903158E+05, 4.478043E+05, 5.137787E+05,
    5.894956E+05, 6.763969E+05, 7.761243E+05, 8.905622E+05, 1.021863E+06, 1.172482E+06,
    1.345229E+06, 1.543303E+06, 1.770360E+06, 2.030568E+06, 2.328675E+06, 2.670086E+06,
    3.060957E+06, 3.508289E+06, 4.020046E+06, 4.605271E+06, 5.274240E+06, 6.038612E+06,
    6.911616E+06, 7.908245E+06, 9.045487E+06, 1.034258E+07, 1.182128E+07, 1.350621E+07,
    1.542519E+07, 1.760965E+07, 2.009505E+07, 2.292142E+07, 2.613387E+07, 2.978325E+07,
    3.392680E+07, 3.862896E+07, 4.396218E+07, 5.000792E+07, 5.685769E+07, 6.461421E+07,
    7.339274E+07, 8.332252E+07, 9.454840E+07, 1.072326E+08, 1.215567E+08, 1.377238E+08,
    1.559610E+08, 1.765221E+08, 1.996904E+08, 2.257822E+08, 2.551502E+08, 2.881877E+08,
    3.253328E+08, 3.670735E+08, 4.139525E+08, 4.665738E+08, 5.256087E+08, 5.918032E+08,
    6.659852E+08, 7.490741E+08, 8.420894E+08, 9.461613E+08, 1.062542E+09, 1.192619E+09,
    1.337927E+09, 1.500164E+09, 1.681208E+09, 1.883134E+09, 2.108234E+09, 2.359039E+09,
    2.638341E+09, 2.949219E+09, 3.295067E+09, 3.679627E+09, 4.107016E+09, 4.581768E+09,
    5.108871E+09, 5.693809E+09, 6.342612E+09, 7.061902E+09, 7.858951E+09, 8.741742E+09,
    9.719033E+09, 1.080043E+10, 1.199645E+10, 1.331863E+10, 1.477960E+10, 1.639318E+10,
    1.817449E+10, 2.014008E+10, 2.230804E+10, 2.469810E+10, 2.733187E+10, 3.023290E+10,
    3.342690E+10, 3.694193E+10, 4.080858E+10, 4.506017E+10, 4.973304E+10, 5.486675E+10,
    6.050436E+10, 6.669275E+10, 7.348290E+10, 8.093024E+10, 8.909503E+10, 9.804276E+10,
    1.078445E+11, 1.185775E+11, 1.303256E+11, 1.431796E+11, 1.572382E+11, 1.726082E+11,
    1.894054E+11, 2.077554E+11, 2.277941E+11, 2.496685E+11, 2.735379E+11, 2.995744E+11,
    3.279644E+11, 3.589092E+11, 3.926263E+11, 4.293508E+11, 4.693366E+11, 5.128578E+11,
    5.602101E+11, 6.117127E+11, 6.677096E+11, 7.285721E+11, 7.947001E+11, 8.665246E+11,
    9.445098E+11, 1.029156E+12, 1.121000E+12, 1.220623E+12, 1.328646E+12, 1.445740E+12,
    1.572626E+12, 1.710077E+12, 1.858927E+12, 2.020068E+12, 2.194462E+12, 2.383137E+12,
    2.587201E+12, 2.807841E+12, 3.046329E+12, 3.304031E+12, 3.582410E+12, 3.883036E+12,
    4.207589E+12, 4.557870E+12,
])

#  ---------------------- M = 27, I = 2 ---------------------------
M = 27
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    7.614480E+00, 4.686721E+02, 1.315543E+03, 2.412801E+03, 3.724027E+03, 5.246064E+03,
    6.990860E+03, 8.976483E+03, 1.122421E+04, 1.375949E+04, 1.661280E+04, 1.982202E+04,
    2.343294E+04, 2.750053E+04, 3.209119E+04, 3.728225E+04, 4.316512E+04, 4.984547E+04,
    5.744633E+04, 6.610916E+04, 7.599690E+04, 8.729731E+04, 1.002252E+05, 1.150289E+05,
    1.319909E+05, 1.514370E+05, 1.737394E+05, 1.993267E+05, 2.286883E+05, 2.623844E+05,
    3.010571E+05, 3.454428E+05, 3.963803E+05, 4.548324E+05, 5.218983E+05, 5.988332E+05,
    6.870716E+05, 7.882488E+05, 9.042323E+05, 1.037152E+06, 1.189433E+06, 1.363837E+06,
    1.563510E+06, 1.792029E+06, 2.053463E+06, 2.352434E+06, 2.694190E+06, 3.084692E+06,
    3.530698E+06, 4.039869E+06, 4.620887E+06, 5.283581E+06, 6.039074E+06, 6.899946E+06,
    7.880414E+06, 8.996537E+06, 1.026644E+07, 1.171059E+07, 1.335203E+07, 1.521675E+07,
    1.733400E+07, 1.973673E+07, 2.246196E+07, 2.555133E+07, 2.905161E+07, 3.301531E+07,
    3.750133E+07, 4.257574E+07, 4.831258E+07, 5.479478E+07, 6.211514E+07, 7.037750E+07,
    7.969796E+07, 9.020622E+07, 1.020472E+08, 1.153824E+08, 1.303924E+08, 1.472781E+08,
    1.662635E+08, 1.875980E+08, 2.115592E+08, 2.384560E+08, 2.686314E+08, 3.024668E+08,
    3.403857E+08, 3.828581E+08, 4.304053E+08, 4.836051E+08, 5.430980E+08, 6.095931E+08,
    6.838754E+08, 7.668134E+08, 8.593671E+08, 9.625978E+08, 1.077678E+09, 1.205900E+09,
    1.348694E+09, 1.507634E+09, 1.684456E+09, 1.881072E+09, 2.099589E+09, 2.342326E+09,
    2.611834E+09, 2.910918E+09, 3.242661E+09, 3.610451E+09, 4.018007E+09, 4.469413E+09,
    4.969148E+09, 5.522123E+09, 6.133724E+09, 6.809848E+09, 7.556956E+09, 8.382119E+09,
    9.293073E+09, 1.029828E+10, 1.140699E+10, 1.262932E+10, 1.397629E+10, 1.545998E+10,
    1.709352E+10, 1.889128E+10, 2.086891E+10, 2.304345E+10, 2.543350E+10, 2.805929E+10,
    3.094287E+10, 3.410821E+10, 3.758140E+10, 4.139081E+10, 4.556728E+10, 5.014429E+10,
    5.515826E+10, 6.064867E+10, 6.665842E+10, 7.323403E+10, 8.042594E+10, 8.828887E+10,
    9.688209E+10, 1.062698E+11, 1.165216E+11, 1.277127E+11, 1.399247E+11, 1.532456E+11,
    1.677709E+11, 1.836035E+11, 2.008548E+11, 2.196451E+11, 2.401044E+11, 2.623730E+11,
    2.866023E+11, 3.129555E+11, 3.416090E+11, 3.727526E+11, 4.065911E+11, 4.433452E+11,
    4.832525E+11, 5.265688E+11, 5.735698E+11, 6.245518E+11, 6.798340E+11, 7.397592E+11,
    8.046964E+11, 8.750421E+11, 9.512223E+11, 1.033695E+12, 1.122951E+12, 1.219518E+12,
    1.323963E+12, 1.436894E+12, 1.558962E+12, 1.690867E+12, 1.833358E+12, 1.987238E+12,
    2.153368E+12, 2.332671E+12,
])

#  ---------------------- M = 28, I = 1 ---------------------------
M = 28
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[4]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    8.000210E+00, 6.080058E+01, 1.634756E+02, 2.965282E+02, 4.537635E+02, 6.319162E+02,
    8.288067E+02, 1.042867E+03, 1.272974E+03, 1.518410E+03, 1.778850E+03, 2.054349E+03,
    2.345319E+03, 2.652478E+03, 2.976809E+03, 3.319511E+03, 3.681956E+03, 4.065668E+03,
    4.472291E+03, 4.903585E+03, 5.361415E+03, 5.847747E+03, 6.364650E+03, 6.914302E+03,
    7.498988E+03, 8.121115E+03, 8.783208E+03, 9.487927E+03, 1.023807E+04, 1.103656E+04,
    1.188652E+04, 1.279117E+04, 1.375395E+04, 1.477846E+04, 1.586846E+04, 1.702794E+04,
    1.826107E+04, 1.957224E+04, 2.096604E+04, 2.244732E+04, 2.402115E+04, 2.569284E+04,
    2.746798E+04, 2.935242E+04, 3.135229E+04, 3.347401E+04, 3.572430E+04, 3.811021E+04,
    4.063911E+04, 4.331870E+04, 4.615704E+04, 4.916257E+04, 5.234409E+04, 5.571079E+04,
    5.927230E+04, 6.303863E+04, 6.702025E+04, 7.122808E+04, 7.567351E+04, 8.036840E+04,
    8.532511E+04, 9.055653E+04, 9.607607E+04, 1.018977E+05, 1.080359E+05, 1.145057E+05,
    1.213230E+05, 1.285039E+05, 1.360654E+05, 1.440252E+05, 1.524013E+05, 1.612128E+05,
    1.704793E+05, 1.802210E+05, 1.904590E+05, 2.012151E+05, 2.125118E+05, 2.243723E+05,
    2.368209E+05, 2.498822E+05, 2.635820E+05, 2.779468E+05, 2.930039E+05, 3.087813E+05,
    3.253080E+05, 3.426140E+05, 3.607297E+05, 3.796869E+05, 3.995178E+05, 4.202557E+05,
    4.419347E+05, 4.645899E+05, 4.882572E+05, 5.129732E+05, 5.387755E+05, 5.657027E+05,
    5.937941E+05, 6.230898E+05, 6.536308E+05, 6.854591E+05, 7.186173E+05, 7.531488E+05,
    7.890981E+05, 8.265101E+05, 8.654309E+05, 9.059069E+05, 9.479856E+05, 9.917151E+05,
    1.037144E+06, 1.084322E+06, 1.133300E+06, 1.184128E+06, 1.236857E+06, 1.291540E+06,
    1.348229E+06, 1.406978E+06, 1.467840E+06, 1.530870E+06, 1.596122E+06, 1.663652E+06,
    1.733515E+06, 1.805768E+06, 1.880466E+06, 1.957667E+06, 2.037427E+06, 2.119805E+06,
    2.204858E+06, 2.292643E+06, 2.383220E+06, 2.476646E+06, 2.572979E+06, 2.672280E+06,
    2.774606E+06, 2.880016E+06, 2.988569E+06, 3.100324E+06, 3.215340E+06, 3.333676E+06,
    3.455390E+06, 3.580542E+06, 3.709190E+06, 3.841392E+06, 3.977207E+06, 4.116693E+06,
    4.259908E+06, 4.406909E+06, 4.557755E+06, 4.712501E+06, 4.871206E+06, 5.033924E+06,
    5.200714E+06, 5.371630E+06, 5.546727E+06, 5.726062E+06, 5.909687E+06, 6.097659E+06,
    6.290029E+06, 6.486851E+06, 6.688177E+06, 6.894061E+06, 7.104553E+06, 7.319705E+06,
    7.539566E+06, 7.764186E+06, 7.993616E+06, 8.227903E+06, 8.467096E+06, 8.711241E+06,
    8.960386E+06, 9.214576E+06, 9.473858E+06, 9.738275E+06, 1.000787E+07, 1.028269E+07,
    1.056277E+07, 1.084817E+07, 1.113891E+07, 1.143504E+07, 1.173659E+07, 1.204362E+07,
    1.235615E+07, 1.267422E+07, 1.299787E+07, 1.332714E+07, 1.366206E+07, 1.400266E+07,
    1.434898E+07, 1.470106E+07, 1.505891E+07, 1.542258E+07, 1.579209E+07, 1.616748E+07,
    1.654877E+07, 1.693599E+07, 1.732917E+07, 1.772834E+07, 1.813351E+07, 1.854472E+07,
    1.896199E+07, 1.938534E+07, 1.981480E+07, 2.025039E+07, 2.069212E+07, 2.114002E+07,
    2.159411E+07, 2.205441E+07, 2.252093E+07, 2.299369E+07, 2.347271E+07, 2.395800E+07,
    2.444958E+07, 2.494747E+07, 2.545167E+07, 2.596220E+07, 2.647907E+07, 2.700230E+07,
    2.753188E+07, 2.806784E+07, 2.861019E+07, 2.915892E+07, 2.971405E+07, 3.027559E+07,
    3.084354E+07, 3.141790E+07, 3.199869E+07, 3.258591E+07,
])

#  ---------------------- M = 29, I = 1 ---------------------------
M = 29
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.300330E+01, 1.060780E+03, 2.994480E+03, 5.499119E+03, 8.466905E+03, 1.183885E+04,
    1.558590E+04, 1.970577E+04, 2.422011E+04, 2.917030E+04, 3.461299E+04, 4.061669E+04,
    4.725957E+04, 5.462826E+04, 6.281747E+04, 7.193003E+04, 8.207734E+04, 9.337996E+04,
    1.059684E+05, 1.199837E+05, 1.355789E+05, 1.529192E+05, 1.721835E+05, 1.935653E+05,
    2.172735E+05, 2.435340E+05, 2.725902E+05, 3.047047E+05, 3.401604E+05, 3.792616E+05,
    4.223355E+05, 4.697336E+05, 5.218332E+05, 5.790385E+05, 6.417827E+05, 7.105294E+05,
    7.857744E+05, 8.680471E+05, 9.579127E+05, 1.055974E+06, 1.162873E+06, 1.279295E+06,
    1.405966E+06, 1.543660E+06, 1.693200E+06, 1.855457E+06, 2.031357E+06, 2.221881E+06,
    2.428067E+06, 2.651016E+06, 2.891891E+06, 3.151923E+06, 3.432411E+06, 3.734728E+06,
    4.060320E+06, 4.410715E+06, 4.787525E+06, 5.192441E+06, 5.627252E+06, 6.093834E+06,
    6.594162E+06, 7.130313E+06, 7.704467E+06, 8.318917E+06, 8.976065E+06, 9.678431E+06,
    1.042866E+07, 1.122953E+07, 1.208393E+07, 1.299491E+07, 1.396564E+07, 1.499946E+07,
    1.609985E+07, 1.727043E+07, 1.851502E+07, 1.983759E+07, 2.124227E+07, 2.273339E+07,
    2.431547E+07, 2.599322E+07, 2.777152E+07, 2.965551E+07, 3.165049E+07, 3.376202E+07,
    3.599588E+07, 3.835806E+07, 4.085482E+07, 4.349265E+07, 4.627831E+07, 4.921882E+07,
    5.232149E+07, 5.559387E+07, 5.904385E+07, 6.267958E+07, 6.650956E+07, 7.054254E+07,
    7.478767E+07, 7.925441E+07, 8.395253E+07, 8.889220E+07, 9.408394E+07, 9.953866E+07,
    1.052676E+08, 1.112825E+08, 1.175954E+08, 1.242189E+08, 1.311658E+08, 1.384496E+08,
    1.460840E+08, 1.540834E+08, 1.624626E+08, 1.712367E+08, 1.804217E+08, 1.900338E+08,
    2.000898E+08, 2.106070E+08, 2.216035E+08, 2.330976E+08, 2.451085E+08, 2.576558E+08,
    2.707598E+08, 2.844415E+08, 2.987224E+08, 3.136247E+08, 3.291713E+08, 3.453860E+08,
    3.622928E+08, 3.799169E+08, 3.982840E+08, 4.174208E+08, 4.373543E+08, 4.581129E+08,
    4.797253E+08, 5.022216E+08, 5.256319E+08, 5.499880E+08, 5.753222E+08, 6.016678E+08,
    6.290589E+08, 6.575307E+08, 6.871194E+08, 7.178619E+08, 7.497965E+08, 7.829623E+08,
    8.173995E+08, 8.531493E+08, 8.902543E+08, 9.287578E+08, 9.687047E+08, 1.010141E+09,
    1.053113E+09, 1.097669E+09, 1.143859E+09, 1.191734E+09, 1.241345E+09, 1.292747E+09,
    1.345992E+09, 1.401139E+09, 1.458245E+09, 1.517367E+09, 1.578567E+09, 1.641907E+09,
    1.707449E+09, 1.775259E+09, 1.845404E+09, 1.917950E+09, 1.992969E+09, 2.070532E+09,
    2.150711E+09, 2.233580E+09, 2.319218E+09, 2.407703E+09, 2.499113E+09, 2.593533E+09,
    2.691044E+09, 2.791734E+09,
])

#  ---------------------- M = 29, I = 2 ---------------------------
M = 29
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.600659E+01, 2.121560E+03, 5.988960E+03, 1.099824E+04, 1.693381E+04, 2.367770E+04,
    3.117187E+04, 3.941173E+04, 4.844066E+04, 5.834143E+04, 6.922734E+04, 8.123543E+04,
    9.452204E+04, 1.092604E+05, 1.256400E+05, 1.438665E+05, 1.641626E+05, 1.867695E+05,
    2.119482E+05, 2.399810E+05, 2.711735E+05, 3.058566E+05, 3.443879E+05, 3.871544E+05,
    4.345740E+05, 4.870983E+05, 5.452144E+05, 6.094476E+05, 6.803632E+05, 7.585704E+05,
    8.447233E+05, 9.395250E+05, 1.043730E+06, 1.158147E+06, 1.283642E+06, 1.421143E+06,
    1.571641E+06, 1.736195E+06, 1.915934E+06, 2.112067E+06, 2.325876E+06, 2.558730E+06,
    2.812083E+06, 3.087485E+06, 3.386577E+06, 3.711105E+06, 4.062920E+06, 4.443983E+06,
    4.856372E+06, 5.302289E+06, 5.784058E+06, 6.304143E+06, 6.865140E+06, 7.469795E+06,
    8.121003E+06, 8.821819E+06, 9.575464E+06, 1.038532E+07, 1.125497E+07, 1.218817E+07,
    1.318886E+07, 1.426120E+07, 1.540954E+07, 1.663848E+07, 1.795281E+07, 1.935759E+07,
    2.085809E+07, 2.245987E+07, 2.416872E+07, 2.599072E+07, 2.793224E+07, 2.999994E+07,
    3.220077E+07, 3.454200E+07, 3.703124E+07, 3.967644E+07, 4.248586E+07, 4.546818E+07,
    4.863242E+07, 5.198798E+07, 5.554466E+07, 5.931272E+07, 6.330278E+07, 6.752592E+07,
    7.199374E+07, 7.671818E+07, 8.171180E+07, 8.698757E+07, 9.255900E+07, 9.844013E+07,
    1.046456E+08, 1.111905E+08, 1.180905E+08, 1.253621E+08, 1.330222E+08, 1.410883E+08,
    1.495787E+08, 1.585123E+08, 1.679087E+08, 1.777882E+08, 1.881719E+08, 1.990815E+08,
    2.105395E+08, 2.225695E+08, 2.351955E+08, 2.484426E+08, 2.623366E+08, 2.769043E+08,
    2.921733E+08, 3.081724E+08, 3.249309E+08, 3.424795E+08, 3.608497E+08, 3.800740E+08,
    4.001862E+08, 4.212210E+08, 4.432141E+08, 4.662025E+08, 4.902245E+08, 5.153195E+08,
    5.415277E+08, 5.688914E+08, 5.974535E+08, 6.272584E+08, 6.583520E+08, 6.907814E+08,
    7.245953E+08, 7.598439E+08, 7.965784E+08, 8.348521E+08, 8.747197E+08, 9.162371E+08,
    9.594623E+08, 1.004455E+09, 1.051276E+09, 1.099989E+09, 1.150657E+09, 1.203349E+09,
    1.258132E+09, 1.315075E+09, 1.374253E+09, 1.435738E+09, 1.499608E+09, 1.565940E+09,
    1.634815E+09, 1.706315E+09, 1.780525E+09, 1.857533E+09, 1.937427E+09, 2.020299E+09,
    2.106243E+09, 2.195356E+09, 2.287737E+09, 2.383486E+09, 2.482710E+09, 2.585513E+09,
    2.692005E+09, 2.802299E+09, 2.916510E+09, 3.034755E+09, 3.157156E+09, 3.283835E+09,
    3.414921E+09, 3.550541E+09, 3.690830E+09, 3.835924E+09, 3.985962E+09, 4.141087E+09,
    4.301445E+09, 4.467185E+09, 4.638461E+09, 4.815431E+09, 4.998252E+09, 5.187090E+09,
    5.382113E+09, 5.583493E+09,
])

#  ---------------------- M = 30, I = 1 ---------------------------
M = 30
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.021972E+02, 8.926883E+03, 2.522955E+04, 4.637336E+04, 7.178418E+04, 1.020099E+05,
    1.386432E+05, 1.842557E+05, 2.424692E+05, 3.181910E+05, 4.180157E+05, 5.508060E+05,
    7.284907E+05, 9.671311E+05, 1.288340E+06, 1.721151E+06, 2.304480E+06, 3.090346E+06,
    4.148075E+06, 5.569754E+06, 7.477295E+06, 1.003153E+07, 1.344388E+07, 1.799125E+07,
    2.403505E+07, 3.204519E+07, 4.263047E+07, 5.657681E+07, 7.489511E+07, 9.888099E+07,
    1.301893E+08, 1.709260E+08, 2.237627E+08, 2.920765E+08, 3.801222E+08, 4.932437E+08,
    6.381305E+08, 8.231307E+08, 1.058630E+09, 1.357512E+09, 1.735707E+09, 2.212863E+09,
    2.813136E+09, 3.566151E+09, 4.508136E+09, 5.683276E+09, 7.145326E+09, 8.959512E+09,
    1.120479E+10, 1.397648E+10, 1.738943E+10, 2.158165E+10, 2.671863E+10, 3.299836E+10,
    4.065719E+10, 4.997670E+10, 6.129157E+10, 7.499890E+10, 9.156880E+10, 1.115568E+11,
    1.356180E+11, 1.645237E+11, 1.991801E+11, 2.406500E+11, 2.901778E+11, 3.492179E+11,
    4.194675E+11, 5.029034E+11, 6.018245E+11, 7.189005E+11, 8.572261E+11, 1.020383E+12,
    1.212513E+12, 1.438393E+12, 1.703531E+12, 2.014262E+12, 2.377871E+12, 2.802713E+12,
    3.298368E+12, 3.875798E+12, 4.547536E+12, 5.327887E+12, 6.233164E+12, 7.281945E+12,
    8.495361E+12, 9.897418E+12, 1.151536E+13, 1.338007E+13, 1.552651E+13, 1.799421E+13,
    2.082787E+13, 2.407789E+13, 2.780112E+13, 3.206156E+13, 3.693125E+13, 4.249113E+13,
    4.883210E+13, 5.605613E+13, 6.427748E+13, 7.362411E+13, 8.423914E+13, 9.628255E+13,
    1.099330E+14, 1.253896E+14, 1.428748E+14, 1.626357E+14, 1.849479E+14, 2.101173E+14,
    2.384842E+14, 2.704260E+14, 3.063617E+14, 3.467555E+14, 3.921214E+14, 4.430285E+14,
    5.001060E+14, 5.640492E+14, 6.356260E+14, 7.156838E+14, 8.051568E+14, 9.050747E+14,
    1.016571E+15, 1.140894E+15, 1.279415E+15, 1.433643E+15, 1.605233E+15, 1.796004E+15,
    2.007949E+15, 2.243255E+15, 2.504315E+15, 2.793752E+15, 3.114435E+15, 3.469502E+15,
    3.862381E+15, 4.296819E+15, 4.776905E+15, 5.307101E+15, 5.892273E+15, 6.537723E+15,
    7.249229E+15, 8.033080E+15, 8.896120E+15, 9.845792E+15, 1.089019E+16, 1.203810E+16,
    1.329907E+16, 1.468347E+16, 1.620253E+16, 1.786845E+16, 1.969444E+16, 2.169481E+16,
    2.388505E+16, 2.628195E+16, 2.890364E+16, 3.176975E+16, 3.490150E+16, 3.832178E+16,
    4.205536E+16, 4.612893E+16, 5.057131E+16, 5.541360E+16, 6.068930E+16, 6.643451E+16,
    7.268815E+16, 7.949209E+16, 8.689142E+16, 9.493463E+16, 1.036739E+17, 1.131653E+17,
    1.234691E+17, 1.346500E+17, 1.467777E+17, 1.599265E+17, 1.741768E+17, 1.896144E+17,
    2.063313E+17, 2.244263E+17,
])

#  ---------------------- M = 31, I = 1 ---------------------------
M = 31
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[6]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.000000E+00, 9.577640E+00, 2.619804E+01, 4.723879E+01, 7.205446E+01, 1.001556E+02,
    1.312023E+02, 1.649444E+02, 2.011901E+02, 2.397898E+02, 2.806276E+02, 3.236172E+02,
    3.686976E+02, 4.158306E+02, 4.649971E+02, 5.161940E+02, 5.694318E+02, 6.247317E+02,
    6.821239E+02, 7.416459E+02, 8.033414E+02, 8.672592E+02, 9.334532E+02, 1.001981E+03,
    1.072904E+03, 1.146289E+03, 1.222203E+03, 1.300719E+03, 1.381913E+03, 1.465863E+03,
    1.552650E+03, 1.642361E+03, 1.735081E+03, 1.830901E+03, 1.929915E+03, 2.032218E+03,
    2.137907E+03, 2.247085E+03, 2.359854E+03, 2.476320E+03, 2.596589E+03, 2.720774E+03,
    2.848985E+03, 2.981338E+03, 3.117948E+03, 3.258936E+03, 3.404422E+03, 3.554529E+03,
    3.709382E+03, 3.869109E+03, 4.033838E+03, 4.203702E+03, 4.378833E+03, 4.559368E+03,
    4.745444E+03, 4.937200E+03, 5.134779E+03, 5.338325E+03, 5.547982E+03, 5.763900E+03,
    5.986229E+03, 6.215121E+03, 6.450730E+03, 6.693213E+03, 6.942728E+03, 7.199438E+03,
    7.463503E+03, 7.735091E+03, 8.014367E+03, 8.301502E+03, 8.596667E+03, 8.900036E+03,
    9.211786E+03, 9.532093E+03, 9.861140E+03, 1.019911E+04, 1.054618E+04, 1.090255E+04,
    1.126840E+04, 1.164392E+04, 1.202931E+04, 1.242476E+04, 1.283047E+04, 1.324664E+04,
    1.367348E+04, 1.411118E+04, 1.455995E+04, 1.502000E+04, 1.549154E+04, 1.597478E+04,
    1.646995E+04, 1.697724E+04, 1.749688E+04, 1.802909E+04, 1.857410E+04, 1.913212E+04,
    1.970338E+04, 2.028811E+04, 2.088653E+04, 2.149888E+04, 2.212539E+04, 2.276628E+04,
    2.342180E+04, 2.409219E+04, 2.477767E+04, 2.547848E+04, 2.619488E+04, 2.692709E+04,
    2.767536E+04, 2.843993E+04, 2.922104E+04, 3.001895E+04, 3.083389E+04, 3.166610E+04,
    3.251585E+04, 3.338337E+04, 3.426891E+04, 3.517272E+04, 3.609505E+04, 3.703614E+04,
    3.799625E+04, 3.897562E+04, 3.997449E+04, 4.099313E+04, 4.203177E+04, 4.309066E+04,
    4.417005E+04, 4.527018E+04, 4.639131E+04, 4.753367E+04, 4.869751E+04, 4.988308E+04,
    5.109061E+04, 5.232035E+04, 5.357253E+04, 5.484740E+04, 5.614520E+04, 5.746615E+04,
    5.881050E+04, 6.017847E+04, 6.157031E+04, 6.298623E+04, 6.442647E+04, 6.589125E+04,
    6.738080E+04, 6.889533E+04, 7.043508E+04, 7.200025E+04, 7.359107E+04, 7.520774E+04,
    7.685048E+04, 7.851950E+04, 8.021500E+04, 8.193718E+04, 8.368625E+04, 8.546241E+04,
    8.726584E+04, 8.909676E+04, 9.095533E+04, 9.284176E+04, 9.475623E+04, 9.669891E+04,
    9.866999E+04, 1.006696E+05, 1.026980E+05, 1.047553E+05, 1.068417E+05, 1.089574E+05,
    1.111024E+05, 1.132770E+05, 1.154813E+05, 1.177155E+05, 1.199797E+05, 1.222740E+05,
    1.245986E+05, 1.269537E+05, 1.293393E+05, 1.317557E+05, 1.342028E+05, 1.366809E+05,
    1.391901E+05, 1.417304E+05, 1.443021E+05, 1.469051E+05, 1.495397E+05, 1.522058E+05,
    1.549037E+05, 1.576334E+05, 1.603950E+05, 1.631886E+05, 1.660143E+05, 1.688721E+05,
    1.717622E+05, 1.746846E+05, 1.776393E+05, 1.806266E+05, 1.836463E+05, 1.866986E+05,
    1.897836E+05, 1.929013E+05, 1.960517E+05,
])

#  ---------------------- M = 31, I = 2 ---------------------------
M = 31
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.000000E+00, 9.858680E+00, 2.626126E+01, 4.730993E+01, 7.213596E+01, 1.002319E+02,
    1.312533E+02, 1.649477E+02, 2.011172E+02, 2.396146E+02, 2.803171E+02, 3.231407E+02,
    3.680172E+02, 4.149141E+02, 4.638055E+02, 5.146919E+02, 5.675764E+02, 6.224870E+02,
    6.794483E+02, 7.384938E+02, 7.996744E+02, 8.630347E+02, 9.286157E+02, 9.964913E+02,
    1.066714E+03, 1.139338E+03, 1.214443E+03, 1.292091E+03, 1.372359E+03, 1.455313E+03,
    1.541049E+03, 1.629623E+03, 1.721154E+03, 1.815684E+03, 1.913338E+03, 2.014177E+03,
    2.118315E+03, 2.225852E+03, 2.336870E+03, 2.451471E+03, 2.569761E+03, 2.691824E+03,
    2.817792E+03, 2.947774E+03, 3.081835E+03, 3.220139E+03, 3.362775E+03, 3.509861E+03,
    3.661488E+03, 3.817802E+03, 3.978929E+03, 4.144958E+03, 4.316016E+03, 4.492262E+03,
    4.673789E+03, 4.860762E+03, 5.053242E+03, 5.251395E+03, 5.455392E+03, 5.665333E+03,
    5.881313E+03, 6.103551E+03, 6.332104E+03, 6.567196E+03, 6.808930E+03, 7.057402E+03,
    7.312850E+03, 7.575326E+03, 7.845073E+03, 8.122195E+03, 8.406841E+03, 8.699164E+03,
    8.999319E+03, 9.307513E+03, 9.623851E+03, 9.948547E+03, 1.028170E+04, 1.062348E+04,
    1.097410E+04, 1.133374E+04, 1.170255E+04, 1.208063E+04, 1.246830E+04, 1.286558E+04,
    1.327280E+04, 1.369003E+04, 1.411755E+04, 1.455545E+04, 1.500391E+04, 1.546326E+04,
    1.593352E+04, 1.641497E+04, 1.690779E+04, 1.741224E+04, 1.792835E+04, 1.845655E+04,
    1.899688E+04, 1.954952E+04, 2.011484E+04, 2.069287E+04, 2.128398E+04, 2.188828E+04,
    2.250597E+04, 2.313735E+04, 2.378263E+04, 2.444192E+04, 2.511562E+04, 2.580384E+04,
    2.650680E+04, 2.722481E+04, 2.795798E+04, 2.870665E+04, 2.947101E+04, 3.025131E+04,
    3.104787E+04, 3.186080E+04, 3.269035E+04, 3.353694E+04, 3.440060E+04, 3.528166E+04,
    3.618048E+04, 3.709719E+04, 3.803214E+04, 3.898543E+04, 3.995757E+04, 4.094855E+04,
    4.195886E+04, 4.298876E+04, 4.403836E+04, 4.510792E+04, 4.619795E+04, 4.730858E+04,
    4.844006E+04, 4.959280E+04, 5.076704E+04, 5.196291E+04, 5.318083E+04, 5.442120E+04,
    5.568414E+04, 5.697007E+04, 5.827913E+04, 5.961172E+04, 6.096828E+04, 6.234896E+04,
    6.375399E+04, 6.518386E+04, 6.663882E+04, 6.811918E+04, 6.962521E+04, 7.115721E+04,
    7.271564E+04, 7.430080E+04, 7.591297E+04, 7.755248E+04, 7.921976E+04, 8.091481E+04,
    8.263842E+04, 8.439059E+04, 8.617196E+04, 8.798264E+04, 8.982298E+04, 9.169347E+04,
    9.359443E+04, 9.552618E+04, 9.748903E+04, 9.948332E+04, 1.015096E+05, 1.035681E+05,
    1.056590E+05, 1.077831E+05, 1.099405E+05, 1.121316E+05, 1.143570E+05, 1.166166E+05,
    1.189111E+05, 1.212410E+05,
])

#  ---------------------- M = 31, I = 3 ---------------------------
M = 31
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    4.000000E+00, 3.938981E+01, 1.049203E+02, 1.890120E+02, 2.881945E+02, 4.004404E+02,
    5.243733E+02, 6.589857E+02, 8.034856E+02, 9.572857E+02, 1.119896E+03, 1.290979E+03,
    1.470264E+03, 1.657621E+03, 1.852946E+03, 2.056241E+03, 2.267518E+03, 2.486890E+03,
    2.714455E+03, 2.950346E+03, 3.194767E+03, 3.447896E+03, 3.709897E+03, 3.981065E+03,
    4.261609E+03, 4.551747E+03, 4.851797E+03, 5.162008E+03, 5.482685E+03, 5.814092E+03,
    6.156613E+03, 6.510473E+03, 6.876144E+03, 7.253798E+03, 7.643934E+03, 8.046793E+03,
    8.462831E+03, 8.892449E+03, 9.335972E+03, 9.793812E+03, 1.026639E+04, 1.075404E+04,
    1.125729E+04, 1.177658E+04, 1.231216E+04, 1.286470E+04, 1.343454E+04, 1.402216E+04,
    1.462791E+04, 1.525240E+04, 1.589611E+04, 1.655942E+04, 1.724280E+04, 1.794692E+04,
    1.867213E+04, 1.941910E+04, 2.018807E+04, 2.097971E+04, 2.179469E+04, 2.263342E+04,
    2.349628E+04, 2.438413E+04, 2.529722E+04, 2.623643E+04, 2.720217E+04, 2.819483E+04,
    2.921536E+04, 3.026397E+04, 3.134163E+04, 3.244875E+04, 3.358594E+04, 3.475379E+04,
    3.595293E+04, 3.718418E+04, 3.844797E+04, 3.974515E+04, 4.107614E+04, 4.244157E+04,
    4.384233E+04, 4.527909E+04, 4.675250E+04, 4.826299E+04, 4.981175E+04, 5.139893E+04,
    5.302576E+04, 5.469267E+04, 5.640061E+04, 5.815005E+04, 5.994168E+04, 6.177680E+04,
    6.365556E+04, 6.557899E+04, 6.754782E+04, 6.956311E+04, 7.162501E+04, 7.373522E+04,
    7.589386E+04, 7.810171E+04, 8.036020E+04, 8.266947E+04, 8.503099E+04, 8.744521E+04,
    8.991295E+04, 9.243536E+04, 9.501329E+04, 9.764719E+04, 1.003387E+05, 1.030882E+05,
    1.058965E+05, 1.087650E+05, 1.116941E+05, 1.146851E+05, 1.177388E+05, 1.208561E+05,
    1.240384E+05, 1.272861E+05, 1.306002E+05, 1.339824E+05, 1.374328E+05, 1.409527E+05,
    1.445436E+05, 1.482059E+05, 1.519411E+05, 1.557495E+05, 1.596333E+05, 1.635923E+05,
    1.676286E+05, 1.717431E+05, 1.759363E+05, 1.802093E+05, 1.845641E+05, 1.890011E+05,
    1.935214E+05, 1.981267E+05, 2.028178E+05, 2.075955E+05, 2.124611E+05, 2.174165E+05,
    2.224620E+05, 2.275994E+05, 2.328292E+05, 2.381530E+05, 2.435725E+05, 2.490884E+05,
    2.547017E+05, 2.604140E+05, 2.662267E+05, 2.721408E+05, 2.781576E+05, 2.842780E+05,
    2.905041E+05, 2.968369E+05, 3.032776E+05, 3.098275E+05, 3.164884E+05, 3.232603E+05,
    3.301462E+05, 3.371462E+05, 3.442629E+05, 3.514967E+05, 3.588490E+05, 3.663217E+05,
    3.739162E+05, 3.816337E+05, 3.894754E+05, 3.974428E+05, 4.055377E+05, 4.137616E+05,
    4.221151E+05, 4.306009E+05, 4.392197E+05, 4.479736E+05, 4.568640E+05, 4.658915E+05,
    4.750581E+05, 4.843661E+05,
])

#  ---------------------- M = 32, I = 1 ---------------------------
M = 32
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    8.098530E+00, 6.168274E+02, 1.738066E+03, 3.189875E+03, 4.909798E+03, 6.862997E+03,
    9.030079E+03, 1.140495E+04, 1.399259E+04, 1.680815E+04, 1.987372E+04, 2.321812E+04,
    2.687559E+04, 3.088517E+04, 3.529073E+04, 4.014075E+04, 4.548994E+04, 5.139632E+04,
    5.792593E+04, 6.514813E+04, 7.314122E+04, 8.198985E+04, 9.178577E+04, 1.026295E+05,
    1.146311E+05, 1.279089E+05, 1.425943E+05, 1.588277E+05, 1.767626E+05, 1.965672E+05,
    2.184237E+05, 2.425277E+05, 2.690945E+05, 2.983565E+05, 3.305651E+05, 3.659933E+05,
    4.049357E+05, 4.477128E+05, 4.946682E+05, 5.461776E+05, 6.026400E+05, 6.644928E+05,
    7.322038E+05, 8.062787E+05, 8.872603E+05, 9.757342E+05, 1.072333E+06, 1.177728E+06,
    1.292652E+06, 1.417886E+06, 1.554264E+06, 1.702691E+06, 1.864125E+06, 2.039604E+06,
    2.230231E+06, 2.437189E+06, 2.661747E+06, 2.905255E+06, 3.169158E+06, 3.455004E+06,
    3.764439E+06, 4.099224E+06, 4.461229E+06, 4.852460E+06, 5.275039E+06, 5.731238E+06,
    6.223470E+06, 6.754297E+06, 7.326455E+06, 7.942838E+06, 8.606536E+06, 9.320818E+06,
    1.008916E+07, 1.091524E+07, 1.180298E+07, 1.275651E+07, 1.378023E+07, 1.487879E+07,
    1.605711E+07, 1.732042E+07, 1.867421E+07, 2.012433E+07, 2.167694E+07, 2.333855E+07,
    2.511606E+07, 2.701671E+07, 2.904820E+07, 3.121862E+07, 3.353650E+07, 3.601086E+07,
    3.865118E+07, 4.146747E+07, 4.447026E+07, 4.767066E+07, 5.108035E+07, 5.471163E+07,
    5.857740E+07, 6.269129E+07, 6.706761E+07, 7.172137E+07, 7.666838E+07, 8.192520E+07,
    8.750927E+07, 9.343889E+07, 9.973322E+07, 1.064124E+08, 1.134976E+08, 1.210109E+08,
    1.289756E+08, 1.374159E+08, 1.463573E+08, 1.558267E+08, 1.658518E+08, 1.764620E+08,
    1.876879E+08, 1.995616E+08, 2.121165E+08, 2.253878E+08, 2.394120E+08, 2.542275E+08,
    2.698744E+08, 2.863944E+08, 3.038314E+08, 3.222307E+08, 3.416402E+08, 3.621095E+08,
    3.836906E+08, 4.064374E+08, 4.304066E+08, 4.556569E+08, 4.822498E+08, 5.102492E+08,
    5.397218E+08, 5.707372E+08, 6.033677E+08, 6.376886E+08, 6.737783E+08, 7.117187E+08,
    7.515949E+08, 7.934953E+08, 8.375118E+08, 8.837404E+08, 9.322806E+08, 9.832362E+08,
    1.036715E+09, 1.092828E+09, 1.151692E+09, 1.213428E+09, 1.278162E+09, 1.346024E+09,
    1.417150E+09, 1.491679E+09, 1.569758E+09, 1.651538E+09, 1.737177E+09, 1.826838E+09,
    1.920689E+09, 2.018907E+09, 2.121673E+09, 2.229176E+09, 2.341612E+09, 2.459182E+09,
    2.582099E+09, 2.710579E+09, 2.844846E+09, 2.985137E+09, 3.131691E+09, 3.284760E+09,
    3.444603E+09, 3.611489E+09, 3.785695E+09, 3.967509E+09, 4.157229E+09, 4.355164E+09,
    4.561629E+09, 4.776956E+09,
])

#  ---------------------- M = 33, I = 1 ---------------------------
M = 33
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    5.594200E-01, 7.688782E+01, 2.147062E+02, 3.927665E+02, 6.034216E+02, 8.422352E+02,
    1.106209E+03, 1.393154E+03, 1.701421E+03, 2.029715E+03, 2.377155E+03, 2.743100E+03,
    3.127307E+03, 3.529659E+03, 3.950367E+03, 4.389807E+03, 4.848487E+03, 5.327087E+03,
    5.826359E+03, 6.347192E+03, 6.890410E+03, 7.456986E+03, 8.047900E+03, 8.664120E+03,
    9.306654E+03, 9.976636E+03, 1.067496E+04, 1.140281E+04, 1.216121E+04, 1.295117E+04,
    1.377383E+04, 1.463032E+04, 1.552167E+04, 1.644903E+04, 1.741356E+04, 1.841643E+04,
    1.945864E+04, 2.054151E+04, 2.166621E+04, 2.283391E+04, 2.404598E+04, 2.530358E+04,
    2.660771E+04, 2.796015E+04, 2.936189E+04, 3.081411E+04, 3.231868E+04, 3.387635E+04,
    3.548880E+04, 3.715773E+04, 3.888414E+04, 4.066950E+04, 4.251530E+04, 4.442335E+04,
    4.639489E+04, 4.843149E+04, 5.053503E+04, 5.270646E+04, 5.494802E+04, 5.726067E+04,
    5.964671E+04, 6.210782E+04, 6.464494E+04, 6.726087E+04, 6.995658E+04, 7.273377E+04,
    7.559498E+04, 7.854117E+04, 8.157495E+04, 8.469771E+04, 8.791125E+04, 9.121830E+04,
    9.461985E+04, 9.811822E+04, 1.017153E+05, 1.054135E+05, 1.092148E+05, 1.131206E+05,
    1.171339E+05, 1.212563E+05, 1.254897E+05, 1.298368E+05, 1.342996E+05, 1.388801E+05,
    1.435812E+05, 1.484048E+05, 1.533526E+05, 1.584280E+05, 1.636325E+05, 1.689691E+05,
    1.744392E+05, 1.800464E+05, 1.857931E+05, 1.916809E+05, 1.977128E+05, 2.038912E+05,
    2.102190E+05, 2.166981E+05, 2.233322E+05, 2.301232E+05, 2.370735E+05, 2.441870E+05,
    2.514648E+05, 2.589109E+05, 2.665280E+05, 2.743178E+05, 2.822846E+05, 2.904301E+05,
    2.987578E+05, 3.072706E+05, 3.159718E+05, 3.248634E+05, 3.339491E+05, 3.432315E+05,
    3.527136E+05, 3.623999E+05, 3.722915E+05, 3.823931E+05, 3.927067E+05, 4.032371E+05,
    4.139854E+05, 4.249565E+05, 4.361534E+05, 4.475791E+05, 4.592368E+05, 4.711305E+05,
    4.832632E+05, 4.956373E+05, 5.082588E+05, 5.211290E+05, 5.342531E+05, 5.476335E+05,
    5.612732E+05, 5.751779E+05, 5.893485E+05, 6.037919E+05, 6.185091E+05, 6.335047E+05,
    6.487834E+05, 6.643485E+05, 6.802023E+05, 6.963508E+05, 7.127974E+05, 7.295461E+05,
    7.465986E+05, 7.639627E+05, 7.816395E+05, 7.996338E+05, 8.179508E+05, 8.365924E+05,
    8.555657E+05, 8.748729E+05, 8.945175E+05, 9.145066E+05, 9.348407E+05, 9.555269E+05,
    9.765689E+05, 9.979710E+05, 1.019738E+06, 1.041873E+06, 1.064382E+06, 1.087269E+06,
    1.110538E+06, 1.134193E+06, 1.158240E+06, 1.182684E+06, 1.207529E+06, 1.232780E+06,
    1.258440E+06, 1.284515E+06, 1.311010E+06, 1.337930E+06, 1.365279E+06, 1.393063E+06,
    1.421286E+06, 1.449955E+06,
])

#  ---------------------- M = 35, I = 1 ---------------------------
M = 35
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.408818E+02, 2.081987E+04, 5.959366E+04, 1.144363E+05, 1.889620E+05, 2.880202E+05,
    4.176320E+05, 5.853070E+05, 8.004051E+05, 1.074538E+06, 1.422010E+06, 1.860338E+06,
    2.410828E+06, 3.099228E+06, 3.956556E+06, 5.019996E+06, 6.333930E+06, 7.951208E+06,
    9.934433E+06, 1.235765E+07, 1.530808E+07, 1.888813E+07, 2.321778E+07, 2.843710E+07,
    3.470920E+07, 4.222351E+07, 5.119940E+07, 6.189045E+07, 7.458876E+07, 8.963031E+07,
    1.074004E+08, 1.283401E+08, 1.529530E+08, 1.818133E+08, 2.155737E+08, 2.549750E+08,
    3.008566E+08, 3.541675E+08, 4.159785E+08, 4.874961E+08, 5.700773E+08, 6.652451E+08,
    7.747069E+08, 9.003730E+08, 1.044378E+09, 1.209104E+09, 1.397202E+09, 1.611625E+09,
    1.855648E+09, 2.132908E+09, 2.447432E+09, 2.803673E+09, 3.206556E+09, 3.661513E+09,
    4.174529E+09, 4.752195E+09, 5.401760E+09, 6.131182E+09, 6.949194E+09, 7.865367E+09,
    8.890177E+09, 1.003509E+10, 1.131261E+10, 1.273641E+10, 1.432140E+10, 1.608379E+10,
    1.804126E+10, 2.021301E+10, 2.261991E+10, 2.528461E+10, 2.823167E+10, 3.148772E+10,
    3.508157E+10, 3.904440E+10, 4.340993E+10, 4.821455E+10, 5.349755E+10, 5.930134E+10,
    6.567158E+10, 7.265748E+10, 8.031197E+10, 8.869200E+10, 9.785877E+10, 1.078780E+11,
    1.188203E+11, 1.307613E+11, 1.437821E+11, 1.579696E+11, 1.734170E+11, 1.902238E+11,
    2.084964E+11, 2.283487E+11, 2.499022E+11, 2.732868E+11, 2.986409E+11, 3.261123E+11,
    3.558583E+11, 3.880465E+11, 4.228556E+11, 4.604755E+11, 5.011085E+11, 5.449694E+11,
    5.922865E+11, 6.433023E+11, 6.982744E+11, 7.574760E+11, 8.211970E+11, 8.897447E+11,
    9.634444E+11, 1.042642E+12, 1.127701E+12, 1.219010E+12, 1.316977E+12, 1.422034E+12,
    1.534639E+12, 1.655275E+12, 1.784452E+12, 1.922709E+12, 2.070615E+12, 2.228770E+12,
    2.397806E+12, 2.578390E+12, 2.771227E+12, 2.977056E+12, 3.196658E+12, 3.430854E+12,
    3.680508E+12, 3.946531E+12, 4.229878E+12, 4.531554E+12, 4.852619E+12, 5.194181E+12,
    5.557407E+12, 5.943522E+12, 6.353813E+12, 6.789628E+12, 7.252383E+12, 7.743565E+12,
    8.264728E+12, 8.817504E+12, 9.403604E+12, 1.002482E+13, 1.068303E+13, 1.138019E+13,
    1.211836E+13, 1.289970E+13, 1.372645E+13, 1.460097E+13, 1.552571E+13, 1.650326E+13,
    1.753631E+13, 1.862765E+13, 1.978023E+13, 2.099711E+13, 2.228149E+13, 2.363672E+13,
    2.506627E+13, 2.657379E+13, 2.816307E+13, 2.983806E+13, 3.160289E+13, 3.346185E+13,
    3.541944E+13, 3.748031E+13, 3.964933E+13, 4.193156E+13, 4.433228E+13, 4.685697E+13,
    4.951135E+13, 5.230137E+13, 5.523320E+13, 5.831329E+13, 6.154832E+13, 6.494525E+13,
    6.851131E+13, 7.225402E+13,
])

#  ---------------------- M = 35, I = 2 ---------------------------
M = 35
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.468098E+02, 2.134900E+04, 6.110945E+04, 1.173478E+05, 1.937703E+05, 2.953497E+05,
    4.282605E+05, 6.002031E+05, 8.207761E+05, 1.101887E+06, 1.458203E+06, 1.907688E+06,
    2.472190E+06, 3.178113E+06, 4.057263E+06, 5.147772E+06, 6.495151E+06, 8.153595E+06,
    1.018730E+07, 1.267221E+07, 1.569774E+07, 1.936891E+07, 2.380877E+07, 2.916094E+07,
    3.559270E+07, 4.329828E+07, 5.250266E+07, 6.346585E+07, 7.648739E+07, 9.191183E+07,
    1.101343E+08, 1.316070E+08, 1.568465E+08, 1.864414E+08, 2.210611E+08, 2.614653E+08,
    3.085149E+08, 3.631828E+08, 4.265672E+08, 4.999054E+08, 5.845887E+08, 6.821789E+08,
    7.944272E+08, 9.232921E+08, 1.070963E+09, 1.239882E+09, 1.432769E+09, 1.652649E+09,
    1.902884E+09, 2.187202E+09, 2.509732E+09, 2.875042E+09, 3.288180E+09, 3.754718E+09,
    4.280793E+09, 4.873164E+09, 5.539263E+09, 6.287254E+09, 7.126088E+09, 8.065583E+09,
    9.116481E+09, 1.029053E+10, 1.160058E+10, 1.306063E+10, 1.468596E+10, 1.649322E+10,
    1.850051E+10, 2.072754E+10, 2.319571E+10, 2.592825E+10, 2.895033E+10, 3.228926E+10,
    3.597459E+10, 4.003830E+10, 4.451495E+10, 4.944187E+10, 5.485937E+10, 6.081089E+10,
    6.734329E+10, 7.450702E+10, 8.235636E+10, 9.094971E+10, 1.003498E+11, 1.106241E+11,
    1.218449E+11, 1.340899E+11, 1.474421E+11, 1.619909E+11, 1.778314E+11, 1.950660E+11,
    2.138038E+11, 2.341615E+11, 2.562637E+11, 2.802435E+11, 3.062431E+11, 3.344137E+11,
    3.649168E+11, 3.979245E+11, 4.336196E+11, 4.721973E+11, 5.138646E+11, 5.588420E+11,
    6.073635E+11, 6.596780E+11, 7.160494E+11, 7.767581E+11, 8.421012E+11, 9.123937E+11,
    9.879697E+11, 1.069183E+12, 1.156408E+12, 1.250041E+12, 1.350501E+12, 1.458233E+12,
    1.573704E+12, 1.697411E+12, 1.829877E+12, 1.971654E+12, 2.123324E+12, 2.285505E+12,
    2.458844E+12, 2.644025E+12, 2.841771E+12, 3.052839E+12, 3.278031E+12, 3.518189E+12,
    3.774199E+12, 4.046993E+12, 4.337553E+12, 4.646909E+12, 4.976146E+12, 5.326403E+12,
    5.698874E+12, 6.094819E+12, 6.515555E+12, 6.962464E+12, 7.436999E+12, 7.940684E+12,
    8.475113E+12, 9.041962E+12, 9.642982E+12, 1.028001E+13, 1.095497E+13, 1.166988E+13,
    1.242685E+13, 1.322807E+13, 1.407587E+13, 1.497265E+13, 1.592093E+13, 1.692337E+13,
    1.798271E+13, 1.910183E+13, 2.028375E+13, 2.153161E+13, 2.284869E+13, 2.423842E+13,
    2.570436E+13, 2.725025E+13, 2.887999E+13, 3.059761E+13, 3.240737E+13, 3.431365E+13,
    3.632107E+13, 3.843441E+13, 4.065863E+13, 4.299897E+13, 4.546079E+13, 4.804976E+13,
    5.077171E+13, 5.363275E+13, 5.663921E+13, 5.979770E+13, 6.311508E+13, 6.659849E+13,
    7.025532E+13, 7.409331E+13,
])

#  ---------------------- M = 36, I = 1 ---------------------------
M = 36
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    3.029540E+00, 2.200969E+01, 4.297675E+01, 6.395552E+01, 8.493845E+01, 1.059240E+02,
    1.269117E+02, 1.479015E+02, 1.688930E+02, 1.898864E+02, 2.108815E+02, 2.318784E+02,
    2.528769E+02, 2.738772E+02, 2.948822E+02, 3.158861E+02, 3.368985E+02, 3.579134E+02,
    3.789385E+02, 3.999708E+02, 4.210197E+02, 4.420868E+02, 4.631741E+02, 4.842933E+02,
    5.054422E+02, 5.266341E+02, 5.478727E+02, 5.691675E+02, 5.905169E+02, 6.119432E+02,
    6.334393E+02, 6.550162E+02, 6.766925E+02, 6.984603E+02, 7.203322E+02, 7.423211E+02,
    7.644255E+02, 7.866515E+02, 8.090126E+02, 8.315068E+02, 8.541484E+02, 8.769350E+02,
    8.998723E+02, 9.229750E+02, 9.462310E+02, 9.696643E+02, 9.932616E+02, 1.017048E+03,
    1.041000E+03, 1.065141E+03, 1.089479E+03, 1.113997E+03, 1.138710E+03, 1.163612E+03,
    1.188731E+03, 1.214038E+03, 1.239547E+03, 1.265277E+03, 1.291206E+03, 1.317352E+03,
    1.343706E+03, 1.370272E+03, 1.397067E+03, 1.424069E+03, 1.451294E+03, 1.478747E+03,
    1.506416E+03, 1.534307E+03, 1.562435E+03, 1.590776E+03, 1.619347E+03, 1.648150E+03,
    1.677191E+03, 1.706455E+03, 1.735962E+03, 1.765682E+03, 1.795650E+03, 1.825853E+03,
    1.856277E+03, 1.886958E+03, 1.917863E+03, 1.949013E+03, 1.980393E+03, 2.012022E+03,
    2.043868E+03, 2.075985E+03, 2.108322E+03, 2.140900E+03, 2.173719E+03, 2.206782E+03,
    2.240090E+03, 2.273646E+03, 2.307451E+03, 2.341488E+03, 2.375759E+03, 2.410284E+03,
    2.445065E+03, 2.480064E+03, 2.515321E+03, 2.550840E+03, 2.586581E+03, 2.622585E+03,
    2.658813E+03, 2.695308E+03, 2.732051E+03, 2.769042E+03, 2.806260E+03, 2.843730E+03,
    2.881473E+03, 2.919448E+03, 2.957655E+03, 2.996140E+03, 3.034859E+03, 3.073813E+03,
    3.113025E+03, 3.152497E+03, 3.192207E+03, 3.232179E+03, 3.272390E+03, 3.312840E+03,
    3.353556E+03, 3.394513E+03, 3.435739E+03, 3.477180E+03, 3.518892E+03, 3.560874E+03,
    3.603075E+03, 3.645548E+03, 3.688268E+03, 3.731208E+03, 3.774424E+03, 3.817915E+03,
    3.861629E+03, 3.905593E+03, 3.949807E+03, 3.994273E+03, 4.039020E+03, 4.083990E+03,
    4.129214E+03, 4.174693E+03, 4.220425E+03, 4.266414E+03, 4.312628E+03, 4.359128E+03,
    4.405856E+03, 4.452842E+03, 4.500085E+03, 4.547587E+03, 4.595349E+03, 4.643340E+03,
    4.691592E+03, 4.740104E+03, 4.788878E+03, 4.837883E+03, 4.887149E+03, 4.936647E+03,
    4.986441E+03, 5.036467E+03, 5.086725E+03, 5.137247E+03, 5.188036E+03, 5.239091E+03,
    5.290379E+03, 5.341935E+03, 5.393723E+03, 5.445780E+03, 5.498071E+03, 5.550631E+03,
    5.603462E+03, 5.656527E+03, 5.709827E+03, 5.763397E+03, 5.817240E+03, 5.871319E+03,
    5.925670E+03, 5.980257E+03,
])

#  ---------------------- M = 37, I = 1 ---------------------------
M = 37
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    5.785130E+00, 4.726760E+02, 1.331909E+03, 2.444466E+03, 3.762449E+03, 5.258748E+03,
    6.916853E+03, 8.727864E+03, 1.068858E+04, 1.280031E+04, 1.506728E+04, 1.749553E+04,
    2.009268E+04, 2.286748E+04, 2.582902E+04, 2.898719E+04, 3.235219E+04, 3.593458E+04,
    3.974451E+04, 4.379322E+04, 4.809173E+04, 5.265152E+04, 5.748313E+04, 6.259898E+04,
    6.800999E+04, 7.372804E+04, 7.976499E+04, 8.613334E+04, 9.284498E+04, 9.991164E+04,
    1.073465E+05, 1.151622E+05, 1.233719E+05, 1.319883E+05, 1.410250E+05, 1.504953E+05,
    1.604130E+05, 1.707918E+05, 1.816466E+05, 1.929925E+05, 2.048425E+05, 2.172130E+05,
    2.301186E+05, 2.435762E+05, 2.576005E+05, 2.722077E+05, 2.874142E+05, 3.032379E+05,
    3.196943E+05, 3.368020E+05, 3.545764E+05, 3.730385E+05, 3.922061E+05, 4.120955E+05,
    4.327284E+05, 4.541215E+05, 4.762955E+05, 4.992694E+05, 5.230665E+05, 5.477044E+05,
    5.732050E+05, 5.995885E+05, 6.268775E+05, 6.550925E+05, 6.842570E+05, 7.143946E+05,
    7.455270E+05, 7.776759E+05, 8.108659E+05, 8.451225E+05, 8.804679E+05, 9.169282E+05,
    9.545270E+05, 9.932902E+05, 1.033245E+06, 1.074416E+06, 1.116828E+06, 1.160511E+06,
    1.205491E+06, 1.251794E+06, 1.299452E+06, 1.348487E+06, 1.398931E+06, 1.450812E+06,
    1.504163E+06, 1.559011E+06, 1.615386E+06, 1.673318E+06, 1.732838E+06, 1.793976E+06,
    1.856768E+06, 1.921245E+06, 1.987436E+06, 2.055375E+06, 2.125098E+06, 2.196636E+06,
    2.270020E+06, 2.345293E+06, 2.422481E+06, 2.501629E+06, 2.582761E+06, 2.665923E+06,
    2.751144E+06, 2.838463E+06, 2.927919E+06, 3.019552E+06, 3.113392E+06, 3.209490E+06,
    3.307870E+06, 3.408585E+06, 3.511666E+06, 3.617155E+06, 3.725094E+06, 3.835528E+06,
    3.948487E+06, 4.064024E+06, 4.182175E+06, 4.302986E+06, 4.426502E+06, 4.552764E+06,
    4.681813E+06, 4.813699E+06, 4.948460E+06, 5.086155E+06, 5.226813E+06, 5.370497E+06,
    5.517235E+06, 5.667093E+06, 5.820104E+06, 5.976328E+06, 6.135809E+06, 6.298591E+06,
    6.464727E+06, 6.634271E+06, 6.807274E+06, 6.983775E+06, 7.163838E+06, 7.347517E+06,
    7.534852E+06, 7.725898E+06, 7.920720E+06, 8.119366E+06, 8.321879E+06, 8.528331E+06,
    8.738766E+06, 8.953250E+06, 9.171827E+06, 9.394565E+06, 9.621507E+06, 9.852731E+06,
    1.008828E+07, 1.032821E+07, 1.057259E+07, 1.082148E+07, 1.107494E+07, 1.133303E+07,
    1.159581E+07, 1.186333E+07, 1.213567E+07, 1.241289E+07, 1.269505E+07, 1.298221E+07,
    1.327444E+07, 1.357180E+07, 1.387436E+07, 1.418219E+07, 1.449535E+07, 1.481390E+07,
    1.513792E+07, 1.546748E+07, 1.580263E+07, 1.614346E+07, 1.649004E+07, 1.684241E+07,
    1.720068E+07, 1.756490E+07,
])

#  ---------------------- M = 37, I = 2 ---------------------------
M = 37
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    5.739990E+00, 4.708308E+02, 1.326733E+03, 2.434981E+03, 3.747861E+03, 5.238419E+03,
    6.890124E+03, 8.694231E+03, 1.064761E+04, 1.275163E+04, 1.501028E+04, 1.743002E+04,
    2.001840E+04, 2.278379E+04, 2.573568E+04, 2.888372E+04, 3.223779E+04, 3.580868E+04,
    3.960721E+04, 4.364360E+04, 4.792918E+04, 5.247539E+04, 5.729323E+04, 6.239415E+04,
    6.778948E+04, 7.349165E+04, 7.951198E+04, 8.586227E+04, 9.255496E+04, 9.960247E+04,
    1.070173E+05, 1.148120E+05, 1.229998E+05, 1.315924E+05, 1.406050E+05, 1.500502E+05,
    1.599417E+05, 1.702941E+05, 1.811203E+05, 1.924363E+05, 2.042563E+05, 2.165956E+05,
    2.294686E+05, 2.428909E+05, 2.568798E+05, 2.714501E+05, 2.866181E+05, 3.024019E+05,
    3.188166E+05, 3.358826E+05, 3.536137E+05, 3.720290E+05, 3.911480E+05, 4.109904E+05,
    4.315709E+05, 4.529114E+05, 4.750329E+05, 4.979522E+05, 5.216906E+05, 5.462676E+05,
    5.717072E+05, 5.980275E+05, 6.252486E+05, 6.533981E+05, 6.824923E+05, 7.125570E+05,
    7.436139E+05, 7.756873E+05, 8.087991E+05, 8.429747E+05, 8.782365E+05, 9.146103E+05,
    9.521193E+05, 9.907930E+05, 1.030652E+06, 1.071728E+06, 1.114040E+06, 1.157621E+06,
    1.202496E+06, 1.248693E+06, 1.296235E+06, 1.345157E+06, 1.395485E+06, 1.447246E+06,
    1.500473E+06, 1.555196E+06, 1.611440E+06, 1.669235E+06, 1.728620E+06, 1.789621E+06,
    1.852269E+06, 1.916593E+06, 1.982635E+06, 2.050416E+06, 2.119981E+06, 2.191356E+06,
    2.264576E+06, 2.339672E+06, 2.416687E+06, 2.495652E+06, 2.576605E+06, 2.659573E+06,
    2.744601E+06, 2.831725E+06, 2.920982E+06, 3.012404E+06, 3.106038E+06, 3.201913E+06,
    3.300075E+06, 3.400561E+06, 3.503412E+06, 3.608665E+06, 3.716362E+06, 3.826546E+06,
    3.939250E+06, 4.054530E+06, 4.172418E+06, 4.292959E+06, 4.416198E+06, 4.542175E+06,
    4.670937E+06, 4.802531E+06, 4.936999E+06, 5.074379E+06, 5.214727E+06, 5.358088E+06,
    5.504508E+06, 5.654031E+06, 5.806702E+06, 5.962583E+06, 6.121705E+06, 6.284127E+06,
    6.449902E+06, 6.619068E+06, 6.791685E+06, 6.967806E+06, 7.147471E+06, 7.330735E+06,
    7.517662E+06, 7.708297E+06, 7.902682E+06, 8.100891E+06, 8.302962E+06, 8.508957E+06,
    8.718940E+06, 8.932943E+06, 9.151047E+06, 9.373291E+06, 9.599747E+06, 9.830455E+06,
    1.006549E+07, 1.030491E+07, 1.054875E+07, 1.079710E+07, 1.105000E+07, 1.130753E+07,
    1.156973E+07, 1.183667E+07, 1.210843E+07, 1.238504E+07, 1.266658E+07, 1.295312E+07,
    1.324472E+07, 1.354143E+07, 1.384333E+07, 1.415049E+07, 1.446297E+07, 1.478083E+07,
    1.510415E+07, 1.543299E+07, 1.576743E+07, 1.610752E+07, 1.645334E+07, 1.680496E+07,
    1.716245E+07, 1.752589E+07,
])

#  ---------------------- M = 38, I = 1 ---------------------------
M = 38
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.964040E+00, 1.867813E+02, 5.232315E+02, 9.584283E+02, 1.473741E+03, 2.058415E+03,
    2.705341E+03, 3.409831E+03, 4.169420E+03, 4.983909E+03, 5.855633E+03, 6.789060E+03,
    7.791043E+03, 8.870171E+03, 1.003696E+04, 1.130344E+04, 1.268335E+04, 1.419187E+04,
    1.584565E+04, 1.766292E+04, 1.966394E+04, 2.187063E+04, 2.430711E+04, 2.699984E+04,
    2.997740E+04, 3.327167E+04, 3.691750E+04, 4.095290E+04, 4.541954E+04, 5.036362E+04,
    5.583538E+04, 6.189039E+04, 6.858858E+04, 7.599728E+04, 8.418919E+04, 9.324438E+04,
    1.032502E+05, 1.143029E+05, 1.265073E+05, 1.399784E+05, 1.548424E+05, 1.712363E+05,
    1.893101E+05, 2.092280E+05, 2.311692E+05, 2.553286E+05, 2.819194E+05, 3.111737E+05,
    3.433445E+05, 3.787068E+05, 4.175601E+05, 4.602299E+05, 5.070709E+05, 5.584681E+05,
    6.148394E+05, 6.766383E+05, 7.443566E+05, 8.185285E+05, 8.997333E+05, 9.885977E+05,
    1.085799E+06, 1.192073E+06, 1.308215E+06, 1.435085E+06, 1.573609E+06, 1.724794E+06,
    1.889722E+06, 2.069561E+06, 2.265577E+06, 2.479128E+06, 2.711680E+06, 2.964814E+06,
    3.240234E+06, 3.539768E+06, 3.865391E+06, 4.219220E+06, 4.603536E+06, 5.020787E+06,
    5.473605E+06, 5.964816E+06, 6.497450E+06, 7.074761E+06, 7.700237E+06, 8.377621E+06,
    9.110918E+06, 9.904421E+06, 1.076273E+07, 1.169076E+07, 1.269377E+07, 1.377741E+07,
    1.494767E+07, 1.621101E+07, 1.757429E+07, 1.904485E+07, 2.063052E+07, 2.233968E+07,
    2.418123E+07, 2.616470E+07, 2.830021E+07, 3.059859E+07, 3.307135E+07, 3.573073E+07,
    3.858981E+07, 4.166247E+07, 4.496350E+07, 4.850860E+07, 5.231451E+07, 5.639898E+07,
    6.078088E+07, 6.548028E+07, 7.051845E+07, 7.591801E+07, 8.170292E+07, 8.789865E+07,
    9.453217E+07, 1.016321E+08, 1.092288E+08, 1.173543E+08, 1.260428E+08, 1.353302E+08,
    1.452548E+08, 1.558569E+08, 1.671794E+08, 1.792673E+08, 1.921686E+08, 2.059338E+08,
    2.206164E+08, 2.362728E+08, 2.529626E+08, 2.707490E+08, 2.896984E+08, 3.098811E+08,
    3.313711E+08, 3.542468E+08, 3.785905E+08, 4.044891E+08, 4.320345E+08, 4.613233E+08,
    4.924573E+08, 5.255437E+08, 5.606957E+08, 5.980320E+08, 6.376782E+08, 6.797660E+08,
    7.244340E+08, 7.718283E+08, 8.221023E+08, 8.754174E+08, 9.319435E+08, 9.918586E+08,
    1.055351E+09, 1.122616E+09, 1.193862E+09, 1.269305E+09, 1.349174E+09, 1.433708E+09,
    1.523159E+09, 1.617789E+09, 1.717876E+09, 1.823709E+09, 1.935593E+09, 2.053846E+09,
    2.178804E+09, 2.310816E+09, 2.450248E+09, 2.597487E+09, 2.752934E+09, 2.917011E+09,
    3.090161E+09, 3.272845E+09, 3.465547E+09, 3.668772E+09, 3.883052E+09, 4.108941E+09,
    4.347015E+09, 4.597885E+09,
])

#  ---------------------- M = 38, I = 2 ---------------------------
M = 38
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.204615E+01, 7.642634E+02, 2.141386E+03, 3.922752E+03, 6.032083E+03, 8.425355E+03,
    1.107346E+04, 1.395721E+04, 1.706650E+04, 2.040053E+04, 2.396885E+04, 2.778974E+04,
    3.189127E+04, 3.630859E+04, 4.108475E+04, 4.626896E+04, 5.191754E+04, 5.809252E+04,
    6.486214E+04, 7.230097E+04, 8.049201E+04, 8.952494E+04, 9.949845E+04, 1.105209E+05,
    1.227093E+05, 1.361942E+05, 1.511181E+05, 1.676367E+05, 1.859206E+05, 2.061588E+05,
    2.285571E+05, 2.533428E+05, 2.807614E+05, 3.110884E+05, 3.446214E+05, 3.816882E+05,
    4.226462E+05, 4.678897E+05, 5.178476E+05, 5.729908E+05, 6.338355E+05, 7.009427E+05,
    7.749268E+05, 8.564597E+05, 9.462743E+05, 1.045169E+06, 1.154017E+06, 1.273768E+06,
    1.405457E+06, 1.550210E+06, 1.709254E+06, 1.883920E+06, 2.075661E+06, 2.286052E+06,
    2.516805E+06, 2.769775E+06, 3.046976E+06, 3.350594E+06, 3.683001E+06, 4.046763E+06,
    4.444651E+06, 4.879678E+06, 5.355099E+06, 5.874432E+06, 6.441474E+06, 7.060342E+06,
    7.735464E+06, 8.471627E+06, 9.274007E+06, 1.014816E+07, 1.110010E+07, 1.213630E+07,
    1.326371E+07, 1.448984E+07, 1.582276E+07, 1.727114E+07, 1.884431E+07, 2.055231E+07,
    2.240589E+07, 2.441664E+07, 2.659695E+07, 2.896014E+07, 3.152050E+07, 3.429333E+07,
    3.729505E+07, 4.054321E+07, 4.405664E+07, 4.785548E+07, 5.196127E+07, 5.639707E+07,
    6.118750E+07, 6.635891E+07, 7.193943E+07, 7.795909E+07, 8.444997E+07, 9.144632E+07,
    9.898462E+07, 1.071038E+08, 1.158455E+08, 1.252538E+08, 1.353759E+08, 1.462619E+08,
    1.579654E+08, 1.705432E+08, 1.840558E+08, 1.985675E+08, 2.141468E+08, 2.308664E+08,
    2.488035E+08, 2.680402E+08, 2.886638E+08, 3.107666E+08, 3.344469E+08, 3.598088E+08,
    3.869628E+08, 4.160260E+08, 4.471227E+08, 4.803842E+08, 5.159500E+08, 5.539677E+08,
    5.945935E+08, 6.379928E+08, 6.843406E+08, 7.338221E+08, 7.866329E+08, 8.429801E+08,
    9.030826E+08, 9.671712E+08, 1.035490E+09, 1.108298E+09, 1.185867E+09, 1.268484E+09,
    1.356452E+09, 1.450092E+09, 1.549742E+09, 1.655757E+09, 1.768513E+09, 1.888406E+09,
    2.015851E+09, 2.151289E+09, 2.295182E+09, 2.448017E+09, 2.610306E+09, 2.782591E+09,
    2.965437E+09, 3.159443E+09, 3.365238E+09, 3.583481E+09, 3.814868E+09, 4.060128E+09,
    4.320030E+09, 4.595378E+09, 4.887019E+09, 5.195844E+09, 5.522783E+09, 5.868820E+09,
    6.234981E+09, 6.622346E+09, 7.032047E+09, 7.465271E+09, 7.923262E+09, 8.407328E+09,
    8.918836E+09, 9.459220E+09, 1.002998E+10, 1.063270E+10, 1.126901E+10, 1.194066E+10,
    1.264944E+10, 1.339724E+10, 1.418606E+10, 1.501796E+10, 1.589510E+10, 1.681977E+10,
    1.779432E+10, 1.882124E+10,
])

#  ---------------------- M = 39, I = 1 ---------------------------
M = 39
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.441130E+01, 8.984785E+02, 2.519098E+03, 4.622384E+03, 7.153276E+03, 1.012212E+04,
    1.356235E+04, 1.751151E+04, 2.200662E+04, 2.708504E+04, 3.278708E+04, 3.915884E+04,
    4.625465E+04, 5.413906E+04, 6.288836E+04, 7.259164E+04, 8.335181E+04, 9.528638E+04,
    1.085284E+05, 1.232272E+05, 1.395500E+05, 1.576826E+05, 1.778316E+05, 2.002254E+05,
    2.251168E+05, 2.527848E+05, 2.835372E+05, 3.177134E+05, 3.556871E+05, 3.978696E+05,
    4.447142E+05, 4.967184E+05, 5.544300E+05, 6.184503E+05, 6.894398E+05, 7.681233E+05,
    8.552960E+05, 9.518298E+05, 1.058680E+06, 1.176894E+06, 1.307616E+06, 1.452099E+06,
    1.611715E+06, 1.787960E+06, 1.982471E+06, 2.197034E+06, 2.433598E+06, 2.694287E+06,
    2.981419E+06, 3.297516E+06, 3.645326E+06, 4.027840E+06, 4.448308E+06, 4.910266E+06,
    5.417556E+06, 5.974349E+06, 6.585171E+06, 7.254935E+06, 7.988968E+06, 8.793039E+06,
    9.673404E+06, 1.063683E+07, 1.169064E+07, 1.284277E+07, 1.410177E+07, 1.547691E+07,
    1.697818E+07, 1.861638E+07, 2.040317E+07, 2.235109E+07, 2.447370E+07, 2.678560E+07,
    2.930251E+07, 3.204136E+07, 3.502035E+07, 3.825912E+07, 4.177870E+07, 4.560178E+07,
    4.975264E+07, 5.425747E+07, 5.914429E+07, 6.444319E+07, 7.018647E+07, 7.640872E+07,
    8.314707E+07, 9.044122E+07, 9.833376E+07, 1.068702E+08, 1.160993E+08, 1.260732E+08,
    1.368476E+08, 1.484821E+08, 1.610404E+08, 1.745904E+08, 1.892048E+08, 2.049609E+08,
    2.219414E+08, 2.402345E+08, 2.599341E+08, 2.811403E+08, 3.039598E+08, 3.285064E+08,
    3.549009E+08, 3.832720E+08, 4.137567E+08, 4.465008E+08, 4.816591E+08, 5.193963E+08,
    5.598872E+08, 6.033178E+08, 6.498853E+08, 6.997992E+08, 7.532821E+08, 8.105694E+08,
    8.719115E+08, 9.375733E+08, 1.007836E+09, 1.082998E+09, 1.163375E+09, 1.249300E+09,
    1.341128E+09, 1.439233E+09, 1.544011E+09, 1.655881E+09, 1.775287E+09, 1.902698E+09,
    2.038608E+09, 2.183542E+09, 2.338052E+09, 2.502722E+09, 2.678169E+09, 2.865044E+09,
    3.064034E+09, 3.275863E+09, 3.501296E+09, 3.741140E+09, 3.996242E+09, 4.267501E+09,
    4.555859E+09, 4.862310E+09, 5.187903E+09, 5.533738E+09, 5.900977E+09, 6.290843E+09,
    6.704620E+09, 7.143662E+09, 7.609388E+09, 8.103298E+09, 8.626961E+09, 9.182033E+09,
    9.770245E+09, 1.039343E+10, 1.105349E+10, 1.175245E+10, 1.249242E+10, 1.327562E+10,
    1.410437E+10, 1.498112E+10, 1.590843E+10, 1.688899E+10, 1.792560E+10, 1.902123E+10,
    2.017898E+10, 2.140208E+10, 2.269394E+10, 2.405813E+10, 2.549835E+10, 2.701854E+10,
    2.862278E+10, 3.031533E+10, 3.210070E+10, 3.398356E+10, 3.596882E+10, 3.806161E+10,
    4.026729E+10, 4.259149E+10,
])

#  ---------------------- M = 40, I = 1 ---------------------------
M = 40
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    3.004878E+01, 1.395096E+03, 3.835885E+03, 7.029882E+03, 1.081742E+04, 1.511632E+04,
    1.987927E+04, 2.508270E+04, 3.072275E+04, 3.681224E+04, 4.337811E+04, 5.045934E+04,
    5.810536E+04, 6.637497E+04, 7.533555E+04, 8.506260E+04, 9.563945E+04, 1.071573E+05,
    1.197152E+05, 1.334205E+05, 1.483892E+05, 1.647463E+05, 1.826266E+05, 2.021755E+05,
    2.235493E+05, 2.469166E+05, 2.724590E+05, 3.003719E+05, 3.308659E+05, 3.641678E+05,
    4.005214E+05, 4.401896E+05, 4.834550E+05, 5.306217E+05, 5.820165E+05, 6.379909E+05,
    6.989225E+05, 7.652168E+05, 8.373091E+05, 9.156666E+05, 1.000790E+06, 1.093217E+06,
    1.193523E+06, 1.302323E+06, 1.420279E+06, 1.548095E+06, 1.686527E+06, 1.836381E+06,
    1.998520E+06, 2.173864E+06, 2.363397E+06, 2.568166E+06, 2.789290E+06, 3.027961E+06,
    3.285448E+06, 3.563104E+06, 3.862368E+06, 4.184771E+06, 4.531944E+06, 4.905616E+06,
    5.307629E+06, 5.739936E+06, 6.204613E+06, 6.703863E+06, 7.240020E+06, 7.815564E+06,
    8.433119E+06, 9.095469E+06, 9.805559E+06, 1.056651E+07, 1.138163E+07, 1.225440E+07,
    1.318853E+07, 1.418791E+07, 1.525668E+07, 1.639920E+07, 1.762006E+07, 1.892414E+07,
    2.031655E+07, 2.180272E+07, 2.338834E+07, 2.507943E+07, 2.688234E+07, 2.880373E+07,
    3.085064E+07, 3.303048E+07, 3.535103E+07, 3.782050E+07, 4.044751E+07, 4.324112E+07,
    4.621087E+07, 4.936677E+07, 5.271934E+07, 5.627963E+07, 6.005924E+07, 6.407034E+07,
    6.832572E+07, 7.283878E+07, 7.762356E+07, 8.269480E+07, 8.806795E+07, 9.375919E+07,
    9.978547E+07, 1.061645E+08, 1.129150E+08, 1.200563E+08, 1.276088E+08, 1.355938E+08,
    1.440336E+08, 1.529514E+08, 1.623717E+08, 1.723198E+08, 1.828225E+08, 1.939074E+08,
    2.056036E+08, 2.179413E+08, 2.309523E+08, 2.446695E+08, 2.591273E+08, 2.743617E+08,
    2.904102E+08, 3.073119E+08, 3.251074E+08, 3.438393E+08, 3.635518E+08, 3.842910E+08,
    4.061049E+08, 4.290435E+08, 4.531588E+08, 4.785052E+08, 5.051389E+08, 5.331187E+08,
    5.625056E+08, 5.933632E+08, 6.257574E+08, 6.597570E+08, 6.954334E+08, 7.328607E+08,
    7.721161E+08, 8.132796E+08, 8.564345E+08, 9.016672E+08, 9.490674E+08, 9.987283E+08,
    1.050747E+09, 1.105223E+09, 1.162261E+09, 1.221968E+09, 1.284458E+09, 1.349847E+09,
    1.418253E+09, 1.489804E+09, 1.564628E+09, 1.642860E+09, 1.724638E+09, 1.810107E+09,
    1.899416E+09, 1.992720E+09, 2.090179E+09, 2.191959E+09, 2.298231E+09, 2.409173E+09,
    2.524970E+09, 2.645811E+09, 2.771894E+09, 2.903422E+09, 3.040606E+09, 3.183664E+09,
    3.332821E+09, 3.488310E+09, 3.650373E+09, 3.819257E+09, 3.995222E+09, 4.178531E+09,
    4.369461E+09, 4.568295E+09,
])

#  ---------------------- M = 40, I = 2 ---------------------------
M = 40
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    3.014586E+01, 1.400380E+03, 3.850470E+03, 7.056645E+03, 1.085863E+04, 1.517397E+04,
    1.995524E+04, 2.517891E+04, 3.084126E+04, 3.695530E+04, 4.354822E+04, 5.065922E+04,
    5.833802E+04, 6.664374E+04, 7.564409E+04, 8.541493E+04, 9.604001E+04, 1.076110E+05,
    1.202274E+05, 1.339971E+05, 1.490367E+05, 1.654718E+05, 1.834380E+05, 2.030813E+05,
    2.245589E+05, 2.480404E+05, 2.737081E+05, 3.017585E+05, 3.324034E+05, 3.658707E+05,
    4.024056E+05, 4.422723E+05, 4.857550E+05, 5.331592E+05, 5.848137E+05, 6.410718E+05,
    7.023131E+05, 7.689453E+05, 8.414060E+05, 9.201650E+05, 1.005726E+06, 1.098629E+06,
    1.199452E+06, 1.308815E+06, 1.427382E+06, 1.555862E+06, 1.695014E+06, 1.845650E+06,
    2.008636E+06, 2.184898E+06, 2.375424E+06, 2.581269E+06, 2.803557E+06, 3.043485E+06,
    3.302331E+06, 3.581455E+06, 3.882303E+06, 4.206416E+06, 4.555432E+06, 4.931091E+06,
    5.335245E+06, 5.769857E+06, 6.237015E+06, 6.738933E+06, 7.277960E+06, 7.856588E+06,
    8.477456E+06, 9.143363E+06, 9.857272E+06, 1.062232E+07, 1.144183E+07, 1.231931E+07,
    1.325848E+07, 1.426327E+07, 1.533782E+07, 1.648652E+07, 1.771400E+07, 1.902515E+07,
    2.042513E+07, 2.191937E+07, 2.351362E+07, 2.521392E+07, 2.702665E+07, 2.895852E+07,
    3.101660E+07, 3.320835E+07, 3.554159E+07, 3.802456E+07, 4.066595E+07, 4.347487E+07,
    4.646089E+07, 4.963410E+07, 5.300508E+07, 5.658492E+07, 6.038531E+07, 6.441848E+07,
    6.869728E+07, 7.323518E+07, 7.804633E+07, 8.314553E+07, 8.854833E+07, 9.427099E+07,
    1.003306E+08, 1.067449E+08, 1.135327E+08, 1.207135E+08, 1.283078E+08, 1.363370E+08,
    1.448235E+08, 1.537908E+08, 1.632633E+08, 1.732667E+08, 1.838276E+08, 1.949741E+08,
    2.067353E+08, 2.191416E+08, 2.322250E+08, 2.460185E+08, 2.605568E+08, 2.758761E+08,
    2.920141E+08, 3.090099E+08, 3.269047E+08, 3.457411E+08, 3.655636E+08, 3.864186E+08,
    4.083543E+08, 4.314211E+08, 4.556712E+08, 4.811593E+08, 5.079420E+08, 5.360784E+08,
    5.656298E+08, 5.966602E+08, 6.292359E+08, 6.634261E+08, 6.993024E+08, 7.369397E+08,
    7.764152E+08, 8.178097E+08, 8.612069E+08, 9.066936E+08, 9.543600E+08, 1.004300E+09,
    1.056611E+09, 1.111393E+09, 1.168752E+09, 1.228795E+09, 1.291637E+09, 1.357393E+09,
    1.426185E+09, 1.498139E+09, 1.573384E+09, 1.652057E+09, 1.734296E+09, 1.820247E+09,
    1.910060E+09, 2.003890E+09, 2.101899E+09, 2.204253E+09, 2.311125E+09, 2.422694E+09,
    2.539145E+09, 2.660669E+09, 2.787464E+09, 2.919735E+09, 3.057694E+09, 3.201561E+09,
    3.351562E+09, 3.507931E+09, 3.670911E+09, 3.840752E+09, 4.017712E+09, 4.202060E+09,
    4.394071E+09, 4.594031E+09,
])

#  ---------------------- M = 41, I = 1 ---------------------------
M = 41
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.315131E+01, 1.079811E+03, 2.965691E+03, 5.436069E+03, 8.385490E+03, 1.180644E+04,
    1.574958E+04, 2.029388E+04, 2.553119E+04, 3.156166E+04, 3.849468E+04, 4.645162E+04,
    5.556906E+04, 6.600179E+04, 7.792578E+04, 9.154102E+04, 1.070745E+05, 1.247831E+05,
    1.449571E+05, 1.679240E+05, 1.940520E+05, 2.237552E+05, 2.574980E+05, 2.958013E+05,
    3.392479E+05, 3.884899E+05, 4.442559E+05, 5.073594E+05, 5.787080E+05, 6.593131E+05,
    7.503014E+05, 8.529264E+05, 9.685822E+05, 1.098818E+06, 1.245353E+06, 1.410094E+06,
    1.595158E+06, 1.802885E+06, 2.035868E+06, 2.296976E+06, 2.589376E+06, 2.916568E+06,
    3.282415E+06, 3.691174E+06, 4.147540E+06, 4.656680E+06, 5.224279E+06, 5.856591E+06,
    6.560485E+06, 7.343508E+06, 8.213936E+06, 9.180849E+06, 1.025420E+07, 1.144487E+07,
    1.276479E+07, 1.422701E+07, 1.584577E+07, 1.763664E+07, 1.961662E+07, 2.180426E+07,
    2.421975E+07, 2.688513E+07, 2.982439E+07, 3.306362E+07, 3.663123E+07, 4.055806E+07,
    4.487764E+07, 4.962639E+07, 5.484381E+07, 6.057274E+07, 6.685963E+07, 7.375479E+07,
    8.131271E+07, 8.959236E+07, 9.865750E+07, 1.085771E+08, 1.194257E+08, 1.312836E+08,
    1.442379E+08, 1.583821E+08, 1.738174E+08, 1.906527E+08, 2.090052E+08, 2.290015E+08,
    2.507774E+08, 2.744793E+08, 3.002646E+08, 3.283022E+08, 3.587741E+08, 3.918751E+08,
    4.278148E+08, 4.668179E+08, 5.091253E+08, 5.549954E+08, 6.047049E+08, 6.585505E+08,
    7.168495E+08, 7.799418E+08, 8.481907E+08, 9.219851E+08, 1.001740E+09, 1.087900E+09,
    1.180939E+09, 1.281364E+09, 1.389714E+09, 1.506566E+09, 1.632536E+09, 1.768278E+09,
    1.914492E+09, 2.071922E+09, 2.241360E+09, 2.423650E+09, 2.619690E+09, 2.830436E+09,
    3.056904E+09, 3.300174E+09, 3.561394E+09, 3.841784E+09, 4.142639E+09, 4.465335E+09,
    4.811331E+09, 5.182177E+09, 5.579515E+09, 6.005089E+09, 6.460744E+09, 6.948439E+09,
    7.470246E+09, 8.028363E+09, 8.625114E+09, 9.262960E+09, 9.944506E+09, 1.067251E+10,
    1.144988E+10, 1.227970E+10, 1.316522E+10, 1.410990E+10, 1.511735E+10, 1.619142E+10,
    1.733617E+10, 1.855587E+10, 1.985503E+10, 2.123842E+10, 2.271106E+10, 2.427825E+10,
    2.594556E+10, 2.771889E+10, 2.960444E+10, 3.160873E+10, 3.373864E+10, 3.600142E+10,
    3.840468E+10, 4.095644E+10, 4.366516E+10, 4.653969E+10, 4.958938E+10, 5.282404E+10,
    5.625398E+10, 5.989006E+10, 6.374364E+10, 6.782671E+10, 7.215183E+10, 7.673218E+10,
    8.158161E+10, 8.671466E+10, 9.214658E+10, 9.789336E+10, 1.039718E+11, 1.103994E+11,
    1.171948E+11, 1.243772E+11, 1.319668E+11, 1.399850E+11, 1.484539E+11, 1.573968E+11,
    1.668382E+11, 1.768036E+11,
])

#  ---------------------- M = 41, I = 2 ---------------------------
M = 41
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    4.756468E+01, 2.228120E+03, 6.122297E+03, 1.122300E+04, 1.731317E+04, 2.437845E+04,
    3.252465E+04, 4.191614E+04, 5.274423E+04, 6.521817E+04, 7.956715E+04, 9.604612E+04,
    1.149423E+05, 1.365814E+05, 1.613342E+05, 1.896217E+05, 2.219220E+05, 2.587765E+05,
    3.007971E+05, 3.486737E+05, 4.031834E+05, 4.651993E+05, 5.357021E+05, 6.157912E+05,
    7.066979E+05, 8.098005E+05, 9.266395E+05, 1.058936E+06, 1.208610E+06, 1.377803E+06,
    1.568902E+06, 1.784563E+06, 2.027740E+06, 2.301720E+06, 2.610149E+06, 2.957077E+06,
    3.346993E+06, 3.784872E+06, 4.276222E+06, 4.827137E+06, 5.444356E+06, 6.135322E+06,
    6.908250E+06, 7.772205E+06, 8.737178E+06, 9.814173E+06, 1.101530E+07, 1.235389E+07,
    1.384459E+07, 1.550347E+07, 1.734820E+07, 1.939814E+07, 2.167453E+07, 2.420060E+07,
    2.700184E+07, 3.010607E+07, 3.354375E+07, 3.734815E+07, 4.155558E+07, 4.620567E+07,
    5.134166E+07, 5.701064E+07, 6.326393E+07, 7.015737E+07, 7.775173E+07, 8.611306E+07,
    9.531315E+07, 1.054300E+08, 1.165481E+08, 1.287594E+08, 1.421633E+08, 1.568678E+08,
    1.729895E+08, 1.906549E+08, 2.100008E+08, 2.311751E+08, 2.543376E+08, 2.796608E+08,
    3.073314E+08, 3.375503E+08, 3.705345E+08, 4.065179E+08, 4.457524E+08, 4.885096E+08,
    5.350814E+08, 5.857824E+08, 6.409505E+08, 7.009493E+08, 7.661692E+08, 8.370298E+08,
    9.139812E+08, 9.975068E+08, 1.088125E+09, 1.186391E+09, 1.292901E+09, 1.408292E+09,
    1.533248E+09, 1.668500E+09, 1.814831E+09, 1.973077E+09, 2.144133E+09, 2.328955E+09,
    2.528564E+09, 2.744052E+09, 2.976580E+09, 3.227394E+09, 3.497816E+09, 3.789260E+09,
    4.103232E+09, 4.441336E+09, 4.805281E+09, 5.196886E+09, 5.618088E+09, 6.070948E+09,
    6.557656E+09, 7.080543E+09, 7.642086E+09, 8.244916E+09, 8.891828E+09, 9.585791E+09,
    1.032996E+10, 1.112766E+10, 1.198246E+10, 1.289812E+10, 1.387861E+10, 1.492818E+10,
    1.605128E+10, 1.725268E+10, 1.853739E+10, 1.991072E+10, 2.137831E+10, 2.294611E+10,
    2.462041E+10, 2.640788E+10, 2.831554E+10, 3.035083E+10, 3.252162E+10, 3.483620E+10,
    3.730333E+10, 3.993226E+10, 4.273276E+10, 4.571512E+10, 4.889021E+10, 5.226948E+10,
    5.586501E+10, 5.968954E+10, 6.375646E+10, 6.807992E+10, 7.267480E+10, 7.755676E+10,
    8.274231E+10, 8.824879E+10, 9.409449E+10, 1.002986E+11, 1.068814E+11, 1.138640E+11,
    1.212689E+11, 1.291194E+11, 1.374404E+11, 1.462576E+11, 1.555982E+11, 1.654910E+11,
    1.759658E+11, 1.870541E+11, 1.987891E+11, 2.112053E+11, 2.243391E+11, 2.382286E+11,
    2.529139E+11, 2.684368E+11, 2.848412E+11, 3.021733E+11, 3.204811E+11, 3.398152E+11,
    3.602284E+11, 3.817762E+11,
])

#  ---------------------- M = 41, I = 3 ---------------------------
M = 41
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    4.640829E+01, 2.165303E+03, 5.949088E+03, 1.090578E+04, 1.682930E+04, 2.371551E+04,
    3.167737E+04, 4.088260E+04, 5.152311E+04, 6.380713E+04, 7.796181E+04, 9.423931E+04,
    1.129235E+05, 1.343366E+05, 1.588447E+05, 1.868645E+05, 2.188692E+05, 2.553945E+05,
    2.970463E+05, 3.445084E+05, 3.985504E+05, 4.600380E+05, 5.299431E+05, 6.093559E+05,
    6.994976E+05, 8.017346E+05, 9.175952E+05, 1.048786E+06, 1.197212E+06, 1.364999E+06,
    1.554512E+06, 1.768388E+06, 2.009559E+06, 2.281283E+06, 2.587180E+06, 2.931268E+06,
    3.318001E+06, 3.752316E+06, 4.239679E+06, 4.786137E+06, 5.398376E+06, 6.083784E+06,
    6.850514E+06, 7.707561E+06, 8.664841E+06, 9.733275E+06, 1.092489E+07, 1.225290E+07,
    1.373184E+07, 1.537769E+07, 1.720796E+07, 1.924188E+07, 2.150051E+07, 2.400695E+07,
    2.678645E+07, 2.986667E+07, 3.327782E+07, 3.705292E+07, 4.122803E+07, 4.584250E+07,
    5.093921E+07, 5.656495E+07, 6.277064E+07, 6.961172E+07, 7.714850E+07, 8.544659E+07,
    9.457722E+07, 1.046178E+08, 1.156524E+08, 1.277720E+08, 1.410755E+08, 1.556700E+08,
    1.716714E+08, 1.892051E+08, 2.084071E+08, 2.294241E+08, 2.524148E+08, 2.775507E+08,
    3.050167E+08, 3.350126E+08, 3.677537E+08, 4.034723E+08, 4.424186E+08, 4.848620E+08,
    5.310927E+08, 5.814227E+08, 6.361877E+08, 6.957486E+08, 7.604932E+08, 8.308379E+08,
    9.072299E+08, 9.901490E+08, 1.080110E+09, 1.177664E+09, 1.283403E+09, 1.397960E+09,
    1.522014E+09, 1.656291E+09, 1.801568E+09, 1.958675E+09, 2.128501E+09, 2.311996E+09,
    2.510172E+09, 2.724114E+09, 2.954978E+09, 3.203997E+09, 3.472486E+09, 3.761848E+09,
    4.073580E+09, 4.409273E+09, 4.770625E+09, 5.159443E+09, 5.577650E+09, 6.027291E+09,
    6.510543E+09, 7.029720E+09, 7.587282E+09, 8.185841E+09, 8.828174E+09, 9.517228E+09,
    1.025613E+10, 1.104820E+10, 1.189697E+10, 1.280617E+10, 1.377975E+10, 1.482192E+10,
    1.593712E+10, 1.713006E+10, 1.840573E+10, 1.976942E+10, 2.122670E+10, 2.278350E+10,
    2.444605E+10, 2.622098E+10, 2.811528E+10, 3.013632E+10, 3.229192E+10, 3.459030E+10,
    3.704019E+10, 3.965075E+10, 4.243169E+10, 4.539323E+10, 4.854616E+10, 5.190187E+10,
    5.547234E+10, 5.927022E+10, 6.330882E+10, 6.760219E+10, 7.216510E+10, 7.701311E+10,
    8.216261E+10, 8.763084E+10, 9.343594E+10, 9.959700E+10, 1.061341E+11, 1.130683E+11,
    1.204218E+11, 1.282180E+11, 1.364812E+11, 1.452374E+11, 1.545134E+11, 1.643377E+11,
    1.747400E+11, 1.857517E+11, 1.974056E+11, 2.097360E+11, 2.227790E+11, 2.365727E+11,
    2.511566E+11, 2.665724E+11, 2.828637E+11, 3.000762E+11, 3.182578E+11, 3.374587E+11,
    3.577313E+11, 3.791307E+11,
])

#  ---------------------- M = 41, I = 4 ---------------------------
M = 41
I = 4
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    9.539916E+01, 4.470892E+03, 1.228501E+04, 2.252154E+04, 3.475468E+04, 4.897554E+04,
    6.541725E+04, 8.442665E+04, 1.064017E+05, 1.317753E+05, 1.610212E+05, 1.946663E+05,
    2.333045E+05, 2.776101E+05, 3.283502E+05, 3.863976E+05, 4.527432E+05, 5.285101E+05,
    6.149681E+05, 7.135503E+05, 8.258710E+05, 9.537460E+05, 1.099215E+06, 1.264566E+06,
    1.452364E+06, 1.665478E+06, 1.907122E+06, 2.180882E+06, 2.490766E+06, 2.841242E+06,
    3.237293E+06, 3.684466E+06, 4.188938E+06, 4.757571E+06, 5.397996E+06, 6.118680E+06,
    6.929018E+06, 7.839421E+06, 8.861424E+06, 1.000779E+07, 1.129264E+07, 1.273156E+07,
    1.434179E+07, 1.614232E+07, 1.815412E+07, 2.040028E+07, 2.290621E+07, 2.569988E+07,
    2.881206E+07, 3.227651E+07, 3.613034E+07, 4.041424E+07, 4.517284E+07, 5.045502E+07,
    5.631431E+07, 6.280930E+07, 7.000402E+07, 7.796850E+07, 8.677916E+07, 9.651945E+07,
    1.072804E+08, 1.191612E+08, 1.322698E+08, 1.467241E+08, 1.626520E+08, 1.801928E+08,
    1.994978E+08, 2.207313E+08, 2.440718E+08, 2.697130E+08, 2.978648E+08, 3.287546E+08,
    3.626291E+08, 3.997549E+08, 4.404208E+08, 4.849391E+08, 5.336473E+08, 5.869099E+08,
    6.451208E+08, 7.087047E+08, 7.781201E+08, 8.538612E+08, 9.364606E+08, 1.026492E+09,
    1.124573E+09, 1.231368E+09, 1.347593E+09, 1.474015E+09, 1.611462E+09, 1.760820E+09,
    1.923041E+09, 2.099150E+09, 2.290241E+09, 2.497492E+09, 2.722164E+09, 2.965606E+09,
    3.229265E+09, 3.514690E+09, 3.823537E+09, 4.157580E+09, 4.518712E+09, 4.908959E+09,
    5.330485E+09, 5.785601E+09, 6.276774E+09, 6.806635E+09, 7.377994E+09, 7.993846E+09,
    8.657381E+09, 9.372002E+09, 1.014133E+10, 1.096923E+10, 1.185980E+10, 1.281742E+10,
    1.384672E+10, 1.495267E+10, 1.614050E+10, 1.741580E+10, 1.878451E+10, 2.025292E+10,
    2.182772E+10, 2.351600E+10, 2.532529E+10, 2.726358E+10, 2.933934E+10, 3.156152E+10,
    3.393966E+10, 3.648380E+10, 3.920463E+10, 4.211343E+10, 4.522214E+10, 4.854343E+10,
    5.209065E+10, 5.587795E+10, 5.992028E+10, 6.423345E+10, 6.883414E+10, 7.373998E+10,
    7.896960E+10, 8.454265E+10, 9.047989E+10, 9.680319E+10, 1.035357E+11, 1.107016E+11,
    1.183268E+11, 1.264382E+11, 1.350644E+11, 1.442354E+11, 1.539828E+11, 1.643401E+11,
    1.753422E+11, 1.870260E+11, 1.994305E+11, 2.125965E+11, 2.265670E+11, 2.413872E+11,
    2.571045E+11, 2.737690E+11, 2.914332E+11, 3.101521E+11, 3.299836E+11, 3.509886E+11,
    3.732309E+11, 3.967775E+11, 4.216988E+11, 4.480684E+11, 4.759638E+11, 5.054661E+11,
    5.366605E+11, 5.696362E+11, 6.044865E+11, 6.413096E+11, 6.802081E+11, 7.212894E+11,
    7.646663E+11, 8.104566E+11,
])

#  ---------------------- M = 42, I = 1 ---------------------------
M = 42
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00, 0.000000E+00,
    0.000000E+00, 0.000000E+00,
])

#  ---------------------- M = 43, I = 1 ---------------------------
M = 43
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.019112E+01, 1.905753E+02, 3.807665E+02, 5.762817E+02, 7.905060E+02, 1.038159E+03,
    1.333345E+03, 1.691617E+03, 2.131702E+03, 2.676627E+03, 3.354709E+03, 4.200671E+03,
    5.256974E+03, 6.575423E+03, 8.219051E+03, 1.026434E+04, 1.280384E+04, 1.594915E+04,
    1.983447E+04, 2.462068E+04, 3.050003E+04, 3.770160E+04, 4.649756E+04, 5.721039E+04,
    7.022113E+04, 8.597887E+04, 1.050115E+05, 1.279381E+05, 1.554830E+05, 1.884915E+05,
    2.279483E+05, 2.749977E+05, 3.309668E+05, 3.973918E+05, 4.760472E+05, 5.689788E+05,
    6.785415E+05, 8.074402E+05, 9.587774E+05, 1.136105E+06, 1.343483E+06, 1.585544E+06,
    1.867567E+06, 2.195558E+06, 2.576338E+06, 3.017647E+06, 3.528251E+06, 4.118066E+06,
    4.798292E+06, 5.581567E+06, 6.482126E+06, 7.515994E+06, 8.701176E+06, 1.005789E+07,
    1.160879E+07, 1.337926E+07, 1.539769E+07, 1.769582E+07, 2.030904E+07, 2.327685E+07,
    2.664322E+07, 3.045709E+07, 3.477285E+07, 3.965090E+07, 4.515827E+07, 5.136922E+07,
    5.836599E+07, 6.623958E+07, 7.509054E+07, 8.502994E+07, 9.618032E+07, 1.086768E+08,
    1.226681E+08, 1.383180E+08, 1.558067E+08, 1.753321E+08, 1.971114E+08, 2.213829E+08,
    2.484083E+08, 2.784735E+08, 3.118921E+08, 3.490069E+08, 3.901925E+08, 4.358585E+08,
    4.864516E+08, 5.424591E+08, 6.044125E+08, 6.728905E+08, 7.485237E+08, 8.319970E+08,
    9.240564E+08, 1.025512E+09, 1.137243E+09, 1.260205E+09, 1.395433E+09, 1.544051E+09,
    1.707273E+09, 1.886418E+09, 2.082910E+09, 2.298291E+09, 2.534229E+09, 2.792522E+09,
    3.075116E+09, 3.384109E+09, 3.721764E+09, 4.090522E+09, 4.493014E+09, 4.932072E+09,
    5.410747E+09, 5.932319E+09, 6.500319E+09, 7.118541E+09, 7.791064E+09, 8.522268E+09,
    9.316850E+09, 1.017986E+10, 1.111670E+10, 1.213319E+10, 1.323554E+10, 1.443039E+10,
    1.572490E+10, 1.712669E+10, 1.864393E+10, 2.028536E+10, 2.206030E+10, 2.397874E+10,
    2.605133E+10, 2.828946E+10, 3.070527E+10, 3.331171E+10, 3.612261E+10, 3.915271E+10,
    4.241771E+10, 4.593434E+10, 4.972043E+10, 5.379494E+10, 5.817805E+10, 6.289125E+10,
    6.795741E+10, 7.340077E+10, 7.924717E+10, 8.552402E+10, 9.226047E+10, 9.948742E+10,
    1.072377E+11, 1.155462E+11, 1.244497E+11, 1.339876E+11, 1.442013E+11, 1.551348E+11,
    1.668348E+11, 1.793507E+11, 1.927348E+11, 2.070423E+11, 2.223319E+11, 2.386657E+11,
    2.561090E+11, 2.747312E+11, 2.946056E+11, 3.158095E+11, 3.384248E+11, 3.625376E+11,
    3.882395E+11, 4.156265E+11, 4.448002E+11, 4.758678E+11, 5.089422E+11, 5.441429E+11,
    5.815953E+11, 6.214321E+11, 6.637927E+11, 7.088241E+11, 7.566812E+11, 8.075271E+11,
    8.615336E+11, 9.188811E+11,
])

#  ---------------------- M = 44, I = 1 ---------------------------
M = 44
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.957662E+01, 5.518291E+02, 1.102633E+03, 1.668279E+03, 2.286281E+03, 2.997450E+03,
    3.838554E+03, 4.846743E+03, 6.063697E+03, 7.537981E+03, 9.326765E+03, 1.149723E+04,
    1.412808E+04, 1.731129E+04, 2.115428E+04, 2.578174E+04, 3.133815E+04, 3.799064E+04,
    4.593183E+04, 5.538330E+04, 6.659913E+04, 7.986998E+04, 9.552786E+04, 1.139507E+05,
    1.355685E+05, 1.608685E+05, 1.904027E+05, 2.247946E+05, 2.647478E+05, 3.110535E+05,
    3.646019E+05, 4.263913E+05, 4.975399E+05, 5.792987E+05, 6.730643E+05, 7.803938E+05,
    9.030217E+05, 1.042876E+06, 1.202097E+06, 1.383060E+06, 1.588392E+06, 1.821003E+06,
    2.084105E+06, 2.381241E+06, 2.716320E+06, 3.093641E+06, 3.517933E+06, 3.994391E+06,
    4.528714E+06, 5.127151E+06, 5.796541E+06, 6.544370E+06, 7.378816E+06, 8.308808E+06,
    9.344088E+06, 1.049527E+07, 1.177392E+07, 1.319260E+07, 1.476499E+07, 1.650593E+07,
    1.843154E+07, 2.055927E+07, 2.290806E+07, 2.549840E+07, 2.835243E+07, 3.149414E+07,
    3.494939E+07, 3.874612E+07, 4.291446E+07, 4.748690E+07, 5.249840E+07, 5.798665E+07,
    6.399214E+07, 7.055842E+07, 7.773230E+07, 8.556400E+07, 9.410744E+07, 1.034204E+08,
    1.135649E+08, 1.246073E+08, 1.366186E+08, 1.496748E+08, 1.638572E+08, 1.792528E+08,
    1.959542E+08, 2.140605E+08, 2.336775E+08, 2.549180E+08, 2.779020E+08, 3.027576E+08,
    3.296212E+08, 3.586378E+08, 3.899621E+08, 4.237580E+08, 4.602001E+08, 4.994740E+08,
    5.417764E+08, 5.873165E+08, 6.363159E+08, 6.890100E+08, 7.456480E+08, 8.064942E+08,
    8.718285E+08, 9.419473E+08, 1.017164E+09, 1.097811E+09, 1.184239E+09, 1.276819E+09,
    1.375943E+09, 1.482026E+09, 1.595504E+09, 1.716839E+09, 1.846519E+09, 1.985058E+09,
    2.132997E+09, 2.290908E+09, 2.459392E+09, 2.639085E+09, 2.830653E+09, 3.034800E+09,
    3.252265E+09, 3.483827E+09, 3.730304E+09, 3.992557E+09, 4.271490E+09, 4.568055E+09,
    4.883248E+09, 5.218118E+09, 5.573767E+09, 5.951348E+09, 6.352075E+09, 6.777217E+09,
    7.228108E+09, 7.706145E+09, 8.212792E+09, 8.749583E+09, 9.318127E+09, 9.920104E+09,
    1.055728E+10, 1.123149E+10, 1.194468E+10, 1.269885E+10, 1.349612E+10, 1.433869E+10,
    1.522888E+10, 1.616909E+10, 1.716185E+10, 1.820978E+10, 1.931564E+10, 2.048228E+10,
    2.171272E+10, 2.301006E+10, 2.437758E+10, 2.581868E+10, 2.733690E+10, 2.893594E+10,
    3.061966E+10, 3.239208E+10, 3.425739E+10, 3.621994E+10, 3.828429E+10, 4.045517E+10,
    4.273749E+10, 4.513640E+10, 4.765723E+10, 5.030553E+10, 5.308708E+10, 5.600790E+10,
    5.907422E+10, 6.229255E+10, 6.566964E+10, 6.921252E+10, 7.292848E+10, 7.682509E+10,
    8.091023E+10, 8.519208E+10,
])

#  ---------------------- M = 44, I = 2 ---------------------------
M = 44
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    2.026479E+01, 3.788677E+02, 7.570816E+02, 1.145507E+03, 1.569915E+03, 2.058345E+03,
    2.636068E+03, 3.328628E+03, 4.164690E+03, 5.177665E+03, 6.406892E+03, 7.898611E+03,
    9.707022E+03, 1.189546E+04, 1.453793E+04, 1.772040E+04, 2.154245E+04, 2.611938E+04,
    3.158409E+04, 3.808955E+04, 4.581120E+04, 5.494989E+04, 6.573509E+04, 7.842821E+04,
    9.332667E+04, 1.107679E+05, 1.311340E+05, 1.548574E+05, 1.824256E+05, 2.143877E+05,
    2.513616E+05, 2.940409E+05, 3.432024E+05, 3.997164E+05, 4.645548E+05, 5.388019E+05,
    6.236663E+05, 7.204925E+05, 8.307742E+05, 9.561694E+05, 1.098515E+06, 1.259846E+06,
    1.442409E+06, 1.648688E+06, 1.881421E+06, 2.143626E+06, 2.438622E+06, 2.770061E+06,
    3.141951E+06, 3.558689E+06, 4.025097E+06, 4.546452E+06, 5.128527E+06, 5.777632E+06,
    6.500654E+06, 7.305109E+06, 8.199187E+06, 9.191807E+06, 1.029267E+07, 1.151234E+07,
    1.286226E+07, 1.435488E+07, 1.600369E+07, 1.782331E+07, 1.982959E+07, 2.203965E+07,
    2.447203E+07, 2.714675E+07, 3.008544E+07, 3.331144E+07, 3.684992E+07, 4.072800E+07,
    4.497489E+07, 4.962204E+07, 5.470328E+07, 6.025497E+07, 6.631617E+07, 7.292885E+07,
    8.013801E+07, 8.799194E+07, 9.654241E+07, 1.058449E+08, 1.159587E+08, 1.269474E+08,
    1.388789E+08, 1.518261E+08, 1.658664E+08, 1.810828E+08, 1.975638E+08, 2.154040E+08,
    2.347039E+08, 2.555711E+08, 2.781199E+08, 3.024721E+08, 3.287576E+08, 3.571142E+08,
    3.876889E+08, 4.206376E+08, 4.561262E+08, 4.943310E+08, 5.354389E+08, 5.796485E+08,
    6.271706E+08, 6.782284E+08, 7.330590E+08, 7.919132E+08, 8.550571E+08, 9.227721E+08,
    9.953563E+08, 1.073125E+09, 1.156413E+09, 1.245571E+09, 1.340974E+09, 1.443015E+09,
    1.552110E+09, 1.668700E+09, 1.793248E+09, 1.926244E+09, 2.068205E+09, 2.219676E+09,
    2.381232E+09, 2.553477E+09, 2.737051E+09, 2.932625E+09, 3.140908E+09, 3.362643E+09,
    3.598617E+09, 3.849653E+09, 4.116620E+09, 4.400430E+09, 4.702043E+09, 5.022468E+09,
    5.362763E+09, 5.724042E+09, 6.107473E+09, 6.514284E+09, 6.945762E+09, 7.403258E+09,
    7.888191E+09, 8.402048E+09, 8.946386E+09, 9.522842E+09, 1.013313E+10, 1.077904E+10,
    1.146245E+10, 1.218534E+10, 1.294977E+10, 1.375789E+10, 1.461196E+10, 1.551435E+10,
    1.646753E+10, 1.747409E+10, 1.853673E+10, 1.965827E+10, 2.084169E+10, 2.209005E+10,
    2.340660E+10, 2.479470E+10, 2.625787E+10, 2.779978E+10, 2.942429E+10, 3.113539E+10,
    3.293726E+10, 3.483426E+10, 3.683096E+10, 3.893208E+10, 4.114258E+10, 4.346763E+10,
    4.591259E+10, 4.848308E+10, 5.118493E+10, 5.402423E+10, 5.700733E+10, 6.014081E+10,
    6.343157E+10, 6.688675E+10,
])

#  ---------------------- M = 44, I = 3 ---------------------------
M = 44
I = 3
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    5.938131E+01, 1.108247E+03, 2.214517E+03, 3.350727E+03, 4.592300E+03, 6.021313E+03,
    7.711766E+03, 9.738490E+03, 1.218549E+04, 1.515071E+04, 1.874951E+04, 2.311752E+04,
    2.841381E+04, 3.482427E+04, 4.256628E+04, 5.189234E+04, 6.309521E+04, 7.651387E+04,
    9.253932E+04, 1.116218E+05, 1.342779E+05, 1.610995E+05, 1.927630E+05, 2.300395E+05,
    2.738067E+05, 3.250609E+05, 3.849317E+05, 4.546970E+05, 5.357995E+05, 6.298651E+05,
    7.387238E+05, 8.644319E+05, 1.009295E+06, 1.175895E+06, 1.367122E+06, 1.586198E+06,
    1.836720E+06, 2.122693E+06, 2.448568E+06, 2.819290E+06, 3.240346E+06, 3.717812E+06,
    4.258411E+06, 4.869576E+06, 5.559511E+06, 6.337263E+06, 7.212800E+06, 8.197089E+06,
    9.302184E+06, 1.054133E+07, 1.192904E+07, 1.348124E+07, 1.521536E+07, 1.715048E+07,
    1.930742E+07, 2.170896E+07, 2.437993E+07, 2.734741E+07, 3.064087E+07, 3.429243E+07,
    3.833699E+07, 4.281248E+07, 4.776010E+07, 5.322453E+07, 5.925423E+07, 6.590171E+07,
    7.322381E+07, 8.128204E+07, 9.014290E+07, 9.987824E+07, 1.105657E+08, 1.222889E+08,
    1.351382E+08, 1.492109E+08, 1.646119E+08, 1.814539E+08, 1.998583E+08, 2.199558E+08,
    2.418865E+08, 2.658011E+08, 2.918612E+08, 3.202405E+08, 3.511248E+08, 3.847136E+08,
    4.212204E+08, 4.608738E+08, 5.039185E+08, 5.506162E+08, 6.012466E+08, 6.561088E+08,
    7.155218E+08, 7.798268E+08, 8.493872E+08, 9.245912E+08, 1.005852E+09, 1.093611E+09,
    1.188338E+09, 1.290532E+09, 1.400726E+09, 1.519485E+09, 1.647413E+09, 1.785148E+09,
    1.933372E+09, 2.092806E+09, 2.264219E+09, 2.448424E+09, 2.646286E+09, 2.858721E+09,
    3.086701E+09, 3.331257E+09, 3.593479E+09, 3.874525E+09, 4.175617E+09, 4.498052E+09,
    4.843200E+09, 5.212513E+09, 5.607523E+09, 6.029851E+09, 6.481212E+09, 6.963415E+09,
    7.478371E+09, 8.028099E+09, 8.614729E+09, 9.240508E+09, 9.907808E+09, 1.061913E+10,
    1.137711E+10, 1.218453E+10, 1.304431E+10, 1.395953E+10, 1.493346E+10, 1.596950E+10,
    1.707126E+10, 1.824252E+10, 1.948726E+10, 2.080968E+10, 2.221419E+10, 2.370541E+10,
    2.528822E+10, 2.696772E+10, 2.874930E+10, 3.063859E+10, 3.264151E+10, 3.476429E+10,
    3.701345E+10, 3.939584E+10, 4.191864E+10, 4.458939E+10, 4.741597E+10, 5.040668E+10,
    5.357018E+10, 5.691557E+10, 6.045237E+10, 6.419056E+10, 6.814058E+10, 7.231337E+10,
    7.672038E+10, 8.137359E+10, 8.628553E+10, 9.146932E+10, 9.693866E+10, 1.027079E+11,
    1.087920E+11, 1.152067E+11, 1.219684E+11, 1.290941E+11, 1.366017E+11, 1.445099E+11,
    1.528383E+11, 1.616071E+11, 1.708377E+11, 1.805521E+11, 1.907737E+11, 2.015265E+11,
    2.128357E+11, 2.247276E+11,
])

#  ---------------------- M = 44, I = 4 ---------------------------
M = 44
I = 4
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    5.938661E+01, 1.108336E+03, 2.214589E+03, 3.350570E+03, 4.591574E+03, 6.019498E+03,
    7.708091E+03, 9.731833E+03, 1.217425E+04, 1.513263E+04, 1.872146E+04, 2.307521E+04,
    2.835138E+04, 3.473390E+04, 4.243751E+04, 5.171140E+04, 6.284408E+04, 7.616915E+04,
    9.207088E+04, 1.109911E+05, 1.334360E+05, 1.599844E+05, 1.912971E+05, 2.281258E+05,
    2.713248E+05, 3.218620E+05, 3.808325E+05, 4.494733E+05, 5.291779E+05, 6.215133E+05,
    7.282402E+05, 8.513320E+05, 9.929969E+05, 1.155703E+06, 1.342204E+06, 1.555566E+06,
    1.799202E+06, 2.076901E+06, 2.392867E+06, 2.751758E+06, 3.158724E+06, 3.619459E+06,
    4.140242E+06, 4.727999E+06, 5.390348E+06, 6.135671E+06, 6.973171E+06, 7.912946E+06,
    8.966061E+06, 1.014463E+07, 1.146190E+07, 1.293235E+07, 1.457177E+07, 1.639738E+07,
    1.842795E+07, 2.068389E+07, 2.318741E+07, 2.596262E+07, 2.903569E+07, 3.243502E+07,
    3.619134E+07, 4.033796E+07, 4.491089E+07, 4.994905E+07, 5.549449E+07, 6.159258E+07,
    6.829223E+07, 7.564616E+07, 8.371114E+07, 9.254824E+07, 1.022231E+08, 1.128064E+08,
    1.243737E+08, 1.370064E+08, 1.507918E+08, 1.658230E+08, 1.822004E+08, 2.000307E+08,
    2.194287E+08, 2.405167E+08, 2.634254E+08, 2.882945E+08, 3.152730E+08, 3.445196E+08,
    3.762037E+08, 4.105057E+08, 4.476175E+08, 4.877435E+08, 5.311009E+08, 5.779207E+08,
    6.284482E+08, 6.829439E+08, 7.416843E+08, 8.049627E+08, 8.730900E+08, 9.463957E+08,
    1.025229E+09, 1.109959E+09, 1.200977E+09, 1.298696E+09, 1.403554E+09, 1.516013E+09,
    1.636560E+09, 1.765712E+09, 1.904012E+09, 2.052033E+09, 2.210381E+09, 2.379692E+09,
    2.560640E+09, 2.753931E+09, 2.960311E+09, 3.180562E+09, 3.415509E+09, 3.666020E+09,
    3.933004E+09, 4.217420E+09, 4.520273E+09, 4.842618E+09, 5.185562E+09, 5.550269E+09,
    5.937956E+09, 6.349900E+09, 6.787441E+09, 7.251982E+09, 7.744989E+09, 8.268002E+09,
    8.822629E+09, 9.410552E+09, 1.003353E+10, 1.069341E+10, 1.139211E+10, 1.213163E+10,
    1.291408E+10, 1.374165E+10, 1.461662E+10, 1.554138E+10, 1.651842E+10, 1.755032E+10,
    1.863980E+10, 1.978967E+10, 2.100286E+10, 2.228244E+10, 2.363158E+10, 2.505360E+10,
    2.655195E+10, 2.813020E+10, 2.979208E+10, 3.154147E+10, 3.338240E+10, 3.531905E+10,
    3.735576E+10, 3.949704E+10, 4.174758E+10, 4.411223E+10, 4.659605E+10, 4.920424E+10,
    5.194225E+10, 5.481567E+10, 5.783035E+10, 6.099230E+10, 6.430779E+10, 6.778327E+10,
    7.142544E+10, 7.524123E+10, 7.923781E+10, 8.342260E+10, 8.780325E+10, 9.238769E+10,
    9.718412E+10, 1.022010E+11, 1.074470E+11, 1.129313E+11, 1.186630E+11, 1.246519E+11,
    1.309079E+11, 1.374411E+11,
])

#  ---------------------- M = 44, I = 5 ---------------------------
M = 44
I = 5
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    6.090077E+01, 1.138773E+03, 2.275792E+03, 3.443921E+03, 4.720878E+03, 6.191305E+03,
    7.931657E+03, 1.001940E+04, 1.254164E+04, 1.560008E+04, 1.931466E+04, 2.382667E+04,
    2.930200E+04, 3.593494E+04, 4.395300E+04, 5.362102E+04, 6.524665E+04, 7.918681E+04,
    9.585408E+04, 1.157246E+05, 1.393459E+05, 1.673468E+05, 2.004476E+05, 2.394715E+05,
    2.853580E+05, 3.391763E+05, 4.021417E+05, 4.756331E+05, 5.612115E+05, 6.606413E+05,
    7.759143E+05, 9.092750E+05, 1.063248E+06, 1.240670E+06, 1.444723E+06, 1.678972E+06,
    1.947406E+06, 2.254480E+06, 2.605165E+06, 3.005004E+06, 3.460163E+06, 3.977501E+06,
    4.564632E+06, 5.230003E+06, 5.982976E+06, 6.833909E+06, 7.794262E+06, 8.876687E+06,
    1.009515E+07, 1.146505E+07, 1.300334E+07, 1.472869E+07, 1.666160E+07, 1.882462E+07,
    2.124247E+07, 2.394228E+07, 2.695376E+07, 3.030945E+07, 3.404492E+07, 3.819908E+07,
    4.281440E+07, 4.793725E+07, 5.361820E+07, 5.991235E+07, 6.687972E+07, 7.458561E+07,
    8.310104E+07, 9.250322E+07, 1.028760E+08, 1.143103E+08, 1.269049E+08, 1.407668E+08,
    1.560118E+08, 1.727656E+08, 1.911638E+08, 2.113534E+08, 2.334931E+08, 2.577542E+08,
    2.843221E+08, 3.133963E+08, 3.451922E+08, 3.799422E+08, 4.178961E+08, 4.593235E+08,
    5.045140E+08, 5.537796E+08, 6.074553E+08, 6.659014E+08, 7.295048E+08, 7.986808E+08,
    8.738752E+08, 9.555661E+08, 1.044266E+09, 1.140524E+09, 1.244930E+09, 1.358112E+09,
    1.480745E+09, 1.613551E+09, 1.757301E+09, 1.912819E+09, 2.080987E+09, 2.262745E+09,
    2.459099E+09, 2.671121E+09, 2.899954E+09, 3.146818E+09, 3.413014E+09, 3.699928E+09,
    4.009035E+09, 4.341907E+09, 4.700218E+09, 5.085748E+09, 5.500390E+09, 5.946158E+09,
    6.425194E+09, 6.939770E+09, 7.492304E+09, 8.085362E+09, 8.721666E+09, 9.404108E+09,
    1.013575E+10, 1.091986E+10, 1.175986E+10, 1.265943E+10, 1.362243E+10, 1.465296E+10,
    1.575536E+10, 1.693424E+10, 1.819446E+10, 1.954117E+10, 2.097981E+10, 2.251614E+10,
    2.415625E+10, 2.590657E+10, 2.777389E+10, 2.976540E+10, 3.188867E+10, 3.415170E+10,
    3.656294E+10, 3.913129E+10, 4.186616E+10, 4.477745E+10, 4.787562E+10, 5.117167E+10,
    5.467720E+10, 5.840445E+10, 6.236629E+10, 6.657628E+10, 7.104869E+10, 7.579854E+10,
    8.084163E+10, 8.619460E+10, 9.187493E+10, 9.790101E+10, 1.042922E+11, 1.110687E+11,
    1.182520E+11, 1.258645E+11, 1.339296E+11, 1.424723E+11, 1.515183E+11, 1.610951E+11,
    1.712311E+11, 1.819564E+11, 1.933025E+11, 2.053023E+11, 2.179905E+11, 2.314034E+11,
    2.455790E+11, 2.605572E+11, 2.763796E+11, 2.930901E+11, 3.107345E+11, 3.293607E+11,
    3.490188E+11, 3.697615E+11,
])

#  ---------------------- M = 44, I = 6 ---------------------------
M = 44
I = 6
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[2]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    4.755283E+01, 8.917192E+02, 1.782495E+03, 2.702949E+03, 3.721375E+03, 4.909994E+03,
    6.337485E+03, 8.078305E+03, 1.021922E+04, 1.286259E+04, 1.613004E+04, 2.016576E+04,
    2.514031E+04, 3.125496E+04, 3.874699E+04, 4.789457E+04, 5.902344E+04, 7.251376E+04,
    8.880723E+04, 1.084165E+05, 1.319339E+05, 1.600430E+05, 1.935301E+05, 2.332972E+05,
    2.803772E+05, 3.359502E+05, 4.013610E+05, 4.781392E+05, 5.680217E+05, 6.729762E+05,
    7.952284E+05, 9.372903E+05, 1.101993E+06, 1.292522E+06, 1.512454E+06, 1.765801E+06,
    2.057055E+06, 2.391235E+06, 2.773948E+06, 3.211440E+06, 3.710667E+06, 4.279359E+06,
    4.926097E+06, 5.660397E+06, 6.492793E+06, 7.434939E+06, 8.499703E+06, 9.701287E+06,
    1.105534E+07, 1.257909E+07, 1.429147E+07, 1.621329E+07, 1.836739E+07, 2.077878E+07,
    2.347487E+07, 2.648563E+07, 2.984384E+07, 3.358525E+07, 3.774889E+07, 4.237729E+07,
    4.751677E+07, 5.321771E+07, 5.953488E+07, 6.652777E+07, 7.426095E+07, 8.280442E+07,
    9.223405E+07, 1.026320E+08, 1.140870E+08, 1.266953E+08, 1.405605E+08, 1.557947E+08,
    1.725188E+08, 1.908630E+08, 2.109676E+08, 2.329838E+08, 2.570740E+08, 2.834131E+08,
    3.121888E+08, 3.436027E+08, 3.778713E+08, 4.152266E+08, 4.559173E+08, 5.002100E+08,
    5.483899E+08, 6.007624E+08, 6.576539E+08, 7.194134E+08, 7.864137E+08, 8.590528E+08,
    9.377556E+08, 1.022975E+09, 1.115194E+09, 1.214928E+09, 1.322724E+09, 1.439167E+09,
    1.564877E+09, 1.700514E+09, 1.846781E+09, 2.004423E+09, 2.174233E+09, 2.357051E+09,
    2.553770E+09, 2.765338E+09, 2.992756E+09, 3.237090E+09, 3.499466E+09, 3.781077E+09,
    4.083186E+09, 4.407130E+09, 4.754323E+09, 5.126259E+09, 5.524518E+09, 5.950769E+09,
    6.406775E+09, 6.894396E+09, 7.415596E+09, 7.972447E+09, 8.567134E+09, 9.201960E+09,
    9.879354E+09, 1.060187E+10, 1.137221E+10, 1.219320E+10, 1.306783E+10, 1.399924E+10,
    1.499073E+10, 1.604578E+10, 1.716804E+10, 1.836135E+10, 1.962973E+10, 2.097741E+10,
    2.240886E+10, 2.392872E+10, 2.554190E+10, 2.725353E+10, 2.906899E+10, 3.099394E+10,
    3.303430E+10, 3.519625E+10, 3.748631E+10, 3.991127E+10, 4.247826E+10, 4.519474E+10,
    4.806852E+10, 5.110775E+10, 5.432098E+10, 5.771714E+10, 6.130557E+10, 6.509603E+10,
    6.909872E+10, 7.332428E+10, 7.778385E+10, 8.248906E+10, 8.745202E+10, 9.268540E+10,
    9.820242E+10, 1.040169E+11, 1.101431E+11, 1.165961E+11, 1.233916E+11, 1.305458E+11,
    1.380757E+11, 1.459989E+11, 1.543341E+11, 1.631002E+11, 1.723174E+11, 1.820064E+11,
    1.921888E+11, 2.028874E+11, 2.141254E+11, 2.259273E+11, 2.383185E+11, 2.513254E+11,
    2.649754E+11, 2.792970E+11,
])

#  ---------------------- M = 45, I = 1 ---------------------------
M = 45
I = 1
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    1.000000E+00, 1.001788E+00, 1.126881E+00, 1.526206E+00, 2.077149E+00, 2.667730E+00,
    3.249956E+00, 3.809254E+00, 4.344827E+00, 4.860660E+00, 5.361676E+00, 5.852264E+00,
    6.335855E+00, 6.814950E+00, 7.291287E+00, 7.766034E+00, 8.239959E+00, 8.713552E+00,
    9.187121E+00, 9.660856E+00, 1.013487E+01, 1.060924E+01, 1.108399E+01, 1.155914E+01,
    1.203472E+01, 1.251071E+01, 1.298713E+01, 1.346397E+01, 1.394124E+01, 1.441895E+01,
    1.489709E+01, 1.537567E+01, 1.585471E+01, 1.633422E+01, 1.681420E+01, 1.729468E+01,
    1.777567E+01, 1.825719E+01, 1.873927E+01, 1.922193E+01, 1.970520E+01, 2.018910E+01,
    2.067367E+01, 2.115894E+01, 2.164494E+01, 2.213171E+01, 2.261929E+01, 2.310771E+01,
    2.359701E+01, 2.408724E+01, 2.457844E+01, 2.507064E+01, 2.556390E+01, 2.605825E+01,
    2.655374E+01, 2.705042E+01, 2.754832E+01, 2.804751E+01, 2.854802E+01, 2.904991E+01,
    2.955321E+01, 3.005797E+01, 3.056425E+01, 3.107208E+01, 3.158152E+01, 3.209261E+01,
    3.260540E+01, 3.311994E+01, 3.363627E+01, 3.415443E+01, 3.467448E+01, 3.519645E+01,
    3.572040E+01, 3.624636E+01, 3.677439E+01, 3.730452E+01, 3.783680E+01, 3.837127E+01,
    3.890798E+01, 3.944696E+01, 3.998826E+01, 4.053192E+01, 4.107798E+01, 4.162648E+01,
    4.217746E+01, 4.273095E+01, 4.328701E+01, 4.384566E+01, 4.440694E+01, 4.497089E+01,
    4.553755E+01, 4.610696E+01, 4.667914E+01, 4.725414E+01, 4.783199E+01, 4.841272E+01,
    4.899637E+01, 4.958297E+01, 5.017256E+01, 5.076516E+01, 5.136081E+01, 5.195955E+01,
    5.256139E+01, 5.316638E+01, 5.377454E+01, 5.438591E+01, 5.500051E+01, 5.561837E+01,
    5.623952E+01, 5.686400E+01, 5.749182E+01, 5.812302E+01, 5.875762E+01, 5.939565E+01,
    6.003713E+01, 6.068210E+01, 6.133059E+01, 6.198260E+01, 6.263818E+01, 6.329734E+01,
    6.396011E+01, 6.462652E+01, 6.529658E+01, 6.597033E+01, 6.664778E+01, 6.732897E+01,
    6.801391E+01, 6.870262E+01, 6.939513E+01, 7.009147E+01, 7.079164E+01, 7.149568E+01,
    7.220361E+01, 7.291544E+01, 7.363121E+01, 7.435092E+01, 7.507461E+01, 7.580228E+01,
    7.653397E+01, 7.726969E+01, 7.800946E+01, 7.875330E+01, 7.950123E+01, 8.025328E+01,
    8.100945E+01, 8.176978E+01, 8.253427E+01, 8.330295E+01, 8.407583E+01, 8.485294E+01,
    8.563429E+01, 8.641990E+01, 8.720979E+01, 8.800398E+01, 8.880249E+01, 8.960532E+01,
    9.041251E+01, 9.122406E+01, 9.204000E+01, 9.286034E+01, 9.368510E+01, 9.451430E+01,
    9.534795E+01, 9.618607E+01, 9.702868E+01, 9.787579E+01, 9.872743E+01, 9.958360E+01,
    1.004443E+02, 1.013096E+02, 1.021795E+02, 1.030540E+02, 1.039331E+02, 1.048168E+02,
    1.057052E+02, 1.065983E+02, 1.074960E+02, 1.083984E+02, 1.093056E+02, 1.102175E+02,
    1.111341E+02, 1.120555E+02, 1.129816E+02, 1.139126E+02, 1.148484E+02, 1.157890E+02,
    1.167344E+02, 1.176847E+02, 1.186398E+02, 1.195998E+02, 1.205648E+02, 1.215346E+02,
    1.225094E+02, 1.234891E+02, 1.244738E+02, 1.254635E+02, 1.264581E+02, 1.274578E+02,
    1.284624E+02, 1.294721E+02, 1.304869E+02, 1.315067E+02, 1.325316E+02, 1.335616E+02,
    1.345967E+02, 1.356369E+02, 1.366822E+02, 1.377327E+02, 1.387884E+02, 1.398492E+02,
    1.409153E+02, 1.419865E+02, 1.430630E+02, 1.441447E+02, 1.452317E+02, 1.463239E+02,
    1.474214E+02, 1.485242E+02, 1.496323E+02, 1.507458E+02, 1.518645E+02, 1.529887E+02,
    1.541181E+02, 1.552530E+02, 1.563932E+02, 1.575389E+02, 1.586900E+02, 1.598465E+02,
    1.610084E+02, 1.621758E+02, 1.633487E+02, 1.645271E+02, 1.657110E+02, 1.669004E+02,
    1.680953E+02, 1.692957E+02, 1.705017E+02, 1.717133E+02, 1.729305E+02, 1.741532E+02,
    1.753815E+02, 1.766155E+02, 1.778551E+02, 1.791004E+02, 1.803513E+02, 1.816078E+02,
    1.828701E+02, 1.841380E+02, 1.854117E+02, 1.866910E+02, 1.879761E+02, 1.892670E+02,
    1.905636E+02, 1.918659E+02, 1.931741E+02, 1.944880E+02, 1.958078E+02, 1.971333E+02,
    1.984647E+02, 1.998019E+02, 2.011450E+02, 2.024939E+02, 2.038487E+02, 2.052094E+02,
    2.065760E+02, 2.079485E+02, 2.093269E+02, 2.107112E+02, 2.121014E+02, 2.134977E+02,
    2.148998E+02, 2.163080E+02, 2.177221E+02, 2.191422E+02, 2.205683E+02, 2.220004E+02,
    2.234386E+02, 2.248827E+02, 2.263329E+02, 2.277892E+02, 2.292515E+02, 2.307199E+02,
    2.321943E+02, 2.336749E+02, 2.351615E+02, 2.366543E+02, 2.381531E+02, 2.396581E+02,
    2.411692E+02, 2.426864E+02, 2.442098E+02, 2.457394E+02, 2.472751E+02, 2.488169E+02,
    2.503650E+02, 2.519192E+02, 2.534797E+02, 2.550463E+02, 2.566191E+02, 2.581982E+02,
    2.597835E+02,
])

#  ---------------------- M = 45, I = 2 ---------------------------
M = 45
I = 2
TIPS_2017_ISOT_HASH[(M,I)] = TIPS_2017_ISOT[1]
TIPS_2017_ISOQ_HASH[(M,I)] = float64([
    6.000000E+00, 6.029347E+00, 6.728818E+00, 8.168283E+00, 9.865998E+00, 1.164890E+01,
    1.346792E+01, 1.530608E+01, 1.715597E+01, 1.901383E+01, 2.087753E+01, 2.274575E+01,
    2.461767E+01, 2.649271E+01, 2.837047E+01, 3.025068E+01, 3.213312E+01, 3.401765E+01,
    3.590412E+01, 3.779247E+01, 3.968262E+01, 4.157454E+01, 4.346818E+01, 4.536356E+01,
    4.726067E+01, 4.915955E+01, 5.106023E+01, 5.296280E+01, 5.486733E+01, 5.677392E+01,
    5.868270E+01, 6.059382E+01, 6.250744E+01, 6.442372E+01, 6.634288E+01, 6.826512E+01,
    7.019067E+01, 7.211977E+01, 7.405266E+01, 7.598962E+01, 7.793091E+01, 7.987682E+01,
    8.182762E+01, 8.378361E+01, 8.574509E+01, 8.771234E+01, 8.968568E+01, 9.166539E+01,
    9.365178E+01, 9.564515E+01, 9.764579E+01, 9.965399E+01, 1.016701E+02, 1.036943E+02,
    1.057269E+02, 1.077682E+02, 1.098186E+02, 1.118781E+02, 1.139472E+02, 1.160261E+02,
    1.181150E+02, 1.202141E+02, 1.223238E+02, 1.244442E+02, 1.265756E+02, 1.287182E+02,
    1.308723E+02, 1.330380E+02, 1.352155E+02, 1.374051E+02, 1.396070E+02, 1.418213E+02,
    1.440482E+02, 1.462879E+02, 1.485407E+02, 1.508066E+02, 1.530859E+02, 1.553786E+02,
    1.576850E+02, 1.600053E+02, 1.623395E+02, 1.646878E+02, 1.670504E+02, 1.694274E+02,
    1.718190E+02, 1.742252E+02, 1.766463E+02, 1.790823E+02, 1.815333E+02, 1.839996E+02,
    1.864811E+02, 1.889781E+02, 1.914906E+02, 1.940188E+02, 1.965627E+02, 1.991225E+02,
    2.016983E+02, 2.042901E+02, 2.068981E+02, 2.095224E+02, 2.121631E+02, 2.148202E+02,
    2.174939E+02, 2.201843E+02, 2.228913E+02, 2.256153E+02, 2.283561E+02, 2.311140E+02,
    2.338889E+02, 2.366811E+02, 2.394905E+02, 2.423172E+02, 2.451614E+02, 2.480230E+02,
    2.509023E+02, 2.537992E+02, 2.567138E+02, 2.596462E+02, 2.625966E+02, 2.655649E+02,
    2.685512E+02, 2.715556E+02, 2.745782E+02, 2.776191E+02, 2.806782E+02, 2.837558E+02,
    2.868518E+02, 2.899664E+02, 2.930995E+02, 2.962513E+02, 2.994219E+02, 3.026112E+02,
    3.058195E+02, 3.090466E+02, 3.122928E+02, 3.155580E+02, 3.188423E+02, 3.221459E+02,
    3.254687E+02, 3.288109E+02, 3.321724E+02, 3.355534E+02, 3.389539E+02, 3.423740E+02,
    3.458138E+02, 3.492732E+02, 3.527525E+02, 3.562516E+02, 3.597706E+02, 3.633095E+02,
    3.668685E+02, 3.704476E+02, 3.740468E+02, 3.776663E+02, 3.813061E+02, 3.849662E+02,
    3.886467E+02, 3.923476E+02, 3.960692E+02, 3.998113E+02, 4.035740E+02, 4.073576E+02,
    4.111618E+02, 4.149870E+02, 4.188330E+02, 4.227001E+02, 4.265882E+02, 4.304974E+02,
    4.344277E+02, 4.383793E+02, 4.423522E+02, 4.463464E+02, 4.503621E+02, 4.543992E+02,
    4.584579E+02, 4.625383E+02, 4.666403E+02, 4.707640E+02, 4.749095E+02, 4.790770E+02,
    4.832663E+02, 4.874777E+02, 4.917111E+02, 4.959667E+02, 5.002444E+02, 5.045444E+02,
    5.088667E+02, 5.132115E+02, 5.175787E+02, 5.219684E+02, 5.263806E+02, 5.308156E+02,
    5.352732E+02, 5.397537E+02, 5.442570E+02, 5.487832E+02, 5.533323E+02, 5.579046E+02,
    5.624999E+02, 5.671184E+02, 5.717602E+02, 5.764253E+02, 5.811137E+02, 5.858256E+02,
    5.905610E+02, 5.953200E+02, 6.001026E+02, 6.049090E+02, 6.097391E+02, 6.145930E+02,
    6.194709E+02, 6.243727E+02, 6.292986E+02, 6.342486E+02, 6.392228E+02, 6.442212E+02,
    6.492439E+02, 6.542909E+02, 6.593625E+02, 6.644585E+02, 6.695791E+02, 6.747243E+02,
    6.798943E+02, 6.850890E+02, 6.903085E+02, 6.955530E+02, 7.008224E+02, 7.061168E+02,
    7.114364E+02, 7.167811