#Code accompanying the paper, "Phenotypic plasticity as a route to population shifts via tipping points".
#Used to plot Figure 2 from the paper. 
#Requires Simulation.csv to be in file (see below) attained from running BlowflyRunner.jl with parameters v1=-4,v2=3.36,v3=-0.50.


using Plots, LsqFit, LaTeXStrings, CSV,DataFrames,Measures
using Statistics, VectorizedStatistics, Distributions

Plots.scalefontsizes()
Plots.scalefontsizes(1.7)

SJdata=CSV.read("SurvivalData.csv",DataFrame)
SJdataN=length(SJdata[!,2])


taus=[0.6,5,10];#E,L,pi+J
tau=sum(taus)
deltas=[0.07,0.004];
SE=exp(-taus[1]*deltas[1]);
SL=exp(-taus[2]*deltas[2]);
Spi=exp(-0.0025*4.1);
deltaA=0.27;

KL=50000

q1=3.95;
q2=6.9;
q3=-0.97;
q4=0.78;

function q(a)
    if a.<exp(-q3/q2)
        out=0
    else
        out= q1.*(q2.*log.(a).+q3).^q4
    end
    return out
end

function qprimeoverq(a)
    return q2.*q4./(a.*(q2.*log.(a).+q3))
end

function aq(q)#Goes from q to the classes a
    exp(((q./q1).^(1/q4)-q3)./q2);
end


function KAfun(a,SJs,qs)#Goes from q to the classes a
    return -(KL.*deltas[2].*SL.*Spi.*SJs)./(deltaA.*(1 .-SL).*a.*log.(deltaA./(qs.*SE.*SL.*Spi.*SJs)))
end

function As(KAs,SJs,qs)
    KAs.*log.(qs.*SE.*SL.*Spi.*SJs./deltaA)
end

function Ls(KAs,SJs,qs)
    (deltaA.*(1 .-SL))./(deltas[2].*SL.*Spi.*SJs).*As(KAs,SJs,qs)
end

function stability(qs,SJs)
    stables=(deltaA.<qs.*SE.*SL.*Spi.*SJs);
    logs=log.(deltaA./(qs.*SE.*SL.*Spi.*SJs));
    possStabs=logs[stables][-1 .<1 ./(1 .+logs[stables]).<1];
    complexStabs=-acos.(1 ./(1 .+possStabs)) .+ deltaA*tau*sqrt.((1 .+possStabs).^2 .- 1);
    stables[stables .& (-1 .<1 ./(1 .+logs).<1)].=(complexStabs.<0);
    
    return stables;
end

function realEigen(qs,SJs)
    logs=log.(deltaA./(qs.*SE.*SL.*Spi.*SJs))
    return logs.<-(1+(1 /(exp(1+deltaA*tau)*deltaA*tau)))
end


function Bifurcation(a,SJs,SJpRatios)
    lR=log.(qs.*SE.*SL*Spi.*SJs./deltaA)
    out=(SJpRatios.+qprimeoverq(a))./lR
    out[lR.<1].=NaN
    return out
end


v1=-4#
v2=3.36#
v3=-0.44 # -0.44 is Normal -0.51 is Modified

function G(x,p)#logistic quadratic regression for G=ln(SJ/(1-SJ))
    p[1].+p[2]*log.(x).+p[3]*log.(x).^2;
end

function Gprime(x,p)
    p[2]./x.+p[3]*2*log.(x)./x
end

function SJ(a,p)#Equation for the survival probability classes
    KLexp=10000;
    exp.(G.(KLexp./(5*a),Ref(p)))./(1 .+exp.(G.(KLexp./(5*a),Ref(p))));#G(KL/5a)
end

function SJprime(a,p)
    KLexperiment=10000
    (-(KLexperiment/5)./a.^2 .*Gprime.(KLexperiment./(5*a),Ref(p)).*exp.(G.(KLexperiment./(5*a),Ref(p))))./((1 .+exp.(G.(KLexperiment./(5*a),Ref(p)))).^2)
end

function SJprimeoverSJ(a,p)
    SJprime.(a,Ref(p))./SJ.(a,Ref(p));
end

function SJHolling1(a,p)
    SJ1=exp(p[1])
    SJ=SJ1.*a
    SJ[SJ.<0].=0
    SJ[SJ.>p[2]].=p[2]
    return SJ
end

function SJprimeoverSJHolling1(a,p)
    SJ1=exp(p[1])
    SJ=SJ1.*a
    SJprime=1 ./a
    SJprime[SJ.<0].=0
    SJprime[SJ.>p[2]].=0
    return SJprime
end

function SJHolling2(a,p)
    SJ1=exp(p[1])
    SJ2=exp(p[2])
    SJ=SJ1.*a./(1 .+SJ1.*SJ2.*a)
    SJ[SJ.<0].=0
    SJ[SJ.>1].=1
    return SJ
end

function SJprimeoverSJHolling2(a,p)
    SJ1=exp(p[1])
    SJ2=exp(p[2])
    SJ=SJ1.*a./(1 .+SJ1.*SJ2.*a)
    SJprime=1 ./(a.*(1 .+SJ1.*SJ2.*a))
    SJprime[SJ.<0].=0
    SJprime[SJ.>1].=0
    return SJprime
end

function SJHolling3(a,p)
    SJ1=exp(p[1])
    SJ2=exp(p[2])
    SJ3=exp(p[3])+1
    SJ=SJ1.*(a.^SJ3)./(1 .+SJ1.*SJ2.*(a.^SJ3))
    SJ[SJ.<0].=0
    SJ[SJ.>1].=1
    return SJ
end

function SJprimeoverSJHolling3(a,p)
    SJ1=exp(p[1])
    SJ2=exp(p[2])
    SJ3=exp(p[3])+1
    SJ=SJ1.*(a.^SJ3)./(1 .+SJ1.*SJ2.*(a.^SJ3))
    SJprime=SJ3./(a.*(1 .+SJ1.*SJ2.*(a.^SJ3)))
    SJprime[SJ.<0].=0
    SJprime[SJ.>1].=0
    return SJprime
end

#set-up
Am=64*4*4*8+1;
class_as=Vector(LinRange(0,250,Am))
qs=q.(class_as)

#Moe
Moep0=[v1,v2,v3]
Moefit=curve_fit(SJ,SJdata[!,2],SJdata[!,7],Moep0)
MoeStd=stderror(Moefit)

σMoe = std(Moefit.resid)
errorMoe = quantile(Normal(0, σMoe), [.025,.975])



#Holling type-1
H1p0=[-3.0,1.0]
H1fit=curve_fit(SJHolling1,SJdata[!,2],SJdata[!,7],H1p0)
H1Std=stderror(H1fit)

σ1 = std(H1fit.resid)
error1 = quantile(Normal(0, σ1), [.025,.975])

#Holling type-2
H2p0=[-1.0, 0.0];
H2fit=curve_fit(SJHolling2,SJdata[!,2],SJdata[!,7],H2p0);
H2Std=stderror(H2fit)

σ2 = std(H2fit.resid)
error2 = quantile(Normal(0, σ2), [.025,.975])

#Holling type-3
H3p0=[0.0, 0.0, 0.0];
H3fit=curve_fit(SJHolling3,SJdata[!,2],SJdata[!,7],H3p0);
H3Std=stderror(H3fit)

σ3 = std(H3fit.resid)
error3 = quantile(Normal(0, σ3), [.025,.975])

SJNames=["Holling type-1" "Holling type-2" "Holling type-3"]
SJShortNames=["H1" "H2" "H3"]
SJfuncts=[SJHolling1,SJHolling2,SJHolling3]
SJpoverSJfuncts=[SJprimeoverSJHolling1,SJprimeoverSJHolling2,SJprimeoverSJHolling3]
SJps=[H1fit.param,H2fit.param,H3fit.param]
SJn=length(SJfuncts)

SJs  =zeros(Am,SJn)
SJpRatios=zeros(Am,SJn)
KAs  =zeros(Am,SJn)
Avals=zeros(Am,SJn)
Lvals=zeros(Am,SJn)
Stabs=zeros(Am,SJn)
RAs=  zeros(Am,SJn)
AtoKA=zeros(Am,SJn)
fs=   zeros(Am,SJn)
gs=   zeros(Am,SJn)
regTable=Array{Any}(undef,3,SJn)

for i in 1:SJn
    SJs[:,i]=SJfuncts[i](class_as,SJps[i])
    SJpRatios[:,i]=SJpoverSJfuncts[i](class_as,SJps[i])
    KAs[:,i]=KAfun.(class_as,SJs[:,i],qs)
    KAs[KAs[:,i].<0,i].=NaN
    Avals[:,i]=As(KAs[:,i],SJs[:,i],qs)
    Lvals[:,i]=Ls(KAs[:,i],SJs[:,i],qs)
    Stabs[:,i]=stability(qs,SJs[:,i])
    RAs[:,i]=KL*SJs[:,i]./class_as./taus[2]
    AtoKA[:,i]=Avals[:,i]./KAs[:,i]
    fs[:,i]=SJpoverSJfuncts[i](class_as,SJps[i]).-1 ./class_as
    gs[:,i]=Bifurcation(class_as,SJs[:,i],SJpRatios[:,i])

    regTable[1,i]=SJNames[i]
    regTable[2,i]=sum((SJfuncts[i](SJdata[!,2],SJps[i]).-SJdata[!,7]).^2)
    regTable[3,i]=2*length(SJps[i])+SJdataN*log(regTable[2,i]/SJdataN)
end

simN=1000;
bound=0.1;
aMat=hcat(fill.(class_as, simN)...)';

SJH1s=zeros(Am,simN);
SJpRatioH1s=zeros(Am,simN);
KAH1s  =zeros(Am,simN);
AvalH1s=zeros(Am,simN);
LvalH1s=zeros(Am,simN);
StabH1s=zeros(Am,simN);
RAH1s=  zeros(Am,simN);
AtoKAH1s=zeros(Am,simN);

H1Samples=rand(MvNormal(SJps[1],H1Std),simN)'
for i in 1:simN
    SJH1s[:,i]=SJHolling1(class_as,H1Samples[i,:])
    SJpRatioH1s[:,i]=SJprimeoverSJHolling1(class_as,H1Samples[i,:])
    KAH1s[:,i]=KAfun.(class_as,SJH1s[:,i],qs)
    KAH1s[KAH1s[:,i].<0,i].=0
    AvalH1s[:,i]=As(KAH1s[:,i],SJH1s[:,i],qs)
    LvalH1s[:,i]=Ls(KAH1s[:,i],SJH1s[:,i],qs)
    StabH1s[:,i]=stability(qs,SJH1s[:,i])
    RAH1s[:,i]=KL*SJH1s[:,i]./class_as./taus[2]
    AtoKAH1s[:,i]=AvalH1s[:,i]./KAH1s[:,i]
end


SJH1Bounds=hcat(vquantile!(copy(SJH1s), 0.5,dims=2),vquantile!(copy(SJH1s), bound,dims=2),vquantile!(copy(SJH1s), 1-bound,dims=2));
AH1Bounds=hcat(vquantile!(copy(AvalH1s), 0.5,dims=2),vquantile!(copy(AvalH1s), bound,dims=2),vquantile!(copy(AvalH1s), 1-bound,dims=2));
KAH1Bounds=hcat(vquantile!(copy(KAH1s), 0.5,dims=2),vquantile!(copy(KAH1s), bound,dims=2),vquantile!(copy(KAH1s), 1-bound,dims=2));

AH1Bounds=AH1Bounds[KAH1Bounds[:,1].>0,:];
KAH1Bounds=KAH1Bounds[KAH1Bounds[:,1].>0,:];

SJH2s=zeros(Am,simN);
SJpRatioH2s=zeros(Am,simN);
KAH2s  =zeros(Am,simN);
AvalH2s=zeros(Am,simN);
LvalH2s=zeros(Am,simN);
StabH2s=zeros(Am,simN);
RAH2s=  zeros(Am,simN);
AtoKAH2s=zeros(Am,simN);

H2Samples=rand(MvNormal(SJps[2],H2Std),simN)'
for i in 1:simN
    SJH2s[:,i]=SJHolling2(class_as,H2Samples[i,:])
    SJpRatioH2s[:,i]=SJprimeoverSJHolling2(class_as,H2Samples[i,:])
    KAH2s[:,i]=KAfun.(class_as,SJH2s[:,i],qs)
    KAH2s[KAH2s[:,i].<0,i].=0
    AvalH2s[:,i]=As(KAH2s[:,i],SJH2s[:,i],qs)
    LvalH2s[:,i]=Ls(KAH2s[:,i],SJH2s[:,i],qs)
    StabH2s[:,i]=stability(qs,SJH2s[:,i])
    RAH2s[:,i]=KL*SJH2s[:,i]./class_as./taus[2]
    AtoKAH2s[:,i]=AvalH2s[:,i]./KAH2s[:,i]
end


SJH2Bounds=hcat(vquantile!(copy(SJH2s), 0.5,dims=2),vquantile!(copy(SJH2s), bound,dims=2),vquantile!(copy(SJH2s), 1-bound,dims=2));
AH2Bounds=hcat(vquantile!(copy(AvalH2s), 0.5,dims=2),vquantile!(copy(AvalH2s), bound,dims=2),vquantile!(copy(AvalH2s), 1-bound,dims=2));
KAH2Bounds=hcat(vquantile!(copy(KAH2s), 0.5,dims=2),vquantile!(copy(KAH2s), bound,dims=2),vquantile!(copy(KAH2s), 1-bound,dims=2));

AH2Bounds=AH2Bounds[KAH2Bounds[:,1].>0,:];
KAH2Bounds=KAH2Bounds[KAH2Bounds[:,1].>0,:];


SJH3s=zeros(Am,simN);
SJpRatioH3s=zeros(Am,simN);
KAH3s  =zeros(Am,simN);
AvalH3s=zeros(Am,simN);
LvalH3s=zeros(Am,simN);
StabH3s=zeros(Am,simN);
RAH3s=  zeros(Am,simN);
AtoKAH3s=zeros(Am,simN);
fminusgH3=zeros(Am,simN);

H3Samples=rand(MvNormal(SJps[3],H3Std),simN)'
for i in 1:simN
    SJH3s[:,i]=SJHolling3(class_as,H3Samples[i,:])
    SJpRatioH3s[:,i]=SJprimeoverSJHolling3(class_as,H3Samples[i,:])
    KAH3s[:,i]=KAfun.(class_as,SJH3s[:,i],qs)
    KAH3s[KAH3s[:,i].<0,i].=0
    AvalH3s[:,i]=As(KAH3s[:,i],SJH3s[:,i],qs)
    LvalH3s[:,i]=Ls(KAH3s[:,i],SJH3s[:,i],qs)
    StabH3s[:,i]=stability(qs,SJH3s[:,i])
    RAH3s[:,i]=KL*SJH3s[:,i]./class_as./taus[2]
    AtoKAH3s[:,i]=AvalH3s[:,i]./KAH3s[:,i]
    fminusgH3[:,i]=SJpRatioH3s[:,i].-1 ./class_as.-Bifurcation(class_as,SJH3s[:,i],SJpRatioH3s[:,i])
end

SJH3Bounds=hcat(vquantile!(copy(SJH3s), 0.5,dims=2),vquantile!(copy(SJH3s), bound,dims=2),vquantile!(copy(SJH3s), 1-bound,dims=2));
AH3Bounds=hcat(vquantile!(copy(AvalH3s), 0.5,dims=2),vquantile!(copy(AvalH3s), bound,dims=2),vquantile!(copy(AvalH3s), 1-bound,dims=2));
KAH3Bounds=hcat(vquantile!(copy(KAH3s), 0.5,dims=2),vquantile!(copy(KAH3s), bound,dims=2),vquantile!(copy(KAH3s), 1-bound,dims=2));

AH3Bounds=AH3Bounds[KAH3Bounds[:,1].>0,:];
KAH3Bounds=KAH3Bounds[KAH3Bounds[:,1].>0,:];

flipH3s=(fminusgH3[1:Am-1,:].*fminusgH3[2:Am,:]).<0



RNPlot=Plots.plot(xlabel=L"Larval food per capita, $\alpha$ (mg)",ylabel=L"Through-juvenile survival, $S_J$",size=[1200,900],legend=false)
Plots.plot!(class_as,SJH1Bounds[:,1],ribbon=(SJH1Bounds[:,1].-SJH1Bounds[:,2],SJH1Bounds[:,3].-SJH1Bounds[:,1]),label="Holling 1",color=1,fillalpha=0.2,linewidth=2)
Plots.plot!(class_as,SJH2Bounds[:,1],ribbon=(SJH2Bounds[:,1].-SJH2Bounds[:,2],SJH2Bounds[:,3].-SJH2Bounds[:,1]),label="Holling 2",color=2,fillalpha=0.2,linewidth=2)
Plots.plot!(class_as,SJH3Bounds[:,1],ribbon=(SJH3Bounds[:,1].-SJH3Bounds[:,2],SJH3Bounds[:,3].-SJH3Bounds[:,1]),label="Holling 3",color=3,fillalpha=0.2,linewidth=2)
Plots.scatter!(aMat[1:Am-1,:][flipH3s],SJH3s[1:Am-1,:][flipH3s],color=3,label="Tipping points - Holling3",alpha=0.4,markersize=2)
Plots.scatter!(SJdata[!,2],SJdata[!,7],yerror=(SJdata[!,7].-SJdata[!,6],SJdata[!,8].-SJdata[!,7]),color=:black)
Plots.plot!(ylims=[0,1],right_margin=5mm,left_margin=5mm)


APlot=Plots.plot(xlabel=L"Adult food supply, $K_A$ (mg)",ylabel=L"Total adult population, $A_{tot}$",size=[1200,900],legend=false)#,xlims=[0,1000],ylims=[0,5000]
Plots.scatter!(KAH3s[1:Am-1,:][flipH3s],AvalH3s[1:Am-1,:][flipH3s],color=3,label="Tipping points - Holling3",alpha=0.4,markersize=2)
Plots.plot!(KAH1Bounds[:,1],AH1Bounds[:,1],ribbon=(AH1Bounds[:,1].-AH1Bounds[:,2],AH1Bounds[:,3].-AH1Bounds[:,1]),label="Holling 1",color=1,fillalpha=0.2,linewidth=2)
Plots.plot!(KAH2Bounds[:,1],AH2Bounds[:,1],ribbon=(AH2Bounds[:,1].-AH2Bounds[:,2],AH2Bounds[:,3].-AH2Bounds[:,1]),label="Holling 2",color=2,fillalpha=0.2,linewidth=2)
Plots.plot!(KAH3Bounds[:,1],AH3Bounds[:,1],ribbon=(AH3Bounds[:,1].-AH3Bounds[:,2],AH3Bounds[:,3].-AH3Bounds[:,1]),label="Holling 3",color=3,fillalpha=0.2,linewidth=2)
Plots.plot!(xlims=[0,3000],ylims=[1,1e4],right_margin=5mm,left_margin=5mm)

#Add in legends
legendsize=[1000,50]
legendplot= Plots.scatter([0,0], [0,1],size=legendsize,legendfontsize=10, legend=:outertopright,xlims=(1,1.1),color="black",linewidth=2,legend_column=-1,label="Experimental data", framestyle=:none)
Plots.plot!(legendplot,[0,0], [0,1],label="Holling Type-1",color=1,linewidth=2)
Plots.plot!(legendplot,[0,0], [0,1],label="Holling Type-2",color=2,linewidth=2)
Plots.plot!(legendplot,[0,0], [0,1],label="Holling Type-3",color=3,linewidth=2)


l= @layout[
    a{0.01h}; b c
]
CombinedPlot=Plots.plot(legendplot,RNPlot,APlot,size=[1000,800],layout=l,margin=10mm,left_margin=10mm,top_margin=0mm)

plot!(CombinedPlot, legend=false, xlabel=L"Larval food per capita, $\alpha$ (mg)", ylabel=L"Through-juvenile survival, $S_J$",xlabelfontsize=9,ylabelfontsize=9, framestyle = :box,xlims=[0,10],ylims=[0,0.8],linewidth=2,
inset = (2, bbox(0, 0.15, 0.5, 0.5, :bottom, :right)),subplot=4)
Plots.scatter!(aMat[1:Am-1,:][flipH3s],SJH3s[1:Am-1,:][flipH3s],color=3,label="Tipping points - Holling3",alpha=0.4,subplot=4)
Plots.plot!(class_as,SJH1Bounds[:,1],ribbon=(SJH1Bounds[:,1].-SJH1Bounds[:,2],SJH1Bounds[:,3].-SJH1Bounds[:,1]),label="Holling 1",color=1,fillalpha=0.2,linewidth=2,subplot=4)
Plots.plot!(class_as,SJH2Bounds[:,1],ribbon=(SJH2Bounds[:,1].-SJH2Bounds[:,2],SJH2Bounds[:,3].-SJH2Bounds[:,1]),label="Holling 2",color=2,fillalpha=0.2,linewidth=2,subplot=4)
Plots.plot!(class_as,SJH3Bounds[:,1],ribbon=(SJH3Bounds[:,1].-SJH3Bounds[:,2],SJH3Bounds[:,3].-SJH3Bounds[:,1]),label="Holling 3",color=3,fillalpha=0.2,linewidth=2,subplot=4)
Plots.scatter!(SJdata[!,2],SJdata[!,7],yerror=(SJdata[!,7].-SJdata[!,6],SJdata[!,8].-SJdata[!,7]),color=:black,subplot=4)



l2= @layout[
    a{0.01h}; b
]
RNFullPlot=Plots.plot(legendplot,RNPlot,size=[1000,1000],layout=l2,margin=10mm,left_margin=10mm,top_margin=0mm)

plot!(RNFullPlot, legend=false, xlabel=L"Larval food per capita, $\alpha$ (mg)", ylabel=L"Through-juvenile survival, $S_J$",xlabelfontsize=9,ylabelfontsize=9, framestyle = :box,xlims=[0,10],ylims=[0,0.8],linewidth=2,
inset = (2, bbox(0, 0.15, 0.5, 0.5, :bottom, :right)),subplot=3)
Plots.scatter!(aMat[1:Am-1,:][flipH3s],SJH3s[1:Am-1,:][flipH3s],color=3,label="Tipping points - Holling3",alpha=0.4,subplot=3)
Plots.plot!(class_as,SJH1Bounds[:,1],ribbon=(SJH1Bounds[:,1].-SJH1Bounds[:,2],SJH1Bounds[:,3].-SJH1Bounds[:,1]),label="Holling 1",color=1,fillalpha=0.2,linewidth=2,subplot=3)
Plots.plot!(class_as,SJH2Bounds[:,1],ribbon=(SJH2Bounds[:,1].-SJH2Bounds[:,2],SJH2Bounds[:,3].-SJH2Bounds[:,1]),label="Holling 2",color=2,fillalpha=0.2,linewidth=2,subplot=3)
Plots.plot!(class_as,SJH3Bounds[:,1],ribbon=(SJH3Bounds[:,1].-SJH3Bounds[:,2],SJH3Bounds[:,3].-SJH3Bounds[:,1]),label="Holling 3",color=3,fillalpha=0.2,linewidth=2,subplot=3)
Plots.scatter!(SJdata[!,2],SJdata[!,7],yerror=(SJdata[!,7].-SJdata[!,6],SJdata[!,8].-SJdata[!,7]),color=:black,subplot=3)


