#Code accompanying the paper, "Phenotypic plasticity as a route to population shifts via tipping points".
#Used to plot Figure 2 from the paper. 
#Requires Simulation.csv to be in file (see below) attained from running BlowflyRunner.jl with parameters v1=-4,v2=3.36,v3=-0.50.


using DelimitedFiles,Statistics,LinearAlgebra,ColorSchemes,LaTeXStrings
using Plots, Measures

#Parameters of the Run
file=string("390to450_speed0.001/");
(LeadIn,ChangeSpeed,Start,End)=readdlm(string(file,"ParamValues.csv"), '\t',Float64);

ChangeTime=abs(End-Start)/ChangeSpeed;
TotalTime=LeadIn+2*ChangeTime;

#args = 1fn, 2.1 TotalTime, 2.2 LeadIn, 2.3 ChangeTime, 3CutOff, 4.1 Start, 4.2 End,
args=[file,[TotalTime,LeadIn,ChangeTime],LeadIn,[Start,End]];

plotsize=[600,600];
dpis=300;
plotInsetsize=[400,400];
plotsizelegend=[1600,60];
Plots.scalefontsizes(); 
Plots.scalefontsizes(1.7);

#Parameters of the Plotting
timeLength=100;#Length of time that needs to be averaged
steadyStates=1;#Whether steady states for larval food/pop are plotted
errorBound=0.00;

plotArgs=[timeLength,steadyStates];

writedlm(string(file,"PlotParamValues.csv"), [timeLength,steadyStates]);

q1=3.95;
q2=6.9;
q3=-0.97;
q4=0.78;

#Function for linear changes to adult food with a lead in
function q(a)#This sets up the adult fecundity classes
    q= q1*(q2*log(a)+q3+0im)^q4;
    if (imag(q) ==0 && real(q)>0) #&& a>=0.14 
        return real(q);
    else
        return 0;
    end
end

v1=-4;
v2=3.36;
v3=-0.50;
function G(x)#logistic quadratic regression for G=ln(SP/(1-SP))
    v1+v2*log(x)+v3*log(x)^2;
end


function SP(a)#Equation for the survival probability classes
    KLexp=10000;
    Gs=exp(G(KLexp/(5*a)))/(1+exp(G(KLexp/(5*a))));#G(KL/5a)
    if isnan(Gs)
        return 0;
    #elseif a<1.4
    #    return 0;
    else
        return Gs;
    end
end

function aq(q)#Goes from q to the classes a
    exp(((q./q1).^(1/q4)-q3)./q2);
end


#Function for linear changes to adult food with a lead in
function KA(t,args)
    #args = 1fn, 2.1 TotalTime, 2.2 LeadIn, 2.3 ChangeTime, 3CutOff, 4.1 Start, 4.2 End,
    if t.<args[2][2]
        return args[4][1];
    elseif t.<args[2][2]+args[2][3]
        return args[4][1].+(t.-args[2][2])*(args[4][2]-args[4][1])/args[2][3];
    else
        return  args[4][2].+(t.-args[2][2]-args[2][3])*(args[4][1]-args[4][2])/args[2][3];
    end
end

function omega(a,abounds)#This sorts the Larvae into classes
    n=size(abounds)[1]-1
    omega=zeros(n)
    for i=1:n
        if abounds[i+1]>=a && a > abounds[i]
            omega[i]=1
            break
        end
    end
    return omega
end



Am=64*4;
qbounds=LinRange(0,60,Am+1);
abounds=aq.(qbounds);
class_as=(abounds[1:Am].+abounds[2:Am+1])/2;
qs=q.(class_as);
SPs=SP.(class_as);

as_min=3.3;
as_max=15.5;
WithinRange=as_min .<class_as.<as_max;
class_as_restricted=class_as[WithinRange];
class_as_restricted=class_as_restricted[1:6:length(class_as_restricted)]
qs_restricted=q.(class_as_restricted);
SPs_restricted=SP.(class_as_restricted);


#To add on steady states for food and larvae
taus=[0.6,5,10];#E,L,pi+J
tau=sum(taus);
deltas=[0.07,0.004];
SE=exp(-taus[1]*deltas[1]);
SL=exp(-taus[2]*deltas[2]);
SJ=exp(-0.0025*4.1)
deltaA=0.27;
KL=50000;
C1=deltaA*(1-SL)/(deltas[2]*SL*SJ);#First constant (For larvae)
C2=deltaA/(SE*SL*SJ);#Second constant (in log)

function KAfroma(a)
    -(KL*SP(a))/(C1*a*log(C1/(q(a)*SP(a))))
end


function AstabFunct(q,SP)
    Qratio=q.*SE*SL.*SP.*SJ
    b=1 .-log.(Qratio)
    stab=zeros(length(q)).+3;
    posbs=(b.^2 .-1 .>=0)
    stab[posbs][tau .<acos.(1 ./b[posbs])./(deltaA*sqrt.(b[posbs].^2 .-1))].=2
    stab[Qratio .<=exp(2)].=1;
    stab[Qratio .<1].=0;
    return(stab)
end

Stabs=AstabFunct(qs,SPs)[WithinRange];
stableStates=(Stabs.==1).||(Stabs.==2);
unstableStates=(Stabs.==0).||(Stabs.==3);

Stabs_restricted=AstabFunct(qs_restricted,SPs_restricted);
stableStates_restricted=(Stabs_restricted.==1).||(Stabs_restricted.==2);
unstableStates_restricted=(Stabs_restricted.==0).||(Stabs_restricted.==3);



stableN=sum(stableStates);
unstableN=sum(unstableStates);
restrictedN=stableN+unstableN;

stableLabel=Array{Union{Nothing, String}}(nothing, stableN);
#stableLabel[stableN]="Analytical: Stable equilibria";

unstableLabel=Array{Union{Nothing, String}}(nothing, unstableN);
#unstableLabel[1]="Analytical: Unstable equilibria";

Cols=cgrad(:grays, [stableN/restrictedN]);
#stableCols=palette([:cyan,:green],stableN);
#unstableCols=palette([:purple,:red],unstableN);

function Plotter(KA,args,plotArgs)
    println("Reading file...")
    data = readdlm(string(args[1],"Simulation.csv"), '\t', Float64);
    println("Read file.")

    n=size(data)[1];#The total number of points
    width=size(data)[2];#The total number of variables

    ATot=sum(data[:,4:width],dims=2);

    Aq=(data[:,4:width]*qs)./ATot;
    ASP=(data[:,4:width]*SPs)./ATot;



    DecreasingClassStart=1;
    IncreasingClassStart=Int64(floor((args[2][3])/plotArgs[1]));

    
    splitN=Int64(floor((args[2][1]-args[3])/plotArgs[1]));#The number of classes we will have
    splitLength=Int64(floor(n/splitN));#The length of the time splits

    #Initialise the variables
    Averages=zeros(splitN,width);
    Maximums=zeros(splitN,width);
    Minimums=zeros(splitN,width);
    ATotmean=zeros(splitN);
    ATotmax=zeros(splitN);
    ATotmin=zeros(splitN);
    qmean=zeros(splitN);
    qmax=zeros(splitN);
    qmin=zeros(splitN);
    SPmean=zeros(splitN);
    SPmax=zeros(splitN);
    SPmin=zeros(splitN);
    println("Starting splits...")
    for i=1:splitN
        #Computes the average across the size splitLength
        for j=1:width
            Averages[i,j]=quantile(data[(i-1)*splitLength+1:(i)*splitLength,j],0.5);
            Maximums[i,j]=quantile(data[(i-1)*splitLength+1:(i)*splitLength,j],1-errorBound);
            Minimums[i,j]=quantile(data[(i-1)*splitLength+1:(i)*splitLength,j],errorBound);
        end
        ATotmean[i]=quantile(ATot[(i-1)*splitLength+1:(i)*splitLength],0.5);
        ATotmax[i]=quantile(ATot[(i-1)*splitLength+1:(i)*splitLength],1-errorBound);
        ATotmin[i]=quantile(ATot[(i-1)*splitLength+1:(i)*splitLength],errorBound);
        qmean[i]=quantile(Aq[(i-1)*splitLength+1:(i)*splitLength],0.5);
        qmax[i]=quantile(Aq[(i-1)*splitLength+1:(i)*splitLength],1-errorBound);
        qmin[i]=quantile(Aq[(i-1)*splitLength+1:(i)*splitLength],errorBound);
        SPmean[i]=quantile(ASP[(i-1)*splitLength+1:(i)*splitLength],0.5);
        SPmax[i]=quantile(ASP[(i-1)*splitLength+1:(i)*splitLength],1-errorBound);
        SPmin[i]=quantile(ASP[(i-1)*splitLength+1:(i)*splitLength],errorBound);
    end

    KAs=zeros(splitN);
    for i=1:splitN
        KAs[i]=KA(Averages[i,1],args);
    end

    netqmean=qmean.*exp.(-ATotmean./KAs);
    totalqmean=qmean.*exp.(-ATotmean./KAs).*ATotmean;

    PopMatRate=deltaA.*ATotmean;#deltas[2].*Averages[:,3].*SL.*SP.(Averages[:,2])./(1-SE);
    PopRepRate=sum((Averages[:,4:width]*qs),dims=2).^exp.(-ATotmean./KAs);
    KAoverA=KAs./ATotmean;


    #xtickfontsize=24,ytickfontsize=24,xguidefontsize=20,yguidefontsize=20,legendfontsize=12
    aplot     =plot(linewidth=4,size=plotsize,margin=10mm,left_margin=10mm,dpi=dpis,xguidefontsize=16,yguidefontsize=16)
    Lplot     =plot(linewidth=4,size=plotsize,margin=10mm,left_margin=15mm,dpi=dpis,xguidefontsize=16,yguidefontsize=16)
    Atotplot  =plot(linewidth=4,size=plotsize,margin=10mm,left_margin=15mm,dpi=dpis,xguidefontsize=16,yguidefontsize=16)
    Afoodplot =plot(linewidth=4,size=plotsize,margin=10mm,left_margin=15mm,dpi=dpis,xguidefontsize=16,yguidefontsize=16)
    indexplot =plot(linewidth=4,size=plotsize,margin=10mm,left_margin=15mm,dpi=dpis,xguidefontsize=16,yguidefontsize=16)
    qplot     =plot(linewidth=4,size=plotsize,margin=10mm,left_margin=15mm,dpi=dpis,xguidefontsize=16,yguidefontsize=16)
    SPplot    =plot(linewidth=4,size=plotsize,margin=10mm,left_margin=15mm,dpi=dpis,xguidefontsize=16,yguidefontsize=16)
    PopMatplot=plot(linewidth=4,size=plotsize,margin=10mm,left_margin=15mm,dpi=dpis,xguidefontsize=16,yguidefontsize=16)
    PopRepplot=plot(linewidth=4,size=plotsize,margin=10mm,left_margin=15mm,dpi=dpis,xguidefontsize=16,yguidefontsize=16)
    qRNplot   =plot(linewidth=4,size=plotsize,margin=10mm,left_margin=15mm,dpi=dpis,xguidefontsize=16,yguidefontsize=16)
    SPRNplot  =plot(linewidth=4,size=plotsize,margin=10mm,left_margin=15mm,dpi=dpis,xguidefontsize=16,yguidefontsize=16)
    
    if plotArgs[2]==1
        for i =1:stableN
            #For each class look at the stability
            indexi=(1:Am)[WithinRange][stableStates][i]
            As=-KAs.*log.(C2./(qs[indexi].*SPs[indexi]))
            Ls=C1*As./SPs[indexi]
            as=KL./Ls

            possible=Vector(abounds[indexi].<as.<abounds[indexi+1])
            iPossible=(zeros(splitN).+indexi).*possible
            iPossible[iPossible.==0].=NaN
            as=as.*possible
            as[as.==0].=NaN
            Ls=Ls.*possible
            Ls[Ls.==0].=NaN
            As=As.*possible
            As[As.==0].=NaN


            plot!(aplot,    KAs,as,       color=Cols[i/restrictedN],label=stableLabel[i])
            plot!(Lplot,    KAs,Ls,       color=Cols[i/restrictedN],label=stableLabel[i])
            plot!(indexplot,KAs,iPossible,color=Cols[i/restrictedN],label=stableLabel[i])
            plot!(Atotplot, KAs,As,       color=Cols[i/restrictedN],label=stableLabel[i])
        end

        for i =1:unstableN
            #For each class look at the stability
            indexi=(1:Am)[WithinRange][unstableStates][i]
            As=-KAs.*log.(C2./(qs[indexi].*SPs[indexi]))
            Ls=C1*As./(SPs[indexi])
            as=KL./Ls

            possible=Vector(abounds[indexi].<as.<abounds[indexi+1])
            iPossible=(zeros(splitN).+indexi).*possible
            iPossible[iPossible.==0].=NaN
            as=as.*possible
            as[as.==0].=NaN
            Ls=Ls.*possible
            Ls[Ls.==0].=NaN
            As=As.*possible
            As[As.==0].=NaN

            plot!(aplot,    KAs,as,       color=Cols[(stableN+i)/restrictedN],label=unstableLabel[i],linestyle=:dot)
            plot!(Lplot,    KAs,Ls,       color=Cols[(stableN+i)/restrictedN],label=unstableLabel[i],linestyle=:dot)
            plot!(indexplot,KAs,iPossible,color=Cols[(stableN+i)/restrictedN],label=unstableLabel[i],linestyle=:dot)
            plot!(Atotplot, KAs,As,       color=Cols[(stableN+i)/restrictedN],label=unstableLabel[i],linestyle=:dot)
        end

    end

    println("Starting plotting...")
    plot!(aplot,KAs[IncreasingClassStart:splitN],Averages[IncreasingClassStart:splitN,2],ribbon=(Averages[IncreasingClassStart:splitN,2]-Minimums[IncreasingClassStart:splitN,2],Maximums[IncreasingClassStart:splitN,2]-Averages[IncreasingClassStart:splitN,2]),color=1,linewidth=3,xlabel=L"Adult food supply, $K_A$ (mg)",ylabel=L"Average larval food per capita, $\alpha$ (mg)",label=false);
    plot!(aplot,KAs[DecreasingClassStart:IncreasingClassStart],Averages[DecreasingClassStart:IncreasingClassStart,2],ribbon=(Averages[DecreasingClassStart:IncreasingClassStart,2]-Minimums[DecreasingClassStart:IncreasingClassStart,2],Maximums[DecreasingClassStart:IncreasingClassStart,2]-Averages[DecreasingClassStart:IncreasingClassStart,2]),color=3,linewidth=3,label=false);
    
    plot!(Afoodplot,KAs[IncreasingClassStart:splitN],KAoverA[IncreasingClassStart:splitN],color=:red,linewidth=3,xlabel=L"Adult food supply, $K_A$ (mg)",ylabel=L"Average adult food per capita, $K_A/A_{tot}$ (mg)",label=false);
    plot!(Afoodplot,KAs[DecreasingClassStart:IncreasingClassStart],KAoverA[DecreasingClassStart:IncreasingClassStart],color=:red,linewidth=3,label=false);
    
    foodPlot=plot(aplot,Afoodplot);

    plot!(Lplot,KAs[IncreasingClassStart:splitN],Averages[IncreasingClassStart:splitN,3],ribbon=(Averages[IncreasingClassStart:splitN,3]-Minimums[IncreasingClassStart:splitN,3],Maximums[IncreasingClassStart:splitN,3]-Averages[IncreasingClassStart:splitN,3]),color=:red,linewidth=3,xlabel=L"Adult food supply, $K_A$ (mg)",ylabel=L"Larval population, $L$" ,label=false);
    plot!(Lplot,KAs[DecreasingClassStart:IncreasingClassStart],Averages[DecreasingClassStart:IncreasingClassStart,3],ribbon=(Averages[DecreasingClassStart:IncreasingClassStart,3]-Minimums[DecreasingClassStart:IncreasingClassStart,3],Maximums[DecreasingClassStart:IncreasingClassStart,3]-Averages[DecreasingClassStart:IncreasingClassStart,3]),color=:red,linewidth=3,label=false);
    
    #To get rid of the zero vectors
    RemovedZeros=Averages[:,4:width];
    RemovedZeros[RemovedZeros.<1].=NaN;

    plot!(Atotplot,KAs[IncreasingClassStart:splitN],ATotmean[IncreasingClassStart:splitN],ribbon = (ATotmean[IncreasingClassStart:splitN]-ATotmin[IncreasingClassStart:splitN],ATotmax[IncreasingClassStart:splitN]-ATotmean[IncreasingClassStart:splitN]),color=:red,linewidth=3,xlabel=L"Adult food supply, $K_A$ (mg)",ylabel=L"Total adult population, $A_{tot}$",label=false);
    plot!(Atotplot,KAs[DecreasingClassStart:IncreasingClassStart],ATotmean[DecreasingClassStart:IncreasingClassStart],ribbon = (ATotmean[DecreasingClassStart:IncreasingClassStart]-ATotmin[DecreasingClassStart:IncreasingClassStart],ATotmax[DecreasingClassStart:IncreasingClassStart]-ATotmean[DecreasingClassStart:IncreasingClassStart]),color=:red,linewidth=3,label=false)

    RemoveZeros=Averages[:,4:width];
    RemoveZeros[RemoveZeros.<1].=0;
    RemoveZeros[RemoveZeros.>=1].=1;
    indexs=RemoveZeros*diagm(Vector(1:Am));
    indexs[indexs.<1].=NaN;

    dLAB = hcat("Decreasing",fill("",1,Am-1));
    iLAB = hcat("Increasing",fill("",1,Am-1));
    #plot!(twinx(),KAs,indexs,ylims=[50,100],ylabel="Adult class index",color="orange",xticks=:none,label=LAB,legend=:right);
    plot!(indexplot,KAs[IncreasingClassStart:splitN],indexs[IncreasingClassStart:splitN,:],xlabel=L"Adult food supply, $K_A$ (mg)",ylabel="Adult class index",color=:red,linewidth=3,label=dLAB);
    plot!(indexplot,KAs[DecreasingClassStart:IncreasingClassStart],indexs[DecreasingClassStart:IncreasingClassStart,:],color=:red,linewidth=3,label=iLAB);

    plot!(qplot,KAs[IncreasingClassStart:splitN],qmean[IncreasingClassStart:splitN],ribbon = (qmean[IncreasingClassStart:splitN]-qmin[IncreasingClassStart:splitN],qmax[IncreasingClassStart:splitN]-qmean[IncreasingClassStart:splitN]),color=:black,linewidth=3,xlabel=L"Adult food supply, $K_A$ (mg) ",ylabel=L"Maximum adult fecundity, $q$",label=false,ylims=[15,120]);
    plot!(qplot,KAs[DecreasingClassStart:IncreasingClassStart],qmean[DecreasingClassStart:IncreasingClassStart],ribbon = (qmean[DecreasingClassStart:IncreasingClassStart]-qmin[DecreasingClassStart:IncreasingClassStart],qmax[DecreasingClassStart:IncreasingClassStart]-qmean[DecreasingClassStart:IncreasingClassStart]),color=:black,linewidth=3,label=false)


    #plot!(qplot,KAs[IncreasingClassStart:splitN],netqmean[IncreasingClassStart:splitN],color=:red,linewidth=3,linestyle=:dot,label=L"$qe^{-A/K_A}$ (indivual)");
    #plot!(qplot,KAs[DecreasingClassStart:IncreasingClassStart],netqmean[DecreasingClassStart:IncreasingClassStart],color=:red,linewidth=3,linestyle=:dot,label=L"$qe^{-A/K_A}$ (individual)")
    
    #Adds in fake plots for labels
    plot!(KAs[IncreasingClassStart:splitN], NaN.*KAs[IncreasingClassStart:splitN], label = "Maximum adult fecundity", linecolor=:black, grid=false, legend=false) 
    plot!(KAs[IncreasingClassStart:splitN], NaN.*KAs[IncreasingClassStart:splitN], label = "Total population fecundity", linecolor=:red, grid=false) 

    tx=twinx()
    plot!(tx,KAs[IncreasingClassStart:splitN],totalqmean[IncreasingClassStart:splitN],color=:royalblue,linewidth=4,ylabel="Total population fecundity",ylims=[0,3500],yguidefontsize=16,legend=:bottomright,label=false);
    plot!(tx,KAs[DecreasingClassStart:IncreasingClassStart],totalqmean[DecreasingClassStart:IncreasingClassStart],color=:royalblue,linewidth=4,label=false)
    
    plot!(tx,y_foreground_color_axis=:royalblue)
    plot!(tx,y_foreground_color_text=:royalblue)
    plot!(tx,y_foreground_color_border=:royalblue)

    plot!(SPplot,KAs[IncreasingClassStart:splitN],SPmean[IncreasingClassStart:splitN],ribbon = (SPmean[IncreasingClassStart:splitN]-SPmin[IncreasingClassStart:splitN],SPmax[IncreasingClassStart:splitN]-SPmean[IncreasingClassStart:splitN]),color=:black,xlabel=L"Adult food supply, $K_A$ (mg)",ylabel=L"Through-pupal survival, $S_P$",ylims=[0,0.55], legend=false,linewidth=3);
    plot!(SPplot,KAs[DecreasingClassStart:IncreasingClassStart],SPmean[DecreasingClassStart:IncreasingClassStart],ribbon = (SPmean[DecreasingClassStart:IncreasingClassStart]-SPmin[DecreasingClassStart:IncreasingClassStart],SPmax[DecreasingClassStart:IncreasingClassStart]-SPmean[DecreasingClassStart:IncreasingClassStart]),color=:black,linewidth=3)



    savefig(foodPlot,string(args[1],"Food",args[4][1],"to",args[4][2],"_speed",args[2][3],".png"));
    savefig(Lplot,string(args[1],"Larval",args[4][1],"to",args[4][2],"_speed",args[2][3],".png"));
    savefig(Atotplot,string(args[1],"AdultTotal",args[4][1],"to",args[4][2],"_speed",args[2][3],".png"));


    savefig(indexplot,string(args[1],"Indexes",args[4][1],"to",args[4][2],"_speed",args[2][3],".png"));
    savefig(qplot,string(args[1],"Fecundity",args[4][1],"to",args[4][2],"_speed",args[2][3],".png"));
    savefig(SPplot,string(args[1],"Survival",args[4][1],"to",args[4][2],"_speed",args[2][3],".png"));


    println("Saving figure...")


    plot!(SPRNplot,class_as,SPs,xlabel=L"Larval food per capita, $\alpha$ (mg)",ylabel=L"Through-pupal survival, $S_P$",xscale=:log2,xticks=(2 .^(0:7), 2 .^(0:7)),color=:black,legend=:bottomright,colorbar=false,label=false,yguidefontsize=20)
    scatter!(class_as_restricted[stableStates_restricted],SPs_restricted[stableStates_restricted],marker_z=class_as_restricted[stableStates_restricted],markercolor=Cols,markersize=9,markerstrokewidth=0,markershape=:square,label="Analytical: Stable equilibria")
    scatter!(class_as_restricted[unstableStates_restricted],SPs_restricted[unstableStates_restricted],marker_z=class_as_restricted[unstableStates_restricted],markercolor=Cols,markersize=9,markerstrokewidth=0,markershape=:utriangle,label="Analytical: Unstable equilibria")
    
    plot!(qRNplot,class_as,qs,xlabel=L"Larval food per capita, $\alpha$ (mg)",ylabel=L"Maximum adult fecundity, $q$",xscale=:log2,xticks=(2 .^(0:7), 2 .^(0:7)),color=:black,legend=:bottomright,colorbar=false,label=false,yguidefontsize=20)
    scatter!(class_as_restricted[stableStates_restricted],qs_restricted[stableStates_restricted],marker_z=class_as_restricted[stableStates_restricted],markercolor=Cols,markersize=9,markerstrokewidth=0,markershape=:square,label="Analytical: Stable equilibria")
    scatter!(class_as_restricted[unstableStates_restricted],qs_restricted[unstableStates_restricted],marker_z=class_as_restricted[unstableStates_restricted],markercolor=Cols,markersize=9,markerstrokewidth=0,markershape=:utriangle,label="Analytical: Unstable equilibria")
    

    #plotfakelegend
    legendplot=plot(linewidth=3,size=plotsizelegend,margin=0mm,dpi=dpis,legendfontsize=20, xlims=(1,1.1),legend_column=-1, framestyle=:none)
    plot!(legendplot,[0,0], [0,1],label="Numerics: Decreasing food",color=1,linewidth=3)
    plot!(legendplot,[0,0], [0,1],label="Numerics: Increasing food",color=3,linewidth=3)
    plot!(legendplot,[0,0], [0,1],label="Analytical: Stable equilibria",color=Cols[0.5],linewidth=3)
    plot!(legendplot,[0,0], [0,1],label="Analytical: Unstable equilibria",color=Cols[0.5],linewidth=3,linestyle=:dot)
    savefig(legendplot,string(args[1],"Legend",args[4][1],"to",args[4][2],"_speed",args[2][3],".png"))




    #Colorbar
    cmap = cgrad(:thermal)
    cbar = heatmap(rand(2,2),
        clims=(as_min,as_max), framestyle=:none, c=cmap,
        cbar=true, cbar_title=L"Larval food per capita, $\alpha$ (mg)",
        lims=(-1,0), thickness_scaling = 1,size=[50,2000]
    )
    savefig(cbar,string(args[1],"Colorbar",args[4][1],"to",args[4][2],"_speed",args[2][3],".png"))

    l=@layout[
     a b; c d
    ];
    
    combinedplot=plot(Lplot,Atotplot,SPplot,qplot,layout=l,size=[1200,900],dpi=600,margin=8mm)

    plot!(combinedplot, ylims=[2500,15000], yformatter = :plain,subplot=1)
    plot!(combinedplot, ylims=[500,1750],subplot=2)
    plot!(combinedplot, xlims=[390,450])

    annotate!(combinedplot, 283,  261, text("a", 20),subplot=4)
    annotate!(combinedplot, 368,  261, text("b", 20),subplot=4)
    annotate!(combinedplot, 283,  125, text("c", 20),subplot=4)
    annotate!(combinedplot, 368,  125, text("d", 20),subplot=4)

    plot!(combinedplot,margin=2mm)
    plot!(combinedplot,grid=false)

    savefig(combinedplot,string(args[1],"Full",args[4][1],"to",args[4][2],"_speed",args[2][3],".png"))


    #Feedback plots


    plot!(PopMatplot,KAs[IncreasingClassStart:splitN],PopMatRate[IncreasingClassStart:splitN],color=1,xlabel=L"Adult food supply, $K_A$ (mg)",ylabel=L"Population maturation rate, $sum(q_kA_k)S_ES_LS_J(\alpha)$", legend=false,linewidth=3);
    plot!(PopMatplot,KAs[DecreasingClassStart:IncreasingClassStart],PopMatRate[DecreasingClassStart:IncreasingClassStart],color=3,linewidth=3)

    plot!(PopRepplot,KAs[IncreasingClassStart:splitN],PopRepRate[IncreasingClassStart:splitN],color=1,xlabel=L"Adult food supply, $K_A$ (mg)",ylabel=L"Population reproducation rate, $sum(q_kA_k)$", legend=false,linewidth=3);
    plot!(PopRepplot,KAs[DecreasingClassStart:IncreasingClassStart],PopRepRate[DecreasingClassStart:IncreasingClassStart],color=3,linewidth=3)



    FeedbackPlot=plot(PopMatplot,PopRepplot)
    savefig(string(args[1],"Feedbacks",args[4][1],"to",args[4][2],"_speed",args[2][3],".png"));

end

Plotter(KA,args,plotArgs)

