#Code accompanying the paper, "Phenotypic plasticity as a route to population shifts via tipping points".
#A stage/phenotypically structured delay-differential equation model that predicts how changes in adult food supply 
#affect the population dynamics of Lucilia cuprina. Designed to be used with BlowflyRunner.jl to change adult food supply
#and BlowflyPlotter.jl to plot these simulations. With parameters v1=-4,v2=3.36,v3=-0.50.

module Blowfly

#Packages
using Logging: global_logger
using TerminalLoggers: TerminalLogger
using ProgressMeter
global_logger(TerminalLogger())
using Statistics, Plots,StatsBase, FFTW,Dates,DelimitedFiles,Interpolations,DifferentialEquations

function q(a)#Sets up the adult fecundity classes
    q= 3.95*(6.90*log(a)-0.97+0im)^0.78;
    if (imag(q) ==0 && real(q)>0) #&& a>=0.14 
        return real(q);
    else
        return 0;
    end
end
v1=-4;
v2=3.36;
v3=-0.50;
function G(x)#logistic quadratic regression for G=ln(SJ/(1-SJ))
    v1+v2*log(x)+v3*log(x)^2;
end

function SJ(a)#Equation for the survival probability classes
    KL=10000;
    Gs=exp(-0.0025*4.1)*exp(G(KL/(5*a)))/(1+exp(G(KL/(5*a))));#G(KL/5a)
    if isnan(Gs)
        return 0;
    #elseif a<1.4
    #    return 0;
    else
        return Gs;
    end
end

function aq(q)#Goes from adult fecundity (q) to the classes for larval food (a)
    exp(((q./3.95).^(1/0.78)+0.97)./6.90);
end

function BlowflyModel(dN, N, h, p, t)#The DDE to be run N=a,L,A1,...,Am
    #p=[m,qs,ps,SJs,taus,deltas,SEL,KA,tspan];
    
    ahistPJ=h(p,t-p[5][3]; idxs=1)

    LhistL=h(p,t-p[5][2]; idxs=2)

    AhistE=h(p,t-p[5][1]; idxs=3:p[1]+2)
    AhistETotal=sum(AhistE)

    AhistEL=h(p,t-p[5][1]-p[5][2]; idxs=3:p[1]+2)
    AhistELTotal=sum(AhistEL)

    AhistELPJ=h(p,t-p[5][1]-p[5][2]-p[5][3]; idxs=3:p[1]+2)
    AhistELPJTotal=sum(AhistELPJ)

    KAhistE=   p[8](t-p[5][1]                ,p[9]);
    KAhistEL=  p[8](t-p[5][1]-p[5][2]        ,p[9]);
    KAhistELPJ=p[8](t-p[5][1]-p[5][2]-p[5][3],p[9]);

    growthE=sum(p[2].*AhistE)
    growthEL=sum(p[2].*AhistEL)
    growthELPJ=sum(p[2].*AhistELPJ)

    RL=(growthE*exp(-AhistETotal/KAhistE)+inoc(t,p[5][1]))*p[7][1]
    RLhistL=(growthEL*exp(-AhistELTotal/KAhistEL)+inoc(t,p[5][1]+p[5][2]))*p[7][1]*p[7][2]

    RA=omega(ahistPJ,p[3]).*p[4]*((growthELPJ*exp(-AhistELPJTotal/KAhistELPJ)+inoc(t,p[5][1]+p[5][2]+p[5][3]))*p[7][1]*p[7][2])

    dN[1]=-N[1]^2*(N[2]-LhistL)/(250000)#/Lmax*tauL
    dN[2]=RL-RLhistL-p[6][2]*N[2]
    dN[3:p[1]+2]=RA.-(0.27.*N[3:p[1]+2])
end

function h(p, t; idxs::Union{Nothing,Int, UnitRange{Int}} = nothing)#History function
    t ≤ 0 || error("history function is only implemented for t ≤ 0")
    if idxs === nothing
        N=zeros(2+p[1])
        N[1]=50000/9500#KL/start number
        N[2]=9500#start number
        return N
    elseif idxs == 1
        50000/9500
    elseif idxs ==2
        9500
    elseif idxs == 3:p[1]+2
        zeros(p[1])
    else
        println("Error indexes out of range")
    end
end

function inoc(t,a)#Initial inoculation
    if t<=a+2 && t>=a-0.2
        return 1
    else
        return 0
    end
end

function omega(a,abounds)#This sorts the Larvae into classes
    n=size(abounds)[1]-1
    omega=zeros(n)
    for i=1:n
        if abounds[i+1]>=a && a > abounds[i]
            omega[i]=1
            break
        end
    end
    return omega
end

function Run(KA,args::Vector{Any})#Runs the problem with i giving the Adult food to give
    #args = 1fn, 2.1 TotalTime, 2.2 LeadIn, 2.3 ChangeTime, 3CutOff, 4.1 Start, 4.2 End,
    
    println("Running blowfly model")
    taus=[0.6,5,10]#E,L,pi+J
    deltas=[0.07,0.004]
    SEL=exp.(-taus[1:2].*deltas)
    lags=[taus[1],taus[2],taus[3],taus[1]+taus[2],taus[1]+taus[2]+taus[3]]


    m=64*4;

    qbounds=LinRange(0,60,m+1)
    aqbounds=aq.(qbounds)
    aqs=(aqbounds[1:m].+aqbounds[2:m+1])/2
    q2s=q.(aqs)
    SJqs=SJ.(aqs)

    p=[m,q2s,aqbounds,SJqs,taus,deltas,SEL,KA,args];

    alg = MethodOfSteps(RK4());#An order 4 level of solving the DDEs that are undeclared
    prob=DDEProblem(BlowflyModel,h(p,0),h,args[2][1],p);#Constant_lags=lags

    println("Starting DDE")
    LongSol=solve(prob,alg,abstol=1e-8,reltol=1e-8,maxiters=1e10, saveat = args[5], progress=true)#Actually does the solving part #, isoutofdomain = (N,p,t)->any(x->x<0,N), # 
    println("Solved DDE")

    #Wanting to avoid the lead in data
    CutOff=Int64(floor(args[3]/savepoints));

    println("Starting Writing")
    writedlm(args[1],hcat(LongSol.t[CutOff+1],transpose(LongSol.u[CutOff+1])));
    
    open(args[1], "a") do io
    @showprogress for i=CutOff+2:length(LongSol.t)
        writedlm(io, hcat(LongSol.t[i],transpose(LongSol.u[i])));
    end
    end
    println("Finished Writing")
    return LongSol
end 


end
