#Code accompanying the paper, "Phenotypic plasticity as a route to population shifts via tipping points".
#Produces two variable parameter sweeps that give Figure 3 and Figures S.3.4-S.3.14
#Creates a contour plot of the relative change in the hysteresis plots as two reaction norm variables are varied
#Then plots a sequence of reaction norms associated with a sample of these variables


using Plots, Measures; pythonplot()
using LaTeXStrings

Plots.scalefontsizes()
Plots.scalefontsizes(1.7)

xvar=[2,2+1]
yvar=[2,3+1]
RNstartVal=[-.2,-.5];
RNendVal=-RNstartVal;
addpoint=true;
insertHeight=false;

#region Set-up functions
taus=[0.6,5,4.1,5.9];#E,L,pi+J
tau=sum(taus);
deltas=[0.07,0.004,0.0025];
SE=exp(-taus[1]*deltas[1]);
SL=exp(-taus[2]*deltas[2]);
SJ=exp(-taus[3]*deltas[3]);
deltaA=0.27;

KL=50000

C1=deltaA*(1-SL)/(deltas[2]*SL)#First constant
C2=deltaA/(SE*SL)


function q(a,p)#This sets up the adult fecundity classes
    q=p[1][1]*(p[1][2]*log(a)+p[1][3]+0im)^p[1][4];
    if (imag(q) ==0 && real(q)>0) #&& a>=0.14 
        return real(q);
    else
        return 0;
    end
end

function qprime(a,p)
    q= (p[1][2]/a)*p[1][4]*p[1][1]*(p[1][2]*log(a)+p[1][3]+0im)^(p[1][4]-1)
    if imag(q) ==0 #&& a>=0.14 
        return real(q);
    else
        return 0;
    end
end

function qvect(a,p)
    q=real.(p[1][1].*(p[1][2].*log.(a).+p[1][3].+0im).^p[1][4]);
    q[q.<0].=0;
    return q;
end

function G(x,p)#logistic quadratic regression for G=ln(SP/(1-SP))
    p[2][2].+ p[2][3]*log.(x).+p[2][4]*log.(x).^2
end

function Gprime(x,p)
    p[2][3]./x.+p[2][4]*2*log.(x)./x
end

function SPvect(a,p)#Equation for the survival probability classes
    Gs=exp.(G(p[2][1]./(5*a),p))./(1 .+exp.(G(p[2][1]./(5*a),p)));#G(KL/5a)
    Gs[isnan.(Gs)].=0
    return Gs
end

function SP(a,p)#Equation for the survival probability classes
    Gs=exp.(G(p[2][1]./(5*a),p))./(1 .+exp.(G(p[2][1]./(5*a),p)));#G(KL/5a)
    if isnan.(Gs)
        Gs=0
    end
    return Gs
end

function SPprime(a,p)
    Gs=(-(p[2][1]/5)./a.^2 .*Gprime(p[2][1]./(5*a),p).*exp.(G(p[2][1]./(5*a),p)))./((1 .+exp.(G(p[2][1]./(5*a),p))).^2)
    if isnan(Gs)
        return 0;
    else
        return Gs;
    end
end

function SPprimeVec(a,p)
    Gs=(-(p[2][1]/5)./a.^2 .*Gprime(p[2][1]./(5*a),p).*exp.(G(p[2][1]./(5*a),p)))./((1 .+exp.(G(p[2][1]./(5*a),p))).^2)
    Gs[isnan.(Gs)].=0
    return Gs
end

function logDR(a,p)
    log(deltaA/(q(a,p)*SE*SL*SJ*SP(a,p)))
end

function logDRprime(a,p)
    -(qprime(a,p)/q(a,p)+SPprime(a,p)/SP(a,p))
end

function aq(q,p)#Goes from q to the classes a
    exp(((q/p[1][1])^(1/p[1][4])-p[1][3])/p[1][2]);
end

function KAfroma(a,p)
    -(KL*SP(a,p))/(C1*a*log(deltaA/(q(a,p)*SE*SL*SP(a,p))))
end

function A(a,p)
    -KAfroma(a,p)*logDR(a,p)
end


q1=3.95;
q2=6.90;
q3=-0.97;
q4=0.78;

KLexp=10000;
SP1=-4;
SP1SE=1.49;
SP2=3.36;
SP2SE=0.59;
SP3=-0.44;
SP3SE=0.06;

p_mean=[[q1,q2,q3,q4],[KLexp,SP1,SP2,SP3]]
pSE=[SP1SE,SP2SE,SP3SE]
pnames=[["h1","h2","h3","h4"],["KLexp","v1","v2","v3"]]
plabelnames=[[L"$h_1$",L"$h_2$",L"$h_3$",L"$h_4$"],["KLexp",L"$v_1$",L"$v_2$",L"$v_3$"]]

x=p_mean[xvar[1]][xvar[2]];
y=p_mean[yvar[1]][yvar[2]];
println(string(pnames[xvar[1]][xvar[2]],",",pnames[yvar[1]][yvar[2]]))

astart=exp(-q3/q2)
aend=exp(1/q2*((59/q1)^(1/q4)-q3))/3

an=1001
as=Vector(LinRange(astart,aend,an))


#endregion
#region Bifurcation analysis

ChangeRange=0.5;
simNumber=201;

xs=LinRange(x*(1-ChangeRange),x*(1+ChangeRange),simNumber);
xsRatio=(xs.-x)./x;
ys=LinRange(y*(1-ChangeRange),y*(1+ChangeRange),simNumber);
ysRatio=(ys.-y)./y;





isBifurcation=zeros(simNumber,simNumber);
BifurcationsKADistance=zeros(simNumber,simNumber);
BifurcationsADistance=zeros(simNumber,simNumber);
BifurcationsARelDistance=zeros(simNumber,simNumber);
BifurcationsKAMid=zeros(simNumber,simNumber);
BifurcationsaDistance=zeros(simNumber,simNumber);
for i=1:simNumber
    for j=1:simNumber
        p=[[q1,q2,q3,q4],[KLexp,SP1,SP2,SP3]];
        p[xvar[1]][xvar[2]]=xs[i]
        p[yvar[1]][yvar[2]]=ys[j]
        grads=zeros(an)
        for k=1:an
            grads[k]=as[k]*((SPprime(as[k],p)/SP(as[k],p))-(logDRprime(as[k],p)/logDR(as[k],p)))-1;
        end
        gradflips=grads[1:an-1].*grads[2:an];
        for k in (1:an-1)[gradflips.<0]
            if KAfroma(as[k],p)<0
                gradflips[k]=NaN
            elseif  KAfroma(as[k],p)>1000
                gradflips[k]=NaN
            end
        end 
        if sum(gradflips.<0)==2
            isBifurcation[i,j]=1

            aflips=[as[1:an-1][gradflips.<0][1],as[1:an-1][gradflips.<0][2]]

            As=A.(aflips,Ref(p))
            BifurcationsaDistance[i,j]=aflips[2]-aflips[1]
            BifurcationsKADistance[i,j]=KAfroma(aflips[2],p)-KAfroma(aflips[1],p)
            BifurcationsADistance[i,j]=As[2]-As[1]
            BifurcationsARelDistance[i,j]=(As[2]-As[1])/As[2]
            BifurcationsKAMid[i,j]=(KAfroma(aflips[2],p)+KAfroma(aflips[1],p))/2
        end
    end
end

BifurcationsKADistance[BifurcationsKADistance.==0].=NaN;
BifurcationsADistance[BifurcationsADistance.==0].=NaN;
maxdrop=maximum(BifurcationsARelDistance);
BifurcationsARelDistance[BifurcationsARelDistance.==0].=NaN;



ticks=(Vector(-.5:0.1:0.5),["-50%","-40%","-30%","-20%","-10%","0%","10%","20%","30%","40%","50%"])
ticks2=(Vector(-.2:0.1:0.2),["-20%","-10%","0%","10%","20%"])

rm=3mm
BifPlot=Plots.contour(xsRatio,ysRatio,transpose(BifurcationsKADistance),
 color=:algae,fill=true,colorbar=true, 
 colorbar_title =L"Difference in adult food, $K_A$, between tipping points (mg)",
 xlabel=latexstring("Percentage difference in ",plabelnames[xvar[1]][xvar[2]]),
 ylabel=latexstring("Percentage difference in ",plabelnames[yvar[1]][yvar[2]]),
 size=[1200,1200],
 legend=:bottomright,
 xticks=ticks,
 yticks=ticks,
 right_margin=rm
)

BifAPlot=Plots.contour(xsRatio,ysRatio,transpose(BifurcationsADistance),
 color=:algae,fill=true,colorbar=true, 
 colorbar_title =L"Difference in total adult population, $A_{tot}$ (i.e. environmental state difference $\Delta E$)",
 xlabel=latexstring("Percentage difference in ",plabelnames[xvar[1]][xvar[2]]),
 ylabel=latexstring("Percentage difference in ",plabelnames[yvar[1]][yvar[2]]),
 size=[1200,1200],
 legend=:bottomright,
 xticks=ticks,
 yticks=ticks,
 right_margin=rm
)

RelDropTicks=(Vector(0:0.1:ceil(maxdrop*10)/10),string.(round.(Vector(0:0.1:ceil(maxdrop*10)/10),digits=1).*100,"%"))


BifARelPlot=Plots.contour(xsRatio,ysRatio,transpose(BifurcationsARelDistance),
 color=cgrad(:algae),fill=true,colorbar=true,
 colorbar_title ="Relative difference in adult abundance between alternative stables states",
 xlabel=latexstring("Percentage difference in ",plabelnames[xvar[1]][xvar[2]]," from mean"),
 ylabel=latexstring("Percentage difference in ",plabelnames[yvar[1]][yvar[2]]," from mean"),
 size=[1200,1200],
 legend=:bottomright,
 xticks=ticks,
 yticks=ticks,
 colorbar_ticks=RelDropTicks,
 levels=Vector(0:0.05:ceil(maxdrop*10)/10),
 right_margin=rm
)

if (yvar[1]==2)&(xvar[1]==2)
    yupperSE=pSE[yvar[2]-1]/y;
    ylowerSE=-pSE[yvar[2]-1]/y;
    xupperSE=(pSE[xvar[2]-1])/x;
    xlowerSE=(-pSE[xvar[2]-1])/x;
    w=xupperSE-xlowerSE
    h=yupperSE-ylowerSE
    plot!(BifARelPlot,Shape(xlowerSE .+ [0,w,w,0,0], ylowerSE .+ [0,0,h,h,0]),fillalpha=0,color=:black,linewidth=3,linestyle=:dash,label="One standard error from mean")
    plot!(BifAPlot,Shape(xlowerSE .+ [0,w,w,0,0], ylowerSE .+ [0,0,h,h,0]),fillalpha=0,color=:black,linewidth=3,linestyle=:dash,label="One standard error from mean")
elseif yvar[1]==2
    upperSE=pSE[yvar[2]-1]/y;
    lowerSE=-pSE[yvar[2]-1]/y;
    hline!(BifARelPlot,[upperSE],linewidth=3,color=:black,linestyle=:dash,label="One standard error from mean")
    hline!(BifARelPlot,[lowerSE],linewidth=3,color=:black,linestyle=:dash,label=false)
    hline!(BifAPlot,[upperSE],linewidth=3,color=:black,linestyle=:dash,label="One standard error from mean")
    hline!(BifAPlot,[lowerSE],linewidth=3,color=:black,linestyle=:dash,label=false)

elseif xvar[1]==2
    upperSE=(pSE[xvar[2]-1])/x;
    lowerSE=(-pSE[xvar[2]-1])/x;
    vline!(BifARelPlot,[upperSE],linewidth=3,color=:black,linestyle=:dash,label="One standard error from mean")
    vline!(BifARelPlot,[lowerSE],linewidth=3,color=:black,linestyle=:dash,label=false)
    vline!(BifAPlot,[upperSE],linewidth=3,color=:black,linestyle=:dash,label="One standard error from mean")
    vline!(BifAPlot,[lowerSE],linewidth=3,color=:black,linestyle=:dash,label=false)
end


#endregion

if sum(isBifurcation)==0
    print("No bifurcations")
else
#region Add in extra point

if addpoint==true
    BPn=1001;
    BPxVals=LinRange(RNstartVal[1],RNendVal[1],BPn);
    BPyVals=LinRange(RNstartVal[2],RNendVal[2],BPn);

    BPisBif=zeros(BPn)
    for i=1:BPn
        p=[[q1,q2,q3,q4],[KLexp,SP1,SP2,SP3]]
        p[xvar[1]][xvar[2]]=BPxVals[i]*x+x;
        p[yvar[1]][yvar[2]]=BPyVals[i]*y+y;

        grads=zeros(an)
        for k=1:an
            grads[k]=as[k]*((SPprime(as[k],p)/SP(as[k],p))-(logDRprime(as[k],p)/logDR(as[k],p)))-1;
        end
        gradflips=grads[1:an-1].*grads[2:an];
        for k in (1:an-1)[gradflips.<0]
            if KAfroma(as[k],p)<0
                gradflips[k]=NaN
            elseif  KAfroma(as[k],p)>1000
                gradflips[k]=NaN
            end
        end 
        if sum(gradflips.<0)==2
            BPisBif[i]=1
        end
    end
    BPfirstX=BPxVals[findfirst(BPisBif.==1)]
    BPfirstY=BPyVals[findfirst(BPisBif.==1)]
end

RNn=41;
RNxVals=Vector(LinRange(RNstartVal[1],RNendVal[1],RNn));
RNyVals=Vector(LinRange(RNstartVal[2],RNendVal[2],RNn));

if addpoint==true
    append!(RNxVals,BPfirstX)
    append!(RNyVals,BPfirstY)
    if RNxVals[1]<RNxVals[RNn]
        sort!(RNxVals)
    else
        sort!(RNxVals)
        reverse!(RNxVals)
    end
    if RNyVals[1]<RNyVals[RNn]
        sort!(RNyVals)
    else
        sort!(RNyVals)
        reverse!(RNyVals)
    end

    RNn=RNn+1
end




#endregion
#region Add in the reaction norms


aStartEnd=zeros(RNn,2);
KAStartEnd=zeros(RNn,2);
qStartEnd=zeros(RNn,2);
SPStartEnd=zeros(RNn,2);
SPprimeMax=zeros(RNn)
SPprimeMaxa=zeros(RNn)

for i=1:RNn
    p=[[q1,q2,q3,q4],[KLexp,SP1,SP2,SP3]]
    p[xvar[1]][xvar[2]]=RNxVals[i]*x+x;
    p[yvar[1]][yvar[2]]=RNyVals[i]*y+y;

    grads=zeros(an)
    for k=1:an
        grads[k]=as[k]*((SPprime(as[k],p)/SP(as[k],p))-(logDRprime(as[k],p)/logDR(as[k],p)))-1;
    end
    gradflips=grads[1:an-1].*grads[2:an];
    for k in (1:an-1)[gradflips.<0]
        if KAfroma(as[k],p)<0
            gradflips[k]=NaN
        elseif  KAfroma(as[k],p)>1000
            gradflips[k]=NaN
        end
    end 
    if sum(gradflips.<0)==2
        isBifurcation=1

        aStartEnd[i,1]=as[findfirst(gradflips.<0)]
        aStartEnd[i,2]=as[findlast(gradflips.<0)]

        KAStartEnd[i,1]=KAfroma(aStartEnd[i,1],p)
        KAStartEnd[i,2]=KAfroma(aStartEnd[i,2],p)

        reorder=sortperm(KAStartEnd[i,:])

        KAStartEnd[i,:]=KAStartEnd[i,:][reorder]
        aStartEnd[i,:]=aStartEnd[i,:][reorder]

        qStartEnd[i,1]=q(aStartEnd[i,1],p)
        qStartEnd[i,2]=q(aStartEnd[i,2],p)

        SPStartEnd[i,1]=SP(aStartEnd[i,1],p)
        SPStartEnd[i,2]=SP(aStartEnd[i,2],p)
    else
        aStartEnd[i,1]=NaN
        aStartEnd[i,2]=NaN        
        
        qStartEnd[i,1]=NaN
        qStartEnd[i,2]=NaN

        SPStartEnd[i,1]=NaN
        SPStartEnd[i,2]=NaN
    end


    SPprimeMax[i]=maximum(SPprimeVec(as,p))
    SPprimeMaxa[i]=as[findfirst(==(maximum(SPprimeMax[i])), SPprimeVec(as,p))]

end

NoBifurcation=isnan.(aStartEnd[:,1]);
NoBifN=sum(NoBifurcation);
BifN=RNn-NoBifN;



SPPlot=Plots.plot(xlabel=L"Larval food available per capita, $\alpha$ (mg)",ylabel=L"Through-pupal survival, $S_P(\alpha)$",size=[600,600],ylims=[0,1]);
qPlot =Plots.plot(xlabel=L"Larval food available per capita, $\alpha$ (mg)",ylabel=L"Maximum adult fecundity, $q(\alpha)$",size=[600,600]);
qSPlot=Plots.plot(xlabel=L"Larval food available per capita, $\alpha$ (mg)",ylabel=L"Maximum reproduction rate, $qS$",size=[600,600]);
SPoveraPlot=Plots.plot(xlabel=L"Larval food available per capita, $\alpha$ (mg)",ylabel=L"Ratio of survival over food, $S_P/\alpha$",size=[600,600],ylims=[0,.1]);


Cxchange=(RNxVals[NoBifN]+RNxVals[NoBifN+1])/2
CSchange=(Cxchange-RNstartVal[1])/(RNendVal[1]-RNstartVal[1])

UnifiedColors=cgrad(:balance,CSchange)

j=1
for i=(1:RNn)[NoBifurcation]
    p=[[q1,q2,q3,q4],[KLexp,SP1,SP2,SP3]]
    p[xvar[1]][xvar[2]]=RNxVals[i]*x+x;
    p[yvar[1]][yvar[2]]=RNyVals[i]*y+y;

    SPs=SPvect(as,p);
    Plots.plot!(SPPlot,as,SPs,color=UnifiedColors[(RNxVals[j]-RNstartVal[1])/(RNendVal[1]-RNstartVal[1])],linewidth=2,label=false);

    qs=qvect(as,p);
    Plots.plot!(qPlot,as,qs,color=UnifiedColors[(RNxVals[j]-RNstartVal[1])/(RNendVal[1]-RNstartVal[1])],linewidth=2,label=false);

    qS=qs.*SPs*SE*SL*SJ
    Plots.plot!(qSPlot,as,qS,color=UnifiedColors[(RNxVals[j]-RNstartVal[1])/(RNendVal[1]-RNstartVal[1])],linewidth=2,label=false);
    j=j+1;
end

j=1
for i=(1:RNn)[NoBifurcation.==false]
    p=[[q1,q2,q3,q4],[KLexp,SP1,SP2,SP3]]
    p[xvar[1]][xvar[2]]=RNxVals[i]*x+x;
    p[yvar[1]][yvar[2]]=RNyVals[i]*y+y;


    SPs=SPvect(as,p);
    Plots.plot!(SPPlot,as,SPs,color=UnifiedColors[(RNxVals[NoBifN+j]-RNstartVal[1])/(RNendVal[1]-RNstartVal[1])],linewidth=2,label=false);

    Plots.plot!(SPoveraPlot,as,SPs./as,color=UnifiedColors[(RNxVals[NoBifN+j]-RNstartVal[1])/(RNendVal[1]-RNstartVal[1])],linewidth=2,label=false);

    qs=qvect(as,p);
    Plots.plot!(qPlot,as,qs,color=UnifiedColors[(RNxVals[NoBifN+j]-RNstartVal[1])/(RNendVal[1]-RNstartVal[1])],linewidth=2,label=false);

    qS=qs.*SPs*SE*SL*SJ
    Plots.plot!(qSPlot,as,qS,color=UnifiedColors[(RNxVals[NoBifN+j]-RNstartVal[1])/(RNendVal[1]-RNstartVal[1])],linewidth=2,label=false);
    j=j+1
end


Plots.scatter!(BifPlot,RNxVals[NoBifurcation.==false],RNyVals[NoBifurcation.==false],color=UnifiedColors[(RNxVals[NoBifurcation.==false].-RNstartVal[1])./(RNendVal[1]-RNstartVal[1])],markersize=12,label="Has tipping points",markershape=:square)
Plots.scatter!(BifPlot,RNxVals[NoBifurcation],        RNyVals[NoBifurcation],        color=UnifiedColors[(RNxVals[NoBifurcation].-RNstartVal[1])./(RNendVal[1]-RNstartVal[1])],        markersize=12,label="No tipping points",markershape=:utriangle)

Plots.scatter!(BifAPlot,RNxVals[NoBifurcation.==false],RNyVals[NoBifurcation.==false],color=UnifiedColors[(RNxVals[NoBifurcation.==false].-RNstartVal[1])./(RNendVal[1]-RNstartVal[1])],markersize=12,label="Has tipping points",markershape=:square)
Plots.scatter!(BifAPlot,RNxVals[NoBifurcation],        RNyVals[NoBifurcation],        color=UnifiedColors[(RNxVals[NoBifurcation].-RNstartVal[1])./(RNendVal[1]-RNstartVal[1])],        markersize=12,label="No tipping points",markershape=:utriangle)

Plots.scatter!(BifARelPlot,RNxVals[NoBifurcation.==false],RNyVals[NoBifurcation.==false],color=UnifiedColors[(RNxVals[NoBifurcation.==false].-RNstartVal[1])./(RNendVal[1]-RNstartVal[1])],markersize=12,label="Has tipping points",markershape=:square)
Plots.scatter!(BifARelPlot,RNxVals[NoBifurcation],        RNyVals[NoBifurcation],        color=UnifiedColors[(RNxVals[NoBifurcation].-RNstartVal[1])./(RNendVal[1]-RNstartVal[1])],        markersize=12,label="No tipping points",markershape=:utriangle)


Plots.scatter!(qPlot,aStartEnd[:,1],qStartEnd[:,1],markersize=8,markerstrokewidth=3,color=:black,markershape=:x,label="High stress tipping point")
Plots.scatter!(qPlot,aStartEnd[:,2],qStartEnd[:,2],markersize=8,markerstrokewidth=2,color=:black,label="Low stress tipping point")

Plots.scatter!(SPPlot,aStartEnd[:,1],SPStartEnd[:,1],markersize=8,markerstrokewidth=3,color=:black,markershape=:x,label="High stress tipping point")
Plots.scatter!(SPPlot,aStartEnd[:,2],SPStartEnd[:,2],markersize=8,markerstrokewidth=2,color=:black,label="Low stress tipping point")

Plots.scatter!(SPoveraPlot,aStartEnd[:,1],SPStartEnd[:,1]./aStartEnd[:,1],markersize=8,markerstrokewidth=3,color=:black,markershape=:x,label="High stress tipping point")
Plots.scatter!(SPoveraPlot,aStartEnd[:,2],SPStartEnd[:,2]./aStartEnd[:,2],markersize=8,markerstrokewidth=2,color=:black,label="Low stress tipping point")


#endregion

#region Plot collater


l=@layout[
    a{0.66h}; [b{0.45w} c{0.35w}]
]

totalAbsPlot=Plots.plot(BifAPlot,qPlot, SPPlot,layout=l,margin=10mm,size=[1200,1600], dpi=300)
totalRelPlot=Plots.plot(BifARelPlot,qPlot, SPPlot,layout=l,margin=10mm,size=[1200,1600], dpi=300)

#region Add in colourbar and insert height

CLims=[0,CSchange,1]

if RNstartVal[1]==RNstartVal[2]
    cbarTitle=latexstring("Percentage difference in ",plabelnames[xvar[1]][xvar[2]],",",plabelnames[yvar[1]][yvar[2]])
else
    cbarTitle=latexstring("Percentage difference in ",plabelnames[xvar[1]][xvar[2]])
end


Plots.scatter!(totalAbsPlot,zeros(RNn), zeros(RNn), zcolor=RNxVals,
    background_color_subplot=:transparent,
    inset=bbox(0.115, 0.04, 0.8, 0.30, :bottom, :right), subplot=4,
    xlims=(1,1.1), label="", margin=0mm,
    color=UnifiedColors[(RNxVals.-RNstartVal[1])./(RNendVal[1]-RNstartVal[1])],thickness_scaling = 0.8,
    colorbar_title=cbarTitle, framestyle=:none,
    colorbar_ticks=ticks
    )

Plots.scatter!(totalRelPlot,zeros(RNn), zeros(RNn), zcolor=RNxVals,
    background_color_subplot=:transparent,
    inset=bbox(0.115, 0.04, 0.8, 0.30, :bottom, :right), subplot=4,
    xlims=(1,1.1), label="", margin=0mm,
    color=UnifiedColors[(RNxVals.-RNstartVal[1])./(RNendVal[1]-RNstartVal[1])],thickness_scaling = 0.8,
    colorbar_title=cbarTitle, framestyle=:none,
    colorbar_ticks=ticks
    )


if insertHeight
Plots.plot!(BifPlot,legend=:left)
Plots.scatter!(RNxVals,aStartEnd[:,2].-aStartEnd[:,1],color=UnifiedColors[(1:RNn)/RNn],
    inset = (1, bbox(0.10, 0.075, 0.35, 0.40, :bottom, :right)),
    markersize=8,markerstrokewidth=2,legend=false,
    subplot = 5, xguidefontsize=16, yguidefontsize=12, 
    xlims=[0,0.5],
    ylabel=L"Difference in $\alpha$ between tipping points",xlabel=latexstring("Difference in ",plabelnames[xvar[1]][xvar[2]],plabelnames[yvar[1]][yvar[2]])
)
end

annotate!(totalAbsPlot, -50,  3.05, text("a", :right, 25),subplot=3)
annotate!(totalAbsPlot, -5,  1, text("c", :right, 25),subplot=3)
annotate!(totalAbsPlot, -50,  1, text("b", :right, 25),subplot=3)

annotate!(totalRelPlot, -50,  3.05, text("a", :right, 25),subplot=3)
annotate!(totalRelPlot, -5,  1, text("c", :right, 25),subplot=3)
annotate!(totalRelPlot, -50,  1, text("b", :right, 25),subplot=3)

plot!(totalRelPlot[1],right_margin=-3mm)
plot!(totalAbsPlot[1],right_margin=-3mm)

Plots.savefig(totalAbsPlot,string(pnames[xvar[1]][xvar[2]],"vs",pnames[yvar[1]][yvar[2]],".png"))
Plots.savefig(totalRelPlot,string(pnames[xvar[1]][xvar[2]],"vs",pnames[yvar[1]][yvar[2]],"Relative",".png"))
#endregion
end
totalRelPlot
