####~load libraries~~~~~~~~~~####
library(ggVennDiagram)
library(GeneOverlap)
library(ggplot2)
library(svglite)
library(reshape2)
library(IsoformSwitchAnalyzeR)
library(UpSetR)
library(ggpubr)

####~housekeeping~~~~~~~~~~~~####
rm(list=ls()) #clear the environment
setwd(dirname(rstudioapi::getActiveDocumentContext()$path)) #set wd to Scripts folder

###~~specify input files~~~~~####
ss_file = "../02_reference_data/sample_sheet.csv"
gtf_file = "../02_reference_data/annotation.gtf"
cpm_file = "../04_edger/cpm.csv"
logcpm_file = "../04_edger/log2cpm.csv"
tx_cpm_file = "../04_edger/transcript_cpm.csv"
tx_logcpm_file = "../04_edger/transcript_log2cpm.csv"
sample_order = c("pre rep","pregnant","post rep")
sample_labels = c("pre-pregnancy","pregnancy","post-parturition")

###~~themes~~~~~~~~~~~~~~~~~~####
theme_j = theme(
  plot.title = element_blank(),
  axis.text.x=element_text(size=10),
  axis.text.y=element_text(size=10),
  axis.title.x=element_text(size=18),
  axis.title.y=element_text(size=18),
  panel.background = element_rect(fill = "white", colour = "lightgrey"),
  panel.grid.major = element_line(linewidth = 0.5, linetype = "solid", colour = "lightgrey"),
  panel.grid.minor = element_line(linewidth =  0.25, linetype = "solid", colour = "lightgrey"),
  legend.key = element_blank(),
  legend.title = element_blank(), 
  legend.text = element_text(size = 18), 
  legend.key.size = unit(1, "cm"))

##~~~colourblind palettes~~~~####
palette_cb1 = c("#999999", "#E69F00", "#56B4E9", "#009E73", "#F0E442", "#0072B2", "#D55E00", "#CC79A7")
palette_cb2 = c("#000000", "#E69F00", "#56B4E9", "#009E73", "#F0E442", "#0072B2", "#D55E00", "#CC79A7")

###~~output directory~~~~~~~~####
output = "../11_summary_and_figures/" #specify where the output should go
dir.create(output) #create directory for output
setwd(output) #set the new output directory as the working directory

####~load data~~~~~~~~~~~~~~~####

###~~all DEGs~~~~~~~~~~~~~~~~####
dge_PREGvNON = read.csv("../06_dge_analysis/01_PREGvNON/sig.csv", row.names = 1)
dge_PREGvPRE = read.csv("../06_dge_analysis/02_PREGvPRE/sig.csv", row.names = 1)
dge_PREGvPOST = read.csv("../06_dge_analysis/03_PREGvPOST/sig.csv", row.names = 1)
dge_POSTvPRE = read.csv("../06_dge_analysis/04_POSTvPRE/sig.csv", row.names = 1)
DEGs = list(
  PREGvNON = dge_PREGvNON$SYMBOL,
  PREGvPRE = dge_PREGvPRE$SYMBOL,
  PREGvPOST = dge_PREGvPOST$SYMBOL,
  POSTvPRE = dge_POSTvPRE$SYMBOL
)

##~~~preg DEGs~~~~~~~~~~~~~~~####
pregDEGs = unique(c(dge_PREGvPRE$SYMBOL,dge_PREGvPOST$SYMBOL))

###~~upregulated DEGs~~~~~~~~####
up_PREGvNON = read.csv("../06_dge_analysis/01_PREGvNON/sig_up.csv")
up_PREGvPRE = read.csv("../06_dge_analysis/02_PREGvPRE/sig_up.csv")
up_PREGvPOST = read.csv("../06_dge_analysis/03_PREGvPOST/sig_up.csv")
up_POSTvPRE = read.csv("../06_dge_analysis/04_POSTvPRE/sig_up.csv")
up_DEGs = list(
  PREGvNON = up_PREGvNON$SYMBOL,
  PREGvPRE = up_PREGvPRE$SYMBOL,
  PREGvPOST = up_PREGvPOST$SYMBOL,
  POSTvPRE = up_POSTvPRE$SYMBOL
)

###~~downregulated DEGs~~~~~~####
down_PREGvNON = read.csv("../06_dge_analysis/01_PREGvNON/sig_down.csv")
down_PREGvPRE = read.csv("../06_dge_analysis/02_PREGvPRE/sig_down.csv")
down_PREGvPOST = read.csv("../06_dge_analysis/03_PREGvPOST/sig_down.csv")
down_POSTvPRE = read.csv("../06_dge_analysis/04_POSTvPRE/sig_down.csv")
down_DEGs = list(
  PREGvNON = down_PREGvNON$SYMBOL,
  PREGvPRE = down_PREGvPRE$SYMBOL,
  PREGvPOST = down_PREGvPOST$SYMBOL,
  POSTvPRE = down_POSTvPRE$SYMBOL
)

###~~dtu genes~~~~~~~~~~~~~~~####
dtu_PREGvPRE = read.delim("../08_dtu_preg_v_pre/DTU_gene_list.txt", header = FALSE)
dtu_PREGvPOST = read.delim("../09_dtu_preg_v_post/DTU_gene_list.txt", header = FALSE)
dtu_POSTvPRE = read.delim("../10_dtu_post_v_pre/DTU_gene_list.txt", header = FALSE)
DTUs = list(
  PREGvPRE = dtu_PREGvPRE$V1,
  PREGvPOST = dtu_PREGvPOST$V1,
  POSTvPRE = dtu_POSTvPRE$V1
)

###~~list all DEGs~~~~~~~~~~~####
allDEGs = unique(c())

###~~load other taxa DEGs~~~~####
other_taxa = read.csv("../02_reference_data/Shared_DE_genes_vertebrates.csv")
CVII_EO = read.csv("../02_reference_data/Recknagel_et_al_2021_DEGs.csv")
candidates=read.csv("../02_reference_data/Recknagel_et_al_2021_candidates.csv",header=FALSE)

###~~sample sheet~~~~~~~~~~~~####
ss = read.csv(ss_file, row.names = 3)

###~~cpm~~~~~~~~~~~~~~~~~~~~~####
cpm = read.csv(cpm_file, row.names = 1)
logcpm = read.csv(logcpm_file, row.names = 1)
cpm.tx = read.csv(tx_cpm_file, row.names = 1)
logcpm.tx = read.csv(tx_logcpm_file, row.names = 1)

####~parse data~~~~~~~~~~~~~~####
###~~em~~~~~~~~~~~~~~~~~~~~~~####
em = cpm
em_scaled = data.frame(t(scale(data.frame(t(em)))))
em.tx = cpm.tx[,-9] #without gene names
em.tx_scaled=data.frame(t(scale(data.frame(t(em.tx)))))

###~~CVII~~~~~~~~~~~~~~~~~~~~####
CVII_EO = subset(CVII_EO, combined_siginficant_padj.0.05_after_corrected == "YES")
CVII_EO = unique(CVII_EO$gene_symbol)

###~~all other taxa~~~~~~~~~~####
ot_gene_lists = list(
  ot_atleast7 = subset(other_taxa, Sum.shared.all > 6)$gene.symbol,
  ot_atleast6 = subset(other_taxa, Sum.shared.all > 5)$gene.symbol,
  ot_atleast5 = subset(other_taxa, Sum.shared.all > 4)$gene.symbol,
  ot_atleast4 = subset(other_taxa, Sum.shared.all > 3)$gene.symbol,
  ot_atleast3 = subset(other_taxa, Sum.shared.all > 2)$gene.symbol)

###~~other squamates~~~~~~~~~####
squam_gene_lists = list(
  squam_atleast4 = subset(other_taxa, sum.squamates.shared > 3)$gene.symbol,
  squam_atleast3 = subset(other_taxa, sum.squamates.shared > 2)$gene.symbol,
  squam_atleast2 = subset(other_taxa, sum.squamates.shared > 1)$gene.symbol)

###~~mammals~~~~~~~~~~~~~~~~~####
mam_gene_lists = list(
  mam_atleast5 = subset(other_taxa, sum.mammals.shared > 4)$gene.symbol,
  mam_atleast4 = subset(other_taxa, sum.mammals.shared > 3)$gene.symbol,
  mam_atleast3 = subset(other_taxa, sum.mammals.shared > 2)$gene.symbol,
  mam_atleast2 = subset(other_taxa, sum.mammals.shared > 1)$gene.symbol)

###~~candidate genes~~~~~~~~~####
names(candidates) = candidates[2,]
candidates = candidates[-c(1,2),]
all_candidate_genes = candidates[,1]
candidate_genes = all_candidate_genes[all_candidate_genes %in% row.names(em)]

####~bar chart~~~~~~~~~~~~~~~####
makeNeg = function(x) x*-1
bars = data.frame(var = rep(c("upregulated","downregulated"), each=4),
                  comp = rep(c("PREGvNON","PREGvPRE","PREGvPOST","PREvPOST"), all=2),
                  v = c(unlist(lapply(up_DEGs, length)),
                        unlist(lapply(lapply(down_DEGs, length), makeNeg))
                        )
                  )
ggplot(data = bars, aes (x = comp, y = v, fill = var)) +
  geom_bar(stat="identity") +
  scale_x_discrete(limits = c("PREGvNON","PREGvPRE","PREGvPOST","PREvPOST")) +
  ylab("DEGs") +
  xlab(element_blank()) +
  theme_j +
  scale_fill_manual(values = c("lightblue","darkred")) +
  theme(axis.text.x = element_text(angle = 45, size = 16, hjust = 1),
        axis.text.y = element_text(size = 16))
ggsave("dge_bar.svg")

####~comparisons~~~~~~~~~~~~~####
compareLists = function(list1, list2) {
  return(list1[list1 %in% list2])
}

###~~DGE analyses~~~~~~~~~~~~####
PvPRE_PvN_shared = compareLists(DEGs$PREGvPRE, DEGs$PREGvNON)
PvPOST_PvN_shared = compareLists(DEGs$PREGvPOST, DEGs$PREGvNON)
PvPRE_PvPOST_shared = compareLists(DEGs$PREGvPOST, DEGs$PREGvPRE)
PvPnPvN_PvPnPvN_shared = compareLists(PvPRE_PvN_shared, PvPOST_PvN_shared)
POSTvPRE_PvPRE_shared = compareLists(DEGs$PREGvPOST, DEGs$POSTvPRE)
POSTvPRE_PvPOST_shared = compareLists(DEGs$PREGvPRE, DEGs$POSTvPRE)

##~~~Venn~~~~~~~~~~~~~~~~~~~~####
ggVennDiagram(DEGs, label_alpha = 0, label = "count") +
  ggplot2::scale_fill_gradient(high = "red", low = "lightgrey") +
  ggplot2::scale_color_manual(values = c("black","black","black","black")) +
  ggplot2::scale_x_continuous(expand = expansion(mult = .2)) +
  theme(legend.position = "none")
ggsave("venn_degs.svg")

###~~DTU analysis~~~~~~~~~~~~####
PvPRE_PvPOST_shared_dtu = compareLists(DTUs$PREGvPRE, DTUs$PREGvPOST)
write(PvPRE_PvPOST_shared_dtu, file = "shared_dtu.txt")

##~~~Venn~~~~~~~~~~~~~~~~~~~~####
ggVennDiagram(DTUs, label_alpha = 0, label = "count") +
  ggplot2::scale_fill_gradient(high = "red", low = "lightgrey") +
  ggplot2::scale_color_manual(values = c("black","black","black")) +
  ggplot2::scale_x_continuous(expand = expansion(mult = .2)) +
  theme(legend.position = "none")
ggsave("venn_dtu.svg")

###~~DGE with DTU~~~~~~~~~~~~####
PREGvPRE_c = compareLists(DEGs$PREGvPRE, DTUs$PREGvPRE)
PREGvPOST_c = compareLists(DEGs$PREGvPOST, DTUs$PREGvPOST)
POSTvPRE_c = compareLists(DEGs$POSTvPRE, DTUs$POSTvPRE)

##~~~Venn~~~~~~~~~~~~~~~~~~~~####
DGE_DTU = list("PREGvPRE DTU" = DTUs$PREGvPRE,
               "PREGvPOST DTU" = DTUs$PREGvPOST,
               "PREGvPRE DGE" = DEGs$PREGvPRE,
               "PREGvPOST DGE" = DEGs$PREGvPOST)
ggVennDiagram(DGE_DTU, label_alpha = 0, label = "count") +
  ggplot2::scale_fill_gradient(high = "red", low = "lightgrey") +
  ggplot2::scale_color_manual(values = c("black","black","black","black","black","black")) +
  ggplot2::scale_x_continuous(expand = expansion(mult = .2)) +
  theme(legend.position = "none")
ggsave("venn_dtu_degs.svg")

DGE_DTU_PREGvPRE = list("PREGvPRE DTU" = DTUs$PREGvPRE,
                        "PREGvPRE DGE" = DEGs$PREGvPRE)
ggVennDiagram(DGE_DTU_PREGvPRE, label_alpha = 0, label = "count") +
  ggplot2::scale_fill_gradient(high = "red", low = "lightgrey") +
  ggplot2::scale_color_manual(values = c("black","black","black","black","black","black")) +
  ggplot2::scale_x_continuous(expand = expansion(mult = .2)) +
  theme(legend.position = "none")
ggsave("venn_PREGvPRE.svg")

DGE_DTU_PREGvPOST = list("PREGvPOST DTU" = DTUs$PREGvPOST,
                        "PREGvPOST DGE" = DEGs$PREGvPOST)
ggVennDiagram(DGE_DTU_PREGvPOST, label_alpha = 0, label = "count") +
  ggplot2::scale_fill_gradient(high = "red", low = "lightgrey") +
  ggplot2::scale_color_manual(values = c("black","black","black","black","black","black")) +
  ggplot2::scale_x_continuous(expand = expansion(mult = .2)) +
  theme(legend.position = "none")
ggsave("venn_PREGvPOST.svg")

##~~~bar chart~~~~~~~~~~~~~~~####
bars = data.frame(var = rep(c("DGE","DTU","shared"), each=3),
                  comp = rep(c("PREGvPRE","PREGvPOST","PREvPOST"), all=3),
                  val = c(length(DEGs$PREGvPRE),
                          length(DEGs$PREGvPOST),
                          length(DEGs$POSTvPRE),
                          length(DTUs$PREGvPRE),
                          length(DTUs$PREGvPOST),
                          length(DTUs$POSTvPRE),
                          length(PREGvPRE_c),
                          length(PREGvPOST_c),
                          length(POSTvPRE_c)))
ggplot(data = bars, aes (x = comp, y = val, fill = var)) +
  geom_bar(stat="identity", position = position_dodge()) +
  scale_x_discrete(limits = c("PREGvPRE","PREGvPOST","PREvPOST")) +
  scale_y_log10() +
  ylab("log10(genes)") +
  xlab(element_blank()) +
  theme_j +
  theme(axis.text.x = element_text(angle = 45, size = 16, hjust = 1),
        axis.text.y = element_text(size = 16)) +
  scale_fill_manual(values = palette_cb1)
ggsave("dge_dtu_bar.svg")

##~~~UpSet plot~~~~~~~~~~~~~~####
DGEvDTU = list("PREGvPRE DTU" = DTUs$PREGvPRE,
               "PREGvPRE DGE" = DEGs$PREGvPRE,
               "PREGvPOST DTU" = DTUs$PREGvPOST,
               "PREGvPOST DGE" = DEGs$PREGvPOST,
               "POSTvPRE DTU" = DTUs$POSTvPRE,
               "POSTvPRE DGE" = DEGs$POSTvPRE)
upsetData = fromList(DGEvDTU) #make upset plot dataset
upsetDTUvsDGE = upset(upsetData,
                      sets = c(
                        "POSTvPRE DTU",
                        "POSTvPRE DGE",
                        "PREGvPOST DTU",
                        "PREGvPOST DGE",
                        "PREGvPRE DTU",
                        "PREGvPRE DGE"
                      ),
                      order.by = "freq",
                      keep.order = TRUE,
                      text.scale = c(1.3, 1.3, 1, 1, 1.25, 1.2),
                      point.size = 3,
                      line.size = 1,
                      main.bar.color = "black",
                      sets.bar.color = "grey",
                      mainbar.y.label = "Unique Genes",
                      sets.x.label = "Total Genes"
                      )

svglite(filename = "upset_lot_dte_dgu.svg")
upsetDTUvsDGE
dev.off()

####~specific genes~~~~~~~~~~####

###~~PREGvPRE~~~~~~~~~~~~~~~~####
load(file = "../08_dtu_preg_v_pre/switchListAnalysed.Rdata")
switchListPREGvPRE = mySwitchList

###~~PREGvPOST~~~~~~~~~~~~~~~####
load(file = "../09_dtu_preg_v_post/switchListAnalysed.Rdata")
switchListPREGvPOST = mySwitchList

###~~POSTvPRE~~~~~~~~~~~~~~~~####
load(file = "../10_dtu_post_v_pre/switchListAnalysed.Rdata")
switchListPOSTvPRE = mySwitchList

getTranscripts4Gene = function(gene) {
  df = subset(cpm, GENEID == gene)
  txs = row.names(df)
  return(txs)
}

makeGeneBoxplot = function(gene, em = logcpm, palette = palette_cb1) {
  #make gene table
  gene_data=data.frame(t(em[gene,]))
  gene_data$sample_group=ss$Condition
  gene_data.m=melt(gene_data,id.vars = "sample_group")
  gene_data.m$sample_group=factor(gene_data$sample_group,levels=sample_order) #reorder
  
  #make boxplot
  boxplot=ggplot(gene_data.m,aes(x=variable,y=value,fill=sample_group))+
    geom_boxplot(outlier.size=0,show.legend=TRUE)+
    theme_j+
    xlab(element_blank())+
    ylab("log2 (CPM)")+
    scale_fill_manual(values=palette)+
    coord_cartesian(clip = "off")+
    theme(axis.text.x=element_text(angle = 45, hjust = 1, size = 14),
          aspect.ratio = 2/1,
          plot.margin = margin(l = 20, b = 40))
  return(boxplot)
}

makeTxBoxplot = function(gene, em = logcpm.tx[,-9], palette = palette_cb1) {
  #get tx IDs
  df = subset(cpm.tx, GENEID == gene)
  txs = row.names(df)

  #make gene table
  gene_data=data.frame(t(em[txs,]))
  gene_data$sample_group=ss$Condition
  gene_data.m=melt(gene_data,id.vars = "sample_group")
  gene_data.m$sample_group=factor(gene_data$sample_group,levels=sample_order) #reorder
  
  #make boxplot
  boxplot=ggplot(gene_data.m,aes(x=variable,y=value,fill=sample_group))+
    geom_boxplot(outlier.size=0,show.legend=TRUE)+
    theme_j+
    xlab(element_blank())+
    ylab("log2(CPM)")+
    scale_fill_manual(values=palette)+
    coord_cartesian(clip = "off")+
    theme(axis.text.x=element_text(angle = 45, hjust = 1, size = 14),
          aspect.ratio = 2/1,
          plot.margin = margin(l = 20, b = 40))
  return(boxplot)
}

makeStackedBarIF = function(gene, palette = palette_cb1) {
  #get tx IDs
  df = subset(cpm.tx, GENEID == gene)
  txs = row.names(df)
  
  #get IF for each state for each tx
  tx.df = cpm.tx[txs,-9]
  tx.df$gene_id = gene
  if.df = isoformToIsoformFraction(
    isoformRepExpression = tx.df
  )
  
  #make gene table
  gene_data=data.frame(t(if.df))
  gene_data$sample_group=ss$Condition
  gene_data.m=melt(gene_data,id.vars = "sample_group")
  gene_data.m$sample_group=factor(gene_data$sample_group,levels=sample_order) #reorder
  
  #make bar plot
  barplot = ggplot(gene_data.m,
                   aes(x = sample_group, 
                       y = value, 
                       fill = variable))+
    geom_bar(position = "fill", stat = "identity")+
    ylab("IF")+
    xlab(element_blank())+
    theme_j+
    scale_fill_manual(values = palette, labels=sample_labels)+
    coord_cartesian(clip = "off")+
    theme(axis.text.x=element_text(angle = 45, hjust = 1, size = 14),
          aspect.ratio = 2/1)
  return(barplot)
}

makeBoxPlotIF = function(gene, palette = palette_cb1) {
  #get tx IDs
  df = subset(cpm.tx, GENEID == gene)
  txs = row.names(df)
  
  #get IF for each state for each tx
  tx.df = cpm.tx[txs,-9]
  tx.df$gene_id = gene
  if.df = isoformToIsoformFraction(
    isoformRepExpression = tx.df
  )
  
  #make gene table
  gene_data=data.frame(t(if.df))
  gene_data$sample_group=ss$Condition
  gene_data.m=melt(gene_data,id.vars = "sample_group")
  gene_data.m$sample_group=factor(gene_data$sample_group,levels=sample_order) #reorder
  
  #make box plot
  boxplot=ggplot(gene_data.m,aes(x=variable,y=value,fill=sample_group))+
    geom_boxplot(outlier.size=0,show.legend=TRUE)+
    theme_j+
    xlab(element_blank())+
    ylab("IF")+
    scale_fill_manual(values=palette, labels=sample_labels)+
    theme(axis.text.x=element_text(angle = 45, hjust = 1, size = 14),
          aspect.ratio = 2/1,
          plot.margin = margin(l = 20, b = 40))
  return(boxplot)
}

makeMySwitchplot = function(gene, switchList) {
  #get tx IDs
  df = subset(cpm.tx, GENEID == gene)
  txs = row.names(df)
  #make switchplot
  mySwitchPlot = switchPlotTranscript(switchList, isoform_id = txs)+
    ggplot2::theme(axis.text = element_text(size = 14),
                   legend.text = element_text(size = 16),
                   legend.title = element_text(size = 18))
  return(mySwitchPlot)
}

saveAllPlots = function(gene) {
  bp = makeGeneBoxplot(gene)
  ggsave(paste(gene,"_bp.svg",sep = ""),bp, width = 4, height = 6)
  bptx = makeTxBoxplot(gene)
  ggsave(paste(gene,"_tx_bp.svg",sep = ""), bptx, width = 4, height = 6)
  bpif = makeBoxPlotIF(gene)
  ggsave(paste(gene,"_if_bp.svg",sep = ""), bpif, width = 4, height = 6)
  sbif = makeStackedBarIF(gene)
  ggsave(paste(gene,"_if.svg",sep = ""), sbif, width = 8, height = 6)
  tx = makeMySwitchplot(gene = gene, switchList = switchListPREGvPRE)
  ggsave(paste(gene,"_tx.svg",sep = ""), tx, width = 12, height = 3)
  combo = ggarrange(bp, bptx, bpif, ncol = 3, common.legend = TRUE)
  ggsave(paste(gene,"_combined.svg",sep = ""), combo, width = 9, height = 6)
  return(print(paste("Plots saved to",getwd())))
}

make1TxBoxplot = function(tx, em = logcpm.tx[,-9], palette = palette_cb1) {
  #make gene table
  gene_data=data.frame(t(em[tx,]))
  gene_data$sample_group=ss$Condition
  gene_data.m=melt(gene_data,id.vars = "sample_group")
  gene_data.m$sample_group=factor(gene_data$sample_group,levels=sample_order) #reorder
  
  #make boxplot
  boxplot=ggplot(gene_data.m,aes(x=variable,y=value,fill=sample_group))+
    geom_boxplot(outlier.size=0,show.legend=TRUE)+
    theme_j+
    xlab(element_blank())+
    ylab("log2(CPM)")+
    scale_fill_manual(values=palette, labels=sample_labels)+
    coord_cartesian(clip = "off")+
    theme(axis.text.x=element_text(angle = 45, hjust = 1, size = 14),
          aspect.ratio = 2/1,
          plot.margin = margin(l = 20, b = 40))
  return(boxplot)
}

make1TxBoxplotIF = function(gene, tx, palette = palette_cb1) {
  #get tx IDs
  df = subset(cpm.tx, GENEID == gene)
  txs = row.names(df)
  
  #get IF for each state for each tx
  tx.df = cpm.tx[txs,-9]
  tx.df$gene_id = gene
  if.df = isoformToIsoformFraction(
    isoformRepExpression = tx.df
  )
  if.df = if.df[tx,] #retain only transcript of interest
  
  #make gene table
  gene_data=data.frame(t(if.df))
  gene_data$sample_group=ss$Condition
  gene_data.m=melt(gene_data,id.vars = "sample_group")
  gene_data.m$sample_group=factor(gene_data$sample_group,levels=sample_order) #reorder
  
  #make box plot
  boxplot=ggplot(gene_data.m,aes(x=variable,y=value,fill=sample_group))+
    geom_boxplot(outlier.size=0,show.legend=TRUE)+
    theme_j+
    xlab(element_blank())+
    ylab("IF")+
    scale_fill_manual(values=palette,labels=sample_labels)+
    theme(axis.text.x=element_text(angle = 45, hjust = 1, size = 14),
          aspect.ratio = 2/1,
          plot.margin = margin(l = 20, b = 40))
  return(boxplot)
}


###~~SRSF~~~~~~~~~~~~~~~~~~~~####
txplot = make1TxBoxplot(tx = "XM_035104565.2")
ggsave("SRSF2_XM_035104565-2.svg", plot = txplot, width = 4, height = 6)

ifplot = make1TxBoxplotIF(gene = "SRSF2", tx = "XM_035104565.2")
ggsave("SRSF2_XM_035104565-2_IF.svg", plot = ifplot, width = 4, height = 6)

saveAllPlots("SRSF2")

gplot = makeGeneBoxplot(gene = "SRSF2")
comboSRSF2 = ggarrange(gplot, txplot, ifplot, ncol = 3, common.legend = TRUE)
ggsave("SRSF2_combined2.svg", plot = comboSRSF2, width = 9, height = 6)

###~getDIF~~~~~~~~~~~~~~~~~~~####
dIFtable = function(gene) {
  #get tx IDs
  df = subset(cpm.tx, GENEID == gene)
  txs = row.names(df)
  
  #get IF for each state for each tx
  tx.df = cpm.tx[txs,-9]
  tx.df$gene_id = gene
  if.df = isoformToIsoformFraction(
    isoformRepExpression = tx.df
  )
  
  #add mean values to IF table
  if.df.PRE = if.df[,c("O1","O2")]
  if.df$PRE = rowMeans(if.df.PRE)
  if.df.PREG = if.df[,c("PREG3","PREG4","PREG5","PREG6")]
  if.df$PREG = rowMeans(if.df.PREG)
  if.df.POST = if.df[,c("PP1A","PP2B")]
  if.df$POST = rowMeans(if.df.POST)
  if.df$PREGvPRE = if.df$PREG - if.df$PRE
  if.df$PREGvPOST = if.df$PREG - if.df$POST
  return(if.df)
}

SRSF2_df = dIFtable("SRSF2")
write.csv(SRSF2_df, file = "SRSF2.csv")

###~~FAU~~~~~~~~~~~~~~~~~~~~~####
saveAllPlots("FAU")
FAU_df = dIFtable("FAU")
write.csv(FAU_df, file = "FAU.csv")

###~~LPP~~~~~~~~~~~~~~~~~~~~~####
saveAllPlots("LPP")
tx = makeMySwitchplot(gene = "LPP", switchList = switchListPREGvPOST)
ggsave("LPP_tx.svg", tx, width = 12, height = 3) #default switchList doesn't have all the isoforms for some raisin
LPP_df = dIFtable("LPP")
write.csv(LPP_df, file = "LPP.csv")

####~other studies~~~~~~~~~~~####

top20_PREGvNON = dge_PREGvNON[1:20,]
up20_PREGvNON = up_PREGvNON[1:20,]
row.names(up20_PREGvNON) =up20_PREGvNON$X

###~~old Zoot vs. preg~~~~~~~####
Z_shared.1 = dge_PREGvPOST$SYMBOL[dge_PREGvPOST$SYMBOL %in% CVII_EO]
Z_shared.2 = compareLists(DEGs$PREGvNON, CVII_EO)
row.names(dge_PREGvNON) = dge_PREGvNON$SYMBOL
dge_PREGvNON.zv_sh = dge_PREGvNON[Z_shared.2,]
up20_PREGvNON[Z_shared.2,]

###~~old DGE vs. preg~~~~~~~~####

##~~~make comparisons~~~~~~~~####
ot_comps = lapply(ot_gene_lists, compareLists, list2 = Z_shared.2)
squam_comps = lapply(squam_gene_lists, compareLists, list2 = Z_shared.2)
mam_comps = lapply(mam_gene_lists, compareLists, list2 = Z_shared.2)

##~~~splicing?~~~~~~~~~~~~~~~####
dtu_vs_dge_old = lapply(ot_gene_lists, compareLists, list2 = PvPRE_PvPOST_shared_dtu) #none shared
dtu_vs_dge_Zold = compareLists(CVII_EO, PvPRE_PvPOST_shared_dtu)

###~~candidate genes~~~~~~~~~####
#find DE candidate genes
candidatesDE = compareLists(candidate_genes, pregDEGs)

##~~~all candidate genes~~~~~####
#make a table for candidate genes
gene_data=data.frame(t(logcpm[candidate_genes,]))
gene_data$sample_group=ss$Condition
gene_data.m=melt(gene_data,id.vars = "sample_group",levels=c())
gene_data.m$sample_group=factor(gene_data$sample_group, 
                                levels=c("pre rep","pregnant","post rep")) #reorder

faceted_boxplot_can=ggplot(gene_data.m,aes(y=value,fill=sample_group))+ 
  geom_boxplot(outlier.size=0,show.legend=TRUE)+
  facet_wrap(~variable,ncol=5)+
  theme_j+
  scale_fill_manual("Sample Group", 
                    values = palette_cb1, 
                    #labels = samples,
                    #                    breaks = c("pre rep","pregnant","post rep"),
  )+
  labs(title = "Candidate genes from Recknagel et al. 2021", y = "Log(CPM)")+
  theme(axis.text.x=element_blank(),axis.ticks.x=element_blank())+
  theme(legend.position = "right")+
  guides(fill = guide_legend(nrow = 3))

ggsave("candidate_genes_expressed.svg",plot=faceted_boxplot_can)

##~~~DE candidate genes~~~~~~####
#make a table for candidate genes
gene_data=data.frame(t(logcpm[candidatesDE,]))
gene_data$sample_group=ss$Condition
gene_data.m=melt(gene_data,id.vars = "sample_group",levels=c())
gene_data.m$sample_group=factor(gene_data$sample_group, 
                                levels=c("pre rep","pregnant","post rep")) #reorder

faceted_boxplot_canDE=ggplot(gene_data.m,aes(y=value,fill=sample_group))+ 
  geom_boxplot(outlier.size=0,show.legend=TRUE)+
  facet_wrap(~variable,ncol=5)+
  theme_j+
  scale_fill_manual("Sample Group", 
                    values = palette_cb1, 
                    #labels = samples,
                    #                    breaks = c("pre rep","pregnant","post rep"),
  )+
  labs(title = "Candidate genes from Recknagel et al. 2021", y = "Log(CPM)")+
  theme(axis.text.x=element_blank(),
        axis.ticks.x=element_blank(),
        legend.position = "right",
        strip.text = element_text(size = 16))+
  guides(fill = guide_legend(nrow = 3))
faceted_boxplot_canDE

ggsave("candidate_genes_DE.svg",plot=faceted_boxplot_canDE)